#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

import sys
import traceback
import logging
from dbus import DBusException

__all__=("AtabakeException", "InvalidSessionError", "DaemonStartError",
         "OwnerCheckError", "PlayerSessionError", "DBusRegisterError",
         "DisposeError", "CreatePlayerError", "ResumePlayerError",
         "PlayerError", "PlayError", "PauseError", "StopError",
         "SeekError", "GetPositionError", "GetDurationError",
         "GetMediaDetailsError", "SetVolumeError",
         "SetVideoWindowError", "SetFullScreenError")

class AtabakeException(Exception):
    """
    Base class for Atabake's exceptions.

    The ones marked with a "(D)" also inherit from DBusException.

    BaseException
     +-- Exception
          +-- AtabakeException
          |    +-- DaemonStartError
          |    +-- OwnerCheckError  (D)
          |    +-- InvalidSessionError (D)
          |    +-- PlayerSessionError  (D)
          |    |    +-- DBusRegisterError
          |    |    +-- DisposeError
          |    |    +-- CreatePlayerError
          |    |    +-- ResumePlayerError
          |    +-- PlayerError (D)
          |    |    +-- PlayError
          |    |    +-- PauseError
          |    |    +-- StopError
          |    |    +-- SeekError
          |    |    +-- GetPositionError
          |    |    +-- GetDurationError
          |    |    +-- GetMediaDetailsError
          |    |    +-- SetVolumeError
          |    |    +-- SetVideoWindowError
          |    |    +-- SetFullScreenError

    """
    (ERROR_UNKNOWN, ERROR_CREATING_PLAYER, ERROR_STOPPING, ERROR_HOLDING,
     ERROR_REGISTERING_DBUS_NAME, ERROR_PLAYING, ERROR_PLAYER_STOPPED,
     ERROR_RESUMING, ERROR_PLAYER_UNAVAILABLE, ERROR_FILE_NOT_FOUND,
     ERROR_FORMAT_NOT_SUPPORTED, ERROR_PLAYER_GENERIC) = range(12)

    def formatException(tb_level=5):
        cls, exc, tb = sys.exc_info()
        try:
            execName = cls.__name__
            execArgs = exc.args
            execTb = traceback.format_tb(tb, tb_level)[0]
            return (execName, execTb, execArgs)
        except Exception:
            return ("", "", "")

    def __init__(self, arg):
        Exception.__init__(arg)
        self.execname, self.exectb, self.execargs = self.formatException()

    def __str__(self):
        return "%s @%s: %s" % (execname, exectb, execargs)


################################################################

class InvalidSessionError(AtabakeException, DBusException):
    def __init__(self, session):
        session_error = KeyError("Invalid Session")
        AtabakeException.__init__(self, session_error)
        DBusException.__init__(self, session_error)
        self.session = session

    def __str__(self):
        return "invalid session: %s @%s" % (self.session, self.exectb)


################################################################

class OwnerCheckError(AtabakeException, DBusException):
    def __init__(self, owner):
        error = ValueError(owner)
        AtabakeException.__init__(self, error)
        DBusException.__init__(self, error)
        self.owner = owner

    def __str_(self):
        return "sender's id is different " \
            "from owner's id (%s) @%s" % (self.owner, self.exectb)

################################################################

class DaemonStartError(AtabakeException):
    """Problems while starting daemon."""
    def __init__(self, pid):
        pid_error = SystemError(pid)
        AtabakeException.__init__(self, pid_error)
        self.pid = pid

    def __str__(self):
        return "pid %s already exists. " \
               "Is it still running ?" % (self.pid)


################################################################

class PlayerSessionError(AtabakeException, DBusException):
    """Base class for PlayerSession's Exceptions"""
    def __init__(self, exception):
        AtabakeException.__init__(self, exception)
        DBusException.__init__(self, exception)


class DBusRegisterError(PlayerSessionError):
    """ """
    def __init__(self, dbus_exception):
        PlayerSessionError.__init__(dbus_exception)
        self.dbus_exception = dbus_exception

    def __str__(self):
        return "error creating session: %s @%s" % (self.execargs, self.exectb)


class DisposeError(PlayerSessionError):
    def __init__(self, msg=""):
        error = StandardError(msg)
        PlayerSessionError.__init__(error)
        self.msg = msg

    def __str__(self):
        return "%s @%s" % (self.msg, self.exectb)


class CreatePlayerError(PlayerSessionError):
    def __str__(self):
        return "error creating player @%s" % self.exectb


class ResumePlayerError(PlayerSessionError):
    def __str__(self):
        return "error resuming player @%s" % self.exectb


################################################################

class PlayerError(AtabakeException, DBusException):
    """Base class for Player's Exceptions"""
    def __init__(self, exception=None):
        AtabakeException.__init__(self, exception)
        DBusException.__init__(self, exception)
        self.e = exception

    def __str__(self):
        return "the backend's player " \
               "is not running: %s" % self.exectb


class PlayError(PlayerError):
    def __str__(self):
        return "failed while playing: %s @%s" % (self.execargs, self.exectb)


class PauseError(PlayerError):
    def __str__(self):
        return "failed while pausing: %s @%s" % (self.execargs, self.exectb)


class StopError(PlayerError):
    def __str__(self):
        return "failed while stopping: %s @%s" % (self.execargs, self.exectb)


class SeekError(PlayerError):
    def __str__(self):
        return "failed while seeking: %s @%s" % (self.execargs, self.exectb)


class GetPositionError(PlayerError):
    def __str__(self):
        return "failed while getting " \
               "position: %s @%s" % (self.execargs, self.exectb)


class GetDurationError(PlayerError):
    def __str__(self):
        return "failed while getting " \
               "duration: %s @%s" % (self.execargs, self.exectb)


class GetMediaDetailsError(PlayerError):
    def __str__(self):
        return "failed while getting " \
               "media details: %s @%s" % (self.execargs, self.exectb)


class SetVolumeError(PlayerError):
    def __str__(self):
        return "failed while setting " \
               "volume: %s @%s" % (self.execargs, self.exectb)


class SetVideoWindowError(PlayerError):
    def __str__(self):
        return "failed while setting video " \
               "window: %s @%s" % (self.execargs, self.exectb)


class SetFullScreenError(PlayerError):
    def __str__(self):
        return "failed while setting " \
               "fullscreen: %s @%s" % (self.execargs, self.exectb)
