#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

import os
import time
import array
import dbus
import dbus.service
import logging
import select
import mimetypes

import atabake.lib.utils
from atabake.lib.errors import *
from atabake.lib.config_file import AtabakeConfigFile

from dbus import DBusException

log = logging.getLogger("atabake.session")

class PlayerSession(dbus.service.Object):
    """
    Class to be used by clients to create and manage sessions.

    This class handles sessions, so the client can create 'player states'
    and reuse it, restore previous states, change player backends, etc...
    """
    DBUS_SERVICE_NAME = "br.org.indt.atabake.PlayerSession"
    DBUS_OBJ_PATH     = "/br/org/indt/atabake/PlayerSession"
    DBUS_IFACE        = "br.org.indt.atabake.PlayerSession"

    # indicating if dbus service name was already registered
    dbus_registered = False
    bus_name = None

    # amount of time to go back on media when resuming session
    resume_const = 1500

    # Player state type
    (STATE_NONE, STATE_PLAYING,
     STATE_PAUSED, STATE_ERROR, STATE_HOLD) = range(5)

    def __init__(self, session_bus, sender, url, player_set):
        self.url = url
        self.dbus_id = sender
        self.session_bus = session_bus
        self.session_path = None
        self.xid = None
        self.dispose_cb = None
        self.preferences = None

        self.player_set = player_set
        self.conf = AtabakeConfigFile()
        self.players = self.detect_fallback_players()
        self.player = self.create_player(url, self.choose_player(url))

        self.id = id(self)
        self.session_path = "%s/%s" % (self.DBUS_OBJ_PATH, id(self))
        self.dbus_register()

        self.state = self.STATE_NONE
        self.resume_state = None
        self.reset_session()

        self.pos = 0
        self.volume = 80
        self.duration = None
        self.fullscreen = False

    def is_paused(self):
        return self.state == self.STATE_PAUSED

    def is_playing(self):
        return self.state == self.STATE_PLAYING

    def is_idle(self):
        return self.state == self.STATE_NONE

    def is_holded(self):
        return self.state == self.STATE_HOLD

    def _owner_check(self, sender=None):
        """
        Function to check if the app that called
        a function is the owner of the session.

        """
        if sender and (self.dbus_id != sender):
            raise OwnerCheckError(self.dbus_id)

    def dbus_register(self):
        try:
            if not self.dbus_registered:
                self.bus_name = dbus.service.BusName(self.DBUS_SERVICE_NAME,
                                                     bus=self.session_bus)
                self.dbus_registered = True

            dbus.service.Object.__init__(self, self.bus_name, self.session_path)
            log.debug("Dbus session registered")
        except DBusException, e:
            raise DBusRegisterError(e)

    def create_player(self, uri=None, player=None):
        """
        Create a player on the bus queue and return it's id.

        """
        try:
            cls_player = self.player_set.get(player)
            obj_player = cls_player(uri, self.xid, self.get_state,
                                    self.eos_signal, self.error_signal,
                                    self.buffering_signal)
        except TypeError, e:
            raise CreatePlayerError(e)

        log.info("Created player, %s: %s", player, id(obj_player))
        return obj_player

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def dispose(self, sender=None):
        self._owner_check(sender)
        if not self.session_path:
            log.warning("It's not possible to dispose one "
                        "session that was not created yet.")
            raise DisposeError("You can't dispose one session"
                               " that was not created yet")
        try:
            if self.player:
                self.player.delete()
            self.dispose_cb(sender, self.session_path)
        except InvalidSessionError, e:
            log.error(e, exc_info=True)
            raise

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def play(self, sender=None):
        """
        Method that changes the session state and
        sends the 'play' command to the player.

        Note that the backend must emit buffering_signal(100.0)
        in order to complete the "play" command.

        """
        self._owner_check(sender)
        try:
            if self.is_idle() or self.is_holded():
                self.player.play()
            elif self.is_paused():
                self.player.pause()

            self.change_state(self.STATE_PLAYING)
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def pause(self, sender=None):
        """
        Method that changes the session state and
        sends the 'pause' command to the player.

        """
        self._owner_check(sender)
        try:
            self.pos = self.get_position()
            self.player.pause()
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        if self.is_paused():
            self.change_state(self.STATE_PLAYING)
        else:
            self.change_state(self.STATE_PAUSED)

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def stop(self, sender=None):
        """
        Method that changes the session state and
        sends the 'stop' command to the player.

        """
        self._owner_check(sender)
        try:
            self.player.stop()
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        self.change_state(self.STATE_NONE)
        self.reset_session()

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def hold(self, sender=None):
        """
        Method to be called when another player
        ask the engine to stop the active player.
        This way we should stop it but we should
        not reset_session.

        """
        self._owner_check(sender)
        try:
            # saving our position a little bit
            # earlier to be on the safe side.
            if self.state == self.STATE_PLAYING:
                pos = self.get_position()
            else:
                pos = self.pos
            self.pos = pos - min(pos, self.resume_const)
            self.resume_state = self.state
            self.player.stop()
            self.change_state(self.STATE_HOLD)
            log.debug("Holded at: %s", self.pos)
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

    def resume(self):
        """
        This method must be called to restore
        the state of a session that was put on hold.
        We must always start the player with play() and
        then pause() it if it was STATE_PAUSED.

        """
        try:
            log.info("Resuming session")
            self.player.play()
            if self.resume_state == self.STATE_PAUSED:
                self.player.pause()
            self.resume_state = None
        except Exception:
            raise ResumePlayerError()

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def is_seekable(self, sender=None):
        """
        Method that checks if the backend is seekable.

        """
        self._owner_check(sender)
        return self.player.is_seekable()

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def seek(self, position, sender=None):
        self._owner_check(sender)
        ret = (0,0)
        try:
            if not self.is_idle() and not self.is_holded() \
                    and self.is_seekable():

                ret = self.player.seek(position)
                self.pos = position
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        log.debug("Seeked to position: %s", self.pos)
        return ret

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def set_player(self, player="mplayer", sender=None):
        """
        Method that changes the player that was created
        inside __init__() to a given player (default=mplayer).

        """
        self._owner_check(sender)
        try:
            if not self.is_idle():
                self.pos = self.get_position()
                self.hold()
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        log.info("Changing player to: %s", player)

        try:
            self.player = self.create_player(self.url, player)
        except CreatePlayerError, e:
            log.error(e, exc_info=True)
            raise

    ## Getters

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def get_preferences(self, sender=None):
        self._owner_check(sender)
        log.debug("Getting preferences: %s", self.preferences)
        return self.preferences

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def get_volume(self, sender=None):
        self._owner_check(sender)
        log.debug("Getting volume: %s", self.volume)
        return self.volume

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def get_position(self, sender=None):
        self._owner_check(sender)
        pos = 0
        try:
            if not self.is_idle():
                pos = self.player.get_position()
        except PlayerError, e:
            log.warning(e, exc_info=True)
            raise

        log.debug("Getting position: %s", pos)
        return pos

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def get_duration(self, sender=None):
        self._owner_check(sender)
        try:
            if not self.is_idle():
                self.duration = self.player.get_duration()
        except PlayerError, e:
            log.warning(e, exc_info=True)
            raise

        log.debug("Getting duration: %s", self.duration)
        return self.duration

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def get_state(self, sender=None):
        """
        Method that returns the current session's state.

        """
        self._owner_check(sender)
        return self.state

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def get_video_window(self, sender=None):
        """Method that returns the current session's window ID."""
        self._owner_check(sender)
        log.debug("Getting video window: %s", self.xid)
        return self.xid

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def get_media_details(self, sender=None):
        """
        Method that returns the media details of the current uri.
        Parse details formats:
        "length", "width", "height", "audio codec", "video codec",
        "encoder", "title", "album", "artist", "genre",
        "comment", "track", "year"

        """
        self._owner_check(sender)
        ret = None
        try:
            ret = self.player.get_media_details()
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        log.debug("Getting media details: %s", ret)
        return ret

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def is_fullscreen(self, sender=None):
        self._owner_check(sender)
        log.debug("Getting fullscreen: %s", self.fullscreen)
        return self.fullscreen

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def load_preferences(self, preferences, sender=None):
        """
        Set some preferences to the player. This can
        be either a dictionary with all the options or
        the path for a config file that will be parsed by
        an object of the class Preferences.

        """
        self._owner_check(sender)
        if isinstance(preferences, dict):
            self.preferences = preferences
        elif os.path.exists(preferences):
            self.preferences = atabake.lib.utils.load_preferences(preferences)
        else:
            raise TypeError("You should provide a dictionary or"
                            " a file with options for the player")

        log.debug("Loaded preferences: %s", self.preferences)
        self.player.load_preferences(self.preferences)

    ## Setters

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def set_uri(self, uri, auto=True, sender=None):
        """
        Sets new uri for player.

        """
        self._owner_check(sender)

        # going to recreate uri as string
        uri_array = array.array("B")
        uri_array.fromlist(uri)
        self.url = uri_array.tostring()

        try:
            if not self.is_idle():
                self.stop()
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        try:
            if auto or not self.player:
                player = self.choose_player(self.url)
                if player != self.player.name:
                    log.debug("Creating another player. Old: %s, New: %s",
                              self.player.name, player)
                    self.player.delete()
                    self.player = self.create_player(self.url, player)
        except CreatePlayerError, e:
            log.error(e, exc_info=True)
            raise

        log.debug("Setting uri: %s", self.url)
        return self.player.set_uri(self.url)

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def set_volume(self, value, sender=None):
        """
        Method that sets the current session's volume.

        """
        self._owner_check(sender)
        try:
            if not self.is_idle():
                self.player.set_volume(value)
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        self.volume = value
        log.debug("Setting volume: %s", self.volume)

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def set_fullscreen(self, status, sender=None):
        """
        Method that sets the current session's fullscreen mode.

        """
        self._owner_check(sender)
        try:
            self.player.set_fullscreen(status)
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        self.fullscreen = status
        log.debug("Setting fullscreen: %s", self.fullscreen)

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def set_video_window(self, xid, sender=None):
        """
        Method that sets the current session's window ID.

        """
        self._owner_check(sender)
        try:
            if self.xid != xid:
                # stopping and resuming the active media player
                play = False
                if not self.is_idle():
                    self.stop()
                    play = True

                self.player.set_video_window(xid)
                if play:
                    self.play()
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        log.debug("Setting xid in player_session to %s", xid)
        self.xid = xid

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def reset_session(self, sender=None):
        """
        Method that resets the main session
        properties, such as position. Currently is
        maintaining the url.

        """
        self._owner_check(sender)
        try:
            if not self.is_idle():
                self.stop()
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        self.pos = 0
        self.duration = None
        self.error = None

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def change_state(self, state, sender=None):
        """
        Method that sets the current session's state and
        call the function that emits a signal.

        """
        self._owner_check(sender)
        if self.player:
            if self.state != state:
                self.state_signal(state)
        else:
            msg = "You need a player in order to change it's state"
            log.error(msg)
            raise AttributeError(msg)

    @dbus.service.signal(DBUS_IFACE)
    def state_signal(self, state):
        """
        Method that sets the current session's state and
        emits a signal using dbus.

        """
        self.state = state
        log.debug("Changed State: %s", state)
        return True

    @dbus.service.signal(DBUS_IFACE)
    def eos_signal(self):
        """
        Method that emits a eos signal using dbus.

        """
        self.change_state(self.STATE_NONE)
        self.reset_session()
        log.debug("Received EOS signal and reseted session")
        return True

    @dbus.service.signal(DBUS_IFACE)
    def error_signal(self, error):
        """
        Method that sets the error code and
        emits a signal using dbus.

        """
        self.error = error
        log.error(error)
        return True

    @dbus.service.signal(DBUS_IFACE)
    def buffering_signal(self, percentage):
        """
        Method that sets the current buffering
        percentage and emits a signal with it.

        When it reaches 100% of buffering, verifies
        if we have a session to resume and if that
        is true, resume that session.

        All backends must emit buffering_signal(100)
        when it start playing, even when there was
        no buffering at all.

        """
        self.buffering = percentage
        try:
            if float(percentage) == 100.0 and \
               self.resume_state:
                # Restore session
                self.resume_state = None
                self.set_volume(self.volume)
                if self.xid is not None:
                    log.debug("Resuming to xid: %x", self.xid)
                    self.set_video_window(self.xid)
                    self.set_fullscreen(self.fullscreen)
                log.debug("Resuming to position: %s", self.pos)
                if self.pos > 0:
                    self.seek(self.pos)
        except PlayerError, e:
            log.error(e, exc_info=True)
            raise
        return True

    def detect_fallback_players(self):
        """
        Detect and set default audio/video players
        """
        def get_player(section, option):
            if not self.conf.has_option(section, option):
                return self.player_set.list[0].name
            else:
                pl = self.conf.get(section, option)
                if self.player_set.get(pl):
                    return pl
                else:
                    return self.player_set.list[0].name

        players = {}
        # init players set
        players["audio"] = get_player("Fallback", "audio")
        players["video"] = get_player("Fallback", "video")
        # general to handle very bad medias
        players["general"] = get_player("Fallback", "general")
        log.info("Read config file por Atabake, using %s", players)

        return players

    def _choose_player_media_section(self, url):
        player = None

        if self.conf.has_section("Media"):
            media_lst = self.conf.items("Media")

            for media, pl in media_lst:
                # if we have a list of players
                if url.lower().endswith(media):
                    if isinstance(pl, list):
                        for p in pl:
                            if self.player_set.get(p):
                                player = p
                                break
                        player = self.players["general"]
                    elif isinstance(pl, str) and self.player_set.get(pl):
                        player = pl
        return player

    def _choose_player_mimetype(self, url):
        player = None
        mimetype = mimetypes.guess_type(url)[0]
        log.debug("mimetype = %s", mimetype)

        if mimetype:
            if mimetype.find("video") >= 0:
                # video
                player = self.players["video"]
            elif mimetype.find("audio") >= 0:
                # audio
                player = self.players["audio"]
            else:
                # bad url!
                player = self.players["general"]
                log.warning("Bad mimetype. Selecting general player")
        else:
            # bad media! could not even detect mymetype
            player = self.players["general"]
            log.warning("Could not handle mimetype = %s", mimetype)

        return player

    def choose_player(self, url):
        """
        Function to choose a player based on rules created here.
        The default ones says that for video we must always use mplayer
        and for audio always use oms. In the future we can specify players
        using the type of file also (separate audio cases in mp3, ogg, etc..)

        """
        try:
            player = self._choose_player_media_section(url)

            if not player:
                # ok, we need to fallback as there is
                # no special player for this media
                player = self._choose_player_mimetype(url)

        except Exception, e:
            log.warning("Error finding mimetype (%s)", e, exc_info=True)
            log.debug("PLAYERS: %s", self.players)
            player = self.players["general"]

        log.debug("Automatic player selection, returning = %s", player)
        return player

    def delete(self):
        self.remove_from_connection()
        self.dispose_cb = None
        self.session_bus = None
        self.player_set = None
        log.debug("Deleted session: %s", self)

    def __str__(self):
        return '%s(player=%r, url=%r, state=%s, position=%s,' \
               ' volume=%s,  duration=%s, fullscreen=%s)' % \
               (self.__class__.__name__, self.player, self.url,
                self.state, self.pos, self.volume, self.duration,
                self.fullscreen)


