#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Carlos Eduardo Aguiar <carlos.aguiar@openbossa.org>
#          Adriano Rezende <adriano.rezende@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import pygst
pygst.require("0.10")
import gst
import logging

from atabake.lib.player import Player
from atabake.lib.errors import AtabakeException
from atabake.lib.player_session import PlayerSession
from atabake.lib.config_file import AtabakeConfigFile

log = logging.getLogger("atabake.player.gstreamer")

class GSTreamerBackend(Player):
    name = "gstreamer"
    exec_name = "atabake"
    priority = -1
    enabled = True
    gst.MSECOND = 1000000.0

    error_bindings = {
        # 1 = Internal data flow error
        1: AtabakeException.ERROR_PLAYER_GENERIC,
        # 3 = Resource not found
        3: AtabakeException.ERROR_FILE_NOT_FOUND,
        # 4 - Resource busy
        4: AtabakeException.ERROR_PLAYING,
        # 9 = Could not read from resource
        9: AtabakeException.ERROR_PLAYING,
        # 12 = Synchronize on the resource failed
        12: AtabakeException.ERROR_PLAYING
        }

    def __init__(self, url, xid, state, eos_cb, error_cb, buff_cb):
        Player.__init__(self, url, xid, state, eos_cb, error_cb, buff_cb)

        self.conf = AtabakeConfigFile()
        self.pipeline = gst.Pipeline('pipeline')
        self.playbin = gst.element_factory_make('playbin2', None)

        if self.conf.has_option("GSTreamer", "audio_sink"):
            audio_option = self.conf.get("GSTreamer", "audio_sink")
            self.audio_sink = gst.element_factory_make(audio_option, None)
            self.playbin.set_property("audio_sink", self.audio_sink)

        if self.conf.has_option("GSTreamer", "video_sink"):
            video_option = self.conf.get("GSTreamer", "video_sink")
            self.video_sink = gst.element_factory_make(video_option, None)
            self.playbin.set_property("video_sink", self.video_sink)

        self.pipeline.add(self.playbin)

        self.volume = None

        self.reset_player()

        bus = self.pipeline.get_bus()
        bus.add_signal_watch()
        bus.enable_sync_message_emission()
        self.watch_id = bus.connect('message', self._handle_bus_message)
        self.watch_xid = bus.connect('sync-message::element', \
                                     self._handle_sync_message)

    def reset_player(self):
        self.info = {}
        self.position = 0
        self.duration = 0
        self.get_media_details()

        return True

    # Player controls

    def play(self):
        if self.xid:
            log.info("Setting xid (0x%x) in gstreamer", self.xid)
        self.pipeline.set_state(gst.STATE_PLAYING)
        return True

    def pause(self):
        if self.get_state() == PlayerSession.STATE_PAUSED:
            self.pipeline.set_state(gst.STATE_PLAYING)
        else:
            self.pipeline.set_state(gst.STATE_PAUSED)
        return True

    def stop(self):
        self.pipeline.set_state(gst.STATE_NULL)
        return True

    def is_seekable(self):
        return True

    def seek(self, position):
        event = gst.event_new_seek(1.0, gst.FORMAT_TIME,
                                   gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_ACCURATE,
                                   gst.SEEK_TYPE_SET, position * gst.MSECOND,
                                   gst.SEEK_TYPE_NONE, 0)

        if not self.playbin.send_event(event):
            log.error("error while trying to seek position %r" % position)
            return False

        return True

    # Getters

    def get_position(self):
        try:
            self.position, format = self.playbin.query_position(gst.FORMAT_TIME, None)
            self.position = int(self.position / gst.MSECOND)
            log.debug("returned position from gstreamer: %d miliseconds", self.position)
        except Exception, e:
            log.error("error while trying to get position from gstreamer: %s", e)
            self.position = 0

        return self.position

    def get_duration(self):
        try:
            self.duration, format = self.playbin.query_duration(gst.FORMAT_TIME, None)
            self.duration = int(self.duration/ gst.MSECOND)
            log.debug("returned duration from gstreamer: %d miliseconds",
                      self.duration)
        except Exception, e:
            log.error("error while trying to get duration from gstreamer.: %s", e)
            self.duration = 0

        return self.duration

    def get_media_details(self):
        # TODO
        return self.info

    # Setters

    def set_volume(self, value):
        self.volume = value
        volume = value / 100.0

        log.debug("trying to set volume in gstreamer to: %f", value)
        self.playbin.set_property('volume', volume)
        return True

    def set_uri(self, uri):
        Player.set_uri(self, uri)

        self.reset_player()

        if not uri:
            return False

        if uri.find('://') < 0:
            uri = 'file://' + uri

        log.debug("trying to set uri in gstreamer to: %s", uri)
        self.playbin.set_property("uri", uri)
        self.pipeline.set_state(gst.STATE_READY)

        # When setting uri, the volume goes back to maximum.
        # So we need to set again to the last volume played.
        # The problem is when setting the same volume it doesn't
        # come back to normal. So we need to do the hack below.
        if self.volume is not None:
            if self.volume != 0:
                self.set_volume(self.volume - 1)
            else:
                self.set_volume(self.volume + 1)

            self.set_volume(self.volume)

        return True

    def set_fullscreen(self, status):
        return True

    def set_video_window(self, xid):
        self.xid = xid
        log.debug("Setting xid in gstreamer")

    def _handle_bus_message(self, bus, message):
        log.debug("bus message: %s" % message)

        if message.type == gst.MESSAGE_EOS:
            log.debug('received EOS from bus in gstreamer')
            if self.eos_callback:
                self.eos_callback()
        elif message.type == gst.MESSAGE_ERROR:
            err, dbg = message.parse_error()
            log.debug("received error from bus in gstreamer: (%d) %s",
                      err.code, err.message)

            err_code = self.error_bindings.get(err.code,
                                               AtabakeException.ERROR_UNKNOWN)

            if self.error_callback:
                self.error_callback(err_code)
        elif message.type == gst.MESSAGE_SEGMENT_DONE:
            log.debug('received SEGMENT_DONE from bus in gstreamer')

    def _handle_sync_message(self, bus, message):
        if message.structure is None:
            return
        message_name = message.structure.get_name()
        if message_name == "prepare-xwindow-id":
            imagesink = message.src
            imagesink.set_property("force-aspect-ratio", True)
            imagesink.set_xwindow_id(self.xid)

    def delete(self):
        Player.delete(self)
        log.debug("Deleting player")

        bus = self.pipeline.get_bus()
        bus.disconnect(self.watch_id)
        bus.disconnect(self.watch_xid)
        bus.remove_signal_watch()
        bus.disable_sync_message_emission()
