#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

import os
import sys
import dbus
import socket
import logging
import mimetypes

from atabake.lib.errors import *
from atabake.lib.player import Player
from atabake.lib.player_session import PlayerSession


log = logging.getLogger("atabake.player.oms")

class OMSBackend(Player):
    """OMS backend for playing audio/video files.
    This class sends a request to the osso-media-server.
    """
    name = "oms"
    exec_name = "osso-media-server"
    priority = 0
    enabled = True

    OMS_SERVICE_NAME         = "com.nokia.osso_media_server"
    OMS_OBJECT_PATH          = "/com/nokia/osso_media_server"
    OMS_MUSIC_INTERFACE_NAME = "com.nokia.osso_media_server.music"
    OMS_VIDEO_INTERFACE_NAME = "com.nokia.osso_media_server.video"

    def __init__(self, url, xid, state, eos_cb, error_cb, buff_cb):
        Player.__init__(self, url, xid, state, eos_cb, error_cb, buff_cb)

        self._init_dbus()
        self._init_signals()
        self.reset_player()
        # to be on the safe side
        # to avoid reconnecting signals
        self._signals_connected = False

        self.state = None
        self.volume = 0
        self.oms_proxy = None
        self.set_uri(url)
        # setting xid in oms
        if xid:
            self.set_video_window(xid)

    def reset_player(self):
        self.duration = 0
        self.info = {}
        Player.reset_player(self)

    def set_interface(self):
        """Defines wich dbus interface we should use: the
        video or the music one based on the extension of the
        file"""
        if not self.url:
            return None

        mimetype = mimetypes.guess_type(self.url)[0]
        if mimetype and mimetype.find("video") >= 0:
            self.is_video = True
            self.oms_iface_name = self.OMS_VIDEO_INTERFACE_NAME
            self.oms_proxy = dbus.Interface(self.oms_object,
                                            self.OMS_VIDEO_INTERFACE_NAME)
        else:
            self.is_video = False
            self.oms_iface_name = self.OMS_MUSIC_INTERFACE_NAME
            self.oms_proxy = dbus.Interface(self.oms_object,
                                            self.OMS_MUSIC_INTERFACE_NAME)

        if self._signals_connected:
            self._disconnect_signals()
        self._connect_signals()

    def play(self):
        """Sends the command 'play' to oms"""
        try:
            self.oms_proxy.play_media(self.url)
        except dbus.DBusException, e:
            log.error(e, exc_info=True)
            raise PlayError(e)

    def pause(self):
        """Sends the command 'pause' to oms"""
        try:
            self.oms_proxy.pause()
        except dbus.DBusException, e:
            log.error(e, exc_info=True)
            raise PauseError(e)

    def stop(self):
        """Sends the command 'stop' to oms"""
        try:
            self.oms_proxy.stop()
        except dbus.DBusException, e:
            log.error(e, exc_info=True)
            raise StopError(e)

    def is_seekable(self):
        """Returns that OMS suppports seeking"""
        return True

    def seek(self, position):
        """Seeks to a given position"""
        try:
            seek_set = dbus.Int32(1)
            dbus_pos = dbus.Int32(position)
            self.oms_proxy.seek(seek_set, dbus_pos)
        except dbus.DBusException, e:
            log.error(e, exc_info=True)
            raise SeekError(e)

        return (dbus_pos, seek_set)

    def get_position(self):
        """Gets the position of the current media"""
        try:
            value = self.oms_proxy.get_position()
            if isinstance(value, int):
                pos = value
                log.debug("Get position: %s", pos)
            else:
                pos, self.duration = value
                log.debug("Get position: %s and duration: %s",
                          pos, self.duration)
        except Exception, e:
            log.error(e, exc_info=True)
            raise GetPositionError(e)

        log.debug("Got position: %s", pos)
        return pos

    def get_duration(self):
        """Gets the duration of the current media"""
        if self.duration > 0:
            return self.duration

        try:
            value = self.oms_proxy.get_position()
            if isinstance(value, tuple):
                log.debug("trying Get duration with 2 return params")
                pos, self.duration = value
            else:
                log.debug("trying Get duration from media_details")
                self.get_media_details()
                if "length" in self.info:
                    self.duration = int(self.info["length"])
        except dbus.DBusException, e:
            log.error(e, exc_info=True)
            raise GetDurationError(e)

        log.debug("Got duration: %s", self.duration)
        return self.duration

    def get_media_details(self):
        """Gets details of the current media"""
        if not self.url:
            return {}

        try:
            details = \
                self.oms_proxy.get_media_details(self.url)
        except dbus.DBusException, e:
            log.error(e, exc_info=True)
            raise GetMediaDetailsError(e)

        # parse details formats:
        # - video: "length", "width", "height", "audio codec",
        # "video codec", "encoder"
        # - audio: "length", "artist", "bitrate", "title"
        # - iradio: "iradio-name", "iradio-url", "iradio-genre"
        if "bitrate" in details:
            # change "bitrate" to "audio bitrate" or "video bitrate"
            value = details.pop("bitrate")
            if self.is_video:
                new_key = "video bitrate"
            else:
                new_key = "audio bitrate"

            details[new_key] = value

        log.info(str(details))
        self.info = details
        return self.info

    def set_volume(self, value):
        """Sets OMS volume"""
        def send_msg(msg):
            s = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            s.connect("/tmp/mediaplayer-engine.socket")
            s.sendall(msg)
            s.close()

        try:
            log.debug("Volume set to %s", value)
            if os.path.exists("/tmp/mediaplayer-engine.socket"):
                send_msg("volume\n%s\n" % value)
            else:
                tmp = value / 100.0
                self.oms_proxy.set_volume(tmp)
            self.volume = value
        except Exception, e:
            log.error(e, exc_info=True)
            raise SetVolumeError(e)

    def set_uri(self, uri):
        """Sets a new uri and also set the new dbus interface
        of OSSO Media Engine"""
        Player.set_uri(self, uri)
        self.reset_player()
        self.set_interface()
        self.get_media_details()
        return True

    # Video methods

    def set_fullscreen(self, status):
        """Sets OMS to fullscreen"""
        return True

    def set_video_window(self, xid):
        """Sets a new window ID"""
        try:
            log.debug("Setting xid (0x%x) in oms", xid)
            video_proxy = \
                dbus.Interface(self.oms_object,
                               self.OMS_VIDEO_INTERFACE_NAME)

            dbus_xid = dbus.UInt32(xid)
            video_proxy.set_video_window(dbus_xid)
        except dbus.DBusException, e:
            log.error(e, exc_info=True)
            raise SetVideoWindowError(e)

    def _init_dbus(self):
        self.bus = dbus.SessionBus()

        # resolving unique bus name for oms
        bus_object = self.bus.get_object('org.freedesktop.DBus',
                                         '/org/freedesktop/DBus',
                                         introspect=False)
        self.oms_bus_name = bus_object.GetNameOwner(
            self.OMS_SERVICE_NAME,
            dbus_interface='org.freedesktop.DBus')

        self.oms_object = self.bus.get_object(self.oms_bus_name,
                                              self.OMS_OBJECT_PATH,
                                              introspect=False)

    def _init_signals(self):
        """ Defining oms signals and callbacks """

        self.oms_status_signals = \
            {"begin_buffering": self._handle_buffering,
             "info_buffering": self._handle_buffering,
             "end_buffering": self._handle_buffering,
             "state_changed": self._handle_state_changed,
             "end_of_stream": self._handle_eos}

        self.oms_error_signals = \
            {"no_media_selected": AtabakeException.ERROR_PLAYER_GENERIC,
             "file_not_found": AtabakeException.ERROR_FILE_NOT_FOUND,
             "type_not_found": AtabakeException.ERROR_FORMAT_NOT_SUPPORTED,
             "unsupported_type": AtabakeException.ERROR_FORMAT_NOT_SUPPORTED,
             "gstreamer": AtabakeException.ERROR_PLAYER_GENERIC,
             "no_connection": AtabakeException.ERROR_PLAYER_GENERIC,
             "dsp": AtabakeException.ERROR_PLAYER_GENERIC,
             "device_unavailable": AtabakeException.ERROR_PLAYER_UNAVAILABLE,
             "corrupted_file": AtabakeException.ERROR_PLAYER_GENERIC,
             "out_of_memory": AtabakeException.ERROR_PLAYER_GENERIC,
             "video_codec_not_supported": AtabakeException.ERROR_FORMAT_NOT_SUPPORTED,
             "audio_codec_not_supported": AtabakeException.ERROR_FORMAT_NOT_SUPPORTED,
             "resolution_not_supported": AtabakeException.ERROR_FORMAT_NOT_SUPPORTED}

    def _connect_signals(self):
        log.info("connecting oms signals...")
        self._connect_status_signals()
        self._connect_error_signals()
        self._signals_connected = True

    def _disconnect_signals(self):
        if not self.oms_proxy:
            return

        log.info("disconnecting oms signals...")
        self._signals_connected = False

        for sig_name in self.oms_status_signals.iterkeys():
            self._disconnect_signal(sig_name)

        for sig_name in self.oms_error_signals.iterkeys():
            self._disconnect_signal(sig_name)

    def _connect_status_signals(self):
        if not self.oms_proxy:
            return

        for (sig, cb) in self.oms_status_signals.iteritems():
            self.oms_proxy.connect_to_signal(sig, cb)

    def _connect_error_signals(self):
        if not self.oms_proxy:
            return

        for error_sig in self.oms_error_signals.iterkeys():
            self.oms_proxy.connect_to_signal(error_sig,
                                             self._handle_error)

    def _disconnect_signal(self, signal_name):
        try:
            self.bus.remove_signal_receiver(None,
                                            signal_name,
                                            self.oms_iface_name,
                                            self.oms_bus_name,
                                            self.OMS_OBJECT_PATH)
        except dbus.DBusException, e:
            # dbus-python version 0.71 has a bug,
            # always raising exception in signal removing
            pass

    def _handle_state_changed(self, state):
        """
        For now just logging state change:
        ["playing", "paused", "stopped", "connecting"]

        Emit buffering_signal(100) when reaches playing state
        as PlayerSession's docs pointed out.

        """
        log.info("Received state-changed signal from oms: %s", state)
        if state == "playing":
            self.buffering_callback(100.0)
        self.state = state

    def _handle_eos(self, value):
        log.debug("In handle eos: %s", value)
        if not self.is_idle() and self.eos_callback:
            self.eos_callback()
        self.reset_player()

    def _handle_buffering(self, percentage):
        log.info("Received buffering signal: %s", percentage)
        self.buffering_callback(percentage)

    def _handle_error(self, msg):
        log.info("Received error signal = %s", msg)
        self.error_callback(
            self.oms_error_signals.get(msg, AtabakeException.ERROR_UNKNOWN))

    def delete(self):
        Player.delete(self)
        log.debug("Deleting player")
        self._disconnect_signals()
