#include <dbus/dbus-glib.h>

#include "bkl-db.h"
#include "bkl-internal-marshal.h"
#include "bkl-source-client.h"

#include "bkl-source-bindings.h"

struct _BklSourceClientPrivate {
    DBusGProxy *proxy;
    char *object_path;

    BklDB *db;
};

enum {
    READY,
    ADDED,
    DELETED,
    CHANGED,
    IN_PROGRESS,
    COMPLETED,
    INDEX_CHANGED,
    LAST_SIGNAL
};

enum {
    PROP_0,
    PROP_PATH,
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BKL_TYPE_SOURCE_CLIENT, BklSourceClientPrivate))
G_DEFINE_TYPE (BklSourceClient, bkl_source_client, G_TYPE_OBJECT);

static guint32 signals[LAST_SIGNAL] = {0, };

static void uri_changed (DBusGProxy      *proxy,
                         const char      *uri,
                         BklSourceClient *client);
static void uri_added (DBusGProxy      *proxy,
                       const char      *uri,
                       BklSourceClient *client);
static void uri_deleted (DBusGProxy      *proxy,
                         const char      *uri,
                         BklSourceClient *client);
static void index_in_progress (DBusGProxy      *proxy,
                               BklSourceClient *client);
static void index_complete (DBusGProxy      *proxy,
                            BklSourceClient *client);

static void
bkl_source_client_finalize (GObject *object)
{
    BklSourceClient *self = (BklSourceClient *) object;
    BklSourceClientPrivate *priv = self->priv;

    if (priv->object_path) {
        g_free (priv->object_path);
    }

    if (priv->db) {
        bkl_db_free (priv->db);
    }

    G_OBJECT_CLASS (bkl_source_client_parent_class)->finalize (object);
}

static void
bkl_source_client_dispose (GObject *object)
{
    BklSourceClient *self = (BklSourceClient *) object;
    BklSourceClientPrivate *priv = self->priv;

    if (priv->proxy) {
        dbus_g_proxy_disconnect_signal (priv->proxy, "UriChanged",
                                        G_CALLBACK (uri_changed), object);
        dbus_g_proxy_disconnect_signal (priv->proxy, "UriAdded",
                                        G_CALLBACK (uri_added), object);
        dbus_g_proxy_disconnect_signal (priv->proxy, "UriDeleted",
                                        G_CALLBACK (uri_deleted), object);
        dbus_g_proxy_disconnect_signal (priv->proxy, "InProgress",
                                        G_CALLBACK (index_in_progress), object);
        dbus_g_proxy_disconnect_signal (priv->proxy, "Complete",
                                        G_CALLBACK (index_complete), object);
        g_object_unref (priv->proxy);
        priv->proxy = NULL;
    }

    G_OBJECT_CLASS (bkl_source_client_parent_class)->dispose (object);
}

static void
bkl_source_client_set_property (GObject      *object,
                                guint         prop_id,
                                const GValue *value,
                                GParamSpec   *pspec)
{
    BklSourceClient *self = (BklSourceClient *) object;
    BklSourceClientPrivate *priv = self->priv;

    switch (prop_id) {
    case PROP_PATH:
        priv->object_path = g_value_dup_string (value);
        break;

    default:
        break;
    }
}

static void
bkl_source_client_get_property (GObject    *object,
                                guint       prop_id,
                                GValue     *value,
                                GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
uri_changed (DBusGProxy      *proxy,
             const char      *uri,
             BklSourceClient *client)
{
    g_signal_emit (client, signals[CHANGED], 0, uri);
}

static void
uri_added (DBusGProxy      *proxy,
           const char      *uri,
           BklSourceClient *client)
{
    g_signal_emit (client, signals[ADDED], 0, uri);
}

static void
uri_deleted (DBusGProxy      *proxy,
             const char      *uri,
             BklSourceClient *client)
{
    g_signal_emit (client, signals[DELETED], 0, uri);
}

static void
index_in_progress (DBusGProxy      *proxy,
                   BklSourceClient *client)
{
    g_signal_emit (client, signals[IN_PROGRESS], 0);
}

static void
index_complete (DBusGProxy      *proxy,
                BklSourceClient *client)
{
    g_signal_emit (client, signals[COMPLETED], 0);
}

static void
index_changed (DBusGProxy      *proxy,
               GPtrArray       *added,
               GPtrArray       *removed,
               BklSourceClient *client)
{
    g_signal_emit (client, signals[INDEX_CHANGED], 0, added, removed);
}

static void
get_db_path_reply (DBusGProxy *proxy,
                   char       *OUT_path,
                   GError     *error,
                   gpointer    userdata)
{
    BklSourceClient *client = (BklSourceClient *) userdata;
    BklSourceClientPrivate *priv = client->priv;
    GError *error2 = NULL;

    if (error) {
        g_warning ("Error getting DB path for %s: %s", priv->object_path,
                   error->message);

        priv->db = NULL;
        g_signal_emit (client, signals[READY], 0);

        g_error_free (error);
        return;
    }

    priv->db = bkl_db_get_for_path (OUT_path, "", &error2);
    if (priv->db == NULL) {
        g_warning ("Error opening DB at %s: %s", OUT_path, error2->message);
        g_error_free (error2);
    }

    g_signal_emit (client, signals[READY], 0);

    g_free (OUT_path);
}

static GObject *
bkl_source_client_constructor (GType                  type,
                               guint                  n_params,
                               GObjectConstructParam *params)
{
    BklSourceClient *client;
    BklSourceClientPrivate *priv;
    DBusGConnection *connection;
    GError *error = NULL;

    client = BKL_SOURCE_CLIENT (G_OBJECT_CLASS (bkl_source_client_parent_class)->constructor
                                (type, n_params, params));
    priv = client->priv;

    connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (connection == NULL) {
        g_warning ("Failed to open connection to D-Bus: %s",
                   error->message);
        g_error_free (error);

        priv->proxy = NULL;
        priv->db = NULL;
        return (GObject *) client;
    }

    priv->proxy = dbus_g_proxy_new_for_name (connection,
                                             BKL_SOURCE_DBUS_SERVICE,
                                             priv->object_path,
                                             BKL_SOURCE_DBUS_INTERFACE);
    dbus_g_proxy_add_signal (priv->proxy, "UriChanged",
                             G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_add_signal (priv->proxy, "UriAdded",
                             G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_add_signal (priv->proxy, "UriDeleted",
                             G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_add_signal (priv->proxy, "InProgress", G_TYPE_INVALID);
    dbus_g_proxy_add_signal (priv->proxy, "Complete", G_TYPE_INVALID);
    dbus_g_proxy_add_signal (priv->proxy, "IndexChanged",
                             G_TYPE_STRV, G_TYPE_STRV, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (priv->proxy, "UriChanged",
                                 G_CALLBACK (uri_changed), client, NULL);
    dbus_g_proxy_connect_signal (priv->proxy, "UriAdded",
                                 G_CALLBACK (uri_added), client, NULL);
    dbus_g_proxy_connect_signal (priv->proxy, "UriDeleted",
                                 G_CALLBACK (uri_deleted), client, NULL);
    dbus_g_proxy_connect_signal (priv->proxy, "InProgress",
                                 G_CALLBACK (index_in_progress), client, NULL);
    dbus_g_proxy_connect_signal (priv->proxy, "Complete",
                                 G_CALLBACK (index_complete), client, NULL);
    dbus_g_proxy_connect_signal (priv->proxy, "IndexChanged",
                                 G_CALLBACK (index_changed), client, NULL);

    org_moblin_Bickley_Source_get_db_path_async (priv->proxy,
                                                 get_db_path_reply, client);

    return (GObject *) client;
}

static void
bkl_source_client_class_init (BklSourceClientClass *klass)
{
    GObjectClass *o_class = (GObjectClass *)klass;

    o_class->dispose = bkl_source_client_dispose;
    o_class->finalize = bkl_source_client_finalize;
    o_class->set_property = bkl_source_client_set_property;
    o_class->get_property = bkl_source_client_get_property;
    o_class->constructor = bkl_source_client_constructor;

    g_type_class_add_private (klass, sizeof (BklSourceClientPrivate));

    dbus_g_object_register_marshaller (bkl_internal_marshal_VOID__POINTER_POINTER,
                                       G_TYPE_NONE, G_TYPE_STRV, G_TYPE_STRV,
                                       G_TYPE_INVALID);
    g_object_class_install_property
        (o_class, PROP_PATH,
         g_param_spec_string ("object-path",
                              "Object path",
                              "The D-Bus path to the object",
                              "",
                              G_PARAM_WRITABLE |
                              G_PARAM_CONSTRUCT_ONLY |
                              G_PARAM_STATIC_STRINGS));

    signals[READY] = g_signal_new ("ready",
                                   G_TYPE_FROM_CLASS (klass),
                                   G_SIGNAL_RUN_FIRST |
                                   G_SIGNAL_NO_RECURSE,
                                   0, NULL, NULL,
                                   g_cclosure_marshal_VOID__VOID,
                                   G_TYPE_NONE, 0);
    signals[CHANGED] = g_signal_new ("uri-changed",
                                     G_TYPE_FROM_CLASS (klass),
                                     G_SIGNAL_RUN_FIRST |
                                     G_SIGNAL_NO_RECURSE,
                                     0, NULL, NULL,
                                     g_cclosure_marshal_VOID__STRING,
                                     G_TYPE_NONE, 1,
                                     G_TYPE_STRING);
    signals[ADDED] = g_signal_new ("uri-added",
                                   G_TYPE_FROM_CLASS (klass),
                                   G_SIGNAL_RUN_FIRST |
                                   G_SIGNAL_NO_RECURSE,
                                   0, NULL, NULL,
                                   g_cclosure_marshal_VOID__STRING,
                                   G_TYPE_NONE, 1,
                                   G_TYPE_STRING);
    signals[DELETED] = g_signal_new ("uri-deleted",
                                     G_TYPE_FROM_CLASS (klass),
                                     G_SIGNAL_RUN_FIRST |
                                     G_SIGNAL_NO_RECURSE,
                                     0, NULL, NULL,
                                     g_cclosure_marshal_VOID__STRING,
                                     G_TYPE_NONE, 1,
                                     G_TYPE_STRING);
    signals[IN_PROGRESS] = g_signal_new ("in-progress",
                                         G_TYPE_FROM_CLASS (klass),
                                         G_SIGNAL_RUN_FIRST |
                                         G_SIGNAL_NO_RECURSE,
                                         0, NULL, NULL,
                                         g_cclosure_marshal_VOID__VOID,
                                         G_TYPE_NONE, 0);
    signals[COMPLETED] = g_signal_new ("complete",
                                       G_TYPE_FROM_CLASS (klass),
                                       G_SIGNAL_RUN_FIRST |
                                       G_SIGNAL_NO_RECURSE,
                                       0, NULL, NULL,
                                       g_cclosure_marshal_VOID__VOID,
                                       G_TYPE_NONE, 0);
    signals[INDEX_CHANGED] = g_signal_new ("index-changed",
                                           G_TYPE_FROM_CLASS (klass),
                                           G_SIGNAL_RUN_FIRST |
                                           G_SIGNAL_NO_RECURSE,
                                           0, NULL, NULL,
                                           bkl_internal_marshal_VOID__POINTER_POINTER,
                                           G_TYPE_NONE, 2, G_TYPE_STRV,
                                           G_TYPE_STRV);
}

static void
bkl_source_client_init (BklSourceClient *self)
{
    self->priv = GET_PRIVATE (self);
}

BklSourceClient *
bkl_source_client_new (const char *object_path)
{
    return g_object_new (BKL_TYPE_SOURCE_CLIENT,
                         "object-path", object_path,
                         NULL);
}

gboolean
bkl_source_client_update_item (BklSourceClient *client,
                               BklItem         *item,
                               GError         **error)
{
    return FALSE;
}

BklDB *
bkl_source_client_get_db (BklSourceClient *client)
{
    BklSourceClientPrivate *priv;

    g_return_val_if_fail (IS_BKL_SOURCE_CLIENT (client), NULL);

    priv = client->priv;

    return priv->db;
}

const char *
bkl_source_client_get_path (BklSourceClient *client)
{
    BklSourceClientPrivate *priv;

    g_return_val_if_fail (IS_BKL_SOURCE_CLIENT (client), NULL);

    priv = client->priv;

    return priv->object_path;
}
