/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <gio/gio.h>

#include <bickley/bkl-db.h>
#include <bickley/bkl-entry.h>
#include <bickley/bkl-item-extended.h>

#include "bkl-finder.h"
#include "bkl-indexer.h"
#include "bkl-orbiter.h"
#include "bkl-path-finder.h"
#include "bkl-source.h"
#include "bkl-source-removable.h"

typedef struct _BklSourceRemovable {
    BklSource source;

    BklFinder *finder;

    int uri_count;
    /* List of the uris in the database and the
       time they were last indexed at */
    GHashTable *last_indexed;
    GHashTable *dead_uris;
} BklSourceRemovable;

typedef struct _BklSourceRemovableClass {
    BklSourceClass parent_class;
} BklSourceRemovableClass;

G_DEFINE_TYPE (BklSourceRemovable, bkl_source_removable, BKL_TYPE_SOURCE);

static void
bkl_source_removable_finalize (GObject *object)
{
    BklSourceRemovable *removable = (BklSourceRemovable *) object;
    BklSource *source = (BklSource *) object;

    if (source->db) {
        bkl_db_free (source->db);
        source->db = NULL;
    }

    g_free (source->name);

    if (removable->finder) {
        bkl_finder_destroy (removable->finder);
        removable->finder = NULL;
    }

    if (removable->last_indexed) {
        g_hash_table_destroy (removable->last_indexed);
        removable->last_indexed = NULL;
    }

    if (removable->dead_uris) {
        g_hash_table_destroy (removable->dead_uris);
        removable->dead_uris = NULL;
    }

    G_OBJECT_CLASS (bkl_source_removable_parent_class)->finalize (object);
}

static void
bkl_source_removable_dispose (GObject *object)
{
    G_OBJECT_CLASS (bkl_source_removable_parent_class)->dispose (object);
}

static void
delete_dead_keys (gpointer key,
                  gpointer value,
                  gpointer userdata)
{
    BklSource *source = userdata;
    BklItem *item = NULL;
    GError *error = NULL;
    char *uri = key;

    item = bkl_db_get_item (source->db, uri, &error);
    if (error != NULL) {
        if (error->code != KOZO_DB_ERROR_KEY_NOT_FOUND) {
            g_warning ("Error getting item for %s: %s", uri, error->message);
            return;
        }

        g_error_free (error);
        error = NULL;
    }

    if (item) {
        bkl_indexer_remove_item (source, item, source->db->db);
    }

    bkl_db_remove (source->db, uri, &error);
    if (error != NULL) {
        g_warning ("(%s): Error removing dead uri %s: %s",
                   source->name, uri, error->message);
        g_error_free (error);
    }
    bkl_source_uri_deleted (source, uri);
}

static gboolean
bkl_source_removable_do_work (BklSource *source)
{
    BklSourceRemovable *removable = (BklSourceRemovable *) source;

    if (source->working == FALSE) {
        source->working = TRUE;
        bkl_source_in_progress (source);
    }

    source->more_work = bkl_finder_lazy_dig (removable->finder);

    if (source->more_work == FALSE) {
        if (removable->dead_uris) {
            /* Now we've spidered everything we know what we can delete, yey! */
            g_print ("(%s): We have %d dead uris\n", source->name,
                     g_hash_table_size (removable->dead_uris));
            g_hash_table_foreach (removable->dead_uris, delete_dead_keys,
                                  removable);

            g_hash_table_destroy (removable->dead_uris);
            removable->dead_uris = NULL;
        }

        /* The index is not complete here, as we are waiting for the
           investigator to finish doing its stuff */
        source->working = FALSE;
    }

    return source->more_work;
}

/* FIXME: Can this be shared in BklSource? */
static void
bkl_source_removable_investigate_uris (BklSource *source,
                                       GPtrArray *files)
{
    BklSourceRemovable *removable = (BklSourceRemovable *) source;
    GPtrArray *fa;
    int i;

    fa = g_ptr_array_new ();

    for (i = 0; i < files->len; i++) {
        InvestigateUri *iu = files->pdata[i];
        char *uri = g_file_get_uri (iu->file);
        gpointer value;

        /* When we see a URI remove it from the dead_uris table */
        if (removable->dead_uris) {
            g_hash_table_remove (removable->dead_uris, uri);
        }

        value = g_hash_table_lookup (removable->last_indexed, uri);
        if (value != NULL) {
            if (iu->mtime.tv_sec <= GPOINTER_TO_INT (value)) {
                g_free (uri);

                continue;
            }
        }

        g_ptr_array_add (fa, uri);
    }

    removable->uri_count += bkl_orbiter_investigate_files (source, fa);
    /* investigate_files takes ownership of the uris */
    g_ptr_array_free (fa, TRUE);
}

/* FIXME: Can this be shared in BklSource? */
static void
bkl_source_removable_remove_uris (BklSource *source,
                                  GPtrArray *files)
{
    BklSourceRemovable *removable = (BklSourceRemovable *) source;
    int i;

    for (i = 0; i < files->len; i++) {
        char *uri;
        GError *error = NULL;
        BklItem *item;

        uri = g_file_get_uri (files->pdata[i]);

        item = bkl_db_get_item (source->db, uri, &error);
        if (error != NULL) {
            if (error->code != KOZO_DB_ERROR_KEY_NOT_FOUND) {
                return;
            }

            g_error_free (error);
            error = NULL;
        }

        bkl_indexer_remove_item (source, item, source->db->db);
        g_object_unref (item);

        bkl_db_remove (source->db, uri, &error);
        if (error) {
            g_warning ("(%s): Error removing %s: %s", source->name, uri,
                       error->message);
            g_error_free (error);
        }

        bkl_source_uri_deleted (source, uri);
        if (removable->dead_uris) {
            g_hash_table_remove (removable->dead_uris, uri);
        }
        g_free (uri);
    }
}

/* FIXME: Can this be shared in BklSource? */
static GPtrArray *
copy_string_array (GPtrArray *original)
{
    GPtrArray *clone;
    int i;

    if (original == NULL) {
        return NULL;
    }

    clone = g_ptr_array_sized_new (original->len);
    for (i = 0; i < original->len; i++) {
        g_ptr_array_add (clone, g_strdup (original->pdata[i]));
    }

    return clone;
}

static gboolean
bkl_source_removable_add_item (BklSource  *source,
                               const char *uri,
                               BklItem    *item,
                               GError    **error)
{
    BklSourceRemovable *removable = (BklSourceRemovable *) source;
    BklItem *old;

    old = bkl_db_get_item (source->db, uri, error);
    if (*error != NULL) {
        if ((*error)->code != KOZO_DB_ERROR_KEY_NOT_FOUND) {
            return FALSE;
        }

        g_error_free (*error);
        *error = NULL;
    }

    if (old == NULL) {
        bkl_db_add_item (source->db, uri, item, error);
        bkl_indexer_index_item (source, item, source->db->db, error);
        bkl_source_uri_added (source, uri);
    } else {
        BklItemExtended *ext = (BklItemExtended *) old;
        GPtrArray *tags;

                /* Copy the extended details from the existing item
           as these shouldn't change on update */
        bkl_item_extended_set_use_count
            ((BklItemExtended *) item, bkl_item_extended_get_use_count (ext));
        bkl_item_extended_set_last_used
            ((BklItemExtended *) item, bkl_item_extended_get_last_used (ext));
        bkl_item_extended_set_rating
            ((BklItemExtended *) item, bkl_item_extended_get_rating (ext));
        bkl_item_extended_set_pinned
            ((BklItemExtended *) item, bkl_item_extended_get_pinned (ext));

        tags = bkl_item_extended_get_tags (ext);
        bkl_item_extended_set_tags ((BklItemExtended *) item,
                                    copy_string_array (tags));

        bkl_db_replace_item (source->db, uri, item, error);
        bkl_indexer_update_item (source, item, old, source->db->db);

        g_object_unref (old);

        bkl_source_uri_changed (source, uri);
    }

    if (--removable->uri_count == 0) {
        bkl_source_completed (source);
    }

    return TRUE;
}

static gboolean
get_last_indexed (KozoDB     *db,
                  const char *key,
                  KozoEntry  *entry,
                  gpointer    userdata)
{
    BklSourceRemovable *source = (BklSourceRemovable *) userdata;
    KozoField *field;
    const char *data;
    glong t;

    field = bkl_entry_get_field (entry, BKL_FIELD_FILE);
    if (field == NULL) {
        return TRUE;
    }

    data = bkl_file_field_get_string (field, BKL_FILE_FIELD_MODIFICATION_TIME);
    if (data == NULL || *data == '\0') {
        return TRUE;
    }

    t = strtol (data, NULL, 10);
    g_hash_table_insert (source->last_indexed, g_strdup (key),
                         GINT_TO_POINTER (t));
    g_hash_table_insert (source->dead_uris, g_strdup (key), NULL);

    kozo_field_free (field);

    return TRUE;
}

static gboolean
kozo_init (BklSourceRemovable *source,
           GMount             *mount)
{
    BklSource *s = (BklSource *) source;
    GError *error = NULL;
    char *dirname, *filename, *name, *enc, *path;
    GFile *root;

    root = g_mount_get_root (mount);
    name = g_mount_get_name (mount);

    path = g_file_get_path (root);
    g_object_unref (root);
    if (path == NULL) {
        g_free (name);
        return FALSE;
    }

    enc = g_compute_checksum_for_string (G_CHECKSUM_MD5, path, -1);

    if (access (path, R_OK | W_OK) == 0) {
        dirname = g_build_filename (path, ".kozo", "databases", NULL);

        if (g_mkdir_with_parents (dirname, 0755) < 0) {
            g_warning ("Error making dir %s", strerror (errno));
        }
        filename = g_build_filename (dirname, enc, NULL);
        g_free (enc);
        g_free (dirname);

        s->db = bkl_db_get_for_path (filename, name, &error);
        g_free (filename);
    } else {
        filename = g_build_filename (g_get_home_dir (), ".kozo", "databases",
                                     enc, NULL);
        g_print ("%s is not writable, falling back to local: %s\n", path, name);
        g_free (enc);

        s->db = bkl_db_get_for_path (filename, name, &error);
        g_free (filename);
    }

    if (s->db == NULL) {
        g_warning ("Error getting Bickley database: %s", error->message);
        g_error_free (error);

        g_free (path);
        g_free (name);
        return FALSE;
    }

    s->name = name;

    source->last_indexed = g_hash_table_new_full (g_str_hash, g_str_equal,
                                                  g_free, NULL);
    /* This could probably be done quicker with some sort of prefix tree */
    source->dead_uris = g_hash_table_new_full (g_str_hash, g_str_equal,
                                               g_free, NULL);
    kozo_db_foreach (s->db->db, get_last_indexed, source);
    return TRUE;
}

static void
bkl_source_removable_class_init (BklSourceRemovableClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    BklSourceClass *s_class = (BklSourceClass *) klass;

    o_class->finalize = bkl_source_removable_finalize;
    o_class->dispose = bkl_source_removable_dispose;

    s_class->do_work = bkl_source_removable_do_work;
    s_class->investigate_files = bkl_source_removable_investigate_uris;
    s_class->remove_files = bkl_source_removable_remove_uris;
    s_class->add_item = bkl_source_removable_add_item;
}

static void
bkl_source_removable_init (BklSourceRemovable *removable)
{
}

BklSource *
bkl_source_removable_new (GMount *mount)
{
    BklSourceRemovable *removable;
    BklSource *source;
    GFile *root;
    char *uri;

    removable = g_object_new (BKL_TYPE_SOURCE_REMOVABLE, NULL);
    source = (BklSource *) removable;

    if (kozo_init (removable, mount) == FALSE) {
        g_free (source->name);
        g_object_unref (removable);
        return NULL;
    }

    source->more_work = TRUE;

    root = g_mount_get_root (mount);
    uri = g_file_get_uri (root);
    g_object_unref (root);

    removable->finder = bkl_path_finder_new (source, uri);
    g_free (uri);
    removable->uri_count = 0;

    source->working = FALSE;

    return source;
}
