/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <gst/gst.h>
#include <gio/gio.h>
#include <string.h>
#include "bkl.h"
#include "bkl-investigator.h"

#include <ctype.h>


static void tag_foreach (const GstTagList * list,
                         const gchar      * tag,
                         const gchar      * path)
{
  gint i, count;

  count = gst_tag_list_get_tag_size (list, tag);

  for (i = 0; i < count; ++i)
    {
      const GValue *value = gst_tag_list_get_value_index (list, tag, i);
      GValue str_value = {0, };

      g_value_init (&str_value, G_TYPE_STRING);
      g_value_transform (value, &str_value);

      bkl_investigator_set_key ("gst", path, tag, g_value_get_string (&str_value)); 

      g_value_unset (&str_value);
    }
}

static void consume_metadata (GstElement  *element,
                              const gchar *uri)
{
  GstBus *bus = gst_element_get_bus (element);
  g_return_if_fail (bus != 0);

  gboolean done = FALSE;
  while (!done)
    {
      GstMessage *message;

      message = gst_bus_pop  (bus);

      if (message == NULL)
        done = TRUE;
      else switch (GST_MESSAGE_TYPE (message))
        {
          case GST_MESSAGE_EOS:
            gst_message_unref (message);
            done = TRUE;
            break;
          case GST_MESSAGE_TAG:
            {
              GstTagList *tag_list = 0;
              gst_message_parse_tag (message, &tag_list);
              if (tag_list)
                {
                  gst_tag_list_foreach (tag_list,
                                  (GstTagForeachFunc) tag_foreach, (void*)uri);
                  gst_tag_list_free (tag_list);
                }
            }
            gst_message_unref (message);
            break;
          case GST_MESSAGE_ERROR:
              {
                GError *error = 0;
                gst_message_parse_error (message, &error, 0);
                g_print ("%s ERROR: %s\n", G_STRLOC, error->message);
                g_error_free (error);
                gst_message_unref (message);
                done = TRUE;
                break;
              }
          default:
            gst_message_unref (message);
            break;
        }
     }
  gst_object_unref (bus);
}

/* The spider is single threaded, during meta data extraction itself when the
 * spider might be blocking itself from execution the database is not locked.
 */

gboolean bkl_task_gst_meta (GFile *file)
{
  gchar *uri;


  uri = g_file_get_uri (file);

    {
      GstStateChangeReturn state;
      GstElement *element, *audiosink, *videosink;

      element    = gst_element_factory_make ("playbin",  "playbin");
      audiosink  = gst_element_factory_make ("fakesink", "fakesink0");
      videosink  = gst_element_factory_make ("fakesink", "fakesink0");

      g_object_set (G_OBJECT (element), "uri", uri, "audio-sink", audiosink, "video-sink", videosink, NULL);
        
      gint change_timeout = 10;

      state = gst_element_set_state (element, GST_STATE_PAUSED);
      while (state == GST_STATE_CHANGE_ASYNC && change_timeout > 0)
        {
          GstState status;
          state = gst_element_get_state (GST_ELEMENT (element), &status, 0, 0.2 * GST_SECOND);
          change_timeout--;
        }

      if (state != GST_STATE_CHANGE_FAILURE &&
          state != GST_STATE_CHANGE_ASYNC)
        {
          GstBus *bus = gst_element_get_bus (GST_ELEMENT (element));
          if (bus)
            {
              gst_bus_post (bus, gst_message_new_application (GST_OBJECT (element), 0));
              gst_object_unref (bus);
            }

          consume_metadata (GST_ELEMENT (element), uri);
        }

      gst_element_set_state (GST_ELEMENT (element), GST_STATE_NULL);
      gst_object_unref (GST_ELEMENT (element));
    }
  g_free (uri);
  return TRUE;
}


void bkl_task_gst_meta_register (void)
{
  {
    const gchar *task = "bkl:task_gst_meta_audio";
    bkl_uri_set_key (task, "bkl:taskPriority", "60");
    bkl_uri_set_key (task, "bkl:taskPattern", "audio/.*");
    bkl_uri_set_key (task, "bkl:taskQueue", "audio-meta");
    bkl_task_set_handler (task, bkl_task_gst_meta);
  }
  {
    const gchar *task = "bkl:task_gst_meta_video";
    bkl_uri_set_key (task, "bkl:taskPriority", "60");
    bkl_uri_set_key (task, "bkl:taskPattern", "video/.*");
    bkl_uri_set_key (task, "bkl:taskQueue", "video-meta");
    bkl_task_set_handler (task, bkl_task_gst_meta);
  }
}
