#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import ecore

from terra.core.model import Model


class Notify(Model):
    terra_type = "Model/Notify"

    def __init__(self, name, message, answer_callback=None):
        self.message_changed_callback = None
        self.answer_callback = answer_callback
        Model.__init__(self, name)
        self._message = message

    def message_get(self):
        return self._message

    def message_set(self, message):
        if self.message_changed_callback is not None:
            self.message_changed_callback(self, message)
        self._message = message

    message = property(message_get, message_set)


class WaitNotify(Notify):
    terra_type = "Model/WaitNotify"

    def __init__(self, message, timeout, answer_callback=None):
        Notify.__init__(self, "Wait", message, answer_callback)
        self.timeout = float(timeout)
        self.callback_stopped = None
        self._timer = None
        self._informed = False
        self._delete_me = False

    def _inform_stop(self):
        self._informed = True
        self._delete_me = True
        if self.callback_stopped is not None:
            self.callback_stopped(self)

    def _timer_informed(self):
        if self._informed:
            return False

        self._inform_stop()
        return False

    def start(self):
        self._timer = ecore.timer_add(self.timeout, self._timer_informed)

    def stop(self):
        if self._informed:
            return

        if self._timer is not None:
            self._timer.delete()
        self._inform_stop()


class ThrobberDialog(Notify):
    terra_type = "Model/ThrobberDialog"

    def __init__(self, message, answer_callback=None):
        Notify.__init__(self, "ThrobberDialog", message, answer_callback)
        self.callback_hide = None

    def hide(self):
        if self.callback_hide is not None:
            self.callback_hide(self)


class YesNoDialog(Notify):
    terra_type = "Model/YesNoDialog"

    def __init__(self, message, answer_callback=None):
        Notify.__init__(self, "YesNo", message, answer_callback)


class EntryDialog(Notify):
    terra_type = "Model/EntryDialog"

    def __init__(self, title, subtitle, text="", answer_callback=None):
        Notify.__init__(self, "Entry", title, answer_callback)
        self.subtitle = subtitle
        self.text = text
