#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging
import feedparser
import os
from urllib2 import HTTPError

import etk
import evas.decorators

from terra.ui.panel import PanelRichList, PanelContentFrame
from terra.ui.modal import Modal
from terra.core.manager import Manager
from terra.core.threaded_func import ThreadedFunction
from terra.core.plugin_prefs import PluginPrefs
from terra.ui.throbber import EtkThrobber

log = logging.getLogger("plugins.canola-core.settings")
mger = Manager()
network = mger.get_status_notifier("Network")
settings = PluginPrefs("settings")
DownloadManager = mger.get_class("DownloadManager")

PanelController = mger.get_class("Controller/Panel")
ModalController = mger.get_class("Controller/Modal")
FeedItemRenderer = mger.get_class("Renderer/EtkList/FeedItem")

download_mger = DownloadManager()


class SettingsEditFeedView(Modal):
    def __init__(self, parent, title, theme=None):
        Modal.__init__(self, parent.view, "Edit", theme,
                       hborder=16, vborder=100)

        self.callback_ok_clicked = None
        self.callback_cancel_clicked = None
        self.callback_escape = None

        label = etk.Label("    Feed title:")
        label.alignment_set(0.0, 1.0)
        label.show()

        self.entry = etk.Entry(text=title)
        self.entry.on_text_activated(self._on_ok_clicked)
        self.entry.show()

        vbox = etk.VBox()
        vbox.border_width_set(25)
        vbox.append(label, etk.VBox.START, etk.VBox.FILL, 0)
        vbox.append(self.entry, etk.VBox.START, etk.VBox.FILL, 10)
        vbox.show()

        self.modal_contents = PanelContentFrame(self.evas)
        self.modal_contents.frame.add(vbox)
        self.ok_button = self.modal_contents.button_add("OK")
        self.ok_button.on_clicked(self._on_button_clicked)
        self.cancel_button = self.modal_contents.button_add("  Cancel  ")
        self.cancel_button.on_clicked(self._on_button_clicked)

        self.modal_contents.handle_key_down = self.handle_key_down
        self.contents_set(self.modal_contents.object)

    def handle_key_down(self, ev):
        if ev.key == "Escape":
            if self.callback_escape:
                self.callback_escape()
            return False
        return True

    def _on_ok_clicked(self, *ignored):
        if self.callback_ok_clicked:
            self.callback_ok_clicked(self.entry.text)

    def _on_button_clicked(self, bt):
        if bt == self.ok_button:
            self._on_ok_clicked()
        elif bt == self.cancel_button:
            if self.callback_cancel_clicked:
                self.callback_cancel_clicked()

    def do_on_focus(self):
        self.modal_contents.object.focus = True

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self.modal_contents.destroy()


class SettingsEditFeedController(ModalController):
    terra_type = "Controller/Settings/Feed"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)
        self.model = model
        self.view = SettingsEditFeedView(parent, model.title, parent.view.theme)
        self.view.callback_ok_clicked = self._on_ok_clicked
        self.view.callback_cancel_clicked = self.close
        self.view.callback_escape = self.close
        self.view.show()

    def close(self):
        def cb(*ignored):
            self.back()
        self.view.hide(end_callback=cb)

    def _on_ok_clicked(self, new_title):
        if self.model.title != new_title:
            self.model.title = new_title
            self.model.commit()

            # XXX
            self.parent.killall()
        self.close()

    def delete(self):
        self.view.delete()
        self.view = None
        self.model = None


class SettingsFeedPanel(PanelRichList):
    def __init__(self, main_window, title, elements, theme=None):
        PanelRichList.__init__(self, main_window, title, None, theme)

        self.list.animated_changes = True
        self._init_header()

        self._network = True

        self.callback_item_clicked = None
        self.callback_edit_clicked = None
        self.callback_delete_clicked = None
        self.callback_text_activated = None
        self.callback_check_network = None
        self.callback_no_connection = None

        self.list.freeze()
        for e in elements:
            self.list_model.append(e)
        self.list.thaw()

    def _on_entry_key_down(self, o, ev):
        if ev.key == "Escape":
            log.debug("Taking focus out of entry.")
            self.entry.unfocus()
            return False
        else:
            return True

    def _init_header(self):
        self.label = etk.Label("  Add URL:")
        self.label.show()

        self.entry = etk.Entry()
        self.entry.on_text_activated(self._on_text_activated)
        self.entry.on_key_down(self._on_entry_key_down)
        self.entry.show()

        self.throbber = EtkThrobber("  Acquiring name")
        self.throbber.show()
        self.throbber_align = \
            etk.Alignment(0.5, 0.5, 0.0, 0.0, child=self.throbber)

        self.box = etk.VBox(size_request=(100, 66))
        self.box.append(self.label, etk.VBox.START, etk.VBox.FILL, 0)
        self.box.append(self.entry, etk.VBox.START, etk.VBox.FILL, 0)
        self.box.append(self.throbber_align, etk.VBox.START,
                        etk.VBox.EXPAND_FILL, 0)
        self.box.show()

        self.header_area_append(self.box)

    def list_columns_get(self):
        r = FeedItemRenderer(text_func=self._get_row_text,
                             item_click=self._on_item_clicked,
                             edit_click=self._on_edit_clicked,
                             delete_click=self._on_delete_clicked)
        return [(100, r, True)]

    def _get_row_text(self, row):
        return row.title

    def _on_text_activated(self, e, *ignored):
        if self.callback_text_activated:
            self.callback_text_activated(e.text)

    def _on_item_clicked(self, row, list):
        if self.callback_item_clicked:
            self.callback_item_clicked(row)

    def _on_edit_clicked(self, row, list):
        if self.callback_edit_clicked:
            self.callback_edit_clicked(row)

    def _on_delete_clicked(self, row, list):
        if self.callback_delete_clicked:
            self.callback_delete_clicked(row)

    def start_loading(self):
        self.label.hide()
        self.entry.hide()
        self.throbber_align.show()
        self.throbber.start()

    def stop_loading(self, end_callback=None):
        def cb():
            self.label.show()
            self.entry.show()
            self.throbber_align.hide()
            if end_callback:
                end_callback()
            self.throbber.callback_animate_hide_finished = None
        self.throbber.callback_animate_hide_finished = cb
        self.throbber.stop()

    def do_on_focus(self):
        if self.callback_check_network:
            self.network = self.callback_check_network()

    def _network_get(self):
        return self._network

    def _network_set(self, value):
        old = self._network
        self._network = value

        if value == old:
            return
        elif value:
            self.entry.disabled = False
        else:
            self.entry.disabled = True
            self.callback_no_connection()

    network = property(_network_get, _network_set)

    def _add_url_enabled_get(self):
        return not self.entry.disabled

    def _add_url_enabled_set(self, value):
        if value and self._network:
            self.entry.disabled = False
        else:
            self.entry.disabled = True

    add_url_enabled = property(_add_url_enabled_get, _add_url_enabled_set)


class SettingsFeedFolderController(PanelController):
    terra_type = "Controller/Settings/Folder/Feed"

    acceptable_mime_types = None # all are accepted

    def __init__(self, model, canvas, parent, view=None):
        PanelController.__init__(self, model, canvas, parent)

        self.model = model
        self.model.load()

        # Use this to mark if the controller is still valid for asynchronous
        # functions know if their result is still wanted.
        self.still_valid = True
        self.view = view
        if not self.view:
            self.view = SettingsFeedPanel(parent.window, model.title,
                                          self.model.children)

        self.view.callback_item_clicked = self._cb_click
        self.view.callback_edit_clicked = self._cb_edit
        self.view.callback_delete_clicked = self._cb_delete
        self.view.callback_text_activated = self._cb_new
        self.view.callback_escape = self.back
        self.view.callback_check_network = self._cb_check_network
        self.view.callback_no_connection = self._cb_no_connection

    def back(self, end_callback=None):
        self.still_valid = False
        PanelController.back(self, end_callback)

    def _cb_no_connection(self):
        self.parent.message("No connection available.")

    def _cb_check_network(self):
        return bool(network and network.status > 0.0)

    def _cb_click(self, model):
        pass # print "clicked in item => %s" % model

    def _cb_edit(self, model):
        controller = SettingsEditFeedController(model, self.evas, self)
        self.use(controller)

    def _cb_delete(self, model):
        self.killall()
        self.view.list_model.remove(model)
        model.delete()

    def _parse(self, filename, orig_uri, mimetype):
        feed_info = feedparser.parse(filename)
        return feed_info

    def _normalize_uri(self, uri):
        if not "://" in uri:
            uri = "http://" + uri
        return uri

    def _target_file_get(self, uri):
        basename = os.path.basename(uri)
        temp_dir = settings.get("temp_dir")
        return os.path.join(temp_dir, basename)

    def _cb_new(self, uri):
        if uri == "":
            return

        uri = self._normalize_uri(uri)
        log.debug("Trying add feed for '%s'" % uri)
        feed_target = self._target_file_get(uri)

        self.view.add_url_enabled = False

        if self.model.child_model_exists_in_db(uri):
            self.view.entry.text = ""
            self.parent.message("This URL already is in your list.")
            self.view.add_url_enabled = True
            return

        if not self._cb_check_network():
            self.view.entry.text = ""
            self.view.network = False
            return

        self.view.start_loading()
        self.view.entry.text = ""

        def cleanup():
            if os.path.exists(feed_target):
                try:
                    os.unlink(feed_target)
                except:
                    pass

        def check_then_cleanup():
            if not self.still_valid:
                log.info("Panel removed from the stack before loading, cancel add URL then.")
                cleanup()
                return False
            else:
                return True

        def remove_callbacks():
            f.on_finished_remove(fetching_finished)
            f.on_filtered_remove(filtered)

        def parse_feed(mimetype):
            try:
                log.debug("Parsing feed with URI : %s" % uri)
                data = self._parse(feed_target, uri, mimetype)
                cleanup()
            except Exception, e:
                log.error("Exception during feed parsing %s", e)
                cleanup()
                return None

            return data

        def fetching_finished(exception, mimetype):
            remove_callbacks()
            if not check_then_cleanup():
                return

            if exception:
                log.error("Exception during feed fetching %s", exception)
                cleanup()
                if isinstance(exception, HTTPError):
                    if exception.code == 401:
                        show_error("Wrong user or password")
                    else:
                        show_error(exception.msg)
                else:
                    show_error()
            else:
                t = ThreadedFunction(parsing_finished, parse_feed, mimetype)
                t.start()

        def filtered(mimetype):
            remove_callbacks()
            if not check_then_cleanup():
                return

            log.info("Filtered, mimetype: " + str(mimetype))

            alt_uri = self.alternative_uri(uri, mimetype)
            if alt_uri is None:
                # XXX
                feed_contents = self.process_non_feed(uri, mimetype)
                if feed_contents is not None:
                    add_feed(feed_contents)
                else:
                    show_error("URL is no feed: %s" % mimetype)
            else:
                # retry with alternative uri
                self._cb_new(alt_uri)

        def parsing_finished(exception, feed_contents):
            if not check_then_cleanup():
                return
            if exception is not None:
                log.error("Exception during feed parsing %s", exception)
                show_error()
            else:
                add_feed(feed_contents)

        def show_error(message=None):
            self.view.entry.text = uri
            if message:
                self.parent.message(message)
            else:
                self.parent.message("No feed available for this URL.")
            self.view.add_url_enabled = True
            self.view.stop_loading()

        def add_feed(feed_contents):
            if feed_contents and feed_contents.feed.has_key("title"):
                self.killall()
                item = self.model.item_new(uri, feed_contents)
                self.view.list_model.prepend(item)
                self.view.add_url_enabled = True
                self.view.stop_loading()
            else:
                show_error()

        f = download_mger.add(uri, feed_target, ignore_limit=True)
        f.set_accepted_mimetypes(self.acceptable_mime_types)
        f.on_finished_add(fetching_finished)
        f.on_filtered_add(filtered)
        f.start()

    def process_non_feed(self, uri, mimetype):
        return None

    def alternative_uri(self, uri, mimetype):
        return None

    def killall(self):
        # XXX: Ask SettingsController to killall tasks, in case some Screen
        # with download progress bars are loaded.
        # This is a bit drastic, some refinement on what is killed could
        # be a good idea OR podcast screen could "de-register" this callbacks
        # and "re-register" them when suspended and resumed.
        self.parent.killall()

    def delete(self):
        self.view.delete()
        self.view = None
        self.model.unload()
