/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *      Neil Jagdish Patel <neil.patel@canonical.com>
 *      Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
/**
 * SECTION:dee-peer
 * @short_description: Finds other objects with the same swarm-name on the bus.
 * @include: dee.h
 *
 * #DeePeer allows you to build objects that can 'find eachother' on D-Bus
 * without the need for an central registration service. Think of it like
 * peer-to-peer for your application. The DBus session bus will also implicitly
 * elect a swarm leader - namely the one owning the swarm name on the bus, but
 * it's up to the consumer of this API to determine whether swarm leadership has
 * any concrete responsibilities associated.
 *
 * Peers find eachother through a well-known "swarm-name", which loosely
 * translates to a DBus name, such as: org.myapp.MyPeers. Choose a namespaced
 * name that would not normally be used outside of your program.
 *
 * For example:
 * <informalexample><programlisting>
 * {
 *   DeePeer *peer;
 *
 *   peer = g_object_new (DBUS_TYPE_PEER,
 *                        "swarm-name", "org.myapp.MyPeers",
 *                        NULL);
 *
 *   g_signal_connect (peer, "peer-found",
 *                     G_CALLBACK (on_peer_found), NULL);
 *   g_signal_connect (peer, "peer-lost",
 *                     G_CALLBACK (on_peer_lost), NULL);
 *
 *   /<!-- -->* Publish this peer and start monitoring for other peers *<!-- -->/
 *   dee_peer_connect (peer);
 * }
 * </programlisting></informalexample>
 *
 * Commonly the callbacks for these signals will create (or tear down) a proxy
 * for the relevant peer. With DBus-GLib it might look like:
 *
 * <informalexample><programlisting>
 * void
 * on_peer_found (DeePeer *peer, const gchar *name)
 * {
 *   DBusGProxy *proxy;
 *
 *   /<!-- -->* We have found a peer, so let's connect to it with a well-known
 *    <!-- -->* interface.
 *    *<!-- -->/
 *   proxy = dbus_g_proxy_new_for_name_owner (connection,
 *                                            name,
 *                                            "/org/myapp/MyPeerIface",
 *                                            "org.myapp.MyPeerIface",
 *                                            NULL);
 * }
 *
 * </programlisting></informalexample>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus.h>
#include "dee-peer.h"
#include "dee-marshal.h"
#include "trace-log.h"

G_DEFINE_TYPE (DeePeer, dee_peer, G_TYPE_OBJECT)

#define DEE_PEER_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, DEE_TYPE_PEER, DeePeerPrivate))

#define _DeePeerIter GSequenceIter

/**
 * DeePeerPrivate:
 *
 * Ignore this structure.
 **/
struct _DeePeerPrivate
{
  DBusGConnection *connection;
  DBusGProxy      *dbus_proxy;
  DBusGProxy      *leader_proxy;

  GHashTable *peers;
  
  gchar  *swarm_name;
  gchar  *swarm_path;
  gchar  *leader_address;

  GSList *match_rules;

  gboolean connected;
  gboolean is_swarm_leader;
};

/* Globals */
enum
{
  PROP_0,
  PROP_SWARM_NAME,
  PROP_SWARM_LEADER
};

enum
{
  CONNECTED,
  PEER_FOUND,
  PEER_LOST,

  /* Private, leaked because of dbus-glib */
  PING,
  PONG,
  BYE,

  LAST_SIGNAL
};

static guint32 _peer_signals[LAST_SIGNAL] = { 0 };

/* Forwards */
static void                 emit_peer_found          (DeePeer    *self,
                                                      const gchar *name);

static void                 on_leadership_lost       (DeePeer    *self);

static void                 on_leadership_acquired   (DeePeer    *self);

static void                 on_join_received         (DeePeer    *self,
                                                      const gchar *peer_address);

static void                 on_bye_received          (DeePeer    *self,
                                                      const gchar *peer_address);

static void                 on_ping_received         (DeePeer    *self,
                                                      const gchar *leader_address);

static void                 on_pong_received         (DeePeer    *self,
                                                      const gchar *peer_address);

static void                 on_list_received         (DBusGProxy  *proxy,
                                                      char       **names,
                                                      GError      *error,
                                                      void        *user_data);

static void                 on_leader_found          (DBusGProxy  *proxy,
                                                      char        *leader_address,
                                                      GError      *error,
                                                      gpointer     userdata);

static void                 set_swarm_name           (DeePeer    *self,
                                                      const gchar *swarm_name);

static void                 emit_ping                (DeePeer    *self);

static void                 emit_pong                (DeePeer    *self);

DBusHandlerResult          _dbus_message_filter      (DBusConnection *dconn,
                                                      DBusMessage    *msg,
                                                      void           *user_data);

/* Globals */
static gboolean dee_peer_types_initialized = FALSE;

/* Stuff to make dbus-glib happy */
static gboolean             _dee_peer_server_list    (DeePeer     *self,
                                                      gchar      ***peers,
                                                      GError       **error);
#include "dee-peer-client.h"
#include "dee-peer-server.h"

/* GObject Init */
static void
dee_peer_finalize (GObject *object)
{
  DeePeerPrivate *priv;
  DBusConnection  *dconn;
  GSList *match_iter;

  priv = DEE_PEER (object)->priv;

  /* Remove match rules from the bus, and free the string repr. of the rule  */
  if (priv->connection)
    {
      dconn = dbus_g_connection_get_connection (priv->connection);

      /* Uninstall filter */
      dbus_connection_remove_filter (dconn, _dbus_message_filter, object);
      
      for (match_iter = priv->match_rules;
           match_iter != NULL;
           match_iter = match_iter->next)
        {
          dbus_bus_remove_match (dconn, (const char*)match_iter->data, NULL);
          g_free (match_iter->data);
        }
    }

  /* Free resources */
  if (priv->swarm_name)
    {
      g_free (priv->swarm_name);
      priv->swarm_name = NULL;
    }
  if (priv->swarm_path)
    {
      g_free (priv->swarm_path);
      priv->swarm_path = NULL;
    }
  if (priv->leader_address)
    {
      g_free (priv->leader_address);
      priv->leader_address = NULL;
    }
  if (priv->peers)
    {
      g_hash_table_destroy (priv->peers);
      priv->peers = NULL;
    }      
  if (priv->dbus_proxy)
    {
      g_object_unref (priv->dbus_proxy);
      priv->dbus_proxy = NULL;
    }
  if (priv->leader_proxy)
    {
      g_object_unref (priv->leader_proxy);
      priv->leader_proxy = NULL;
    }
  if (priv->connection)
    {
      dbus_g_connection_unref (priv->connection);
      priv->connection = NULL;
    }
  if (priv->match_rules)
    g_slist_free (priv->match_rules);
  
  
  G_OBJECT_CLASS (dee_peer_parent_class)->finalize (object);
}

static void
dee_peer_set_property (GObject       *object,
                       guint          id,
                       const GValue  *value,
                       GParamSpec    *pspec)
{
  DeePeerPrivate *priv = DEE_PEER (object)->priv;
  
  
  switch (id)
    {
    case PROP_SWARM_NAME:
      set_swarm_name (DEE_PEER (object), g_value_get_string (value));
      break;
    case PROP_SWARM_LEADER:
      if (priv->leader_address) g_free (priv->leader_address);
      priv->leader_address = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_peer_get_property (GObject     *object,
                        guint        id,
                        GValue      *value,
                        GParamSpec  *pspec)
{
  switch (id)
    {
    case PROP_SWARM_NAME:
      g_value_set_string (value, DEE_PEER (object)->priv->swarm_name);
      break;
    case PROP_SWARM_LEADER:
      g_value_set_string (value, DEE_PEER (object)->priv->leader_address);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_peer_class_init (DeePeerClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = dee_peer_finalize;
  obj_class->set_property = dee_peer_set_property;
  obj_class->get_property = dee_peer_get_property;

  /* Add Signals */
  /**
   * DeePeer::connected
   * @self: the #DeePeer on which the signal is emitted
   * @unique_name: the unique name of the connection
   *
   * Connect to this signal to be notified when the peer connects to DBus.
   *   Sub-classes should connect to this signal so they register themselves
   *   on the bus at the same time.
   **/
  _peer_signals[CONNECTED] =
    g_signal_new ("connected",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DeePeerClass, connected),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

  /**
   * DeePeer::peer-found:
   * @self: the #DeePeer on which the signal is emitted
   * @name: the DBus name of the object found
   *
   * Connect to this signal to be notified of existing and new peers that are
   *   in your swarm.
   **/
  _peer_signals[PEER_FOUND] =
    g_signal_new ("peer-found",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DeePeerClass, peer_found),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

  /**
   * DeePeer::peer-lost:
   * @self: the #DeePeer on which the signal is emitted
   * @name: the DBus name of the object that disconnected
   *
   * Connect to this signal to be notified when peers disconnect from the swarm
   **/
  _peer_signals[PEER_LOST] =
    g_signal_new ("peer-lost",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DeePeerClass, peer_lost),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

  /**
   * DeePeer::ping: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  /*< private >*/
  _peer_signals[PING] =
    g_signal_new ("ping",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  0,
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

  /**
   * DeePeer::pong: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  /*< private >*/
  _peer_signals[PONG] =
    g_signal_new ("pong",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  0,
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

 /**
   * DeePeer::pong: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  /*< private >*/
  _peer_signals[PONG] =
    g_signal_new ("bye",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  0,
                  NULL, NULL,
                  _dee_marshal_VOID__STRING_STRING,
                  G_TYPE_NONE, 2,
                  G_TYPE_STRING,
                  G_TYPE_STRING);
  
  /* Add properties */
  /**
   * DeePeer::swarm-name:
   *
   * The name of the swarm that this peer is connected to. All swarm members
   * will try and own this name on the session bus. The one owning the name
   * is the swarm leader.
   **/
  pspec = g_param_spec_string ("swarm-name", "Swarm Name",
                               "Well-known name to find other peers with",
                               NULL,
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT
                               | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (obj_class, PROP_SWARM_NAME, pspec);

  /**
   * DeePeer::swarm-leader:
   *
   * The name of the swarm that this peer is connected to. All swarm members
   * will try and own this name on the session bus. The one owning the name
   * is the swarm leader.
   **/
  pspec = g_param_spec_string ("swarm-leader", "Swarm Leader",
                               "Unique DBus address of the swarm leader",
                               NULL,
                               G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (obj_class, PROP_SWARM_LEADER, pspec);

  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DeePeerPrivate));

  /* Register as a DBus object */
  dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
                                   &dbus_glib__dee_peer_server_object_info);
}

static void
dee_peer_init (DeePeer *peer)
{
  DeePeerPrivate *priv;

  priv = peer->priv = DEE_PEER_GET_PRIVATE (peer);

  priv->swarm_name = NULL;
  priv->leader_address = NULL;
  priv->match_rules = NULL;
  priv->peers = g_hash_table_new_full (g_str_hash,
                                       g_str_equal,
                                       (GDestroyNotify)g_free,
                                       NULL);
  
  priv->connected = FALSE;
  priv->is_swarm_leader = FALSE;
}

/* Private Methods */

static void
signal_peer_lost_hfunc (gchar *peer, gchar *ignored, DeePeer *self)
{
  g_signal_emit (self, _peer_signals[PEER_LOST], 0, peer);
}

/* Async callback for com.canonical.Dee.Peer.List */
static void
on_list_received (DBusGProxy  *proxy,
                  char       **names,
                  GError      *error,
                  void        *user_data)
{
  DeePeer        *self = DEE_PEER (user_data);
  DeePeerPrivate *priv;
  GHashTable      *peers;
  GSList          *new_peers, *iter;
  guint            i;

  g_return_if_fail (DEE_IS_PEER (self));
  priv = self->priv;

  if (error)
    {
      g_warning ("%s: Unable to list peers: %s",
                 G_STRLOC,
                 error->message);
      return;
    }
  else
    {
      trace_object (self, "Got list of %d peers", g_strv_length (names));
    }
  
  /* We diff the current list of peers against the new list
   * and emit signals accordingly. New peers are added to new_peers,
   * and lost peers will remain in priv->peers: */
  new_peers = NULL;
  peers = g_hash_table_new_full (g_str_hash,
                                 g_str_equal,
                                 (GDestroyNotify)g_free,
                                 NULL);                                     
  for (i = 0; names[i]; i++)
    {
      g_hash_table_insert (peers, g_strdup (names[i]), NULL);
      if (!g_hash_table_remove (priv->peers, names[i]))
        {
          /* The peer was not previously known */
          new_peers = g_slist_prepend (new_peers, names[i]);
        }      
    }

  /* Signal about lost peers */
  g_hash_table_foreach (priv->peers, (GHFunc)signal_peer_lost_hfunc, self);

  /* Signal about new peers */
  for (iter = new_peers; iter; iter = iter->next)
    {
      emit_peer_found (self, (const gchar*)iter->data);
    }

  /* Free just the array, not the strings - they are owned by 'peers' now */
  g_slist_free (new_peers);
  g_hash_table_destroy (priv->peers);
  priv->peers = peers;
}

/* Async callback for DBus.GetNameOwner(swarm_name) */
static void
on_leader_found (DBusGProxy  *proxy,
                 char        *leader_address,
                 GError      *error,
                 gpointer     userdata)
{
  DeePeer        *self;
  DeePeerPrivate *priv;
  
  g_return_if_fail (DEE_IS_PEER (userdata));

  if (error != NULL)
    {
      g_warning ("Error detecting leader: %s", error->message);
      return;
    }
  
  self = DEE_PEER (userdata);
  priv = self->priv;

  trace_object (self, "Found leader (%s): %s",
                priv->swarm_name, leader_address);
  
  if (priv->leader_address != NULL)
    {
      if (!g_str_equal (priv->leader_address, leader_address))
        {
          g_free (priv->leader_address);
          priv->leader_address = leader_address;
          g_object_notify (G_OBJECT (self), "swarm-leader");
        }
    }
  else
    {
      priv->leader_address = g_strdup (leader_address);
      g_object_notify (G_OBJECT (self), "swarm-leader");
    }
}

static void
install_match_rule (DeePeer *self, const char *rule, ...)
{
  DeePeerPrivate *priv;
  DBusConnection  *dconn;
  char            *f_rule;
  va_list args;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (rule != NULL);

  priv = self->priv;
  dconn = dbus_g_connection_get_connection (priv->connection);
  
	va_start (args, rule);

  f_rule = g_strdup_vprintf (rule, args);

  /* By setting the error argument to NULL libdbus will use async mode
   * for adding the match rule. We want that. */
  dbus_bus_add_match (dconn, f_rule, NULL);
  priv->match_rules = g_slist_prepend (priv->match_rules, f_rule);
  
  va_end (args);
}

static gboolean
dee_peer_connect_real (DeePeer *self)
{
  DeePeerPrivate *priv;
  GError          *error = NULL;  
  gboolean         is_leader = FALSE;
  guint32          ret = 0;

  g_return_val_if_fail (DEE_IS_PEER (self), FALSE);
  priv = self->priv;
  g_return_val_if_fail (priv->swarm_name != NULL, FALSE);
  g_return_val_if_fail (priv->connected == FALSE, FALSE);

  if (priv->connection)
    {
      g_warning ("Peer already connected to session bus");
      return FALSE;
    }
  
  if (!dee_peer_types_initialized)
    {
      dee_peer_types_initialized = TRUE;

      /* Marshaller for Ping and Pong */
      dbus_g_object_register_marshaller (g_cclosure_marshal_VOID__STRING,
                                         G_TYPE_NONE,
                                         G_TYPE_STRING,
                                         G_TYPE_INVALID);
    }

  priv->connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
  
  /* Create a proxy for the DBus API unless we already tried */
  if (priv->dbus_proxy == NULL)
    {
      // FIXME: Sync DBus call?
      priv->dbus_proxy = dbus_g_proxy_new_for_name_owner (priv->connection,
                                                          DBUS_SERVICE_DBUS,
                                                          DBUS_PATH_DBUS,
                                                          DBUS_INTERFACE_DBUS,
                                                          &error);
      if (error)
        {
          g_error ("%s: Unable to get dbus proxy from session bus: %s",
                   G_STRLOC,
                   error->message);
          g_error_free (error);
          return FALSE;
        }
    }
  
  /* Try and own the swarm leader name. Allow other peers to forcefully
   * own the name if they really really want it */
  // FIXME: Sync DBus call
  if (!org_freedesktop_DBus_request_name (priv->dbus_proxy,
                                          priv->swarm_name,
                                          DBUS_NAME_FLAG_ALLOW_REPLACEMENT,
                                          &ret,
                                          &error))
    {
      g_warning ("%s: Unable to request name (%s) from DBus: %s",
                 G_STRLOC,
                 priv->swarm_name,
                 error->message);
      g_error_free (error);

      return FALSE;
    }

  /* Check if we became swarm leaders */
  switch (ret)
    { 
      case DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER:
      case DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER:
        is_leader = TRUE; /* Fall through */
      case DBUS_REQUEST_NAME_REPLY_IN_QUEUE:
        break;
      default:
        g_critical ("Swarm protocol error. Failed to apply for leadership"); 
    }

  /* Filter messages matching out match rules  */
  dbus_connection_add_filter (dbus_g_connection_get_connection (priv->connection),
                              _dbus_message_filter,
                              self,
                              NULL);
  
  /* Detect if we lose the swarm name, hence the leadership */
  install_match_rule (self,
                      "interface='"DBUS_INTERFACE_DBUS"',"
                      "member='NameLost',"
                      "arg0='%s'",
                      priv->swarm_name);

  /* Detect if we win the swarm name, hence the leadership */
  install_match_rule (self,
                      "interface='"DBUS_INTERFACE_DBUS"',"
                      "member='NameAcquired',"
                      "arg0='%s'",
                      priv->swarm_name);

  /* Detect when someone joins the swarm */
  install_match_rule (self,
                      "interface='"DBUS_INTERFACE_DBUS"',"
                      "member='RequestName',"
                      "arg0='%s'",
                      priv->swarm_name);

  /* Detect when swarm leader notifies us that someone left */
  install_match_rule (self,
                      "interface='"DEE_PEER_DBUS_IFACE"'"
                      ",member='Bye',"
                      "arg0='%s'",
                      priv->swarm_name);

  /* Detect when swarm leader performs a head count */
  install_match_rule (self,
                      "interface='"DEE_PEER_DBUS_IFACE"',"
                      "member='Ping',"
                      "arg0='%s'",
                      priv->swarm_name);    

  /* Export Swarm methods */
  dbus_g_connection_register_g_object (priv->connection,
                                       priv->swarm_path,
                                       G_OBJECT (self));

  /* If we are not leaders. Request a swarm roster from the leader -
   * that way we also learn her unique name */
  if (!is_leader)
    {
      org_freedesktop_DBus_get_name_owner_async (priv->dbus_proxy,
                                                 priv->swarm_name,
                                                 on_leader_found,
                                                 self);
      
      priv->leader_proxy =
               dbus_g_proxy_new_for_name_owner (priv->connection,
                                                priv->swarm_name,
                                                priv->swarm_path,
                                                DEE_PEER_DBUS_IFACE,
                                                &error);
      if (error != NULL)
        {
          g_warning ("%s: Unable to connect to swarm leader '%s': %s",
                     G_STRLOC,
                     priv->swarm_name,
                     error->message);
          g_error_free (error);
          return FALSE;
        }
      
      com_canonical_Dee_Peer_list_async (priv->leader_proxy,
                                        on_list_received,
                                        self);
    } 
  else
    {
      on_leadership_acquired (self);
    }   
  
  priv->connected = TRUE;
  g_signal_emit (self, _peer_signals[CONNECTED], 0,
                 dbus_bus_get_unique_name (
                       dbus_g_connection_get_connection (priv->connection)));
  
  return FALSE;
}


/* Public Methods */
/**
 * dee_peer_connect:
 * @self: a #DeePeer
 *
 * Will cause @self to connect to the swarm and begin monitoring peers.
 **/
void
dee_peer_connect (DeePeer *self)
{
  g_return_if_fail (DEE_IS_PEER (self));

  dee_peer_connect_real (self);
}

/**
 * dee_peer_is_swarm_leader
 * @self: a #DeePeer
 *
 * Return value: %TRUE if and only if this peer owns the swarm name on the session bus
 **/ 
const gboolean
dee_peer_is_swarm_leader  (DeePeer *self)
{
  g_return_val_if_fail (DEE_PEER (self), FALSE);

  return self->priv->is_swarm_leader;
}

/**
 * dee_peer_get_swarm_leader
 * @self: a #DeePeer
 *
 * Gets the unique DBus address of the current swarm leader.
 *
 * This function can only be used after dee_peer_connect() has been called.
 *
 * Return value: Unique DBus address of the current swarm leader, possibly %NULL
 *    if the leader has not been detected yet
 **/
const gchar *
dee_peer_get_swarm_leader (DeePeer *self)
{
  g_return_val_if_fail (DEE_PEER (self), NULL);

  return self->priv->leader_address;
}

/**
 * dee_peer_get_swarm_name:
 * @self: a #DeePeer
 *
 * Gets the unique name for this swarm. The swarm leader is the Peer owning
 * this name on the session bus.
 *
 * Return value: The swarm name
 **/
const gchar *
dee_peer_get_swarm_name (DeePeer *self)
{
  g_return_val_if_fail (DEE_PEER (self), NULL);

  return self->priv->swarm_name;
}

static void
emit_peer_found (DeePeer    *self,
                 const gchar *name)
{
  DeePeerPrivate *priv;
  const gchar     *own_name;

  g_return_if_fail (DEE_IS_PEER(self));
  g_return_if_fail (name != NULL);

  priv = self->priv;
  own_name = dbus_bus_get_unique_name (
                            dbus_g_connection_get_connection(priv->connection));
  
  if (!g_str_equal (name, own_name))
    {
      g_signal_emit (self, _peer_signals[PEER_FOUND], 0, name);
    }
}

static void
set_swarm_name (DeePeer    *self,
                const gchar *swarm_name)
{
  DeePeerPrivate *priv;
  gchar           *dummy;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (swarm_name != NULL);
  priv = self->priv;

  if (priv->swarm_name)
    {
      g_warning ("%s: Unable to set previously set swarm_name (%s) to (%s)",
                 G_STRLOC,
                 priv->swarm_name,
                 swarm_name);
      return;
    }

  /* If swarm_name is org.example.MyService then the swarm_path will
   * become /com/canonical/dee/org/example/MyService. Note that
   * the actual object path of the peer is not used in the Swarm spec */
  
  priv->swarm_name = g_strdup (swarm_name);
  dummy = g_strdelimit (g_strdup(swarm_name), ".", '/');
  priv->swarm_path = g_strdup_printf ("/com/canonical/dee/peer/%s", dummy);

  g_free (dummy);
}

/* DBus callback: When loosing leadership */
static void
on_leadership_lost (DeePeer *self)
{
  DeePeerPrivate *priv;
  GError          *error;
  
  g_return_if_fail (DEE_IS_PEER (self));

  priv = self->priv;
  
  if (priv->is_swarm_leader)
    {      
      /* We signal the change of swarm leadership in in_ping_received(),
       * only at that point do we know the unique name of the new leader */
      trace_object (self, "Lost swarm leadership");      
      // FIXME. We ought to remove the Pong match rule, but it's not paramount
      priv->is_swarm_leader = FALSE;      
    }
  else
    {
      g_warning ("Internal error: Leadership lost, but we are not leaders");
    }

  if (priv->leader_proxy == NULL)
    {
      error = NULL;
      priv->leader_proxy = dbus_g_proxy_new_for_name_owner (priv->connection,
                                                            priv->swarm_name,
                                                            priv->swarm_path,
                                                            DEE_PEER_DBUS_IFACE,
                                                            &error);
      if (error != NULL)
        {
          g_warning ("Failed to connect to leader: %s", error->message);
          g_error_free (error);
        }
    }
}

/* DBus callback: When we become leaders */
static void
on_leadership_acquired (DeePeer *self)
{
  DeePeerPrivate *priv;
  
  g_return_if_fail (DEE_IS_PEER (self));

  priv = self->priv;
  
  if (priv->is_swarm_leader)
    {
      trace_object (self, "Leadership acquired, but we are already leaders");
    }
  else
    {
      trace_object (self, "Got swarm leadership");
      priv->is_swarm_leader = TRUE;
      g_object_notify (G_OBJECT (self), "swarm-leader");

      /* Start listening for Pongs, so we can do head-counts */
      install_match_rule (self,
                          "interface='"DEE_PEER_DBUS_IFACE"',"
                          "member='Pong',"
                          "arg0='%s'",
                          priv->swarm_name);

      /* Listen for disconnects from the bus,
       * so we can emit Bye when appropriate */
      install_match_rule (self,
                          "interface='"DBUS_INTERFACE_DBUS"',"
                          "member='NameOwnerChanged',"
                          "arg2=''");      
      
      emit_ping (self);                                                      
    }
}

/* Callback from _dbus_message_filter() for custom match rules
 * Indicates that @peer_address joined the warm.
 * Caller owns @peer_address */
static void
on_join_received (DeePeer    *self,
                  const gchar *peer_address)
{
  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (peer_address != NULL);

  trace_object (self, "Found peer %s", peer_address);

  /* If the peer is already known it must have tried to acquire the swarm name
   * twice...  Just ignore it */
  if (g_hash_table_lookup_extended (self->priv->peers, peer_address, NULL, NULL))
    return;

  g_hash_table_insert (self->priv->peers, g_strdup (peer_address), NULL);
  emit_peer_found (self, peer_address);
}

/* Callback from _dbus_message_filter() for custom match rules
 * Indicates that @peer_address left the swarm.
 * Caller owns @peer_address */
static void
on_bye_received (DeePeer    *self,
                 const gchar *peer_address)
{
  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (peer_address != NULL);

  trace_object (self, "Bye %s", peer_address);
  
  if (g_hash_table_remove (self->priv->peers, peer_address))
    {
      g_signal_emit (self, _peer_signals[PEER_LOST], 0, peer_address);
    }
  else
    {
      trace_object (self, "Unknown peer '%s' dropped out of the swarm",
                    peer_address);
    }
}

/* Callback from _dbus_message_filter() for custom match rules
 * Indicates that @leader_address send a Ping to the swarm to
 * initiate a head count.
 * Caller owns @leader_address */
static void
on_ping_received (DeePeer    *self,
                  const gchar *leader_address)
{
  DeePeerPrivate *priv;
  
  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (leader_address != NULL);
  
  priv = self->priv;

  trace_object (self, "Got Ping from: %s. Old leader was: %s",
                leader_address, priv->leader_address);
  
  /* Check if we have a new leader */
  if (priv->leader_address != NULL &&
      !g_str_equal (leader_address, priv->leader_address)) 
    {
      if (priv->leader_address) g_free (priv->leader_address);

      priv->leader_address = g_strdup (leader_address);
      g_object_notify (G_OBJECT (self), "swarm-leader");
    }

  emit_pong(self);
}

/* Callback from _dbus_message_filter() for custom match rules
 * Indicates that @peer_address has emitted a Pong.
 * Caller owns @peer_address */
static void
on_pong_received (DeePeer    *self,
                  const gchar *peer_address)
{
  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (peer_address != NULL);

  trace_object (self, "Got pong %s", peer_address);
  
  if (!g_hash_table_lookup_extended (self->priv->peers, peer_address, NULL, NULL))
    {
      g_hash_table_insert (self->priv->peers, g_strdup (peer_address), NULL);
      emit_peer_found (self, peer_address);
    }
}

/* Broadcast a Ping signal to do a head-count on the swarm.
 * Only call this method as swarm leader - that's the contract
 * of the Swarm spec */ 
static void
emit_ping (DeePeer    *self)
{
  DeePeerPrivate *priv;
  DBusConnection  *dconn;
  DBusMessage     *ping;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (self->priv->is_swarm_leader);
  g_return_if_fail (self->priv->connection != NULL);

  trace_object (self, "Emit ping");
  
  priv = self->priv;
  dconn = dbus_g_connection_get_connection (priv->connection);
  ping = dbus_message_new_signal (priv->swarm_path,
                                  DEE_PEER_DBUS_IFACE,
                                  "Ping");
  dbus_message_append_args (ping,
                            DBUS_TYPE_STRING, &priv->swarm_name, 
                            DBUS_TYPE_INVALID);
  dbus_connection_send (dconn, ping, NULL);
  dbus_message_unref (ping);
}

/* Broadcast a Pong signal as a response to a Ping */ 
static void
emit_pong (DeePeer    *self)
{
  DeePeerPrivate *priv;
  DBusConnection  *dconn;
  DBusMessage     *ping;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (self->priv->connection != NULL);

  trace_object (self, "Emit pong");
  
  priv = self->priv;
  dconn = dbus_g_connection_get_connection (priv->connection);
  ping = dbus_message_new_signal (priv->swarm_path,
                                  DEE_PEER_DBUS_IFACE,
                                  "Pong");
  dbus_message_append_args (ping,
                            DBUS_TYPE_STRING, &priv->swarm_name, 
                            DBUS_TYPE_INVALID);
  dbus_connection_send (dconn, ping, NULL);
  dbus_message_unref (ping);
}

/* Broadcast a Bye signal to to notify the swarm that someone left.
 * Only call this method as swarm leader - that's the contract
 * of the Swarm spec */ 
static void
emit_bye (DeePeer    *self,
          const gchar *peer_address)
{
  DeePeerPrivate *priv;
  DBusConnection  *dconn;
  DBusMessage     *bye;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (self->priv->is_swarm_leader);
  g_return_if_fail (self->priv->connection != NULL);
  g_return_if_fail (peer_address != NULL);

  trace_object (self, "Emit Bye(%s)", peer_address);
  
  priv = self->priv;
  dconn = dbus_g_connection_get_connection (priv->connection);
  bye = dbus_message_new_signal (priv->swarm_path,
                                 DEE_PEER_DBUS_IFACE,
                                 "Bye");
  dbus_message_append_args (bye,
                            DBUS_TYPE_STRING, &priv->swarm_name,
                            DBUS_TYPE_STRING, &peer_address,
                            DBUS_TYPE_INVALID);
  dbus_connection_send (dconn, bye, NULL);
  dbus_message_unref (bye);
} 

static gboolean
_dee_peer_server_list (DeePeer    *self,
                       gchar     ***peers,
                       GError     **error)
{
  DeePeerPrivate  *priv;
  GHashTableIter    iter;
  gchar           **names;
  gpointer          dummy1;
  gpointer          dummy2;
  gint              i;

  g_return_val_if_fail (DEE_IS_PEER (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  /* Copy all keys from the 'peers' hash table */
  priv = self->priv;
  names = g_new0 (gchar*, g_hash_table_size (priv->peers) + 1);
  i = 0;
  g_hash_table_iter_init (&iter, priv->peers);
  while (g_hash_table_iter_next (&iter, &dummy1, &dummy2)) 
  {
    names[i] = g_strdup((gchar*)dummy1);
    i++;    
  }

  *peers = names;
  
  return TRUE;
}

/* Callback applied to all incoming dbus messages. We use this to grab
 * messages for our match rules and dispatch to the right on_*_received function
 * - dbus-glib can not work with match rules - sigh */
DBusHandlerResult
_dbus_message_filter (DBusConnection *dconn,
                      DBusMessage    *msg,
                      void           *user_data)
{
  DeePeer        *self;
  DeePeerPrivate *priv;
  DBusError       derror;
  const char      *swarm_name = NULL;
  const char      *peer_address = NULL;

  /*trace ("FILTER: %p", user_data);
  trace ("Msg filter: From: %s, Iface: %s, Member: %s",
         dbus_message_get_sender (msg),
         dbus_message_get_interface (msg),
         dbus_message_get_member (msg));*/
  
  g_return_val_if_fail (DEE_IS_PEER (user_data),
                        DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

  self = DEE_PEER (user_data);
  priv = self->priv;

  /* Important note: Apps consuming this lib will likely install custom match
   *                 rules which will trigger this filter. Hence we must do very
   *                 strict matching before we dispatch our methods */
  
  dbus_error_init (&derror);
  if (dbus_message_is_method_call (msg, DBUS_INTERFACE_DBUS, "RequestName"))
    {
      peer_address = dbus_message_get_sender(msg);
      dbus_message_get_args (msg, &derror,
                             DBUS_TYPE_STRING, &swarm_name,
                             DBUS_TYPE_INVALID);
      if (dbus_error_is_set (&derror))
        {
          g_critical ("Bad RequestName call: %s", derror.message);
          dbus_error_free (&derror);
        }
      else if (g_str_equal (swarm_name, priv->swarm_name))
        {
          on_join_received (self, peer_address);
          return DBUS_HANDLER_RESULT_HANDLED;
        }
    }
  else if (dbus_message_is_signal (msg, DEE_PEER_DBUS_IFACE, "Bye"))
    {
      dbus_message_get_args (msg, &derror,
                             DBUS_TYPE_STRING, &swarm_name,
                             DBUS_TYPE_STRING, &peer_address,
                             DBUS_TYPE_INVALID);
      if (dbus_error_is_set (&derror))
        {
          g_critical ("Bad Bye signal: %s", derror.message);
          dbus_error_free (&derror);
        }
      else if (g_str_equal (swarm_name, priv->swarm_name))
        {
          on_bye_received (self, peer_address);
          return DBUS_HANDLER_RESULT_HANDLED;
        }      
    }
  else if (dbus_message_is_signal (msg, DEE_PEER_DBUS_IFACE, "Ping"))
    {
      peer_address = dbus_message_get_sender(msg);
      dbus_message_get_args (msg, &derror,
                             DBUS_TYPE_STRING, &swarm_name,
                             DBUS_TYPE_INVALID);
      if (dbus_error_is_set (&derror))
        {
          g_critical ("Bad Ping signal: %s", derror.message);
          dbus_error_free (&derror);
        }
      else if (g_str_equal (swarm_name, priv->swarm_name))
        {
          on_ping_received (self, peer_address);
          return DBUS_HANDLER_RESULT_HANDLED;
        }
    }
  else if (dbus_message_is_signal (msg, DEE_PEER_DBUS_IFACE, "Pong"))
    {
      peer_address = dbus_message_get_sender(msg);
      dbus_message_get_args (msg, &derror,
                             DBUS_TYPE_STRING, &swarm_name,
                             DBUS_TYPE_INVALID);
      if (dbus_error_is_set (&derror))
        {
          g_critical ("Bad Pong signal: %s", derror.message);
          dbus_error_free (&derror);
        }
      else if (g_str_equal (swarm_name, priv->swarm_name))
        {
          on_pong_received (self, peer_address);
          return DBUS_HANDLER_RESULT_HANDLED;
        }
    }
  else if (dbus_message_is_signal (msg, DBUS_INTERFACE_DBUS, "NameAcquired"))
    {
      peer_address = dbus_message_get_sender(msg);
      dbus_message_get_args (msg, &derror,
                             DBUS_TYPE_STRING, &swarm_name,
                             DBUS_TYPE_INVALID);
      if (dbus_error_is_set (&derror))
        {
          g_critical ("Bad NameAcquired signal: %s", derror.message);
          dbus_error_free (&derror);
        }
      else if (g_str_equal (swarm_name, priv->swarm_name))
        {
          on_leadership_acquired (self);
          return DBUS_HANDLER_RESULT_HANDLED;
        }
    }
  else if (dbus_message_is_signal (msg, DBUS_INTERFACE_DBUS, "NameLost"))
    {
      peer_address = dbus_message_get_sender(msg);
      dbus_message_get_args (msg, &derror,
                             DBUS_TYPE_STRING, &swarm_name,
                             DBUS_TYPE_INVALID);
      if (dbus_error_is_set (&derror))
        {
          g_critical ("Bad NameLost signal: %s", derror.message);
          dbus_error_free (&derror);
        }
      else if (g_str_equal (swarm_name, priv->swarm_name))
        {
          on_leadership_lost (self);
          return DBUS_HANDLER_RESULT_HANDLED;
        }
    }
  else if (dbus_message_is_signal (msg, DBUS_INTERFACE_DBUS, "NameOwnerChanged"))
    {      
      const char *old_address, *new_address;
      dbus_message_get_args (msg, &derror,
                             DBUS_TYPE_STRING, &peer_address,
                             DBUS_TYPE_STRING, &old_address,
                             DBUS_TYPE_STRING, &new_address,
                             DBUS_TYPE_INVALID);
      if (dbus_error_is_set (&derror))
        {
          g_critical ("Bad NameOwnerChanged signal: %s", derror.message);
          dbus_error_free (&derror);
        }
      else if (g_str_equal (peer_address, old_address) &&
               g_str_equal (new_address, "") &&
               g_hash_table_lookup_extended (priv->peers,
                                             peer_address,
                                             NULL,
                                             NULL))
        {          
          if (priv->is_swarm_leader)
            {
              emit_bye (self, peer_address);                      
              return DBUS_HANDLER_RESULT_HANDLED;
            }
        }
    }
  
  /* The message is not one matched by our match rules, pass it on */
  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}
