/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus.h>
#include <dee.h>
#include <gtx.h>

#define TIMEOUT 100
#define MODEL_NAME "com.canonical.DeeModel.Tests.Interactions"

/* A command line that launches the appropriaye model-helper-* executable,
 * giving $name as first argument */
#define MODEL_HELPER(helper,name) \
  (gchar *[]) { "./model-helper-"#helper, name, NULL }


typedef struct
{
  DeeModel *model;

} Fixture;

/* GTest normally fails if any warnings are logged, but DBus-Glib likes
 * to log warnings because we use it a bit unconventioanlly. We this this
 * special log handler to ignore warnings from dbus-glib */
static gboolean fatal_log_handler        (const gchar *log_domain,
                                          GLogLevelFlags log_level,
                                          const gchar *message,
                                          gpointer user_data);

static void model_setup         (Fixture *fix, gconstpointer data);
static void model_teardown      (Fixture *fix, gconstpointer data);
static void model_setup_null    (Fixture *fix, gconstpointer data);
static void model_teardown_null (Fixture *fix, gconstpointer data);

static void test_ready        (Fixture *fix, gconstpointer data);
static void test_clone        (Fixture *fix, gconstpointer data);
static void test_row_added    (Fixture *fix, gconstpointer data);
static void test_row_changed  (Fixture *fix, gconstpointer data);
static void test_row_removed  (Fixture *fix, gconstpointer data);
static void test_model_clear  (Fixture *fix, gconstpointer data);
static void test_row_inserted (Fixture *fix, gconstpointer data);
static void test_uninitialized_leader (Fixture *fix, gconstpointer data);
static void test_introspect   (Fixture *fix, gconstpointer data);

void
test_model_interactions_create_suite (void)
{
#define DOMAIN "/Model/Interactions"

  g_test_add (DOMAIN"/Ready", Fixture, 0,
              model_setup, test_ready, model_teardown);
  g_test_add (DOMAIN"/Clone", Fixture, 0,
              model_setup, test_clone, model_teardown);
  g_test_add (DOMAIN"/RowAdded", Fixture, 0,
              model_setup, test_row_added, model_teardown);
  g_test_add (DOMAIN"/RowChanged", Fixture, 0,
              model_setup, test_row_changed, model_teardown);
  g_test_add (DOMAIN"/RowRemoved", Fixture, 0,
                model_setup, test_row_removed, model_teardown);
  g_test_add (DOMAIN"/Clear", Fixture, 0,
              model_setup, test_model_clear, model_teardown);
  g_test_add (DOMAIN"/RowInserted", Fixture, 0,
              model_setup, test_row_inserted, model_teardown);
  g_test_add (DOMAIN"/UninitializedLeader", Fixture, 0,
              model_setup_null, test_uninitialized_leader, model_teardown_null);
  g_test_add (DOMAIN"/Introspect", Fixture, 0,
              model_setup, test_introspect, model_teardown);
}

static gboolean
fatal_log_handler (const gchar *log_domain,
                   GLogLevelFlags log_level,
                   const gchar *message,
                   gpointer user_data)
{
  /* The warnings we ignore from dbus-glib starts with 'signal ' */
  return !g_str_has_prefix (message, "signal ");
}

static void
model_setup (Fixture *fix, gconstpointer data)
{   
  fix->model = dee_shared_model_new (MODEL_NAME,
                                      2,
                                      G_TYPE_INT,
                                      G_TYPE_STRING);

  g_assert (DEE_IS_MODEL (fix->model));

  /* We need to ignore certain warnings from dbus-glib */
  g_test_log_set_fatal_handler (fatal_log_handler, NULL);
}

static void
model_teardown (Fixture *fix, gconstpointer data)
{
  gtx_assert_last_unref (fix->model);
}

static void
model_setup_null (Fixture *fix, gconstpointer data)
{   
  fix->model = NULL;

  /* We need to ignore certain warnings from dbus-glib */
  g_test_log_set_fatal_handler (fatal_log_handler, NULL);
}

static void
model_teardown_null (Fixture *fix, gconstpointer data)
{
  g_assert (fix->model == NULL);
}

static void
test_ready (Fixture *fix, gconstpointer data)
{   
  if (!gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Ready signal emitted, but we have not connected yet!");

  dee_shared_model_connect (DEE_SHARED_MODEL (fix->model));
  if (gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model never emitted 'ready' signal");

  if (!gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model emitted 'ready' signal twice");
}

static gboolean
_add3rows (DeeModel *model)
{
  g_return_val_if_fail (DEE_IS_MODEL (model), FALSE);

  dee_model_append (model, 0, 0, 1, "zero", -1);
  dee_model_append (model, 0, 1, 1, "one", -1);
  dee_model_append (model, 0, 2, 1, "two", -1);
  return FALSE;
}

static gboolean
_add5rows (DeeModel *model)
{
  g_return_val_if_fail (DEE_IS_MODEL (model), FALSE);

  dee_model_append (model, 0, 0, 1, "zero", -1);
  dee_model_append (model, 0, 1, 1, "one", -1);
  dee_model_append (model, 0, 2, 1, "two", -1);
  dee_model_append (model, 0, 3, 1, "three", -1);
  dee_model_append (model, 0, 4, 1, "four", -1);
  return FALSE;
}

static gboolean
_change3rows (DeeModel *model)
{
  DeeModelIter *iter;

  g_return_val_if_fail (DEE_IS_MODEL (model), FALSE);

  iter = dee_model_get_iter_at_row (model, 0);
  dee_model_set (model, iter, 1, "changed_zero", -1);

  iter = dee_model_get_iter_at_row (model, 1);
  dee_model_set (model, iter, 1, "changed_one", -1);

  iter = dee_model_get_iter_at_row (model, 2);
  dee_model_set (model, iter, 1, "changed_two", -1);
  
  return FALSE;
}

/* Assumes a model with 5 rows. Removes rows 0, 4, and 2
 * in that order. Accounting for rows shifts this becomes
 * 0, 3, and 1. Leaving the original rows 1 and 3 now in
 * positions 0 and 1 */
static gboolean
_remove3rows (DeeModel *model)
{
  DeeModelIter *iter0, *iter4, *iter2;
  DeeModelIter *orig_iter1, *orig_iter3;

  g_return_val_if_fail (DEE_IS_MODEL (model), FALSE);
  g_return_val_if_fail (dee_model_get_n_rows (model) == 5, FALSE);

  iter0 = dee_model_get_iter_at_row (model, 0);
  iter4 = dee_model_get_iter_at_row (model, 4);
  iter2 = dee_model_get_iter_at_row (model, 2);

  orig_iter1 = dee_model_get_iter_at_row (model, 1);
  orig_iter3 = dee_model_get_iter_at_row (model, 3);

  dee_model_remove (model, iter0);
  dee_model_remove (model, iter4);
  dee_model_remove (model, iter2);

  g_assert_cmpint (dee_model_get_n_rows (model), ==, 2);
  g_assert (dee_model_get_iter_at_row (model, 0) == orig_iter1);
  g_assert (dee_model_get_iter_at_row (model, 1) == orig_iter3);

  return FALSE;
}

static gboolean
_insert1row (DeeModel *model)
{
  g_return_val_if_fail (DEE_IS_MODEL (model), FALSE);

  dee_model_insert (model, 1, 0, 27, 1, "twentyseven", -1);
  return FALSE;
}

static gboolean
_clear_model (DeeModel *model)
{
  g_return_val_if_fail (DEE_IS_MODEL (model), FALSE);

  dee_model_clear (model);
  
  return FALSE;
}

static void
test_clone (Fixture *fix, gconstpointer data)
{ 
  dee_shared_model_connect (DEE_SHARED_MODEL (fix->model));
  if (gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model never emitted 'ready' signal");

  _add3rows (fix->model);

  if (gtx_wait_for_command (TESTDIR,
                            MODEL_HELPER (clone3rows, MODEL_NAME),
                            1000))
    g_critical ("Model helper timed out");

  gtx_assert_last_command_status (0);

  /* We test that we can do this two times */
  /*if (gtx_wait_for_command (TESTDIR, MODEL_HELPER (3rows, MODEL_NAME), 1000))
    g_critical ("Model helper timed out");

  gtx_assert_last_command_status (0);*/
}

static void
test_row_added (Fixture *fix, gconstpointer data)
{ 
  dee_shared_model_connect (DEE_SHARED_MODEL (fix->model));
  if (gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model never emitted 'ready' signal");

  g_timeout_add (500, (GSourceFunc)_add3rows, fix->model);

  if (gtx_wait_for_command (TESTDIR,
                            MODEL_HELPER (add3rows, MODEL_NAME),
                            2000))
    g_critical ("Model helper timed out");

  gtx_assert_last_command_status (0);
}

static void
test_row_changed (Fixture *fix, gconstpointer data)
{ 
  dee_shared_model_connect (DEE_SHARED_MODEL (fix->model));
  if (gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model never emitted 'ready' signal");

  _add3rows (fix->model);
  g_timeout_add (500, (GSourceFunc)_change3rows, fix->model);

  //const gchar *cmd[] = {"dbus-monitor", NULL};
  //const gchar *cmd[] = {"sleep", "1",NULL};
  if (gtx_wait_for_command (TESTDIR,
                            MODEL_HELPER (change3rows, MODEL_NAME),
                            2000))
    g_critical ("Model helper timed out");
  
  gtx_assert_last_command_status (0);  
}

static void
test_row_removed (Fixture *fix, gconstpointer data)
{
  dee_shared_model_connect (DEE_SHARED_MODEL (fix->model));
  if (gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model never emitted 'ready' signal");

  _add5rows (fix->model);
  g_timeout_add (500, (GSourceFunc)_remove3rows, fix->model);

  if (gtx_wait_for_command (TESTDIR,
                            MODEL_HELPER (remove3rows, MODEL_NAME),
                            2000))
    g_critical ("Model helper timed out");

  gtx_assert_last_command_status (0);
}

static void
test_model_clear (Fixture *fix, gconstpointer data)
{ 
  dee_shared_model_connect (DEE_SHARED_MODEL (fix->model));
  if (gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model never emitted 'ready' signal");

  _add3rows (fix->model);
  g_timeout_add (500, (GSourceFunc)_clear_model, fix->model);
  
  if (gtx_wait_for_command (TESTDIR,
                            MODEL_HELPER (clear3rows, MODEL_NAME),
                            2000))
    g_critical ("Model helper timed out");
  
  gtx_assert_last_command_status (0);  
}

static void
test_row_inserted (Fixture *fix, gconstpointer data)
{ 
  dee_shared_model_connect (DEE_SHARED_MODEL (fix->model));
  if (gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model never emitted 'ready' signal");

  _add3rows (fix->model);
  g_timeout_add (500, (GSourceFunc)_insert1row, fix->model);
  
  if (gtx_wait_for_command (TESTDIR,
                            MODEL_HELPER (insert1row, MODEL_NAME),
                            2000))
    g_critical ("Model helper timed out");

  gtx_assert_last_command_status (0);
}

static void
_ready (DeeModel *model, GSList **rows_so_far)
{
  /* We must not have any rows when 'ready' is emitted */
  g_assert (*rows_so_far == NULL);
}

static void
_collect_row (DeeModel *model, DeeModelIter *iter, GSList **rows_so_far)
{
  /* Yes, I _know_ that append() is slow, but this is a test! */
  *rows_so_far = g_slist_append (*rows_so_far, iter);
}

/* This case must run without a Fixture  */
static void
test_uninitialized_leader (Fixture *fix, gconstpointer data)
{
  DeeModel     *model;
  DeeModelIter *iter;
  GSList        *rows_added;
  
  g_assert (fix->model == NULL);

  /* Set up a clean model without column type info */
  model = dee_shared_model_new_with_name (MODEL_NAME);
  dee_shared_model_connect (DEE_SHARED_MODEL (model));

  /* Listen for changes */
  rows_added = NULL;
  g_signal_connect (model, "row-added", G_CALLBACK (_collect_row), &rows_added);
  g_signal_connect (model, "ready", G_CALLBACK (_ready), &rows_added);
  
  /* Remote process defines column types and adds two rows */
  if (gtx_wait_for_command (TESTDIR,
                            MODEL_HELPER (uninitialized, MODEL_NAME),
                            2000))
    g_critical ("Model helper timed out");

  /* Check that we got what we expected */
  g_assert_cmpint (g_slist_length (rows_added), == , 2);
  g_assert_cmpint (dee_model_get_n_rows (model), ==, 2);
  g_assert_cmpint (dee_model_get_n_columns (model), ==, 2);

  iter = (DeeModelIter*) g_slist_nth (rows_added, 0)->data;
  g_assert_cmpint (dee_model_get_position (model, iter), == , 0);
  g_assert_cmpint (dee_model_get_int (model, iter, 0), == , 27);
  g_assert_cmpstr (dee_model_get_string (model, iter, 1), == , "skunkworks");

  iter = (DeeModelIter*) g_slist_nth (rows_added, 1)->data;
  g_assert_cmpint (dee_model_get_position (model, iter), == , 1);
  g_assert_cmpint (dee_model_get_int (model, iter, 0), == , 68);
  g_assert_cmpstr (dee_model_get_string (model, iter, 1), == , "wumbo");

  gtx_assert_last_unref (model);
  g_slist_free (rows_added);
}

static void
test_introspect (Fixture *fix, gconstpointer data)
{ 
  dee_shared_model_connect (DEE_SHARED_MODEL (fix->model));
  if (gtx_wait_for_signal (G_OBJECT (fix->model), TIMEOUT, "ready"))
    g_critical ("Model never emitted 'ready' signal");

  if (gtx_wait_for_command (TESTDIR,
                            MODEL_HELPER (introspect, MODEL_NAME),
                            2000))
    g_critical ("Model helper timed out");

  gtx_assert_last_command_status (0);
}
