/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <gio/gio.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>

#include "launcher-app.h"

#include "clutter-focus-manager.h"
#include "launcher-background.h"
#include "launcher-catbar.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-iconview.h"
#include "launcher-icon.h"
#include "launcher-link-editor.h"
#include "launcher-menu.h"
#include "launcher-startup.h"
#include "launcher-util.h"
#include "launcher-wm.h"


G_DEFINE_TYPE (LauncherApp, launcher_app, CLUTTER_TYPE_GROUP);

#define LAUNCHER_APP_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_APP, \
  LauncherAppPrivate))

struct _LauncherAppPrivate
{
  GFile *watch_dir;
  GFileMonitor *dir_monitor;

  gboolean windowed;
  gboolean lock;

  WnckScreen     *screen;
  LauncherConfig *config;
  LauncherWm     *wm;
  LauncherMenu   *menu;
  
  ClutterActor *focus_manager;
  
  ClutterActor *bg;
  ClutterActor *shadow;
  ClutterActor *catbar;
  ClutterActor *iconview;

  ClutterTimeline       *time;
  ClutterEffectTemplate *temp;

  ClutterTimeline       *timeslow;
  ClutterEffectTemplate *tempslow;

  guint eat_tag;

  ClutterTimeline *playing;
};

void      
launcher_app_lock (LauncherApp *app, gboolean lock)
{
  g_return_if_fail (LAUNCHER_IS_APP (app));

  app->priv->lock = lock;
}

static void
check_status (ClutterActor *catbar, LauncherApp *app)
{
  g_return_if_fail (LAUNCHER_IS_APP (app));

  if (CLUTTER_IS_ACTOR (app->priv->iconview))
    clutter_actor_destroy (app->priv->iconview);

  launcher_catbar_unset (LAUNCHER_CATBAR (app->priv->catbar));
  launcher_util_start_input ();
}

static void
activate_view (LauncherApp *app, gboolean activate)
{
  LauncherAppPrivate *priv = app->priv;
    
  launcher_util_stop_input ();

  if (activate)
  { 
    priv->playing = clutter_effect_move (priv->tempslow, priv->catbar, 
                                         0, 0, 
                                         NULL, NULL);
    launcher_background_blur (LAUNCHER_BACKGROUND (priv->bg));
  }
  else
  {
    if (CLUTTER_IS_ACTOR (priv->iconview))
    {
      launcher_iconview_disable (LAUNCHER_ICONVIEW (priv->iconview));
      clutter_effect_fade (priv->temp, priv->iconview, 0,
                           (ClutterEffectCompleteFunc)clutter_actor_destroy, 
                           NULL);
    }

    launcher_background_unblur (LAUNCHER_BACKGROUND (priv->bg));
    launcher_catbar_unset (LAUNCHER_CATBAR (priv->catbar));
    priv->playing = clutter_effect_move (priv->tempslow, priv->catbar, 
                         0, CSH()*0.1, 
                         (ClutterEffectCompleteFunc)check_status, app);
  }
}

void
launcher_app_reset (LauncherApp *app)
{
  g_return_if_fail (LAUNCHER_IS_APP (app));
  
  activate_view (app, FALSE);
}

/* Callbacks */
static gboolean
on_scroll_event (ClutterActor       *actor,
                 ClutterScrollEvent *event,
                 LauncherApp        *app)
{
  g_return_val_if_fail (LAUNCHER_IS_APP (app), FALSE);

  return launcher_iconview_scroll_event (actor, 
                                         event, 
                                       LAUNCHER_ICONVIEW (app->priv->iconview));
}

static gboolean
on_bg_clicked (ClutterActor       *actor,
               ClutterButtonEvent *event,
               LauncherApp        *app)
{
  g_return_val_if_fail (LAUNCHER_IS_APP (app), FALSE);
  
  if (event->button != 1)
    return FALSE;
 
  activate_view (app, FALSE);
  
  return TRUE;
}

static void
on_new_category (LauncherCatbar       *catbar,
                 LauncherMenuCategory *category,
                 LauncherApp          *app)
{
  LauncherAppPrivate *priv;
  LauncherMenu *menu = launcher_menu_get_default ();
  LauncherConfig *cfg = launcher_config_get_default ();
  gint view_width = cfg->iconview_width;
  gboolean first_run = TRUE;

  g_return_if_fail (LAUNCHER_IS_APP (app));
  g_return_if_fail (category);
  priv = app->priv;

  priv->lock = FALSE;

  if (CLUTTER_IS_ACTOR (priv->iconview))
  {
      launcher_iconview_disable (LAUNCHER_ICONVIEW (priv->iconview));
      clutter_effect_fade (priv->temp, priv->iconview, 0,
                         (ClutterEffectCompleteFunc)clutter_actor_destroy, 
                         NULL);
    first_run = FALSE;
  }

  if (!launcher_menu_get_n_categories (menu))
  {
    activate_view (app, FALSE);
    return;
  }
  if (first_run)
    activate_view (app, TRUE);

  priv->iconview = launcher_iconview_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (app),
                               priv->iconview);
  clutter_actor_set_position (priv->iconview,
                              (CSW()/2)-(view_width/2),
                              ((CSH()/2)-(cfg->iconview_height/2))
                                +cfg->cat_height/2 + PANEL_HEIGHT/2);

  clutter_actor_show (priv->iconview);
  clutter_focus_manager_insert_group(CLUTTER_FOCUS_MANAGER(priv->focus_manager),
                                     CLUTTER_FOCUS_GROUP (priv->iconview), 2);

  launcher_iconview_set_category (LAUNCHER_ICONVIEW (priv->iconview), 
                                  category,
                                  first_run);
}

static gboolean
eat_event (ClutterActor *stage, ClutterEvent *event, LauncherApp *app)
{
  gint opacity = clutter_actor_get_opacity (CLUTTER_ACTOR (app));

  if (opacity == 255)
  {
    g_signal_handlers_disconnect_by_func (stage, eat_event, app);
    clutter_ungrab_pointer ();
    return FALSE;
  }
  

  return TRUE;
}

static void
ungrab (ClutterActor *actor, LauncherApp *app)
{
  clutter_ungrab_pointer ();
  g_signal_handler_disconnect (clutter_stage_get_default (),
                               app->priv->eat_tag);
}


static void
on_active_window_changed (WnckScreen  *screen,
                          WnckWindow  *old_window,
                          LauncherApp *app)
{
  LauncherAppPrivate *priv;
  ClutterActor *stage = clutter_stage_get_default ();
  WnckWindow *window;
  WnckWindowType type = WNCK_WINDOW_DESKTOP;
  static ClutterTimeline *old_time = NULL;
    
  g_return_if_fail (LAUNCHER_IS_APP (app));
  priv = app->priv;

  window = wnck_screen_get_active_window (screen);
  if (WNCK_IS_WINDOW (window))
    type = wnck_window_get_window_type (window);

  if (priv->lock)
  {
    if (WNCK_IS_WINDOW (window) && type == WNCK_WINDOW_DESKTOP)
    {
      priv->lock = FALSE;
    }
    return;
  }
  
  if (priv->windowed)
    return;

  if (CLUTTER_IS_TIMELINE (old_time))
  {
    clutter_timeline_stop (old_time);
    g_object_unref (old_time);
  }

  if (!WNCK_IS_WINDOW (window) 
        || type == WNCK_WINDOW_DESKTOP
        || type == WNCK_WINDOW_DOCK)
  {
    old_time = clutter_effect_fade (priv->temp, CLUTTER_ACTOR (app), 
                                    255, 
                                    (ClutterEffectCompleteFunc)ungrab, app);
  }
  else
  {
    old_time = clutter_effect_fade (priv->temp, CLUTTER_ACTOR (app), 
                                    50, NULL, NULL);
    clutter_grab_pointer (stage);
    priv->eat_tag = g_signal_connect (stage, "captured-event", 
                                      G_CALLBACK (eat_event), app);
  }
}

/*
 * External link watching 
 */
static void
on_link_reponse (GtkWidget *dialog, gint res, LauncherApp *app)
{
  GList *cats;

  if (!LAUNCHER_IS_APP (app))
  {
    gtk_widget_destroy (dialog);
    return;
  }
  cats = launcher_menu_get_categories (app->priv->menu);

  if (res == GTK_RESPONSE_YES)
  {

     gchar *name = NULL, *exec = NULL, *icon_name = NULL;
     gint cat = 0;
     g_object_get (dialog, 
                   "link-category", &cat,
                   "link-name", &name,
                   "link-exec", &exec,
                   "link-icon", &icon_name,
                   NULL);
     launcher_menu_add_application (launcher_menu_get_default (),
                                    g_list_nth_data (cats, cat), 
                                    name, exec, icon_name);
     g_free (name);
     g_free (exec);
     g_free (icon_name);
  }
  app->priv->lock = FALSE;
  gtk_widget_destroy (dialog);
}


static void
on_watch_dir_changed (GFileMonitor      *monitor,
                      GFile             *file,
                      GFile             *other_file,
                      GFileMonitorEvent  event,
                      LauncherApp       *app)
{
  LauncherAppPrivate *priv;
  gchar *link_path;

  g_return_if_fail (LAUNCHER_IS_APP (app));
  priv = app->priv;

  if (event != G_FILE_MONITOR_EVENT_CREATED)
    return;

  link_path = g_file_get_path (file);
  if (link_path && strstr (link_path, ".desktop"))
  {
    GKeyFile *file;
    
    file = g_key_file_new ();
    if (g_key_file_load_from_file (file, link_path, 0, NULL))
    {
      GtkWidget *dialog;
      gchar *icon_path;
      
      icon_path = g_build_filename (LINK_WATCH_DIR, 
                                    g_key_file_get_string (file, 
                                                      G_KEY_FILE_DESKTOP_GROUP,
                                           G_KEY_FILE_DESKTOP_KEY_ICON, NULL),
                                    NULL);
      dialog = launcher_link_editor_new ();
      g_object_set (dialog, 
                "title", _("Add Shortcut"),
                "link-name", 
                  g_key_file_get_string (file, G_KEY_FILE_DESKTOP_GROUP,
                                         G_KEY_FILE_DESKTOP_KEY_NAME, NULL),
                "link-exec",
                  g_key_file_get_string (file, G_KEY_FILE_DESKTOP_GROUP,
                                         G_KEY_FILE_DESKTOP_KEY_URL, NULL),
                "link-icon", icon_path,
                NULL);
      gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
                              "gtk-cancel", GTK_RESPONSE_CANCEL,
                              GTK_STOCK_APPLY, GTK_RESPONSE_YES,
                              NULL);
      g_signal_connect (dialog, "response",
                        G_CALLBACK (on_link_reponse), app);
                                   
      launcher_util_present_window (GTK_WINDOW (dialog));
      g_free (icon_path);
    }
    g_key_file_free (file);
  }
 

  g_free (link_path);
}

static void
watch_links (LauncherApp *app)
{
  LauncherAppPrivate *priv = app->priv;
  GFile *dir;

  g_mkdir_with_parents (LINK_WATCH_DIR, 0700);

  dir = priv->watch_dir = g_file_new_for_path (LINK_WATCH_DIR);

  priv->dir_monitor = g_file_monitor_directory (dir, 0, NULL, NULL);

  g_signal_connect (priv->dir_monitor, "changed",
                    G_CALLBACK (on_watch_dir_changed), app);
}

/* GObject stuff */
static void
launcher_app_finalize (GObject *object)
{
  LauncherAppPrivate *priv;

  priv = LAUNCHER_APP_GET_PRIVATE (object);

  g_object_unref (priv->wm);
  g_object_unref (priv->menu);
  g_object_unref (priv->watch_dir);
  g_object_unref (priv->dir_monitor);

  G_OBJECT_CLASS (launcher_app_parent_class)->finalize (object);
}

static void
launcher_app_class_init (LauncherAppClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = launcher_app_finalize;

  g_type_class_add_private (obj_class, sizeof (LauncherAppPrivate));
}

static void
real_init (LauncherApp *app)
{
  LauncherAppPrivate *priv;
  ClutterActor *stage = clutter_stage_get_default ();
  	
  priv = app->priv = LAUNCHER_APP_GET_PRIVATE (app);

  priv->lock = FALSE;

  /* The desktop background */
  priv->bg = launcher_background_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), priv->bg);
  clutter_actor_set_size (priv->bg, CSW(), CSH());
  clutter_actor_set_position (priv->bg, 0, 0);
  clutter_actor_show (priv->bg);
  clutter_actor_set_reactive (priv->bg, TRUE);
  g_signal_connect (priv->bg, "scroll-event",
                    G_CALLBACK (on_scroll_event),app);
  g_signal_connect (priv->bg, "button-release-event",
                    G_CALLBACK (on_bg_clicked),app);


  clutter_actor_set_anchor_point_from_gravity (priv->bg, 
                                               CLUTTER_GRAVITY_CENTER);
  clutter_actor_set_position (priv->bg, CSW()/2, CSH()/2);

  /* Main container group */
  clutter_actor_set_position (CLUTTER_ACTOR (app), 0, 0);
  clutter_actor_set_size (CLUTTER_ACTOR (app), CSW(), CSH());
  clutter_actor_show (CLUTTER_ACTOR (app));
  clutter_actor_raise_top (CLUTTER_ACTOR (app));

  stage = CLUTTER_ACTOR (app);

  /* Category bar */
  priv->catbar = launcher_catbar_get_default ();
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), priv->catbar);
  clutter_actor_set_position (priv->catbar, 0, CSH()*0.1);
  clutter_actor_show (priv->catbar);

  g_signal_connect (priv->catbar, "category-selected",
                    G_CALLBACK (on_new_category), app);
}

static void
on_config_changed (LauncherConfig *config, LauncherApp *app)
{
  LauncherAppPrivate *priv = app->priv;

  clutter_actor_destroy (priv->bg);
  clutter_actor_destroy (priv->catbar);
  if (priv->iconview) clutter_actor_destroy (priv->iconview);
  real_init (app);
}

static void
launcher_app_init (LauncherApp *app)
{
  LauncherAppPrivate *priv;
  ClutterActor *stage = clutter_stage_get_default ();
  const gchar *winmode;
  	
  priv = app->priv = LAUNCHER_APP_GET_PRIVATE (app);

  winmode = g_getenv ("LAUNCHER_WINDOWED");
  priv->windowed = winmode ? atoi (winmode) : FALSE;
  priv->screen = wnck_screen_get_default ();
  priv->lock = FALSE;

  priv->config = launcher_config_get_default ();

  g_signal_connect (priv->config, "config-changed", 
                    G_CALLBACK (on_config_changed), app);

  /* The desktop background */
  priv->bg = launcher_background_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), priv->bg);
  clutter_actor_set_size (priv->bg, CSW(), CSH());
  clutter_actor_set_position (priv->bg, 0, 0);
  clutter_actor_show (priv->bg);
  clutter_actor_set_reactive (priv->bg, TRUE);
  g_signal_connect (priv->bg, "scroll-event",
                    G_CALLBACK (on_scroll_event),app);
  g_signal_connect (priv->bg, "button-release-event",
                    G_CALLBACK (on_bg_clicked),app);


  clutter_actor_set_anchor_point_from_gravity (priv->bg, 
                                               CLUTTER_GRAVITY_CENTER);
  clutter_actor_set_position (priv->bg, CSW()/2, CSH()/2);

  /* Main container group */
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), CLUTTER_ACTOR (app));
  clutter_actor_set_position (CLUTTER_ACTOR (app), 0, 0);
  clutter_actor_set_size (CLUTTER_ACTOR (app), CSW(), CSH());
  clutter_actor_show (CLUTTER_ACTOR (app));

  /* Focus manager */
  priv->focus_manager = clutter_focus_manager_get_default ();
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), priv->focus_manager);

  stage = CLUTTER_ACTOR (app);

  /* Shadow 
  priv->shadow = launcher_util_texture_new_from_file (PKGDATADIR"/shadow.png");
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), priv->shadow);
  clutter_actor_set_position (priv->shadow, 0, PANEL_HEIGHT);
  clutter_actor_show (priv->shadow);
  */

  /* Init the WM interface */
  priv->wm = launcher_wm_get_default ();
  g_signal_connect (priv->screen, "active-window-changed",
                    G_CALLBACK (on_active_window_changed), app);

  /* Application menu */
  priv->menu = launcher_menu_get_default ();

  /* Category bar */
  priv->catbar = launcher_catbar_get_default ();
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), priv->catbar);
  clutter_actor_set_position (priv->catbar, 0, CSH()*0.1);
  clutter_actor_show (priv->catbar);
  clutter_focus_manager_insert_group(CLUTTER_FOCUS_MANAGER(priv->focus_manager),
                                     CLUTTER_FOCUS_GROUP (priv->catbar),
                                     0);

  g_signal_connect (priv->catbar, "category-selected",
                    G_CALLBACK (on_new_category), app);

  priv->time = launcher_util_timeline_new_fast ();
  priv->temp = clutter_effect_template_new (priv->time, 
                                            clutter_sine_inc_func);

  priv->timeslow = launcher_util_timeline_new_slow ();
  priv->tempslow  = clutter_effect_template_new (priv->time, 
                                            clutter_sine_inc_func);

  watch_links (app);
}

LauncherApp *
launcher_app_get_default (void)

{
  static LauncherApp *app = NULL;

  if (!app)
    app = g_object_new (LAUNCHER_TYPE_APP, 
                       NULL);

  return app;
}
