/*
 * Copyright (C) 2008  Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * In addition, as a special exception, Intel gives permission to link
 * the code of portions of this program with the OpenSSL project's
 * "OpenSSL" library (or with modified versions of it that use the same
 * license as the "OpenSSL" library), and distribute the linked
 * executables.  You must obey the GNU General Public License in all
 * respects for all of the code used other than "OpenSSL".  If you modify
 * this file, you may extend this exception to your version of the file,
 * but you are not obligated to do so.  If you do not wish to do so,
 * delete this exception statement from your version.
 */


/**
 * 
 * @file AccelHelper.cpp
 *
 * Implementation of class AccelHelper
 */

#include <assert.h>
#include "AccelHelper.h"

/* const values */
const int AccelHelper::ORIENTATION_SHRESHOLD = 2;
const char * const AccelHelper::MODULE_NAME = "Freescale MMA7360L";
const int AccelHelper::G_SELECT_1P5G = 0;
const int AccelHelper::G_SELECT_6G = 1;

const char AccelHelper::GRAVITY_MEASUREMENT_1P5G = 62;
const int AccelHelper::ACCEL_SENSE_1P5G = 0x0A;
const int AccelHelper::GRAVITY_DIV_SQRT2_1P5G =
	(int)(GRAVITY_MEASUREMENT_1P5G / 1.4);
const int AccelHelper::SQR_GRAVITY_DIV2_1P5G =
	GRAVITY_MEASUREMENT_1P5G * GRAVITY_MEASUREMENT_1P5G / 2;

const char AccelHelper::GRAVITY_MEASUREMENT_6G = 16;
const int AccelHelper::ACCEL_SENSE_6G = 0x03;
const int AccelHelper::GRAVITY_DIV_SQRT2_6G =
	(int)(GRAVITY_MEASUREMENT_6G / 1.4);
const int AccelHelper::SQR_GRAVITY_DIV2_6G =
	GRAVITY_MEASUREMENT_6G * GRAVITY_MEASUREMENT_6G / 2;

#ifdef WIN32

const wchar_t * const AccelHelper::EVENT_NAME = L"AccelEvent";
const wchar_t * const AccelHelper::DEVICE_PATH = L"\\\\.\\Accelerometer";

DWORD WINAPI AccelEventHandler(LPVOID lpParameter)
{
	HANDLE handles[2] = {AccelHelper::GetInstance().m_hAccelEvent,
		AccelHelper::GetInstance().m_hExitEvent};

	while (WAIT_OBJECT_0 == 
		WaitForMultipleObjects(2, handles, FALSE, INFINITE)) {
		/* AccelEvent was set by driver. Reset it for next notification */
		ResetEvent(AccelHelper::GetInstance().m_hAccelEvent);

		/* get and call DCS to dispatch accel data */
		DCS_Accel_Data data;
		if (!AccelHelper::GetInstance().GetAccelData(data)) {
			return DCS_FAIL_OPERATION;
		}

        if (AccelHelper::GetInstance().IsEventEnabled()) {
    		assert(NULL != AccelHelper::GetInstance().m_pFuncs);
	    	AccelHelper::GetInstance().m_pFuncs->dispatchEvent(DCS_ACCELEROMETER,
		    	ACCEL_DATA_CHANGED, &data, sizeof(DCS_Accel_Data));
        }

		AccelHelper::GetInstance().CheckAndNotifySystemOrientation(data);
	}

	return DCS_SUCCESS;
}

/**
 * Initialize accel plugin. During initialization, handles and threads
 * will be created. Accel will be started.
 *
 * @param pDispatcher	a point to DCS callback struct
 * @return	DCS_SUCCESS if success; DCS_FAIL_OPERATION if error
 */
int AccelHelper::Initialize(DCS_Funcs *pDispatcher)
{
	if (m_hAccelEvent || m_hExitEvent || m_hThread || m_hDevice) {
		return DCS_FAIL_OPERATION;
	}

	m_hAccelEvent = CreateEvent(NULL, TRUE, FALSE, EVENT_NAME);
	if (NULL == m_hAccelEvent) {
		return DCS_FAIL_OPERATION;
	}

	m_hExitEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
	if (NULL == m_hExitEvent) {
		CloseHandles();
		return DCS_FAIL_OPERATION;
	}

	m_hDevice = CreateFile(DEVICE_PATH, GENERIC_READ | GENERIC_WRITE,
		0, NULL, OPEN_EXISTING, 0, NULL);
	if (INVALID_HANDLE_VALUE == m_hDevice) {
		CloseHandles();
		return DCS_FAIL_OPERATION;
	}

	if (SetAccelGSelect(G_SELECT_1P5G)) {
		m_GSelect = G_SELECT_1P5G;
		m_AccelSense = ACCEL_SENSE_1P5G;
		m_GravityDivSqrt2 = GRAVITY_DIV_SQRT2_1P5G;
		m_SqrGravityDiv2 = SQR_GRAVITY_DIV2_1P5G;
		m_GravityMeasurement = GRAVITY_MEASUREMENT_1P5G;
		m_Orientation = DCS_SYSTEM_ORIENTATION_NORMAL;
	} else {
		CloseHandles();
		return DCS_FAIL_OPERATION;
	}

	if (!SetAccelSense(m_AccelSense)) {
		CloseHandles();
		return DCS_FAIL_OPERATION;
	}

	if (!StartAccel()) {
		CloseHandles();
		return DCS_FAIL_OPERATION;
	}

	m_hThread = CreateThread(NULL, 0, AccelEventHandler, NULL, 0, NULL);
	if (NULL == m_hThread) {
		CloseHandles();
		return DCS_FAIL_OPERATION;
	}

	m_pFuncs = pDispatcher;
	return DCS_SUCCESS;
}

void AccelHelper::CloseHandles()
{
	if (m_hAccelEvent) {
		CloseHandle(m_hAccelEvent);
		m_hAccelEvent = NULL;
	}

	if (m_hExitEvent) {
		CloseHandle(m_hExitEvent);
		m_hExitEvent = NULL;
	}

	if (m_hThread) {
		CloseHandle(m_hThread);
		m_hThread = NULL;
	}

	if (m_hDevice) {
		CloseHandle(m_hDevice);
		m_hThread = NULL;
	}
}

/** 
 * Shutdown accel plugin. After cleanup, accel will be stopped. Handles
 * will be closed and thread will exit.
 *
 * @return	DCS_SUCCESS if success; DCS_FAIL_OPERATION if error
 */
int AccelHelper::Cleanup()
{
	if (!SetEvent(m_hExitEvent)) {
		return DCS_FAIL_OPERATION;
	}

	if (!StopAccel()) {
		return DCS_FAIL_OPERATION;
	}

	/* wait 3 seconds for thread to return */
	if (WAIT_OBJECT_0 != WaitForSingleObject(m_hThread, 3000)) {
		return DCS_FAIL_OPERATION;
	}

	CloseHandles();
	
	return DCS_SUCCESS;
}

BOOL AccelHelper::StartAccel()
{
	ULONG  ulLength;

	if (!(DeviceIoControl(m_hDevice, IOCTL_ACCEL_START, NULL, 0, NULL, 0,
		&ulLength, NULL))) {
		return FALSE;
	}

	return TRUE;
}

BOOL AccelHelper::StopAccel()
{
	ULONG  ulLength;

	if (!(DeviceIoControl(m_hDevice, IOCTL_ACCEL_STOP, NULL, 0, NULL, 0,
		&ulLength, NULL))) {
		return FALSE;
	}

	return TRUE;
}

BOOL AccelHelper::GetAccelData(DCS_Accel_Data &data)
{
	ACCEL_RAW_DATA rawData;
	ULONG  ulLength;

	if (!(DeviceIoControl(m_hDevice, IOCTL_ACCEL_READ_DATA, NULL, 0,
		&rawData, sizeof(ACCEL_RAW_DATA), &ulLength, NULL))) {
		return FALSE;
	}

	/* translate ACCEL_RAW_DATA to DCS_ACCEL_DATA */
	/* reverse X and Y axis in alpha due to HW issue in EVT system */
	data.AccelX = rawData.AccelRawX - 128;
	data.AccelY = rawData.AccelRawY - 128;
	data.AccelZ = rawData.AccelRawZ - 128;
	return TRUE;
}

BOOL AccelHelper::SetAccelSense(int sense)
{
	ULONG  ulLength;

	if (!(DeviceIoControl(m_hDevice, IOCTL_ACCEL_SET_SENSE,
		&sense, sizeof(int), NULL, 0, &ulLength, NULL))) {
		return FALSE;
	}

	return TRUE;
}

BOOL AccelHelper::SetAccelGSelect(int select)
{
	ULONG  ulLength;

	if (!(DeviceIoControl(m_hDevice, IOCTL_ACCEL_SET_G_SELECT,
		&select, sizeof(int), NULL, 0, &ulLength, NULL))) {
		return FALSE;
	}

	return TRUE;
}

BOOL AccelHelper::GetAccelGSelect(int *select)
{
	ULONG  ulLength;

	if (!(DeviceIoControl(m_hDevice, IOCTL_ACCEL_GET_G_SELECT, NULL, 0,
		select, sizeof(int), &ulLength, NULL))) {
		return FALSE;
	}

	return TRUE;
}

#else

const char * const AccelHelper::DEVICE_PATH = "/proc/acpi/accel";

void *AccelEventHandler(void *)
{
	DCS_Accel_Data data;
	
	while (AccelHelper::GetInstance().GetAccelData(data)) {
        if (AccelHelper::GetInstance().IsEventEnabled()) {
    		assert(NULL != AccelHelper::GetInstance().m_pFuncs);
	    	AccelHelper::GetInstance().m_pFuncs->dispatchEvent(DCS_ACCELEROMETER,
		    	ACCEL_DATA_CHANGED, &data, sizeof(DCS_Accel_Data));
        }

		AccelHelper::GetInstance().CheckAndNotifySystemOrientation(data);
	}

	return NULL;
}

int AccelHelper::Initialize(DCS_Funcs *pDispatcher)
{
	m_Device = open(DEVICE_PATH, O_RDWR | O_SYNC);
	if (-1 == m_Device) {	
		return DCS_FAIL_OPERATION;
	}

	if (SetAccelGSelect(G_SELECT_1P5G)) {
		m_GSelect = G_SELECT_1P5G;
		m_AccelSense = ACCEL_SENSE_1P5G;
		m_GravityDivSqrt2 = GRAVITY_DIV_SQRT2_1P5G;
		m_SqrGravityDiv2 = SQR_GRAVITY_DIV2_1P5G;
		m_GravityMeasurement = GRAVITY_MEASUREMENT_1P5G;
		m_Orientation = DCS_SYSTEM_ORIENTATION_NORMAL;
	} else {
		goto fail0;
	}

	if (!SetAccelSense(m_AccelSense)) {
		goto fail0;
	}

	if (!StartAccel()) {
		goto fail0;
	}

	if (pthread_create(&m_Thread, NULL, AccelEventHandler, NULL)) {
		goto fail0;
	}

	m_pFuncs = pDispatcher;
	return DCS_SUCCESS;

fail0:
	if (m_Device >= 0) {
		close(m_Device);
		m_Device = -1;
	}
	return DCS_FAIL_OPERATION;
}

int AccelHelper::Cleanup()
{
	if (m_Device >= 0) {
		close(m_Device);
		m_Device = -1;
	}
	
	if (0 != m_Thread) {	
		if (pthread_join(m_Thread, NULL)) {
			m_Thread = 0;
			return DCS_FAIL_OPERATION;
		}
		m_Thread = 0;
	}

	return DCS_SUCCESS;
}

BOOL AccelHelper::StartAccel()
{
	if (-1 == ioctl(m_Device, IOCTL_ACCEL_START)) {
		return FALSE;
	}

	return TRUE;
}

BOOL AccelHelper::StopAccel()
{
	if (-1 == ioctl(m_Device, IOCTL_ACCEL_STOP)) {
		return FALSE;
	}

	return TRUE;
}

BOOL AccelHelper::GetAccelData(DCS_Accel_Data &data)
{
	struct accel_raw_data raw_data;
	int size = sizeof(raw_data);

	if (size != read(m_Device, &raw_data, size)) {
		return FALSE;
	}

	/* translate ACCEL_RAW_DATA to DCS_ACCEL_DATA */
	/* reverse X and Y axis in alpha due to HW issue in EVT system */
	data.AccelX = raw_data.accel_raw_x - 128;
	data.AccelY = raw_data.accel_raw_y - 128;
	data.AccelZ = raw_data.accel_raw_z - 128;
	return TRUE;
}

BOOL AccelHelper::SetAccelSense(int sense)
{
	if (-1 == ioctl(m_Device, IOCTL_ACCEL_SET_SENSE, sense)) {
		return FALSE;
	}

	return TRUE;
}

BOOL AccelHelper::SetAccelGSelect(int select)
{
	if (-1 == ioctl(m_Device, IOCTL_ACCEL_SET_G_SELECT, select)) {
		return FALSE;
	}

	return TRUE;
}

BOOL AccelHelper::GetAccelGSelect(int *select)
{
	if (-1 == ioctl(m_Device, IOCTL_ACCEL_GET_G_SELECT, select)) {
		return FALSE;
	}

	return TRUE;
}

#endif /* WIN32 */

DCS_SystemOrientation AccelHelper::GetSystemOrientation(
	const DCS_Accel_Data &data)
{
	if ((data.AccelZ <= m_GravityDivSqrt2) && (((data.AccelX * data.AccelX) 
		+ (data.AccelY * data.AccelY)) >= m_SqrGravityDiv2)) {
		if (abs(data.AccelX) >= abs(data.AccelY) * ORIENTATION_SHRESHOLD) {
			if (data.AccelX > 0) {
				return DCS_SYSTEM_ORIENTATION_CW90;
			} else {
				return DCS_SYSTEM_ORIENTATION_CW270;
			}
		} else if (abs(data.AccelY) >=
			abs(data.AccelX) * ORIENTATION_SHRESHOLD) {
			if (data.AccelY > 0) {
				return DCS_SYSTEM_ORIENTATION_NORMAL;
			} else {
				return DCS_SYSTEM_ORIENTATION_CW180;
			}
		}
	}

	return m_Orientation;
}

void AccelHelper::CheckAndNotifySystemOrientation(const DCS_Accel_Data &data)
{
	DCS_SystemOrientation orientation;

	/* check if the system orientation changed if it was enabled*/
	if (IsUserIdEmpty()) {
		orientation = GetSystemOrientation(data);
		if (m_Orientation != orientation) {
			/* orientation changed */
			m_Orientation = orientation;
			m_pFuncs->dispatchEvent(
				DCS_ACCELEROMETER, ACCEL_SYSTEM_ORIENTATION_CHANGED,
				&orientation, sizeof(orientation));
		}
	}
}

/**
 * Handle predefined DCS requests. According to request ID, return data
 * or operate accel accordingly.
 *
 * @param pRequest	a point to request struct
 * @return	DCS_SUCCESS if success; DCS_FUNC_NOTEXIST if no ID matched;
 * 			DCS_FAIL_OPERATION if error
 */
DCS_Return_Code AccelHelper::HandleRequest(DCS_RequestData *pRequest)
{
	switch (pRequest->funcId) {
	case ACCEL_GET_DATA:
		DCS_Accel_Data data;
		if (!GetAccelData(data)) {
			return DCS_FAIL_OPERATION;
		}

		*(PDCS_Accel_Data)(pRequest->pReturnData) = data;
		pRequest->pReturnDataLen = sizeof(DCS_Accel_Data);
		break;
	
	case ACCEL_GET_MEASUREMENT:
		*(char *)(pRequest->pReturnData) = m_GravityMeasurement;
		pRequest->pReturnDataLen = sizeof(char);
		break;

	case ACCEL_ENABLE_DESKTOP_AUTOROTATION:
		EmptyUserId();
		break;

	case ACCEL_DISABLE_DESKTOP_AUTOROTATION:
		AddUserId(pRequest->cliId);
		break;

	case ACCEL_UNDISABLE_DESKTOP_AUTOROTATION:
		RemoveUserId(pRequest->cliId);
		break;

	case ACCEL_GET_DESKTOP_AUTOROTATION_STATUS:
		if (IsUserIdEmpty()) {
			*(int *)(pRequest->pReturnData) = 1;
			pRequest->pReturnDataLen = sizeof(int);
		} else {
			*(int *)(pRequest->pReturnData) = 0;
			pRequest->pReturnDataLen = sizeof(int);
		}
		break;

	case ACCEL_GET_SYSTEM_ORIENTATION:
		*(DCS_SystemOrientation *)(pRequest->pReturnData) = m_Orientation;
		pRequest->pReturnDataLen = sizeof(m_Orientation);
		break;

    case ACCEL_ENABLE_DATA_EVENT:
        m_bIsEventEnabled = true;
        break;

    case ACCEL_DISABLE_DATA_EVENT:
        m_bIsEventEnabled = false;
        break;

	default:
		return DCS_FUNC_NOTEXIST;
	}

	return DCS_SUCCESS;
}

/**
 * Remove a client ID of DCS from disabled desktop orientation notification
 * list. When a client drop from DCS, this function will be called.
 *
 * @param cliId	a DCS client ID
 */
int AccelHelper::DestroyClient(int cliId)
{
	RemoveUserId(cliId);
	return DCS_SUCCESS;
}

