# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import pgm
import math
import gobject

from node import Node
from drawable import Drawable

class NodeNotInGroup(Exception):
    pass

class Group(Node):
    """
    DOCME

    Emits signals:
        - 'resized': when the size of the group changes; parameters are its
                     absolute width and height before the change.
        - 'repositioned': when the position of the group changes: parameters
                          are its absolute x, y and z coordinates before the
                          change.

    DOCME more signals.
    """

    # FIXME: add other Pigment signals: pressured, changed

    __gsignals__ = {
        'drag-begin': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT, gobject.TYPE_UINT)),
        'drag-motion': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT, gobject.TYPE_UINT)),
        'drag-end': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT)),
        'clicked': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT, gobject.TYPE_UINT)),
        'pressed': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT, gobject.TYPE_UINT)),
        'released': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT)),
        'double-clicked': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT)),
        'scrolled': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ScrollDirection, gobject.TYPE_UINT)),
        'resized': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT)),
        'repositioned': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT)),
    }

    def __init__(self):
        self._canvas = None
        self._layer = pgm.DRAWABLE_MIDDLE
        self._children = []

        # A dict instance->[<signal id>,...] to keep track of connected
        # signals and be able to disconnect from them.
        self._children_signal_id = {}
        
        super(Group, self).__init__()

    def canvas__set(self, canvas):
        old_canvas = self._canvas
        new_canvas = canvas

        self._canvas = canvas

        if old_canvas != None:
            for child in self._children:
                if isinstance(child, Drawable):
                    old_canvas.remove(child)
                elif isinstance(child, Group):
                    child.canvas = None

        if new_canvas != None:
            for child in self._children:
                if isinstance(child, Drawable):
                    new_canvas.add(self._layer, child)
                elif isinstance(child, Group):
                    child.canvas = new_canvas

    def canvas__get(self):
        return self._canvas

    canvas = property(canvas__get, canvas__set)

    def layer__set(self, layer):
        if layer == self._layer:
            return

        self._layer = layer

        for child in self._children:
            if isinstance(child, Drawable):
                self._canvas.remove(child)
                self._canvas.add(self._layer, child)
            elif isinstance(child, Group):
                child.layer = self._layer

    def layer__get(self):
        if self._canvas != None:
            return self._layer
        else:
            return pgm.DRAWABLE_UNBOUND

    layer = property(layer__get, layer__set)

    def add(self, child, forward_signals=True):
        self._children.append(child)
        child.parent = self

        if forward_signals and type(child) is not Node:
            id_list = []
            for signal_name in ('double-clicked', 'drag-end', 'released'):
                signal_id = child.connect(signal_name,
                                          self._proxy_child_signal, signal_name)
                id_list.append(signal_id)

            for signal_name in ('clicked', 'pressed', 'drag-begin', 'drag-motion'):
                signal_id = child.connect(signal_name,
                                      self._proxy_child_signal_with_pressure,
                                      signal_name)
                id_list.append(signal_id)
            signal_id = child.connect('scrolled',
                                      self._proxy_child_signal_scrolled,
                                      'scrolled')
            id_list.append(signal_id)

            self._children_signal_id[child] = id_list

        if self._canvas != None:
            if isinstance(child, Drawable):
                self._canvas.add(self._layer, child)
            elif isinstance(child, Group):
                child.canvas = self._canvas
                child.layer = self._layer

    def _proxy_child_signal_scrolled(self, drawable, x, y, z, direction, time, signal_name):
        if self.absolute_visible and self.absolute_opacity != 0:
            return self.emit(signal_name, x, y, z, direction, time)

    def _proxy_child_signal(self, drawable, x, y, z, button, time, signal_name):
        if self.absolute_visible and self.absolute_opacity != 0:
            return self.emit(signal_name, x, y, z, button, time)

    def _proxy_child_signal_with_pressure(self, drawable, x, y, z, button, time, pressure, signal_name):
        if self.absolute_visible and self.absolute_opacity != 0:
            return self.emit(signal_name, x, y, z, button, time, pressure)

    def remove(self, child):
        try:
            self._children.remove(child)
        except:
            raise NodeNotInGroup

        if child in self._children_signal_id:
            for signal_id in self._children_signal_id[child]:
                child.disconnect(signal_id)
            del self._children_signal_id[child]

        child.parent = None

        if self._canvas != None:
            if isinstance(child, Drawable):
                self._canvas.remove(child)
            elif isinstance(child, Group):
                child.canvas = None

    def empty(self):
        for child in self._children:
            child.parent = None
            if isinstance(child, Drawable):
                self._canvas.remove(child)
            elif isinstance(child, Group):
                child.canvas = None

        self._children = []

    def __len__(self):
        return len(self._children)

    def __iter__(self):
        return self._children.__iter__()

    def update_absolute_x(self, value=None):
        x, y, z = self.absolute_position
        super(Group, self).update_absolute_x(value)
        for child in self._children:
            child.update_absolute_x()
        self.emit("repositioned", x, y, z)

    def update_absolute_y(self, value=None):
        x, y, z = self.absolute_position
        super(Group, self).update_absolute_y(value)
        for child in self._children:
            child.update_absolute_y()
        self.emit("repositioned", x, y, z)

    def update_absolute_z(self, value=None):
        x, y, z = self.absolute_position
        super(Group, self).update_absolute_z(value)
        for child in self._children:
            child.update_absolute_z()
        self.emit("repositioned", x, y, z)

    def update_absolute_position(self, value=None):
        x, y, z = self.absolute_position
        super(Group, self).update_absolute_position(value)
        for child in self._children:
            child.update_absolute_position()
        self.emit("repositioned", x, y, z)

    def update_absolute_width(self, value=None):
        width, height = self.absolute_size
        super(Group, self).update_absolute_width(value)
        for child in self._children:
            child.update_absolute_width()
            # FIXME: write test before uncommenting
            child.update_absolute_x()
        self.emit("resized", width, height)

    def update_absolute_height(self, value=None):
        width, height = self.absolute_size
        super(Group, self).update_absolute_height(value)
        for child in self._children:
            child.update_absolute_height()
            # FIXME: write test before uncommenting
            child.update_absolute_y()
        self.emit("resized", width, height)

    def update_absolute_size(self, value=None):
        width, height = self.absolute_size
        super(Group, self).update_absolute_size(value)
        for child in self._children:
            child.update_absolute_size()
            # FIXME: write test before uncommenting
            child.update_absolute_position()
        self.emit("resized", width, height)

    def update_absolute_visible(self, value=None):
        super(Group, self).update_absolute_visible(value)
        for child in self._children:
            child.update_absolute_visible()

    def update_absolute_opacity(self, value=None):
        super(Group, self).update_absolute_opacity(value)
        for child in self._children:
            child.update_absolute_opacity()

    # FIXME: write tests
    def regenerate(self):
        for child in self._children:
            child.regenerate()
    
    def clean(self):
        # We need to create a new list since  we modify it in the loop.
        for child in list(self._children):
            self.remove(child)
            child.clean()

        return super(Group, self).clean()
