# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.components.controller import Controller
from elisa.core.utils.defer import Deferred, DeferredList, deferred_accumulator
from elisa.plugins.pigment.widgets.widget import Widget
from pgm.timing import implicit
from elisa.core import common
from elisa.core.input_event import *

import gobject

class PigmentController(Controller, gobject.GObject):
    """
    Specialised L{elisa.core.components.controller.Controller} for the Pigment
    toolkit.
    A L{PigmentController} creates a user interface in its L{widget}
    widget using other Pigment widgets.

    @ivar frontend:         frontend that created the controller
    @type frontend:         L{elisa.core.components.frontend.Frontend}
    @ivar sensitive:        whether this controller should react on click events
    @type sensitive:        C{bool}
    @ivar path:             path for which that controller was selected
    @type path:             C{str}
    @ivar widget:           Pigment widget containing the UI of the controller
    @type widget:           L{elisa.plugins.pigment.widgets.widget.Widget}
    @ivar widget_animated:  implicitly animated version of L{widget}
    @type widget_animated:  L{pgm.timing.implicit.AnimatedObject}
    """

    __gsignals__ = {
        'clean': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_PYOBJECT,
                  (), deferred_accumulator)
    }

    def __init__(self):
        super(PigmentController, self).__init__()
        self.sensitive = False
        self.widget = Widget()
        self.widget.visible = True
        self.widget_animated = implicit.AnimatedObject(self.widget)
        self.widget_animated.mode = implicit.REPLACE
        self.pending_decorator_deferreds = []

    def set_frontend(self, frontend):
        """
        Set the frontend for the controller.

        @param frontend: frontend
        @type frontend: L{elisa.core.components.frontend.Frontend}
        """
        self.frontend = frontend

    def set_path(self, path):
        """
        Set the path for the controller.

        @param path: path
        @type path:  C{str}
        """
        self.path = path

    def handle_input(self, input_manager, input_event):
        return False

    def has_focus(self):
        return self.widget.focus or self.widget.focus_child != None

    def prepare(self):
        """
        Called after L{widget} is added to the canvas. It is sensible to do
        aspect ratio dependent operations at that point.
        """
        pass

    def removed(self):
        """
        Called after L{widget} is removed from the canvas.
        """
        pass

    def clean(self):
        dfr = DeferredList(self.emit('clean') + 
                self.pending_decorator_deferreds)
        self.widget.clean()
        self.widget = None
        
        def call_parent(ignored):
            return super(PigmentController, self).clean()

        dfr.addCallback(call_parent)

        return dfr

