# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


from elisa.core.components.model import Model

class FileModel(Model):
    """
    A file model represents a file in a file system.

    @ivar name:  the name of the file
    @type name:  C{unicode}
    @ivar size:  a size in bytes
    @type size:  C{long}
    @ivar mtime: the time the file was accessed the last time (see note)
    @type mtime: C{long}
    @ivar atime: the time the file was modified the last time (see note)
    @type atime: C{long}

    Note: These values can be different from one system to the other. See the
    documentation of C{os.stat}.

    To represent a directory please use the DirectoryModel.
    """
    def __init__(self):
        super(FileModel, self).__init__()
        self.name = u''
        self.size = 0  # size in bytes
        self.mtime = 0 # time of last access
        self.atime = 0 # time of last modification
        self.media_type = None # a string describing the type of media this file
                               # is. Usually one of 'audio', 'video', 'image',
        self.thumbnail = None
        self.media_type = None

class DirectoryModel(FileModel):
    """
    A directory model is a certain type of 
    L{elisa.plugins.base.models.file.FileModel} with the difference that
    it can contain other FileModels. They get stored in C{files}.

    You can identify a DirectoryModel by doing the isinstance or check if it has
    the C{files} attribute.

    Another slight difference between a Directory and a FileModel is that the
    size of a directory is always 0 (platform independent).

    @ivar files:    files in the directory
    @type files:    C{list} of L{FileModel}s
    """
    def __init__(self):
        super(DirectoryModel, self).__init__()
        self.files = []
        self.media_type = u"directory"

    def get_size(self):
        return 0

    def _null(self, *args):
        pass

    # ignore setting of media_type and size
    size = property(fget=get_size, fset=_null)
