/*
 * $Id: sig_pci_bus.h,v 1.45 2009-01-27 17:44:20 potyra Exp $
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __SIG_PCI_BUS_H_INCLUDED
#define __SIG_PCI_BUS_H_INCLUDED

#include <inttypes.h>

#include "sig_boolean.h"
#include "sig_boolean_or.h"
#include "sig_gen.h"

enum sig_pci_bus_main_cycle {
	/* Must be the same as in sig_host_bus.h! */
	SIG_PCI_BUS_C0R = 0,
	SIG_PCI_BUS_C0W,
	SIG_PCI_BUS_C1R,
	SIG_PCI_BUS_C1W,
	SIG_PCI_BUS_IOR,
	SIG_PCI_BUS_IOW,
	SIG_PCI_BUS_MR,
	SIG_PCI_BUS_MW,
	SIG_PCI_BUS_INTA,
};

struct sig_pci_bus_main_funcs {
	int (*type_addr)(void *s, unsigned int type, uint32_t addr);
	int (*read_data)(void *s, unsigned int bs, uint32_t *valp);
	int (*write_data)(void *s, unsigned int bs, uint32_t val);

	int (*c0r)(void *s, uint32_t addr, unsigned int bs, uint32_t *valp);
	int (*c0w)(void *s, uint32_t addr, unsigned int bs, uint32_t val);

	int (*c1r)(void *s, uint32_t addr, unsigned int bs, uint32_t *valp);
	int (*c1w)(void *s, uint32_t addr, unsigned int bs, uint32_t val);

	int (*ior)(void *s, uint32_t port, unsigned int bs, uint32_t *valp);
	int (*iow)(void *s, uint32_t port, unsigned int bs, uint32_t val);

	int (*ior_info)(void *s, uint32_t port, unsigned int bs,
			int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
			void **csp);
	int (*iow_info)(void *s, uint32_t port, unsigned int bs,
			int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
			void **csp);

	void (*ior_info_flush)(void *s, uint32_t port, unsigned int bs);
	void (*iow_info_flush)(void *s, uint32_t port, unsigned int bs);

	int (*mr)(void *s, uint32_t addr, unsigned int bs, uint32_t *valp);
	int (*mw)(void *s, uint32_t addr, unsigned int bs, uint32_t val);

	int (*map)(void *s, unsigned long pa, unsigned int len,
			char **haddr_mr_p, char **haddr_mw_p);

	int (*unmap)(void *s, unsigned long pa, unsigned long len);

	int (*inta_addr)(void *s);
	int (*inta_data)(void *s, uint8_t *valp);
};

#define SIG_PCI_BUS_HASH_SIZE	(1 << 10)

struct sig_pci_bus_main {
	struct {
		void *s;
		const struct sig_pci_bus_main_funcs *f;
	} member[32];
	unsigned int member_count;

	struct sig_pci_bus_main_map {
		struct sig_pci_bus_main_map *lru_prev;
		struct sig_pci_bus_main_map *lru_next;

		struct sig_pci_bus_main_map *hash_prev;
		struct sig_pci_bus_main_map *hash_next;

		uint32_t addr;
		uint32_t *va_mr;
		uint32_t *va_mw;
	} map[256];
	struct sig_pci_bus_main_map *map_lru_first;
	struct sig_pci_bus_main_map *map_lru_last;
	struct sig_pci_bus_main_map *map_hash_first[SIG_PCI_BUS_HASH_SIZE];
	struct sig_pci_bus_main_map *map_hash_last[SIG_PCI_BUS_HASH_SIZE];

	struct sig_pci_bus_main_ior {
		struct sig_pci_bus_main_ior *lru_prev;
		struct sig_pci_bus_main_ior *lru_next;

		struct sig_pci_bus_main_ior *hash_prev;
		struct sig_pci_bus_main_ior *hash_next;

		uint32_t port;
		unsigned int bs;
		int (*f)(void *s,
				uint32_t port, unsigned int bs, uint32_t *valp);
		void *s;
	} ior[256];
	struct sig_pci_bus_main_ior *ior_lru_first;
	struct sig_pci_bus_main_ior *ior_lru_last;
	struct sig_pci_bus_main_ior *ior_hash_first[SIG_PCI_BUS_HASH_SIZE];
	struct sig_pci_bus_main_ior *ior_hash_last[SIG_PCI_BUS_HASH_SIZE];

	struct sig_pci_bus_main_iow {
		struct sig_pci_bus_main_iow *lru_prev;
		struct sig_pci_bus_main_iow *lru_next;

		struct sig_pci_bus_main_iow *hash_prev;
		struct sig_pci_bus_main_iow *hash_next;

		uint32_t port;
		unsigned int bs;
		int (*f)(void *s,
				uint32_t port, unsigned int bs, uint32_t val);
		void *s;
	} iow[256];
	struct sig_pci_bus_main_iow *iow_lru_first;
	struct sig_pci_bus_main_iow *iow_lru_last;
	struct sig_pci_bus_main_iow *iow_hash_first[SIG_PCI_BUS_HASH_SIZE];
	struct sig_pci_bus_main_iow *iow_hash_last[SIG_PCI_BUS_HASH_SIZE];
};

struct sig_pci_bus_main_merge {
	struct sig_pci_bus_main *s0;
	struct sig_pci_bus_main *s1;
};

extern int
sig_pci_bus_main_type_addr(struct sig_pci_bus_main *b, void *s,
		unsigned int type, uint32_t addr);
extern int
sig_pci_bus_main_read_data(struct sig_pci_bus_main *b, void *s,
		unsigned int bs, uint32_t *valp);
extern int
sig_pci_bus_main_write_data(struct sig_pci_bus_main *b, void *s,
		unsigned int bs, uint32_t val);

extern int
sig_pci_bus_main_c0r(struct sig_pci_bus_main *b, void *s,
		uint32_t addr, unsigned int bs, uint32_t *valp);
extern int
sig_pci_bus_main_c0w(struct sig_pci_bus_main *b, void *s,
		uint32_t addr, unsigned int bs, uint32_t val);

extern int
sig_pci_bus_c1r(struct sig_pci_bus_main *b, void *s,
		uint32_t addr, unsigned int bs, uint32_t *valp);
extern int
sig_pci_bus_c1w(struct sig_pci_bus_main *b, void *s,
		uint32_t addr, unsigned int bs, uint32_t val);

extern int
sig_pci_bus_ior(struct sig_pci_bus_main *b, void *s,
		uint32_t port, unsigned int bs, uint32_t *valp);
extern int
sig_pci_bus_iow(struct sig_pci_bus_main *b, void *s,
		uint32_t port, unsigned int bs, uint32_t val);

extern int
sig_pci_bus_ior_info(struct sig_pci_bus_main *b, void *s,
		uint32_t port, unsigned int bs,
		int (**cf)(void *, uint32_t, unsigned int, uint32_t *),
		void **cs);
extern int
sig_pci_bus_iow_info(struct sig_pci_bus_main *b, void *s,
		uint32_t port, unsigned int bs,
		int (**cf)(void *, uint32_t, unsigned int, uint32_t),
		void **cs);

extern void
sig_pci_bus_ior_info_flush(struct sig_pci_bus_main *b, void *s,
		uint32_t port, unsigned int bs);
extern void
sig_pci_bus_iow_info_flush(struct sig_pci_bus_main *b, void *s,
		uint32_t port, unsigned int bs);

extern int
sig_pci_bus_mr(struct sig_pci_bus_main *b, void *s,
		uint32_t addr, unsigned int bs, uint32_t *valp);
extern int
sig_pci_bus_mw(struct sig_pci_bus_main *b, void *s,
		uint32_t addr, unsigned int bs, uint32_t val);
extern int
sig_pci_bus_map(struct sig_pci_bus_main *b, void *s,
		unsigned long pa, unsigned int len,
		char **haddr_mr_p, char **haddr_mw_p);

extern int
sig_pci_bus_unmap(struct sig_pci_bus_main *b,
		void *s, unsigned long pa, unsigned long len);

extern int
sig_pci_bus_inta_addr(struct sig_pci_bus_main *b, void *s);
extern int
sig_pci_bus_inta_data(struct sig_pci_bus_main *b, void *s,
		unsigned char *valp);

extern void
sig_pci_bus_main_connect(struct sig_pci_bus_main *b,
		void *s, const struct sig_pci_bus_main_funcs *f);

extern struct sig_pci_bus_main *
sig_pci_bus_main_init(const char *name, int nr);

extern struct sig_pci_bus_main_merge *
sig_pci_bus_main_merge(
	struct sig_pci_bus_main *s0,
	struct sig_pci_bus_main *s1
);
extern void
sig_pci_bus_main_split(struct sig_pci_bus_main_merge *m);

extern void
sig_pci_bus_main_create(const char *name, int nr);
extern void
sig_pci_bus_main_destroy(const char *name, int nr);

struct sig_pci_bus_idsel_funcs {
	int (*c0r)(void *s, uint32_t addr, unsigned int bs, uint32_t *valp);
	int (*c0w)(void *s, uint32_t addr, unsigned int bs, uint32_t val);
};

struct sig_pci_bus_idsel {
	struct {
		void *s;
		const struct sig_pci_bus_idsel_funcs *f;
	} member[10];
	unsigned int member_count;
};

struct sig_pci_bus_idsel_merge {
	struct sig_pci_bus_idsel *s0;
	struct sig_pci_bus_idsel *s1;
};

extern int
sig_pci_bus_idsel_c0r(struct sig_pci_bus_idsel *b, void *s,
		uint32_t addr, unsigned int bs, uint32_t *valp);
extern int
sig_pci_bus_idsel_c0w(struct sig_pci_bus_idsel *b, void *s,
		uint32_t addr, unsigned int bs, uint32_t val);

extern void
sig_pci_bus_idsel_connect(struct sig_pci_bus_idsel *b,
		void *s, const struct sig_pci_bus_idsel_funcs *f);

extern struct sig_pci_bus_idsel *
sig_pci_bus_idsel_init(const char *name, int nr);

extern struct sig_pci_bus_idsel_merge *
sig_pci_bus_idsel_merge(
	struct sig_pci_bus_idsel *s0,
	struct sig_pci_bus_idsel *s1
);
extern void
sig_pci_bus_idsel_split(struct sig_pci_bus_idsel_merge *m);

extern void
sig_pci_bus_idsel_create(const char *name, int nr);
extern void
sig_pci_bus_idsel_destroy(const char *name, int nr);

struct sig_pci_bus {
	enum sig_gen_type type;
	struct sig_boolean *p5V;
	struct sig_boolean *p12V;
	struct sig_boolean *m12V;
	struct sig_boolean *n_reset;
	struct sig_pci_bus_idsel *idsel;
	struct sig_pci_bus_main *main;
	struct sig_boolean_or *intA;
	struct sig_boolean_or *intB;
	struct sig_boolean_or *intC;
	struct sig_boolean_or *intD;
};

struct sig_pci_bus *
sig_pci_bus_init(const char *name, unsigned int nr);

extern void
sig_pci_bus_create(const char *name, unsigned int nr);
extern void
sig_pci_bus_destroy(const char *name, unsigned int nr);

#endif /* __SIG_PCI_BUS_H_INCLUDED */
