/*
 * $Id: simnodes.c,v 1.5 2009-01-27 17:06:42 potyra Exp $
 * 
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <fcntl.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue-main.h"
#include "simnodes.h"

static char def_user[32];
static char def_host[256];

struct simnode *simnode_first;
struct simnode *simnode_last;

static void
simnodes_entry(
	const char *name,
	const char *user, 
	const char *host, 
	const char *path
)
{
	struct simnode *node;

	for (node = simnode_first; ; node = node->next) {
		if (! node) {
			node = malloc(sizeof(*node));
			assert(node);

			if (name) {
				node->name = strdup(name);
				assert(node->name);
			} else {
				node->name = NULL;
			}

			node->user = strdup(user);
			assert(node->user);
			node->host = strdup(host);
			assert(node->host);
			node->path = strdup(path);
			assert(node->path);

			node->prev = simnode_last;
			node->next = NULL;
			if (node->prev) {
				node->prev->next = node;
			} else {
				simnode_first = node;
			}
			simnode_last = node;
			break;
		}
		if (strcmp(node->user, user) == 0
		 && strcmp(node->host, host) == 0
		 && strcmp(node->path, path) == 0) {
			if (node->name == NULL) {
				node->name = strdup(name);
			} else {
				fprintf(stderr, "%s: simulation.nodes: %s@%s:%s: Used more than once.\n",
						"faum-node-pc",  /* FIXME */
						user, host, path);
				exit(1);
			}
			break;
		}
	}
}

static void
simnodes_default(void)
{
	simnodes_entry(NULL, def_user, def_host, "node.def");
}

static void
simnodes_add(const char *name, char *user_host_path)
{
	char *user;
	char *host;
	char *path;

	if (strchr(user_host_path, ':')) {
		path = strchr(user_host_path, ':');
		*path++ = '\0';

		if (strchr(user_host_path, '@')) {
			user = user_host_path;
			host = strchr(user_host_path, '@');
			*host++ = '\0';
		} else {
			user = def_user;
			host = user_host_path;
		}
	} else {
		user = def_user;
		host = def_host;
		path = user_host_path;
	}

	simnodes_entry(name, user, host, path);
}

void
simnodes_read(void)
{
	struct passwd *pw;
	int fd;
	int ret;

	/*
	 * Get Default User
	 */
	pw = getpwuid(getuid());
	assert(pw);

	strcpy(def_user, pw->pw_name);

	/*
	 * Get Default Host
	 */
	ret = gethostname(def_host, sizeof(def_host));
	assert(0 <= ret);

	/*
	 * Read List of Nodes
	 */
	simnodes_default();

	fd = open("simulation.nodes", O_RDONLY);
	if (0 <= fd) {
		char buffer[16*1024];
		char name[32];
		char user_host_path[32 + 1 + 64 + 1 + 1024];
		unsigned int x;
		const char *p;

		ret = read(fd, buffer, sizeof(buffer) - 1);
		assert(0 <= ret);
		buffer[ret] = '\0';

		for (p = buffer; *p; ) {
			switch (*p) {
			case '\n':
				/* Skip return. */
				p++;
				break;
			case '\t':
			case ' ':
				/* Skip white space. */
				p++;
				break;
			case '#':
				/* Skip comment. */
				while (*p != '\0'
				    && *p != '\n') {
					p++;
				}
				break;
			default:
				/*
				 * Entry Found
				 */
				/* Read node name. */
				x = 0;
				while (*p != '\0'
				    && *p != '\t'
				    && *p != '\n'
				    && *p != '\r'
				    && *p != ' '
				    && *p != '=') {
					name[x++] = *p++;
				}
				name[x] = '\0';

				/* Skip white space. */
				while (*p == '\t'
				    || *p == ' ') {
					p++;
				}

				/* Skip '=' (if present). */
				if (*p == '=') {
					p++;
				}

				/* Skip white space. */
				while (*p == '\t'
				    || *p == ' ') {
					p++;
				}

				/* Read user@host:path. */
				x = 0;
				while (*p != '\0'
				    && *p != '\t'
				    && *p != '\n'
				    && *p != '\r'
				    && *p != ' ') {
					user_host_path[x++] = *p++;
				}
				user_host_path[x] = '\0';

				simnodes_add(name, user_host_path);
			}
		}

		ret = close(fd);
		assert(0 <= ret);
	}

	if (! simnode_first->name) {
		simnode_first->name = strdup("<default>");
		assert(simnode_first->name);
	}
}
