-- $Id: expect.vhdl,v 1.1 2009-01-30 10:31:44 potyra Exp $
-- vim:tabstop=8:shiftwidth=8:textwidth=72:encoding=utf8:

-- Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
--
-- This program is free software. You can redistribute it and/or modify it
-- under the terms of the GNU General Public License, either version 2 of
-- the License, or (at your option) any later version. See COPYING.

library expect;
use expect.types.all;

PACKAGE procedures IS

-- send the string send to the keyboard, using timeout as delay between
-- each key getting pressed.
PROCEDURE send_keyboard(
	SIGNAL key: OUT boolean_array(0 to 128);
	CONSTANT kbd_type: IN  integer;
	CONSTANT send: IN string;
	CONSTANT timeout: IN time
);


-- detect mouse (local procedure)
-- FIXME potyra: why does it need to be declared in the
-- package, and cannot just get declared in the package body
-- FIXME potyra: should acutally be local to control_mouse!
PROCEDURE detect_mouse(
	SIGNAL mouse_match: IN boolean;
	SIGNAL mouse_event: IN  boolean;
	SIGNAL dx: OUT integer;
	SIGNAL dy: OUT integer
);

-- move the mouse to the coordinates (target_x/target_y).
-- mouse_match: state signal of mouse_matcher
-- mouse_event: explicit event signal of mouse_matcher
-- mouse_x, mouse_y: x/y coordinate of mouse_matcher
-- dx, dy: (x/y) delta inputs of mouse.
PROCEDURE control_mouse(
	VARIABLE target_x: IN  integer;
	VARIABLE target_y: IN  integer;
	SIGNAL mouse_match: IN  boolean;
	SIGNAL mouse_event: IN  boolean;
	SIGNAL mouse_x: IN  integer;
	SIGNAL mouse_y: IN  integer;
	SIGNAL dx: OUT integer;
	SIGNAL dy: OUT integer
);


-- send a string to a str_comm signal 
-- implemented in glue-vhdl.
PROCEDURE send_string(
	SIGNAL dest : OUT cstring;
	CONSTANT s : IN string
);
ATTRIBUTE foreign of send_string : PROCEDURE IS "send_string";

-- terminate the simulation
-- implemented in glue-vhdl
PROCEDURE terminate;
ATTRIBUTE foreign of terminate : PROCEDURE IS "terminate";

-- toggle debugging on/off
PROCEDURE debug;
ATTRIBUTE foreign of debug : PROCEDURE IS "debug";

-- lazy connect a boolean signal to a the component with pathname comp
-- to the port name comp_port.
-- implemented in glue-vhdl.
PROCEDURE shortcut_bool_out(
	SIGNAL s : OUT boolean;
	CONSTANT comp : IN string;
	CONSTANT comp_port : IN string
);
ATTRIBUTE foreign of shortcut_bool_out : PROCEDURE IS "shortcut_bool_out";

-- lazy connect an integer signal to a the component with pathname comp
-- to the port name comp_port.
-- implemented in glue-vhdl.
PROCEDURE shortcut_int_out(
	SIGNAL s : OUT integer;
	CONSTANT comp : IN string;
	CONSTANT comp_port : IN string
);
ATTRIBUTE foreign of shortcut_int_out : PROCEDURE IS "shortcut_int_out";

END procedures;

PACKAGE BODY procedures IS

-- *************************************
-- ***          send_keyboard        ***
-- *************************************
PROCEDURE send_keyboard(
	SIGNAL key: OUT boolean_array(0 to 128);
	CONSTANT kbd_type: IN  integer;
	CONSTANT send: IN string;
	CONSTANT timeout: IN time
) IS
	type escapes is (NOT_ESCAPED, ESC_DIGIT_1, ESC_DIGIT_2);
	VARIABLE esc : escapes := NOT_ESCAPED;
	VARIABLE digit1 : character;
BEGIN
	-- us keyboard
	assert kbd_type = 0 report "only kbd type 0 (us) supported" 
		severity failure;

	FOR i IN send'range LOOP
		case esc is
		when NOT_ESCAPED =>
			case send(i) is
			when '\' =>
				esc := ESC_DIGIT_1;

			when 'a' =>
				key(30) <= true;
				WAIT FOR timeout;
				key(30) <= false;
				WAIT FOR timeout;
			when 'b' =>
				key(48) <= true;
				WAIT FOR timeout;
				key(48) <= false;
				WAIT FOR timeout;
			when 'c' =>
				key(46) <= true;
				WAIT FOR timeout;
				key(46) <= false;
				WAIT FOR timeout;
			when 'd' =>
				key(32) <= true;
				WAIT FOR timeout;
				key(32) <= false;
				WAIT FOR timeout;
			when 'e' =>
				key(18) <= true;
				WAIT FOR timeout;
				key(18) <= false;
				WAIT FOR timeout;
			when 'f' =>
				key(33) <= true;
				WAIT FOR timeout;
				key(33) <= false;
				WAIT FOR timeout;
			when 'g' =>
				key(34) <= true;
				WAIT FOR timeout;
				key(34) <= false;
				WAIT FOR timeout;
			when 'h' =>
				key(35) <= true;
				WAIT FOR timeout;
				key(35) <= false;
				WAIT FOR timeout;
			when 'i' =>
				key(23) <= true;
				WAIT FOR timeout;
				key(23) <= false;
				WAIT FOR timeout;
			when 'j' =>
				key(36) <= true;
				WAIT FOR timeout;
				key(36) <= false;
				WAIT FOR timeout;
			when 'k' =>
				key(37) <= true;
				WAIT FOR timeout;
				key(37) <= false;
				WAIT FOR timeout;
			when 'l' =>
				key(38) <= true;
				WAIT FOR timeout;
				key(38) <= false;
				WAIT FOR timeout;
			when 'm' =>
				key(50) <= true;
				WAIT FOR timeout;
				key(50) <= false;
				WAIT FOR timeout;
			when 'n' =>
				key(49) <= true;
				WAIT FOR timeout;
				key(49) <= false;
				WAIT FOR timeout;
			when 'o' =>
				key(24) <= true;
				WAIT FOR timeout;
				key(24) <= false;
				WAIT FOR timeout;
			when 'p' =>
				key(25) <= true;
				WAIT FOR timeout;
				key(25) <= false;
				WAIT FOR timeout;
			when 'q' =>
				key(16) <= true;
				WAIT FOR timeout;
				key(16) <= false;
				WAIT FOR timeout;
			when 'r' =>
				key(19) <= true;
				WAIT FOR timeout;
				key(19) <= false;
				WAIT FOR timeout;
			when 's' =>
				key(31) <= true;
				WAIT FOR timeout;
				key(31) <= false;
				WAIT FOR timeout;
			when 't' =>
				key(20) <= true;
				WAIT FOR timeout;
				key(20) <= false;
				WAIT FOR timeout;
			when 'u' =>
				key(22) <= true;
				WAIT FOR timeout;
				key(22) <= false;
				WAIT FOR timeout;
			when 'v' =>
				key(47) <= true;
				WAIT FOR timeout;
				key(47) <= false;
				WAIT FOR timeout;
			when 'w' =>
				key(17) <= true;
				WAIT FOR timeout;
				key(17) <= false;
				WAIT FOR timeout;
			when 'x' =>
				key(45) <= true;
				WAIT FOR timeout;
				key(45) <= false;
				WAIT FOR timeout;
			when 'y' =>
				key(21) <= true;
				WAIT FOR timeout;
				key(21) <= false;
				WAIT FOR timeout;
			when 'z' =>
				key(44) <= true;
				WAIT FOR timeout;
				key(44) <= false;
				WAIT FOR timeout;

			when 'A' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(30) <= true;
				WAIT FOR timeout;
				key(30) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'B' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(48) <= true;
				WAIT FOR timeout;
				key(48) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'C' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(46) <= true;
				WAIT FOR timeout;
				key(46) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'D' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(32) <= true;
				WAIT FOR timeout;
				key(32) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'E' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(18) <= true;
				WAIT FOR timeout;
				key(18) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'F' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(33) <= true;
				WAIT FOR timeout;
				key(33) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'G' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(34) <= true;
				WAIT FOR timeout;
				key(34) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'H' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(35) <= true;
				WAIT FOR timeout;
				key(35) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'I' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(23) <= true;
				WAIT FOR timeout;
				key(23) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'J' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(36) <= true;
				WAIT FOR timeout;
				key(36) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'K' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(37) <= true;
				WAIT FOR timeout;
				key(37) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'L' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(38) <= true;
				WAIT FOR timeout;
				key(38) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'M' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(50) <= true;
				WAIT FOR timeout;
				key(50) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'N' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(49) <= true;
				WAIT FOR timeout;
				key(49) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'O' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(24) <= true;
				WAIT FOR timeout;
				key(24) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'P' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(25) <= true;
				WAIT FOR timeout;
				key(25) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'Q' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(16) <= true;
				WAIT FOR timeout;
				key(16) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'R' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(19) <= true;
				WAIT FOR timeout;
				key(19) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'S' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(31) <= true;
				WAIT FOR timeout;
				key(31) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'T' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(20) <= true;
				WAIT FOR timeout;
				key(20) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'U' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(22) <= true;
				WAIT FOR timeout;
				key(22) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'V' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(47) <= true;
				WAIT FOR timeout;
				key(47) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'W' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(17) <= true;
				WAIT FOR timeout;
				key(17) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'X' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(45) <= true;
				WAIT FOR timeout;
				key(45) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'Y' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(21) <= true;
				WAIT FOR timeout;
				key(21) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when 'Z' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(44) <= true;
				WAIT FOR timeout;
				key(44) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;


			when '1' =>
				key(2) <= true;
				WAIT FOR timeout;
				key(2) <= false;
				WAIT FOR timeout;
			when '2' =>
				key(3) <= true;
				WAIT FOR timeout;
				key(3) <= false;
				WAIT FOR timeout;
			when '3' =>
				key(4) <= true;
				WAIT FOR timeout;
				key(4) <= false;
				WAIT FOR timeout;
			when '4' =>
				key(5) <= true;
				WAIT FOR timeout;
				key(5) <= false;
				WAIT FOR timeout;
			when '5' =>
				key(6) <= true;
				WAIT FOR timeout;
				key(6) <= false;
				WAIT FOR timeout;
			when '6' =>
				key(7) <= true;
				WAIT FOR timeout;
				key(7) <= false;
				WAIT FOR timeout;
			when '7' =>
				key(8) <= true;
				WAIT FOR timeout;
				key(8) <= false;
				WAIT FOR timeout;
			when '8' =>
				key(9) <= true;
				WAIT FOR timeout;
				key(9) <= false;
				WAIT FOR timeout;
			when '9' =>
				key(10) <= true;
				WAIT FOR timeout;
				key(10) <= false;
				WAIT FOR timeout;
			when '0' =>
				key(11) <= true;
				WAIT FOR timeout;
				key(11) <= false;
				WAIT FOR timeout;
			when '-' =>
				key(12) <= true;
				WAIT FOR timeout;
				key(12) <= false;
				WAIT FOR timeout;
			when '=' =>
				key(13) <= true;
				WAIT FOR timeout;
				key(13) <= false;
				WAIT FOR timeout;

			when '!' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(2) <= true;
				WAIT FOR timeout;
				key(2) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '@' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(3) <= true;
				WAIT FOR timeout;
				key(3) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '#' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(4) <= true;
				WAIT FOR timeout;
				key(4) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '$' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(5) <= true;
				WAIT FOR timeout;
				key(5) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '%' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(6) <= true;
				WAIT FOR timeout;
				key(6) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '^' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(7) <= true;
				WAIT FOR timeout;
				key(7) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '&' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(8) <= true;
				WAIT FOR timeout;
				key(8) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '*' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(9) <= true;
				WAIT FOR timeout;
				key(9) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '(' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(10) <= true;
				WAIT FOR timeout;
				key(10) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when ')' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(11) <= true;
				WAIT FOR timeout;
				key(11) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '_' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(12) <= true;
				WAIT FOR timeout;
				key(12) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '+' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(13) <= true;
				WAIT FOR timeout;
				key(13) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;

			when '[' =>
				key(26) <= true;
				WAIT FOR timeout;
				key(26) <= false;
				WAIT FOR timeout;
			when ']' =>
				key(27) <= true;
				WAIT FOR timeout;
				key(27) <= false;
				WAIT FOR timeout;
			when ';' =>
				key(39) <= true;
				WAIT FOR timeout;
				key(39) <= false;
				WAIT FOR timeout;
			when ''' =>
				key(40) <= true;
				WAIT FOR timeout;
				key(40) <= false;
				WAIT FOR timeout;
			when ',' =>
				key(51) <= true;
				WAIT FOR timeout;
				key(51) <= false;
				WAIT FOR timeout;
			when '.' =>
				key(52) <= true;
				WAIT FOR timeout;
				key(52) <= false;
				WAIT FOR timeout;
			when '/' =>
				key(53) <= true;
				WAIT FOR timeout;
				key(53) <= false;
				WAIT FOR timeout;
			when '`' =>
				key(41) <= true;
				WAIT FOR timeout;
				key(41) <= false;
				WAIT FOR timeout;

			when '{' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(26) <= true;
				WAIT FOR timeout;
				key(26) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '}' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(27) <= true;
				WAIT FOR timeout;
				key(27) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when ':' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(39) <= true;
				WAIT FOR timeout;
				key(39) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '"' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(40) <= true;
				WAIT FOR timeout;
				key(40) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '|' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(43) <= true;
				WAIT FOR timeout;
				key(43) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '<' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(51) <= true;
				WAIT FOR timeout;
				key(51) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '>' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(52) <= true;
				WAIT FOR timeout;
				key(52) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '?' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(53) <= true;
				WAIT FOR timeout;
				key(53) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;
			when '~' =>
				key(42) <= true;
				WAIT FOR timeout;
				key(41) <= true;
				WAIT FOR timeout;
				key(41) <= false;
				WAIT FOR timeout;
				key(42) <= false;
				WAIT FOR timeout;

			when ' ' =>
				key(57) <= true;
				WAIT FOR timeout;
				key(57) <= false;
				WAIT FOR timeout;

			when others =>
				assert false report "unknown character"
					severity failure;
			end case;

		when ESC_DIGIT_1 =>
			case send(i) is
			when '\' =>	-- use "\\" to escape backslash
				key(43) <= true;
				WAIT FOR timeout;
				key(43) <= false;
				WAIT FOR timeout;
				esc := NOT_ESCAPED;

			when others =>
				digit1 := send(i);
				esc := ESC_DIGIT_2;
			end case;

		when ESC_DIGIT_2 =>
			case digit1 is
			when '0' =>
				case send(i) is
				when '1' =>	-- home
					key(102) <= true;
					WAIT FOR timeout;
					key(102) <= false;
					WAIT FOR timeout;

				when '3' =>	-- enter
					key(96) <= true;
					WAIT FOR timeout;
					key(96) <= false;
					WAIT FOR timeout;

				when '4' =>	-- end
					key(107) <= true;
					WAIT FOR timeout;
					key(107) <= false;
					WAIT FOR timeout;

				when '8' =>	-- backspace
					key(14) <= true;
					WAIT FOR timeout;
					key(14) <= false;
					WAIT FOR timeout;

				when '9' =>	-- tab
					key(15) <= true;
					WAIT FOR timeout;
					key(15) <= false;
					WAIT FOR timeout;

				when 'b' =>	-- page up
					key(104) <= true;
					WAIT FOR timeout;
					key(104) <= false;
					WAIT FOR timeout;

				when 'c' =>	-- page down
					key(109) <= true;
					WAIT FOR timeout;
					key(109) <= false;
					WAIT FOR timeout;

				when 'd' =>	-- carriage return
					key(28) <= true;
					WAIT FOR timeout;
					key(28) <= false;
					WAIT FOR timeout;

				when 'f' =>	-- ESC escape
					key(1) <= true;
					WAIT FOR timeout;
					key(1) <= false;
					WAIT FOR timeout;


				when others => 
					assert false report
						"unknown escape"
						severity failure;
				end case;

			when '1' =>
				case send(i) is
				when '0' =>	-- F1
					key(59) <= true;
					WAIT FOR timeout;
					key(59) <= false;
					WAIT FOR timeout;

				when '1' =>	-- F2
					key(60) <= true;
					WAIT FOR timeout;
					key(60) <= false;
					WAIT FOR timeout;

				when '2' => 	-- F3
					key(61) <= true;
					WAIT FOR timeout;
					key(61) <= false;
					WAIT FOR timeout;

				when '3' =>	-- F4
					key(62) <= true;
					WAIT FOR timeout;
					key(62) <= false;
					WAIT FOR timeout;

				when '4' =>	-- F5
					key(63) <= true;
					WAIT FOR timeout;
					key(63) <= false;
					WAIT FOR timeout;

				when '5' =>	-- F6
					key(64) <= true;
					WAIT FOR timeout;
					key(64) <= false;
					WAIT FOR timeout;

				when '6' =>	-- F7
					key(65) <= true;
					WAIT FOR timeout;
					key(65) <= false;
					WAIT FOR timeout;

				when '7' =>	-- F8
					key(66) <= true;
					WAIT FOR timeout;
					key(66) <= false;
					WAIT FOR timeout;

				when '8' =>	-- F9
					key(67) <= true;
					WAIT FOR timeout;
					key(67) <= false;
					WAIT FOR timeout;

				when '9' =>	-- F10
					key(68) <= true;
					WAIT FOR timeout;
					key(68) <= false;
					WAIT FOR timeout;

				when 'a' =>	-- F11
					key(87) <= true;
					WAIT FOR timeout;
					key(87) <= false;
					WAIT FOR timeout;

				when 'b' =>	-- F12
					key(88) <= true;
					WAIT FOR timeout;
					key(8) <= false;
					WAIT FOR timeout;

				when 'c' =>	-- left arrow
					key(105) <= true;
					WAIT FOR timeout;
					key(105) <= false;
					WAIT FOR timeout;

				when 'd' =>	-- right arrow
					key(106) <= true;
					WAIT FOR timeout;
					key(106) <= false;
					WAIT FOR timeout;

				when 'e' =>	-- up arrow
					key(103) <= true;
					WAIT FOR timeout;
					key(103) <= false;
					WAIT FOR timeout;

				when 'f' =>	-- down arrow
					key(108) <= true;
					WAIT FOR timeout;
					key(108) <= false;
					WAIT FOR timeout;

				when others => 
					assert false report
						"unknown escape"
						severity failure;
				end case;

			when '2' =>
				case send(i) is
				when '2' =>	-- '"'
					key(42) <= true;
					WAIT FOR timeout;
					key(40) <= true;
					WAIT FOR timeout;
					key(40) <= false;
					WAIT FOR timeout;
					key(42) <= false;
					WAIT FOR timeout;

				when others =>
					assert false report
						"unknown escape"
						severity failure;
				end case;

			when '8' =>
				case send(i) is
				when '0' =>	-- up arrow (keypad)
					key(72) <= true;
					WAIT FOR timeout;
					key(72) <= false;
					WAIT FOR timeout;

				when '1' =>	-- right arrow (keypad)
					key(77) <= true;
					WAIT FOR timeout;
					key(77) <= false;
					WAIT FOR timeout;

				when '2' =>	-- down arrow (keypad)
					key(80) <= true;
					WAIT FOR timeout;
					key(80) <= false;
					WAIT FOR timeout;

				when '3' =>	-- left arrow (keypad)
					key(75) <= true;
					WAIT FOR timeout;
					key(75) <= false;
					WAIT FOR timeout;

				when others =>
					assert false report
						"unknown escape"
						severity failure;
				end case;

			when others =>
				assert false report 
					"unknown escape sequence"
					severity failure;
			end case;
			esc := NOT_ESCAPED;
			null;

		end case;
	END LOOP;

	-- sanity check
	case esc is
	when NOT_ESCAPED => 
		null;
	when others =>
		assert false report send severity note;
		assert false report "broken escape sequence" severity
			failure;
	end case;
END send_keyboard;

-- *************************************
-- ***          detect_mouse         ***
-- *************************************
PROCEDURE detect_mouse(
	SIGNAL mouse_match: IN boolean;
	SIGNAL mouse_event: IN  boolean;
	SIGNAL dx: OUT integer;
	SIGNAL dy: OUT integer
)
IS
BEGIN
	-- First: try tiny step.
	IF not mouse_match THEN
		dx <= 1;
		WAIT ON mouse_event FOR 500 ms;
	END IF;
	IF not mouse_match THEN
		dy <= -1;
		WAIT ON mouse_event FOR 500 ms;
	END IF;
	IF not mouse_match THEN
		dx <= -1;
		WAIT ON mouse_event FOR 500 ms;
	END IF;
	IF not mouse_match THEN
		dy <= 1;
		WAIT ON mouse_event FOR 500 ms;
	END IF;

	-- Second: try big step.
	IF not mouse_match THEN
		dx <= 25;
		WAIT ON mouse_event FOR 500 ms;
	END IF;
	IF not mouse_match THEN
		dy <= -25;
		WAIT ON mouse_event FOR 500 ms;
	END IF;
	IF not mouse_match THEN
		dx <= -25;
		WAIT ON mouse_event FOR 500 ms;
	END IF;
	IF not mouse_match THEN
		dy <= 25;
		WAIT ON mouse_event FOR 500 ms;
	END IF;
END detect_mouse;

-- *************************************
-- ***          control_mouse        ***
-- *************************************
PROCEDURE control_mouse (
	VARIABLE target_x: IN  integer;
	VARIABLE target_y: IN  integer;
	SIGNAL mouse_match: IN  boolean;
	SIGNAL mouse_event: IN  boolean;
	SIGNAL mouse_x: IN  integer;
	SIGNAL mouse_y: IN  integer;
	SIGNAL dx: OUT integer;
	SIGNAL dy: OUT integer
)
IS
	VARIABLE delta_x : integer;
	VARIABLE delta_y : integer;
BEGIN
	WHILE not mouse_match LOOP
		detect_mouse(mouse_match,
				mouse_event,
				dx, dy);
		WAIT FOR 1000 ms;
	END LOOP;

	WHILE ((mouse_x - target_x) * (mouse_x - target_x) > 16)
	   OR ((mouse_y - target_y) * (mouse_y - target_y) > 16) LOOP
		delta_x := target_x - mouse_x;
		delta_y := mouse_y - target_y;

		-- Calculate delta_x.
		IF delta_x <= -32 THEN
			delta_x := -8;
		ELSIF delta_x <= -4 THEN
			delta_x := delta_x / 4;
		ELSIF delta_x <= -1 THEN
			delta_x := -1;

		ELSIF delta_x >= 32 THEN
			delta_x := 8;
		ELSIF delta_x >= 4 THEN
			delta_x := delta_x / 4;
		ELSIF delta_x >= 1 THEN
			delta_x := 1;
		END IF;

		-- Calculate delta_y.
		IF delta_y <= -32 THEN
			delta_y := -8;
		ELSIF delta_y <= -4 THEN
			delta_y := delta_y / 4;
		ELSIF delta_y <= -1 THEN
			delta_y := -1;

		ELSIF delta_y >= 32 THEN
			delta_y := 8;
		ELSIF delta_y >= 4 THEN
			delta_y := delta_y / 4;
		ELSIF delta_y >= 1 THEN
			delta_y := 1;
		END IF;

		-- Move mouse.
		dx <= delta_x;
		dy <= delta_y;

		-- Wait for mouse to move...
		WAIT ON mouse_event FOR 500 ms;

		WHILE not mouse_match LOOP
			detect_mouse(mouse_match,
				mouse_event, 
				dx, dy);
			WAIT FOR 1000 ms;
		END LOOP;
	END LOOP;
END control_mouse;


END procedures;
