/* $Id: chip_harris_82c55a.c,v 1.36 2009-01-28 12:59:17 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */
 
#include "chip_harris_82c55a.h"
#include <inttypes.h>

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue-log.h"
#include "glue-shm.h"
 
#define CHIPNAME "chip_harris_82c55a"

/* Debugging stuff */
#define WARNINGS 0x0
#define DEBUGMASK 0x0

/* Add the following values together to select which debug messages you want */
#define DEBUG_IO		0x01
#define DEBUG_OUTB		0x02
#define DEBUG_INB		0x04
#define DEBUG_MODESET	0x08
#define DEBUG_SET		0x10
#define DEBUG_OTHER		0x20
 
/* Debugging */
#if (DEBUGMASK > 0)
#define TRACE(lvl, fmt, arg...) \
	if ((lvl & DEBUGMASK)) { \
		char tmplog[2]; \
		tmplog[0] = '-'; \
		tmplog[1] = 0; \
		faum_log(FAUM_LOG_DEBUG, "harris_82c55a", \
		 tmplog, "[ %04x  ] " fmt , \
		 lvl, arg); \
	}
#else
#define TRACE(lvl, fmt, arg...) while(0) { } /* ; */
#endif /* DEBUGMASK > 0 */

struct css {
	/*
	 * Config
	 */
	uint8_t portA, portB, portC;
	uint8_t control;

	/*
	 * Signals
	 */
	struct sig_boolean *port_power;
	struct sig_boolean *port_n_reset;
	struct sig_cs *port_cs;
	struct sig_std_logic *p[24];

	/*
	 * State
	 */
	unsigned int nr;
	unsigned int state_power;
	uint8_t portA_state;
	uint8_t portB_state;
	uint8_t portC_state;
};

static void
chip_harris_82c55a_port_write(void *_css, uint8_t val, unsigned int pr)
{
	/*
	 * pr selects the port register, valid values are
	 * 0 -> Port A
	 * 1 -> Port B
	 * 2 -> Port C
	 */
	assert(pr < 3);

	struct css *css = (struct css *) _css;
	unsigned int i;

	if ((css->control & 0x60) == 0) {
		/* mode 0 logic */
		if (pr == 0) {
			/* port A */
			css->portA = val;
			if ((css->control & 0x10) == 0) {
				/* port A is configured for output -> set value & update pins */
				TRACE(DEBUG_OUTB,"PORT WRITE: port register: %d, val: %02x\n",pr , val);
				for (i=0; i<8; i++) sig_std_logic_set(css->p[i], css, ((css->portA >> i) & 0x1 ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0));
			} else {
				/* port A is configured for input */
				TRACE(DEBUG_OUTB,"PORT WRITE: ignoring write (%02x) to port register %d\n",val , pr);
			}
		} else if (pr == 1) {
			/* port B */
			css->portB = val;
			if ((css->control & 0x2) == 0) {
				/* port B is configured for output -> set value & update pins */
				TRACE(DEBUG_OUTB,"PORT WRITE: port register: %d, val: %02x\n",pr , val);
				for (i=0; i<8; i++) sig_std_logic_set(css->p[i+8], css, ((css->portB >> i) & 0x1 ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0));
			} else {
				/* port B is configured for input */
				TRACE(DEBUG_OUTB,"PORT WRITE: ignoring write (%02x) to port register %d\n",val , pr);
			}
		} else {
			/*
			 * port C can is devided into 2 4 bit ports
			 * each can be configured for input or output
			 */
			css->portC = val;
			/* high order 4 bits of port C */
			if ((css->control & 0x8) == 0) {
				/* port C_h is configured for output -> set value & update pins */
				TRACE(DEBUG_OUTB,"PORT WRITE: port register: %d high, val: %02x\n",pr , (val&0xf0));
				for (i=4; i<8; i++) sig_std_logic_set(css->p[i+16], css, ((css->portC >> i) & 0x1 ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0));
			} else {
				/* port C_h is configured for input */
				TRACE(DEBUG_OUTB,"PORT WRITE: ignoring write (%02x) to port register %d high\n",(val&0xf0) , pr);
			}
			/* low order 4 bits of port C */
			if ((css->control & 0x1) == 0) {
				/* port C_l is configured for output -> set value & update pins */
				TRACE(DEBUG_OUTB,"PORT WRITE: port register: %d low, val: %02x\n",pr , (val&0x0f));
				for (i=0; i<4; i++) sig_std_logic_set(css->p[i+16], css, ((css->portC >> i) & 0x1 ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0));
			} else {
				/* port C_l is configured for input */
				TRACE(DEBUG_OUTB,"PORT WRITE: ignoring write (%02x) to port register %d low\n",(val&0x0f) , pr);
			}	
		}
	} else {
		/* only mode 0 is currently supported */
		assert(0);
	}
}

static void
chip_harris_82c55a_mode_set(void *_css, uint8_t val)
{
	uint8_t mode;
	unsigned int i;
	struct css *css = (struct css *) _css;

	/* group a (port a and upper port c) */
	mode = val & 0x78;
	if ((mode & 0x60) == 0) {
		/* mode 0 */

		/* upper port c */
		if ((mode & 0x08) == 0x08) {
			/* input */
			for (i=20; i<24; i++) sig_std_logic_set(css->p[i], css, SIG_STD_LOGIC_Z);
		} else {
			/* output */
			css->portC &= ~0xf0;
			for (i=20; i<24; i++) sig_std_logic_set(css->p[i], css, SIG_STD_LOGIC_0);
		}

		/* port a */
		if ((mode & 0x10) == 0x10) {
			/* input */
			for (i=0; i<8; i++) sig_std_logic_set(css->p[i], css, SIG_STD_LOGIC_Z);
		} else {
			/* output */
			css->portA = 0;
			for (i=0; i<8; i++) sig_std_logic_set(css->p[i], css, SIG_STD_LOGIC_0);
		}
	} else {
		/* mode 1 or mode 2 */
		/* FIXME: not supported yet */
		assert(0);
	}

	/* group b (port b and lower port c) */
	mode = val & 0x07;
	if ((mode & 0x04) == 0) {
		/* mode 0 */

		/* lower port c */
		if ((mode & 0x01) == 1) {
			/* input */
			for (i=16; i<20; i++) sig_std_logic_set(css->p[i], css, SIG_STD_LOGIC_Z);
		} else {
			/* output */
			css->portC &= ~0x0f;
			for (i=16; i<20; i++) sig_std_logic_set(css->p[i], css, SIG_STD_LOGIC_0);
		}

		/* port b */
		if ((mode & 0x02) == 2) {
			/* input */
			for (i=8; i<16; i++) sig_std_logic_set(css->p[i], css, SIG_STD_LOGIC_Z);
		} else {
			/* output */
			css->portB = 0;
			for (i=8; i<16; i++) sig_std_logic_set(css->p[i], css, SIG_STD_LOGIC_0);
		}
	} else {
		/* mode 1 */
		/* FIXME: not supported yet */
		assert(0);
	}

	css->control = val;
}

static void
chip_harris_82c55a_portC_set(void *_css, uint8_t val)
{
	/*
	 *  port C single bit set function
	 */
	unsigned int bit = (val &  0xe) >> 1;
	struct css *css = (struct css *) _css;

	if (val & 0x1) {
		css->portC |= (0x1 << bit);
	} else {
		css->portC &= ~(0x1 << bit);
	}

	/* as long as we support only mode 0, we update all portC bits */
	assert((css->control & 0x60) == 0);
	chip_harris_82c55a_port_write(_css, css->portC, 2);
}

static void
chip_harris_82c55a_power_set(void *_css, unsigned int val)
{
	TRACE(DEBUG_OTHER,  "chip_harris_82c55a power set (%u)\n",val);

	struct css *css = (struct css *) _css;

	css->state_power = val;

	if (val) {
		/* set default values */
		chip_harris_82c55a_mode_set(css, 0x9b);
	}
}

static void
chip_harris_82c55a_n_reset_set(void *_css, unsigned int val)
{
	TRACE(DEBUG_OTHER,  "chip_harris_82c55a resetted (%u)\n",val);

	struct css *css = (struct css *) _css;

	if (val) {
		/* set default values */
		chip_harris_82c55a_mode_set(css, 0x9b);
	}
}

static int
chip_harris_82c55a_inb(void *_css, uint8_t *valp, unsigned long port)
{
	/*
	 * inb: cpu tries to read val from address given in port,
	 * on success values is written to address given with valp
	 * and function returns 0
	 */ 

	 struct css *css = (struct css *) _css;
	 unsigned int addr = (unsigned int) port & 0x3;

	 switch (addr) {
		case 0x0:
			/* port A */
			*valp = css->portA_state;
			break;
		case 0x1:
			/* port B */
			*valp = css->portB_state;
			break;
		case 0x2:
			/* port C */
			*valp = css->portC_state;
			break;
		case 0x3:
			/* control register */
			*valp = css->control;
			break;
		default:	 
			TRACE(DEBUG_INB,"UNKNOW INB READ: addr: %16lu, val: %02x\n",port, *valp);
			assert(0);
			break;
	}

	TRACE(DEBUG_INB,"INB READ: addr: %16lu, val: %02x\n",port, *valp);
	/* FIXME: error return val ? */
	return 0;
}

static int
chip_harris_82c55a_outb(void *_css, uint8_t val, unsigned long port)
{
	/*
	 * outb: cpu writes byte val to address given in port
	 * on success function returns 0
	 */
	
	unsigned int addr = (unsigned int) port & 0x3;
	TRACE(DEBUG_OUTB,"OUTB WRITE: addr: %16u, val: %02x\n",addr, val); 
	 
	switch (addr) {
		case 0x0:
				/* port A */
				chip_harris_82c55a_port_write(_css, val, 0);
				break;
		case 0x1:
				/* port B */
				chip_harris_82c55a_port_write(_css, val, 1);
				break;
		case 0x2:
				/* port C */
				chip_harris_82c55a_port_write(_css, val, 2);				
				break;
		case 0x3:
				/* control word */
				/*
				 * control word bit 7 handling
				 * case bit is 0:
				 * writes to the lower 7 bits of the control word are set / reset operations to port C
				 * values are not stored in the control register. only affects bits from port C
				 * case bit is 1:
				 * writes to the lower 7 bits of ther contral word are mode set operations
				 * values are stored in the control register
				 */
				 if (val & 0x80) {
				 	/* mode set operations */
				 	chip_harris_82c55a_mode_set(_css, val);
				 } else {
				 	/* set / reset operations to port C */
				 	chip_harris_82c55a_portC_set(_css, val);
				 }
				break;
		default:	 
			TRACE(DEBUG_OUTB,"UNKNOW OUTB WRITE: addr: %16lu, val: %02x\n",port, val);
			assert(0);
			break;
	}

	/* FIXME: error return val ? */
	return 0;
}

static void
chip_harris_82c55a_set(void  *_css, unsigned int val, unsigned int nr)
{
	struct css *css = (struct css *) _css;

	switch (nr) {
		case 0 ... 7:
			TRACE(DEBUG_IO,  "Port A: Bit %d changed to %d\n",nr,val);
			if ((val == SIG_STD_LOGIC_1)||(val == SIG_STD_LOGIC_H)) {
				/* set bit */
				css->portA_state |= (1 << nr);
			} else if ((val == SIG_STD_LOGIC_0)||(SIG_STD_LOGIC_L)) {
				/* reset bit */
				css->portA_state &= ~(1 << nr);
			} else if (val == SIG_STD_LOGIC_Z) {
				/* pin open or no one driving, read 1 */
				css->portA_state |= (1 << nr);
			} else {
				TRACE(DEBUG_IO,  "Port A: logic level not supported%s","\n");
			}
			break;
		case 8 ... 15:
			TRACE(DEBUG_IO,  "Port B Bit %d changed to %d\n",(nr-8),val);
			if ((val == SIG_STD_LOGIC_1)||(val == SIG_STD_LOGIC_H)) {
				/* set bit */
				css->portB_state |= (1 << (nr-8));
			} else if ((val == SIG_STD_LOGIC_0)||(SIG_STD_LOGIC_L)) {
				/* reset bit */
				css->portB_state &= ~(1 << (nr-8));
			} else if (val == SIG_STD_LOGIC_Z) {
				/* pin open or no one driving, read 1 */
				css->portB_state |= (1 << (nr-8));
			} else {
				TRACE(DEBUG_IO,  "Port B: logic level not supported%s","\n");
			}
			break;
		case 16 ... 23:
			TRACE(DEBUG_IO,  "Port C: Bit %d changed to %d\n",(nr-16),val);
			if ((val == SIG_STD_LOGIC_1)||(val == SIG_STD_LOGIC_H)) {
				/* set bit */
				css->portC_state |= (1 << (nr-16));
			} else if ((val == SIG_STD_LOGIC_0)||(SIG_STD_LOGIC_L)) {
				/* reset bit */
				css->portC_state &= ~(1 << (nr-16));
			} else if (val == SIG_STD_LOGIC_Z) {
				/* pin open or no one driving, read 1 */
				css->portC_state |= (1 << (nr-16));
			} else {
				TRACE(DEBUG_IO,  "Port C: logic level not supported%s","\n");
			}
			break;
		default:
			/* we have only 24 bit */
			assert(0);
			break;
	}
}


#define def_set(X) \
void chip_harris_82c55a_set ## X (void *_css, unsigned int val) \
{ \
	chip_harris_82c55a_set(_css, val, atoi(# X) ); \
}

def_set(00);
def_set(01);
def_set(02);
def_set(03);
def_set(04);
def_set(05);
def_set(06);
def_set(07);
def_set(08);
def_set(09);
def_set(10);
def_set(11);
def_set(12);
def_set(13);
def_set(14);
def_set(15);
def_set(16);
def_set(17);
def_set(18);
def_set(19);
def_set(20);
def_set(21);
def_set(22);
def_set(23);

#undef def_set

void
chip_harris_82c55a_init(
	unsigned int nr,
	struct sig_boolean *port_power,
	struct sig_boolean *port_n_reset,
	struct sig_cs *port_cs,
	struct sig_std_logic *port_pA0,
	struct sig_std_logic *port_pA1,
	struct sig_std_logic *port_pA2,
	struct sig_std_logic *port_pA3,
	struct sig_std_logic *port_pA4,
	struct sig_std_logic *port_pA5,
	struct sig_std_logic *port_pA6,
	struct sig_std_logic *port_pA7,
	struct sig_std_logic *port_pB0,
	struct sig_std_logic *port_pB1,
	struct sig_std_logic *port_pB2,
	struct sig_std_logic *port_pB3,
	struct sig_std_logic *port_pB4,
	struct sig_std_logic *port_pB5,
	struct sig_std_logic *port_pB6,
	struct sig_std_logic *port_pB7,
	struct sig_std_logic *port_pC0,
	struct sig_std_logic *port_pC1,
	struct sig_std_logic *port_pC2,
	struct sig_std_logic *port_pC3,
	struct sig_std_logic *port_pC4,
	struct sig_std_logic *port_pC5,
	struct sig_std_logic *port_pC6,
	struct sig_std_logic *port_pC7
)
{	
	/*
	 * functions needed
	 * reset/power function:
	 *  since the chip has the same behavior on power on and reset,
	 *  this could be one function and 2 wrappers to keep the style
	 *  .set -> chip_harris_82c55a_power_set -> _chip_harris_82c55a_start (done)
	 *  .set -> chip_harris_82c55a_n_reset_set -> _chip_harris_82c55a_start (done)
	 * cs function for reading/writing chip registers:
	 *  .readb -> chip_harris_82c55a_inb
	 *  .writeb -> chip_harris_82c55a_outb
	 * sig_std_logic set/get functions
	 * .set -> chip_harris_82c55a_xx_set
	 */

	static const struct sig_boolean_funcs power_funcs = {
		.set = chip_harris_82c55a_power_set,
	};
	static const struct sig_boolean_funcs n_reset_funcs = {
		.set = chip_harris_82c55a_n_reset_set,
	};
	static const struct sig_cs_funcs cs_funcs = {
		.readb = chip_harris_82c55a_inb,
		.writeb = chip_harris_82c55a_outb,
	};

	#define def_set(X) \
	static const struct sig_std_logic_funcs func ## X = { \
		.set = chip_harris_82c55a_set ## X, \
	}

	def_set(00);
	def_set(01);
	def_set(02);
	def_set(03);
	def_set(04);
	def_set(05);
	def_set(06);
	def_set(07);
	def_set(08);
	def_set(09);
	def_set(10);
	def_set(11);
	def_set(12);
	def_set(13);
	def_set(14);
	def_set(15);
	def_set(16);
	def_set(17);
	def_set(18);
	def_set(19);
	def_set(20);
	def_set(21);
	def_set(22);
	def_set(23);

	#undef def_set

	struct css *css;

	css = shm_map(CHIPNAME, nr, sizeof(*css), 0);

	css->nr = nr;
	css->port_power = port_power;
	sig_boolean_connect_in(port_power, css, &power_funcs);
	css->port_n_reset = port_n_reset;
	sig_boolean_connect_in(port_n_reset, css, &n_reset_funcs);
	css->port_cs = port_cs;
	sig_cs_connect(port_cs, css, &cs_funcs);

	css->p[0 * 8 + 0] = port_pA0;
	css->p[0 * 8 + 1] = port_pA1;
	css->p[0 * 8 + 2] = port_pA2;
	css->p[0 * 8 + 3] = port_pA3;
	css->p[0 * 8 + 4] = port_pA4;
	css->p[0 * 8 + 5] = port_pA5;
	css->p[0 * 8 + 6] = port_pA6;
	css->p[0 * 8 + 7] = port_pA7;
	css->p[1 * 8 + 0] = port_pB0;
	css->p[1 * 8 + 1] = port_pB1;
	css->p[1 * 8 + 2] = port_pB2;
	css->p[1 * 8 + 3] = port_pB3;
	css->p[1 * 8 + 4] = port_pB4;
	css->p[1 * 8 + 5] = port_pB5;
	css->p[1 * 8 + 6] = port_pB6;
	css->p[1 * 8 + 7] = port_pB7;
	css->p[2 * 8 + 0] = port_pC0;
	css->p[2 * 8 + 1] = port_pC1;
	css->p[2 * 8 + 2] = port_pC2;
	css->p[2 * 8 + 3] = port_pC3;
	css->p[2 * 8 + 4] = port_pC4;
	css->p[2 * 8 + 5] = port_pC5;
	css->p[2 * 8 + 6] = port_pC6;
	css->p[2 * 8 + 7] = port_pC7;

	sig_std_logic_connect_out(css->p[0], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[1], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[2], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[3], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[4], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[5], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[6], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[7], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[8], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[9], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[10], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[11], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[12], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[13], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[14], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[15], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[16], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[17], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[18], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[19], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[20], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[21], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[22], css, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_out(css->p[23], css, SIG_STD_LOGIC_Z);

	sig_std_logic_connect_in(css->p[0], css, &func00);
	sig_std_logic_connect_in(css->p[1], css, &func01);
	sig_std_logic_connect_in(css->p[2], css, &func02);
	sig_std_logic_connect_in(css->p[3], css, &func03);
	sig_std_logic_connect_in(css->p[4], css, &func04);
	sig_std_logic_connect_in(css->p[5], css, &func05);
	sig_std_logic_connect_in(css->p[6], css, &func06);
	sig_std_logic_connect_in(css->p[7], css, &func07);
	sig_std_logic_connect_in(css->p[8], css, &func08);
	sig_std_logic_connect_in(css->p[9], css, &func09);
	sig_std_logic_connect_in(css->p[10], css, &func10);
	sig_std_logic_connect_in(css->p[11], css, &func11);
	sig_std_logic_connect_in(css->p[12], css, &func12);
	sig_std_logic_connect_in(css->p[13], css, &func13);
	sig_std_logic_connect_in(css->p[14], css, &func14);
	sig_std_logic_connect_in(css->p[15], css, &func15);
	sig_std_logic_connect_in(css->p[16], css, &func16);
	sig_std_logic_connect_in(css->p[17], css, &func17);
	sig_std_logic_connect_in(css->p[18], css, &func18);
	sig_std_logic_connect_in(css->p[19], css, &func19);
	sig_std_logic_connect_in(css->p[20], css, &func20);
	sig_std_logic_connect_in(css->p[21], css, &func21);
	sig_std_logic_connect_in(css->p[22], css, &func22);
	sig_std_logic_connect_in(css->p[23], css, &func23);
}

unsigned int
chip_harris_82c55a_create(void)
{
	static unsigned int nr = 0;

	shm_create(CHIPNAME, nr, sizeof(struct css));

	return nr++;
}

void
chip_harris_82c55a_destroy(unsigned int nr)
{
	shm_destroy(CHIPNAME, nr);	
}
