/*
 * Copyright (C) 2010 Freescale Semiconductor, Inc. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    buffer_allocator.c
 *
 * Description:    Implementation physical buffer allocator
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 */

#include <gst/gst.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>


#include "buffer_allocator.h"


#include "fsl_debug.h"



#define MAX_ALLOCATOR 3

typedef struct _HWBufAllocator{
    GMutex *    lock;
    int mem_fd;
    int cnt;
    int freecnt;
    HWBufDesc * free;
    HWBufDesc * all;
    int descontruct;
    struct _HWBufAllocator * next;
}HWBufAllocator;



#define FSL_MM_IOCTL(device, request, errorroute, ...)\
    do{\
        int ret;\
        if ((ret = ioctl((device), (request), ##__VA_ARGS__))<0){\
            DEBUG_ERROR("%s:%d ioctl error, return %d\n", __FILE__, __LINE__, ret);\
            goto errorroute;\
        }\
    }while(0)


static HWBufAllocator * glocalallocator = NULL;

static void _destroyHWBufferAllocator_LOCK(void * allochandle, int force)
{
    /* did not free any software resource when called by deconstructor, 
       due to other element may access it, and it will finally free 
       by loader */
    HWBufAllocator * allocator = (HWBufAllocator *)allochandle;
    HWBufDesc * tmp = allocator->all;
    HWBufDesc * tmpnext;
    if (force==0){
        HWBufAllocator * al1 = glocalallocator;
        if (al1==allocator){
            glocalallocator = allocator->next;
        }else{
            while(al1->next!=allocator)
                al1=al1->next;
            al1->next = allocator->next;
        }
    }
    
    while(tmp){
        allocator->cnt--;
        tmpnext = tmp->link;
        munmap(tmp->virtaddr, tmp->size);
        mem_desc * memdesc = &tmp->memdesc;
        FSL_MM_IOCTL(allocator->mem_fd, PHY_MEM_FREE_IOCTL_ID, error, memdesc);
        if (force==0)
            g_free(tmp);
        tmp = tmpnext;
    }
    allocator->free=NULL;
    allocator->all=NULL;

    g_mutex_unlock(allocator->lock);
    
    if (force==0){
        g_mutex_free(allocator->lock);
        g_free(allocator);
    }
    
    DEBUG_MESSAGE("Hwbuf Allocator destroied, force=%d!\n", force);
error:
    return;
}




void * createHWBufferAllocator()
{
    HWBufAllocator * allocator = NULL;
    allocator = g_malloc(sizeof(HWBufAllocator));
    if (allocator==NULL){
        DEBUG_ERROR("can not allocate memory for hwbuf allocator\n");
        goto error;
    }

    memset(allocator, 0, sizeof(HWBufAllocator));
    allocator->lock = g_mutex_new();

    
    if ((allocator->mem_fd=open(MEMORY_DEVICE_NAME, O_RDWR))<0){
        DEBUG_ERROR("can not open memory device\n");
        goto error;
    }

    allocator->next = glocalallocator;
    glocalallocator = allocator;

    return allocator;
    
error:
    if (allocator)
        g_free(allocator);
    return NULL;
}



void destroyHWBufferAllocator(void * allochandle)
{
    HWBufAllocator * allocator = (HWBufAllocator *)allochandle;
    g_mutex_lock(allocator->lock);
    _destroyHWBufferAllocator_LOCK(allochandle, 0);
}


void setHWBufferAllocatorSelfDescontruct(void * allochandle)
    
{
    HWBufAllocator * allocator = (HWBufAllocator *)allochandle;
    allocator->descontruct=1;
}


HWBufDesc * newHwBuffer(void * ahandle, int size)
{
    HWBufAllocator * allocator = (HWBufAllocator *)ahandle;
    HWBufDesc * bufdesc;

    if (ahandle==NULL)
        return NULL;


    g_mutex_lock(allocator->lock);
    
    bufdesc = allocator->free;
    while(bufdesc){
        if (bufdesc->size>=size)
            break;
        bufdesc=bufdesc->next;
    }
    if (bufdesc){
        if (bufdesc->prev){
            bufdesc->prev->next = bufdesc->next;
        }else{
            allocator->free = bufdesc->next;
        }
        if (bufdesc->next){
            bufdesc->next->prev = bufdesc->prev;
        }
        allocator->freecnt--;
    }else{
        
        bufdesc = g_malloc(sizeof(HWBufDesc));
        if (bufdesc==NULL){
            goto error;
        }
        mem_desc * memdesc =  &bufdesc->memdesc;
        memset(memdesc, 0, sizeof(mem_desc));
        DESC2SIZE(memdesc) = size;
        FSL_MM_IOCTL(allocator->mem_fd, PHY_MEM_ALLOC_IOCTL_ID, error, memdesc);
        bufdesc->phyaddr = DESC2PHYADDRESS(memdesc);
        bufdesc->virtaddr =  mmap(NULL, size,
				    PROT_READ | PROT_WRITE, MAP_SHARED,
				    allocator->mem_fd, bufdesc->phyaddr);
        if ((int)bufdesc->virtaddr==-1){
            DEBUG_ERROR("CAN NOT MAP VIRTADDR %d %p\n", size, bufdesc->phyaddr);
            FSL_MM_IOCTL(allocator->mem_fd, PHY_MEM_FREE_IOCTL_ID, error, memdesc);
            goto error;
        }
        bufdesc->size = size;
        bufdesc->priv = allocator;
        bufdesc->link = allocator->all;
        allocator->all = bufdesc;
        allocator->cnt++;
        
    }
    g_mutex_unlock(allocator->lock);
    return bufdesc;
    
error:
    if (bufdesc){
        g_free(bufdesc);
    }
    g_mutex_unlock(allocator->lock);
    return NULL;
}

void freeHwBuffer( HWBufDesc * bufdesc)
{
    HWBufAllocator * allocator = (HWBufAllocator *)bufdesc->priv;

    g_mutex_lock(allocator->lock);
    bufdesc->prev = NULL;
    bufdesc->next = allocator->free;
    if (allocator->free){
        allocator->free->prev=bufdesc;
    }
    allocator->free = bufdesc;
    allocator->freecnt++;
  

    if ((allocator->freecnt==allocator->cnt)&&(allocator->descontruct)){
        _destroyHWBufferAllocator_LOCK(allocator, 0);
        return;
    }
    
    g_mutex_unlock(allocator->lock);
}


void __attribute__ ((destructor)) buffer_allocator_deconstructor(void);

void buffer_allocator_deconstructor(void)
{
    HWBufAllocator * hd = glocalallocator; 
    HWBufAllocator * hdnext;
    while(hd){
        hdnext = hd->next;
        g_mutex_lock(hd->lock);
        _destroyHWBufferAllocator_LOCK(hd, 1);
        hd = hdnext;
    }
    
}


