/*
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_mpg2_demuxer.c  
 *
 * Description:    mpg2 Demuxer plug-in for Gstreamer.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 */


/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <string.h>
#include <gst/gst.h>
#include <gst/base/gstadapter.h>

#include "mpg2_demuxer/mpg_demuxer_api.h"
#include "mfw_gst_utils.h"
#include "mfw_gst_mpg2demuxer.h"

/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/

/* None */

/*=============================================================================
                           STATIC VARIABLES
=============================================================================*/
#define MFW_GST_MPG2_CAPS \
        "video/mpeg,"\
		"mpegversion = (int) [ 1,2 ], "\
		"systemstream = (boolean){true, false} ;"\
		\
		"video/mpegts,"\
		"systemstream = (boolean)true ;"\
		\
		"video/x-cdxa"  


static GstStaticPadTemplate mfw_gst_mpg2_demuxer_sink_factory =
GST_STATIC_PAD_TEMPLATE("sink",
			GST_PAD_SINK,
			GST_PAD_ALWAYS,
			GST_STATIC_CAPS(MFW_GST_MPG2_CAPS)
					
    );


/*=============================================================================
                            LOCAL MACROS
=============================================================================*/
//input buffer defines
#define MPG2_INPUT_BUFFER_SIZE_IN_BYTE 8192

#define MPG2_INBUFNUMBER_SHIFT  6
#define MPG2_INBUFNUMBER (1<<MPG2_INBUFNUMBER_SHIFT)
#define MPG2_INBUFNUMBER_MASK (MPG2_INBUFNUMBER-1)
#define MPG2_INBUFINDEX_NEXT(index) (((index)+1) & MPG2_INBUFNUMBER_MASK)

#define MPG2_STREAM_VIDEO_ON (1<<0)
#define MPG2_STREAM_AUDIO_ON (1<<1)

#define MPG2_VBUFFER_MAXSIZE 8192
#define MPG2_ABUFFER_MAXSIZE 8192
#define MPG2_STARTCODE       0x00000100

#define MPG2_PTS_SAMPLERATE 90000
#define MPG2_PTS_TO_TIMESTAMP(pts) gst_util_uint64_scale_int(pts, GST_SECOND, MPG2_PTS_SAMPLERATE)

#define MPG2_CHECKDURATION_BACKWARD_LEN 500000
#define MPG2_CHECKIFRAME_BACKWARD_LEN 100000

#define MPG2_IFRAMETYPE     0x001
#define MPG2_SWITCH_OPMODE(demuxer, op) (demuxer)->opmode = (op)

#define MPG2_FLOW g_print


#define MFW_GST_MPG2_VIDEO_CAPS \
        "video/mp2v" 

#define MFW_GST_MPG2_AUDIO_CAPS \
        "audio/mpeg ;" \
        "audio/x-ac3" 

enum {
    AUDIO_TARCK_NUM = 1, /* Select the Audio track count */
    VIDEO_TARCK_NUM,
};
        
/* define a tag name */
#define	MFW_GST_TAG_WIDTH		        "width"
#define MFW_GST_TAG_HEIGHT	            "height"
#define MFW_GST_TAG_FRAMERATE           "framerate"
#define MFW_GST_TAG_SAMPLING_FREQUENCY  "sampling frequency"
#define MFW_GST_TAG_YEAR                "year"

#define MPG2_DEMUXER_FLOW 

#define GET_SEEK_TIME_FROM_OFFSET(cur,basetime,duration)    \
do {                                                        \
       if  ( ((basetime) != 0) && ((basetime)<(duration)))  \
           (cur) = (basetime)  + ((duration)-(basetime))*   \
       ((float)(cur) / (duration));                         \
} while(0);

#ifndef MPG2_DEMUXER_FLOW
#define MPG2_DEMUXER_FLOW(...)
#endif


#define DEMUX_WAIT_INTERVAL 10000

/*=============================================================================
                           LOCAL VARIABLES
=============================================================================*/

/* None. */

/*=============================================================================
                        STATIC FUNCTION PROTOTYPES
=============================================================================*/
#define GST_CAT_DEFAULT mfw_gst_mpg2demuxer_debug

GST_DEBUG_CATEGORY_STATIC(mfw_gst_mpg2demuxer_debug);
static void mfw_gst_mpg2_demuxer_class_init(MFW_GST_MPG2DEMUX_INFO_CLASS_T
					   *);
static void mfw_gst_mpg2_demuxer_base_init(MFW_GST_MPG2DEMUX_INFO_CLASS_T *);
static void mfw_gst_mpg2_demuxer_init(MFW_GST_MPG2DEMUX_INFO_T *,
				     MFW_GST_MPG2DEMUX_INFO_CLASS_T *);
static gboolean mfw_gst_mpg2_demuxer_set_caps(GstPad *, GstCaps *);
static GstStateChangeReturn mfw_gst_mpg2demuxer_change_state(GstElement *,
							    GstStateChange);
static gboolean mfw_gst_mpg2demuxer_activate(GstPad *);
static gboolean mfw_gst_mpg2demuxer_activate_pull(GstPad *, gboolean);
static gboolean mfw_gst_mpg2demuxer_sink_event(GstPad *, GstEvent *);

static gboolean MPG2WriteAudioData(MFW_GST_MPG2DEMUX_INFO_T * demuxer, GstPad * pad, GstBuffer * gstbuf);
static gboolean MPG2WriteVideoData(MFW_GST_MPG2DEMUX_INFO_T * demuxer, GstPad * pad, GstBuffer * gstbuf);

static GstPadTemplate *audio_src_templ(void);
static GstPadTemplate *video_src_templ(void);
static gboolean MPG2DemuxerFillStreamInfo(MFW_GST_MPG2DEMUX_INFO_T *);
static gboolean MPG2DemuxerActivePads(MFW_GST_MPG2DEMUX_INFO_T * demuxer);
static gboolean mpg2_demuxer_set_pad_caps(MFW_GST_MPG2DEMUX_INFO_T *);
static void MPG2CreateTagList(MFW_GST_MPG2DEMUX_INFO_T *);
static gboolean mfw_gst_mpg2demuxer_parse(MFW_GST_MPG2DEMUX_INFO_T *);
static gboolean mfw_gst_mpg2demuxer_handle_src_query(GstPad *,
						    GstQuery *);

static void mfw_gst_mpg2_demuxer_close(MFW_GST_MPG2DEMUX_INFO_T *);
static const GstQueryType *gst_mpg2demuxer_get_src_query_types(GstPad *);
							      
static gboolean mfw_gst_mpg2demuxer_handle_src_event(GstPad*,GstEvent*);

//Input buffer manage functions
static gboolean MPG2FeedInBuffer(MFW_GST_MPG2DEMUX_INFO_T * demuxer);
static void MPG2RecycleInBuffer(MFW_GST_MPG2DEMUX_INFO_T * demuxer);
static void MPG2RecycleAllInBuffer(MFW_GST_MPG2DEMUX_INFO_T * demuxer);
static gboolean MPG2SeekToTimeStamp(MFW_GST_MPG2DEMUX_INFO_T* demuxer, guint64 timetoseek);
static gint MPG2FindStartCode(MFW_GST_MPG2DEMUX_INFO_T * demuxer, char * buffer, int len);
static void MPG2ResyncDemuxer(MFW_GST_MPG2DEMUX_INFO_T * demuxer);

/*=============================================================================
                            STATIC VARIABLES
=============================================================================*/

static GstElementClass *parent_class = NULL;

/*=============================================================================
FUNCTION:           appDemuxMalloc

DESCRIPTION:        Allocate memory for demuxer core library.

ARGUMENTS PASSED:   pAppContext - pointer to application context. 
                    Size - size in byte requested to allocate
  
RETURN VALUE:       void * - return pointer to memory if success. 
                             return NULL if failed.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    Callback function for demuxer core library.
==============================================================================*/
VOID * appDemuxMalloc(VOID * pAppContext, U32 Size)
{
    return (VOID *)(MM_MALLOC(Size));
}

/*=============================================================================
FUNCTION:           appDemuxFree

DESCRIPTION:        Free memory for demuxer core library.

ARGUMENTS PASSED:   pAppContext - pointer to application context
                    Ptr - pointer to memory requested to free
        
RETURN VALUE:       None
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    Callback function for demuxer core library.
==============================================================================*/
VOID appDemuxFree(VOID * pAppContext, VOID *Ptr)
{
    MM_FREE(Ptr);
}

/*=============================================================================
FUNCTION:           appDemuxRequestBuffer

DESCRIPTION:        Allocate output buffer for demuxer core library.

ARGUMENTS PASSED:   pAppContext - pointer to application context
                    StreamID - the ID of the stream the ES buffer associated
                    Size - size in byte of requested out buffer
                    NewPacket - indicate whether is a new packet request or not
        
RETURN VALUE:       void * - return pointer to output buffer if success. 
                             return NULL if failed.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    Callback function for demuxer core library.
==============================================================================*/
VOID * appDemuxRequestBuffer(VOID * pAppContext,U32 StreamID,U32 Size, U32 NewPacket)
{
    MFW_GST_MPG2DEMUX_INFO_T * demuxer = (MFW_GST_MPG2DEMUX_INFO_T *)pAppContext;
    GstPad * pad=NULL;
    GstBuffer * gstbuf = NULL;
    void * buffer = NULL;
    GstFlowReturn ret;
    FSL_MPG_DEMUX_RET_TYPE_T mpg_ret;
    
    if (demuxer->streamID[demuxer->audio_pad_idx]==StreamID){
        pad = demuxer->srcpad[demuxer->audio_pad_idx];

        ret = gst_pad_alloc_buffer(pad, 0, Size, GST_PAD_CAPS(pad),&gstbuf);
        if (ret==GST_FLOW_OK){
            MM_REGRES(gstbuf,RES_GSTBUFFER);
            demuxer->outGstbuf[demuxer->audio_pad_idx] = gstbuf;
            buffer = (void *)GST_BUFFER_DATA(gstbuf);
        } else if ( ret == GST_FLOW_NOT_LINKED) {
            GST_ERROR("Cannot allocate audio buffer from next elment(%d) -- not linked, \
                disable %d stream.\n", ret,demuxer->audio_pad_idx);
            mpg_ret = FSL_MPG_DEMUX_Control(demuxer->parser_object, 
                FSL_MPG_DEMUX_CMD_DISABLE_STREAM, &demuxer->audio_pad_idx);
            if (mpg_ret != FSL_MPG_DEMUX_RT_SUCCESS) {
                GST_WARNING("FSL_MPG_DEMUX_Control failed.\n");
            }
            demuxer->escape = ESCAPE_NOT_LINKED;
        }
        else {
            GST_ERROR("Cannot allocate audio buffer from next elment(%d).\n",ret);
            demuxer->escape = ESCAPE_ERROR;
        }

    }else if (demuxer->streamID[demuxer->video_pad_idx]==StreamID){
        pad = demuxer->srcpad[demuxer->video_pad_idx];
        ret = gst_pad_alloc_buffer(pad, 0, Size, GST_PAD_CAPS(pad),&gstbuf);
        
        if (ret==GST_FLOW_OK){
            MM_REGRES(gstbuf,RES_GSTBUFFER);
            demuxer->outGstbuf[demuxer->video_pad_idx] = gstbuf;
            buffer = (void *)GST_BUFFER_DATA(gstbuf);
        } else if (ret == GST_FLOW_NOT_LINKED) {
            GST_ERROR("Cannot allocate video buffer from next elment --not linked(%d), \
                disable %d stream.\n", ret,demuxer->audio_pad_idx);
            mpg_ret = FSL_MPG_DEMUX_Control(demuxer->parser_object, 
                FSL_MPG_DEMUX_CMD_DISABLE_STREAM, &demuxer->video_pad_idx);
            if (mpg_ret != FSL_MPG_DEMUX_RT_SUCCESS) {
                GST_WARNING("FSL_MPG_DEMUX_Control failed.\n");
            }

            demuxer->escape = ESCAPE_NOT_LINKED;
        }
        else {
            GST_ERROR("Cannot allocate audio buffer from next elment(%d).\n",ret);
            demuxer->escape = ESCAPE_ERROR;
        }
        
    }

    return buffer;
}

/*=============================================================================
FUNCTION:           appDemuxOutput

DESCRIPTION:        Output buffer from demuxer core library.

ARGUMENTS PASSED:   pAppContext - pointer to application context
                    StreamID - the ID of the stream the ES buffer associated
                    pES - pointer to the ES buffer passing out
                    Size - the valid ES stream in the buffer
                    PTS - presentation timestamp(33 bits)
                    DTS - decoding  timestamp(33 bits)
                    Flag - versatile flag
        
RETURN VALUE:       S32 - 0 if output successfully 
                          -1 if failed.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    Callback function for demuxer core library.
==============================================================================*/

S32 appDemuxOutput(VOID * pAppContext,U32 StreamID, VOID *pES, U32 Size,U64 PTS,U64 DTS,U32 Flag)
{
    MFW_GST_MPG2DEMUX_INFO_T * demuxer = (MFW_GST_MPG2DEMUX_INFO_T *)pAppContext;
    GstPad * pad=NULL;
    GstBuffer * gstbuf = NULL;
    GST_DEBUG("video stream id:%d,audio id:%d,coming id:%d.\n",demuxer->streamID[demuxer->video_pad_idx],
            demuxer->streamID[demuxer->audio_pad_idx], StreamID);

    switch (demuxer->opmode){
    case OP_NORMAL:
        
        if (demuxer->streamID[demuxer->audio_pad_idx]==StreamID){
            pad =  demuxer->srcpad[demuxer->audio_pad_idx];
            gstbuf = demuxer->outGstbuf[demuxer->audio_pad_idx];
            if (demuxer->new_seg_flag_audio){
                if ((Flag & FSL_MPG_DEMUX_PTS_VALID) && (MPG2_PTS_TO_TIMESTAMP(PTS)>=demuxer->seektime)){
                    demuxer->atimestamp = MPG2_PTS_TO_TIMESTAMP(PTS);
                }else{
                MM_UNREGRES(gstbuf,RES_GSTBUFFER);
                    gst_buffer_unref(gstbuf);
                    return 0;
                }
            }
            if (MPG2WriteAudioData(demuxer, pad, gstbuf)){
                return 0;
            }else{
                GST_ERROR("Can not push to audio pad\n");
                demuxer->escape = ESCAPE_ERROR;
                return -1;
            }
            
        }else if (demuxer->streamID[demuxer->video_pad_idx]==StreamID){

        
            pad =  demuxer->srcpad[demuxer->video_pad_idx];
            gstbuf = demuxer->outGstbuf[demuxer->video_pad_idx];
            
            if ((Flag & (FSL_MPG_DEMUX_PTS_VALID|FSL_MPG_DEMUX_NEW_PES))==(FSL_MPG_DEMUX_PTS_VALID|FSL_MPG_DEMUX_NEW_PES)){//mark timestarmp if pts present
                //g_print("video %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(MPG2_PTS_TO_TIMESTAMP(PTS)));
                //demuxer->vtimestamp = demuxer->vtimestampnew;
                demuxer->vtimestamp = (MPG2_PTS_TO_TIMESTAMP(PTS));
		//if( (demuxer->vtimestamp-u64_LastTS) > 10000000000 )
		if( demuxer->vtimestamp > demuxer->videoduration )
		{
			demuxer->vtimevalid = FALSE;
			g_print("Warning: video timestamp is large than video duration!\n");
			g_print("video timestamp: %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(demuxer->vtimestamp));
			g_print("video duration: %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(demuxer->videoduration));
		}
		else
		{
                demuxer->vtimevalid = TRUE;
		}
            }
            if (MPG2WriteVideoData(demuxer, pad, gstbuf)){
                return 0;
            }else{
                demuxer->escape = ESCAPE_ERROR;
                GST_ERROR("Can not push to video pad\n");
                return -1;
            }
        }else{
            demuxer->escape = ESCAPE_ERROR;
            GST_ERROR("output stream ID not associated.\n");
            return -1;
        }
        break;

    case OP_PROBE_DURATION:
        if (demuxer->streamID[demuxer->audio_pad_idx]==StreamID){
            if (Flag & FSL_MPG_DEMUX_PTS_VALID){
                if ((MPG2_PTS_TO_TIMESTAMP(PTS))>demuxer->audioduration)
                    demuxer->audioduration=MPG2_PTS_TO_TIMESTAMP(PTS);
            }
            MM_UNREGRES((demuxer->outGstbuf[demuxer->audio_pad_idx]),RES_GSTBUFFER);
            gst_buffer_unref(demuxer->outGstbuf[demuxer->audio_pad_idx]);
            return 0;
        }else if (demuxer->streamID[demuxer->video_pad_idx]==StreamID){
            if (Flag & FSL_MPG_DEMUX_PTS_VALID){
                if ((MPG2_PTS_TO_TIMESTAMP(PTS))>demuxer->videoduration)
                    demuxer->videoduration=MPG2_PTS_TO_TIMESTAMP(PTS);
            }
            MM_UNREGRES((demuxer->outGstbuf[demuxer->video_pad_idx]),RES_GSTBUFFER);
            gst_buffer_unref(demuxer->outGstbuf[demuxer->video_pad_idx]);
            return 0;
        }else{
            demuxer->escape = ESCAPE_ERROR;
            GST_ERROR("output stream ID not associated.\n");
            return -1;
        }
        break;
    
    case OP_PROBE_I_FRAME:
        if (demuxer->streamID[demuxer->audio_pad_idx]==StreamID){
            MM_UNREGRES((demuxer->outGstbuf[demuxer->audio_pad_idx]),RES_GSTBUFFER);
            gst_buffer_unref(demuxer->outGstbuf[demuxer->audio_pad_idx]);
        }else if (demuxer->streamID[demuxer->video_pad_idx]==StreamID){
            gstbuf = demuxer->outGstbuf[demuxer->video_pad_idx];
            if (demuxer->escape==NO_ESCAPE){
                char * buf; 
                gint len, pos;
                
                buf = GST_BUFFER_DATA(gstbuf);
                len = GST_BUFFER_SIZE(gstbuf);
                while ((pos = MPG2FindStartCode(demuxer, buf, len))>=0){
                    if (Flag & FSL_MPG_DEMUX_PTS_VALID){
                        if (len-pos>6){
                            unsigned char key;
                            key = buf[pos+5];
                            key <<=2;
                            key >>=5;
                            if (key==MPG2_IFRAMETYPE){
                                GST_DEBUG("find Iframe with time %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(MPG2_PTS_TO_TIMESTAMP(PTS)));

                                if (MPG2_PTS_TO_TIMESTAMP(PTS)<=demuxer->seektime){
                                    GST_DEBUG("found the header.\n");
                                    demuxer->findHead = 1;
                                }else{
                                    if ((demuxer->findHead==0) && (demuxer->syncstart)){
                                        if (demuxer->syncstart>MPG2_CHECKIFRAME_BACKWARD_LEN)
                                            demuxer->syncstart-=MPG2_CHECKIFRAME_BACKWARD_LEN;
                                        else
                                            demuxer->syncstart=0;
                                        demuxer->escape = ESCAPE_RESYNC;
                                        break;
                                    }else{
                                    //ready to go
                                        demuxer->vtimestamp = MPG2_PTS_TO_TIMESTAMP(PTS);
                                        demuxer->vtimevalid = TRUE;
                                        demuxer->seektime = demuxer->vtimestamp;
                                        GST_BUFFER_DATA(gstbuf) =&buf[pos];
                                        GST_BUFFER_SIZE(gstbuf) = len-pos;
                                        MM_UNREGRES(gstbuf, RES_GSTBUFFER);
                                        gst_adapter_push(demuxer->vAdapter, gstbuf);
                                        MPG2_SWITCH_OPMODE(demuxer, OP_NORMAL);
                                        demuxer->searchstart = 4;
                                        return 0;
                                    }
                                }
                            }
                            
                        }else{
                            GST_DEBUG("waiting for i type\n");
                        }
                        
                    }else{
                        GST_DEBUG("find startcode without time\n");
                    }
                    demuxer->searchstart+=pos;
                
                }
                demuxer->searchstart=0;
            }
            MM_UNREGRES(gstbuf,RES_GSTBUFFER);
            gst_buffer_unref(gstbuf);
        }else{
            demuxer->escape = ESCAPE_ERROR;
            GST_ERROR("output stream ID not associated.\n");
            return -1;
        }
        break;
        
    case OP_PROBE_AUDIO:
        if (demuxer->streamID[demuxer->video_pad_idx]==StreamID){
            MM_UNREGRES((demuxer->outGstbuf[demuxer->video_pad_idx]),RES_GSTBUFFER);
            gst_buffer_unref(demuxer->outGstbuf[demuxer->video_pad_idx]);
        }else if (demuxer->streamID[demuxer->audio_pad_idx]==StreamID){
            gstbuf = demuxer->outGstbuf[demuxer->audio_pad_idx];
            if ((demuxer->escape==NO_ESCAPE) && (Flag & FSL_MPG_DEMUX_PTS_VALID)){
                if (MPG2_PTS_TO_TIMESTAMP(PTS)<=demuxer->seektime){
                    demuxer->findHead = 1;
                }else{
                    if ((demuxer->findHead==0) && (demuxer->syncstart)){
                        if (demuxer->syncstart>MPG2_CHECKIFRAME_BACKWARD_LEN)
                            demuxer->syncstart-=MPG2_CHECKIFRAME_BACKWARD_LEN;
                        else
                            demuxer->syncstart=0;
                        demuxer->escape = ESCAPE_RESYNC;
                        gst_buffer_unref(gstbuf);
                        break;
                    }else{
                    //ready to go
                        demuxer->atimestamp = MPG2_PTS_TO_TIMESTAMP(PTS);
                        pad = demuxer->srcpad[demuxer->audio_pad_idx];
                        
                        demuxer->opmode = OP_NORMAL;
                        if (MPG2WriteAudioData(demuxer, pad, gstbuf)){
                            return 0;
                        }else{
                            GST_ERROR("Can not push to audio pad\n");
                            demuxer->escape = ESCAPE_ERROR;
                            return -1;
                        }
                    }
                }
            }
        }else{
            demuxer->escape = ESCAPE_ERROR;
            GST_ERROR("output stream ID not associated.\n");
            return -1;
        }
        break;
        
    default:
        demuxer->escape = ESCAPE_ERROR;
        GST_ERROR("output stream ID not associated.\n");
        return -1;
    
    }//end of switch
    
}

/*=============================================================================
FUNCTION:           appDemuxOutputIndex

DESCRIPTION:        Output buffer from demuxer core library.

ARGUMENTS PASSED:   pAppContext - pointer to application context
                    StreamID - the ID of the stream the ES buffer associated
                    BufIndex - The buffer Index
                    BufOffset - The buffer offset
                    PTS - presentation timestamp(33 bits)
                    DTS - decoding  timestamp(33 bits)
                    Flag - versatile flag
        
RETURN VALUE:       S32 - 0 if output successfully 
                          -1 if failed.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    Callback function for demuxer core library.
==============================================================================*/

S32 appDemuxOutputIndex(VOID * pAppContext,U32  
            StreamID, U32 BufIndex, S32 BufOffset,U64 PTS,U64 
            DTS,U32 Flag)

{
    MFW_GST_MPG2DEMUX_INFO_T * demuxer = (MFW_GST_MPG2DEMUX_INFO_T *)pAppContext;
    GstPad * pad=NULL;

    if (demuxer->streamID[demuxer->audio_pad_idx]==StreamID){
        if (Flag & FSL_MPG_DEMUX_PTS_VALID){
            if ((MPG2_PTS_TO_TIMESTAMP(PTS))>demuxer->audioduration)
                demuxer->audioduration=MPG2_PTS_TO_TIMESTAMP(PTS);
        }
        return 0;
    }else if (demuxer->streamID[demuxer->video_pad_idx]==StreamID){
        if (Flag & FSL_MPG_DEMUX_PTS_VALID){
            if ((MPG2_PTS_TO_TIMESTAMP(PTS))>demuxer->videoduration)
                demuxer->videoduration=MPG2_PTS_TO_TIMESTAMP(PTS);
        }
        return 0;
    }else {
        GST_ERROR("output stream ID not associated.\n");
        return -1;
    }
    
}
/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/

/*=============================================================================
FUNCTION:           MPG2FindStartcode

DESCRIPTION:        Find startcode pattern in buffer.

ARGUMENTS PASSED:   demuxer - pointer to demuxer context
                    buffer - pointer to buffer need to be searched
                    len - size in byte of valid data in buffer
  
RETURN VALUE:       gint - return no-nagative if startcode found, means 
                            startcode end position based on buffer pointer.
                            return -1 if no startcode found.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/


static gint MPG2FindStartCode(MFW_GST_MPG2DEMUX_INFO_T * demuxer, char * buffer, int len)
{
    guint32 off = demuxer->searchstart;
    guint32 lastword = demuxer->lastword;
    gint ret = -1;
    
    while(off<len){
        lastword = ((lastword<<8) | buffer[off++]);
        if (lastword==MPG2_STARTCODE){
            ret = off-4;
            break;
        }
    }
    
    demuxer->searchstart = off;
    demuxer->lastword = lastword;
    return ret;
}


static gboolean MPG2IsFromStartCode(guint32 * buffer, int startcode)
{
    if ((*buffer&0xffffffff) == startcode)
        return TRUE;
    else
        return FALSE;

}
#define AC3_BIG_STARTCODE 0x00000b77
#define AC3_LIT_STARTCODE 0x0000770b

/*=============================================================================
FUNCTION:           AC3FindStartCode

DESCRIPTION:        Find startcode pattern in AC3 track.

ARGUMENTS PASSED:   demuxer - pointer to demuxer context
                    buffer - pointer to buffer need to be searched
                    len - size in byte of valid data in buffer
                    startcode - start code context.
  
RETURN VALUE:       gint - return no-nagative if startcode found, means 
                            startcode end position based on buffer pointer.
                            return -1 if no startcode found.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/
static gint AC3FindStartCode(MFW_GST_MPG2DEMUX_INFO_T * demuxer, char * buffer, int len, int startcode)
{
    guint32 off = demuxer->ac3_sstart;
    guint32 lastword = demuxer->ac3_lastword;
    gint ret = -1;
    
    while(off<len){
        lastword = ((lastword<<8) | buffer[off++]);
        if ((lastword&0x0000ffff)==startcode){
            ret = off - 2;
            GST_DEBUG("Found startcode, offset is %d.\n",ret);
            break;
        }
    }
    
    demuxer->ac3_sstart = off;
    demuxer->ac3_lastword = lastword;
    return ret;
}

/*=============================================================================
FUNCTION:           AC3IsFromStartCode

DESCRIPTION:        Check the buffer is AC3 format or not.

ARGUMENTS PASSED:   
                    buffer - pointer to buffer need to be checked.
                    startcode - start code context.
  
RETURN VALUE:       gboolean - return TRUE if the first code is startcode.
                               return FALSE if the first code is not startcode.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/
static gboolean AC3IsFromStartCode(guint32 * buffer, int startcode)
{
    if ((*buffer&0x0000ffff) == startcode)
        return TRUE;
    else
        return FALSE;

}


/*=============================================================================
FUNCTION:            MPG2UpdateFileInfo
        
DESCRIPTION:         This function update file file information. 

ARGUMENTS PASSED:   None
  
RETURN VALUE:       None

PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None
   	    
IMPORTANT NOTES:    None
=============================================================================*/

void MPG2UpdateFileInfo(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{
    GstPad *my_peer_pad = NULL;
    GstFormat fmt = GST_FORMAT_BYTES;

    my_peer_pad = gst_pad_get_peer(demuxer->sinkpad);
    gst_pad_query_duration(my_peer_pad, &fmt,
			   &(demuxer->file_info.length));

	gst_object_unref(my_peer_pad);
}

/*=============================================================================
FUNCTION:           MPG2WriteVideoData

DESCRIPTION:        Output video data to video codec plugin.

ARGUMENTS PASSED:   demuxer - pointer to demuxer context
                    pad - video pad for write
                    gstbuf - gst buffer need to be output
        
RETURN VALUE:       gboolean - TRUE if success
                               FALSE if failed
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/
static gboolean MPG2WriteVideoData(MFW_GST_MPG2DEMUX_INFO_T * demuxer, GstPad * pad, GstBuffer * gstbuf)
{
    guint buflen;
    gint position;
    guint8 * data;
    GstBuffer *pushbuffer;
    gboolean ret = TRUE;
    GstAdapter * adapter = demuxer->vAdapter;
    static int av_ts_nonesync = 1;
    // If audio and video are not synchronous with difference base time stamp larger than 10 seconds, then do synchronization.
    if( ((guint64)(demuxer->baseATS-demuxer->baseVTS)>(guint64)1000000000000) && (1==av_ts_nonesync) )
    {
        printf("demuxer->baseVTS==%lld,demuxer->baseATS==%lld,demuxer->vtimestamp==%lld,demuxer->atimestamp==%lld\n",demuxer->baseVTS,demuxer->baseATS,demuxer->vtimestamp,demuxer->atimestamp);
        GST_DEBUG("demuxer->baseVTS==%lld,demuxer->baseATS==%lld,demuxer->vtimestamp==%lld,demuxer->atimestamp==%lld\n",demuxer->baseVTS,demuxer->baseATS,demuxer->vtimestamp,demuxer->atimestamp);
        av_ts_nonesync = 0;

        GST_DEBUG("demuxer->first_media_pts = %x(%x,%lld).\n",demuxer->first_media_pts
            , demuxer->first_media_pts & FIRST_VIDEO_PTS_MASK, demuxer->baseVTS);
        GST_DEBUG("video new segment timestamp %" GST_TIME_FORMAT "\n", GST_TIME_ARGS(demuxer->vtimestamp));

        {
            GstObject *parent;
            //demuxer->baseVTS = demuxer->baseATS;
            parent = gst_element_get_parent(GST_ELEMENT(demuxer));

            if (parent != NULL) {
                gst_pipeline_set_new_stream_time(GST_PIPELINE(parent),demuxer->baseATS);
                gst_object_unref(parent);
            }
            GST_WARNING("set video new basetime %" GST_TIME_FORMAT "\n", GST_TIME_ARGS(demuxer->baseATS));

        }
        if ( !gst_pad_push_event(pad,
                                 gst_event_new_new_segment(FALSE, 
                                        1.0, GST_FORMAT_TIME, 
                                        demuxer->baseATS,
                                        GST_CLOCK_TIME_NONE, 
                                        demuxer->baseATS)
                                )
           ) 
        {
            GST_ERROR("\nCannot send new segment to the src pad\n");
            goto War_Msg;
        }
    }

    if  (demuxer->new_seg_flag_video) {
        GST_DEBUG("demuxer->first_media_pts = %x(%x,%lld).\n",demuxer->first_media_pts
            , demuxer->first_media_pts & FIRST_VIDEO_PTS_MASK, demuxer->baseVTS);
        GST_DEBUG("video new segment timestamp %" GST_TIME_FORMAT "\n", GST_TIME_ARGS(demuxer->vtimestamp));

        if ( ((demuxer->first_media_pts & FIRST_VIDEO_PTS_MASK) != FIRST_VIDEO_PTS_MASK) 
            && (demuxer->baseVTS == 0) ) {
            GstObject *parent;
            demuxer->baseVTS = demuxer->vtimestamp;
            parent = gst_element_get_parent(GST_ELEMENT(demuxer));

            if (parent != NULL) {
                gst_pipeline_set_new_stream_time(GST_PIPELINE(parent),demuxer->baseVTS);
                gst_object_unref(parent);
            }
            demuxer->first_media_pts |= FIRST_VIDEO_PTS_MASK;
            GST_WARNING("set video new basetime %" GST_TIME_FORMAT "\n", GST_TIME_ARGS(demuxer->baseVTS));

        }
        if ( !gst_pad_push_event(pad,
                                 gst_event_new_new_segment(FALSE, 
                                        1.0, GST_FORMAT_TIME, 
                                        demuxer->vtimestamp,
                                        GST_CLOCK_TIME_NONE, 
                                        demuxer->vtimestamp)
                                )
           ) 
        {
            GST_ERROR("\nCannot send new segment to the src pad\n");
            demuxer->new_seg_flag_video = FALSE;
            goto War_Msg;
        }
        demuxer->new_seg_flag_video = FALSE;
    }
    gst_adapter_push(adapter, gstbuf);
    MM_UNREGRES(gstbuf,RES_GSTBUFFER);

    buflen = gst_adapter_available(adapter);
    data = (guint8 *)gst_adapter_peek(adapter, buflen);

#if 1
    while ((position = MPG2FindStartCode(demuxer, data, buflen))>=0){
        pushbuffer = gst_adapter_take_buffer(adapter, position);
        if ( !MPG2IsFromStartCode((guint32 *)GST_BUFFER_DATA(pushbuffer), 0x00010000))
        {
            GST_WARNING("It is not start from the start code (%08x).\n",*(guint32 *)GST_BUFFER_DATA(pushbuffer));
        }
        demuxer->searchstart -= position;
        gst_buffer_set_caps(pushbuffer, GST_PAD_CAPS(pad));
        if (demuxer->vtimevalid){
            demuxer->vtimevalid = FALSE;
            GST_BUFFER_TIMESTAMP(pushbuffer) = demuxer->vtimestamp;
            GST_DEBUG("GST buffer time = %"GST_TIME_FORMAT" \n", GST_TIME_ARGS(GST_BUFFER_TIMESTAMP(pushbuffer)));

        }   

        if (gst_pad_push(pad, pushbuffer)!=GST_FLOW_OK){
            ret = FALSE;
            GST_ERROR("Error in push buffer to video sink pad\n");
            goto War_Msg;
        }
        demuxer->videosent = TRUE;
        buflen = gst_adapter_available(adapter);
        data = (guint8 *)gst_adapter_peek(adapter, buflen);
    }
#else
{
   gint p1,p2;
   gint32 off;
   guint32 lastword;
   
   while (1) {
        off = demuxer->searchstart;
        lastword = demuxer->lastword;

        if (((p1 = MPG2FindStartCode(demuxer, data, buflen))>=0) && 
            (p1 <= 100)
            )
            {
            pushbuffer = gst_adapter_take_buffer(adapter, p1);
            if ( !MPG2IsFromStartCode((guint32 *)GST_BUFFER_DATA(pushbuffer), 0x00010000))
            {
                g_print("It is not start from the start code,discard it(%08x).\n",*(guint32 *)GST_BUFFER_DATA(pushbuffer));
            }
            demuxer->searchstart -= p1;
            gst_buffer_set_caps(pushbuffer, gst_pad_get_caps(pad));
            if (demuxer->vtimevalid){
                demuxer->vtimevalid = FALSE;
                GST_BUFFER_TIMESTAMP(pushbuffer) = demuxer->vtimestamp;
                GST_DEBUG("GST buffer time = %"GST_TIME_FORMAT" \n", GST_TIME_ARGS(GST_BUFFER_TIMESTAMP(pushbuffer)));

            }   
            if (gst_pad_push(pad, pushbuffer)!=GST_FLOW_OK){
                ret = FALSE;
                GST_ERROR("Error in push buffer to video sink pad\n");
                goto War_Msg;
            }
            buflen = gst_adapter_available(adapter);
            data = (guint8 *)gst_adapter_peek(adapter, buflen);
            break;
            }
        
         if (((p1 = MPG2FindStartCode(demuxer, data, buflen))>=0)
          && ((p2 = MPG2FindStartCode(demuxer, data, buflen))>=0) ) {
                // g_print("buflen:%d,p1:%d,p2:%d.\n",buflen,p1,p2);
                pushbuffer = gst_adapter_take_buffer(adapter, p1);
                if ( !MPG2IsFromStartCode((guint32 *)GST_BUFFER_DATA(pushbuffer), 0x00010000))
                {
                    g_print("It is not start from the start code,discard it(%08x).\n",*(guint32 *)GST_BUFFER_DATA(pushbuffer));
                }
                gst_buffer_set_caps(pushbuffer, gst_pad_get_caps(pad));
                if (demuxer->vtimevalid){
                    demuxer->vtimevalid = FALSE;
                    GST_BUFFER_TIMESTAMP(pushbuffer) = demuxer->vtimestamp;
                    GST_DEBUG("GST buffer time = %"GST_TIME_FORMAT" \n", GST_TIME_ARGS(GST_BUFFER_TIMESTAMP(pushbuffer)));

                }   
                if (gst_pad_push(pad, pushbuffer)!=GST_FLOW_OK){
                    ret = FALSE;
                    GST_ERROR("Error in push buffer to video sink pad\n");
                    goto War_Msg;
                }
                buflen = gst_adapter_available(adapter);
                data = (guint8 *)gst_adapter_peek(adapter, buflen);
                demuxer->searchstart = 4;
                
        }
        else {
       demuxer->searchstart = off;
       demuxer->lastword = lastword;
        break;
        }
    }
}   
#endif    
    return ret;
    
War_Msg:
	{
    GstMessage *message = NULL;
	GError *gerror = NULL;
	gerror = g_error_new_literal(1, 0, "pads are not negotiated!");

	message =
	    gst_message_new_warning(GST_OBJECT(GST_ELEMENT(demuxer
)), gerror,
				  "video pads are not negotiated!");
	gst_element_post_message(GST_ELEMENT(demuxer), message);
	g_error_free(gerror);
	return FALSE;
    }
}


/*=============================================================================
FUNCTION:           MPG2WriteAudioData

DESCRIPTION:        Output video data to video codec plugin.

ARGUMENTS PASSED:   demuxer - pointer to demuxer context
                    pad - video pad for write
                    gstbuf - gst buffer need to be output
        
RETURN VALUE:       gboolean - TRUE if success
                               FALSE if failed
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/
static gboolean MPG2WriteAudioData(MFW_GST_MPG2DEMUX_INFO_T * demuxer, GstPad * pad, GstBuffer * gstbuf)
{
    guint buflen;
    gint position;
    guint8 * data;
    GstBuffer *pushbuffer;
    gboolean ret = TRUE;
    FILE *fp,*fp1;

    if  (demuxer->new_seg_flag_audio) {
        if ((demuxer->first_media_pts & FIRST_AUDIO_PTS_MASK) != FIRST_AUDIO_PTS_MASK) {
            demuxer->baseATS = demuxer->atimestamp;
            demuxer->first_media_pts |= FIRST_AUDIO_PTS_MASK;
            GST_WARNING("set audio new basetime %" GST_TIME_FORMAT "\n", GST_TIME_ARGS(demuxer->atimestamp));

        }
        if ( !gst_pad_push_event(pad,
                                 gst_event_new_new_segment(FALSE, 
                                        1.0, GST_FORMAT_TIME, 
                                        demuxer->atimestamp,
                                        GST_CLOCK_TIME_NONE, 
                                        demuxer->atimestamp)
                                )
           ) 
        {
            GST_ERROR("\nCannot send new segment to the src pad\n");
            demuxer->new_seg_flag_audio = FALSE;
            goto War_Msg;
        }
        
        demuxer->new_seg_flag_audio = FALSE;
    }
            

    if (demuxer->audio_type[demuxer->audio_pad_idx] == FSL_MPG_DEMUX_AC3_AUDIO) {
        GstAdapter * adapter = demuxer->aAdapter;

        gst_adapter_push(adapter, gstbuf);
        MM_UNREGRES(gstbuf,RES_GSTBUFFER);

        buflen = gst_adapter_available(adapter);
        data = (guint8 *)gst_adapter_peek(adapter, buflen);
        
        while ((position = AC3FindStartCode(demuxer, data, buflen, AC3_BIG_STARTCODE))>0){ 
                /* should check is zero ? */
                pushbuffer = gst_adapter_take_buffer(adapter, position);
                demuxer->ac3_sstart -= position;

                /* Check the buffer is start with startcode */
                if ( !AC3IsFromStartCode((guint32 *)GST_BUFFER_DATA(pushbuffer), AC3_LIT_STARTCODE))
                {
                    GST_WARNING("It is not start from the start code,discard it(%08x).\n",*(guint32 *)GST_BUFFER_DATA(pushbuffer));
                    gst_buffer_unref(pushbuffer);
                    break;
                }
                gst_buffer_set_caps(pushbuffer, GST_PAD_CAPS(pad));
         
                if (gst_pad_push(pad, pushbuffer)!=GST_FLOW_OK){
                    ret = FALSE;
                    GST_ERROR("Error in push buffer to audio sink pad\n");
                    goto War_Msg;
                }


                buflen = gst_adapter_available(adapter);
                data = (guint8 *)gst_adapter_peek(adapter, buflen);
                
        }
    }
    else {
        MM_UNREGRES(gstbuf,RES_GSTBUFFER);
            if (gst_pad_push(pad, gstbuf)!=GST_FLOW_OK){
                ret = FALSE;
                GST_ERROR("Error in push buffer to audio sink pad\n");
            }  
            
    }

    // gst_buffer_unref(gstbuf);
    return ret;
    
War_Msg:
	{
    GstMessage *message = NULL;
	GError *gerror = NULL;
	gerror = g_error_new_literal(1, 0, "pads are not negotiated!");

	message =
	    gst_message_new_warning(GST_OBJECT(GST_ELEMENT(demuxer
)), gerror,
				  "audio pads are not negotiated!");
	gst_element_post_message(GST_ELEMENT(demuxer), message);
	g_error_free(gerror);
	return TRUE;
    }
}

/*=============================================================================
FUNCTION:           MPG2FeedInBuffer

DESCRIPTION:        Feed buffer to demuxer core library.

ARGUMENTS PASSED:   demuxer - pointer to demuxer context
        
RETURN VALUE:       gboolean - TRUE if one buffer is successfully feeded.
                               FALSE if there's no space for feed buffer.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/
static gboolean MPG2FeedInBuffer(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{
    guint index = demuxer->inWriteIndex;
    FSL_MPG_DEMUX_BD_T * inbuf_descriptor = demuxer->inbuf_object;
    GstBuffer * gstbuf;
    
    if (inbuf_descriptor[index].pCnxt==NULL){
        if (gst_pad_pull_range(demuxer->sinkpad, demuxer->file_info.offset,
			                    MPG2_INPUT_BUFFER_SIZE_IN_BYTE, &gstbuf) == GST_FLOW_OK){
            MM_REGRES(gstbuf,RES_GSTBUFFER);
			                    
			demuxer->file_info.offset += GST_BUFFER_SIZE(gstbuf);
            
            inbuf_descriptor[index].pPtr = GST_BUFFER_DATA(gstbuf);
            inbuf_descriptor[index].uliFill = GST_BUFFER_SIZE(gstbuf);
            inbuf_descriptor[index].pCnxt = gstbuf;
            demuxer->inWriteIndex = MPG2_INBUFINDEX_NEXT(index);

            if ((demuxer->file_info.length) 
                && (demuxer->file_info.offset>=demuxer->file_info.length)){
                demuxer->eosfile = TRUE;
                GST_WARNING("eos of file\n");
            }else if (GST_BUFFER_SIZE(gstbuf)<MPG2_INPUT_BUFFER_SIZE_IN_BYTE){
                demuxer->eosfile = TRUE;
                GST_WARNING("eos of file\n");
            }
			return TRUE;   
        }else{
            GST_ERROR("pull range failed \n");
            demuxer->escape = ESCAPE_ERROR;
            return FALSE;
        }
    }else{
        GST_ERROR("can not feed buffer since ring is full\n");
        return FALSE;
    }
}

/*=============================================================================
FUNCTION:           MPG2RecycleInBuffer

DESCRIPTION:        Recycle input buffer which is be consumed by demuxer core 
                    library.

ARGUMENTS PASSED:   demuxer - pointer to demuxer context
        
RETURN VALUE:       gboolean - TRUE if at least one buffer is successfully recycled.
                               FALSE if there's no buffer can be recycled.
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/
static void MPG2RecycleInBuffer(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{
    FSL_MPG_DEMUX_BD_T * inbuf_descriptor = demuxer->inbuf_object;
    GstBuffer * gstbuf;
    guint index = demuxer->inReadIndex;

    for (;;){
        if ((gstbuf = inbuf_descriptor[index].pCnxt)!=NULL){
            if (inbuf_descriptor[index].uliFill == 0){
                MM_UNREGRES(gstbuf,RES_GSTBUFFER);
                gst_buffer_unref(gstbuf);
                inbuf_descriptor[index].pPtr = NULL;
                inbuf_descriptor[index].pCnxt = NULL;
                index = MPG2_INBUFINDEX_NEXT(index);
            }else{
              break;
            }
        }else{
            break;
        }
    }
    demuxer->inReadIndex = index;
}   

/*=============================================================================
FUNCTION:           MPG2RecycleAllInBuffer

DESCRIPTION:        Recycle all input buffer previous feeded to demuxer core 
                    library whatever it was consumed or not.

ARGUMENTS PASSED:   demuxer - pointer to demuxer context
        
RETURN VALUE:       None 
                    
PRE-CONDITIONS:     None
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/
static void MPG2RecycleAllInBuffer(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{
    int i;
    GstBuffer * gstbuf;
    FSL_MPG_DEMUX_BD_T * inbuf_descriptor = demuxer->inbuf_object;
    for (i=0;i<MPG2_INBUFNUMBER;i++){
        if ((gstbuf = inbuf_descriptor[i].pCnxt)!=NULL){
            MM_UNREGRES(gstbuf,RES_GSTBUFFER);
            gst_buffer_unref(gstbuf);
            inbuf_descriptor[i].pCnxt = NULL;
            demuxer->inbuf_object[i].uliFill = 0;
        }
    }
    demuxer->inReadIndex = demuxer->inWriteIndex = 0;
}


/*=============================================================================
FUNCTION:           MPG2ResyncDemuxer

DESCRIPTION:        Recycle all input buffer previous feeded to demuxer core 
                    library whatever it was consumed or not.

ARGUMENTS PASSED:   demuxer - pointer to demuxer context
        
RETURN VALUE:       None 
                    
PRE-CONDITIONS:     sync point offset should be stored in demuxer->syncstart
 
POST-CONDITIONS:    None

IMPORTANT NOTES:    None
==============================================================================*/
static void MPG2ResyncDemuxer(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{
    demuxer->file_info.offset = demuxer->syncstart;
    gst_adapter_clear(demuxer->vAdapter);
    gst_adapter_clear(demuxer->aAdapter);
    
    demuxer->lastword = 0xffffffff;
    demuxer->searchstart = 0;
    demuxer->eosfile = FALSE;
    demuxer->escape = NO_ESCAPE;

    demuxer->ac3_lastword = 0xffffffff;
    demuxer->ac3_sstart = 0;
    
    GST_WARNING("resync request to %lld\n", (gint64)demuxer->syncstart);
    FSL_MPG_DEMUX_Control(demuxer->parser_object, FSL_MPG_DEMUX_CMD_RESYNC, NULL);
    MPG2RecycleAllInBuffer(demuxer);
    
}

/*=============================================================================
FUNCTION:            mfw_gst_mpg2demuxer_parse
        
DESCRIPTION:         This is the main functions, where calls to MPG2 Parser 
					 are made.

ARGUMENTS PASSED:    demuxer - The context of the main mpg2 demuxer element.
  
RETURN VALUE:        None

PRE-CONDITIONS:      None
 
POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/
static gboolean mfw_gst_mpg2demuxer_parse(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{
    GstPad *src_pad;
    gboolean ret_value;
    FSL_MPG_DEMUX_RET_TYPE_T Err = FSL_MPG_DEMUX_RT_SUCCESS;
    guint track_count;		/* track counter */
    guint stream_count;		/* number of streams */
	guint64 duration;     /* stream duration */ 
	gchar *text_msg = NULL;
    
    FSL_MPG_DEMUX_T * parser_object;
    FSL_MPG_DEMUX_BD_T * inbuf_object;
    int i;
    int value;

    /* Allocate the memories for the objects */
    parser_object =
        (FSL_MPG_DEMUX_T *) MM_MALLOC(sizeof(FSL_MPG_DEMUX_T));

    if (parser_object == NULL) {
        GST_ERROR
            ("Error in allocating the memory for mpg2 parser object type\n");
        text_msg = " Unable to allocate the memory ";
        goto Err_Msg;
	}

    inbuf_object = 
        (FSL_MPG_DEMUX_BD_T *)MM_MALLOC(sizeof(FSL_MPG_DEMUX_BD_T)*MPG2_INBUFNUMBER);

    if (inbuf_object == NULL) {
        GST_ERROR
            ("Error in allocating the memory for mpg2 input buffer\n");
        text_msg = " Unable to allocate the memory ";
        goto Err_Msg;
	}

    //store system objects.
    demuxer->parser_object = parser_object;
    demuxer->inbuf_object = inbuf_object;

    demuxer->inReadIndex = demuxer->inWriteIndex = 0;

    //initialize system variables
    for (i=0;i<MPG2_INBUFNUMBER;i++){
        inbuf_object[i].pPtr = NULL;
        inbuf_object[i].uliFill = 0;
        inbuf_object[i].uliLen = MPG2_INPUT_BUFFER_SIZE_IN_BYTE;
        inbuf_object[i].pCnxt = NULL;
    }

    memset(parser_object, 0, sizeof(FSL_MPG_DEMUX_T));

    parser_object->InputBuffer.pPtr = inbuf_object;
    parser_object->InputBuffer.usiCntBDSet = MPG2_INBUFNUMBER;

    parser_object->pAppContext = demuxer;

    parser_object->DemuxMalloc = appDemuxMalloc;
    parser_object->DemuxFree = appDemuxFree;
    parser_object->DemuxRequestBuffer = appDemuxRequestBuffer;
    parser_object->DemuxOutput = appDemuxOutput;
    /* Add for new demuxer API */
    parser_object->DemuxOutputIndex = appDemuxOutputIndex;

    MPG2UpdateFileInfo(demuxer);

    /* Initialize the mpg2 parser */
    Err = FSL_MPG_DEMUX_Open(parser_object);
    if (Err!=FSL_MPG_DEMUX_RT_SUCCESS) {
    	GST_ERROR("Error in FSL_MPG_DEMUX_Open, ret=%d\n", Err);
    	text_msg = " Unable to open demuxer ";
    	goto Err_Msg;
    }

    while(MPG2FeedInBuffer(demuxer)){
        Err = FSL_MPG_DEMUX_GetPSI(parser_object);
        MPG2RecycleInBuffer(demuxer);
        if ((Err!=FSL_MPG_DEMUX_RT_MOREDATA) || (demuxer->eosfile))
            break;
    };
    
    if (Err!=FSL_MPG_DEMUX_RT_SUCCESS){
        GST_ERROR
            ("Error in FSL_MPG_DEMUX_GetPSI, ret=%d\n", Err);
        text_msg = "Unable to getPSIs the vectors ";
        goto Err_Msg;
    }
    
    {
        FSL_MPG_DEMUX_PSI_T * psi_info = &parser_object->TS_PSI;
        if (psi_info->IsTS){
            g_print("TS stream found, total %d programs\n", psi_info->Programs);
            for (i=0;i<psi_info->Programs;i++){
                if (psi_info->ProgramNumber[i]!=0){
                    FSL_MPG_DEMUX_Control(parser_object,FSL_MPG_DEMUX_CMD_SELECT_PROGRAM,&psi_info->ProgramNumber[i]);
                    break;
                }
            }
            if (i==psi_info->Programs){
                GST_ERROR
                    ("Can not select a program\n");
                 text_msg = "Unable to getPSIs the vectors ";
                goto Err_Msg;
            }
            
        }
    }

    MPG2ResyncDemuxer(demuxer);
    
    while(MPG2FeedInBuffer(demuxer)){
        Err = FSL_MPG_DEMUX_Probe(parser_object);
        MPG2RecycleInBuffer(demuxer);
        if ((Err!=FSL_MPG_DEMUX_RT_MOREDATA) || (demuxer->eosfile))
            break;
    };

    if ((Err == FSL_MPG_DEMUX_RT_MOREDATA) && (demuxer->eosfile == TRUE)){
        GST_WARNING("Not a standard clips");
    }else if (Err!=FSL_MPG_DEMUX_RT_SUCCESS){
        GST_ERROR
            ("Error in FSL_MPG_DEMUX_Probe, ret=%d\n", Err);
        text_msg = "Unable to probe the vectors ";
        goto Err_Msg;
    }

    if (demuxer->stop_request == FALSE) {
        /**** Setting Pads for the streams present ****/
    	ret_value = MPG2DemuxerFillStreamInfo(demuxer);
    	if (!ret_value) {
    	    GST_ERROR("Error in setting the pad capability in fill stream info\n");
    	    text_msg = " media formats are not matching";

    		goto Err_Msg;
    	}
        MPG2DemuxerActivePads(demuxer);
        
    } else {
    	/* reseting the capabilities of the pad */
        GST_WARNING("Reset all the pads.\n");
    	ret_value = mpg2_demuxer_set_pad_caps(demuxer);
    	if (!ret_value) {
    	    GST_ERROR("Error in setting the pad capability \n");
    		text_msg = " media formats are not matching";

    		goto Err_Msg;
    	    
    	}
    }
    
    return TRUE;

	/* send an error message for an unrecoverable error */
Err_Msg:
	{
    GstMessage *message = NULL;
	GError *gerror = NULL;

	gerror = g_error_new_literal(1, 0, text_msg);

	message =
	    gst_message_new_error(GST_OBJECT(GST_ELEMENT(demuxer)), gerror,
				  "debug none");
	gst_element_post_message(GST_ELEMENT(demuxer), message);
	g_error_free(gerror);
	return FALSE;
	}
 }

/*=============================================================================
FUNCTION:            mfw_gst_mpg2demuxer_taskfunc
        
DESCRIPTION:         This is the main functions, where audio and video data are
                     Read from the file 

ARGUMENTS PASSED:    demuxer - The context of the main mpg2 demuxer element.
        
  
RETURN VALUE:        
                     None
        
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

void mfw_gst_mpg2demuxer_taskfunc(MFW_GST_MPG2DEMUX_INFO_T *demuxer)
{
	gboolean  ret_value;
	FILE      *fp;
	GstPad    *src_pad;
	guint     stream_count;
	guint     track_count;
    GstAdapter * adapter = demuxer->vAdapter;
    guint len;
    FSL_MPG_DEMUX_RET_TYPE_T Err;

 //   g_mutex_lock(demuxer->lock_mutex);
    MPG2FeedInBuffer(demuxer);
    Err = FSL_MPG_DEMUX_Process(demuxer->parser_object);
    MPG2RecycleInBuffer(demuxer);
//    g_mutex_unlock(demuxer->lock_mutex);
    
    if ((Err == FSL_MPG_DEMUX_RT_EOS) || (demuxer->eosfile)){
        if (demuxer->opmode==OP_NORMAL){
            if (len = gst_adapter_available(adapter)){
                g_print("last %d\n", len);
                gst_pad_push(demuxer->srcpad[demuxer->video_pad_idx], gst_adapter_take_buffer(adapter, len));
            }
        if (demuxer->srcpad[demuxer->audio_pad_idx]){
            src_pad = demuxer->srcpad[demuxer->audio_pad_idx];
            if (src_pad){
                /*Handle an event on the sink pad with event as End of Stream */
                ret_value = mfw_gst_mpg2demuxer_sink_event(src_pad, gst_event_new_eos());

                if (ret_value != TRUE) {
                    GST_ERROR("\n Error in pushing the event, result is %d\n", ret_value);
                }
            }
        }

        if (demuxer->srcpad[demuxer->video_pad_idx]){
            src_pad = demuxer->srcpad[demuxer->video_pad_idx];
            if (src_pad){
                /*Handle an event on the sink pad with event as End of Stream */
                ret_value = mfw_gst_mpg2demuxer_sink_event(src_pad, gst_event_new_eos());

                if (ret_value != TRUE) {
                    GST_ERROR("\n Error in pushing the event, result is %d\n", ret_value);
                }
            }
        }
              
            goto pause_task;
        }else if ((demuxer->opmode==OP_PROBE_I_FRAME)||(demuxer->opmode==OP_PROBE_AUDIO)){//seeking
            if (demuxer->findHead ==0){
                if (demuxer->syncstart>MPG2_CHECKIFRAME_BACKWARD_LEN)
                    demuxer->syncstart-=MPG2_CHECKIFRAME_BACKWARD_LEN;
                else
                    demuxer->syncstart=0;
                MPG2ResyncDemuxer(demuxer);
            }else{
                demuxer->seektime = 0;
                MPG2_SWITCH_OPMODE(demuxer, OP_NORMAL);
                MPG2ResyncDemuxer(demuxer);
            }
        }
        
    }
#if 0
    else if ((Err == FSL_MPG_DEMUX_RT_DISCONTINUITY_AUDIO) || (Err ==FSL_MPG_DEMUX_RT_DISCONTINUITY_VIDEO)){
        demuxer->escape=NO_ESCAPE;
    }
#endif    
    else if (demuxer->escape==ESCAPE_ERROR){
        //goto pause_task;
        
    }else if ((demuxer->escape==ESCAPE_RESYNC)){
        MPG2ResyncDemuxer(demuxer);
    }

    if (demuxer->videosent) {
        /* Keep the balance of demuxer and codec */    
        usleep(DEMUX_WAIT_INTERVAL);
        demuxer->videosent = FALSE;
    }

    return;

    
pause_task:
	{
	 /* Pause the task active on the pad */
    ret_value = gst_pad_pause_task(demuxer->sinkpad);
    if (FALSE == ret_value)
    GST_ERROR("\n There is no task on this sink pad !! \n");
	return;
	}

}

/*=============================================================================
FUNCTION:            MPG2DemuxerFillStreamInfo
        
DESCRIPTION:         Add and Creates the audio pad sets the 
					 corresponding capabilities.

ARGUMENTS PASSED:    demuxer      - The main structure of mpg2 parser plugin context.
        
  
RETURN VALUE:
       TRUE       -	 if the function is performed properly
	   FALSE	  -	 if there is any errors encounterd
        

PRE-CONDITIONS:
   	    None
 
POST-CONDITIONS:
   	    None

IMPORTANT NOTES:
   	    None
=============================================================================*/

static gboolean MPG2DemuxerFillStreamInfo(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{

    GstPadTemplate *templ = NULL;
    GstCaps *caps = NULL;
    GstPad *pad = NULL;
    gchar *padname = NULL;
    guint total_pads;
    GstPad *src_pad;
    guint32 version;
    gboolean set;
    guint frame_width;
    guint frame_height;
    float frame_rate;
	gchar *text_msg=NULL;
	guint video_offset=0;
	guint audio_offset=0;
	guint sampling_frequency=0;
    gboolean type_find;

    const char *media_type;

    FSL_MPG_DEMUX_SYSINFO_T * sysinfo;
    FSL_MPEGSTREAM_T * streaminfo;
    int i;

    TRACK_INFO_T track_id;
    sysinfo = &(demuxer->parser_object->SystemInfo);

    if (sysinfo->uliPSFlag){
        MPG2_FLOW("PS stream.\n");
	}
	else if (sysinfo->uliNoStreams>0){
        MPG2_FLOW("PES stream.\n");
	}
	else{
        GST_ERROR("Not PS/PES. Not support \n");
        return FALSE;
	}

    GST_DEBUG("1 total stream number %d\n", sysinfo->uliNoStreams);

    // searching for audio stream
    for (i=0;i<sysinfo->uliNoStreams;i++){
        caps = NULL;
        streaminfo = &(sysinfo->Stream[i]);
        if (i >= MAX_SRC_PADS) {
            g_print("The stream track exceed the capability of demuxer.\n");
            break;
        }

        if ((demuxer->streamMask & MPG2_STREAM_AUDIO_ON)
             &&(streaminfo->enuStreamType == FSL_MPG_DEMUX_AUDIO_STREAM) 
             && (streaminfo->uliPropertyValid)
            ) {

            FSL_AUDIO_PROPERTY_T * audioinfo = &(streaminfo->MediaProperty.AudioProperty);

            GST_DEBUG("Audio index and required: %d,%d\n",demuxer->audio_track_idx,demuxer->audio_track_num);
            GST_DEBUG("Audio Type: %d, stream count : %d.\n",audioinfo->enuAudioType,i);

            if  ((audioinfo->enuAudioType==FSL_MPG_DEMUX_MP1_AUDIO)
                 || (audioinfo->enuAudioType==FSL_MPG_DEMUX_MP2_AUDIO)
                 || (audioinfo->enuAudioType==FSL_MPG_DEMUX_MP3_AUDIO)){

                /* 
                 * Only supported tracked will be saved, the unsupported will 
                 *    be discard silently.
                 *
                 */
                demuxer->streamID[i] = audioinfo->uliAudioID;
                demuxer->streamType[i] = AUDIO_INDEX;
                demuxer->audio_type[i] = audioinfo->enuAudioType;
                demuxer->audio_track_num++;    

                media_type="audio/mpeg";
                version = 1;
                
                padname = g_strdup_printf(PARSER_AUDIOPAD_TEMPLATE_NAME, i);
                templ = audio_src_templ();
                caps = gst_caps_new_simple(media_type,
    				       "mpegversion", G_TYPE_INT, version,
    				       "framed", G_TYPE_INT, 1,
    				       "bitrate", G_TYPE_INT,
    				       audioinfo->uliAudioBitRate,
    					   "rate", G_TYPE_INT,
    					   audioinfo->uliAudioSampleRate,
                           "channels",G_TYPE_INT,//MONO,
                           audioinfo->usiAudioChannels,
    				       NULL);
            }
            else if (audioinfo->enuAudioType==FSL_MPG_DEMUX_AC3_AUDIO)
            {

                demuxer->streamID[i] = audioinfo->uliAudioID;
                demuxer->streamType[i] = AUDIO_INDEX;
                demuxer->audio_type[i] = audioinfo->enuAudioType;
                demuxer->audio_track_num++;    

                media_type="audio/x-ac3";
                version = 1;
                padname = g_strdup_printf(PARSER_AUDIOPAD_TEMPLATE_NAME, i);
                templ = audio_src_templ();
                caps = gst_caps_new_simple(media_type,
                           "channels",G_TYPE_INT,//MONO,
                           audioinfo->usiAudioChannels,
    				       NULL);
                demuxer->new_seg_flag_audio = TRUE;
            }

        }
        if ((demuxer->streamMask & MPG2_STREAM_VIDEO_ON)
             &&(streaminfo->enuStreamType == FSL_MPG_DEMUX_VIDEO_STREAM) 
             && (streaminfo->uliPropertyValid)
             ) {
            FSL_VIDEO_PROPERTY_T * videoinfo = &(streaminfo->MediaProperty.VideoProperty);

            if  (videoinfo->enuVideoType==FSL_MPG_DEMUX_MPEG2_VIDEO) {
                demuxer->streamID[i] = videoinfo->uliVideoID;
                demuxer->streamType[i] = VIDEO_INDEX;
                demuxer->video_track_num++;

                GST_DEBUG("Video index and required: %d,%d\n",demuxer->video_track_idx,demuxer->video_track_num);
                GST_DEBUG("Video Type: %d, stream count : %d.\n",videoinfo->enuVideoType,i);

                media_type="video/mp2v";
                version = 1;
                padname = g_strdup_printf(PARSER_VIDEOPAD_TEMPLATE_NAME, i);
                templ = video_src_templ();
                caps = gst_caps_new_simple(media_type,
    				       "mpegversion", G_TYPE_INT, 1,
    				       "systemstream", G_TYPE_BOOLEAN, FALSE,
    				       "height", G_TYPE_INT, videoinfo->uliVideoHeight,
    				       "width", G_TYPE_INT, videoinfo->uliVideoWidth, 
    				       "framerate", GST_TYPE_FRACTION,
    				       (gint32) videoinfo->uliFRNumerator, 
    				       (gint32) videoinfo->uliFRDenominator,
                           NULL);

            }
        }

        

        if (caps){
            demuxer->caps[i] = caps;
            
            pad = demuxer->srcpad[i] = gst_pad_new_from_template(templ, padname);
            
	        g_free(padname);

            gst_pad_use_fixed_caps(pad);

            /* set the stream_ptr on to the pad */
            gst_pad_set_element_private(pad, demuxer->srcpad[i]);

            /* set the capabilities of the pad */
            set = gst_pad_set_caps(pad, caps);
            if (set == FALSE) {
                GST_ERROR
                    ("\n unable to set the capability of the pad:result is %d\n", set);
                return set;
            }

            GstCaps *temp_caps;
            temp_caps = gst_pad_get_caps(pad);
            /* check if the caps represents media formats */
            if (gst_caps_is_empty(temp_caps)){
                GST_ERROR("\n caps is empty\n");
                    gst_caps_unref(caps);
                    gst_caps_unref(temp_caps);
                    return FALSE;
            }
            gst_caps_unref(temp_caps);
            
            /* unref a GstCap */
            gst_caps_unref(caps);
            
        }
    }

    /* Send Signal to upper layer */
    track_id.audio_track_num = demuxer->audio_track_num;
    track_id.video_track_num = demuxer->video_track_num;
    g_signal_emit (&demuxer->element, demuxer->track_signal_id, 
        0 /* details. No details means the detail is same as the signal name "track_id". */, 
        (void *) (&track_id));

    return TRUE;
}

static gboolean MPG2DemuxerActivePads(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{

    GstPad *pad = NULL;
    int i;
    TRACK_INFO_T track_id;
    gint value;
    FSL_MPG_DEMUX_T * parser_object = demuxer->parser_object;
    gint Err;
    FSL_MPG_DEMUX_SYSINFO_T * sysinfo = NULL;
    FSL_MPG_DEMUX_RET_TYPE_T mpg_ret;
    if (demuxer->parser_object)  
        sysinfo = &(demuxer->parser_object->SystemInfo);
    else 
        return FALSE;  // happens on gst-inspect

    
    track_id.audio_track_num = 0;
    track_id.video_track_num = 0;

    GST_DEBUG("%s: actived audio track:%d, video track:%d\n",
        __FUNCTION__,demuxer->audio_track_idx,demuxer->video_track_idx);

    for (i=0;i<sysinfo->uliNoStreams;i++)
    {
        if ( demuxer->srcpad[i] != NULL) {
            if (demuxer->streamType[i] == VIDEO_INDEX)
            {
                if (demuxer->video_track_idx == track_id.video_track_num) {
                    demuxer->video_pad_idx = i;

                    mpg_ret = FSL_MPG_DEMUX_Control(demuxer->parser_object, FSL_MPG_DEMUX_CMD_ENABLE_STREAM, &i);
                    if (mpg_ret != FSL_MPG_DEMUX_RT_SUCCESS) {
                        GST_WARNING("FSL_MPG_DEMUX_Control failed.\n");
                    }
                    demuxer->new_seg_flag_video = TRUE;

                    GST_WARNING("****\nEnabled Video stream :%d.\n",i);
                    GST_DEBUG("Enabled Video stream id:%d\n",demuxer->streamID[demuxer->video_pad_idx]);
                }
                else {
                    mpg_ret = FSL_MPG_DEMUX_Control(demuxer->parser_object, FSL_MPG_DEMUX_CMD_DISABLE_STREAM, &i);
                    if (mpg_ret != FSL_MPG_DEMUX_RT_SUCCESS) {
                        GST_WARNING("FSL_MPG_DEMUX_Control failed.\n");
                    }
                    GST_DEBUG("get pad name :%s.\n",gst_pad_get_name(demuxer->srcpad[i]));

                    gst_pad_set_active(demuxer->srcpad[i], FALSE);
                    if (gst_element_get_pad(GST_ELEMENT(demuxer),
                        gst_pad_get_name(demuxer->srcpad[i]))) 
                    {
                        gst_element_remove_pad(GST_ELEMENT(demuxer), demuxer->srcpad[i]);
                    }
                    GST_DEBUG("Disabled Video stream :%d.\n",i);
                }
                track_id.video_track_num++;
            }

            else if (demuxer->streamType[i] == AUDIO_INDEX )
            {
                if (demuxer->audio_track_idx == track_id.audio_track_num) {
                    demuxer->audio_pad_idx = i;

                    mpg_ret = FSL_MPG_DEMUX_Control(demuxer->parser_object, FSL_MPG_DEMUX_CMD_ENABLE_STREAM, &i);
                    if (mpg_ret != FSL_MPG_DEMUX_RT_SUCCESS) {
                        GST_WARNING("FSL_MPG_DEMUX_Control failed.\n");
                    }

                    demuxer->new_seg_flag_audio = TRUE;
                    GST_WARNING("****\nEnabled Audio stream :%d (type:%d).\n",i,demuxer->audio_type[i]);
                    GST_DEBUG("Enabled Audio stream id:%d\n",demuxer->streamID[demuxer->audio_pad_idx]);

                }
                else {
                    mpg_ret = FSL_MPG_DEMUX_Control(demuxer->parser_object, FSL_MPG_DEMUX_CMD_DISABLE_STREAM, &i);
                    if (mpg_ret != FSL_MPG_DEMUX_RT_SUCCESS) {
                        GST_WARNING("FSL_MPG_DEMUX_Control failed.\n");
                    }
                    gst_pad_set_active(demuxer->srcpad[i], FALSE);
                    if (gst_element_get_pad(GST_ELEMENT(demuxer),
                        gst_pad_get_name(demuxer->srcpad[i]))) 
                    {
                        gst_element_remove_pad(GST_ELEMENT(demuxer), demuxer->srcpad[i]);
                    }
                   
                    GST_DEBUG("Disabled Audio stream :%d.\n",i);

                }
                track_id.audio_track_num++;
            }
        }
        else {
            mpg_ret = FSL_MPG_DEMUX_Control(demuxer->parser_object, FSL_MPG_DEMUX_CMD_DISABLE_STREAM, &i);
            if (mpg_ret != FSL_MPG_DEMUX_RT_SUCCESS) {
                GST_WARNING("FSL_MPG_DEMUX_Control failed.\n");
            }
            
            GST_DEBUG("Disabled stream :%d.\n",i);

        }
    }

    if ( demuxer->srcpad[demuxer->audio_pad_idx] != NULL) {
            pad = demuxer->srcpad[demuxer->audio_pad_idx];
            {
                /* Activates the given pad */
                if (gst_pad_set_active(pad, TRUE) == TRUE) {
                    GST_DEBUG("\nThe pad was activated successfully\n");
                }
                /* adds a pad to the element */
                gst_element_add_pad(GST_ELEMENT(demuxer), pad);
            }
            
    }

    if ( demuxer->srcpad[demuxer->video_pad_idx] != NULL) {

            pad = demuxer->srcpad[demuxer->video_pad_idx];
            /* Activates the given pad */
            {
                if (gst_pad_set_active(pad, TRUE) == TRUE) {
                    GST_DEBUG("\nThe pad was activated successfully\n");
                }
                /* adds a pad to the element */
                gst_element_add_pad(GST_ELEMENT(demuxer), pad);
            }
             
    }

    gst_element_no_more_pads(GST_ELEMENT (demuxer));
    //initialize core demuxer parameters to normal operation
    value = 1;
    FSL_MPG_DEMUX_Control(parser_object, FSL_MPG_DEMUX_CMD_SET_SKIPFLAG, &value);

    value = MPG2_VBUFFER_MAXSIZE;
    FSL_MPG_DEMUX_Control(parser_object, FSL_MPG_DEMUX_CMD_SET_VBUFSIZE, &value);
    
    value = MPG2_ABUFFER_MAXSIZE;
    FSL_MPG_DEMUX_Control(parser_object, FSL_MPG_DEMUX_CMD_SET_ABUFSIZE, &value);


    if (demuxer->file_info.length>MPG2_CHECKDURATION_BACKWARD_LEN){
        demuxer->syncstart= demuxer->file_info.length 
                                    - MPG2_CHECKDURATION_BACKWARD_LEN;
    }else{
        demuxer->syncstart = 0;
    }

    //try to get duration for search tail of file;
    MPG2_SWITCH_OPMODE(demuxer, OP_PROBE_DURATION);
    MPG2ResyncDemuxer(demuxer);
    while(MPG2FeedInBuffer(demuxer)){
        /* 
         * Set the Demuxer to Scan mode, in this case, the
         *   output callback will be appDemuxOutputIndex. 
         */

        Err = FSL_MPG_DEMUX_Scan(parser_object);
        // Err = FSL_MPG_DEMUX_Process(parser_object);
        MPG2RecycleInBuffer(demuxer);
        if (((Err!=FSL_MPG_DEMUX_RT_MOREDATA) && (Err!=FSL_MPG_DEMUX_RT_SUCCESS)) 
            || (demuxer->eosfile))
            break;
    };
    
    if (demuxer->videoduration){
        gst_segment_set_duration(&demuxer->segment, GST_FORMAT_TIME, demuxer->videoduration);
    }else if (demuxer->audioduration){
        g_print("set audio as duration\n");
        gst_segment_set_duration(&demuxer->segment, GST_FORMAT_TIME, demuxer->audioduration);
    }
    else 
        GST_WARNING("there is no duration in clips.\n");

   
    
    GST_WARNING("\tAudio Duration: %"GST_TIME_FORMAT", \n\tVideo Duration: %"GST_TIME_FORMAT"\n", 
        GST_TIME_ARGS(demuxer->audioduration), GST_TIME_ARGS(demuxer->videoduration));

    demuxer->syncstart = 0;
    
    //switch to normal mode
    MPG2_SWITCH_OPMODE(demuxer, OP_NORMAL);
    /* 
     * Reset the Demuxer to Normal mode, in this case, the
     *   output callback will be appDemuxOutput. 
     */
    MPG2ResyncDemuxer(demuxer);
    

    if (demuxer->srcpad[demuxer->audio_pad_idx]){
    	gst_pad_set_query_type_function(demuxer->srcpad[demuxer->audio_pad_idx],
    					                GST_DEBUG_FUNCPTR
    					                 (gst_mpg2demuxer_get_src_query_types));

    	gst_pad_set_query_function(demuxer->srcpad[demuxer->audio_pad_idx],
    				                    GST_DEBUG_FUNCPTR
    				                     (mfw_gst_mpg2demuxer_handle_src_query));

    	gst_pad_set_event_function(demuxer->srcpad[demuxer->audio_pad_idx],
                                        GST_DEBUG_FUNCPTR (
    		                             mfw_gst_mpg2demuxer_handle_src_event));
    }

    if (demuxer->srcpad[demuxer->video_pad_idx]){
    	gst_pad_set_query_type_function(demuxer->srcpad[demuxer->video_pad_idx],
    					                GST_DEBUG_FUNCPTR
    					                 (gst_mpg2demuxer_get_src_query_types));

    	gst_pad_set_query_function(demuxer->srcpad[demuxer->video_pad_idx],
    				                    GST_DEBUG_FUNCPTR
    				                     (mfw_gst_mpg2demuxer_handle_src_query));

    	gst_pad_set_event_function(demuxer->srcpad[demuxer->video_pad_idx],
                                        GST_DEBUG_FUNCPTR (
    		                             mfw_gst_mpg2demuxer_handle_src_event));
    }

    

	/* Add the user data information to the taglist */
	MPG2CreateTagList(demuxer);

    return TRUE;
}




/*=============================================================================
FUNCTION:            mpg2_demuxer_set_pad_caps
        
DESCRIPTION:         Reset the capabilities of the pads created before.

ARGUMENTS PASSED:    demuxer      - The main structure of mpg2 parser plugin context.
        
  
RETURN VALUE:
       TRUE       -	 if the function is performed properly
	   FALSE	  -	 if there is any errors encounterd
        

PRE-CONDITIONS:
   	    None
 
POST-CONDITIONS:
   	    None

IMPORTANT NOTES:
   	    None
   	
=============================================================================*/

static gboolean mpg2_demuxer_set_pad_caps(MFW_GST_MPG2DEMUX_INFO_T * demuxer)
{
    gboolean set;

    if (demuxer->srcpad[demuxer->audio_pad_idx]){
        set = gst_pad_set_caps(demuxer->srcpad[demuxer->audio_pad_idx],
            demuxer->caps[demuxer->audio_pad_idx]);
        if (set == FALSE) {
            GST_ERROR("\n unable to set the capability of the pad:result is %d\n",
                 set);
            return FALSE;
	    }
    }

    if (demuxer->srcpad[demuxer->video_pad_idx]){
        set = gst_pad_set_caps(demuxer->srcpad[demuxer->video_pad_idx],
            demuxer->caps[demuxer->video_pad_idx]);
        if (set == FALSE) {
            GST_ERROR("\n unable to set the capability of the pad:result is %d\n",
                 set);
            return FALSE;
	    }
    }
    return TRUE;
}

/*=============================================================================
FUNCTION:            audio_src_templ
        
DESCRIPTION:         defines the audio source pad template.

ARGUMENTS PASSED:    None
  
RETURN VALUE:        Pointer to GstPadTemplate. 

PRE-CONDITIONS:      None
   	    
POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/
static GstPadTemplate *audio_src_templ(void)
{
    static GstPadTemplate *templ = NULL;

    if (!templ) {
	GstCaps *caps = NULL;
    GstCaps *ac3caps = NULL;
	GstStructure *structure;

	caps = gst_caps_new_simple("audio/mpeg",
				   "mpegversion", GST_TYPE_INT_RANGE, 0, 1,
				   "framed", GST_TYPE_INT_RANGE, 0, 1,
				   "bitrate", GST_TYPE_INT_RANGE, 0,
				   G_MAXINT,NULL);

	ac3caps = gst_caps_new_simple("audio/x-ac3",
				   NULL);

    gst_caps_append(caps, ac3caps);
    
	templ = gst_pad_template_new(PARSER_AUDIOPAD_TEMPLATE_NAME, GST_PAD_SRC,
				     GST_PAD_SOMETIMES, caps);
    }
    return templ;
}

/*=============================================================================
FUNCTION:            video_src_templ
        
DESCRIPTION:         defines the video source pad template.

ARGUMENTS PASSED:    None
  
RETURN VALUE:        Pointer to GstPadTemplate. 

PRE-CONDITIONS:      None
 
POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/
static GstPadTemplate *video_src_templ()
{
    static GstPadTemplate *templ = NULL;

    if (!templ) {
	GstCaps *caps      = NULL;
	GstStructure *structure;

	caps     = gst_caps_new_simple("video/mp2v",
				   NULL
				   );
				   
	templ = gst_pad_template_new(PARSER_VIDEOPAD_TEMPLATE_NAME, GST_PAD_SRC,
				     GST_PAD_SOMETIMES, caps);
    }
    return templ;
}

/*=============================================================================
FUNCTION:            mfw_gst_mpg2demuxer_sink_event
        
DESCRIPTION:         This functions handles the events that triggers the
				     sink pad of the mpg2demuxer element.

ARGUMENTS PASSED:
        pad      -   pointer to pad
        event    -   pointer to event
        
RETURN VALUE:
       TRUE       -	 if event is sent to sink properly
	   FALSE	  -	 if event is not sent to sink properly

PRE-CONDITIONS:
   	    None
 
POST-CONDITIONS:
   	    None

IMPORTANT NOTES:
   	    None
=============================================================================*/
static gboolean mfw_gst_mpg2demuxer_sink_event(GstPad * pad,
					      GstEvent * event)
{

    gboolean result = TRUE;
    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_NEWSEGMENT:
	{
	    GstFormat format;
	    gst_event_parse_new_segment(event, NULL, NULL, &format, NULL,
					NULL, NULL);
	    if (format == GST_FORMAT_TIME) {
		GST_ERROR("\nCame to the FORMAT_TIME call\n");
		gst_event_unref(event);
		result = TRUE;
	    } else {
		GST_ERROR("dropping newsegment event in format %s",
			gst_format_get_name(format));
		gst_event_unref(event);
		result = TRUE;
	    }
	    break;
	}

    case GST_EVENT_EOS:
	{
	    GST_DEBUG("\nDemuxer: Sending EOS to Decoder\n");
	    result = gst_pad_push_event(pad, event);

	    if (result != TRUE) {
		GST_ERROR("\n Error in pushing the event, result is %d\n",
			  result);
	    }
	    break;
	}

    default:
	{
	    result = gst_pad_event_default(pad, event);
	    return TRUE;
	}

    }

    return result;
}

/*=============================================================================
FUNCTION:   	     mfw_gst_mpg2demuxer_activate

DESCRIPTION:         it will call gst_pad_activate_pull which activates or
                     deactivates the given pad in pull mode via dispatching to the
                     pad's activepullfunc

ARGUMENTS PASSED:
         pad       - pointer to GstPad to activate or deactivate

RETURN VALUE:
         TRUE      - if operation was successful
         FALSE     - if operation was unsuccessful

PRE-CONDITIONS:      None
        

POST-CONDITIONS:     None
        

IMPORTANT NOTES:     None
        
=============================================================================*/

static gboolean mfw_gst_mpg2demuxer_activate(GstPad * pad)
{


    if (gst_pad_check_pull_range(pad)) {
	return gst_pad_activate_pull(pad, TRUE);
    } else {
	return FALSE;
    }

    return TRUE;

}

/*=============================================================================
FUNCTION:			 mfw_gst_mpg2demuxer_activate_pull

DESCRIPTION:		 This will start the demuxer task function in "pull" based	
					 scheduling model.

ARGUMENTS PASSED:
	pad		         sink pad, where the data will be pulled into, from	
					 file source.
	active 	         pad is active or not(TRUE/FALSE)
 

RETURN VALUE:		
		TRUE      -  if operation was successful
        FALSE     -  if operation was unsuccessful

PRE-CONDITIONS:      None

POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/

static gboolean
mfw_gst_mpg2demuxer_activate_pull(GstPad * pad, gboolean active)
{
    gboolean ret_val = TRUE;
    MFW_GST_MPG2DEMUX_INFO_T *demuxer_data =
	MFW_GST_MPG2_DEMUXER(GST_PAD_PARENT(pad));

    if (active) {

    /* read the track information present in the container */
    ret_val=mfw_gst_mpg2demuxer_parse(demuxer_data);

	if(ret_val==FALSE)
	{
	    /* if the parsing failed due to unsupported stream or 
		   corrupted stream,return back */
		
		mfw_gst_mpg2_demuxer_close(demuxer_data);
		return FALSE;
	}
	
	/* start a Task, which will read each frame of audio/video data from the 
	   container */
	ret_val = gst_pad_start_task((demuxer_data->sinkpad),
				     (GstTaskFunction)
				     mfw_gst_mpg2demuxer_taskfunc,
				     demuxer_data);

	if (ret_val == FALSE) {
	    GST_ERROR("Task could not be started \n");
		//G2ParserFreeAtom(demuxer_data->parser_info.mpg2_parser_object_type);
	    mfw_gst_mpg2_demuxer_close(demuxer_data);
	    return ret_val;
	}
    } else {
	/* Stop the Task */
	    return gst_pad_pause_task(demuxer_data->sinkpad);

    }

    return TRUE;
}

/*=============================================================================
FUNCTION:   	     MPG2CreateTagList
	
DESCRIPTION:	     This function Add the user data information of the mpg2 container
                     to the taglist.

ARGUMENTS PASSED:
      demuxer_info   : - The main structure of mpg2 parser plugin context.
	  
     
RETURN VALUE       
                     None
        

PRE-CONDITIONS:      None
        

POST-CONDITIONS:     None
        

IMPORTANT NOTES:     None
        
=============================================================================*/

static void MPG2CreateTagList(MFW_GST_MPG2DEMUX_INFO_T *demuxer_info)
{
    //NOT IMPLEMENTED
	return;
}
/*=============================================================================
FUNCTION:   	     mfw_gst_mpg2demuxer_change_state
	
DESCRIPTION:	     This function keeps track of different states of pipeline.

ARGUMENTS PASSED:
      element    :   pointer to the mpg2 demuxer element.
	  transition :   state of the pipeline.
     
RETURN VALUE       
        GST_STATE_CHANGE_FAILURE    - the state change failed
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded
        GST_STATE_CHANGE_ASYNC      - the state change will happen
                                        asynchronously
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled
        

PRE-CONDITIONS:      None
        

POST-CONDITIONS:     None
        

IMPORTANT NOTES:     None
        
=============================================================================*/

static GstStateChangeReturn
mfw_gst_mpg2demuxer_change_state(GstElement * element,
				GstStateChange transition)
{
    GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
    MFW_GST_MPG2DEMUX_INFO_T *mpg2_demuxer = MFW_GST_MPG2_DEMUXER(element);
    guint32 stream_count;

    GstFormat fmt = GST_FORMAT_TIME;
    gint64 pos = 0, len = 0;
    gboolean ret_value;
    guint count;

	/***** UPWARDS *****/
    switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
	{
	    GST_DEBUG
		("\nCame to Demuxer GST_STATE_CHANGE_NULL_TO_READY\n");
        MM_INIT_DBG_MEM("mpg2dem");
	    mpg2_demuxer->file_info.length = 0;
	    mpg2_demuxer->file_info.offset = 0;
	    mpg2_demuxer->segment.last_stop = 0;
	    mpg2_demuxer->new_seg_flag_audio = FALSE;
		mpg2_demuxer->new_seg_flag_video = FALSE;
	    mpg2_demuxer->stop_request = FALSE;
	    mpg2_demuxer->inbuff = NULL;
		mpg2_demuxer->seek_flag = FALSE;
        mpg2_demuxer->streamMask = (MPG2_STREAM_AUDIO_ON | MPG2_STREAM_VIDEO_ON);
        mpg2_demuxer->vAdapter = gst_adapter_new();
        mpg2_demuxer->aAdapter = gst_adapter_new();
        mpg2_demuxer->eosfile = FALSE;
        mpg2_demuxer->searchstart = 0;
        mpg2_demuxer->lastword = 0xffffffff;
        mpg2_demuxer->escape = NO_ESCAPE;
        mpg2_demuxer->vtimestamp = 0;
        mpg2_demuxer->atimestamp = 0;
        mpg2_demuxer->videoduration = 0;
        mpg2_demuxer->audioduration = 0;
        mpg2_demuxer->syncstart = 0;
        mpg2_demuxer->findHead = 0;
        mpg2_demuxer->vtimevalid = FALSE;
        mpg2_demuxer->is_seeking = FALSE;

       
        mpg2_demuxer->first_media_pts = 0;        
        mpg2_demuxer->is_get_stream_info = FALSE; 
        mpg2_demuxer->videosent = FALSE;
        mpg2_demuxer->audio_pad_idx = MAX_SRC_PADS -1 ;
        mpg2_demuxer->video_pad_idx = MAX_SRC_PADS -1;
        
	    break;
	}

    case GST_STATE_CHANGE_READY_TO_PAUSED:
	{

	    GST_DEBUG
		("\nCame to Demuxer GST_STATE_CHANGE_READY_TO_PAUSED\n");
       
        /* Registers a new tag type for the use with GStreamer's type system */
	    gst_tag_register (MFW_GST_TAG_WIDTH, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "image width","image width(pixel)", NULL);
        gst_tag_register (MFW_GST_TAG_HEIGHT, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "image height","image height(pixel)", NULL); 
	    gst_tag_register (MFW_GST_TAG_FRAMERATE, GST_TAG_FLAG_DECODED,G_TYPE_FLOAT,
            "video framerate","number of video frames in a second", NULL);
	    gst_tag_register (MFW_GST_TAG_SAMPLING_FREQUENCY, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "sampling frequency","number of audio samples per frame per second", NULL);
	    gst_tag_register (MFW_GST_TAG_YEAR, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "year","year of creation", NULL);

        

	    /* Initialize segment to its default values. */

	    gst_segment_init(&mpg2_demuxer->segment, GST_FORMAT_UNDEFINED);

	    /* Sets the given activate function for the pad. The activate function will
	       dispatch to activate_push or activate_pull to perform the actual activation. */

	    gst_pad_set_activate_function(mpg2_demuxer->sinkpad,
					  mfw_gst_mpg2demuxer_activate);

	    /* Sets the given activate_pull function for the pad. An activate_pull
	       function prepares the element and any upstream connections for pulling. */

	    gst_pad_set_activatepull_function(mpg2_demuxer->sinkpad,
					      GST_DEBUG_FUNCPTR
					      (mfw_gst_mpg2demuxer_activate_pull));

        break;
	}

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
	{


	    GST_DEBUG
		("\nCame to Demuxer GST_STATE_CHANGE_PAUSED_TO_PLAYING\n");
	    break;
	}

    default:
	break;

    }

    
    ret = parent_class->change_state(element, transition);
	


	 /***** DOWNWARDS *****/
    switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
	{
	    break;
	}
    case GST_STATE_CHANGE_PAUSED_TO_READY:
	{
		/* received stop request,do the following */
        mfw_gst_mpg2_demuxer_close(mpg2_demuxer);
	   
	    /* Initialize segment to its default values. */
	    gst_segment_init(&mpg2_demuxer->segment, GST_FORMAT_UNDEFINED);

	    mpg2_demuxer->file_info.length = 0;
	    mpg2_demuxer->file_info.offset = 0;
	    mpg2_demuxer->segment.last_stop = 0;
	    mpg2_demuxer->new_seg_flag_video = FALSE;
		mpg2_demuxer->new_seg_flag_audio = FALSE;
	    mpg2_demuxer->stop_request = TRUE;
	    mpg2_demuxer->inbuff = NULL;
		mpg2_demuxer->seek_flag=FALSE;

        g_mutex_free(mpg2_demuxer->lock_mutex);
        
 
	    break;

	}

    case GST_STATE_CHANGE_READY_TO_NULL:
	{
        MM_DEINIT_DBG_MEM();
	    break;
	}

    default:
	break;
    }

    return ret;
}

/*=============================================================================
FUNCTION:   mfw_gst_mpg2demuxer_handle_sink_query   

DESCRIPTION:    performs query on src pad.    

ARGUMENTS PASSED:
        pad     -   pointer to GstPad
        query   -   pointer to GstQuery        
            
RETURN VALUE:
        TRUE    -   success
        FALSE   -   failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/

static const GstQueryType *gst_mpg2demuxer_get_src_query_types(GstPad * pad)
{
    static const GstQueryType types[] = {
	GST_QUERY_DURATION,
	0
    };

    return types;
}

/*=============================================================================
FUNCTION:   mfw_gst_mpg2demuxer_handle_sink_query   

DESCRIPTION:    performs query on src pad.    

ARGUMENTS PASSED:
        pad     -   pointer to GstPad
        query   -   pointer to GstQuery        
            
RETURN VALUE:
        TRUE    -   success
        FALSE   -   failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/

static gboolean mfw_gst_mpg2demuxer_handle_src_query(GstPad * pad,
						    GstQuery * query)
{
    MFW_GST_MPG2DEMUX_INFO_T *demuxer_info;
    gboolean res = TRUE;
    demuxer_info = MFW_GST_MPG2_DEMUXER(gst_pad_get_parent(pad));

    GstFormat format;
    format = GST_FORMAT_TIME;
    gint64 dest_value;

    //g_print("handling %s query\n",
	//      gst_query_type_get_name(GST_QUERY_TYPE(query)));

    switch (GST_QUERY_TYPE(query)) {
    case GST_QUERY_DURATION:
	{
		
	    /* save requested format */
	    gst_query_parse_duration(query, &format, NULL);
	  	    

	    if (format == GST_FORMAT_TIME) {

		gst_query_set_duration(query, GST_FORMAT_TIME,
				       demuxer_info->segment.duration);
	    } else if (format == GST_FORMAT_BYTES) {
		gst_query_set_duration(query, GST_FORMAT_BYTES,
				       demuxer_info->segment.duration);

	    }

	    res = TRUE;
	    break;
	}    
    default:
	res = FALSE;

    }
    gst_object_unref(demuxer_info);
    return res;

}

/*=============================================================================
FUNCTION:   	mfw_gst_mpg2demuxer_handle_src_event

DESCRIPTION:	Handles an event on the source pad.

ARGUMENTS PASSED:
        pad        -    pointer to pad
        event      -    pointer to event
RETURN VALUE:
        TRUE       -	if event is sent to source properly
	    FALSE	   -	if event is not sent to source properly

PRE-CONDITIONS:    None

POST-CONDITIONS:   None

IMPORTANT NOTES:   None
=============================================================================*/

static gboolean mfw_gst_mpg2demuxer_handle_src_event(GstPad* src_pad, GstEvent* event)
{
    gboolean res = TRUE;

	MFW_GST_MPG2DEMUX_INFO_T *demuxer_info = MFW_GST_MPG2_DEMUXER(GST_PAD_PARENT (src_pad));
 
	switch (GST_EVENT_TYPE (event)) 
	{
		case GST_EVENT_SEEK:
		{
			GstSeekFlags flags;
	  	    gboolean     flush;
			gdouble      rate;
			GstSeekType  cur_type;
	        GstSeekType	 stop_type;
	        gint64       cur;
	        GstFormat    format;
	        gint64       stop;
           
            GST_DEBUG("Handling the seek event\n");

			gst_event_ref (event);

            
            if ((demuxer_info->is_seeking== TRUE) && ((demuxer_info->srcpad[AUDIO_INDEX]) && (demuxer_info->srcpad[VIDEO_INDEX])))
			 {
				 
				 gst_event_unref (event);
				 return TRUE;
			 }
            demuxer_info->is_seeking= TRUE;

			/* parse a seek event */
            gst_event_parse_seek(event, &rate, &format, &flags,
			&cur_type, &cur, &stop_type, &stop);
            /* demuxer cannot do seek if format is not bytes */
            if ((format != GST_FORMAT_TIME))
            {
                GST_WARNING("SEEK event not TIME format.\n");
                gst_event_unref (event);
                return TRUE;
 
            }

			/* if the seek time is greater than the stream duration */
            if(cur > demuxer_info->segment.duration) {
                GST_WARNING("SEEK event exceed the maximum duration.\n");
                cur = demuxer_info->segment.duration;
            }  
            g_print("duration %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(demuxer_info->segment.duration));

            g_print("original seeking to %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(cur));

            g_print("base %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(demuxer_info->baseVTS));

            GET_SEEK_TIME_FROM_OFFSET(cur, demuxer_info->baseVTS, demuxer_info->segment.duration);
            
            g_print("new seeking to %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(cur));
            
            flush = flags & GST_SEEK_FLAG_FLUSH;


		    if(flush)
            {
                guint strem_count;

				/* sends an event to upstream elements with event as new flush start event*/
                res = gst_pad_push_event (demuxer_info->sinkpad, gst_event_new_flush_start ());
                if (!res) 
				{
                    GST_WARNING_OBJECT (demuxer_info, "Failed to push event upstream!");
                }
                
                for(strem_count = 0; strem_count < MAX_SRC_PADS; strem_count++)
                {
                    if (demuxer_info->srcpad[strem_count]){
                        /* sends an event to downstream elements with event as new flush start event*/                   
                        res = gst_pad_push_event (demuxer_info->srcpad[strem_count], 
    						                      gst_event_new_flush_start());
                        if (!res)
    					{
                            GST_WARNING_OBJECT (demuxer_info, "Failed to push event downstream!");
                        }
                    }
                }
            }

			

            gst_pad_pause_task (demuxer_info->sinkpad);

			/*Lock the stream lock of sinkpad*/
			GST_PAD_STREAM_LOCK(demuxer_info->sinkpad);
            
			/* perform seek */

            res = MPG2SeekToTimeStamp(demuxer_info, cur);

			if(res == FALSE)
			{
              GST_ERROR("Failed in demuxer seek !!\n");

			   /*Unlock the stream lock of sink pad*/	
              GST_PAD_STREAM_UNLOCK(demuxer_info->sinkpad);
			  return FALSE;
			}
           
			demuxer_info->seek_flag=TRUE;

			demuxer_info->new_seg_flag_audio=TRUE;
			demuxer_info->new_seg_flag_video=TRUE;
            
			/* sends an event to upstream elements with event as new flush stop event
			   , to resume data flow */
			res = gst_pad_push_event (demuxer_info->sinkpad, gst_event_new_flush_stop ());
   	        
            if(res == FALSE)
			{
              GST_WARNING_OBJECT (demuxer_info, "Failed to push event upstream!");
			}
			
            
			 /* send flush stop to down stream elements,to enable data flow*/
            if(flush)
			{
              guint strem_count;
              for(strem_count = 0; strem_count < MAX_SRC_PADS; strem_count++)
              {
                   if (demuxer_info->srcpad[strem_count]){       
                      res = gst_pad_push_event (demuxer_info->srcpad[strem_count],
    					                      gst_event_new_flush_stop());
                      if (!res) 
    				  {
                          GST_WARNING_OBJECT (demuxer_info, "Failed to push event downstream!");
                      }
                   }
              }
          }
         
		 /*Unlock the stream lock of sink pad*/	
         GST_PAD_STREAM_UNLOCK(demuxer_info->sinkpad);

          /* streaming can continue now */
          gst_pad_start_task (demuxer_info->sinkpad,
              (GstTaskFunction)mfw_gst_mpg2demuxer_taskfunc,demuxer_info);

          break;
		}

		default:
		res = FALSE;
		break;
	}
		
        demuxer_info->is_seeking= FALSE;
	gst_event_unref (event);
	return TRUE;
}

static gboolean MPG2SeekToTimeStamp(MFW_GST_MPG2DEMUX_INFO_T* demuxer, guint64 timetoseek)
{
    FSL_MPG_DEMUX_RET_TYPE_T Err ;
    gint ret;
    guint64 bytepos;
    if (demuxer->segment.duration){
        bytepos = (guint64)(((gfloat)(timetoseek))*((gfloat)(demuxer->file_info.length))/((gfloat)demuxer->segment.duration));
    }else{
        GST_ERROR("duration is zero, can not seek\n");
        return FALSE;
    }
    demuxer->findHead = 0;
    if (bytepos>MPG2_CHECKIFRAME_BACKWARD_LEN){
        bytepos-=MPG2_CHECKIFRAME_BACKWARD_LEN;
    }else{
        bytepos = 0;
    }
    
    demuxer->seektime = timetoseek;
    demuxer->syncstart = bytepos;
    if (demuxer->srcpad[demuxer->video_pad_idx]){
        MPG2_SWITCH_OPMODE(demuxer, OP_PROBE_I_FRAME);
    }else{//AUDIO ONLY
        MPG2_SWITCH_OPMODE(demuxer, OP_PROBE_AUDIO);
    }
    MPG2ResyncDemuxer(demuxer);

    return TRUE;
}

/*=============================================================================
FUNCTION:            mfw_gst_mpg2_demuxer_close
        
DESCRIPTION:         This function  Frees All Memories Allocated and Resourses. 

ARGUMENTS PASSED:    None
        
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

void mfw_gst_mpg2_demuxer_close(MFW_GST_MPG2DEMUX_INFO_T *mpg2_demuxer)
{
    guint32 stream_count;

    GST_DEBUG("Freeing all the allocated memories()\n");

    if (mpg2_demuxer->parser_object){
        FSL_MPG_DEMUX_Close(mpg2_demuxer->parser_object);

    }
    MPG2RecycleAllInBuffer(mpg2_demuxer);

    if (mpg2_demuxer->inbuf_object){
        MM_FREE(mpg2_demuxer->inbuf_object);
        mpg2_demuxer->inbuf_object = NULL;
    }

    if (mpg2_demuxer->vAdapter){
        gst_adapter_clear(mpg2_demuxer->vAdapter);
        g_object_unref(mpg2_demuxer->vAdapter);
        mpg2_demuxer->vAdapter = NULL;
    }
    
    if (mpg2_demuxer->aAdapter){
        gst_adapter_clear(mpg2_demuxer->aAdapter);
        g_object_unref(mpg2_demuxer->aAdapter);
        mpg2_demuxer->aAdapter = NULL;
    }
    
    if (mpg2_demuxer->parser_object){
        MM_FREE(mpg2_demuxer->parser_object);
        mpg2_demuxer->parser_object = NULL;
    }
    
    return;

}
/*=============================================================================
FUNCTION: mfw_gst_mpg2_demuxer_set_property

DESCRIPTION: sets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property set by the application
        pspec      - pointer to the attributes of the property

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/

static void
mfw_gst_mpg2_demuxer_set_property(GObject * object, guint prop_id,
				   const GValue * value,
				   GParamSpec * pspec)
{

    MFW_GST_MPG2DEMUX_INFO_T *demuxer_info;
    demuxer_info = MFW_GST_MPG2_DEMUXER(object);

    switch (prop_id) {
    case AUDIO_TARCK_NUM:
	demuxer_info->audio_track_idx = g_value_get_int(value);
	g_print("audio_track_idx=%d\n", demuxer_info->audio_track_idx);
	break;
    case VIDEO_TARCK_NUM:
	demuxer_info->video_track_idx = g_value_get_int(value);
	g_print("video_track_idx=%d\n", demuxer_info->video_track_idx);
	break;

    default:
	G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	break;
    }

}

/*=============================================================================
FUNCTION: mfw_gst_mpg2_demuxer_get_property

DESCRIPTION: gets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property to be set for the next element
        pspec      - pointer to the attributes of the property

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/
static void
mfw_gst_mpg2_demuxer_get_property(GObject * object, guint prop_id,
				   GValue * value, GParamSpec * pspec)
{

    MFW_GST_MPG2DEMUX_INFO_T *demuxer_info;
    demuxer_info = MFW_GST_MPG2_DEMUXER(object);

    switch (prop_id) {
    case AUDIO_TARCK_NUM:
	g_value_set_int(value,demuxer_info->audio_track_idx);
    MPG2DemuxerActivePads(demuxer_info);
	GST_DEBUG("audio_track_idx=%d\n", demuxer_info->audio_track_idx);
	break;
    case VIDEO_TARCK_NUM:
	g_value_set_int(value,demuxer_info->video_track_idx);
    MPG2DemuxerActivePads(demuxer_info);

	GST_DEBUG("video_track_idx=%d\n", demuxer_info->video_track_idx);
	break;

    default:
	G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	break;
    }

}

/*=============================================================================
FUNCTION:            mfw_gst_type_mpg2_demuxer_get_type
        
DESCRIPTION:         intefaces are initiated in this function.you can register one
                     or more interfaces  after having registered the type itself.


ARGUMENTS PASSED:    None
        
  
RETURN VALUE:        A numerical value ,which represents the unique identifier of this
                     element(mpg2demuxer)

        

PRE-CONDITIONS:      None
   	 
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

GType mfw_gst_type_mpg2_demuxer_get_type(void)
{
    static GType mpg2demuxer_type = 0;

    if (!mpg2demuxer_type) {
	static const GTypeInfo mpg2demuxer_info = {
	    sizeof(MFW_GST_MPG2DEMUX_INFO_CLASS_T),
	    (GBaseInitFunc) mfw_gst_mpg2_demuxer_base_init,
	    NULL,
	    (GClassInitFunc) mfw_gst_mpg2_demuxer_class_init,
	    NULL,
	    NULL,
	    sizeof(MFW_GST_MPG2DEMUX_INFO_T),
	    0,
	    (GInstanceInitFunc) mfw_gst_mpg2_demuxer_init,
	};
	mpg2demuxer_type = g_type_register_static(GST_TYPE_ELEMENT,
						 "mfw_mpg2demuxer",
						 &mpg2demuxer_info, 0);

	GST_DEBUG_CATEGORY_INIT(mfw_gst_mpg2demuxer_debug, "mfw_mpg2demuxer",
				0, "mpg2 demuxer");
    }
    return mpg2demuxer_type;
}

/*=============================================================================
FUNCTION:            mfw_gst_mpg2_demuxer_base_init    
        
DESCRIPTION:         mpg2demuxer element details are registered with the plugin during
                     _base_init ,This function will initialise the class and child
                     class properties during each new child class creation

ARGUMENTS PASSED:
        Klass   -    pointer to mpg2demuxer plug-in class
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

static void
mfw_gst_mpg2_demuxer_base_init(MFW_GST_MPG2DEMUX_INFO_CLASS_T * klass)
{

    GST_DEBUG("Registering the element details with the plugin\n");

    static GstElementDetails mfw_gst_mpg2demuxer_details = {
	"Freescale MPEG demuxer",
	"Codec/Demuxer",
	"Demultiplex an mpg2 data into audio and video data",
	FSL_GST_MM_PLUGIN_AUTHOR
    };

    GstElementClass *element_class = GST_ELEMENT_CLASS(klass);
    gst_element_class_add_pad_template(element_class, audio_src_templ());
	gst_element_class_add_pad_template(element_class, video_src_templ());

    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&mfw_gst_mpg2_demuxer_sink_factory));

    gst_element_class_set_details(element_class,
				  &mfw_gst_mpg2demuxer_details);

    return;
}

/*=============================================================================
FUNCTION:            mfw_gst_mpg2_demuxer_class_init
        
DESCRIPTION:         Initialise the class only once (specifying what signals,
                     arguments and virtual functions the class has and setting up
                     global state)

ARGUMENTS PASSED:    klass   - pointer to mpg2demuxer element class
        
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

static void
mfw_gst_mpg2_demuxer_class_init(MFW_GST_MPG2DEMUX_INFO_CLASS_T * klass)
{
    GObjectClass *gobject_class = NULL;
    GstElementClass *gstelement_class = NULL;

    GST_DEBUG("Initialise the demuxer class\n");

    klass->track_signal_id = g_signal_new ("track_id", G_TYPE_FROM_CLASS (klass), 
                    G_SIGNAL_RUN_LAST,
                    0, NULL, NULL,
                    gst_marshal_VOID__POINTER, G_TYPE_NONE, 1, G_TYPE_POINTER);

    gobject_class = (GObjectClass *) klass;
    gstelement_class = (GstElementClass *) klass;
    parent_class = g_type_class_peek_parent(klass);

    gstelement_class->change_state = mfw_gst_mpg2demuxer_change_state;

    gobject_class->set_property = mfw_gst_mpg2_demuxer_set_property;
    gobject_class->get_property = mfw_gst_mpg2_demuxer_get_property;



    g_object_class_install_property(gobject_class, AUDIO_TARCK_NUM,
				    g_param_spec_int("audio-track",
						       "audio_track",
						       "set the audio track number to output, start from 0...",
						       0,
						       NO_MEDIA_SUPPORT_MAX, 0,
						       G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, VIDEO_TARCK_NUM,
				    g_param_spec_int("video-track",
						       "video_track",
						       "set the video track number to output, start from 0...",
						       0,
						       NO_MEDIA_SUPPORT_MAX, 0,
						       G_PARAM_READWRITE));

    return;
}

/*=============================================================================
FUNCTION:            mfw_gst_mpg2_demuxer_init
        
DESCRIPTION:         This function creates the pads on the elements and register the
			         function pointers which operate on these pads. 

ARGUMENTS PASSED:    
      demuxer_info  :pointer to the mpg2demuxer plugin structure.
	  gclass        :pointer to mpg2demuxer element class.
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None 
   	  
 
POST-CONDITIONS:     None
 

IMPORTANT NOTES:     None
   	   
=============================================================================*/
static void
mfw_gst_mpg2_demuxer_init(MFW_GST_MPG2DEMUX_INFO_T * demuxer_info,
			 MFW_GST_MPG2DEMUX_INFO_CLASS_T * gclass)
{

    gchar *padname = NULL;
    GstCaps *caps = NULL;
    GstPadTemplate *templ = NULL;
    GstPad *pad = NULL;
    gboolean set;

    GstElementClass *klass = GST_ELEMENT_GET_CLASS(demuxer_info);

    demuxer_info->track_signal_id = 
        ((MFW_GST_MPG2DEMUX_INFO_CLASS_T *)klass)->track_signal_id;

    demuxer_info->sinkpad =
	gst_pad_new_from_template(gst_element_class_get_pad_template
				  (klass, "sink"), "sink");

    GST_DEBUG("Register the function pointers on to the sink pad\n");

    gst_pad_set_setcaps_function(demuxer_info->sinkpad,
				 mfw_gst_mpg2_demuxer_set_caps);

    gst_element_add_pad(GST_ELEMENT(demuxer_info), demuxer_info->sinkpad);

    gst_pad_set_event_function(demuxer_info->sinkpad,
			       GST_DEBUG_FUNCPTR
			       (mfw_gst_mpg2demuxer_sink_event));

#define MFW_GST_MPG_DEMUXER_PLUGIN VERSION
    PRINT_CORE_VERSION(FSL_MPG_DEMUX_Version_Info());
    PRINT_PLUGIN_VERSION(MFW_GST_MPG_DEMUXER_PLUGIN);

    demuxer_info->audio_track_idx = 0;
    demuxer_info->video_track_idx = 0;

    demuxer_info->audio_track_num = 0;
    demuxer_info->video_track_num = 0;
    demuxer_info->parser_object  = NULL;

    demuxer_info->lock_mutex = g_mutex_new();
    
    return;
}

/*=============================================================================
FUNCTION:            mfw_gst_mpg2_demuxer_set_caps
         
DESCRIPTION:         this function handles the link with other plug-ins and used for
                     capability negotiation  between pads  

ARGUMENTS PASSED:    
        pad      -   pointer to GstPad
        caps     -   pointer to GstCaps
        
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

static gboolean mfw_gst_mpg2_demuxer_set_caps(GstPad * pad, GstCaps * caps)
{


    return TRUE;
}

/*=============================================================================
FUNCTION:            plugin_init
        
ESCRIPTION:          special function , which is called as soon as the plugin or
                     element is loaded and information returned by this function
                     will be cached in central registry


ARGUMENTS PASSED:    
          plugin -   pointer to container that contains features loaded
                     from shared object module
        
  
RETURN VALUE:        return TRUE or FALSE depending on whether it loaded initialized any
                     dependency correctly
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	   

IMPORTANT NOTES:     None
   	    
=============================================================================*/


static gboolean plugin_init(GstPlugin * plugin)
{
    return gst_element_register(plugin, "mfw_mpg2demuxer",
				FSL_GST_RANK_HIGH,
				MFW_GST_TYPE_MPG2_DEMUXER);
}

/*   This is used to define the entry point and meta data of plugin   */

GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,	/* major version of Gstreamer  */
		  GST_VERSION_MINOR,	/* minor version of Gstreamer  */
		  "mfw_mpg2demuxer",	/* name of the plugin          */
		  "demuxes video and audio streams from mpg2file",	/* what plugin actually does   */
		  plugin_init,	/* first function to be called */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  FSL_GST_MM_PLUGIN_PACKAGE_NAME, FSL_GST_MM_PLUGIN_PACKAGE_ORIG)

