/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */


#define ROW_OFFSET    5
#define ROW_HEIGHT    40

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <gio/gio.h>

#include <clutter/clutter.h>
#include <cogl/cogl.h>

#include <nbtk/nbtk.h>

#include <bognor/br-queue.h>

#include "hrn.h" /* FIXME */
#include "hrn-drag-n-drop.h"
#include "hrn-queue.h"
#include "hrn-source-manager.h"
#include "hrn-texture-cache.h"
#include "hrn-queue-header.h"
#include "hrn-popup.h"

#define    QUEUE_TIP_HEIGHT    47  /* used to determine drops on tip */


G_DEFINE_TYPE (HrnQueue, hrn_queue, NBTK_TYPE_BIN);

#define HRN_QUEUE_GET_PRIVATE(obj)                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                HRN_TYPE_QUEUE, \
                                HrnQueuePrivate))

struct _HrnQueuePrivate
{
    BrQueue *br_queue;
    HrnSourceManager *manager;

    ClutterActor *top_group;
    NbtkWidget     *toptable;
    NbtkWidget     *header;
    NbtkWidget     *table;
    NbtkWidget     *view;
    NbtkWidget     *scrollview;
    ClutterGroup   *children;
    NbtkWidget     *playpause;
    NbtkWidget *highlight;

    guint   reposition_id;

    BklItem    *active_audio;

    /* FIXME: Queue needs to be a GQueue */
    GList *queue;
};

typedef struct QueueItem
{
  HrnQueue     *queue;
  BklItem      *item;
  ClutterActor *actor;
}
QueueItem;

static GObject * hrn_queue_constructor (GType                  type,
                                        guint                  n_params,
                                        GObjectConstructParam *params);
static void hrn_queue_dispose          (GObject               *object);
static void hrn_queue_allocate         (ClutterActor          *self,
                                        const ClutterActorBox *box,
                                        ClutterAllocationFlags flags);
static void hrn_queue_reposition       (HrnQueue              *queue);
static void add_item                   (HrnQueue              *queue,
                                        QueueItem             *queue_item);

static void
hrn_queue_class_init (HrnQueueClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->dispose      = hrn_queue_dispose;
  gobject_class->constructor  = hrn_queue_constructor;

  actor_class->allocate       = hrn_queue_allocate;

  g_type_class_add_private (gobject_class, sizeof (HrnQueuePrivate));
}

static void
hrn_queue_init (HrnQueue *self)
{
  self->priv = HRN_QUEUE_GET_PRIVATE (self);
}

static void
expanded_clicked_cb (HrnQueueHeader *header,
                     gboolean        expanded,
                     HrnQueue       *queue)
{
  HrnQueuePrivate *priv = queue->priv;

  if (expanded) {
      clutter_actor_animate ((ClutterActor *) queue,
                             CLUTTER_EASE_IN_OUT_CUBIC, 350,
                             "y", -200.0,
                             "height", 362.0,
                             NULL);
      clutter_actor_show (CLUTTER_ACTOR (priv->scrollview));
  } else {
      clutter_actor_animate ((ClutterActor *) queue,
                             CLUTTER_EASE_IN_OUT_CUBIC, 350,
                             "y", 0.0,
                             "height", 74.0,
                             "signal-swapped::completed",
                             clutter_actor_hide, priv->scrollview,
                             NULL);
    }
}

#if 0
static gboolean
playpause_clicked_cb (NbtkButton *button, HrnQueue     *queue)
{
  hrn_queue_set_playing (queue, nbtk_button_get_checked (button));
  return TRUE;
}
#endif

static gboolean
active_audio_drop (ClutterActor *actor,
                   ClutterActor *dropped_on,
                   gint          x,
                   gint          y,
                   gpointer      queue)
{
  BklItem *active_audio = hrn_queue_get_active_audio (queue);

  if (active_audio)
    {
      if (!dropped_on)
        {
          hrn_queue_set_active_audio (queue, NULL, "");
          return TRUE;
        }
      else
        return hrn_item_drop (actor, dropped_on, x, y, active_audio);
    }
  return FALSE;
}


static gboolean
hrn_item_can_drop_active_audio (ClutterActor *actor,
                                ClutterActor *dropped_on,
                                gint          x,
                                gint          y,
                                gpointer      queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (!priv->active_audio)
    return FALSE;
  return hrn_item_can_drop (actor, dropped_on, x, y, queue);
}

static void
empty_queue (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_clear (priv->br_queue);
  hrn_popup_close ();
}

static gint shuffle_sort (gconstpointer a,
                          gconstpointer b)
{
  return g_random_int_range (0, 52)<26;
}

static void
randomize_queue (HrnQueue *queue)
{
  gint i;
  gint l = hrn_queue_get_length (queue);

  GList *iter, *list = NULL;

  /* create a copy of all the items in the queue */
  for (i = 0; i < l; i++)
    {
      BklItem *item = hrn_queue_get (queue, i);
      list = g_list_prepend (list, item);
      g_object_ref (item);
    }

  empty_queue (queue);

  /* we shuffle the list by sorting it*/
  for (i = 0; i < 3; i++)
     list = g_list_sort (list, shuffle_sort);

  /* reinsert the queue items by appending all */
  for (iter = list; iter; iter = iter->next)
    {
      hrn_queue_append (queue, iter->data);
      g_object_unref (iter->data);
    }

  g_list_free (list);

  hrn_popup_close ();
}

#define QUEUE_WIDTH         190
#define QUEUE_WIDTH2        200 /* including padding */
#define QUEUE_ITEM_WIDTH    QUEUE_WIDTH - 40 /* minus interior padding to avoid
                                                horizontal scroll */

#define QUEUE_HEADER_HEIGHT 74

static gboolean
queue_release (ClutterActor *actor,
               ClutterEvent *event,
               gpointer      userdata)
{
  if (event->button.button == 3)
    {
      gpointer actions[] = {
        N_("Empty"),   empty_queue,
        N_("Shuffle"), randomize_queue,
        NULL
      };
      hrn_popup_actor (event->button.x, event->button.y,
                       hrn_popup_actions (actions, actor));
      return TRUE;
    }
  return FALSE;
}

static void
progress_notify (HrnQueueHeader *header,
                 double          position,
                 HrnQueue       *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_set_position (priv->br_queue, position);
}

/* only paint the children that are actually visible */
static void
hrn_queue_list_paint (ClutterActor *actor,
                      gpointer      userdata)
{
  GList         *child_item, *children;
  gint           sh;

  NbtkScrollBar *scrollbar =
    NBTK_SCROLL_BAR (nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (
                clutter_actor_get_parent (clutter_actor_get_parent (actor)))));
  gfloat scroll = nbtk_adjustment_get_value (
    nbtk_scroll_bar_get_adjustment (scrollbar));


  sh = clutter_actor_get_height (clutter_stage_get_default ());

  children = clutter_container_get_children (CLUTTER_CONTAINER (actor));

  for (child_item = children;
       child_item != NULL;
       child_item = child_item->next)
    {
      ClutterActor *child = child_item->data;

      g_assert (child != NULL);

      if (CLUTTER_ACTOR_IS_VISIBLE (child))
        {
          gint y;
          y = clutter_actor_get_y (child);
          if (y - scroll > -64 &&
              y - scroll < 300)
            clutter_actor_paint (child);
        }
    }
  g_list_free (children);

  g_signal_stop_emission_by_name (actor, "paint");
}

static GObject *
hrn_queue_constructor (GType                  type,
                       guint                  n_params,
                       GObjectConstructParam *params)
{
  HrnQueuePrivate *priv;
  GObject         *object;
  HrnQueue        *queue;
  NbtkAdjustment  *vadj;

  object = G_OBJECT_CLASS (hrn_queue_parent_class)->constructor (type,
                                                                 n_params,
                                                                 params);

  clutter_actor_set_reactive (CLUTTER_ACTOR (object), TRUE);
  g_object_set_data (object, "HRN_DROP_MASK",
                     GINT_TO_POINTER (HRN_DROP_MASK_QUEUE));
  g_signal_connect (object, "button-release-event",
                    G_CALLBACK (queue_release), NULL);

  queue = HRN_QUEUE (object);
  priv  = HRN_QUEUE_GET_PRIVATE (queue);

  priv->top_group = clutter_group_new ();
  clutter_container_add_actor ((ClutterContainer *) queue, priv->top_group);

#if 0
  priv->playpause = nbtk_button_new ();
  nbtk_button_set_toggle_mode (NBTK_BUTTON (priv->playpause), TRUE);
  nbtk_widget_set_style_class_name (priv->playpause, "HrnPlayPause");
  clutter_actor_set_size (CLUTTER_ACTOR (priv->playpause), 37, 45);
  priv->play_pause_id = g_signal_connect (priv->playpause, "clicked",
                                          G_CALLBACK (playpause_clicked_cb),
                                          queue);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->playpause), 0, 2,
                                        "row-span", 2,
                                        "col-span", 1,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "y-fill", FALSE,
                                        "y-expand", FALSE,
                                        "x-fill", FALSE,
                                        "x-expand", FALSE,
                                        NULL);
#endif

  priv->header = g_object_new (HRN_TYPE_QUEUE_HEADER, NULL);
  g_signal_connect (priv->header, "expand",
                    G_CALLBACK (expanded_clicked_cb), queue);
  g_signal_connect (priv->header, "progress",
                    G_CALLBACK (progress_notify), queue);
  clutter_container_add_actor ((ClutterContainer *) priv->top_group,
                               (ClutterActor *) priv->header);

  priv->scrollview = nbtk_scroll_view_new ();
  nbtk_widget_set_style_class_name (priv->scrollview, "HrnQueueView");
  clutter_container_add_actor ((ClutterContainer *) priv->top_group,
                               (ClutterActor *) priv->scrollview);
  /* FIXME: Add padding */
  clutter_actor_set_position ((ClutterActor *) priv->scrollview,
                              0, QUEUE_HEADER_HEIGHT);

  priv->view = (NbtkWidget *) nbtk_viewport_new ();
  clutter_container_add (CLUTTER_CONTAINER (priv->scrollview),
                         CLUTTER_ACTOR (priv->view), NULL);
  nbtk_scrollable_get_adjustments (NBTK_SCROLLABLE (priv->view), NULL, &vadj);
  g_object_set (vadj,
                "step-increment", (double) ROW_HEIGHT,
                "page-increment", (double) ROW_HEIGHT,
                NULL);

  priv->children = CLUTTER_GROUP (clutter_group_new ());
  g_signal_connect (priv->children, "paint",
                    G_CALLBACK (hrn_queue_list_paint), NULL);
  clutter_container_add (CLUTTER_CONTAINER (priv->view),
                         CLUTTER_ACTOR (priv->children), NULL);

  priv->highlight = nbtk_label_new ("");
  clutter_actor_set_size (CLUTTER_ACTOR (priv->highlight),
                          QUEUE_ITEM_WIDTH, 45);
  clutter_actor_set_anchor_point_from_gravity (CLUTTER_ACTOR (priv->highlight),
                                               CLUTTER_GRAVITY_WEST);
  nbtk_widget_set_style_class_name (priv->highlight, "HrnQueueHighlight");
  clutter_container_add (CLUTTER_CONTAINER (priv->children),
                         CLUTTER_ACTOR (priv->highlight), NULL);
  clutter_actor_hide (CLUTTER_ACTOR (priv->highlight));

  /* hack to sync up size */
  {
    clutter_actor_set_size (CLUTTER_ACTOR (priv->header),
                            QUEUE_WIDTH2, QUEUE_HEADER_HEIGHT);
    clutter_actor_set_size (CLUTTER_ACTOR (object),
                            QUEUE_WIDTH2, 260);
    clutter_actor_set_size (CLUTTER_ACTOR (priv->scrollview),
                            QUEUE_WIDTH2, 200);
  }

  hrn_queue_hide_audio_progress (queue);

  return object;
}

static void
hrn_queue_allocate (ClutterActor *self, const ClutterActorBox *box,
                    ClutterAllocationFlags flags)
{
  HrnQueuePrivate   *priv = HRN_QUEUE_GET_PRIVATE (self);
  ClutterActorClass *parent_class;
  ClutterActorBox    actor_box;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_queue_parent_class);
  parent_class->allocate (self, box, flags);

  actor_box    = *box;
  actor_box.y2 = box->y2 - box->y1;
  actor_box.y1 = 0;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->top_group), &actor_box, flags);

#if 0
  actor_box.y1 = 60;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->scrollview), &actor_box, flags);
#endif
}

static void
hrn_queue_dispose (GObject *object)
{
  HrnQueuePrivate   *priv = HRN_QUEUE_GET_PRIVATE (object);

  if (priv->reposition_id)
    {
      g_source_remove (priv->reposition_id);
      priv->reposition_id = 0;
    }

  G_OBJECT_CLASS (hrn_queue_parent_class)->dispose (object);
}

static void
brq_uri_added (BrQueue    *brq,
               const char *uri,
               int         position,
               HrnQueue   *queue)
{
    HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
    BklItem         *item;

    item = hrn_source_manager_get_item_for_uri (priv->manager, uri, NULL);
    if (item) {
        QueueItem *queue_item = g_new0 (QueueItem, 1);

        queue_item->item  = g_object_ref (item);
        queue_item->queue = queue;

        priv->queue = g_list_insert (priv->queue, queue_item, position);

        add_item (queue, queue_item);

        hrn_queue_header_expand ((HrnQueueHeader *) priv->header, TRUE);
    }
}

static void
brq_uri_removed (BrQueue    *brq,
                 const char *uri,
                 int         position,
                 HrnQueue   *queue)
{
    HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
    GList           *queue_list;
    QueueItem       *queue_item;

    queue_list = g_list_nth (priv->queue, position);
    if (queue_list == NULL) {
        return;
    }

    queue_item = queue_list->data;

    priv->queue = g_list_remove_link (priv->queue, queue_list);
    g_list_free (queue_list);

    g_object_unref (queue_item->item);
    clutter_actor_destroy (queue_item->actor);
    g_free (queue_item);

    hrn_queue_reposition (queue);
}

static void
brq_position_changed (BrQueue  *brq,
                      double    position,
                      HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  hrn_queue_header_set_progress ((HrnQueueHeader *) priv->header, position);
}

static void
brq_now_playing_changed (BrQueue     *brq,
                         const char  *uri,
                         HrnQueue *queue)
{
  HrnQueuePrivate *priv = queue->priv;
  BklItem *item;

  /* Reset the position of the appropriate slider to 0 */
  brq_position_changed (brq, 0.0, queue);

  if (uri == NULL || *uri == '\0') /* finished playback of this media type */
    {
        hrn_queue_header_show_progress ((HrnQueueHeader *) priv->header, FALSE);
        hrn_queue_set_active_audio (queue, NULL, "");
        return;
    }

  item = hrn_source_manager_get_item_for_uri (priv->manager, uri, NULL);
  hrn_queue_set_active_audio (queue, item, uri);
  hrn_queue_header_show_progress ((HrnQueueHeader *) priv->header, TRUE);
}

static void
set_name_reply (BrQueue *brq,
                char    *name,
                GError  *error,
                gpointer data)
{
}

static void
list_uris_reply (BrQueue *brq,
                 char   **uris,
                 GError  *error,
                 gpointer data)
{
  HrnQueue        *queue = (HrnQueue *) data;
  HrnQueuePrivate *priv  = queue->priv;
  int              i;

  if (error != NULL)
    {
      g_warning ("Error listing queue uris: %s", error->message);
      return;
    }

  for (i = 0; uris[i]; i++)
    {
      BklItem *item = hrn_source_manager_get_item_for_uri (priv->manager,
                                                           uris[i], NULL);
      if (item)
        {
          QueueItem *queue_item = g_new0 (QueueItem, 1);

          queue_item->item  = g_object_ref (item);
          queue_item->queue = queue;

          priv->queue = g_list_append (priv->queue, queue_item);

          add_item (queue, queue_item);

          hrn_queue_header_expand ((HrnQueueHeader *) priv->header, TRUE);
        }
    }
}

static void
get_now_playing_reply (BrQueue *brq,
                       char    *audio_uri,
                       GError  *error,
                       gpointer data)
{
    HrnQueue *queue = (HrnQueue *) data;

    if (error != NULL) {
        g_warning ("Error getting now playing: %s", error->message);
        return;
    }

    brq_now_playing_changed (brq, audio_uri, queue);
}

static void
source_ready_cb (HrnSourceManager *manager,
                 HrnQueue         *queue)
{
    HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

    br_queue_list_uris (priv->br_queue, list_uris_reply, queue);
    br_queue_get_now_playing (priv->br_queue, get_now_playing_reply, queue);
}

HrnQueue *
hrn_queue_new (BrQueue          *br_queue,
               HrnSourceManager *manager)
{
  HrnQueue        *queue = g_object_new (HRN_TYPE_QUEUE, NULL);
  HrnQueuePrivate *priv  = HRN_QUEUE_GET_PRIVATE (queue);

  priv->br_queue = g_object_ref (br_queue);
  g_signal_connect (priv->br_queue, "uri-added",
                    G_CALLBACK (brq_uri_added), queue);
  g_signal_connect (priv->br_queue, "uri-removed",
                    G_CALLBACK (brq_uri_removed), queue);
  g_signal_connect (priv->br_queue, "now-playing-changed",
                    G_CALLBACK (brq_now_playing_changed), queue);
  g_signal_connect (priv->br_queue, "position-changed",
                    G_CALLBACK (brq_position_changed), queue);

  /* Set the name of the queue from Bognor */
  br_queue_get_name (br_queue, set_name_reply, queue);

  priv->manager = manager;

  if (hrn_source_manager_is_ready (manager)) {
      source_ready_cb (manager, queue);
  } else {
      g_signal_connect (manager, "ready",
                        G_CALLBACK (source_ready_cb), queue);
  }

  return queue;
}

void
hrn_queue_set_name (HrnQueue *queue, const gchar *name)
{
}

const gchar *
hrn_queue_get_name (HrnQueue *queue)
{
  g_warning ("FIXME: hrn_queue_get_name");
  return "FIXME:hrn_queue_get_name";
}

gboolean
hrn_item_can_drop (ClutterActor *actor, ClutterActor *dropped_on, gint x,
                   gint y,
                   gpointer item)
{
  ClutterActor    *queue = dropped_on;
  gint             no    = 0;
  HrnQueuePrivate *priv;

  while (queue && !HRN_IS_QUEUE (queue))
    queue = clutter_actor_get_parent (queue);

  if (!queue || !item)
    return FALSE;

  priv = HRN_QUEUE_GET_PRIVATE (queue);

  {
    gfloat         xu, yu;
    gdouble        foo;
    NbtkScrollBar *scrollbar = NBTK_SCROLL_BAR (
      nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scrollview)));
    foo = nbtk_adjustment_get_value (nbtk_scroll_bar_get_adjustment (scrollbar));

    clutter_actor_transform_stage_point (CLUTTER_ACTOR (queue),
                                         x, y, &xu, &yu);

    no = ((floor (yu) + (gint) foo) - ROW_OFFSET) / ROW_HEIGHT - 1;

    if (no == -1)
      return TRUE;

    if (no > g_list_length (priv->queue) - (G_IS_OBJECT (item) ? 0 : 1))
      no = g_list_length (priv->queue) - (G_IS_OBJECT (item) ? 0 : 1);

    clutter_actor_set_y (CLUTTER_ACTOR (priv->highlight), no * ROW_HEIGHT);

    clutter_actor_show (CLUTTER_ACTOR (priv->highlight));
    clutter_actor_set_opacity (CLUTTER_ACTOR (priv->highlight), 0xff);

    clutter_actor_animate (CLUTTER_ACTOR (
                             priv->highlight), CLUTTER_EASE_IN_OUT_CUBIC, 2000,
                           "opacity", 0,
                           "signal-swapped::completed",
                           clutter_actor_hide, priv->highlight,
                           NULL);
  }

  return TRUE;
}


gboolean
hrn_item_drop (ClutterActor *actor, ClutterActor *dropped_on, gint x, gint y,
               gpointer item)
{
  ClutterActor    *queue = dropped_on;
  gint             no    = 0;
  HrnQueuePrivate *priv;

  while (queue && !HRN_IS_QUEUE (queue))
    queue = clutter_actor_get_parent (queue);

  if (!queue || !item)
    return FALSE;

  priv = HRN_QUEUE_GET_PRIVATE (queue);

  {
    gfloat         xu, yu;
    gdouble        foo;
    NbtkScrollBar *scrollbar = NBTK_SCROLL_BAR (
      nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scrollview)));
    foo = nbtk_adjustment_get_value (nbtk_scroll_bar_get_adjustment (scrollbar));

    clutter_actor_transform_stage_point (CLUTTER_ACTOR (queue),
                                         x, y, &xu, &yu);

    no = ((floor (yu) + (gint) foo) - ROW_OFFSET) / ROW_HEIGHT - 1;
    if (yu < QUEUE_TIP_HEIGHT)
      no = -1;
  }

  g_assert (HRN_IS_QUEUE (queue));

  g_warning ("FIXME: hrn_item_drop");
#if 0
  if (hrn_view_has_selected () &&
      hrn_view_is_selected (bkl_item_get_uri (item)))
    {
      GList *s, *selected = hrn_view_get_selected (HRN_VIEW (hrn_view));

      for (s = selected; s; s = s->next)
        {
          hrn_queue_insert (HRN_QUEUE (queue), no, s->data);
          no++;
        }
      g_list_free (selected);
      /* there is a selection */
    }
  else
    {
      hrn_queue_insert (HRN_QUEUE (queue), no, item);
    }
#endif
  return TRUE;
}


gboolean
hrn_cluster_dropped (ClutterActor *actor, ClutterActor *dropped_on, gint x,
                     gint y,
                     gpointer item)
{
  ClutterActor    *queue = dropped_on;
  gint             no    = 0;
  HrnQueuePrivate *priv;

  while (queue && !HRN_IS_QUEUE (queue))
    queue = clutter_actor_get_parent (queue);
  if (!queue)
    {
      return FALSE;
    }

  priv = HRN_QUEUE_GET_PRIVATE (queue);

  {
    gfloat         xu, yu;
    gdouble        foo;
    NbtkScrollBar *scrollbar = NBTK_SCROLL_BAR (
      nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scrollview)));
    foo = nbtk_adjustment_get_value (nbtk_scroll_bar_get_adjustment (scrollbar));

    clutter_actor_transform_stage_point (CLUTTER_ACTOR (queue),
                                         xu, yu, &xu, &yu);

    no = ((floor (yu) + (gint) foo) - ROW_OFFSET) / ROW_HEIGHT - 1;
    if (yu < QUEUE_TIP_HEIGHT)
      no = -1;
  }

  g_assert (HRN_IS_QUEUE (queue));
#if 0
  {
    GList *iter;
    for (iter = HRN_CLUSTER (item)->results; iter; iter = iter->next)
      hrn_queue_insert (HRN_QUEUE (queue), no++, iter->data);
  }
#endif
  return TRUE;
}

static gboolean
hrn_item_queue_start_drag (ClutterActor *actor, gint x, gint y,
                           QueueItem    *queue_item)
{
  clutter_actor_hide (actor);
  hrn_queue_reposition (queue_item->queue);
  return TRUE;
}

static gboolean
from_queue_drop (ClutterActor *actor, ClutterActor *dropped_on, gint x, gint y,
                 QueueItem    *queue_item)
{
  gint             no   = 0;
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue_item->queue);

  if (dropped_on == NULL)
    {
      GList *iter;
      for (iter = priv->queue; iter; iter = iter->next)
        {
          if (iter->data == queue_item)
            break;
          no++;
        }
      hrn_queue_remove (queue_item->queue, no);
      return TRUE;
    }

  {
    gfloat         xu, yu;
    gdouble        foo;
    NbtkScrollBar *scrollbar = NBTK_SCROLL_BAR (
      nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scrollview)));
    foo = nbtk_adjustment_get_value (nbtk_scroll_bar_get_adjustment (scrollbar));

    clutter_actor_transform_stage_point (CLUTTER_ACTOR (queue_item->queue),
                                         x, y, &xu, &yu);

    no = 0;
    {
      GList *iter;
      for (iter = priv->queue; iter; iter = iter->next)
        {
          if (iter->data == queue_item)
            break;
          no++;
        }
    }

    {
      HrnQueue        *queue = HRN_QUEUE (dropped_on);
      HrnQueuePrivate *qp    = HRN_QUEUE_GET_PRIVATE (queue);
      BklItem         *item  = queue_item->item;

      g_object_ref (item);
      hrn_queue_remove (queue_item->queue, no);

      no = ((floor (yu) + (gint) foo) - ROW_OFFSET) / ROW_HEIGHT - 1;
      if (yu < QUEUE_TIP_HEIGHT)
        no = -1;

      if (no == -1)
        {
          br_queue_stop (qp->br_queue);
          br_queue_play_uri (qp->br_queue, bkl_item_get_uri (item),
                             bkl_item_get_mimetype (item));
        }
      else
        {
          hrn_queue_insert (queue, no, item);
        }

      g_object_unref (item);
    }
  }
  return TRUE;
}


static gboolean
hrn_queue_reposition_idle (gpointer queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  GList           *iter;
  gint             no = 0;

  priv->reposition_id = 0;

  for (iter = priv->queue; iter; iter = iter->next)
    {
      QueueItem *item = iter->data;
      if (CLUTTER_ACTOR_IS_VISIBLE (item->actor))
        {
          gfloat x = 7;
          gfloat y = no * ROW_HEIGHT + ROW_OFFSET;
          if (clutter_actor_get_x (item->actor) == 0 &&
              clutter_actor_get_y (item->actor) == 0)

            {
              clutter_actor_set_position (CLUTTER_ACTOR (item->actor), x, y);
            }
          else
            {
              clutter_actor_animate (CLUTTER_ACTOR (item->actor),
                                     CLUTTER_LINEAR, 400,
                                     "x", x, "y", y,
                                     NULL);
            }
          clutter_actor_animate (CLUTTER_ACTOR (item->actor),
                                 CLUTTER_LINEAR, 400,
                                 "opacity", 0xff,
                                 NULL);
          no++;
        }
    }
  return FALSE;
}


static void
hrn_queue_reposition (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (priv->reposition_id != 0)
    return;

  priv->reposition_id = g_idle_add_full (G_PRIORITY_HIGH_IDLE,
                                      hrn_queue_reposition_idle, queue, NULL);
}

static char *
get_title (BklItem *item)
{
    const char *title;

    title = bkl_item_audio_get_title ((BklItemAudio *) item);
    if (title == NULL || *title == '\0') {
        const char *uri = bkl_item_get_uri (item);
        char *basename, *unesc;

        basename = g_path_get_basename (uri);
        unesc = g_uri_unescape_string (basename, NULL);
        g_free (basename);

        return unesc;
    } else {
        return g_strdup (title);
    }
}

static char *
get_artist (BklItem *item)
{
    GPtrArray *artists;

    artists = bkl_item_audio_get_artists ((BklItemAudio *) item);
    if (artists == NULL) {
        return g_strdup (_("Unknown"));
    } else {
        return g_strdup (artists->pdata[0]);
    }
}

static ClutterActor *
get_thumbnail (BklItem *item)
{
    HrnTextureCache *cache = hrn_texture_cache_get_default ();
    const char *uri;

    uri = bkl_item_extended_get_thumbnail ((BklItemExtended *) item);
    if (uri == NULL) {
        return hrn_texture_cache_get_default_texture (cache,
                                                      BKL_ITEM_TYPE_AUDIO);
    } else {
        return hrn_texture_cache_get_texture (cache, uri);
    }
}

static void
add_item (HrnQueue  *queue,
          QueueItem *queue_item)
{
  HrnQueuePrivate *priv          = HRN_QUEUE_GET_PRIVATE (queue);
  BklItem         *item          = queue_item->item;
  ClutterColor     backing_color = { 0xdd, 0xdd, 0xdd, 0x00 };
  ClutterActor    *backing       = clutter_rectangle_new ();
  ClutterActor    *group         = clutter_group_new ();
  ClutterActor *actor;
  char            *title;
  char            *artist;
  NbtkWidget      *text;
  NbtkWidget      *meta_text;
  gfloat           w, h;
  gdouble          scale;

  title = get_title (item);
  text = nbtk_label_new (title);
  nbtk_widget_set_style_class_name (text, "HrnQueueItemLabel");
  g_free (title);

  artist = get_artist (item);
  meta_text = nbtk_label_new (artist);
  nbtk_widget_set_style_class_name (meta_text, "HrnQueueItemMetaLabel");
  g_free (artist);

  actor = get_thumbnail (item);
  clutter_actor_get_size (actor, &w, &h);
  if (w == 0 && h == 0)
    {
      w = h = 36;
      clutter_actor_set_size (actor, 36, 36);
    }

  clutter_rectangle_set_color (CLUTTER_RECTANGLE (backing), &backing_color);
  clutter_actor_set_opacity (backing, 0xbb);

  clutter_actor_set_size (group, QUEUE_ITEM_WIDTH, 36);
  scale = 36.0 / w;
  clutter_actor_set_size (backing, QUEUE_ITEM_WIDTH, 36);
  clutter_actor_set_scale (actor, scale, scale);

  clutter_group_add (CLUTTER_GROUP (group), backing);
  clutter_group_add (CLUTTER_GROUP (group), actor);
  clutter_group_add (CLUTTER_GROUP (group), text);

  clutter_actor_set_width (CLUTTER_ACTOR (text), QUEUE_ITEM_WIDTH - 42);
  clutter_actor_set_position (CLUTTER_ACTOR (text), 42, 2);

  clutter_group_add (CLUTTER_GROUP (group), meta_text);
  clutter_actor_set_width (CLUTTER_ACTOR (meta_text), QUEUE_ITEM_WIDTH - 42);
  clutter_actor_set_position (CLUTTER_ACTOR (meta_text), 42, 2 +
                              clutter_actor_get_height (CLUTTER_ACTOR (text)));
  clutter_actor_set_position (actor, 2, (36 - h * scale) / 2);

  clutter_container_add_actor (CLUTTER_CONTAINER (priv->children),
                               (ClutterActor *) group);

  clutter_actor_set_opacity (group, 0);

  hrn_actor_make_draggable (
    CLUTTER_ACTOR (group),
    group,
    HRN_DROP_MASK_QUEUE,
    G_CALLBACK (hrn_item_queue_start_drag),
    G_CALLBACK (hrn_item_can_drop),
    G_CALLBACK (from_queue_drop),
    queue_item);

  g_object_ref (item);
  queue_item->actor = group;

  clutter_actor_queue_relayout (CLUTTER_ACTOR (queue));
  hrn_queue_reposition (queue);
}


void
hrn_queue_insert (HrnQueue *queue, gint pos, BklItem     *item)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_insert_uri (priv->br_queue, bkl_item_get_uri (item),
                       bkl_item_get_mimetype (item), pos);
}


void
hrn_queue_append (HrnQueue *queue, BklItem  *item)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_add_uri (priv->br_queue, bkl_item_get_uri (item),
                    bkl_item_get_mimetype (item));
}

void
hrn_queue_prepend_list (HrnQueue *queue, GList       *items)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  GList           *iter;

  for (iter = g_list_last (items); iter; iter = iter->prev)
    {
      BklItem *item = iter->data;

      br_queue_insert_uri (priv->br_queue, bkl_item_get_uri (item),
                           bkl_item_get_mimetype (item), 0);
    }

  hrn_queue_header_expand ((HrnQueueHeader *) priv->header, TRUE);
}

gint
hrn_queue_get_length (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  return g_list_length (priv->queue);
}



void
hrn_queue_remove (HrnQueue *queue, gint pos)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_remove (priv->br_queue, pos);
}

BklItem *
hrn_queue_get (HrnQueue *queue, gint pos)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  QueueItem       *item;

  item = g_list_nth_data (priv->queue, pos);
  if (item)
    {
      return item->item;
    }
  else
    {
      return NULL;
    }
}

BklItem     *
hrn_queue_proceed (HrnQueue *queue, gint item_type_mask)
{
  gint len = hrn_queue_get_length (queue);

  if (len)
    {
      gint i;
      for (i = 0; i < len; i++)
        {
          BklItem *ret = hrn_queue_get (queue, i);
          if (bkl_item_get_item_type (ret) & item_type_mask)
            {
              hrn_queue_remove (queue, i);
              return ret;
            }
        }
    }
  return NULL;
}

gboolean
hrn_queue_get_playing (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  return nbtk_button_get_checked (NBTK_BUTTON (priv->playpause));
}

void
hrn_queue_set_playing (HrnQueue *queue, gboolean playing)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  if (playing)
    {
      br_queue_play (priv->br_queue);
    }
  else
    {
      br_queue_stop (priv->br_queue);
    }
}

void
hrn_queue_set_active_audio (HrnQueue *queue,
                            BklItem     *item,
                            const gchar *uri)
{
    HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

    if (priv->active_audio) {
        g_object_unref (priv->active_audio);
    }

    priv->active_audio = NULL;

    if (item) {
        char *title, *artist;
        ClutterActor *thumbnail;

        priv->active_audio = g_object_ref (item);
        title = get_title (item);
        artist = get_artist (item);
        thumbnail = get_thumbnail (item);

        g_object_set (priv->header,
                      "primary", title,
                      "secondary", artist,
                      "thumbnail", thumbnail,
                      NULL);
        g_free (title);
        g_free (artist);
    } else {
        if (uri && uri[0] != '\0') {
            char *basename;
            basename = g_path_get_basename (uri);

            g_object_set (priv->header,
                          "primary", basename,
                          "secondary", "",
                          NULL);
            g_free (basename);
        } else {
            g_object_set (priv->header,
                          "primary", _("Playqueue"),
                          "secondary", "",
                          "thumbnail", NULL,
                          NULL);
/* FIXME: add that later, when string freeze is over */
#if 0
            g_object_set (priv->header,
                          "primary",
                          _("Music Playqueue"),     /* 10 ems ('M's) */
                          "secondary",
                          _("Add music to start"),  /* 11 ems ('M's) */
                          "thumbnail", NULL,
                          NULL);
#endif
        }
    }
}

BklItem     *
hrn_queue_get_active_audio (HrnQueue *queue)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  return priv->active_audio;
}

void
hrn_queue_hide_audio_progress (HrnQueue *queue)
{
}

void
hrn_queue_show_audio_progress (HrnQueue *queue)
{
}

void
hrn_queue_audio_finished (HrnQueue *queue)
{
  hrn_queue_set_active_audio (queue, NULL, "");
  hrn_queue_hide_audio_progress (queue);
}

void
hrn_queue_play_uri_now (HrnQueue    *queue,
                        const gchar *uri)
{
  GFile *file;
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);
  char *mimetype = NULL;

  file = g_file_new_for_commandline_arg (uri);

  /* FIXME: Do we even need to pass a mimetype to BR now? */
  mimetype = hrn_resolve_mimetype (uri);
  br_queue_play_uri (priv->br_queue, g_file_get_uri (file), mimetype);

  g_free (mimetype);
  g_object_unref (file);
}

void
hrn_queue_play_now (HrnQueue *queue,
                    BklItem  *item)
{
  HrnQueuePrivate *priv = HRN_QUEUE_GET_PRIVATE (queue);

  br_queue_play_uri (priv->br_queue, bkl_item_get_uri (item),
                     bkl_item_get_mimetype (item));
}
