/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "hrn.h"
#include <math.h>

void hrn_actor_make_draggable (ClutterActor *actor, ClutterActor *drag_pad,
                               guint mask, GCallback start_drag_cb,
                               GCallback can_drop_cb, GCallback drop_cb,
                               gpointer data);

typedef struct DraggableData
{
  ClutterActor *actor;
  guint         mask;     /* the masks are looked up on entered/left
                             actors using data set with
                             g_object_set_data (G_OBJECT(actor),"HRN_DRAG_MASK",
                             GINT_TO_POINTER(mask);

                             if the mask of the drop target matches
                             the one registered when creating the
                             draggable actor t

                           */

  gboolean (*start_drag)(ClutterActor *actor, gint x, gint y, gpointer userdata);

  gboolean (*can_drop)(ClutterActor *actor, ClutterActor *drop_target, gint x,
                       gint y, gpointer userdata);

  gboolean (*drop)(ClutterActor *actor, ClutterActor *dropped_on, gint x,
                   gint y, gpointer userdata);
  gpointer data;
} DraggableData;

typedef struct HrnDragData
{
  gpointer userdata;

  ClutterActor  *dragged;
  ClutterActor  *ghost;
  ClutterEvent   start_event;  /* press   */
  ClutterEvent   end_event;    /* release */
  gint           x_actor_start;
  gint           y_actor_start;
  guint          handler;
  gint           startx, starty;
  DraggableData *drag_data;

  ClutterActor *potential_target;

  guint emit_delayed_press : 1;
  guint in_drag            : 1;
} HrnDragData;

static gboolean
delayed_release (gpointer dragdata)
{
  HrnDragData *drag = dragdata;

  clutter_do_event (&drag->end_event);

  g_free (drag);
  return FALSE;
}

static void
hide_when_done (ClutterAnimation *animation, ClutterActor     *actor)
{
  clutter_actor_hide (actor);
}


static gboolean
drag_capture (ClutterActor *stage, ClutterEvent *event, gpointer userdata)
{
  HrnDragData *drag = userdata;

  switch (event->type)
    {
      case CLUTTER_MOTION:

      {
        gint rel_x = drag->start_event.motion.x - event->motion.x;
        gint rel_y = drag->start_event.motion.y - event->motion.y;

        if (rel_x * rel_x + rel_y * rel_y > 16 && !drag->in_drag)
          {
            gfloat x0, y0;
            gfloat w, h;
            drag->in_drag = TRUE;

            drag->ghost = clutter_clone_new (drag->drag_data->actor);

            clutter_actor_get_transformed_position (drag->drag_data->actor, &x0,
                                                    &y0);
            clutter_actor_get_transformed_size (drag->drag_data->actor, &w, &h);

            /* This will work as long as we are not scaling, a proper fix needs
             * to go all the way down to clutter itself.
             */
            {
              ClutterActor *iter = drag->drag_data->actor;
              while (iter)
                {
                  if (NBTK_IS_VIEWPORT (iter))
                    {
                      gint x_offset;
                      gint y_offset;

                      nbtk_viewport_get_origin (NBTK_VIEWPORT (
                                                  iter), &x_offset, &y_offset,
                                                NULL);

                      y0 -= y_offset;
                      x0 -= x_offset;
                    }
                  iter = clutter_actor_get_parent (iter);
                }
            }

            clutter_group_add (stage, drag->ghost);

            drag->x_actor_start = x0;
            drag->y_actor_start = y0;

            clutter_actor_set_position (drag->ghost, x0, y0);
            {
              gdouble scale = hrn_actor_get_abs_scale (drag->drag_data->actor);
              clutter_actor_set_scale (drag->ghost, scale, scale);
              scale = 140.0 / clutter_actor_get_width (drag->drag_data->actor);
              clutter_actor_animate (drag->ghost, CLUTTER_LINEAR, 200,
                                     "scale-x", scale,
                                     "scale-y", scale,
                                     NULL);
            }

            clutter_actor_set_anchor_point_from_gravity (drag->ghost,
                                                         CLUTTER_GRAVITY_CENTER);

            if (!hrn_theatre_get_active (HRN_THEATRE (hrn_theatre)))
              {
                clutter_actor_show (hrn_theatre_lowlight);
                clutter_actor_animate (hrn_theatre_lowlight, CLUTTER_LINEAR,
                                       500, "opacity", 92,
                                       NULL);
              }

            if (drag->drag_data->start_drag)
              {
                drag->drag_data->start_drag (drag->dragged, event->motion.x,
                                             event->motion.y,
                                             drag->drag_data->data);
              }
          }
      }

        if (!drag->in_drag)
          {
            return TRUE;
          }


        clutter_actor_set_position (drag->ghost, event->motion.x,
                                    event->motion.y);
        /*drag->x_actor_start - rel_x,
           drag->y_actor_start - rel_y);*/


        if (drag->drag_data->can_drop)
          drag->drag_data->can_drop (drag->dragged, drag->potential_target,
                                     event->motion.x, event->motion.y,
                                     drag->drag_data->data);

        if (event->any.source == hrn_sidebar)
          {
            return FALSE;
          }
        return TRUE;
        break;

      case CLUTTER_ENTER:
      {
        ClutterActor *iter = event->any.source;

        drag->potential_target = NULL;
        while (iter)
          {
            gint mask =
              GPOINTER_TO_INT (g_object_get_data (G_OBJECT (iter),
                                                  "HRN_DROP_MASK"));
            if (mask & drag->drag_data->mask)
              {
                drag->potential_target = iter;
              }
            iter = clutter_actor_get_parent (iter);
          }
        if (drag->potential_target)
          return FALSE;
      }
      break;

      case CLUTTER_LEAVE:
        return FALSE;
        break;

      case CLUTTER_BUTTON_RELEASE:

        if (drag->handler)
          g_signal_handler_disconnect (clutter_actor_get_stage (
                                         drag->drag_data
                                         ->actor),
                                       drag->handler);

        if (!drag->in_drag)
          {
            drag->end_event = *event;
            g_idle_add_full (0, delayed_release, drag, NULL);
            return TRUE;
          }

        /* we emit also when we have no potential target, the drop handler
         * must handle this, this is used to drag items out of a container
         */
        if (drag->drag_data->drop)
          drag->drag_data->drop (drag->dragged, drag->potential_target,
                                 event->motion.x, event->motion.y,
                                 drag->drag_data->data);

        if (!hrn_theatre_get_active (HRN_THEATRE (hrn_theatre)))
          {
            clutter_actor_animate (hrn_theatre_lowlight, CLUTTER_LINEAR, 500,
                                   "opacity", 0,
                                   "signal::completed", hide_when_done,
                                   hrn_theatre_lowlight,
                                   NULL);
          }

        if (drag->ghost)
          {
            clutter_actor_destroy (drag->ghost);
          }
        g_free (drag);
        break;

      default:
        g_debug ("unhandled captured event %i", event->type);
        return FALSE;
        break;
    }
  return TRUE;
}

static void
drag_start (ClutterActor *actor, DraggableData *drag_data,
            ClutterEvent  *start_event)
{
  HrnDragData  *drag  = g_malloc0 (sizeof (HrnDragData));
  ClutterActor *stage = clutter_stage_get_default ();

  drag->dragged   = actor;
  drag->drag_data = drag_data;

  drag->start_event = *start_event;

  drag->handler =
    g_signal_connect_after (stage, "captured-event", G_CALLBACK (
                              drag_capture), drag);
  drag->in_drag = FALSE;

  { /* invoke a leave event upon press (bit hacky but fixes hornsey) */
    ClutterEvent event;
    event.type = CLUTTER_LEAVE;
    clutter_actor_event (drag->dragged, &event, FALSE);
  }
}


static gboolean
draggable_press (ClutterActor *actor, ClutterEvent *event, gpointer userdata)
{
  if (clutter_event_get_state (event) &
      (CLUTTER_SHIFT_MASK | CLUTTER_CONTROL_MASK))
    return FALSE;
  if (hrn_view_has_selected ())
    {
      g_debug ("Should be dragging group");
    }

  drag_start (actor, userdata, event);
  return TRUE;
}


void
hrn_actor_make_draggable (ClutterActor *actor, ClutterActor *drag_pad,
                          guint mask, GCallback start_drag_cb,
                          GCallback can_drop_cb, GCallback drop_cb,
                          gpointer data)
{
  DraggableData *drag_data = g_malloc0 (sizeof (DraggableData));

  clutter_actor_set_reactive (drag_pad, TRUE);
  drag_data->actor      = actor;
  drag_data->start_drag = (void*) start_drag_cb;
  drag_data->can_drop   = (void*) can_drop_cb;
  drag_data->drop       = (void*) drop_cb;
  drag_data->data       = data;
  drag_data->mask       = mask;

  g_signal_connect (drag_pad, "button-press-event", G_CALLBACK (
                      draggable_press), drag_data);

  g_object_set_data_full (G_OBJECT (drag_pad), "drag-info", drag_data, g_free);
}
