/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2007, 2008, 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * SECTION:hrn-texture-frame
 * @short_description: Stretch a texture to fit the entire allocation
 *
 * #HrnSquareClone
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cogl/cogl.h>

#include "hrn-square-clone.h"

enum
{
  PROP_0,

  PROP_PARENT_TEXTURE,
};

G_DEFINE_TYPE (HrnSquareClone, hrn_square_clone, CLUTTER_TYPE_ACTOR);

#define HRN_SQUARE_CLONE_GET_PRIVATE(obj)    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                                                           HRN_TYPE_SQUARE_CLONE, \
                                                                           HrnSquareClonePrivate))

struct _HrnSquareClonePrivate
{
  ClutterTexture *parent_texture;
  CoglHandle      material;
  gboolean        can_pinch;
};


static void
hrn_square_clone_get_preferred_width (ClutterActor *self, gfloat for_height,
                                      gfloat       *min_width_p,
                                      gfloat       *natural_width_p)
{
  HrnSquareClonePrivate *priv = HRN_SQUARE_CLONE (self)->priv;

  if (G_UNLIKELY (priv->parent_texture == NULL))
    {
      if (min_width_p)
        *min_width_p = 0;

      if (natural_width_p)
        *natural_width_p = 0;
    }
  else
    {
      ClutterActorClass *klass;

      /* by directly querying the parent texture's class implementation
       * we are going around any override mechanism the parent texture
       * might have in place, and we ask directly for the original
       * preferred width
       */
      klass = CLUTTER_ACTOR_GET_CLASS (priv->parent_texture);
      klass->get_preferred_width (CLUTTER_ACTOR (priv->parent_texture),
                                  for_height,
                                  min_width_p,
                                  natural_width_p);
    }
}

static void
hrn_square_clone_get_preferred_height (
  ClutterActor *self, gfloat for_width, gfloat       *min_height_p,
  gfloat       *natural_height_p)
{
  HrnSquareClonePrivate *priv = HRN_SQUARE_CLONE (self)->priv;

  if (G_UNLIKELY (priv->parent_texture == NULL))
    {
      if (min_height_p)
        *min_height_p = 0;

      if (natural_height_p)
        *natural_height_p = 0;
    }
  else
    {
      ClutterActorClass *klass;

      /* by directly querying the parent texture's class implementation
       * we are going around any override mechanism the parent texture
       * might have in place, and we ask directly for the original
       * preferred height
       */
      klass = CLUTTER_ACTOR_GET_CLASS (priv->parent_texture);
      klass->get_preferred_height (CLUTTER_ACTOR (priv->parent_texture),
                                   for_width,
                                   min_height_p,
                                   natural_height_p);
    }
}

void
hrn_square_clone_set_can_pinch (ClutterActor *self,
                                gboolean      state)
{
  HrnSquareClonePrivate *priv = HRN_SQUARE_CLONE (self)->priv;
  priv->can_pinch = state;
}

static void
hrn_square_clone_realize (ClutterActor *self)
{
  HrnSquareClonePrivate *priv = HRN_SQUARE_CLONE (self)->priv;

  if (priv->material != COGL_INVALID_HANDLE)
    return;

  priv->material = cogl_material_new ();

  CLUTTER_ACTOR_SET_FLAGS (self, CLUTTER_ACTOR_REALIZED);
}

static void
hrn_square_clone_unrealize (ClutterActor *self)
{
  HrnSquareClonePrivate *priv = HRN_SQUARE_CLONE (self)->priv;

  if (priv->material == COGL_INVALID_HANDLE)
    return;

  cogl_material_unref (priv->material);
  priv->material = COGL_INVALID_HANDLE;

  CLUTTER_ACTOR_UNSET_FLAGS (self, CLUTTER_ACTOR_REALIZED);
}

#include "hrn.h"


static void
hrn_square_clone_paint (ClutterActor *self)
{
  HrnSquareClonePrivate *priv = HRN_SQUARE_CLONE (self)->priv;
  CoglHandle             tex  = COGL_INVALID_HANDLE;
  ClutterActorBox        box  = { 0, };
  guint8                 opacity;

  gfloat                 tex_width, tex_height;

  /* no need to paint stuff if we don't have a texture */
  if (G_UNLIKELY (priv->parent_texture == NULL))
    return;

  /* parent texture may have been hidden, so need to make sure it gets
   * realized
   */
  if (!CLUTTER_ACTOR_IS_REALIZED (priv->parent_texture))
    clutter_actor_realize (CLUTTER_ACTOR (priv->parent_texture));

  tex = clutter_texture_get_cogl_texture (priv->parent_texture);
  if (tex == COGL_INVALID_HANDLE)
    return;

  tex_width  = cogl_texture_get_width (tex);
  tex_height = cogl_texture_get_height (tex);

  {
    gint  bw, bh;
    gint  aw, ah;
    float v;
    float tx1, tx2, ty1, ty2;

    clutter_actor_get_allocation_box (self, &box);
    /*tex = clutter_texture_get_cogl_texture (CLUTTER_TEXTURE (self));*/

    bw = cogl_texture_get_width (tex); /* base texture width */
    bh = cogl_texture_get_height (tex); /* base texture height */

    aw = (box.x2 - box.x1); /* allocation width */
    ah = (box.y2 - box.y1); /* allocation height */

    /* no comment */
    if ((float) bw / bh < (float) aw / ah)
      {
        /* fit width */
        v   = (((float) ah * bw) / ((float) aw * bh)) / 2;
        tx1 = 0;
        tx2 = 1;
        ty1 = (0.5 - v);
        ty2 = (0.5 + v);
      }
    else
      {
        /* fit height */
        v   = (((float) aw * bh) / ((float) ah * bw)) / 2;
        tx1 = (0.5 - v);
        tx2 = (0.5 + v);
        ty1 = 0;
        ty2 = 1;
      }


    opacity = clutter_actor_get_paint_opacity (self);

    g_assert (priv->material != COGL_INVALID_HANDLE);

    /* set the source material using the parent texture's COGL handle */
    cogl_material_set_color4ub (priv->material,
                                opacity, opacity, opacity, opacity);
    cogl_material_set_layer (priv->material, 0, tex);
    cogl_set_source (priv->material);

    {
      gdouble pinch = hrn_get_pinch ();
      if (pinch <= 0.1 || priv->can_pinch == FALSE)
        {
          cogl_rectangle_with_texture_coords (0, 0,
                                              aw, ah,
                                              tx1, ty1,
                                              tx2, ty2);
        }
      else
        {
          CoglTextureVertex polygon[4] = {
            { pinch,      pinch * 5.2,      0.0,        tx1, ty1, },
            { aw - pinch, pinch * 5.2,      0.0,        tx2, ty1, },
            { aw,         ah,             0.0,        tx2, ty2, },
            { 0.0,        ah,             0.0,        tx1, ty2, },
          };
          cogl_polygon (polygon, 4, FALSE);
        }
    }
  }
}

static inline void
hrn_square_clone_set_frame_internal (HrnSquareClone *frame, gfloat left,
                                     gfloat top, gfloat right,
                                     gfloat bottom)
{
  GObject *gobject = G_OBJECT (frame);
  gboolean changed = FALSE;

  g_object_freeze_notify (gobject);

  if (changed && CLUTTER_ACTOR_IS_VISIBLE (frame))
    clutter_actor_queue_redraw (CLUTTER_ACTOR (frame));

  g_object_thaw_notify (gobject);
}

static void
hrn_square_clone_set_property (GObject *gobject, guint prop_id,
                               const GValue *value,
                               GParamSpec   *pspec)
{
  HrnSquareClone *frame = HRN_SQUARE_CLONE (gobject);

  switch (prop_id)
    {
      case PROP_PARENT_TEXTURE:
        hrn_square_clone_set_parent_texture (frame,
                                             g_value_get_object (value));
        break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
        break;
    }
}

static void
hrn_square_clone_get_property (GObject *gobject, guint prop_id,
                               GValue     *value,
                               GParamSpec *pspec)
{
  HrnSquareClonePrivate *priv = HRN_SQUARE_CLONE (gobject)->priv;

  switch (prop_id)
    {
      case PROP_PARENT_TEXTURE:
        g_value_set_object (value, priv->parent_texture);
        break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
        break;
    }
}

static void
hrn_square_clone_dispose (GObject *gobject)
{
  HrnSquareClonePrivate *priv = HRN_SQUARE_CLONE (gobject)->priv;

  if (priv->parent_texture)
    {
      g_object_unref (priv->parent_texture);
      priv->parent_texture = NULL;
    }

  if (priv->material)
    {
      cogl_material_unref (priv->material);
      priv->material = COGL_INVALID_HANDLE;
    }

  G_OBJECT_CLASS (hrn_square_clone_parent_class)->dispose (gobject);
}

static void
hrn_square_clone_class_init (HrnSquareCloneClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);
  GParamSpec        *pspec;

  g_type_class_add_private (gobject_class, sizeof (HrnSquareClonePrivate));

  actor_class->get_preferred_width =
    hrn_square_clone_get_preferred_width;
  actor_class->get_preferred_height =
    hrn_square_clone_get_preferred_height;
  actor_class->realize   = hrn_square_clone_realize;
  actor_class->unrealize = hrn_square_clone_unrealize;
  actor_class->paint     = hrn_square_clone_paint;

  gobject_class->set_property = hrn_square_clone_set_property;
  gobject_class->get_property = hrn_square_clone_get_property;
  gobject_class->dispose      = hrn_square_clone_dispose;

  pspec = g_param_spec_object ("parent-texture",
                               "Parent Texture",
                               "The parent ClutterTexture",
                               CLUTTER_TYPE_TEXTURE,
                               G_PARAM_READWRITE |
                               G_PARAM_CONSTRUCT);
  g_object_class_install_property (gobject_class, PROP_PARENT_TEXTURE, pspec);
}

static void
hrn_square_clone_init (HrnSquareClone *self)
{
  HrnSquareClonePrivate *priv;

  self->priv = priv = HRN_SQUARE_CLONE_GET_PRIVATE (self);

  priv->material = COGL_INVALID_HANDLE;
}

/**
 * hrn_square_clone_new:
 * @texture: a #ClutterTexture or %NULL
 *
 * A #HrnSquareClone is a specialized texture that efficiently clones
 * an area of the given @texture while keeping preserving portions of the
 * same texture.
 *
 * A #HrnSquareClone can be used to make a rectangular texture fit a
 * given size without stretching its borders.
 *
 * Return value: the newly created #HrnSquareClone
 */
ClutterActor*
hrn_square_clone_new (ClutterTexture *texture)
{
  g_return_val_if_fail (texture == NULL || CLUTTER_IS_TEXTURE (texture), NULL);

  return g_object_new (HRN_TYPE_SQUARE_CLONE,
                       "parent-texture", texture,
                       NULL);
}

ClutterTexture *
hrn_square_clone_get_parent_texture (HrnSquareClone *frame)
{
  g_return_val_if_fail (HRN_IS_SQUARE_CLONE (frame), NULL);

  return frame->priv->parent_texture;
}

void
hrn_square_clone_set_parent_texture (HrnSquareClone *frame,
                                     ClutterTexture *texture)
{
  HrnSquareClonePrivate *priv;
  gboolean               was_visible;

  g_return_if_fail (HRN_IS_SQUARE_CLONE (frame));
  g_return_if_fail (texture == NULL || CLUTTER_IS_TEXTURE (texture));

  priv = frame->priv;

  was_visible = CLUTTER_ACTOR_IS_VISIBLE (frame);

  if (priv->parent_texture == texture)
    return;

  if (priv->parent_texture)
    {
      g_object_unref (priv->parent_texture);
      priv->parent_texture = NULL;

      if (was_visible)
        clutter_actor_hide (CLUTTER_ACTOR (frame));
    }

  if (texture)
    {
      priv->parent_texture = g_object_ref (texture);

      if (was_visible && CLUTTER_ACTOR_IS_VISIBLE (priv->parent_texture))
        clutter_actor_show (CLUTTER_ACTOR (frame));
    }

  clutter_actor_queue_relayout (CLUTTER_ACTOR (frame));

  g_object_notify (G_OBJECT (frame), "parent-texture");
}
