/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmäki <antti.kaijanmaki@canonical.com>
 * Kalle Valo       <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "manager.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <locale.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <glib/gi18n.h>
#include <libnotify/notify.h>

#include <libdbusmenu-glib/server.h>

#include "marshal.h"
#include "service.h"
#include "dbus-shared-names.h"
#include "service-manager.h"
#include "network-menu.h"
#include "log.h"
#include "manager-proxy.h"

#include "ui-proxy.h"

#include "android-service.h"
#include "android-telephony-service.h"


G_DEFINE_TYPE(Manager, manager, G_TYPE_OBJECT)

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE((o), TYPE_MANAGER, ManagerPrivate))

typedef struct _ManagerPrivate ManagerPrivate;

struct _ManagerPrivate {
  ManagerProxy *ns_dbus;
  ServiceManager *service_manager;
  AndroidService *android_service;
  AndroidTelephonyService *telephony_service;

  UIProxy *ui;

  DbusmenuServer *server;

  NotifyNotification *notification;
  guint animation_timer;
  gint animation_counter;
  gint connecting_stage;
  gint connecting_state;
  gchar *connecting_icons[CONNECTING_ICON_STAGES][CONNECTING_ICON_STATES];
  gint signal_strength;
  ManagerState manager_state;
  NetworkMenu *network_menu;

  AndroidNetworkState old_wifi_state;
  AndroidNetworkState old_mobile_state;

  const gchar *network_icon;
  const gchar *mobile_icon;

  TelephonySignalLevel telephony_signal_level;
};

enum {
  TECHNOLOGY_STATE_CHANGED,
  TECHNOLOGY_AVAILABILITY_CHANGED,
  LAST_SIGNAL,
};

static guint signals[LAST_SIGNAL] = { 0 };

#define CONNECTING_ANIMATION_TIMEOUT 100

static void update_network_icon(Manager *self, const gchar *name);
static void update_mobile_icon(Manager *self, const gchar *name);
static void update_indicator_icons(Manager *self);

static void update_mobile_signal_level(Manager *self);

static void stop_connecting_animation(Manager *self);
static void start_agent(Manager *self);

static void manager_dispose(GObject *object)
{
  Manager *self = MANAGER(object);
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_debug("%s", __func__);

  if(priv->android_service != NULL) {
    stop_agent(self);
    g_object_unref(priv->android_service);
    priv->android_service = NULL;
  }

  if(priv->telephony_service != NULL) {
    g_object_unref(priv->telephony_service);
    priv->telephony_service = NULL;
  }

  if (priv->network_menu != NULL) {
    g_object_unref(priv->network_menu);
    priv->network_menu = NULL;
  }

  if (priv->service_manager != NULL) {
    g_object_unref(priv->service_manager);
    priv->service_manager = NULL;
  }

  if (priv->ns_dbus != NULL) {
    g_object_unref(priv->ns_dbus);
    priv->ns_dbus = NULL;
  }

  if (priv->ui != NULL) {
    g_object_unref(priv->ui);
    priv->ui = NULL;
  }

  if (priv->server != NULL) {
    g_object_unref(priv->server);
    priv->server = NULL;
  }

  if (notify_is_initted())
    notify_uninit();

  G_OBJECT_CLASS(manager_parent_class)->dispose(object);
}

static void manager_finalize(GObject *object)
{
  Manager *manager = MANAGER(object);
  ManagerPrivate *priv = GET_PRIVATE(manager);
  int i, j;

  for (i = 0; i < CONNECTING_ICON_STAGES; i++) {
    for (j = 0; j < CONNECTING_ICON_STATES; j++) {
      g_free(priv->connecting_icons[i][j]);
      priv->connecting_icons[i][j] = NULL;
    }
  }

  G_OBJECT_CLASS(manager_parent_class)->finalize(object);
}

static void manager_class_init(ManagerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);
  guint signal;

  g_type_class_add_private(klass, sizeof(ManagerPrivate));

  object_class->dispose = manager_dispose;
  object_class->finalize = manager_finalize;

  signal = g_signal_new("technology-state-changed", G_TYPE_FROM_CLASS(klass),
			G_SIGNAL_RUN_LAST, 0, NULL, NULL,
			_marshal_VOID__STRING_BOOLEAN,
			G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_BOOLEAN);
  signals[TECHNOLOGY_STATE_CHANGED] = signal;

  signal = g_signal_new("technology-availability-updated",
			G_TYPE_FROM_CLASS(klass), G_SIGNAL_RUN_LAST,
			0, NULL, NULL, _marshal_VOID__STRING_BOOLEAN,
			G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_BOOLEAN);
  signals[TECHNOLOGY_AVAILABILITY_CHANGED] = signal;

}

static void manager_init(Manager *self)
{
  if (!notify_init("network-indicator")) {
    g_critical("libnotify initialisation failed");
  }
}

const gchar *
get_mobile_icon(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);
  
  TelephonyRadioTechnology tech;
  tech = android_telephony_service_get_radio_technology(priv->telephony_service);

  switch (tech) {

  case TELEPHONY_RADIO_TECHNOLOGY_UNKNOWN:
    return ICON_MOBILE_2G;
    
  case TELEPHONY_RADIO_TECHNOLOGY_GPRS:
  case TELEPHONY_RADIO_TECHNOLOGY_EDGE:
  case TELEPHONY_RADIO_TECHNOLOGY_IS95A: 
  case TELEPHONY_RADIO_TECHNOLOGY_IS95B:
    return ICON_MOBILE_2G;

  case TELEPHONY_RADIO_TECHNOLOGY_UMTS:
  case TELEPHONY_RADIO_TECHNOLOGY_1xRTT:
  case TELEPHONY_RADIO_TECHNOLOGY_EVDO_0:
  case TELEPHONY_RADIO_TECHNOLOGY_EVDO_A:
  case TELEPHONY_RADIO_TECHNOLOGY_EVDO_B:
  case TELEPHONY_RADIO_TECHNOLOGY_HSPA:
  case TELEPHONY_RADIO_TECHNOLOGY_HSPAP:
  case TELEPHONY_RADIO_TECHNOLOGY_HSDPA:
  case TELEPHONY_RADIO_TECHNOLOGY_HSUPA:
  case TELEPHONY_RADIO_TECHNOLOGY_EHRPD:
    return ICON_MOBILE_3G;

  case TELEPHONY_RADIO_TECHNOLOGY_LTE:
  default:
    return ICON_MOBILE_4G;
  }
}

const gchar *
manager_wifi_icon_name(gint signal)
{
    if (signal > 75)
       return ICON_CONNECTED_WIFI_100;
    else if (signal > 50)
      return ICON_CONNECTED_WIFI_75;
    else if (signal > 25)
      return ICON_CONNECTED_WIFI_50;
    else if (signal > 0)
      return ICON_CONNECTED_WIFI_25;
    else
      return ICON_CONNECTED_WIFI_0;
}

const gchar *manager_icon_name(Manager *self, AndroidNetworkType type, gint signal)
{
  switch (type) {
  case ANDROID_NETWORK_TYPE_WIFI:
    return manager_wifi_icon_name(signal);
  case ANDROID_NETWORK_TYPE_MOBILE:
    return get_mobile_icon(self);
  default:
    return ICON_CONNECTED_DEFAULT;
  }
}

gchar *manager_accessible_desc(AndroidNetworkType type, gint signal)
{
  /** @todo this is really annoying that we have to return g_strdup() of 
            string literals. Check with design if the signal strength is
	    indeed needed with WIFI and use plain string literals if not.
	    Otherwise rework this in some saner way.
	    Probably in update_icon().
  */
  switch (type) {
  case ANDROID_NETWORK_TYPE_WIFI:
    return g_strdup_printf(_("Network (Wireless, %d%%)"), signal);
  case ANDROID_NETWORK_TYPE_MOBILE:
    return g_strdup(_("Network (Mobile)"));
  default:
    return g_strdup(_("Network (Connected)"));
  }
}

static void set_animation_icon(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);
  gchar *icon;

  g_return_if_fail(self != NULL);
  g_return_if_fail(priv->ns_dbus != NULL);
  g_return_if_fail(priv->connecting_stage < CONNECTING_ICON_STAGES);
  g_return_if_fail(priv->connecting_state < CONNECTING_ICON_STATES);

  icon = priv->connecting_icons[priv->connecting_stage][priv->connecting_state];
  update_network_icon(self, icon);
}

static gboolean connecting_animation_timeout(gpointer user_data)
{
  Manager *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(self != NULL, FALSE);

  priv->connecting_state++;
  if (priv->connecting_state >= CONNECTING_ICON_STATES)
    priv->connecting_state = 0;

  set_animation_icon(self);

  priv->animation_counter++;

  /* fail safe in case animation is not stopped properly */
  if (priv->animation_counter > 3000) {
    g_warning("connecting animation running for too long!");
    stop_connecting_animation(self);
    return FALSE;
  }

  return TRUE;
}

static void start_connecting_animation(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);

  if (priv->animation_timer != 0)
    return;

  priv->connecting_stage = 0;
  priv->connecting_state = 0;
  priv->animation_counter = 0;

  set_animation_icon(self);

  priv->animation_timer = g_timeout_add(CONNECTING_ANIMATION_TIMEOUT,
					connecting_animation_timeout,
					self);

  if (priv->animation_timer == 0)
    g_warning("failed to add timeout for icon animation");
}

static void stop_connecting_animation(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);

  if (priv->animation_timer == 0)
	  return;

  g_source_remove(priv->animation_timer);
  priv->animation_timer = 0;
}

void manager_notify(Manager *self, const gchar *summary, const gchar *body,
		    const gchar *icon)
{
  ManagerPrivate *priv = GET_PRIVATE(self);
  GError *error = NULL;
  gboolean result;

  g_return_if_fail(self != NULL);
  g_return_if_fail(priv != NULL);

  g_debug("NOTIFY: %s %s %s", summary, body, icon);

  if (priv->notification == NULL) {
    priv->notification = notify_notification_new(summary, body, icon);
  } else {
    notify_notification_update(priv->notification, summary, body, icon);
  }

  result = notify_notification_show(priv->notification, &error);
  if (!result) {
    g_warning("Failed to show notification '%s/%s': %s", summary, body,
	      error->message);
    g_error_free(error);
  }
}

static void update_icon(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);
  AndroidNetworkType type;
  const gchar *icon_name;
  gchar *accessible_desc = NULL;

  AndroidNetworkState wifi_state;
  AndroidNetworkState mobile_state;

  AndroidNetworkType  primary_type;
  AndroidNetworkState primary_state;

  g_return_if_fail(self);

  wifi_state = android_service_get_state(priv->android_service,
					 ANDROID_NETWORK_TYPE_WIFI);
  mobile_state = android_service_get_state(priv->android_service,
					   ANDROID_NETWORK_TYPE_MOBILE);
  switch(wifi_state) {
  case ANDROID_NETWORK_STATE_SUSPENDED:
  case ANDROID_NETWORK_STATE_DISCONNECTING:
  case ANDROID_NETWORK_STATE_DISCONNECTED:
  case ANDROID_NETWORK_STATE_UNKNOWN:
    primary_type  = ANDROID_NETWORK_TYPE_MOBILE;
    primary_state = mobile_state;
    break;
  default:
    primary_type  = ANDROID_NETWORK_TYPE_WIFI;
    primary_state = wifi_state;
  }

  if (primary_state == ANDROID_NETWORK_STATE_CONNECTING &&
      primary_type  != ANDROID_NETWORK_TYPE_MOBILE) { // Android claims mobile to be connecting even in fligh mode. just ignore it.
    start_connecting_animation(self);
    manager_proxy_set_accessible_desc(priv->ns_dbus, _("Network (Connecting)"));
    return;
  }

  stop_connecting_animation(self);

  if (primary_state != ANDROID_NETWORK_STATE_CONNECTED) {
    icon_name = ICON_DISCONNECTED;
    accessible_desc = g_strdup(_("Network (Disconnected)"));
    goto done;
  }


  type = ANDROID_NETWORK_TYPE_WIFI;
  if (primary_type == ANDROID_NETWORK_TYPE_MOBILE)
    type = ANDROID_NETWORK_TYPE_MOBILE;

  icon_name = manager_icon_name(self, type, priv->signal_strength);
  accessible_desc = manager_accessible_desc(type, priv->signal_strength);

 done:

  g_debug("wifi_state: %d", wifi_state);
  g_debug("mobi_state: %d", mobile_state);
  g_debug("icon_name: %s", icon_name);
  g_debug("icon_desc: %s", accessible_desc);

  update_network_icon(self, icon_name);
  manager_proxy_set_accessible_desc(priv->ns_dbus, accessible_desc);

  if (accessible_desc != NULL)
    g_free(accessible_desc);
}

static void state_changed(ServiceManager *sm, gpointer user_data)
{
  Manager *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(self != NULL);
  g_return_if_fail(priv != NULL);

  update_icon(self);
}

/* FIXME: should follow priv->default_service strength */
static void strength_updated(ServiceManager *sm, guint strength,
			     gpointer user_data)
{
  Manager *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);

  priv->signal_strength = strength;

  update_icon(self);
}

static void update_services(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);
  const GSList *networks;

  g_return_if_fail(priv != NULL);

  networks = android_service_get_networks(priv->android_service);

  service_manager_update_services(priv->service_manager, networks);

  update_icon(self);
}

static void ui_connected_notify(UIProxy *ui, GParamSpec *pspec,
				gpointer user_data)
{
  if (ui_proxy_is_connected(ui))
    g_debug("ui connected");
  else
    g_debug("ui disconnected");

  /* FIXME: start ui again if we didn't stop it */
}

static void create_ui_proxy(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(self != NULL);
  g_return_if_fail(priv != NULL);

  /* this should not be called agent_proxy is already created */
  g_return_if_fail(priv->ui == NULL);

  priv->ui = ui_proxy_new();

  if (priv->ui == NULL) {
    g_warning("failed to create ui proxy");
    return;
  }

  g_signal_connect(priv->ui, "notify::connected",
		   G_CALLBACK(ui_connected_notify),
		   self);

  g_debug("ui proxy created");
}

void start_agent(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->ui != NULL);

  ui_proxy_start(priv->ui);
}

void stop_agent(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->ui != NULL);

  ui_proxy_stop(priv->ui);
}

void manager_request_scan(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);
  
  android_service_start_scan(priv->android_service);
}

void manager_set_debug_level(Manager *self, gint level)
{
  g_return_if_fail(IS_MANAGER(self));

  log_set_debug(level > 0);
}

UIProxy *manager_get_ui(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(IS_MANAGER(self), NULL);
  g_return_val_if_fail(priv != NULL, NULL);

  return priv->ui;
}

ServiceManager *manager_get_service_manager(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(self != NULL, NULL);
  g_return_val_if_fail(priv != NULL, NULL);

  return priv->service_manager;
}

AndroidService *
manager_get_android_service(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);
  return priv->android_service;
}

static void
network_state_changed_cb(gpointer *object, gpointer user_data)
{
  Manager *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);

  AndroidNetworkState wifi_state, mobile_state;

  const gchar *icon, *summary, *body;

  wifi_state = android_service_get_state(priv->android_service,
					 ANDROID_NETWORK_TYPE_WIFI);
  mobile_state = android_service_get_state(priv->android_service,
					   ANDROID_NETWORK_TYPE_MOBILE);

  if (wifi_state == ANDROID_NETWORK_STATE_CONNECTED &&
      priv->old_wifi_state != ANDROID_NETWORK_STATE_CONNECTED) {

    icon = ICON_CONNECTED_WIFI;
    summary = _("Connection Established");
    body = android_service_get_connected_name(priv->android_service);

    if (g_strcmp0(body, "") == 0) {
      /// @todo hack, hack hack.. GetConnectedName has not returned yet on android-service... :/
      return;
    }

    manager_notify(self, summary, body, icon);

  } else if (wifi_state == ANDROID_NETWORK_STATE_DISCONNECTED &&
	     priv->old_wifi_state != ANDROID_NETWORK_STATE_DISCONNECTED) {

    icon = ICON_CONNECTED_WIFI;
    manager_notify(self,
		   _("Disconnected"),
		   "",
		   icon);

  } else if (mobile_state == ANDROID_NETWORK_STATE_CONNECTED &&
	     priv->old_mobile_state != ANDROID_NETWORK_STATE_CONNECTED) {

    icon = get_mobile_icon(self);
    summary = _("Connection Established");
    body = android_telephony_service_get_operator_name(priv->telephony_service);
    manager_notify(self, summary, body, icon);

  }
  /// @todo disconnected from mobile

  priv->old_wifi_state   = wifi_state;
  priv->old_mobile_state = mobile_state;

  update_services(self); // also calls update_icon()
}

static void
scan_results_ready_cb(gpointer *object, gpointer user_data)
{
  Manager *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);
  
  g_debug("SCAN RESULTS READY");

  update_services(self);
}

static void
rssi_changed_cb(gpointer *object, gpointer user_data)
{
  Manager *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);

  priv->signal_strength = android_service_get_rssi(priv->android_service);

  update_icon(self);
}

static void
connectivity_cb(gpointer *object, gpointer user_data)
{
  Manager *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);

  update_services(self); // also calls update_icon()
}

static void
airplane_mode_changed_cb(gpointer *object, gboolean value, gpointer user_data)
{
  Manager        *self = MANAGER(user_data);

  update_mobile_signal_level(self);

  if (value == TRUE) {
    manager_notify(self, _("Flight Mode ON"), NULL, ICON_AIRPLANE_MODE);
  } else {
    manager_notify(self, _("Flight Mode OFF"), NULL, ICON_AIRPLANE_MODE);
  }

}

static void
mobile_data_enabled_changed_cb(gpointer *object, gboolean value, gpointer user_data)
{
  Manager        *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);

  gboolean airplane_mode =
    android_service_get_airplane_mode(priv->android_service);
  if (airplane_mode) { // only valid as ufa startup and android in airplane mode
    return;
  }

  if (value == TRUE) {
    manager_notify(self,
		   _("Mobile Data ON"),
		   NULL,
		   ICON_NOTIFICATION_CELLULAR_CONNECTED);
  } else {
    manager_notify(self,
		   _("Mobile Data OFF"),
		   NULL,
		   ICON_NOTIFICATION_CELLULAR_CONNECTED);
  }
}

static void
wifi_enabled_changed_cb(gpointer *object, gboolean value, gpointer user_data)
{
  Manager        *self = MANAGER(user_data);

  if (value == TRUE) {
    manager_notify(self,
		   _("Wi-Fi Enabled"),
		   NULL,
		   ICON_NOTIFICATION_WIFI_FULL);
  } else {
    manager_notify(self,
		   _("Wi-Fi Disabled"),
		   NULL,
		   ICON_NOTIFICATION_WIFI_DISCONNECTED);
  }
}

static void
radio_technology_changed_cb(gpointer *object, gint value, gpointer user_data)
{
  Manager        *self = MANAGER(user_data);

  update_icon(self);
}

static void
operator_name_changed_cb(gpointer *object, gpointer user_data)
{
  Manager        *self = MANAGER(user_data);
  ManagerPrivate *priv = GET_PRIVATE(self);
  const gchar    *operator_name;

  operator_name = android_telephony_service_get_operator_name(priv->telephony_service);

  network_menu_set_operator(priv->network_menu,
			    operator_name);
}

static void
update_mobile_signal_level(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  gboolean airplane_mode;

  priv->telephony_signal_level = android_telephony_service_get_signal_level(priv->telephony_service);
  
  airplane_mode = android_service_get_airplane_mode(priv->android_service);

  if (airplane_mode) {
    update_mobile_icon(self, ICON_AIRPLANE_MODE);
    return;
  }

  switch (priv->telephony_signal_level) {
  case TELEPHONY_SIGNAL_LEVEL_UNKNOWN:
    update_mobile_icon(self, ICON_TELEPHONY_SIGNAL_LEVEL_UNKNOWN);
    break;
  case TELEPHONY_SIGNAL_LEVEL_POOR:
    update_mobile_icon(self, ICON_TELEPHONY_SIGNAL_LEVEL_POOR);
    break;
  case TELEPHONY_SIGNAL_LEVEL_MODERATE:
    update_mobile_icon(self, ICON_TELEPHONY_SIGNAL_LEVEL_MODERATE);
    break;
  case TELEPHONY_SIGNAL_LEVEL_GOOD:
    update_mobile_icon(self, ICON_TELEPHONY_SIGNAL_LEVEL_GOOD);
    break;
  case TELEPHONY_SIGNAL_LEVEL_GREAT:
    update_mobile_icon(self, ICON_TELEPHONY_SIGNAL_LEVEL_GREAT);
    break;
  default:
    // some android have more level
    if (priv->telephony_signal_level > TELEPHONY_SIGNAL_LEVEL_GREAT) {
      update_mobile_icon(self, ICON_TELEPHONY_SIGNAL_LEVEL_GREAT);
    } else {
      g_assert_not_reached();
    }
  }
}

static void
telephony_signal_level_changed_cb(gpointer instance,
				  gint value,
				  gpointer user_data)
{
  Manager        *self = MANAGER(user_data);

  update_mobile_signal_level(self);
}

Manager *manager_new(void)
{
  Manager *self;
  ManagerPrivate *priv;
  gchar *icon;
  int i, j;

  self = g_object_new(TYPE_MANAGER, NULL);
  priv = GET_PRIVATE(self);

  priv->android_service  = android_service_new();
  priv->old_wifi_state   = ANDROID_NETWORK_STATE_DISCONNECTED;
  priv->old_mobile_state = ANDROID_NETWORK_STATE_DISCONNECTED;

  priv->telephony_service = android_telephony_service_new();

  g_signal_connect(G_OBJECT(priv->android_service),
		   "connectivity",
		   G_CALLBACK(connectivity_cb),
		   self);
  g_signal_connect(G_OBJECT(priv->android_service),
		   "rssi-changed",
		   G_CALLBACK(rssi_changed_cb),
		   self);
  g_signal_connect(G_OBJECT(priv->android_service),
		   "scan-results-ready",
		   G_CALLBACK(scan_results_ready_cb),
		   self);
  g_signal_connect(G_OBJECT(priv->android_service),
		   "network-state-changed",
		   G_CALLBACK(network_state_changed_cb),
		   self);
  
  g_signal_connect(G_OBJECT(priv->android_service),
		   "airplane-mode-changed",
		   G_CALLBACK(airplane_mode_changed_cb),
		   self);

  g_signal_connect(G_OBJECT(priv->android_service),
		   "mobile-data-enabled-changed",
		   G_CALLBACK(mobile_data_enabled_changed_cb),
		   self);

  g_signal_connect(G_OBJECT(priv->android_service),
		   "wifi-enabled-changed",
		   G_CALLBACK(wifi_enabled_changed_cb),
		   self);

  g_signal_connect(G_OBJECT(priv->telephony_service),
		   "signal-level-changed",
		   G_CALLBACK(telephony_signal_level_changed_cb),
		   self);

  g_signal_connect(G_OBJECT(priv->telephony_service),
		   "radio-technology-changed",
		   G_CALLBACK(radio_technology_changed_cb),
		   self);

  g_signal_connect(G_OBJECT(priv->telephony_service),
		   "operator-name-changed",
		   G_CALLBACK(operator_name_changed_cb),
		   self);

  priv->ns_dbus = manager_proxy_new(self);
  priv->service_manager = service_manager_new(self);
  priv->network_menu = network_menu_new(self);

  g_signal_connect(G_OBJECT(priv->service_manager), "state-changed",
		   G_CALLBACK(state_changed), self);
  g_signal_connect(G_OBJECT(priv->service_manager), "strength-updated",
		   G_CALLBACK(strength_updated), self);

  create_ui_proxy(self);

  /* create icon names */
  for (i = 0; i < CONNECTING_ICON_STAGES; i++) {
    for (j = 0; j < CONNECTING_ICON_STATES; j++) {
      icon = g_malloc(MAX_ICON_NAME_LEN);
      snprintf(icon, MAX_ICON_NAME_LEN,
               "nm-stage%02d-connecting%02d", i + 1, j + 1);
      priv->connecting_icons[i][j] = icon;
    }
  }

  priv->signal_strength = android_service_get_rssi(priv->android_service);

  update_mobile_signal_level(self);
  update_icon(self);

  priv->server = dbusmenu_server_new(INDICATOR_NETWORK_DBUS_OBJECT);
  dbusmenu_server_set_root(priv->server, DBUSMENU_MENUITEM(priv->network_menu));
  dbusmenu_menuitem_property_set_bool(DBUSMENU_MENUITEM(priv->network_menu),
				      DBUSMENU_MENUITEM_PROP_VISIBLE,
				      TRUE);

  /* otherwise wait for the connected signal */

  network_menu_set_operator(priv->network_menu,
			    android_telephony_service_get_operator_name(priv->telephony_service));
  network_menu_enable(priv->network_menu);

  manager_request_scan(self);

  return self;
}

static void
update_network_icon(Manager *self, const gchar *name)

{
  ManagerPrivate *priv = GET_PRIVATE(self);

  priv->network_icon = name;

  update_indicator_icons(self);
}

static void
update_mobile_icon(Manager *self, const gchar *name)
{
  ManagerPrivate *priv = GET_PRIVATE(self);

  priv->mobile_icon = name;

  update_indicator_icons(self);
}

static void
update_indicator_icons(Manager *self)
{
  ManagerPrivate *priv = GET_PRIVATE(self);
  gchar *icon;

  icon = g_strdup_printf("%s %s",
			 priv->mobile_icon,
			 priv->network_icon);
  manager_proxy_set_icon(priv->ns_dbus, icon);
  g_free(icon);
}
