/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmäki <antti.kaijanmaki@canonical.com>
 * Kalle Valo       <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/** @todo security combobox is commented out as the security
          bits are not implemented ATM
*/

#include "wireless-connect-dialog.h"

#include "service.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <locale.h>
#include <string.h>

G_DEFINE_TYPE(WirelessConnectDialog, wireless_connect_dialog, GTK_TYPE_DIALOG);

typedef struct _WirelessConnectDialogPriv WirelessConnectDialogPriv;

struct _WirelessConnectDialogPriv
{
  GtkWidget *network_combo;
  GtkWidget *count_label;
  GtkWidget *security_combo;

  guint service_count;
  GtkListStore *store;

  gboolean service_selected;
  gboolean connecting;

  const gchar *identifier;
};

#define WIRELESS_CONNECT_DIALOG_GET_PRIVATE(o)				\
  (G_TYPE_INSTANCE_GET_PRIVATE((o),					\
			       WIRELESS_CONNECT_DIALOG_TYPE,		\
			       WirelessConnectDialogPriv))

enum {
  COLUMN_NAME,
  COLUMN_IDENTIFIER,
  N_COLUMNS,
};

static void wireless_connect_dialog_init(WirelessConnectDialog *self)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  priv->network_combo = NULL;
  priv->count_label = NULL;
  priv->security_combo = NULL;

  priv->service_count = 0;
  priv->store = NULL;

  priv->service_selected = FALSE;
  priv->connecting = FALSE;
}

static void wireless_connect_dialog_dispose(GObject *object)
{
  WirelessConnectDialog *self = WIRELESS_CONNECT_DIALOG(object);
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  /* all the gtkwidgets are destroyed at the same time as this dialog */

  /** @todo FIXME: what about priv->store? make sure it's properly destroyed
            is list_store_clear() enough or g_object_unref() or is the store
	    destroyed by the assosiated GtkTreeView??
  */

  G_OBJECT_CLASS(wireless_connect_dialog_parent_class)->dispose(object);
}

static void wireless_connect_dialog_finalize(GObject *object)
{
  G_OBJECT_CLASS(wireless_connect_dialog_parent_class)->finalize(object);
}

static void wireless_connect_dialog_class_init(WirelessConnectDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private(object_class, sizeof(WirelessConnectDialogPriv));

  object_class->dispose = wireless_connect_dialog_dispose;
  object_class->finalize = wireless_connect_dialog_finalize;

  g_assert(klass != NULL);
}

static void network_changed(GtkComboBox *combo, gpointer user_data)
{
  WirelessConnectDialog *self = WIRELESS_CONNECT_DIALOG(user_data);
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  int security;
  const gchar *identifier;
  GtkTreeIter iter;
  gboolean result;

  result = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(priv->network_combo),
					 &iter);
  priv->service_selected = result;

  if (!result)
    return;

    gtk_tree_model_get(GTK_TREE_MODEL(priv->store), &iter,
		       COLUMN_IDENTIFIER, &identifier, -1);

    priv->identifier = identifier;

#if 0
  gtk_tree_model_get(GTK_TREE_MODEL(priv->store), &iter,
		     COLUMN_SERVICE, &service, -1);


  security = void_service_get_security(service);

  gtk_combo_box_set_active(GTK_COMBO_BOX(priv->security_combo), security);
#endif
}

static void security_changed(GtkComboBox *combo, gpointer user_data)
{
  WirelessConnectDialog *self = WIRELESS_CONNECT_DIALOG(user_data);
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  int active, security;
  void *service;
  GtkTreeIter iter;
  gboolean result;

  result = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(priv->network_combo),
					 &iter);
  if (!result) {
    priv->service_selected = FALSE;
    return;
  }

#if 0
  gtk_tree_model_get(GTK_TREE_MODEL(priv->store), &iter,
		     COLUMN_SERVICE, &service, -1);


  g_return_if_fail(VOID_IS_SERVICE(service));
#endif

  active = gtk_combo_box_get_active(GTK_COMBO_BOX(priv->security_combo));
#if 0
  security = void_service_get_security(service);
#endif

  if (active != security)
    priv->service_selected = FALSE;
  else
    priv->service_selected = TRUE;
}

static void update_count_label(WirelessConnectDialog *self)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  gchar buf[200];

  /* FIXME: use g_strdup_printf()? */
  g_snprintf(buf, sizeof(buf), _("%d networks detected"), priv->service_count);

  gtk_label_set_text(GTK_LABEL(priv->count_label), buf);
}

void wireless_connect_dialog_show(WirelessConnectDialog *self, GList *services)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  Service *service;
  GtkTreeIter tree_iter;
  GList *service_iter;

  g_return_if_fail(IS_WIRELESS_CONNECT_DIALOG(self));
  g_return_if_fail(priv != NULL);

  if (gtk_widget_get_visible(GTK_WIDGET(self))) {
    g_warning("Already showing wireless connect dialog, canceling request");
    return;
  }

  priv->service_selected = FALSE;

  //gtk_combo_box_set_active(GTK_COMBO_BOX(priv->security_combo), 0);


  gtk_list_store_clear(priv->store);


  priv->service_count = 0;

  if (services == NULL)
    goto out;
 
  for (service_iter  =  services;
       service_iter != NULL; 
       service_iter  = service_iter->next) {
    service = service_iter->data;

    gtk_list_store_append(priv->store, &tree_iter);
    gtk_list_store_set(priv->store, &tree_iter,
		       COLUMN_NAME, service_get_name(service),
		       COLUMN_IDENTIFIER, service_get_identifier(service),
		       -1);
    priv->service_count++;
  }
  gtk_combo_box_set_active(GTK_COMBO_BOX(priv->network_combo), 0);

 out:
  update_count_label(self);

  gtk_widget_show_all(GTK_WIDGET(self));
}

WirelessConnectDialog *wireless_connect_dialog_new(void)
{
  GtkWidget *vbox, *label, *table;
  WirelessConnectDialogPriv *priv;
  WirelessConnectDialog *self;
  GtkCellRenderer *renderer;

  self = g_object_new(WIRELESS_CONNECT_DIALOG_TYPE, NULL);
  priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  gtk_window_set_title(GTK_WINDOW(self), _("Connect to Wireless Network"));
  gtk_window_set_icon_name(GTK_WINDOW(self), "nm-signal-100");
  gtk_window_set_position(GTK_WINDOW(self), GTK_WIN_POS_CENTER);
  gtk_window_set_deletable(GTK_WINDOW(self), FALSE);
  
  gtk_container_set_border_width(GTK_CONTAINER(self), 12);

  gtk_dialog_add_buttons(GTK_DIALOG(self),
			 GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			 GTK_STOCK_CONNECT, GTK_RESPONSE_ACCEPT,
			 NULL);

  gtk_dialog_set_default_response(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT);

  vbox = gtk_dialog_get_content_area(GTK_DIALOG(self));
  gtk_box_set_spacing(GTK_BOX(vbox), 6);

  table = gtk_table_new(3, 2, FALSE);
  gtk_table_set_col_spacings(GTK_TABLE(table), 6);
  gtk_table_set_row_spacing(GTK_TABLE(table), 0, 3);
  gtk_table_set_row_spacing(GTK_TABLE(table), 1, 6);
  
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(table), TRUE, TRUE, 5);

  label = gtk_label_new(_("Network name:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);

  priv->store = gtk_list_store_new(N_COLUMNS,
				   G_TYPE_STRING,
				   G_TYPE_STRING);

  priv->network_combo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(priv->store));
  
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(priv->network_combo),
			     renderer,
			     TRUE);
  gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(priv->network_combo),
				 renderer,
				 "text",
				 COLUMN_NAME,
				 NULL);


  g_signal_connect(G_OBJECT(priv->network_combo), "changed",
		   G_CALLBACK(network_changed), self);
  gtk_table_attach(GTK_TABLE(table), priv->network_combo, 1, 2, 0, 1,
		    GTK_FILL, GTK_FILL, 0, 0);

  priv->count_label = gtk_label_new(_("0 networks detected"));
  gtk_misc_set_alignment(GTK_MISC(priv->count_label), 0, 0.5);
  update_count_label(self);
  gtk_table_attach(GTK_TABLE(table), priv->count_label, 1, 2, 1, 2,
		    GTK_FILL, 0, 0, 0);

#if 0
  label = gtk_label_new(_("Wireless security:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
		    GTK_FILL, 0, 0, 0);


  priv->security_combo = gtk_combo_box_text_new();

  gtk_combo_box_text_insert_text(GTK_COMBO_BOX_TEXT(priv->security_combo),
			    SERVICE_SECURITY_NONE,
			    _("None"));
  gtk_combo_box_text_insert_text(GTK_COMBO_BOX_TEXT(priv->security_combo),
			    SERVICE_SECURITY_WEP,
			    _("WEP"));
  gtk_combo_box_text_insert_text(GTK_COMBO_BOX_TEXT(priv->security_combo),
			    SERVICE_SECURITY_PSK,
			    _("WPA-PSK"));
  gtk_combo_box_set_active(GTK_COMBO_BOX(priv->security_combo),
			   SERVICE_SECURITY_NONE);

  g_signal_connect(G_OBJECT(priv->security_combo), "changed",
		   G_CALLBACK(security_changed), self);
  gtk_table_attach(GTK_TABLE(table), priv->security_combo, 1, 2, 2, 3,
		    GTK_FILL, GTK_FILL, 0, 0);
#endif

  return self;
}

const gchar *
wireless_connect_dialog_get_selected(WirelessConnectDialog *self)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  return priv->identifier;
}
