# -*- coding: UTF-8 -*-

'''oslib tests'''

# (c) 2007 Canonical Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import unittest, os

from jockey.oslib import OSLib
import sandbox

class OSLibTest(sandbox.LogTestCase):
    '''OSLib tests'''

    def tearDown(self):
        OSLib.inst.hal_get_property_path = OSLib.inst.hal_get_property_path_vapor

    def test_module_blacklisting(self):
        '''module_blacklisted() and blacklist_module()'''

        for m in sandbox.fake_modinfo:
            self.failIf(OSLib.inst.module_blacklisted(m))

            # remove nonexisting entry
            OSLib.inst.blacklist_module(m, False)
            self.failIf(OSLib.inst.module_blacklisted(m))

            # double addition
            OSLib.inst.blacklist_module(m, True)
            self.assert_(OSLib.inst.module_blacklisted(m))
            OSLib.inst.blacklist_module(m, True)
            self.assert_(OSLib.inst.module_blacklisted(m))

        self.assertEqual(open(OSLib.inst.module_blacklist_file).read(), 
            ''.join(['blacklist %s\n' % m for m in sorted(sandbox.fake_modinfo.keys())]))

        # file is removed once it becomes empty
        for m in sandbox.fake_modinfo:
            OSLib.inst.blacklist_module(m, False)
            self.failIf(OSLib.inst.module_blacklisted(m))

        self.failIf(os.path.exists(OSLib.inst.module_blacklist_file))

        # directory gets created if it does not exist
        os.rmdir(os.path.dirname(OSLib.inst.module_blacklist_file))
        OSLib.inst.blacklist_module('vanilla', True)
        self.assert_(OSLib.inst.module_blacklisted('vanilla'))
        OSLib.inst.blacklist_module('vanilla', False)
        self.failIf(os.path.exists(OSLib.inst.module_blacklist_file))

    def test_module_blacklist_load(self):
        '''module blacklist loading.'''

        f = open(OSLib.inst.module_blacklist_file, 'w')
        try:
            f.write('''blacklist vanilla
# FOO BAR

 blacklist  cherry # we do not like red fruits
# blacklist mint
''')
            f.close()

            OSLib.inst._load_module_blacklist()

            self.assert_(OSLib.inst.module_blacklisted('vanilla'))
            self.assert_(OSLib.inst.module_blacklisted('cherry'))
            self.failIf(OSLib.inst.module_blacklisted('chocolate'))
            self.failIf(OSLib.inst.module_blacklisted('mint'))
            self.failIf(OSLib.inst.module_blacklisted('vanilla3d'))
        finally:
            os.unlink(OSLib.inst.module_blacklist_file)
            OSLib.inst._load_module_blacklist()

    def test_module_blacklist_load_thirdparty(self):
        '''module blacklist loading of other files in modprobe.d/.'''

        path = os.path.join(os.path.dirname(OSLib.inst.module_blacklist_file),
            'blacklist-custom')
        f = open(path, 'w')
        try:
            f.write('''blacklist vanilla
# FOO BAR

 blacklist  cherry # we do not like red fruits
# blacklist mint
''')
            f.close()

            OSLib.inst._load_module_blacklist()

            self.assert_(OSLib.inst.module_blacklisted('vanilla'))
            self.assert_(OSLib.inst.module_blacklisted('cherry'))
            self.failIf(OSLib.inst.module_blacklisted('chocolate'))
            self.failIf(OSLib.inst.module_blacklisted('mint'))
            self.failIf(OSLib.inst.module_blacklisted('vanilla3d'))
        finally:
            os.unlink(path)
            OSLib.inst._load_module_blacklist()

    def test_get_system_vendor_product(self):
        '''get_system_vendor_product()'''

        # with existing system name
        v, p = OSLib.inst.get_system_vendor_product()
        self.assertEqual(v, 'VaporTech Inc.')
        self.assertEqual(p, 'Virtualtop X-1')

        # without existing system name
        OSLib.inst.hal_get_property_path = OSLib.inst.hal_get_property_path_empty
        v, p = OSLib.inst.get_system_vendor_product()
        self.assertEqual(v, '')
        self.assertEqual(p, '')

        # with failing hal-get-property
        OSLib.inst.hal_get_property_path = OSLib.inst.hal_get_property_path_fail
        v, p = OSLib.inst.get_system_vendor_product()
        self.assertEqual(v, '')
        self.assertEqual(p, '')

        # with nonexisting hal-get-property
        OSLib.inst.hal_get_property_path = '/nonexisting'
        v, p = OSLib.inst.get_system_vendor_product()
        self.assertEqual(v, '')
        self.assertEqual(p, '')

        # actual system's implementation is a string
        v, p = OSLib(client_only=True).get_system_vendor_product()
        self.assert_(hasattr(v, 'startswith'))
        self.assert_(hasattr(p, 'startswith'))

    def test_package_query(self):
        '''package querying'''

        # use real OSLib here, not test suite's fake implementation
        o = OSLib()

        self.assertEqual(o.package_installed('coreutils'), True)
        self.assertEqual(o.package_installed('nonexisting'), False)

        self.assertEqual(o.is_package_free('coreutils'), True)
        self.assertRaises(ValueError, o.is_package_free, 'nonexisting')

        (short, long) = o.package_description('bash')
        self.assert_('sh' in short.lower())
        self.assert_('shell' in long) 
        self.failIf('size:' in long) 
        self.assertRaises(ValueError, o.package_description, 'nonexisting')

        self.assertRaises(ValueError, o.package_files, 'nonexisting')
        coreutils_files = o.package_files('coreutils')
        self.assert_('/bin/cat' in coreutils_files)
        self.assert_('/bin/tail' in coreutils_files or 
            '/usr/bin/tail' in coreutils_files)

    def test_package_install(self):
        '''package installation/removal

        This will just do some very shallow tests if this is not run as root.
        '''
        # test package; this is most likely not installed yet (test suite will
        # abort if it is)
        test_package = 'lrzsz'

        # use real OSLib here, not test suite's fake implementation
        o = OSLib()

        self.assertRaises(SystemError, o.install_package, 'nonexisting', None)
        # this should not crash, since it is not installed
        o.remove_package('nonexisting', None)

        if os.getuid() != 0:
            return

        self.failIf(o.package_installed(test_package), 
            '%s must not be installed for this test' % test_package)

        # test without progress reporting
        o.install_package(test_package, None)
        self.assert_(o.package_installed(test_package))
        o.remove_package(test_package, None)
        self.failIf(o.package_installed(test_package))
