/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-gadget-item.h"

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <clutk/clutk.h>
#include <launcher/launcher.h>
#include <netbook-launcher/netbook-launcher.h>
#include <clutter-gtk/clutter-gtk.h>
#include <libgadget/libgadget.h>

#include "nl-gadget-icon.h"

G_DEFINE_TYPE (NlGadgetItem, nl_gadget_item, G_TYPE_OBJECT);

#define NL_GADGETS_ITEM_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE \
        ((obj), NL_TYPE_GADGETS_ITEM, NlGadgetItemPrivate))

struct _NlGadgetItemPrivate
{
  NlShell       *shell;
  GadgetManager *manager;

  NlPixbufCache *cache;
  ClutterActor  *icon_view;
  ClutterActor  *item;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */
static void on_item_clicked (ClutterActor *actor, NlGadgetItem *item);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlGadgetItemPrivate *priv;

  g_return_if_fail (NL_IS_GADGETS_ITEM (object));
  priv = NL_GADGETS_ITEM_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlGadgetItemPrivate *priv;

  g_return_if_fail (NL_IS_GADGETS_ITEM (object));
  priv = NL_GADGETS_ITEM_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}


static void
nl_gadget_item_finalize (GObject *object)
{
  NlGadgetItemPrivate *priv;

  priv = NL_GADGETS_ITEM_GET_PRIVATE (object);

  if (priv->cache)
    {
      g_object_unref (priv->cache);
      priv->cache = NULL;
    }
  if (CLUTTER_IS_ACTOR (priv->icon_view))
    {
      g_object_unref (priv->icon_view);
      priv->icon_view = NULL;
    }
  if (priv->manager)
    {
      g_object_unref (priv->manager);
      priv->manager = NULL;
    }

  G_OBJECT_CLASS (nl_gadget_item_parent_class)->finalize (object);
}

static void
on_content_view_changed (NlContentView *view,
                         ClutterActor  *new_view,
                         NlGadgetItem  *self)
{
  g_return_if_fail (NL_IS_GADGETS_ITEM (self));

  nl_sidebar_item_set_active (NL_SIDEBAR_ITEM (self->priv->item),
                              self->priv->icon_view == new_view ? TRUE : FALSE);
}

static void
nl_gadget_item_constructed (GObject *object)
{
  NlGadgetItemPrivate *priv = NL_GADGETS_ITEM (object)->priv;
  ClutterActor     *item;
  GdkPixbuf        *pixbuf;
  ClutterContainer *container;

  priv->manager = gadget_manager_get_default ();

  container = (ClutterContainer *)nl_shell_get_sidebar (priv->shell);

  pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache,
                                          "unr-applications-widgets",
                                          48);
  if (!pixbuf)
    pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache,
                                          "weather-few-clouds",
                                          48);

  item = priv->item = nl_sidebar_item_new (_("Widgets"),
                      _("Browse widgets and add them "
                        "to your favorites"),
                      pixbuf,
                      5);
  clutter_container_add_actor (container, item);
  clutter_actor_show (item);

  g_signal_connect (item, "clicked", G_CALLBACK (on_item_clicked), object);
  g_signal_connect (nl_shell_get_content_view (priv->shell), "changed",
                    G_CALLBACK (on_content_view_changed), object);

  g_object_unref (pixbuf);
}

static void
nl_gadget_item_class_init (NlGadgetItemClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = nl_gadget_item_finalize;
  obj_class->constructed  = nl_gadget_item_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  /* Install properties */
  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlGadgetItemPrivate));
}

static void
nl_gadget_item_init (NlGadgetItem *self)
{
  NlGadgetItemPrivate *priv;

  priv = self->priv = NL_GADGETS_ITEM_GET_PRIVATE (self);

  priv->cache = nl_pixbuf_cache_get_default ();
}

/*
 * Public methods
 */
NlGadgetItem *
nl_gadget_item_new (NlShell *shell)

{
  NlGadgetItem *gadget_item = NULL;

  gadget_item = g_object_new (NL_TYPE_GADGETS_ITEM,
                              "shell", shell,
                              NULL);

  return gadget_item;
}

/*
 * Private methods
 */
static gint
sort_gadgets (GadgetInfo *a, GadgetInfo *b)
{
  if (!a)
    return -1;
  if (!b)
    return 1;
  if (G_UNLIKELY (a ==b))
    return 0;

  return g_strcmp0 (a->name, b->name);
}

static void
on_item_clicked (ClutterActor *actor, NlGadgetItem *self)
{
  NlGadgetItemPrivate *priv;
  NlContentView *view;
  GList         *gadgets, *g;
  ClutterActor  *scroll;
  ClutterActor  *icon_view;
  GdkPixbuf     *icon;

  g_return_if_fail (NL_IS_GADGETS_ITEM (self));
  priv = self->priv;

  view = nl_shell_get_content_view (NL_SHELL (priv->shell));

  scroll = nl_scroll_view_new ();

  icon_view = ctk_icon_view_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (scroll), icon_view);
  clutter_actor_show (icon_view);

  /* Just in case the sources haven't been loaded before */
  gadget_manager_load_sources (priv->manager);

  gadgets = gadget_manager_get_available_gadgets (priv->manager);
  gadgets = g_list_sort (gadgets, (GCompareFunc)sort_gadgets);

  icon= nl_pixbuf_cache_icon_for_name (priv->cache,
                                       "weather-few-clouds",
                                       128);

  for (g = gadgets; g; g = g->next)
    {
      GadgetInfo   *info = g->data;
      ClutterActor *button;

      if (!info)
        continue;

      button = nl_gadget_icon_new (priv->shell, info);
      clutter_container_add_actor (CLUTTER_CONTAINER (icon_view), button);
      clutter_actor_show (button);
    }
  g_object_unref (icon);

  g_list_foreach (gadgets, (GFunc)gadget_info_free, NULL);
  g_list_free (gadgets);

  priv->icon_view = scroll;
  nl_content_view_set_child (view, scroll);
}
