/* TI BQ20Z75 Gauge Driver
 *
 * Copyright (C) 20011 Quanta Computer Inc.
 * Author: Wayne Lin <wayne.lin@quantatw.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*-----------------------------------------------------------------------------
 * Global Include files
 *---------------------------------------------------------------------------*/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/power_supply.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/bq20z75.h>

/*-----------------------------------------------------------------------------
 * Local Include files
 *---------------------------------------------------------------------------*/
#include "bq20z75_battery.h"

/*-----------------------------------------------------------------------------
 * Constants
 *---------------------------------------------------------------------------*/
//#define DEBUG_MSG
#define TI_BQ20Z75_DRV_NAME "TI Bq20z75 gauge IC"						// "/sys/bus/i2c/drivers/"
#define TI_BQ20Z75_DEV_NAME "bq20z75-gauge"							// "/sys/bus/i2c/devices/"
#define TI_BQ20Z75_POWER_SUPPLY_DRV_NAME "bq20z75-battery"					// "/sys/class/power_supply/battery"
#define AC_POWER_SUPPLY_DRV_NAME "ac"								// "/sys/class/power_supply/ac"

/*-----------------------------------------------------------------------------
 * Marcos
 *---------------------------------------------------------------------------*/
#define to_bq20z75_device_info(x) container_of((x), struct bq20z75_drv_info, \
                                              power_supply);
#if defined(DEBUG_MSG)
#define pr_error_i2c_read(func, addr)					pr_err("%s: i2c read at address 0x%x failed\n",(func), (addr))
#define pr_error_i2c_write(func, addr)					pr_err("%s: i2c write at address 0x%x failed\n",(func), (addr))
#define pr_error_get_bat_prop(func, line, addr)				pr_err("%s %d: Get address 0x%x properties failed\n",(func), (line), (addr))
#define pr_error_get_bat_prop_range(func, line, addr)			pr_err("%s %d: Get address 0x%x properties range exceeding\n",(func), (line), (addr))
#define pr_error_invalid_prop(func, line) 				pr_err("%s %d: invalid property\n", (func), (line))
#else
#define pr_error_i2c_read(func, addr)
#define pr_error_i2c_write(func, addr)
#define pr_error_get_bat_prop(func, line, addr)
#define pr_error_get_bat_prop_range(func, line, addr)
#define pr_error_invalid_prop(func, line)
#endif

/*-----------------------------------------------------------------------------
 * Global variables
 *---------------------------------------------------------------------------*/
struct bq20z75_drv_info *bq20z75_drv = NULL;
static enum power_supply_property bq20z75_gauge_properties[] = {
        POWER_SUPPLY_PROP_STATUS,
        POWER_SUPPLY_PROP_HEALTH,
        POWER_SUPPLY_PROP_PRESENT,
        POWER_SUPPLY_PROP_TECHNOLOGY,
        POWER_SUPPLY_PROP_VOLTAGE_NOW,
        POWER_SUPPLY_PROP_CURRENT_NOW,
        POWER_SUPPLY_PROP_CAPACITY,
        POWER_SUPPLY_PROP_TEMP,
        POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG,
        POWER_SUPPLY_PROP_TIME_TO_FULL_AVG,
        POWER_SUPPLY_PROP_SERIAL_NUMBER,
        POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN,
        POWER_SUPPLY_PROP_CHARGE_NOW,
        POWER_SUPPLY_PROP_CHARGE_FULL,
        POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN,
	POWER_SUPPLY_PROP_MODEL_NAME,
	POWER_SUPPLY_PROP_MANUFACTURER,
};
static enum power_supply_property ac_properties[] = {
	POWER_SUPPLY_PROP_ONLINE,
};

static char SerialNum[5] = "";						// It will be used to store serial number global variable
static char ModelName[7] = "";						// It will be used to store manufacture name global variable
/*-----------------------------------------------------------------------------
 * Local Functions
 *---------------------------------------------------------------------------*/

static int bq20z75_gauge_read_word_data(struct i2c_client *client, u8 address)
{
	s32 ret = -1;

	ret = i2c_smbus_read_word_data(client, address);
	if (ret < 0) {
		pr_error_i2c_read(__func__, address);
		return ret;
	}
	return le16_to_cpu(ret);
}

static int bq20z75_gauge_write_word_data(struct i2c_client *client, u8 address,
	u16 value)
{
	s32 ret = -1;

	ret = i2c_smbus_write_word_data(client, address, le16_to_cpu(value));
	if (ret < 0) {
		pr_error_i2c_write(__func__, address);
		return ret;
	}
	return 0;
}

/*-----------------------------------------------------------------------------
 * Battery properties get functions
 *---------------------------------------------------------------------------*/

static int bq20z75_gauge_get_status(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;
	
	val->intval = POWER_SUPPLY_STATUS_UNKNOWN;
	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_STATUS);
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_STATUS);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_STATUS_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_STATUS);
		return ret;
	}
	if (gpio_get_value(bq20z75_drv->gpio_chg_pg))
		val->intval = POWER_SUPPLY_STATUS_DISCHARGING;
	else if (ret & BQ20Z75_REG_STATUS_FULL)
		val->intval = POWER_SUPPLY_STATUS_FULL;
	else
		val->intval = POWER_SUPPLY_STATUS_CHARGING;

	return 0;
}

static int bq20z75_gauge_get_health(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->intval = POWER_SUPPLY_HEALTH_UNKNOWN;

	ret = bq20z75_gauge_write_word_data(client, BQ20Z75_REG_MANUFACTURE_ACCESS, 
									BQ20Z75_REG_MANUFACTURE_ACCESS_STATUS);
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_MANUFACTURE_ACCESS);
		return ret;
	}

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_MANUFACTURE_ACCESS);
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_MANUFACTURE_ACCESS);
		return ret;
	}

	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_MANUFACTURE_ACCESS_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_MANUFACTURE_ACCESS);
		return ret;
	}
	
	ret = ret & BQ20Z75_REG_MANUFACTURE_ACCESS_STATUS_MASK;
	if (ret == BQ20Z75_REG_MANUFACTURE_ACCESS_STATUS_OVER_TEMP)
		val->intval = POWER_SUPPLY_HEALTH_OVERHEAT;
	else if (ret == BQ20Z75_REG_MANUFACTURE_ACCESS_STATUS_OVER_CUR)
		val->intval = POWER_SUPPLY_HEALTH_UNSPEC_FAILURE;
	else if (ret == BQ20Z75_REG_MANUFACTURE_ACCESS_STATUS_FAILURE)
		val->intval = POWER_SUPPLY_HEALTH_DEAD;
	else
		val->intval = POWER_SUPPLY_HEALTH_GOOD;

	return 0;
}

static int bq20z75_gauge_get_present(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->intval = 0;

	ret = bq20z75_gauge_write_word_data(client, BQ20Z75_REG_MANUFACTURE_ACCESS,
						BQ20Z75_REG_MANUFACTURE_ACCESS_STATUS);
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_MANUFACTURE_ACCESS);
		return ret;
	}

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_MANUFACTURE_ACCESS);
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_MANUFACTURE_ACCESS);
		return ret;
	}

	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_MANUFACTURE_ACCESS_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_MANUFACTURE_ACCESS);
		return ret;
	}

	ret = ret & BQ20Z75_REG_MANUFACTURE_ACCESS_STATUS_MASK;
	if (ret == BQ20Z75_REG_MANUFACTURE_ACCESS_STATUS_BAT_REMOVE)
		val->intval = 0;
	else
		val->intval = 1;

	return 0;
}

static int bq20z75_gauge_get_technology(union power_supply_propval *val)
{
	val->intval = POWER_SUPPLY_TECHNOLOGY_LION;
	return 0;
}

static int bq20z75_gauge_get_voltage_now(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_VOLTAGE);
	
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_VOLTAGE);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_VOLTAGE_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_VOLTAGE);
		return ret;
	}
	val->intval = ret * 1000;
	return 0;
}

static int bq20z75_gauge_get_current_now(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_STATUS);
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_STATUS);
		return ret;
	}

	if ((ret & BQ20Z75_REG_STATUS_FULL) && !gpio_get_value(bq20z75_drv->gpio_chg_pg)){
		val->intval = 0;
		return 0;
	}

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_CURRENT);
	
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_CURRENT);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_CURRENT_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_CURRENT);
		return ret;
	}

	if (ret & 0x8000)
		val->intval = -( 32768 - (ret & 0x7fff)) * 1000;
	else
		val->intval = (ret & 0x7fff) * 1000;
	return 0;
}

static int bq20z75_gauge_get_capacity(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;
	
	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_REL_CHARGE);
	
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_REL_CHARGE);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_REL_CHARGE_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_REL_CHARGE);
		return ret;
	}

	val->intval = ret;
	return 0;
}

static int bq20z75_gauge_get_temp(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;
	
	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_TEMP);
	
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_TEMP);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_TEMP_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_TEMP);
		return ret;
	}

	/* tempture transfer 0.1oK->0.1oC*/
	val->intval = ((ret) - 2732);
	return 0;
}

static int bq20z75_gauge_get_time_to_empty_avg(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;
	
	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_AVG_TIME_TO_EMPTY);
	
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_AVG_TIME_TO_EMPTY);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_AVG_TIME_TO_EMPTY_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_AVG_TIME_TO_EMPTY);
		return ret;
	}
	
	val->intval = ret * 1000;
	return 0;
}

static int bq20z75_gauge_get_time_to_full_avg(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;
	
	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_AVG_TIME_TO_FULL);
	
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_AVG_TIME_TO_FULL);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_AVG_TIME_TO_FULL_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_AVG_TIME_TO_FULL);
		return ret;
	}

	val->intval = ret * 1000;
	return 0;
}

static int bq20z75_gauge_get_serial_number(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;
	
	val->strval = "Unknown";
	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_SERIAL_NUM);
	
	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_SERIAL_NUM);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_SERIAL_NUM_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_SERIAL_NUM);
		return ret;
	}
	ret = sprintf(SerialNum, "%04x", ret);
	val->strval = SerialNum;
	return 0;
}

static int bq20z75_gauge_get_voltage_max_design(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_CHARGING_VOLTAGE);

	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_CHARGING_VOLTAGE);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_CHARGING_VOLTAGE_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_CHARGING_VOLTAGE);
		return ret;
	}
	val->intval = ret * 1000;
	return 0;
}

static int bq20z75_gauge_get_charge_now(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_REMAIN_CAPACITY);

	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_REMAIN_CAPACITY);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_REMAIN_CAPACITY_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_REMAIN_CAPACITY);
		return ret;
	}
	val->intval = ret * 1000;
	return 0;
}

static int bq20z75_gauge_get_charge_full(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_FULL_CAPACITY);

	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_FULL_CAPACITY);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_FULL_CAPACITY_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_FULL_CAPACITY);
		return ret;
	}
	val->intval = ret * 1000;
	return 0;
}

static int bq20z75_gauge_get_charge_full_design(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->intval = 0;

	ret = bq20z75_gauge_read_word_data(client, BQ20Z75_REG_DESIGN_CAPACITY);

	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_DESIGN_CAPACITY);
		return ret;
	}
	// To check if the return value exceeding
	if (ret > BQ20Z75_REG_DESIGN_CAPACITY_MAX) {
		pr_error_get_bat_prop_range(__func__, __LINE__, BQ20Z75_REG_DESIGN_CAPACITY);
		return ret;
	}
	val->intval = ret * 1000;
	return 0;
}

static int bq20z75_gauge_get_manufacturer(union power_supply_propval *val)
{
	val->strval = "Vodafone";
	return 0;
}

static int bq20z75_gauge_get_model_name(union power_supply_propval *val)
{
	struct i2c_client *client = bq20z75_drv->client;
	s32 ret = -1;

	val->strval = "Unknown";
	ret = i2c_smbus_read_i2c_block_data(client,
					BQ20Z75_REG_DEVICE_NAME,
					sizeof(ModelName),
					ModelName);

	if (ret < 0) {
		pr_error_get_bat_prop(__func__, __LINE__, BQ20Z75_REG_DEVICE_NAME);
		return ret;
	}

	val->strval = ModelName;
	return 0;
}

static int ac_get_online(union power_supply_propval *val)
{
	struct bq20z75_drv_info *pdata = bq20z75_drv;

	/* AC power plug in? */
	if (!gpio_get_value(pdata->gpio_chg_pg))
		val->intval = 1;	// Plug-in
	else
		val->intval = 0;	// Remove

	return 0;
}

/*-----------------------------------------------------------------------------
 * Global Functions
 *---------------------------------------------------------------------------*/

static int bq20z75_gauge_get_property(struct power_supply *psy,
	enum power_supply_property psp,
	union power_supply_propval *val)
{
/*	int ret;
	struct bq20z75_drv_info *pdata = to_bq20z75_device_info(psy);
	struct i2c_client *client = pdata->client;
*/
	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		bq20z75_gauge_get_status(val);
		break;
	case POWER_SUPPLY_PROP_HEALTH:
		bq20z75_gauge_get_health(val);
		break;
	case POWER_SUPPLY_PROP_PRESENT:
		bq20z75_gauge_get_present(val);
		break;
	case POWER_SUPPLY_PROP_TECHNOLOGY:
		bq20z75_gauge_get_technology(val);
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		bq20z75_gauge_get_voltage_now(val);
		break;
	case POWER_SUPPLY_PROP_CURRENT_NOW:
		bq20z75_gauge_get_current_now(val);
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		bq20z75_gauge_get_capacity(val);
		break;
	case POWER_SUPPLY_PROP_TEMP:
		bq20z75_gauge_get_temp(val);
		break;
	case POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG:
		bq20z75_gauge_get_time_to_empty_avg(val);
		break;
	case POWER_SUPPLY_PROP_TIME_TO_FULL_AVG:
		bq20z75_gauge_get_time_to_full_avg(val);
		break;
	case POWER_SUPPLY_PROP_SERIAL_NUMBER:
		bq20z75_gauge_get_serial_number(val);
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN:
		bq20z75_gauge_get_voltage_max_design(val);
		break;
	case POWER_SUPPLY_PROP_CHARGE_NOW:
		bq20z75_gauge_get_charge_now(val);
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL:
		bq20z75_gauge_get_charge_full(val);
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN:
		bq20z75_gauge_get_charge_full_design(val);
		break;
	case POWER_SUPPLY_PROP_MANUFACTURER:
		bq20z75_gauge_get_manufacturer(val);
		break;
	case POWER_SUPPLY_PROP_MODEL_NAME:
		bq20z75_gauge_get_model_name(val);
		break;
	default:
		pr_error_invalid_prop(__func__, __LINE__);
		return -EINVAL;
	}

	return 0;
}

static int ac_get_property(struct power_supply *psy,
	enum power_supply_property psp,
	union power_supply_propval *val)
{
	switch (psp) {
	case POWER_SUPPLY_PROP_ONLINE:
		ac_get_online(val);
		break;
	default:
		pr_error_invalid_prop(__func__, __LINE__);
		return -EINVAL;
	}

	return 0;
}

static irqreturn_t ac_powerin_interrupt_handler(int irq, void *dev)
{
	struct bq20z75_drv_info *pdata = (struct bq20z75_drv_info *)dev;

	disable_irq_nosync(pdata->irq_chg_pg);
	mutex_lock(&pdata->lock);
	pdata->stat_chg = true;
	mutex_unlock(&pdata->lock);
	schedule_delayed_work(&pdata->workqueue, 20);
	return IRQ_HANDLED;
}

static irqreturn_t bat_powerin_interrupt_handler(int irq, void *dev)
{
	struct bq20z75_drv_info *pdata = (struct bq20z75_drv_info *)dev;

	disable_irq_nosync(pdata->irq_batin);
	mutex_lock(&pdata->lock);
	pdata->stat_batin = true;
	mutex_unlock(&pdata->lock);
	schedule_delayed_work(&pdata->workqueue, 20);
	return IRQ_HANDLED;
}

static void power_change_work_handler(struct work_struct *work)
{
	struct bq20z75_drv_info *pdata =
		container_of(work, struct bq20z75_drv_info, workqueue.work);

	mutex_lock(&pdata->lock);
	if (pdata->stat_chg)
		power_supply_changed(&pdata->power_supply_ac);
	else if (pdata->stat_batin)
		power_supply_changed(&pdata->power_supply_bat);
	mutex_unlock(&pdata->lock);
	if (pdata->stat_chg)
	{
		pdata->stat_chg = false;
		enable_irq(pdata->irq_chg_pg);
	}else if (pdata->stat_batin)
	{
		pdata->stat_batin = false;
		enable_irq(pdata->irq_batin);
	}
}

static int bq20z75_gauge_probe(struct i2c_client *client,
	const struct i2c_device_id *id)
{
	struct bq20z75_dev_info *pdata = NULL;
	int rc = 0;

	/* Get platform data */
	pdata = (struct bq20z75_dev_info *)client->dev.platform_data;
	if (pdata == NULL) {
		pr_err("%s lack of platform data!\n", __func__);
		return -ENODEV;
	}

	/* Driver data initialization */
	bq20z75_drv = kzalloc(sizeof(struct bq20z75_drv_info), GFP_KERNEL);
	if (!bq20z75_drv) {
		pr_err("%s drv data initial failed!\n", __func__);
		return -ENOMEM;
	}

	/* Power supply structure initialization */
	bq20z75_drv->irq_chg_pg = pdata->irq_chg_pg;
	bq20z75_drv->irq_batin = pdata->irq_batin;
	bq20z75_drv->stat_chg = false;
	bq20z75_drv->stat_batin = false;
	bq20z75_drv->client = client;
	bq20z75_drv->power_supply_bat.name = TI_BQ20Z75_POWER_SUPPLY_DRV_NAME;
	bq20z75_drv->power_supply_bat.type = POWER_SUPPLY_TYPE_BATTERY;
	bq20z75_drv->power_supply_bat.properties = bq20z75_gauge_properties;
	bq20z75_drv->power_supply_bat.num_properties =
		ARRAY_SIZE(bq20z75_gauge_properties);
	bq20z75_drv->power_supply_bat.get_property = bq20z75_gauge_get_property;
	bq20z75_drv->power_supply_bat.use_for_apm = 1;
	
	bq20z75_drv->power_supply_ac.name = AC_POWER_SUPPLY_DRV_NAME;
	bq20z75_drv->power_supply_ac.type = POWER_SUPPLY_TYPE_MAINS;
	bq20z75_drv->power_supply_ac.properties = ac_properties;
	bq20z75_drv->power_supply_ac.num_properties =
		ARRAY_SIZE(ac_properties);
	bq20z75_drv->power_supply_ac.get_property = ac_get_property;

	i2c_set_clientdata(client, bq20z75_drv);

	rc = power_supply_register(&client->dev, &bq20z75_drv->power_supply_bat);
	if (rc) {
		pr_err("Register %s power supply fail %s %d\n",TI_BQ20Z75_DRV_NAME, __func__, __LINE__);
		i2c_set_clientdata(client, NULL);
		kfree(bq20z75_drv);
		pdata = NULL;
		return rc;
	}

	rc = power_supply_register(&client->dev, &bq20z75_drv->power_supply_ac);
	if (rc) {
		pr_err("Register %s power supply fail %s %d\n",TI_BQ20Z75_DRV_NAME, __func__, __LINE__);
		power_supply_unregister(&bq20z75_drv->power_supply_bat);
		i2c_set_clientdata(client, NULL);
		kfree(bq20z75_drv);
		pdata = NULL;
		return rc;
	}

	/* Work queue and lock initial */
	INIT_DELAYED_WORK(&bq20z75_drv->workqueue, power_change_work_handler);
	mutex_init(&bq20z75_drv->lock);

	/* AC power good related setting */
	bq20z75_drv->gpio_chg_pg = irq_to_gpio(bq20z75_drv->irq_chg_pg);

	rc = request_irq(bq20z75_drv->irq_chg_pg,
			  ac_powerin_interrupt_handler,
			  IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
			  TI_BQ20Z75_DRV_NAME,
			  bq20z75_drv);
	if (rc) {
		pr_err("Register ac power good irq failed\n");
		power_supply_unregister(&bq20z75_drv->power_supply_bat);
		power_supply_unregister(&bq20z75_drv->power_supply_ac);
		i2c_set_clientdata(client, NULL);
		kfree(bq20z75_drv);
		pdata = NULL;
		return rc;
	}

	/* BAT-IN related setting */
	bq20z75_drv->gpio_batin = irq_to_gpio(bq20z75_drv->irq_batin);

	rc = request_irq(bq20z75_drv->irq_batin,
			  bat_powerin_interrupt_handler,
			  IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
			  TI_BQ20Z75_DRV_NAME,
			  bq20z75_drv);
	if (rc) {
		pr_err("Register bat in irq failed\n");
		power_supply_unregister(&bq20z75_drv->power_supply_bat);
		power_supply_unregister(&bq20z75_drv->power_supply_ac);
		free_irq(pdata->irq_chg_pg, bq20z75_drv);
		i2c_set_clientdata(client, NULL);
		kfree(bq20z75_drv);
		pdata = NULL;
		return rc;
	}

	pr_info("Register %s power supply successfully\n",TI_BQ20Z75_DRV_NAME);
	return 0;
}

static int bq20z75_gauge_remove(struct i2c_client *client)
{
	struct bq20z75_drv_info *pdata = i2c_get_clientdata(client);

	power_supply_unregister(&pdata->power_supply_bat);
	power_supply_unregister(&pdata->power_supply_ac);
	free_irq(pdata->irq_chg_pg, pdata);
	free_irq(pdata->irq_batin, pdata);
	i2c_set_clientdata(client, NULL);
	pdata = NULL;
	kfree(pdata);

	return 0;
}

#if defined CONFIG_PM
static int bq20z75_gauge_suspend(struct i2c_client *client,
	pm_message_t state)
{
	return 0;
}

static int bq20z75_gauge_resume(struct i2c_client *client)
{
	return 0;
}
#endif

static const struct i2c_device_id bq20z75_gauge_id[] = {
	{ TI_BQ20Z75_DEV_NAME, 0 },
	{}
};

static struct i2c_driver bq20z75_gauge_driver = {
	.probe		= bq20z75_gauge_probe,
	.remove		= bq20z75_gauge_remove,
#if defined CONFIG_PM
	.suspend	= bq20z75_gauge_suspend,
	.resume		= bq20z75_gauge_resume,
#endif
	.id_table	= bq20z75_gauge_id,
	.driver = {
		.owner = THIS_MODULE,
		.name  = TI_BQ20Z75_DRV_NAME,
	},
};

static int __init bq20z75_gauge_init(void)
{
	return i2c_add_driver(&bq20z75_gauge_driver);
}

static void __exit bq20z75_gauge_exit(void)
{
	i2c_del_driver(&bq20z75_gauge_driver);
}

module_init(bq20z75_gauge_init);
module_exit(bq20z75_gauge_exit);

MODULE_AUTHOR("Quanta Computer Inc.");
MODULE_DESCRIPTION("TI BQ20Z75 Driver");
MODULE_VERSION("0.1.6");
MODULE_LICENSE("GPL v2");
