#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/platform_device.h>
#include <linux/spi/spi.h>
#include <linux/jiffies.h>
#include <asm/delay.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>
#include <sound/tlv.h>
#include <asm/div64.h>

#include "rt5632.h"

#if REALTEK_HWDEP

#include <linux/ioctl.h>
#include <linux/types.h>

#endif

#define RT_VOL_RESCALE  1

#define AUDIO_NAME "rt5632"
#define RT5632_VERSION "0.06 alsa 1.0.23"
#define ALSA_SOC_VERSION "1.0.23"


static int rt5632_ChangeCodecPowerStatus(struct snd_soc_codec *codec,int power_state);
static int rt5632_AudioOutEnable(struct snd_soc_codec *codec,unsigned short int WavOutPath,int Mute);
static int Enable_ADC_Input_Source(struct snd_soc_codec *codec,unsigned short int ADC_Input_Sour,int Enable);
static void hp_depop_mode2(struct snd_soc_codec *codec);

struct rt5632_priv {
	unsigned int stereo_sysclk;
	unsigned int voice_sysclk;
};

struct rt5632_init_reg {
	u8 reg_index;
	u16 reg_value;	
};

static struct rt5632_init_reg rt5632_init_list[] = {

	{RT5632_HP_OUT_VOL, 		0x8888},	//HP Output Volume
	{RT5632_SPK_OUT_VOL, 		0x8080},	//SPK Output Volume
	{RT5632_DAC_AND_MIC_CTRL, 	0xee03},	//DAC Mic Route
	{RT5632_OUTPUT_MIXER_CTRL, 	0x0748},	//Output Mixer Control
	{RT5632_MIC_CTRL, 			0x0500},	//Mic Control
	{RT5632_VOICE_DAC_OUT_VOL, 	0x6008},	//Voice DAC Volume
	{RT5632_ADC_REC_MIXER, 		0x3f3f},	//ADC Rec Mixer
#ifdef RT_VOL_RESCALE
	{RT5632_GEN_CTRL_REG1, 		0x0c06},    //General Control
	{RT5632_DMIC_BOOST_CTRL, 	0xD803}     //Default Dmic and boost 18DB // default : 0xD803 - modified by Johnny
#else    	
	{RT5632_GEN_CTRL_REG1, 		0x0c0a},    //General Control
	{RT5632_DMIC_BOOST_CTRL, 	0xD805}     //Default Dmic and boost 30DB // default : 0xD805
#endif	

};

#define RT5632_INIT_REG_NUM ARRAY_SIZE(rt5632_init_list)

/*
 *	bit[0]  for linein playback switch
 *	bit[1] phone
 *	bit[2] mic1
 *	bit[3] mic2
 *	bit[4] vopcm
 *	
 */
#define HPL_MIXER 0x80
#define HPR_MIXER 0x82
static unsigned int reg80 = 0, reg82 = 0;

/*
 *	bit[0][1][2] use for aec control
 *  bit[3] for none  
 *	bit[4] for SPKL pga
 *	bit[5] for SPKR pga
 *	bit[6] for hpl pga
 *	bit[7] for hpr pga
 *  bit[8] for dump dsp
 */
 #define VIRTUAL_REG_FOR_MISC_FUNC 0x84
static unsigned int reg84 = 0;


static const u16 rt5632_reg[] = {
	0x59b4, 0x8080, 0x8080, 0x8080,		/*reg00-reg06*/
	0xc800, 0xe808, 0x1010, 0x0808,		/*reg08-reg0e*/
	0xe0ef, 0xcbcb, 0x7f7f, 0x0000,		/*reg10-reg16*/
	0xe010, 0x0000, 0x8008, 0x2007,		/*reg18-reg1e*/
	0x0000, 0x0000, 0x00c0, 0xef00,		/*reg20-reg26*/
	0x0000, 0x0000, 0x0000, 0x0000,		/*reg28-reg2e*/
	0x0000, 0x0000, 0x0000, 0x0000,		/*reg30-reg36*/
	0x0000, 0x0000, 0x0000, 0x0000, 	/*reg38-reg3e*/
	0x0c0a, 0x0000, 0x0000, 0x0000,		/*reg40-reg46*/
	0x0029, 0x0000, 0xbe3e, 0x3e3e,		/*reg48-reg4e*/
	0x0000, 0x0000, 0x803a, 0x0000,		/*reg50-reg56*/
	0x0000, 0x0009, 0x0000, 0x3000,		/*reg58-reg5e*/
	0x3075, 0x1010, 0x3110, 0x0000,		/*reg60-reg66*/
	0x0553, 0x0000, 0x0000, 0x0000,		/*reg68-reg6e*/
	0x0000, 0x0000, 0x0000, 0x0000,		/*reg70-reg76*/
	0x0000, 0x0000, 0x0000, 0x0000,     /*reg78-reg7e*/
};


static struct snd_soc_device *rt5632_socdev;

static inline unsigned int rt5632_read_reg_cache(struct snd_soc_codec *codec, 
	unsigned int reg)
{
	u16 *cache = codec->reg_cache;

	if (reg > 0x7e)
		return 0;
	return cache[reg / 2];
}


static unsigned int rt5632_read_hw_reg(struct snd_soc_codec *codec, unsigned int reg) 
{
	u8 data[2] = {0};
	unsigned int value = 0x0;
	
	data[0] = reg;
	if (codec->hw_write(codec->control_data, data, 1) == 1)
	{
		i2c_master_recv(codec->control_data, data, 2);
		value = (data[0] << 8) | data[1];
//		printk(KERN_DEBUG "%s read reg%x = %x\n", __func__, reg, value);
		return value;
	}
	else
	{
		printk(KERN_DEBUG "%s failed\n", __func__);
		return -EIO;
	}
}


static unsigned int rt5632_read(struct snd_soc_codec *codec, unsigned int reg)
{
	if ((reg == 0x80)
		|| (reg == 0x82)
		|| (reg == 0x84))
		return (reg == 0x80) ? reg80 : ((reg == 0x82) ? reg82 : reg84);
	
		return rt5632_read_hw_reg(codec, reg);
}


static inline void rt5632_write_reg_cache(struct snd_soc_codec *codec,
	unsigned int reg, unsigned int value)
{
	u16 *cache = codec->reg_cache;
	if (reg > 0x7E)
		return;
	cache[reg / 2] = value;
}

static int rt5632_write(struct snd_soc_codec *codec, unsigned int reg,
	unsigned int value)
{
	u8 data[3];
	unsigned int *regvalue = NULL;

	data[0] = reg;
	data[1] = (value & 0xff00) >> 8;
	data[2] = value & 0x00ff;
	
	if ((reg == 0x80)
		|| (reg == 0x82)
		|| (reg == 0x84))
	{		
		regvalue = ((reg == 0x80) ? &reg80 : ((reg == 0x82) ? &reg82 : &reg84));
		*regvalue = value;
//		printk(KERN_INFO "rt5632_write ok, reg = %x, value = %x\n", reg, value);
		return 0;
	}
	rt5632_write_reg_cache(codec, reg, value);

	if (codec->hw_write(codec->control_data, data, 3) == 3)
	{
//		printk(KERN_INFO "rt5632_write ok, reg = %x, value = %x\n", reg, value);
		return 0;
	}
	else 
	{
		printk(KERN_ERR "rt5632_write fail \n");
		return -EIO;
	}
}

#define rt5632_write_mask(c, reg, value, mask) snd_soc_update_bits(c, reg, mask, value)

#define rt5632_reset(c) rt5632_write(c, RT5632_RESET, 0)

static int rt5632_reg_init(struct snd_soc_codec *codec)
{
	int i;

	for (i = 0; i < RT5632_INIT_REG_NUM; i++)
		rt5632_write(codec, rt5632_init_list[i].reg_index, rt5632_init_list[i].reg_value);

	return 0;
}


static const char *rt5632_fun_path_sel[] = {"disable","playbacking", "recording","special func 1","special func 2"};		/*0*/
static const char *rt5632_spk_out_sel[] = {"Class AB", "Class D"}; 					/*1*/
static const char *rt5632_spk_l_source_sel[] = {"LPRN", "LPRP", "LPLN", "MM"};		/*2*/	
static const char *rt5632_spkmux_source_sel[] = {"VMID", "HP Mixer", 
							"SPK Mixer", "Mono Mixer"};               					/*3*/
static const char *rt5632_hplmux_source_sel[] = {"VMID","HPL Mixer"};				/*4*/
static const char *rt5632_hprmux_source_sel[] = {"VMID","HPR Mixer"};				/*5*/
static const char *rt5632_auxmux_source_sel[] = {"VMID", "HP Mixer", 
							"SPK Mixer", "Mono Mixer"};							/*6*/
static const char *rt5632_spkamp_ratio_sel[] = {"2.25 Vdd", "2.00 Vdd",
					"1.75 Vdd", "1.50 Vdd", "1.25 Vdd", "1.00 Vdd"};				/*7*/
static const char *rt5632_mic1_boost_sel[] = {"Bypass", "+20db", "+30db", "+40db"};	/*8*/
static const char *rt5632_mic2_boost_sel[] = {"Bypass", "+20db", "+30db", "+40db"};	/*9*/
static const char *rt5632_dmic_boost_sel[] = {"Bypass", "+6db", "+12db", "+18db", 
					"+24db", "+30db"};						/*10*/

static const struct soc_enum rt5632_enum[] = {

SOC_ENUM_SINGLE(VIRTUAL_REG_FOR_MISC_FUNC, 0, 5, rt5632_fun_path_sel),		/*0*/
SOC_ENUM_SINGLE(RT5632_OUTPUT_MIXER_CTRL, 13, 2, rt5632_spk_out_sel),		/*1*/ // default : 2
SOC_ENUM_SINGLE(RT5632_OUTPUT_MIXER_CTRL, 14, 4, rt5632_spk_l_source_sel),	/*2*/ // default : 4
SOC_ENUM_SINGLE(RT5632_OUTPUT_MIXER_CTRL, 10, 4, rt5632_spkmux_source_sel),/*3*/
SOC_ENUM_SINGLE(RT5632_OUTPUT_MIXER_CTRL, 9, 2, rt5632_hplmux_source_sel),	/*4*/
SOC_ENUM_SINGLE(RT5632_OUTPUT_MIXER_CTRL, 8, 2, rt5632_hprmux_source_sel),/*5*/
SOC_ENUM_SINGLE(RT5632_OUTPUT_MIXER_CTRL, 6, 4, rt5632_auxmux_source_sel),/*6*/
#ifdef RT_VOL_RESCALE
//	SOC_ENUM_SINGLE(RT5632_GEN_CTRL_REG1, 1, 4, rt5632_spkamp_ratio_sel),		/*7*/
#else
	SOC_ENUM_SINGLE(RT5632_GEN_CTRL_REG1, 1, 6, rt5632_spkamp_ratio_sel),		/*7*/
#endif
SOC_ENUM_SINGLE(RT5632_MIC_CTRL, 10, 4,  rt5632_mic1_boost_sel),			/*8*/	// default : 4
SOC_ENUM_SINGLE(RT5632_MIC_CTRL, 8, 4, rt5632_mic2_boost_sel),				/*9*/	// default : 4
#ifdef RT_VOL_RESCALE
    SOC_ENUM_SINGLE(RT5632_DMIC_BOOST_CTRL, 0, 4, rt5632_dmic_boost_sel),		/*10*/ 	// default : 6
#else
    SOC_ENUM_SINGLE(RT5632_DMIC_BOOST_CTRL, 0, 6, rt5632_dmic_boost_sel),		/*10*/ 	// default : 6
#endif
};



//*****************************************************************************
//function:Enable the Voice PCM interface Path
//*****************************************************************************
static int ConfigPcmVoicePath(struct snd_soc_codec *codec,unsigned int bEnableVoicePath,unsigned int mode)
{

	if(bEnableVoicePath)
	 {
			//Power on DAC reference
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,PWR_DAC_REF|PWR_VOICE_DF2SE,PWR_DAC_REF|PWR_VOICE_DF2SE);
			//Power on Voice DAC/ADC 
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD2,PWR_VOICE_CLOCK,PWR_VOICE_CLOCK);
			//routing voice to HPMixer
			rt5632_write_mask(codec,RT5632_VOICE_DAC_OUT_VOL,0,M_V_DAC_TO_HP_MIXER);
					
		switch(mode)		
		{
			case PCM_SLAVE_MODE_B:	//8kHz sampling rate,16 bits PCM mode and Slave mode,PCM mode is B,MCLK=24.576MHz from Oscillator.
								//CSR PSKEY_PCM_CONFIG32 (HEX) = 0x08C00000,PSKEY_FORMAT=0x0060				

				//Enable GPIO 1,3,4,5 to voice interface
				//Set I2S to Slave mode
				//Voice I2S SYSCLK Source select Main SYSCLK
				//Set voice I2S VBCLK Polarity to Invert
				//Set Data length to 16 bit
				//set Data Fomrat to PCM mode B
				//the register 0x36 value's should is 0xC083
				rt5632_write(codec,RT5632_EXTEND_SDP_CTRL,0xC083);

				//Set LRCK voice select divide 32
				//set voice blck select divide 6 and 8 
				//voice filter clock divide 3 and 16
				//the register 0x64 value's should is 0x5524
				rt5632_write(codec,RT5632_VOICE_DAC_PCMCLK_CTRL1,0x5524);
		
				break;

			case PCM_SLAVE_MODE_A:	//8kHz sampling rate,16 bits PCM and Slave mode,PCM mode is A,MCLK=24.576MHz from Oscillator.
								//CSR PSKEY_PCM_CONFIG32 (HEX) = 0x08C00004,PSKEY_FORMAT=0x0060				

				//Enable GPIO 1,3,4,5 to voice interface
				//Set I2S to Slave mode
				//Voice I2S SYSCLK Source select Main SYSCLK
				//Set voice i2s VBCLK Polarity to Invert
				//Set Data length to 16 bit
				//set Data Fomrat to PCM mode A
				//the register 0x36 value's should is 0xC082
				rt5632_write(codec,RT5632_EXTEND_SDP_CTRL,0xC082);

				//Set LRCK voice select divide 64
				//set voice blck select divide 6 and 8 
				//voice filter clock divide 3 and 16
				//the register 0x64 value's should is 0x5524
				rt5632_write(codec,RT5632_VOICE_DAC_PCMCLK_CTRL1,0x5524);
		
				break;

			case PCM_MASTER_MODE_B:	//8kHz sampling rate,16 bits PCM and Master mode,PCM mode is B,Clock from PLL OUT
								//CSR PSKEY_PCM_CONFIG32 (HEX) = 0x08000002,PSKEY_FORMAT=0x0060	

				//Enable GPIO 1,3,4,5 to voice interface
				//Set I2S to master mode
				//Set voice i2s VBCLK Polarity to Invert
				//Set Data length to 16 bit
				//set Data Fomrat to PCM mode B
				//the register 0x36 value's should is 0x8083
				rt5632_write(codec,RT5632_EXTEND_SDP_CTRL,0x8083);

				//Set LRCK voice select divide 64
				//set voice blck select divide 6 and 8 
				//voice filter clock divide 3 and 16
				//the register 0x64 value's should is 0x5524
				rt5632_write(codec,RT5632_VOICE_DAC_PCMCLK_CTRL1,0x5524);
		
				break;

			default:
				//do nothing		
				break;

			}
		}
 		else
		{	
			//Power down Voice Different to sing-end power
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,0,PWR_VOICE_DF2SE);
			//Power down Voice DAC/ADC 
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD2,0,PWR_VOICE_CLOCK);
			//Disable Voice PCM interface	
			rt5632_write_mask(codec,RT5632_EXTEND_SDP_CTRL,0,EXT_I2S_FUNC_ENABLE);			
		}
	
	return 0;
}

//****************************************************************************************************************
//*function:playbacking function from dac to speaker,hp
//*PATH:DAC-->HPMixer-->Speaker,hp
//*
//****************************************************************************************************************
static int rt5632_playback_config(struct snd_soc_codec *codec,int Enable)
{

//	printk("rt5632_playback_config %d\n",Enable);

	if(Enable)
	{		
			rt5632_write(codec, 0x10, 0xe0e3);	//Enable DAC to HP mixer
		
			rt5632_write_mask(codec, 0x0c, 0x0000,0x8080);	//unmute DAC
			
			rt5632_write(codec, 0x1c, 0x0748);	//speaker,HP from HPmixer

			hp_depop_mode2(codec);				
	
			rt5632_ChangeCodecPowerStatus(codec,POWER_STATE_D1_PLAYBACK);	//power on Dac,HPMixer and output power

			rt5632_AudioOutEnable(codec,RT_WAVOUT_SPK,0);	//unmute speaker

			rt5632_AudioOutEnable(codec,RT_WAVOUT_HP,0);	//unmute hp			
	}
	else
	{
			rt5632_AudioOutEnable(codec,RT_WAVOUT_SPK,1);	//mute speaker

			rt5632_AudioOutEnable(codec,RT_WAVOUT_HP,1);	//mute hp		

			rt5632_write(codec, 0x10, 0xe0ef);	//Disable DAC to HP mixer			

			rt5632_ChangeCodecPowerStatus(codec,POWER_STATE_D2_PLAYBACK);	//power off Dac,HPMixer and output power
	}
		
	return 0;
}
//****************************************************************************************************************
//*function:Recording function from Mic to ADC
//*PATH:Mic1-->ADCMixer-->ADC
//*
//****************************************************************************************************************
static int rt5632_record_config(struct snd_soc_codec *codec,int Enable)
{

//printk("rt5632_record_config %d\n",Enable);

	if(Enable)
	{		
		rt5632_write(codec, 0x22, 0x0500);//Mic boost 20db	
		
        rt5632_ChangeCodecPowerStatus(codec,POWER_STATE_D1_RECORD);
        
		Enable_ADC_Input_Source(codec,RT_WAVIN_R_MIC1 | RT_WAVIN_L_MIC1,1);//enable Mic1 to ADC
	}
	else
	{
		Enable_ADC_Input_Source(codec,RT_WAVIN_R_MIC1 | RT_WAVIN_L_MIC1,0);//disable Mic1 to ADC

		rt5632_ChangeCodecPowerStatus(codec,POWER_STATE_D2_RECORD);

	}

	return 0;
}
//****************************************************************************************************************
//*function:rt5632_special_func_1
//*	enable LineIn to speaker & HP
//*
//****************************************************************************************************************
static int rt5632_special_func_1(struct snd_soc_codec *codec,int Enable)
{
//	printk("rt5632_special_func_1 =%d\n",Enable);	

	if(Enable)
	{					
		//enable line In volume power 
		rt5632_write_mask(codec, 0x3e, 0x00c0,0x00c0);	
		//enable line In to HPmixer
		rt5632_write_mask(codec, 0x0a, 0x0000,0x8000);
		//enable HPmixer power and output 
		rt5632_ChangeCodecPowerStatus(codec,POWER_STATE_D1_PLAYBACK);
		//unmute speaker
		rt5632_AudioOutEnable(codec,RT_WAVOUT_SPK,0);	
		//unmute hp		
		rt5632_AudioOutEnable(codec,RT_WAVOUT_HP,0);			
	}
	else
	{
		//mute speaker
		rt5632_AudioOutEnable(codec,RT_WAVOUT_SPK,1);	
		//mute hp		
		rt5632_AudioOutEnable(codec,RT_WAVOUT_HP,1);		
		//mute lineIn to hpmixer	
		rt5632_write_mask(codec, 0x0a, 0x8000,0x8000);		
		//disable HPmixer power and output 
		rt5632_ChangeCodecPowerStatus(codec,POWER_STATE_D2_PLAYBACK);
		//enable line In power 
		rt5632_write_mask(codec, 0x3e, 0x0000,0x00c0);	
	}

	return 0;
}

//****************************************************************************************************************
//*
//*function:rt5632_special_func_2
//*
//****************************************************************************************************************
static int rt5632_special_func_2(struct snd_soc_codec *codec,int Enable)
{	
//	printk("rt5632_special_func_2 =%d\n",Enable);

	if(Enable)
	{
		//TDB
	}
	else
	{
		//TDB
	}
	
	return 0;
}

//****************************************************************************************************************
//*
//*function:disable rt5632's function.
//*
//*
//****************************************************************************************************************
static int rt5632_func_sel_disable(struct snd_soc_codec *codec,int mode)
{

	switch(mode)
	{		
		case RT5632_PLAYBACKING:

			rt5632_playback_config(codec,0);	//disable playback path and power		

		break;

		case RT5632_RECORDING:
	
			rt5632_record_config(codec,0);		//disable record path and power
		
		break;

		case RT5632_SPECIAL_FUNC1:

			rt5632_special_func_1(codec,0);		//disable customer special path and power

		break;

		case RT5632_SPECIAL_FUNC2:			

			rt5632_special_func_2(codec,0);		//disable customer special path and power

		break;	

		default:

		break;
	}

	return 0;
}


static int rt5632_get_func_mode(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	 /*cause we choose bit[0][1][2] to store the mode type*/
	int mode = (rt5632_read(codec, VIRTUAL_REG_FOR_MISC_FUNC)) & 0x07;  

	ucontrol->value.integer.value[0] = mode;
	return 0;
}


static int rt5632_set_func_mode(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	u16 Virtual_reg = rt5632_read(codec, VIRTUAL_REG_FOR_MISC_FUNC);
	int rt5632_mode=(Virtual_reg)&0x07;

//	printk("1rt5632_mode=%d,value[0]=%ld,Virtual_reg=%x\n",rt5632_mode,ucontrol->value.integer.value[0],Virtual_reg);

	if ( rt5632_mode == ucontrol->value.integer.value[0])
		return 0;

	switch(ucontrol->value.integer.value[0])
	{
		case RT5632_PLAYBACKING:

			rt5632_playback_config(codec,1);	//enable playback	

		break;

		case RT5632_RECORDING:
	
			rt5632_record_config(codec,1);		//enable record function
		
		break;

		case RT5632_SPECIAL_FUNC1:

			rt5632_special_func_1(codec,1);			//enable customer special function

		break;

		case RT5632_SPECIAL_FUNC2:

			rt5632_special_func_2(codec,1);			//enable customer special function

		break;

		case RT5632_FUNC_DISABLE:
	
			rt5632_func_sel_disable(codec,rt5632_mode);		//disable previous select function.	
	
		break;	

		default:

		break;
	}

	Virtual_reg &= 0xfff8;
	Virtual_reg |= (ucontrol->value.integer.value[0]);
	rt5632_write(codec, VIRTUAL_REG_FOR_MISC_FUNC, Virtual_reg);

//	printk("2rt5632_mode=%d,value[0]=%ld,Virtual_reg=%x\n",rt5632_mode,ucontrol->value.integer.value[0],Virtual_reg);
	return 0;
}


#ifdef RT_VOL_RESCALE

#define PCM_MAXIMUM_VOL_GAIN		0x3f
#define PCM_MIXER_RANGE 	    	0x2f 
#define SPEAKER_MAXIMUM_VOL_GAIN	0x1f
#define SPEAKER_MIXER_RANGE			0x1e 

static int rt5632_pcm_vol_get(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	int reg_val = rt5632_read(codec, RT5632_STEREO_DAC_VOL) & 0x3f;

	ucontrol->value.integer.value[0] = PCM_MAXIMUM_VOL_GAIN - reg_val; 

	printk("%s():reg=%d display=%d\n", __func__, reg_val, ucontrol->value.integer.value[0]);

	return 0;
}

static int rt5632_pcm_vol_put(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	int reg_val, disp_vol;

	disp_vol = ucontrol->value.integer.value[0];
	reg_val = PCM_MAXIMUM_VOL_GAIN - disp_vol;

	rt5632_write_mask(codec, RT5632_STEREO_DAC_VOL, (reg_val << 8) | reg_val, 0x3f3f);
	printk("%s():reg=%d display=%d\n", __func__, reg_val, ucontrol->value.integer.value[0]);

	return 0;
}

static int rt5632_speaker_vol_get(struct snd_kcontrol * kcontrol, struct snd_ctl_elem_value * ucontrol )
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	int reg_val = rt5632_read(codec, RT5632_SPK_OUT_VOL) & 0x1f;

	ucontrol->value.integer.value[0] = SPEAKER_MAXIMUM_VOL_GAIN - reg_val;

	printk("%s():reg=%d display=%d\n", __func__, reg_val, ucontrol->value.integer.value[0]);

	return 0;
}

static int rt5632_speaker_vol_put(struct snd_kcontrol * kcontrol, struct snd_ctl_elem_value * ucontrol )
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	int reg_val, disp_vol;

	disp_vol = ucontrol->value.integer.value[0];
	reg_val = SPEAKER_MAXIMUM_VOL_GAIN - disp_vol;

	rt5632_write_mask(codec, RT5632_SPK_OUT_VOL, (reg_val << 8) | reg_val, 0x1f1f);
	printk("%s():reg=%d display=%d\n", __func__, reg_val, ucontrol->value.integer.value[0]);

	return 0;
}

#endif//RT_VOL_RESCALE

static const struct snd_kcontrol_new rt5632_snd_controls[] = {
SOC_ENUM_EXT("rt5632 func sel", rt5632_enum[0], rt5632_get_func_mode, rt5632_set_func_mode),
SOC_ENUM("SPK Amp Type", rt5632_enum[1]),
SOC_ENUM("Left SPK Source", rt5632_enum[2]),

#ifdef RT_VOL_RESCALE
	SOC_ENUM("Mic1 Boost", rt5632_enum[7]),
	SOC_ENUM("Mic2 Boost", rt5632_enum[8]),
	SOC_ENUM("Dmic Boost", rt5632_enum[9]),
#else
	SOC_ENUM("SPK Amp Ratio", rt5632_enum[7]),
	SOC_ENUM("Mic1 Boost", rt5632_enum[8]),
	SOC_ENUM("Mic2 Boost", rt5632_enum[9]),
	SOC_ENUM("Dmic Boost", rt5632_enum[10]),
#endif

#ifdef RT_VOL_RESCALE
	SOC_SINGLE_EXT("PCM Playback Volume", RT5632_STEREO_DAC_VOL, 0, PCM_MIXER_RANGE, 0, rt5632_pcm_vol_get, rt5632_pcm_vol_put),
#else
	SOC_DOUBLE("PCM Playback Volume", RT5632_STEREO_DAC_VOL, 8, 0, 63, 1),
#endif//RT_VOL_RESCALE
SOC_DOUBLE("LineIn Playback Volume", RT5632_LINE_IN_VOL, 8, 0, 0, 1),	// default : 31
SOC_SINGLE("Phone Playback Volume", RT5632_PHONEIN_VOL, 8, 31, 1),
SOC_SINGLE("Mic1 Playback Volume", RT5632_MIC_VOL, 8, 31, 1),		// default : 31
SOC_SINGLE("Mic2 Playback Volume", RT5632_MIC_VOL, 0, 31, 1),		// default : 31
SOC_DOUBLE("PCM Capture Volume", RT5632_ADC_REC_GAIN, 8, 0, 31, 1),
#ifdef RT_VOL_RESCALE
	SOC_SINGLE_EXT("SPKOUT Playback Volume", RT5632_SPK_OUT_VOL, 0, SPEAKER_MIXER_RANGE, 0, rt5632_speaker_vol_get, rt5632_speaker_vol_put ),
#else
	SOC_DOUBLE("SPKOUT Playback Volume", RT5632_SPK_OUT_VOL, 8, 0, 31, 1),
#endif
SOC_DOUBLE("SPKOUT Playback Switch", RT5632_SPK_OUT_VOL, 15, 7, 1, 1),
SOC_DOUBLE("HPOUT Playback Volume", RT5632_HP_OUT_VOL, 8, 0, 31, 1),
SOC_DOUBLE("HPOUT Playback Switch", RT5632_HP_OUT_VOL, 15, 7, 1, 1),
SOC_DOUBLE("AUXOUT Playback Volume", RT5632_AUX_OUT_VOL, 8, 0, 31, 1),
SOC_DOUBLE("AUXOUT Playback Switch", RT5632_AUX_OUT_VOL, 15, 7, 1, 1),
#ifdef RT_VOL_RESCALE
	// disable the ADC_REC_GAIN
#else
	SOC_DOUBLE("ADC Record Gain", RT5632_ADC_REC_GAIN, 8, 0, 31, 0),	// default : 31
#endif
};

static int rt5632_add_controls(struct snd_soc_codec *codec)
{
	int err, i;

	for (i = 0; i < ARRAY_SIZE(rt5632_snd_controls); i++){
		err = snd_ctl_add(codec->card, 
				snd_soc_cnew(&rt5632_snd_controls[i],
						codec, NULL));
		if (err < 0)
			return err;
	}
	return 0;
}

static void hp_depop_mode2(struct snd_soc_codec *codec)
{
        rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD1, PWR_MAIN_BIAS, PWR_MAIN_BIAS);
        rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD2, PWR_MIXER_VREF, PWR_MIXER_VREF);
        rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD1, PWR_SOFTGEN_EN, PWR_SOFTGEN_EN);
        rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD3, PWR_HP_R_OUT_VOL|PWR_HP_L_OUT_VOL,
                PWR_HP_R_OUT_VOL|PWR_HP_L_OUT_VOL);
        rt5632_write_mask(codec, RT5632_MISC_CTRL, HP_DEPOP_MODE2_EN, HP_DEPOP_MODE2_EN);
        schedule_timeout_uninterruptible(msecs_to_jiffies(300));

        rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD1, PWR_HP_OUT_AMP|PWR_HP_OUT_ENH_AMP,
                PWR_HP_OUT_AMP|PWR_HP_OUT_ENH_AMP);
        rt5632_write_mask(codec, RT5632_MISC_CTRL, 0, HP_DEPOP_MODE2_EN);

}


//*****************************************************************************
//
//function:Change audio codec power status
//
//*****************************************************************************
static int rt5632_ChangeCodecPowerStatus(struct snd_soc_codec *codec,int power_state)
{
	unsigned short int PowerDownState=0;

	switch(power_state)
	{
		case POWER_STATE_D0:			//FULL ON-----power on all power
			
			rt5632_write(codec,RT5632_PWR_MANAG_ADD1,~PowerDownState);
			rt5632_write(codec,RT5632_PWR_MANAG_ADD2,~PowerDownState);
			rt5632_write(codec,RT5632_PWR_MANAG_ADD3,~PowerDownState);

		break;	

		case POWER_STATE_D1:		//LOW ON-----


			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,PWR_DAC_DF2SE_R|PWR_DAC_DF2SE_L|PWR_DAC_REF | PWR_MAIN_BIAS | PWR_MIC_BIAS1 | PWR_I2S_INTERFACE | PWR_HP_OUT_AMP |PWR_HP_OUT_ENH_AMP | PWR_AMP_POWER		
											   		     ,PWR_DAC_DF2SE_R|PWR_DAC_DF2SE_L|PWR_DAC_REF | PWR_MAIN_BIAS | PWR_MIC_BIAS1 | PWR_I2S_INTERFACE | PWR_HP_OUT_AMP |PWR_HP_OUT_ENH_AMP | PWR_AMP_POWER);
										   
											
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD2,PWR_R_ADC_REC_MIXER | PWR_L_ADC_REC_MIXER | PWR_R_HP_MIXER | PWR_L_HP_MIXER | PWR_R_ADC_CLK | PWR_L_ADC_CLK | 
													      PWR_R_DAC_CLK | PWR_L_DAC_CLK | PWR_MIXER_VREF
											   		 	 ,PWR_R_ADC_REC_MIXER | PWR_L_ADC_REC_MIXER | PWR_R_HP_MIXER | PWR_L_HP_MIXER | PWR_R_ADC_CLK | PWR_L_ADC_CLK | 
													  	  PWR_R_DAC_CLK | PWR_L_DAC_CLK | PWR_MIXER_VREF);

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD3,PWR_MIC1_BOOST | PWR_HP_R_OUT_VOL | PWR_HP_L_OUT_VOL | PWR_SPK_OUT | PWR_SPK_OUT_N
											   		 	 ,PWR_MIC1_BOOST | PWR_HP_R_OUT_VOL | PWR_HP_L_OUT_VOL | PWR_SPK_OUT | PWR_SPK_OUT_N);	
		break;

		case POWER_STATE_D1_PLAYBACK:	//Low on of Playback


			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,PWR_DAC_DF2SE_R|PWR_DAC_DF2SE_L|PWR_DAC_REF | PWR_MAIN_BIAS | PWR_I2S_INTERFACE | PWR_HP_OUT_AMP |PWR_HP_OUT_ENH_AMP | PWR_AMP_POWER | PWR_MAIN_BIAS
													 	 ,PWR_DAC_DF2SE_R|PWR_DAC_DF2SE_L|PWR_DAC_REF | PWR_MAIN_BIAS | PWR_I2S_INTERFACE | PWR_HP_OUT_AMP |PWR_HP_OUT_ENH_AMP | PWR_AMP_POWER | PWR_MAIN_BIAS);
											
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD2,PWR_R_HP_MIXER | PWR_L_HP_MIXER | PWR_R_DAC_CLK | PWR_L_DAC_CLK | PWR_MIXER_VREF 
											   		 	 ,PWR_R_HP_MIXER | PWR_L_HP_MIXER | PWR_R_DAC_CLK | PWR_L_DAC_CLK | PWR_MIXER_VREF);

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD3,PWR_HP_R_OUT_VOL | PWR_HP_L_OUT_VOL | PWR_SPK_OUT | PWR_SPK_OUT_N
											   		 	 ,PWR_HP_R_OUT_VOL | PWR_HP_L_OUT_VOL | PWR_SPK_OUT | PWR_SPK_OUT_N);	
									

		break;

		case POWER_STATE_D1_RECORD:	//Low on of Record

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,PWR_MIC_BIAS1 | PWR_MAIN_BIAS | PWR_I2S_INTERFACE
														 ,PWR_MIC_BIAS1 | PWR_MAIN_BIAS | PWR_I2S_INTERFACE);
											
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD2,PWR_R_ADC_REC_MIXER | PWR_L_ADC_REC_MIXER | PWR_R_ADC_CLK | PWR_L_ADC_CLK | PWR_MIXER_VREF
														 ,PWR_R_ADC_REC_MIXER | PWR_L_ADC_REC_MIXER | PWR_R_ADC_CLK | PWR_L_ADC_CLK | PWR_MIXER_VREF);

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD3,PWR_MIC1_BOOST,PWR_MIC1_BOOST);
	
		break;

		case POWER_STATE_D2:		//STANDBY----

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD3,0,PWR_MIC1_BOOST |PWR_HP_R_OUT_VOL | PWR_HP_L_OUT_VOL | PWR_SPK_OUT | PWR_SPK_OUT_N);	
											//																								
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,0,PWR_DAC_DF2SE_R|PWR_DAC_DF2SE_L|PWR_DAC_REF | PWR_MIC_BIAS1 | PWR_HP_OUT_AMP |PWR_HP_OUT_ENH_AMP | PWR_I2S_INTERFACE | PWR_AMP_POWER);
											//
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD2,0,PWR_R_ADC_REC_MIXER | PWR_L_ADC_REC_MIXER | PWR_R_HP_MIXER | PWR_L_HP_MIXER | PWR_R_ADC_CLK | PWR_L_ADC_CLK | 
															PWR_R_DAC_CLK | PWR_L_DAC_CLK);
		
		break;

		case POWER_STATE_D2_PLAYBACK:	//STANDBY of playback

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD3,0,PWR_HP_R_OUT_VOL | PWR_HP_L_OUT_VOL |PWR_SPK_OUT | PWR_SPK_OUT_N);

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,0,PWR_DAC_DF2SE_R|PWR_DAC_DF2SE_L|PWR_DAC_REF | PWR_HP_OUT_AMP |PWR_HP_OUT_ENH_AMP|PWR_AMP_POWER);
											//
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD2,0,PWR_R_HP_MIXER | PWR_L_HP_MIXER | PWR_R_DAC_CLK | PWR_L_DAC_CLK);


		break;

		case POWER_STATE_D2_RECORD:		//STANDBY of record

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD3,0,PWR_MIC1_BOOST);	

			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,0,PWR_MIC_BIAS1);
											//
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD2,0,PWR_R_ADC_REC_MIXER | PWR_L_ADC_REC_MIXER | PWR_R_ADC_CLK | PWR_L_ADC_CLK);
	

		break;		

		case POWER_STATE_D3:		//SLEEP
		case POWER_STATE_D4:		//OFF----power off all power
			rt5632_write_mask(codec,RT5632_PWR_MANAG_ADD1,0,PWR_HP_OUT_AMP |PWR_HP_OUT_ENH_AMP);		
			
			rt5632_write(codec,RT5632_PWR_MANAG_ADD3,PowerDownState);
			rt5632_write(codec,RT5632_PWR_MANAG_ADD1,PowerDownState);
			rt5632_write(codec,RT5632_PWR_MANAG_ADD2,PowerDownState);

			PowerDownState=RT_PWR_PR0 | RT_PWR_PR1 | RT_PWR_PR2 | RT_PWR_PR3 | RT_PWR_PR5 | RT_PWR_PR6 | RT_PWR_PR7; 		
			rt5632_write(codec,RT5632_PD_CTRL_STAT,PowerDownState);				
						
		break;	

		default:

		break;
	}

	return 0;
}

//*****************************************************************************
//
//function AudioOutEnable:Mute/Unmute audio out channel
//							WavOutPath:output channel
//							Mute :Mute/Unmute output channel											
//
//*****************************************************************************
static int rt5632_AudioOutEnable(struct snd_soc_codec *codec,unsigned short int WavOutPath,int Mute)
{
	int RetVal=0;	

	if(Mute)
	{
		switch(WavOutPath)
		{
			case RT_WAVOUT_ALL_ON:

				RetVal=rt5632_write_mask(codec,RT5632_SPK_OUT_VOL,RT_L_MUTE|RT_R_MUTE,RT_L_MUTE|RT_R_MUTE);	//Mute Speaker right/left channel
				RetVal=rt5632_write_mask(codec,RT5632_HP_OUT_VOL,RT_L_MUTE|RT_R_MUTE,RT_L_MUTE|RT_R_MUTE);	//Mute headphone right/left channel
				RetVal=rt5632_write_mask(codec,RT5632_AUX_OUT_VOL,RT_L_MUTE|RT_R_MUTE,RT_L_MUTE|RT_R_MUTE);	//Mute auxout channel
				RetVal=rt5632_write_mask(codec,RT5632_STEREO_DAC_VOL,RT_L_MUTE|RT_R_MUTE,RT_L_MUTE|RT_R_MUTE);	//Mute DAC right/left channel
		
			break;
		
			case RT_WAVOUT_HP:
				//mute/unmute depop enable
				rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD1, PWR_SOFTGEN_EN, PWR_SOFTGEN_EN);
				rt5632_write_mask(codec, RT5632_MISC_CTRL, M_UM_DEPOP_EN|HP_R_M_UM_DEPOP_EN|HP_L_M_UM_DEPOP_EN,M_UM_DEPOP_EN|HP_R_M_UM_DEPOP_EN|HP_L_M_UM_DEPOP_EN);
				RetVal=rt5632_write_mask(codec,RT5632_HP_OUT_VOL,RT_L_MUTE|RT_R_MUTE,RT_L_MUTE|RT_R_MUTE);	//Mute headphone right/left channel
				mdelay(30);
				//mute/unmute depop disable
				rt5632_write_mask(codec, RT5632_MISC_CTRL,0,M_UM_DEPOP_EN|HP_R_M_UM_DEPOP_EN|HP_L_M_UM_DEPOP_EN);
			
			break;

			case RT_WAVOUT_SPK:
				
				RetVal=rt5632_write_mask(codec,RT5632_SPK_OUT_VOL,RT_L_MUTE|RT_R_MUTE,RT_L_MUTE|RT_R_MUTE);	//Mute Speaker right/left channel		

			break;
			
			case RT_WAVOUT_AUXOUT:

				RetVal=rt5632_write_mask(codec,RT5632_AUX_OUT_VOL,RT_L_MUTE|RT_R_MUTE,RT_L_MUTE|RT_R_MUTE);	//Mute auxout channel	

			break;

			case RT_WAVOUT_DAC:

				RetVal=rt5632_write_mask(codec,RT5632_STEREO_DAC_VOL,RT_L_MUTE|RT_R_MUTE,RT_L_MUTE|RT_R_MUTE);	//Mute DAC 

			break;
			default:

				return 0;

		}
	}
	else
	{
		switch(WavOutPath)
		{

			case RT_WAVOUT_ALL_ON:

				RetVal=rt5632_write_mask(codec,RT5632_SPK_OUT_VOL	,0,RT_L_MUTE|RT_R_MUTE);	//Mute Speaker right/left channel
				RetVal=rt5632_write_mask(codec,RT5632_HP_OUT_VOL 	,0,RT_L_MUTE|RT_R_MUTE);	//Mute headphone right/left channel
				RetVal=rt5632_write_mask(codec,RT5632_AUX_OUT_VOL	,0,RT_L_MUTE|RT_R_MUTE);	//Mute auxout channel	
				RetVal=rt5632_write_mask(codec,RT5632_STEREO_DAC_VOL,0,RT_L_MUTE|RT_R_MUTE);	//Mute DAC
		
			break;
		
			case RT_WAVOUT_HP:
				//mute/unmute depop enable
				rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD1, PWR_SOFTGEN_EN, PWR_SOFTGEN_EN);
				rt5632_write_mask(codec, RT5632_MISC_CTRL, M_UM_DEPOP_EN|HP_R_M_UM_DEPOP_EN|HP_L_M_UM_DEPOP_EN,M_UM_DEPOP_EN|HP_R_M_UM_DEPOP_EN|HP_L_M_UM_DEPOP_EN);

				RetVal=rt5632_write_mask(codec,RT5632_HP_OUT_VOL,0,RT_L_MUTE|RT_R_MUTE);	//unMute headphone right/left channel
				mdelay(30);
				//mute/unmute depop disable
				rt5632_write_mask(codec, RT5632_MISC_CTRL,0,M_UM_DEPOP_EN|HP_R_M_UM_DEPOP_EN|HP_L_M_UM_DEPOP_EN);
					
			break;

			case RT_WAVOUT_SPK:
				
				RetVal=rt5632_write_mask(codec,RT5632_SPK_OUT_VOL,0,RT_L_MUTE|RT_R_MUTE);	//unMute Speaker right/left channel			

			break;
			
			case RT_WAVOUT_AUXOUT:

				RetVal=rt5632_write_mask(codec,RT5632_AUX_OUT_VOL,0,RT_L_MUTE|RT_R_MUTE);	//unMute auxout channel	

			break;
			case RT_WAVOUT_DAC:

				RetVal=rt5632_write_mask(codec,RT5632_STEREO_DAC_VOL,0,RT_L_MUTE|RT_R_MUTE);	//unMute DAC

			break;
			
			default:
				return 0;
		}
	}
	
	return RetVal;
}

//*****************************************************************************
//
//function:Enable/Disable ADC input source control
//
//*****************************************************************************
static int Enable_ADC_Input_Source(struct snd_soc_codec *codec,unsigned short int ADC_Input_Sour,int Enable)
{
	int bRetVal=0;
	
	if(Enable)
	{
		//Enable ADC source 
		bRetVal=rt5632_write_mask(codec,RT5632_ADC_REC_MIXER,0,ADC_Input_Sour);
	}
	else
	{
		//Disable ADC source		
		bRetVal=rt5632_write_mask(codec,RT5632_ADC_REC_MIXER,ADC_Input_Sour,ADC_Input_Sour);
	}

	return bRetVal;
}


#if USE_DAPM_CONTROL
/*
 * _DAPM_ Controls
 */
 /*Left ADC Rec mixer*/
static const struct snd_kcontrol_new rt5632_left_adc_rec_mixer_controls[] = {
SOC_DAPM_SINGLE("Mic1 Capture Switch", RT5632_ADC_REC_MIXER, 14, 1, 1),
SOC_DAPM_SINGLE("Mic2 Capture Switch", RT5632_ADC_REC_MIXER, 13, 1, 1),
SOC_DAPM_SINGLE("LineIn Capture Switch", RT5632_ADC_REC_MIXER, 12, 1, 1),
SOC_DAPM_SINGLE("Phone Capture Switch", RT5632_ADC_REC_MIXER, 11, 1, 1),
SOC_DAPM_SINGLE("HP Mixer Capture Switch", RT5632_ADC_REC_MIXER, 10, 1, 1),
SOC_DAPM_SINGLE("MoNo Mixer Capture Switch", RT5632_ADC_REC_MIXER, 8, 1, 1),
SOC_DAPM_SINGLE("SPK Mixer Capture Switch", RT5632_ADC_REC_MIXER, 9, 1, 1),

};

/*Left ADC Rec mixer*/
static const struct snd_kcontrol_new rt5632_right_adc_rec_mixer_controls[] = {
SOC_DAPM_SINGLE("Mic1 Capture Switch", RT5632_ADC_REC_MIXER, 6, 1, 1),
SOC_DAPM_SINGLE("Mic2 Capture Switch", RT5632_ADC_REC_MIXER, 5, 1, 1),
SOC_DAPM_SINGLE("LineIn Capture Switch", RT5632_ADC_REC_MIXER, 4, 1, 1),
SOC_DAPM_SINGLE("Phone Capture Switch", RT5632_ADC_REC_MIXER, 3, 1, 1),
SOC_DAPM_SINGLE("HP Mixer Capture Switch", RT5632_ADC_REC_MIXER, 2, 1, 1),
SOC_DAPM_SINGLE("MoNo Mixer Capture Switch", RT5632_ADC_REC_MIXER, 0, 1, 1),
SOC_DAPM_SINGLE("SPK Mixer Capture Switch", RT5632_ADC_REC_MIXER, 1, 1, 1),
};

static const struct snd_kcontrol_new rt5632_left_hp_mixer_controls[] = {
SOC_DAPM_SINGLE("ADC Playback Switch", RT5632_ADC_REC_GAIN, 15, 1, 1),
SOC_DAPM_SINGLE("LineIn Playback Switch", HPL_MIXER, 0, 1, 0),
SOC_DAPM_SINGLE("Phone Playback Switch", HPL_MIXER, 1, 1, 0),
SOC_DAPM_SINGLE("Mic1 Playback Switch", HPL_MIXER, 2, 1, 0),
SOC_DAPM_SINGLE("Mic2 Playback Switch", HPL_MIXER, 3, 1, 0),
SOC_DAPM_SINGLE("HIFI DAC Playback Switch", RT5632_DAC_AND_MIC_CTRL, 3, 1, 1),
SOC_DAPM_SINGLE("Voice DAC Playback Switch", HPL_MIXER, 4, 1, 0),
};

static const struct snd_kcontrol_new rt5632_right_hp_mixer_controls[] = {
SOC_DAPM_SINGLE("ADC Playback Switch", RT5632_ADC_REC_GAIN, 7, 1, 1),
SOC_DAPM_SINGLE("LineIn Playback Switch", HPR_MIXER, 0, 1, 0),
SOC_DAPM_SINGLE("Phone Playback Switch", HPR_MIXER, 1, 1, 0),
SOC_DAPM_SINGLE("Mic1 Playback Switch", HPR_MIXER, 2, 1, 0),
SOC_DAPM_SINGLE("Mic2 Playback Switch", HPR_MIXER, 3, 1, 0),
SOC_DAPM_SINGLE("HIFI DAC Playback Switch", RT5632_DAC_AND_MIC_CTRL, 2, 1, 1),
SOC_DAPM_SINGLE("Voice DAC Playback Switch", HPR_MIXER, 4, 1, 0),
};

static const struct snd_kcontrol_new rt5632_mono_mixer_controls[] = {
SOC_DAPM_SINGLE("ADCL Playback Switch", RT5632_ADC_REC_GAIN, 14, 1, 1),
SOC_DAPM_SINGLE("ADCR Playback Switch", RT5632_ADC_REC_GAIN, 6, 1, 1),
SOC_DAPM_SINGLE("Line Mixer Playback Switch", RT5632_LINE_IN_VOL, 13, 1, 1),
SOC_DAPM_SINGLE("Mic1 Playback Switch", RT5632_DAC_AND_MIC_CTRL, 13, 1, 1),
SOC_DAPM_SINGLE("Mic2 Playback Switch", RT5632_DAC_AND_MIC_CTRL, 9, 1, 1),
SOC_DAPM_SINGLE("DAC Mixer Playback Switch", RT5632_DAC_AND_MIC_CTRL, 0, 1, 1),
SOC_DAPM_SINGLE("Voice DAC Playback Switch", RT5632_VOICE_DAC_OUT_VOL, 13, 1, 1),
};

static const struct snd_kcontrol_new rt5632_spk_mixer_controls[] = {
SOC_DAPM_SINGLE("Line Mixer Playback Switch", RT5632_LINE_IN_VOL, 14, 1, 1),	
SOC_DAPM_SINGLE("Phone Playback Switch", RT5632_PHONEIN_VOL, 14, 1, 1),
SOC_DAPM_SINGLE("Mic1 Playback Switch", RT5632_DAC_AND_MIC_CTRL, 14, 1, 1),
SOC_DAPM_SINGLE("Mic2 Playback Switch", RT5632_DAC_AND_MIC_CTRL, 10, 1, 1),
SOC_DAPM_SINGLE("DAC Mixer Playback Switch", RT5632_DAC_AND_MIC_CTRL, 1, 1, 1),
SOC_DAPM_SINGLE("Voice DAC Playback Switch", RT5632_VOICE_DAC_OUT_VOL, 14, 1, 1),
};

static int mixer_event(struct snd_soc_dapm_widget *w, struct snd_kcontrol *k, int event)
{
	struct snd_soc_codec *codec = w->codec;
	unsigned int l, r;

//	printk(KERN_INFO "enter %s\n", __func__);

	l= rt5632_read(codec, HPL_MIXER);
	r = rt5632_read(codec, HPR_MIXER);
	
	if ((l & 0x1) || (r & 0x1))
		rt5632_write_mask(codec, 0x0a, 0x0000, 0x8000);
	else
		rt5632_write_mask(codec, 0x0a, 0x8000, 0x8000);

	if ((l & 0x2) || (r & 0x2))
		rt5632_write_mask(codec, 0x08, 0x0000, 0x8000);
	else
		rt5632_write_mask(codec, 0x08, 0x8000, 0x8000);

	if ((l & 0x4) || (r & 0x4))
		rt5632_write_mask(codec, 0x10, 0x0000, 0x8000);
	else
		rt5632_write_mask(codec, 0x10, 0x8000, 0x8000);

	if ((l & 0x8) || (r & 0x8))
		rt5632_write_mask(codec, 0x10, 0x0000, 0x0800);
	else
		rt5632_write_mask(codec, 0x10, 0x0800, 0x0800);

	if ((l & 0x10) || (r & 0x10))
		rt5632_write_mask(codec, 0x18, 0x0000, 0x8000);
	else
		rt5632_write_mask(codec, 0x18, 0x8000, 0x8000);

	return 0;
}


/*
 *	bit[0][1] use for aec control
 *	bit[2][3] for ADCR func
 *	bit[4] for SPKL pga
 *	bit[5] for SPKR pga
 *	bit[6] for hpl pga
 *	bit[7] for hpr pga
 */
static int spk_pga_event(struct snd_soc_dapm_widget *w, struct snd_kcontrol *k, int event)
 {
	struct snd_soc_codec *codec = w->codec;
	int reg;
	
//	printk(KERN_DEBUG "enter %s\n", __func__);
	reg = rt5632_read(codec, VIRTUAL_REG_FOR_MISC_FUNC) & (0x3 << 4);
	if ((reg >> 4) != 0x3 && reg != 0)
		return 0;

	switch (event)
	{
		case SND_SOC_DAPM_POST_PMU:
//			printk(KERN_INFO "after virtual spk power up!\n");
			rt5632_write_mask(codec, 0x3e, 0x3000, 0x3000);
			rt5632_write_mask(codec, 0x02, 0x0000, 0x8080);
			rt5632_write_mask(codec, 0x3a, 0x0400, 0x0400);                  //power on spk amp
			break;
		case SND_SOC_DAPM_POST_PMD:
//			printk(KERN_INFO "aftet virtual spk power down!\n");
			rt5632_write_mask(codec, 0x3a, 0x0000, 0x0400);//power off spk amp
			rt5632_write_mask(codec, 0x02, 0x8080, 0x8080);
			rt5632_write_mask(codec, 0x3e, 0x0000, 0x3000);                 
			break;
		default:
			return 0;
	}
	return 0;
}




static int hp_pga_event(struct snd_soc_dapm_widget *w, struct snd_kcontrol *k, int event)
{
	struct snd_soc_codec *codec = w->codec;
	int reg;

//	printk(KERN_DEBUG "enter %s\n", __func__);
	reg = rt5632_read(codec, VIRTUAL_REG_FOR_MISC_FUNC) & (0x3 << 6);
	if ((reg >> 6) != 0x3 && reg != 0)
		return 0;
	
	switch (event)
	{
		case SND_SOC_DAPM_POST_PMD:
//			printk(KERN_INFO "aftet virtual hp power down!\n");
			rt5632_write_mask(codec, 0x04, 0x8080, 0x8080);
			rt5632_write_mask(codec, 0x3e, 0x0000, 0x0600);
			rt5632_write_mask(codec, 0x3a, 0x0000, 0x0030);
			break;
		case SND_SOC_DAPM_POST_PMU:	
//			printk(KERN_INFO "after virtual hp power up!\n");
			hp_depop_mode2(codec);
			rt5632_write_mask(codec ,0x04, 0x0000, 0x8080);
			break;
		default:
			return 0;
	}	
	return 0;
}



static int aux_pga_event(struct snd_soc_dapm_widget *w, struct snd_kcontrol *k, int event)
{
	return 0;
}

/*SPKOUT Mux*/
static const struct snd_kcontrol_new rt5632_spkout_mux_out_controls = 
SOC_DAPM_ENUM("Route", rt5632_enum[3]);

/*HPLOUT MUX*/
static const struct snd_kcontrol_new rt5632_hplout_mux_out_controls = 
SOC_DAPM_ENUM("Route", rt5632_enum[4]);

/*HPROUT MUX*/
static const struct snd_kcontrol_new rt5632_hprout_mux_out_controls = 
SOC_DAPM_ENUM("Route", rt5632_enum[5]);
/*AUXOUT MUX*/
static const struct snd_kcontrol_new rt5632_auxout_mux_out_controls = 
SOC_DAPM_ENUM("Route", rt5632_enum[6]);

static const struct snd_soc_dapm_widget rt5632_dapm_widgets[] = {
SND_SOC_DAPM_INPUT("Left LineIn"),
SND_SOC_DAPM_INPUT("Right LineIn"),
SND_SOC_DAPM_INPUT("Phone"),
SND_SOC_DAPM_INPUT("Mic1"),
SND_SOC_DAPM_INPUT("Mic2"),

SND_SOC_DAPM_PGA("Mic1 Boost", RT5632_PWR_MANAG_ADD3, 1, 0, NULL, 0),
SND_SOC_DAPM_PGA("Mic2 Boost", RT5632_PWR_MANAG_ADD3, 0, 0, NULL, 0),

SND_SOC_DAPM_DAC("Left DAC", "Left HiFi Playback DAC", RT5632_PWR_MANAG_ADD2, 9, 0),
SND_SOC_DAPM_DAC("Right DAC", "Right HiFi Playback DAC", RT5632_PWR_MANAG_ADD2, 8, 0),
SND_SOC_DAPM_DAC("Voice DAC", "Voice Playback DAC", RT5632_PWR_MANAG_ADD2, 10, 0),

SND_SOC_DAPM_PGA("Left LineIn PGA", RT5632_PWR_MANAG_ADD3, 7, 0, NULL, 0),
SND_SOC_DAPM_PGA("Right LineIn PGA", RT5632_PWR_MANAG_ADD3, 6, 0, NULL, 0),
SND_SOC_DAPM_PGA("Phone PGA", RT5632_PWR_MANAG_ADD3, 5, 0, NULL, 0),
SND_SOC_DAPM_PGA("Mic1 PGA", RT5632_PWR_MANAG_ADD3, 3, 0, NULL, 0),
SND_SOC_DAPM_PGA("Mic2 PGA", RT5632_PWR_MANAG_ADD3, 2, 0, NULL, 0),
SND_SOC_DAPM_PGA("Left DAC PGA", RT5632_PWR_MANAG_ADD1, 15, 0, NULL, 0),
SND_SOC_DAPM_PGA("Right DAC PGA", RT5632_PWR_MANAG_ADD1, 14, 0, NULL, 0),
SND_SOC_DAPM_PGA("VoDAC PGA", RT5632_PWR_MANAG_ADD1, 7, 0, NULL, 0),
SND_SOC_DAPM_MIXER("Left Rec Mixer", RT5632_PWR_MANAG_ADD2, 1, 0,
	&rt5632_left_adc_rec_mixer_controls[0], ARRAY_SIZE(rt5632_left_adc_rec_mixer_controls)),
SND_SOC_DAPM_MIXER("Right Rec Mixer", RT5632_PWR_MANAG_ADD2, 0, 0,
	&rt5632_right_adc_rec_mixer_controls[0], ARRAY_SIZE(rt5632_right_adc_rec_mixer_controls)),
SND_SOC_DAPM_MIXER_E("Left HP Mixer", RT5632_PWR_MANAG_ADD2, 5, 0,
	&rt5632_left_hp_mixer_controls[0], ARRAY_SIZE(rt5632_left_hp_mixer_controls),
	mixer_event, SND_SOC_DAPM_POST_REG),
SND_SOC_DAPM_MIXER_E("Right HP Mixer", RT5632_PWR_MANAG_ADD2, 4, 0,
	&rt5632_right_hp_mixer_controls[0], ARRAY_SIZE(rt5632_right_hp_mixer_controls),
	mixer_event, SND_SOC_DAPM_POST_REG),
SND_SOC_DAPM_MIXER("MoNo Mixer", RT5632_PWR_MANAG_ADD2, 2, 0, 
	&rt5632_mono_mixer_controls[0], ARRAY_SIZE(rt5632_mono_mixer_controls)),
SND_SOC_DAPM_MIXER("SPK Mixer", RT5632_PWR_MANAG_ADD2, 3, 0,
	&rt5632_spk_mixer_controls[0], ARRAY_SIZE(rt5632_spk_mixer_controls)),
	
/*hpl mixer --> hp mixer-->spkout mux, hpr mixer-->hp mixer -->spkout mux
   hpl mixer-->hp mixer-->Auxout Mux, hpr muxer-->hp mixer-->auxout mux*/	
SND_SOC_DAPM_MIXER("HP Mixer", SND_SOC_NOPM, 0, 0, NULL, 0),
SND_SOC_DAPM_MIXER("DAC Mixer", SND_SOC_NOPM, 0, 0, NULL, 0),
SND_SOC_DAPM_MIXER("Line Mixer", SND_SOC_NOPM, 0, 0, NULL, 0),

SND_SOC_DAPM_MUX("SPKOUT Mux", SND_SOC_NOPM, 0, 0, &rt5632_spkout_mux_out_controls),
SND_SOC_DAPM_MUX("HPLOUT Mux", SND_SOC_NOPM, 0, 0, &rt5632_hplout_mux_out_controls),
SND_SOC_DAPM_MUX("HPROUT Mux", SND_SOC_NOPM, 0, 0, &rt5632_hprout_mux_out_controls),
SND_SOC_DAPM_MUX("AUXOUT Mux", SND_SOC_NOPM, 0, 0, &rt5632_auxout_mux_out_controls),

SND_SOC_DAPM_PGA_E("SPKL Out PGA", VIRTUAL_REG_FOR_MISC_FUNC, 4, 0, NULL, 0,
				spk_pga_event, SND_SOC_DAPM_POST_PMU | SND_SOC_DAPM_POST_PMD),
SND_SOC_DAPM_PGA_E("SPKR Out PGA", VIRTUAL_REG_FOR_MISC_FUNC, 5, 0, NULL, 0,
				spk_pga_event, SND_SOC_DAPM_POST_PMU | SND_SOC_DAPM_POST_PMD),
SND_SOC_DAPM_PGA_E("HPL Out PGA",VIRTUAL_REG_FOR_MISC_FUNC, 6, 0, NULL, 0, 
				hp_pga_event, SND_SOC_DAPM_POST_PMD | SND_SOC_DAPM_POST_PMU),
SND_SOC_DAPM_PGA_E("HPR Out PGA",VIRTUAL_REG_FOR_MISC_FUNC, 7, 0, NULL, 0, 
				hp_pga_event, SND_SOC_DAPM_POST_PMD | SND_SOC_DAPM_POST_PMU),
SND_SOC_DAPM_PGA_E("AUX Out PGA",RT5632_PWR_MANAG_ADD3, 14, 0, NULL, 0, 
				aux_pga_event, SND_SOC_DAPM_PRE_PMD | SND_SOC_DAPM_POST_PMU),
				

SND_SOC_DAPM_ADC("Left ADC", "Left ADC HiFi Capture", RT5632_PWR_MANAG_ADD2, 7, 0),
SND_SOC_DAPM_ADC("Right ADC", "Right ADC HiFi Capture", RT5632_PWR_MANAG_ADD2, 6, 0),
SND_SOC_DAPM_OUTPUT("SPKL"),
SND_SOC_DAPM_OUTPUT("SPKR"),
SND_SOC_DAPM_OUTPUT("HPL"),
SND_SOC_DAPM_OUTPUT("HPR"),
SND_SOC_DAPM_OUTPUT("AUX"),
SND_SOC_DAPM_MICBIAS("Mic1 Bias", RT5632_PWR_MANAG_ADD1, 3, 0),
SND_SOC_DAPM_MICBIAS("Mic2 Bias", RT5632_PWR_MANAG_ADD1, 2, 0),
};

static const struct snd_soc_dapm_route audio_map[] = {
		/*Input PGA*/

		{"Left LineIn PGA", NULL, "Left LineIn"},
		{"Right LineIn PGA", NULL, "Right LineIn"},
		{"Phone PGA", NULL, "Phone"},
		{"Mic1 Boost", NULL, "Mic1"},
		{"Mic2 Boost", NULL, "Mic2"},
		{"Mic1 PGA", NULL, "Mic1 Boost"},
		{"Mic2 PGA", NULL, "Mic2 Boost"},
		{"Left DAC PGA", NULL, "Left DAC"},
		{"Right DAC PGA", NULL, "Right DAC"},
		{"VoDAC PGA", NULL, "Voice DAC"},
		
		/*Left ADC mixer*/
		{"Left Rec Mixer", "LineIn Capture Switch", "Left LineIn"},
		{"Left Rec Mixer", "Phone Capture Switch", "Phone"},
		{"Left Rec Mixer", "Mic1 Capture Switch", "Mic1"},
		{"Left Rec Mixer", "Mic2 Capture Switch", "Mic2"},
		{"Left Rec Mixer", "HP Mixer Capture Switch", "Left HP Mixer"},
		{"Left Rec Mixer", "SPK Mixer Capture Switch", "SPK Mixer"},
		{"Left Rec Mixer", "MoNo Mixer Capture Switch", "MoNo Mixer"},

		/*Right ADC Mixer*/
		{"Right Rec Mixer", "LineIn Capture Switch", "Right LineIn"},
		{"Right Rec Mixer", "Phone Capture Switch", "Phone"},
		{"Right Rec Mixer", "Mic1 Capture Switch", "Mic1"},
		{"Right Rec Mixer", "Mic2 Capture Switch", "Mic2"},
		{"Right Rec Mixer", "HP Mixer Capture Switch", "Right HP Mixer"},
		{"Right Rec Mixer", "SPK Mixer Capture Switch", "SPK Mixer"},
		{"Right Rec Mixer", "MoNo Mixer Capture Switch", "MoNo Mixer"},
		
		/*HPL mixer*/
		{"Left HP Mixer", "ADC Playback Switch", "Left Rec Mixer"},
		{"Left HP Mixer", "LineIn Playback Switch", "Left LineIn PGA"},
		{"Left HP Mixer", "Phone Playback Switch", "Phone PGA"},
		{"Left HP Mixer", "Mic1 Playback Switch", "Mic1 PGA"},
		{"Left HP Mixer", "Mic2 Playback Switch", "Mic2 PGA"},
		{"Left HP Mixer", "HIFI DAC Playback Switch", "Left DAC PGA"},
		{"Left HP Mixer", "Voice DAC Playback Switch", "VoDAC PGA"},
		
		/*HPR mixer*/
		{"Right HP Mixer", "ADC Playback Switch", "Right Rec Mixer"},
		{"Right HP Mixer", "LineIn Playback Switch", "Right LineIn PGA"},	
		{"Right HP Mixer", "HIFI DAC Playback Switch", "Right DAC PGA"},
		{"Right HP Mixer", "Phone Playback Switch", "Phone PGA"},
		{"Right HP Mixer", "Mic1 Playback Switch", "Mic1 PGA"},
		{"Right HP Mixer", "Mic2 Playback Switch", "Mic2 PGA"},
		{"Right HP Mixer", "Voice DAC Playback Switch", "VoDAC PGA"},

		/*DAC Mixer*/
		{"DAC Mixer", NULL, "Left DAC PGA"},
		{"DAC Mixer", NULL, "Right DAC PGA"},

		/*line mixer*/
		{"Line Mixer", NULL, "Left LineIn PGA"},
		{"Line Mixer", NULL, "Right LineIn PGA"},

		/*spk mixer*/
		{"SPK Mixer", "Line Mixer Playback Switch", "Line Mixer"},
		{"SPK Mixer", "Phone Playback Switch", "Phone PGA"},
		{"SPK Mixer", "Mic1 Playback Switch", "Mic1 PGA"},
		{"SPK Mixer", "Mic2 Playback Switch", "Mic2 PGA"},
		{"SPK Mixer", "DAC Mixer Playback Switch", "DAC Mixer"},
		{"SPK Mixer", "Voice DAC Playback Switch", "VoDAC PGA"},

		/*mono mixer*/
		{"MoNo Mixer", "Line Mixer Playback Switch", "Line Mixer"},
		{"MoNo Mixer", "ADCL Playback Switch","Left Rec Mixer"},
		{"MoNo Mixer", "ADCR Playback Switch","Right Rec Mixer"},
		{"MoNo Mixer", "Mic1 Playback Switch", "Mic1 PGA"},
		{"MoNo Mixer", "Mic2 Playback Switch", "Mic2 PGA"},
		{"MoNo Mixer", "DAC Mixer Playback Switch", "DAC Mixer"},
		{"MoNo Mixer", "Voice DAC Playback Switch", "VoDAC PGA"},
		
		/*hp mixer*/
		{"HP Mixer", NULL, "Left HP Mixer"},
		{"HP Mixer", NULL, "Right HP Mixer"},

		/*spkout mux*/
		{"SPKOUT Mux", "HP Mixer", "HP Mixer"},
		{"SPKOUT Mux", "SPK Mixer", "SPK Mixer"},
		{"SPKOUT Mux", "Mono Mixer", "MoNo Mixer"},
		
		/*hpl out mux*/
		{"HPLOUT Mux", "HPL Mixer", "Left HP Mixer"},
		
		/*hpr out mux*/
		{"HPROUT Mux", "HPR Mixer", "Right HP Mixer"},

		/*aux out mux*/
		{"AUXOUT Mux", "HP Mixer", "HP Mixer"},
		{"AUXOUT Mux", "SPK Mixer", "SPK Mixer"},
		{"SPKOUT Mux", "Mono Mixer", "MoNo Mixer"},

		/*spkl out pga*/
		{"SPKL Out PGA", NULL, "SPKOUT Mux"},
		
		
		/*spkr out pga*/
		{"SPKR Out PGA", NULL, "SPKOUT Mux"},
		
		/*hpl out pga*/
		{"HPL Out PGA", NULL, "HPLOUT Mux"},

		/*hpr out pga*/
		{"HPR Out PGA", NULL, "HPROUT Mux"},

		/*aux out pga*/
		{"AUX Out PGA", NULL, "AUXOUT Mux"}, 
		
		/*left adc*/
		{"Left ADC", NULL, "Left Rec Mixer"},
		
		/*right adc*/
		{"Right ADC", NULL, "Right Rec Mixer"},
		
		/*output*/
		{"SPKL", NULL, "SPKL Out PGA"},
		{"SPKR", NULL, "SPKR Out PGA"},
		{"HPL", NULL, "HPL Out PGA"},
		{"HPR", NULL, "HPR Out PGA"},
		{"AUX", NULL, "AUX Out PGA"},
};


static int rt5632_add_widgets(struct snd_soc_codec *codec)
{
	snd_soc_dapm_new_controls(codec, rt5632_dapm_widgets, 
				ARRAY_SIZE(rt5632_dapm_widgets));
	snd_soc_dapm_add_routes(codec, audio_map, ARRAY_SIZE(audio_map));
//	snd_soc_dapm_new_widgets(codec);

	return 0;
}

#endif


#if !USE_DAPM_CONTROL

static int rt5632_pcm_hw_prepare(struct snd_pcm_substream *substream, struct snd_soc_dai *codec_dai)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	int stream = substream->stream;


	switch (stream)
	{
		case SNDRV_PCM_STREAM_PLAYBACK:	

			//enable playback function
			rt5632_playback_config(codec,1);

			break;
		case SNDRV_PCM_STREAM_CAPTURE:

			//enable record function
			rt5632_record_config(codec,1);	

			break;
		default:
			return 0;
	}
	return 0;	
}

#endif

struct _pll_div{
	u32 pll_in;
	u32 pll_out;
	u16 regvalue;
};


/**************************************************************
  *	watch out!
  *	our codec support you to select different source as pll input, but if you 
  *	use both of the I2S audio interface and pcm interface instantially. 
  *	The two DAI must have the same pll setting params, so you have to offer
  *	the same pll input, and set our codec's sysclk the same one, we suggest 
  *	24576000.
  **************************************************************/
static const struct _pll_div codec_master_pll1_div[] = {
		
	{  2048000,  24576000,	0x2ea0},
	{  3686400,  24576000,	0xee27},	
	{ 12000000,  24576000,	0x2915}, 
	{ 12000000,  22579200,	0x7e2f},     
	{ 13000000,  24576000,	0x772e},
	{ 13100000,	 24576000,	0x0d20},	
	{ 18432000,  24576000, 	0x0290},
	{ 12288000,  24576000, 0x0490},
	{ 7872000, 	 24576000, 0x6519},
	{ 6144000,   24576000, 0x0a90},
	{ 4096000,   24576000, 0x1090},	
	{ 26000000,  24576000,	0x2027},
	{ 26000000,  22579200,	0x392f},	
	{ 24576000,  22579200,	0x0921},	
	{ 24576000,  24576000,	0x02a0},	
};

static const struct _pll_div codec_bclk_pll1_div[] = {

	{  1411200,	 22579200,	0x3ea0},	
	{  3072000,	 22579200,	0x1ea0},
	{  1536000,	 24576000,	0x3ea0},	
	{  3072000,	 24576000,	0x1ea0},
};

static const struct _pll_div codec_vbclk_pll1_div[] = {
	{  1411200,	 22579200,	0x3ea0},	
	{  3072000,	 22579200,	0x1ea0},
	{  1536000,	 24576000,	0x3ea0},	
	{  3072000,	 24576000,	0x1ea0},
};


struct _coeff_div_stereo {
	unsigned int mclk;
	unsigned int rate;
	unsigned int reg60;
	unsigned int reg62;
};

struct _coeff_div_voice {
	unsigned int mclk;
	unsigned int rate;
	unsigned int reg64;
};

static const struct _coeff_div_stereo coeff_div_stereo[] = {
		/*bclk is config to 32fs, if codec is choose to be slave mode , input bclk should be 32*fs */
		{24576000, 48000, 0x3174, 0x1010},      
		{12288000, 48000, 0x1174, 0x0000},
		{18432000, 48000, 0x2174, 0x1111},
		{36864000, 48000, 0x2274, 0x2020},
		{49152000, 48000, 0xf074, 0x3030},
        {24576000, 48000, 0x3172,0x1010},
        {24576000,  8000, 0xB274,0x2424},
        {24576000, 16000, 0xB174,0x2222},
        {24576000, 32000, 0xB074,0x2121},
        {22579200, 11025, 0X3374,0x1414},
        {22579200, 22050, 0X3274,0x1212},
        {22579200, 44100, 0X3174,0x1010},

		{0, 0, 0, 0},
};

static const struct _coeff_div_voice coeff_div_voice[] = {
		/*bclk is config to 32fs, if codec is choose to be slave mode , input bclk should be 32*fs */
		{24576000, 16000, 0x2622}, 
		{24576000, 8000, 0x2824},
		{0, 0, 0},
};

static int get_coeff(unsigned int mclk, unsigned int rate, int mode)
{
	int i;

//	printk("get_coeff mclk = %d, rate = %d\n", mclk, rate);
	if (!mode){
		for (i = 0; i < ARRAY_SIZE(coeff_div_stereo); i++) {
			if ((coeff_div_stereo[i].rate == rate) && (coeff_div_stereo[i].mclk == mclk))
				return i;
		}
	}
	else {
		for (i = 0; i< ARRAY_SIZE(coeff_div_voice); i++) {
			if ((coeff_div_voice[i].rate == rate) && (coeff_div_voice[i].mclk == mclk))
				return i;
		}
	}

	return -EINVAL;
	printk(KERN_ERR "can't find a matched mclk and rate in %s\n", 
				(mode ? "coeff_div_voice[]" : "coeff_div_audio[]"));
}


static int rt5632_codec_set_dai_pll(struct snd_soc_dai *codec_dai, 
		int pll_id,int source, unsigned int freq_in, unsigned int freq_out)
{
	int i;
	int ret = -EINVAL;
	struct snd_soc_codec *codec = codec_dai->codec;

//	printk(KERN_DEBUG "enter %s\n", __func__);

	if (pll_id < RT5632_PLL1_FROM_MCLK || pll_id > RT5632_PLL1_FROM_VBCLK)
		return -EINVAL;

	if (!freq_in || !freq_out)
		return 0;

	if (RT5632_PLL1_FROM_MCLK == pll_id)
	{
		for (i = 0; i < ARRAY_SIZE(codec_master_pll1_div); i ++)
		{
			if ((freq_in == codec_master_pll1_div[i].pll_in) && (freq_out == codec_master_pll1_div[i].pll_out))
			{
				rt5632_write(codec, RT5632_GEN_CTRL_REG2, 0x0000);                    			 /*PLL source from MCLK*/
				rt5632_write(codec, RT5632_PLL_CTRL, codec_master_pll1_div[i].regvalue);   /*set pll code*/
				rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD2, 0x8000, 0x8000);			/*enable pll1 power*/
				ret = 0;
			}
		}
	}
	else if (RT5632_PLL1_FROM_BCLK == pll_id)
	{
		for (i = 0; i < ARRAY_SIZE(codec_bclk_pll1_div); i ++)
		{
			if ((freq_in == codec_bclk_pll1_div[i].pll_in) && (freq_out == codec_bclk_pll1_div[i].pll_out))
			{
				rt5632_write(codec, RT5632_GEN_CTRL_REG2, 0x2000);    					/*PLL source from BCLK*/
				rt5632_write(codec, RT5632_PLL_CTRL, codec_bclk_pll1_div[i].regvalue);   /*set pll1 code*/
				rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD2, 0x8000, 0x8000);       	 /*enable pll1 power*/
				ret = 0;
			}
		}
	}
	else if (RT5632_PLL1_FROM_VBCLK == pll_id)
	{
		for (i = 0; i < ARRAY_SIZE(codec_vbclk_pll1_div); i ++)
		{
			if ((freq_in == codec_vbclk_pll1_div[i].pll_in) && (freq_out == codec_vbclk_pll1_div[i].pll_out))
			{
				rt5632_write(codec, RT5632_GEN_CTRL_REG2, 0x2000);    					/*PLL source from BCLK*/
				rt5632_write(codec, RT5632_PLL_CTRL, codec_vbclk_pll1_div[i].regvalue);   /*set pll1 code*/
				rt5632_write_mask(codec, RT5632_PWR_MANAG_ADD2, 0x8000, 0x8000);       	 /*enable pll1 power*/
				ret = 0;
			}
		}
	}
	
	rt5632_write_mask(codec, RT5632_GEN_CTRL_REG1, 0x8000, 0x8000);
	return ret;
}


static int rt5632_hifi_codec_set_dai_sysclk(struct snd_soc_dai *codec_dai, 
		int clk_id, unsigned int freq, int dir)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct rt5632_priv * rt5632 = codec->drvdata;
//	printk(KERN_DEBUG "enter %s\n", __func__);
	
	if ((freq >= (256 * 8000)) && (freq <= (512 * 48000))) {
		rt5632->stereo_sysclk = freq;
		return 0;
	}
	
	printk(KERN_ERR "unsupported sysclk freq %u for audio i2s\n", freq);
	return 0;
}


static int rt5632_hifi_pcm_hw_params(struct snd_pcm_substream *substream, 
			struct snd_pcm_hw_params *params,
			struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_device *socdev = rtd->socdev;
	struct snd_soc_codec *codec = socdev->card->codec;
	struct rt5632_priv *rt5632 = codec->drvdata;

	unsigned int iface = rt5632_read(codec, RT5632_MAIN_SDP_CTRL) & 0xfff3;
	int rate = params_rate(params);
	int coeff = get_coeff(rt5632->stereo_sysclk, rate, 0);
	
//	printk(KERN_DEBUG "enter %s\n", __func__);


	switch (params_format(params))
	{
		case SNDRV_PCM_FORMAT_S16_LE:
			break;
		case SNDRV_PCM_FORMAT_S20_3LE:
			iface |= 0x0004;
		case SNDRV_PCM_FORMAT_S24_LE:
			iface |= 0x0008;
		case SNDRV_PCM_FORMAT_S8:
			iface |= 0x000c;
	}


	
	rt5632_write(codec, RT5632_MAIN_SDP_CTRL, iface);
	rt5632_write_mask(codec, 0x3a, 0x0801, 0x0801);   /*power i2s and dac ref*/
	if (coeff >= 0) {
		rt5632_write(codec, RT5632_STEREO_DAC_CLK_CTRL1, coeff_div_stereo[coeff].reg60);
		rt5632_write(codec, RT5632_STEREO_DAC_CLK_CTRL2, coeff_div_stereo[coeff].reg62);
	}
	
	return 0;
}


static int rt5632_hifi_codec_set_dai_fmt(struct snd_soc_dai *codec_dai, unsigned int fmt)
{

	struct snd_soc_codec *codec = codec_dai->codec;
	u16 iface = 0;

//	printk(KERN_DEBUG "enter %s\n", __func__);

	/*set master/slave interface*/
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK)
	{
		case SND_SOC_DAIFMT_CBM_CFM:
			iface = 0x0000;
			break;
		case SND_SOC_DAIFMT_CBS_CFS:
			iface = 0x8000;
			break;
		default:
			return -EINVAL;
	}

	/*interface format*/
	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK)
	{
		case SND_SOC_DAIFMT_I2S:
			iface |= 0x0000;
			break;
		case SND_SOC_DAIFMT_LEFT_J:
			iface |= 0x0001;
			break;
		case SND_SOC_DAIFMT_DSP_A:
			iface |= 0x0002;
			break;
		case SND_SOC_DAIFMT_DSP_B:
			iface |= 0x0003;
			break;
		default:
			return -EINVAL;			
	}

	/*clock inversion*/
	switch (fmt & SND_SOC_DAIFMT_INV_MASK)
	{
		case SND_SOC_DAIFMT_NB_NF:
			iface |= 0x0000;
			break;
		case SND_SOC_DAIFMT_IB_NF:
			iface |= 0x0080;
			break;
		default:
			return -EINVAL;
	}

	rt5632_write(codec, RT5632_MAIN_SDP_CTRL, iface);
	return 0;
}


static int rt5632_hifi_codec_mute(struct snd_soc_dai *dai, int mute)
{
	struct snd_soc_codec *codec = dai->codec;
	
	if (mute) 
		rt5632_write_mask(codec, RT5632_STEREO_DAC_VOL, 0x8080, 0x8080);
	else
		rt5632_write_mask(codec, RT5632_STEREO_DAC_VOL, 0x0000, 0x8080);
	return 0;
}

#if USE_DAPM_CONTROL

static int rt5632_set_bias_level(struct snd_soc_codec *codec, 
			enum snd_soc_bias_level level)
{
	switch(level) {
	case SND_SOC_BIAS_ON:
	case SND_SOC_BIAS_PREPARE:
		break;
	case SND_SOC_BIAS_STANDBY:
		break;
	case SND_SOC_BIAS_OFF:
		rt5632_write_mask(codec, 0x04, 0x8080, 0x8080);        /*mute hp*/
		rt5632_write_mask(codec, 0x02, 0x8080, 0x8080);        /*mute spk*/
		rt5632_write(codec, 0x3e, 0x0000);					//power off all bit
		rt5632_write(codec, 0x3a, 0x0000);					//power off all bit
		rt5632_write(codec, 0x3c, 0x0000);					//power off all bit
		
		break;
	}
	codec->bias_level = level;
	return 0;
}

#else

static int rt5632_set_bias_level(struct snd_soc_codec *codec, 
			enum snd_soc_bias_level level)
{
	switch(level) {
	case SND_SOC_BIAS_ON:
	case SND_SOC_BIAS_PREPARE:
		break;
	case SND_SOC_BIAS_STANDBY:
		break;
	case SND_SOC_BIAS_OFF:
		
		rt5632_write_mask(codec, 0x04, 0x8080, 0x8080);        /*mute hp*/
		rt5632_write_mask(codec, 0x02, 0x8080, 0x8080);        /*mute spk*/
		rt5632_write(codec, 0x3e, 0x0000);					//power off all bit
		rt5632_write(codec, 0x3a, 0x0000);					//power off all bit
		rt5632_write(codec, 0x3c, 0x0000);					//power off all bit
		break;
	}
	codec->bias_level = level;
	return 0;
}

#endif

static void rt5632_hifi_pcm_shutdown(struct snd_pcm_substream *substream,
		struct snd_soc_dai *codec_dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_device *socdev = rtd->socdev;
	struct snd_soc_codec *codec = socdev->card->codec;
	int stream = substream->stream;

	switch (stream)
	{
		case SNDRV_PCM_STREAM_PLAYBACK:	
			//disable playback function
			rt5632_playback_config(codec,0);

			break;
		case SNDRV_PCM_STREAM_CAPTURE:			
			//disable record function
			rt5632_record_config(codec,0);

			break;
	}

}

#define RT5632_STEREO_RATES	SNDRV_PCM_RATE_8000_48000

#define RT5632_FORMATS (SNDRV_PCM_FMTBIT_S16_LE |\
			SNDRV_PCM_FMTBIT_S20_3LE |\
			SNDRV_PCM_FMTBIT_S24_LE |\
			SNDRV_PCM_FMTBIT_S8)


static struct snd_soc_dai_ops rt5632_dai_ops_hifi = {

	.hw_params	= rt5632_hifi_pcm_hw_params,
#if !USE_DAPM_CONTROL
	.prepare = rt5632_pcm_hw_prepare,
	.shutdown =rt5632_hifi_pcm_shutdown,
#endif
//	.digital_mute	= rt5632_hifi_codec_mute,
	.set_fmt	= rt5632_hifi_codec_set_dai_fmt,
	.set_pll	= rt5632_codec_set_dai_pll,
	.set_sysclk	= rt5632_hifi_codec_set_dai_sysclk,

};


struct snd_soc_dai rt5632_dai[] = {
	/*hifi codec dai*/
	{
		.name = "RT5632 HiFi",
		.id = 1,
		.playback = {
			.stream_name = "HiFi Playback",
			.channels_min = 1,
			.channels_max = 2,
			.rates = RT5632_STEREO_RATES,
			.formats = RT5632_FORMATS,
		},
		.capture = {
			.stream_name = "HiFi Capture",
			.channels_min = 1,
			.channels_max = 2,
			.rates = RT5632_STEREO_RATES,
			.formats = RT5632_FORMATS,
		},


		.ops = &rt5632_dai_ops_hifi,
	},

	{
		.name = "RT5632 Reserved",
		.id = 2,
	}
};

EXPORT_SYMBOL_GPL(rt5632_dai);


static void rt5632_work(struct work_struct *work)
{
	struct snd_soc_codec *codec =
		 container_of(work, struct snd_soc_codec, delayed_work.work);
	rt5632_set_bias_level(codec, codec->bias_level);
}


#if defined(CONFIG_SND_HWDEP)

#if REALTEK_HWDEP

#define RT_CE_CODEC_HWDEP_NAME "rt56xx hwdep "


static int rt56xx_hwdep_open(struct snd_hwdep *hw, struct file *file)
{
//	printk("enter %s\n", __func__);
	return 0;
}

static int rt56xx_hwdep_release(struct snd_hwdep *hw, struct file *file)
{
//	printk("enter %s\n", __func__);
	return 0;
}


static int rt56xx_hwdep_ioctl_common(struct snd_hwdep *hw, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct rt56xx_cmd rt56xx;
	struct rt56xx_cmd __user *_rt56xx = arg;
	struct rt56xx_reg_state *buf;
	struct rt56xx_reg_state *p;
	struct snd_soc_codec *codec = hw->private_data;

	if (copy_from_user(&rt56xx, _rt56xx, sizeof(rt56xx)))
		return -EFAULT;
	buf = kmalloc(sizeof(*buf) * rt56xx.number, GFP_KERNEL);
	if (buf == NULL)
		return -ENOMEM;
	if (copy_from_user(buf, rt56xx.buf, sizeof(*buf) * rt56xx.number)) {
		goto err;
	}
	switch (cmd) {
		case RT_READ_CODEC_REG_IOCTL:
			for (p = buf; p < buf + rt56xx.number; p++)
			{
				p->reg_value = codec->read(codec, p->reg_index);
			}
			if (copy_to_user(rt56xx.buf, buf, sizeof(*buf) * rt56xx.number))
				goto err;
				
			break;
		case RT_WRITE_CODEC_REG_IOCTL:
			for (p = buf; p < buf + rt56xx.number; p++)
				codec->write(codec, p->reg_index, p->reg_value);
			break;
	}

	kfree(buf);
	return 0;

err:
	kfree(buf);
	return -EFAULT;
	
}

static int rt56xx_codec_dump_reg(struct snd_hwdep *hw, struct file *file, unsigned long arg)
{
	struct rt56xx_cmd rt56xx;
	struct rt56xx_cmd __user *_rt56xx = arg;
	struct rt56xx_reg_state *buf;
	struct snd_soc_codec *codec = hw->private_data;
	int number = codec->reg_cache_size;
	int i;

//	printk(KERN_DEBUG "enter %s, number = %d\n", __func__, number);	
	if (copy_from_user(&rt56xx, _rt56xx, sizeof(rt56xx)))
		return -EFAULT;
	
	buf = kmalloc(sizeof(*buf) * number, GFP_KERNEL);
	if (buf == NULL)
		return -ENOMEM;

	for (i = 0; i < number; i++)
	{
		buf[i].reg_index = i << 1;
		buf[i].reg_value = codec->read(codec, buf[i].reg_index);
	}
	if (copy_to_user(rt56xx.buf, buf, sizeof(*buf) * i))
		goto err;
	rt56xx.number = number;
	if (copy_to_user(_rt56xx, &rt56xx, sizeof(rt56xx)))
		goto err;
	kfree(buf);
	return 0;
err:
	kfree(buf);
	return -EFAULT;
	
}

static int rt56xx_hwdep_ioctl(struct snd_hwdep *hw, struct file *file, unsigned int cmd, unsigned long arg)
{
	if (cmd == RT_READ_ALL_CODEC_REG_IOCTL)
	{
		return rt56xx_codec_dump_reg(hw, file, arg);
	}
	else
	{
		return rt56xx_hwdep_ioctl_common(hw, file, cmd, arg);
	}
}

int realtek_ce_init_hwdep(struct snd_soc_codec *codec)
{
	struct snd_hwdep *hw;
	struct snd_card *card = codec->card;
	int err;

	if ((err = snd_hwdep_new(card, RT_CE_CODEC_HWDEP_NAME, 0, &hw)) < 0)
		return err;
	
	strcpy(hw->name, RT_CE_CODEC_HWDEP_NAME);
	hw->private_data = codec;
	hw->ops.open = rt56xx_hwdep_open;
	hw->ops.release = rt56xx_hwdep_release;
	hw->ops.ioctl = rt56xx_hwdep_ioctl;
	return 0;
}

#endif

#endif

static int rt5632_init(struct snd_soc_device *socdev)
{

	struct snd_soc_codec *codec = socdev->card->codec;

	int ret = 0;

	codec->name = "RT5632";
	codec->owner = THIS_MODULE;
	codec->read = rt5632_read;
	codec->write = rt5632_write;
	codec->set_bias_level = rt5632_set_bias_level;
	codec->dai= rt5632_dai;
	codec->num_dai = 2;
	codec->reg_cache_step = 2;		
	codec->reg_cache_size = ARRAY_SIZE(rt5632_reg)*2;
	codec->reg_cache = kmemdup(rt5632_reg, sizeof(rt5632_reg), GFP_KERNEL);
	if (codec->reg_cache == NULL)
		return -ENOMEM;

	rt5632_reset(codec);

	ret = snd_soc_new_pcms(socdev, SNDRV_DEFAULT_IDX1, SNDRV_DEFAULT_STR1);
	if (ret < 0 )
	{
		printk(KERN_ERR "rt5632:  failed to create pcms\n");
		goto pcm_err;
	}
	
	rt5632_write(codec, RT5632_PD_CTRL_STAT, 0);
	rt5632_write(codec, RT5632_PWR_MANAG_ADD1, PWR_MAIN_BIAS);
	rt5632_write(codec, RT5632_PWR_MANAG_ADD2, PWR_MIXER_VREF);
	rt5632_reg_init(codec);
	rt5632_set_bias_level(codec, SND_SOC_BIAS_PREPARE);
	codec->bias_level = SND_SOC_BIAS_STANDBY;
	schedule_delayed_work(&codec->delayed_work, msecs_to_jiffies(80));
	
	rt5632_add_controls(codec);

	#if USE_DAPM_CONTROL

	rt5632_add_widgets(codec);

	#endif

	#if defined(CONFIG_SND_HWDEP)

		#if REALTEK_HWDEP

		 realtek_ce_init_hwdep(codec);

		#endif

	#endif

#if 1
	return 0;
#else
	ret = snd_soc_init_card(socdev);

	if (ret < 0)
	{
		printk(KERN_ERR "rt5632: failed to register card\n");
		goto card_err;
	}
	printk(KERN_DEBUG "rt5632: initial ok\n");
	return ret;


	card_err:
		snd_soc_free_pcms(socdev);
		snd_soc_dapm_free(socdev);
#endif	
	pcm_err:
//		kfree(codec->reg_cache);
		return ret;
	
	
}


static int rt5632_i2c_probe(struct i2c_client *i2c, const struct i2c_device_id *id)
{
	struct snd_soc_device *socdev = rt5632_socdev;

	struct snd_soc_codec *codec = socdev->card->codec;

	int ret;

	i2c_set_clientdata(i2c, codec);
	codec->control_data = i2c;

	ret = rt5632_init(socdev);
	if (ret < 0)
		pr_err("failed to initialise rt5632\n");

	return ret;
}

static int rt5632_i2c_remove(struct i2c_client *client)
{
	struct snd_soc_codec *codec = i2c_get_clientdata(client);
	kfree(codec->reg_cache);
	return 0;
}

static const struct i2c_device_id rt5632_i2c_id[] = {
		{"rt5632", 0},
		{}
};
MODULE_DEVICE_TABLE(i2c, rt5632_i2c_id);
static struct i2c_driver rt5632_i2c_driver = {
	.driver = {
		.name = "RT5632 I2C Codec",
		.owner = THIS_MODULE,
	},
	.probe =    rt5632_i2c_probe,
	.remove =   rt5632_i2c_remove,
	.id_table = rt5632_i2c_id,
};


static int rt5632_add_i2c_device(struct platform_device *pdev,
				 const struct rt5632_setup_data *setup)
{
#if 0
	struct i2c_board_info info;
	struct i2c_adapter *adapter;
	struct i2c_client *client;
#endif
	int ret;

	ret = i2c_add_driver(&rt5632_i2c_driver);
	if (ret != 0) {
		dev_err(&pdev->dev, "can't add i2c driver\n");
		return ret;
	}

#if 0
	memset(&info, 0, sizeof(struct i2c_board_info));
	info.addr = setup->i2c_address;
	strlcpy(info.type, "rt5632", I2C_NAME_SIZE);

	adapter = i2c_get_adapter(setup->i2c_bus);
	if (!adapter) {
		dev_err(&pdev->dev, "can't get i2c adapter %d\n",
			setup->i2c_bus);
		goto err_driver;
	}

	client = i2c_new_device(adapter, &info);
	i2c_put_adapter(adapter);
	if (!client) {
		dev_err(&pdev->dev, "can't add i2c device at 0x%x\n",
			(unsigned int)info.addr);
		goto err_driver;
	}
#endif
	return 0;

#if 0
err_driver:
	i2c_del_driver(&rt5632_i2c_driver);
	return -ENODEV;
#endif
}


static int rt5632_probe(struct platform_device *pdev)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct rt5632_setup_data *setup = socdev->codec_data;
	struct snd_soc_codec *codec;
	struct rt5632_priv *rt5632;
	int ret;

	pr_info("RT5632 Audio Codec %s", RT5632_VERSION);

	codec = kzalloc(sizeof(struct snd_soc_codec), GFP_KERNEL);
	if (codec == NULL)
		return -ENOMEM;

	rt5632 = kzalloc(sizeof(struct rt5632_priv), GFP_KERNEL);
	if (rt5632 == NULL) {
		kfree(codec);
		return -ENOMEM;
	}

	codec->drvdata = rt5632;

	socdev->card->codec = codec;

	mutex_init(&codec->mutex);
	INIT_LIST_HEAD(&codec->dapm_widgets);
	INIT_LIST_HEAD(&codec->dapm_paths);
	rt5632_socdev = socdev;
	INIT_DELAYED_WORK(&codec->delayed_work, rt5632_work);

	ret = -ENODEV;
	if (setup->i2c_address) {
		codec->hw_write = (hw_write_t)i2c_master_send;
//		codec->hw_read = (hw_read_t)i2c_master_recv;
		ret = rt5632_add_i2c_device(pdev, setup);
	}

	if (ret != 0) {
		kfree(codec->drvdata);
		kfree(codec);
		socdev->card->codec = NULL;
	}
	return ret;
}

static int run_delayed_work(struct delayed_work *dwork)
{
	int ret;

	/* cancel any work waiting to be queued. */
	ret = cancel_delayed_work(dwork);

	/* if there was any work waiting then we run it now and
	 * wait for it's completion */
	if (ret) {
		schedule_delayed_work(dwork, 0);
		flush_scheduled_work();
	}
	return ret;
}


static int rt5632_remove(struct platform_device *pdev)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct snd_soc_codec *codec = socdev->card->codec;


	if (codec->control_data)
		rt5632_set_bias_level(codec, SND_SOC_BIAS_OFF);
	run_delayed_work(&codec->delayed_work);
	snd_soc_free_pcms(socdev);
	snd_soc_dapm_free(socdev);
	i2c_unregister_device(codec->control_data);
	i2c_del_driver(&rt5632_i2c_driver);
	kfree(codec->drvdata);
	kfree(codec);

	return 0;
}


static int rt5632_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);

	struct snd_soc_codec *codec = socdev->card->codec;

	rt5632_set_bias_level(codec, SND_SOC_BIAS_OFF);

	return 0;
}

static int rt5632_resume(struct platform_device *pdev)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct snd_soc_codec *codec = socdev->card->codec;
	
	int i;
	u8 data[3];
	u16 *cache = codec->reg_cache;

	/* Sync reg_cache with the hardware */
	for (i = 0; i < ARRAY_SIZE(rt5632_reg); i++) {
		if (i == RT5632_RESET)
			continue;
		data[0] = i << 1;
		data[1] = (0xff00 & cache[i]) >> 8;
		data[2] = 0x00ff & cache[i];
		codec->hw_write(codec->control_data, data, 3);
	}

	rt5632_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	/* charge rt5632 caps */
	if (codec->suspend_bias_level == SND_SOC_BIAS_ON) {
		rt5632_set_bias_level(codec, SND_SOC_BIAS_PREPARE);
		codec->bias_level = SND_SOC_BIAS_ON;
		schedule_delayed_work(&codec->delayed_work,
					msecs_to_jiffies(1000));
	}

	return 0;
}


struct snd_soc_codec_device soc_codec_dev_rt5632 = {
	.probe = 	rt5632_probe,
	.remove = 	rt5632_remove,
	.suspend = 	rt5632_suspend,
	.resume =	rt5632_resume,
};

EXPORT_SYMBOL_GPL(soc_codec_dev_rt5632);

static int __init rt5632_modinit(void)
{
	return snd_soc_register_dais(rt5632_dai, ARRAY_SIZE(rt5632_dai));
}

static void __exit rt5632_exit(void)
{
	snd_soc_unregister_dais(rt5632_dai, ARRAY_SIZE(rt5632_dai));
}

module_init(rt5632_modinit);
module_exit(rt5632_exit);
MODULE_LICENSE("GPL");

