/*
 * stm32l051.c - Intel micro controller driver
 *
 * Copyright (c) 2016 Intel Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/mutex.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/acpi.h>
#include <linux/cdev.h>
#include <linux/i2c.h>
#include <linux/fs.h>
#include <linux/io.h>
#include <linux/of_gpio.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include<linux/circ_buf.h>
#include <linux/interrupt.h>
#include <linux/gpio/consumer.h>
#include <linux/mfd/intel_soc_pmic.h>
#include <linux/regulator/consumer.h>
//#include <linux/vlv2_plat_clock.h>
//#include <linux/atomisp_gmin_platform.h>
#include <linux/timer.h>
#include <linux/kfifo.h>
#include <linux/time.h>
#include <linux/poll.h>


#include <linux/uaccess.h>
#include <linux/atomic.h>
#include <linux/errno.h>
#include <linux/kthread.h>


#include "stm32l051.h"

MODULE_AUTHOR("Nael Masalha <nael.masalha@intel.com>");
MODULE_DESCRIPTION("Intel Timestap Micro Controller Driver");
MODULE_LICENSE("GPL v2");

static int i2c_addr = 0x21;



/* driver parameters */
static int power_up_retries = 5;
static int i2c_read_retry_count = 3;
static int i2c_write_retry_count = 3;
static int uctl_major = 0;
static int uctl_base_minor = 0;
static int uctl_live_devs = 0;

module_param(i2c_addr, int, S_IRUGO);
module_param(power_up_retries, int, S_IRUGO);
module_param(i2c_read_retry_count, int, S_IRUGO);
module_param(i2c_write_retry_count, int, S_IRUGO);

#define DEVICE_NAME "stm32l051"
#define DEVICE_NODE_NAME "stm32l051-%d"

static struct class *uctl_dev_class;
/*
 * Manage driver versioning
 */
const u32 stm32l051_driver_version = 0x03000100;

#define INTERRUPT_FALLBACK_TIMEOUT_MS 180
#define SLEEP_TIME_AFTER_REGULATOR_POWER_ON_MS 200
#define PRINT_REGISTER_TRANSACTIONS 0
#define PRINT_SENSOR_INFO 0
#define STATISTICS_ENABLED 0

/* TODO: not sure about this, taken from cherytrail regulator driver */
#define WCOVE_V1P8SX_CTRL				0x57
#define WCOVE_V2P8SX_CTRL				0x5d
#define WCOVE_CTRL_MASK					0x07
#define WCOVE_CTRL_ENABLE				0x02
#define WCOVE_CTRL_DISABLE				0x00

/* uctl i2c control registers */
#define UCTL_REG_UCTRL_CFG				0x00
#define UCTL_REG_INT_ID_CONFIG				0x04
#define UCTL_REG_INT_TYPE_CONFIG			0x08
#define UCTL_REG_EEPROM_CMD				0x0C
#define UCTL_REG_EEPROM_DATA				0xD0
#define UCTL_REG_TIMER_PRESCALER			0x14
#define UCTL_REG_TIMER_RESET_VALUE			0x18
#define UCTL_REG_GYRO_BYPASS_CMD1			0x1C
#define UCTL_REG_GYRO_BYPASS_CMD2			0x20
#define UCTL_REG_GYRO_BYPASS_RDOUT1			0x24
#define UCTL_REG_GYRO_BYPASS_RDOUT2			0x28
#define UCTL_REG_ACC_BYPASS_CMD1			0x2C
#define UCTL_REG_ACC_BYPASS_CMD2			0x30
#define UCTL_REG_ACC_BYPASS_RDOUT1			0x34
#define UCTL_REG_ACC_BYPASS_RDOUT2			0x38
#define UCTL_REG_REQ_LTR				0x3C
#define UCTL_REG_STATUS_REG				0x40
#define UCTL_ERROR_DETECTED				0x01
#define UCTL_ONE_FIFO_REACHED_LTR			0x02
#define UCTL_REAUIRED_CONFIGURATION_DONE		0x04
#define UCTL_READY_FOR_NEXT_EEPROM			0x08
#define UCTL_POWER_MODE_CHANGE_DONE			0x10
#define UCTL_REG_FIFO_RD				0x44
#define UCTL_NUM_FIFO_ENTRIES_MASK			0x3F
#define UCTL_NUM_TS_FIFO_SHIFT				0
#define UCTL_NUM_IMU_FIFO_SHIFT				6
#define UCTL_REG_FIRST_INT_IGNORE			0x48
#define UCTL_REG_IAP_REG				0x4C
#define UCTL_REG_INT_ENABLE				0x50
#define UCTL_INT_ENABLE_DEFAULT				0x0F
#define UCTL_INT_G2VSYNC_ENABLE				0x40
#define UCTL_INT_G2VSYNC_DISABLE			0xBF
#define UCTL_REG_CURR_PWR_STATE				0x54
#define UCTL_REG_NEXT_PWR_STATE				0x58
#define UCTL_PWR_STATE_DNR				0x00
#define UCTL_PWR_STATE_INIT				0x02
#define UCTL_PWR_STATE_ACTIVE				0x03
#define UCTL_PWR_STATE_PAUSE				0x04
#define UCTL_PWR_STATE_IAP				0x05
#define UCTL_REG_ACC_BW					0x5C
#define UCTL_REG_GYRO_BW				0x60
#define UCTL_REG_ACC_RANGE				0x64
#define UCTL_REG_GYRO_RANGE				0x68
#define UCTL_REG_IMG_VER				0x6C
#define UCTL_REG_TIMESTAMP				0x74

/* uctl i2c data registers */
#define UCTL_FIFO_DATA_TS				0xEC
#define UCTL_FIFO_DATA_IMU				0xF4

/* uctl i2c error handling register */
#define UCTL_ERROR_REG					0x70

#define UCTL_JUMP_TO_APP_REG				0x77

/* register length in bytes */
#define UCTL_8BIT					0x01
#define UCTL_16BIT					0x02
#define UCTL_32BIT					0x04

/* FIFO size in entries
 * number of entries in TS FIFO in FW is 128
 * number of entries in IMU FIFO in FW is 150
 * allocating FIFO with power 2 number of entries,
 * this is the minimal power 2 which is bigger than 278
 */
#define UCTL_FIFO_NUM_ENTRIES				0x400
/* size of data entry in bytes */

#define ACC_BW_125_HZ					0xB
#define ACC_BW_250_HZ					0xC
#define ACC_BW_500_HZ					0xD
#define ACC_BW_1000_HZ					0xE
#define ACC_BW_2000_HZ					0xF


#define ACC_RANGE_4					0x5
#define ACC_RANGE_8					0x7
#define ACC_RANGE_16					0xC



#define GYRO_BW_100_HZ					0x07
#define GYRO_BW_200_HZ					0x06
#define GYRO_BW_400_HZ					0x03
#define GYRO_BW_1000_HZ					0x02
#define GYRO_BW_2000_HZ					0x00




#define GYRO_RANGE_2000					0x0
#define GYRO_RANGE_1000					0x1

#define HW_SYNC_ENABLED					0x8

#define FIFO_LENGTH_MASK				0x7F
#define IMU_FIFO_SHIFT					7

#define G0_FALLING_EDGE_MSK				0x10
#define G1_FALLING_EDGE_MSK				0x20

#define SEC_TO_NANOSEC 1000000000
#define IMU_FREQUENCY  32000000

#define STM32L051_NUM_ERRORS 32
static char *g_error_codes[STM32L051_NUM_ERRORS] = {
	"Driver FIFO full",
	"FW error",
	"Frame Loss detected",
	"I2C timeout",
	"I2C remote IO error",
	"I2C remote unspecified error",
	"FW update failed write",
	"FW update failed FW corrupted",
	"Unknown error 8",
	"Unknown error 9",
	"Unknown error 10",
	"Unknown error 11",
	"Unknown error 12",
	"Unknown error 13",
	"Unknown error 14",
	"Unknown error 15",
	"Unknown error 16",
	"Unknown error 17",
	"Unknown error 18",
	"Unknown error 19",
	"Unknown error 20",
	"Unknown error 21",
	"Unknown error 22",
	"Unknown error 23",
	"Unknown error 24",
	"Unknown error 25",
	"Unknown error 26",
	"Unknown error 27",
	"Unknown error 28",
	"Unknown error 29",
	"Unknown error 30",
	"Unknown error 31",
};

enum stm32l051_fw_error {
	STM32L051_FW_ERR_IMU_NOT_RESPONDING = 0x1,
	STM32L051_FW_ERR_ILLEGAL_CONFIGURATION = 0x2,
	STM32L051_FW_ERR_TS_FIFO_OVERFLOW = 0x4,
	STM32L051_FW_ERR_IMU_FIFO_OVERFLOW = 0x8,
	STM32L051_FW_ERR_WATCH_DOG_RESET = 0x10,
	STM32L051_FW_ERR_CRC_ERROR = 0x20,
	STM32L051_FW_ERR_MISSING_DATA_FROM_IMU = 0x40,
	STM32L051_FW_ERR_BOOT_LOADER_FAILURE = 0x80,
	STM32L051_FW_ERR_EEPROM_ERROR = 0x100,
	STM32L051_FW_ERR_BIST_CHECK_FAILED = 0x200,
};

#define NUM_FW_ERRORS 32
static char *g_fw_error_codes[NUM_FW_ERRORS] = {
	"IMU not responding",
	"illegal configuration",
	"TS FIFO overflow",
	"IMU FIFO overflow",
	"watch dog reset",
	"CRC error",
	"missing data from IMU",
	"Boot loader failure",
	"EEPROM error",
	"BIST check failed",
	"Unknown error 10",
	"Unknown error 11",
	"Unknown error 12",
	"Unknown error 13",
	"Unknown error 14",
	"Unknown error 15",
	"Unknown error 16",
	"Unknown error 17",
	"Unknown error 18",
	"Unknown error 19",
	"Unknown error 20",
	"Unknown error 21",
	"Unknown error 22",
	"Unknown error 23",
	"Unknown error 24",
	"Unknown error 25",
	"Unknown error 26",
	"Unknown error 27",
	"Unknown error 28",
	"Unknown error 29",
	"Unknown error 30",
	"Unknown error 31",
};

enum sensor_interrupt_mask {
	SENSOR_INT_MASK_ACC = 0x1,
	SENSOR_INT_MASK_GYRO = 0x2,
	SENSOR_INT_MASK_DEPTH = 0x4,
	SENSOR_INT_MASK_MOTION = 0x8,
	SENSOR_INT_MASK_G0_VSYNC = 0x10,
	SENSOR_INT_MASK_G1_VSYNC = 0x20,
	SENSOR_INT_MASK_G2_VSYNC = 0x40,
};

union id_config_register {
	struct {
		u32 acc_id:4;		/* 00::02 */
		u32 gyro_id:4;		/* 04::06 */
		u32 depth_id:4;		/* 08::10 */
		u32 motion_id:4;	/* 12::14 */
		u32 G0Vsync_id:4;	/* 16::18 */
		u32 G1Vsync_id:4;	/* 20::22 */
		u32 G2Vsync_id:4;	/* 24::26 */
		u32 reserved:4;		/* 27::31*/
	} bit_fields;
	u32 value;
};

struct sensor_ctrl {
	__u32 handle;
	char name[STM32L051_SENSOR_NAME_MAX_LENGTH];
	enum stm32l051_sensor_type type;
	__u32 frame_id_multiplier;
	__u16 next_frame_id;
	bool was_reset;
	bool filter_out;
	enum sensor_interrupt_mask interrupt_mask;
};

#define NUM_ACCELEROMETERS 1
#define NUM_GYROSCOPES 1
#define NUM_DEPTH_CAMERAS 1
#define NUM_MOTION_CAMERAS 1
#define NUM_EXTERNAL_SENSORS 2

struct external_sensor_ctrl {
	struct sensor_ctrl common;
	/* external control index, used to access the specific GPIO registers */
	u32 index;
};

struct external_sensors_ctrl {
	int handle_root;
	struct external_sensor_ctrl sensors[NUM_EXTERNAL_SENSORS];
};

#define NUM_SENSORS	(NUM_ACCELEROMETERS + \
			 NUM_GYROSCOPES + \
			 NUM_DEPTH_CAMERAS + \
			 NUM_MOTION_CAMERAS + \
			 NUM_EXTERNAL_SENSORS)

/**
 * This struct contains all device sensors.
 */
struct sensors_ctrl {
	struct sensor_ctrl accelerometer;
	struct sensor_ctrl gyroscope;
	struct sensor_ctrl depth_camera;
	struct sensor_ctrl motion_camera;
	struct external_sensors_ctrl external_sensors;

	/* This array holds reference to the fields above, key is the handle. */
	struct sensor_ctrl *controls[NUM_SENSORS];
};

/**
 * Device State Enum.
 * enumerations are used as bitwise for checking valid states.
 */
enum device_state {
	DEVICE_STATE_CLOSED	   = 0x1,
	DEVICE_STATE_UNINITIALIZED = 0x2,
	DEVICE_STATE_STOPPED	   = 0x4,
	DEVICE_STATE_PAUSED	   = 0x8,
	DEVICE_STATE_STREAMING	   = 0x10,
	DEVICE_STATE_STOPPING	   = 0x20,
	DEVICE_STATE_IAP	   = 0x40,
	DEVICE_STATE_FW_CORRUPTED  = 0x80,
};

struct statistic_accumulator {
	u64 count;
	u64 samples;
	u64 min;
	u64 max;
	u64 presamples;
};

struct device_statistics {
	/* Accumulates the delta between hard interrupts. */
	struct statistic_accumulator hard_interrupt_delta;
	/* Accumulates the delta between hard interrupts. */
	struct statistic_accumulator soft_interrupt_delta;
	/* Accumulates the delta between fallback timer expirations. */
	struct statistic_accumulator fallback_timer_delta;
	/* Accumulates the read IMU FIFO times */
	struct statistic_accumulator read_imu_fifo_delta;
	/* Accumulates the read TS times */
	struct statistic_accumulator read_ts_fifo_delta;
	/* Accumulates the number of IMU entries per read. */
	struct statistic_accumulator imu_entries;
	/* Accumulates the number of TS entries per read. */
	struct statistic_accumulator ts_entries;
	/* Counts the number of read HW FIFO iterations. */
	struct statistic_accumulator read_hw_fifo_iterations;
	/* Accumulates the delta between poll while FIFO is empty. */
	struct statistic_accumulator poll_delta;
	/* Counts frame losses. */
	u64 sensor_frame_loss[NUM_SENSORS];
};

struct device_timing_measurements {
	u64 hard_irq_time;
	u64 soft_irq_time;
	u64 fallback_timer_time;
	u64 poll_time;
};

struct device_status {
	struct mutex device_status_lock;
	u32 major_error;
	u32 fw_error;
	bool error_reported;
};

union raw_sensor_header {
	struct fields {
		u16 handle:3;
		u16 frame_num:12;
		u16 reserved:1;
	}__attribute__((packed, aligned(1))) bit_fields;
	u16 value;
}__attribute__((packed, aligned(1)));

struct raw_sensor_common_data {
	union raw_sensor_header header;
	__le32 timestamp;
}__attribute__((packed, aligned(1)));;

struct raw_imu_data {
	struct raw_sensor_common_data common;
	__le16 x;
	__le16 y;
	__le16 z;
}__attribute__((packed, aligned(1)));

struct raw_ts_data {
	struct raw_sensor_common_data common;
}__attribute__((packed, aligned(1)));

/* FIFO RD has 6 bits for every FIFO */
#define MAX_HW_FIFO_ENTRIES (1 << 6)
#define IMU_ENTRY_SIZE (sizeof(struct raw_imu_data))
#define TS_ENTRY_SIZE (sizeof(struct raw_ts_data))
/* ((max fifo entries = 64) * (max entry size = 12)) = 768	 */
#define MAX_HW_FIFO_READ_SIZE (MAX_HW_FIFO_ENTRIES * IMU_ENTRY_SIZE)
#define MAX_IMU_READ_SIZE \
	((((size_t)MAX_HW_FIFO_READ_SIZE) / IMU_ENTRY_SIZE) * IMU_ENTRY_SIZE)
#define MAX_TS_READ_SIZE \
	((((size_t)MAX_HW_FIFO_READ_SIZE) / TS_ENTRY_SIZE) * TS_ENTRY_SIZE)

/**
 * Driver Context
 */
struct device_context {
	int major;
	struct mutex device_handle_lock;
	struct mutex hw_events_lock;

	struct cdev cdev;
	struct device *chrdev;
	struct i2c_client *i2c_client;

	STRUCT_KFIFO(struct stm32l051_sensor_data, UCTL_FIFO_NUM_ENTRIES) fifo;
	u8 hw_fifo_read_buffer[MAX_HW_FIFO_READ_SIZE];

	struct gpio_desc *pa8_gpiod;
	struct gpio_desc *pa9_gpiod;
	struct gpio_desc *pa10_gpiod;

	struct regulator *v1p8_reg;
	struct regulator *v3p3_reg;
	int regulator_was_enabled;
	bool pmic_powered_on;

	enum device_state state;

	struct sensors_ctrl sensors;

	u32 status_register;
	u32 timer_prescaler;
    struct stm32l051_uc_timestamp last_uc_timestamp_value;
    bool uc_timestamp_polling_enabled;

	struct raw_imu_data gyro_last_read;
	u32 gyro_overflow_count;
	struct raw_imu_data accel_last_read;
	u32 accel_overflow_count;
	struct raw_ts_data ts_last_read;
	u32 ts_overflow_count;
	u64 softirq_time_stamp;
	wait_queue_head_t block_read_wait_queue;
	wait_queue_head_t wait_queue;
	wait_queue_head_t wait_power_up;
	bool power_change_int_received;
	struct timer_list fallback_timer;
	struct task_struct *process_hw_events_thread;

	bool interrupt_enabled;
    u32 timestamp_polling_counter;
	struct input_dev *idev;
	struct device_status dev_status;

	struct device_statistics statistics;
	struct device_timing_measurements timing_measurements;
};

struct read_fifo_configuration {
	u32 num_entries;
	size_t entry_size;
	size_t max_read_size;
	u32 fifo_address;
	int (*parse_entry)(struct device_context *dev_ctxt,
			   struct stm32l051_sensor_data *dest,
			   u8 *src);
	const char *name;
};

enum vsync_mode {
	vsync_mode_na,
	vsync_mode_timestamp,
	vsync_mode_interrupt,
	vsync_mode_uart_tx,
	vsync_mode_uart_rx
};

struct vsync_info {
	enum vsync_mode g0_mode;
	enum vsync_mode g1_mode;
	enum vsync_mode g2_mode;
};

/* Function Headers */
static void init_sensor_ctrl(struct sensors_ctrl *sensors, struct sensor_ctrl *ctrl, __u32 handle, char *name, enum stm32l051_sensor_type type, enum sensor_interrupt_mask interrupt_mask);
static void reset_sensor_ctrl(struct sensor_ctrl *ctrl);
static int uctl_i2c_read(struct i2c_client *client, u8 addr, u16 len, u8 *buf);
static int uctl_read_reg(struct i2c_client *client, u16 reg, u16 data_length, u32 *val);
static int uctl_read_reg_with_retries(struct i2c_client *client, u16 reg, u16 data_length, u32 *val, int retries, int sleep_ms);
static int uctl_i2c_write(struct i2c_client *client, u16 len, u8 *data);
static int uctl_write_reg(struct i2c_client *client, u8 reg, u16 data_length, u32 val);
static int pmic_ctrl(struct device_context *dev_ctxt, bool power_on);
static int check_power_up(struct device_context *dev_ctxt);
static void fill_common_sensor_info(struct device_context *dev_ctxt, struct stm32l051_sensor_t *info, struct sensor_ctrl *ctrl);
static char __user *add_sensor_info_to_sensors_list(struct device_context *dev_ctxt, char __user *buf, struct sensor_ctrl *sensor);
static int init_accelerometer(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle);
static int init_gyroscope(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle);
static int init_depth_camera(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle);
static int init_motion_camera(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle);
static int init_external_sensors(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle);
static int init_sensors(struct device_context *dev_ctxt);
static void reset_sensors(struct device_context *dev_ctxt);
static int init_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int uninit(struct device_context *dev_ctxt);
static int uninit_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int get_device_info_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int get_sensors_list_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int set_state_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int handle_to_sensor_ctrl_index(struct device_context *dev_ctxt, int handle, int handle_root, int num_sensors);
static int set_accl_config_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int set_gyro_config_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int set_ext_sensor_config_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int set_depth_cam_config_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int set_motion_cam_config_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int set_hw_sync_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int set_ltr_config_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int set_timestamp_config_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int update_firmware_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int read_eeprom_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int write_eeprom_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int get_drv_error_cmd(struct device_context *dev_ctxt, unsigned long arg);
static int uctl_probe(struct i2c_client *client, const struct i2c_device_id *id);
static int uctl_remove(struct i2c_client *client);
static void uctl_shutdown(struct i2c_client *client);
static int read_hw_fifo(struct i2c_client *client);
static int read_fifo(struct device_context *dev_ctxt, const struct read_fifo_configuration *cfg);
static int start_stream(struct device_context *dev_ctxt);
static int pause_stream(struct device_context *dev_ctxt);
static int stop_stream(struct device_context *dev_ctxt);
static void reg_to_vsync_info(struct device_context *dev_ctxt, struct vsync_info *info, u32 uctrl_cfg_reg);
static int vsync_info_to_reg(struct device_context *dev_ctxt, struct vsync_info *info, u32 *reg);
static int __init uctl_init(void);
static void __exit uctl_exit(void);
static int parse_imu_entry(struct device_context *dev_ctxt, struct stm32l051_sensor_data *dest, u8 *src);
static irqreturn_t uctl_hard_irq_handler(int irq, void *dev_id, void *data);
static irqreturn_t uctl_soft_irq_handler(int irq, void *dev_id, void *data);
static int uctl_irq_init(struct device_context *dev_ctxt);
static void fallback_timer_handler(unsigned long context);
static int process_hw_events_thread_handler(void *data);
static int process_hw_events(struct device_context *dev_ctxt);
static void enable_int(struct device_context *dev_ctxt, const char *caller);
static void disable_int(struct device_context *dev_ctxt, const char *caller);
static int process_status_register(struct device_context *dev_ctxt);
static ssize_t sysfs_show_error(struct device *dev, struct device_attribute *attr, char *buf);
static ssize_t sysfs_show_fw_error(struct device *dev, struct device_attribute *attr, char *buf);
static ssize_t sysfs_show_version(struct device *dev, struct device_attribute *attr, char *buf);
static void init_device_status(struct device_context *dev_ctxt);
static void destroy_device_status(struct device_context *dev_ctxt);
static void set_driver_error(struct device_context *dev_ctxt, enum stm32l051_error_t error);
static void update_fw_error(struct device_context *dev_ctxt);
static ssize_t stm32l051_read(struct file *filp, char __user *buff, size_t len, loff_t *offset);
static ssize_t stm32l051_write(struct file *filp, const char __user *buff, size_t count, loff_t *f_pos);
static long stm32l051_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);
static int stm32l051_open(struct inode *node, struct file *filp);
static int stm32l051_release(struct inode *node, struct file *filp);
static int query_timer_prescaler(struct device_context *dev_ctxt);
static int config_sensor_interrupt(struct device_context *dev_ctxt, enum sensor_interrupt_mask mask, bool enable);
static bool is_sensor_interrupt_enabled(struct device_context *dev_ctxt, enum sensor_interrupt_mask mask);

#if STATISTICS_ENABLED
static void statistic_accumulator_add(struct statistic_accumulator *accumulator, u64 count);
#define reset_statistic_accumulator(_stat_acc) {memset(&_stat_acc, 0, sizeof(_stat_acc)); }
static void reset_statistics(struct device_context *dev_ctxt);
static void reset_statistics_accumulator(struct statistic_accumulator *accumulator);
static void reset_timing_measurements(struct device_context *dev_ctxt);
static void update_time_delta(u64 *time_sample, struct statistic_accumulator *accumulator);
#define statistic_accumulator_avg(_acc) ((_acc.samples > 0) ? (_acc.count / _acc.samples) : 0)
static u64 get_time_ms(void);
static ssize_t sysfs_reset_statistics(struct device *dev, struct device_attribute *attr, const char *buf, size_t count);
static ssize_t sysfs_show_statistics(struct device *dev, struct device_attribute *attr, char *buf);
static int sysfs_print_statistics_accumulator(char *buf, const char *message, const char *units, const struct statistic_accumulator *accumulator);
#else /* STATISICS_ENABLED */
#define statistic_accumulator_add(_accumulator, _count)
#define update_time_delta(_time_sample, _accumulator)
#define reset_statistics(_dev_ctxt)
#define reset_timing_measurements(_dev_ctxt)
#endif /* STATISICS_ENABLED */

static int create_main_thread(struct device_context *dev_ctxt)
{
	dev_ctxt->process_hw_events_thread =
		kthread_run(process_hw_events_thread_handler,
						(void *) dev_ctxt, "st32l051");

	if (IS_ERR(dev_ctxt->process_hw_events_thread)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to create fallback thread\n", __func__);
		dev_ctxt->process_hw_events_thread = NULL;
		return -ERESTARTSYS;
	}

	return 0;
}

static void stop_main_thread(struct device_context *dev_ctxt)
{
	if (dev_ctxt->process_hw_events_thread != NULL) {
		dev_info(&dev_ctxt->i2c_client->dev, "%s stopping thread\n", __func__);
		kthread_stop(dev_ctxt->process_hw_events_thread);
		dev_ctxt->process_hw_events_thread = NULL;
		dev_info(&dev_ctxt->i2c_client->dev, "%s thread stopped\n", __func__);
	} else {
		dev_info(&dev_ctxt->i2c_client->dev, "%s thread is alrady stopped\n", __func__);
	}
}

static int power_up(struct device_context *dev_ctxt)
{
	int i, res;
	res = pmic_ctrl(dev_ctxt, true);
	enable_int(dev_ctxt, __func__);

	switch(res) {
	case -EALREADY:
		dev_info(&dev_ctxt->i2c_client->dev, "%s(): already on!\n", __func__);
		dev_ctxt->power_change_int_received = 1;
		return create_main_thread(dev_ctxt);
	case 0:
		break;
	default:
		return res;
	}
	res = create_main_thread(dev_ctxt);
	if (res != 0)
			return res;
	dev_ctxt->power_change_int_received = 0;
	dev_info(&dev_ctxt->i2c_client->dev, "%s going to wait\n", __func__);
	for (i=0; i<10;i++) {
        process_hw_events(dev_ctxt);
		res = wait_event_interruptible_timeout(dev_ctxt->wait_power_up, dev_ctxt->power_change_int_received != 0, HZ/10);
        
		if (res > 0) {
			dev_info(&dev_ctxt->i2c_client->dev, "%s i=%d, wait returned %d, pwr state=%d\n", __func__, i, res, dev_ctxt->power_change_int_received);
			return 0;
		}
	}
	dev_info(&dev_ctxt->i2c_client->dev, "%s GRRRR! i=%d, wait returned %d, pwr state=%d\n", __func__, i, res, dev_ctxt->power_change_int_received);
	return -ERESTARTSYS;
}

static int power_down(struct device_context *dev_ctxt)
{
	disable_int(dev_ctxt, __func__);
	stop_main_thread(dev_ctxt);
	return pmic_ctrl(dev_ctxt, false);
}

#if STATISTICS_ENABLED

static u64 get_time_ms(void)
{
	struct timespec time = current_kernel_time();
	return timespec_to_ns(&time) / 1000000;
}

static void statistic_accumulator_add(struct statistic_accumulator *accumulator, u64 count)
{
	if((accumulator->presamples++) < 100)
		return;
	accumulator->count += count;
	accumulator->samples++;
	if (count > accumulator->max)
		accumulator->max = count;

	if (count < accumulator->min)
		accumulator->min = count;
}

static void update_time_delta(u64 *time_sample, struct statistic_accumulator *accumulator)
{
	u64 time = get_time_ms();

	/* accumulator can be NULL if the user only wants to get the time. */
	if (accumulator) {
		if ((*time_sample) != 0) {
			u64 delta = (time - (*time_sample));
			statistic_accumulator_add(accumulator, delta);
		}
	}
	(*time_sample) = time;
}

static void reset_statistics(struct device_context *dev_ctxt)
{
	memset(&dev_ctxt->statistics, 0, sizeof(dev_ctxt->statistics));
	/* reset function will set the accumulators min & max fields. */
	reset_statistics_accumulator(&dev_ctxt->statistics.hard_interrupt_delta);
	reset_statistics_accumulator(&dev_ctxt->statistics.soft_interrupt_delta);
	reset_statistics_accumulator(&dev_ctxt->statistics.fallback_timer_delta);
	reset_statistics_accumulator(&dev_ctxt->statistics.read_imu_fifo_delta);
	reset_statistics_accumulator(&dev_ctxt->statistics.read_ts_fifo_delta);
	reset_statistics_accumulator(&dev_ctxt->statistics.imu_entries);
	reset_statistics_accumulator(&dev_ctxt->statistics.ts_entries);
	reset_statistics_accumulator(&dev_ctxt->statistics.read_hw_fifo_iterations);
	reset_statistics_accumulator(&dev_ctxt->statistics.poll_delta);
}

static void reset_statistics_accumulator(struct statistic_accumulator *accumulator)
{
	accumulator->count = 0;
	accumulator->samples = 0;
	accumulator->min = (u64) (-1);
	accumulator->max = 0;
	accumulator->presamples = 0;
}

static void reset_timing_measurements(struct device_context *dev_ctxt)
{
	memset(&dev_ctxt->timing_measurements, 0, sizeof(dev_ctxt->timing_measurements));
}

#endif /*STATISTICS_ENABLED*/

static void init_device_status(struct device_context *dev_ctxt)
{
	memset(&dev_ctxt->dev_status, 0, sizeof(dev_ctxt->dev_status));
	mutex_init(&dev_ctxt->dev_status.device_status_lock);
}
static void destroy_device_status(struct device_context *dev_ctxt)
{
	mutex_destroy(&dev_ctxt->dev_status.device_status_lock);
}

static void set_driver_error(struct device_context *dev_ctxt, enum stm32l051_error_t error)
{
	bool is_new_error = false;
	if (mutex_lock_interruptible(&dev_ctxt->dev_status.device_status_lock) == 0) {
		is_new_error = ((dev_ctxt->dev_status.major_error & error) == 0);
		if(is_new_error) {
			dev_ctxt->dev_status.error_reported = false;
			dev_err(&dev_ctxt->i2c_client->dev,
				"%s New driver error detected 0x%X\n",
				__func__, error);
		}
		dev_ctxt->dev_status.major_error |= error;
		mutex_unlock(&dev_ctxt->dev_status.device_status_lock);
	}

	if (is_new_error) {
		wake_up_interruptible(&dev_ctxt->wait_queue);
	}
}

static void update_fw_error(struct device_context *dev_ctxt)
{
	u32 error_reg = 0;

	int res = uctl_read_reg(dev_ctxt->i2c_client,
		UCTL_ERROR_REG,
		UCTL_32BIT,
		&error_reg);

	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev,
			"%s Failed to read FW error register with status %d\n",
			__func__, res);
	}
/*
	dev_info(&dev_ctxt->i2c_client->dev,
		"%s : error status reg = 0x%X\n",
		__func__, error_reg);
*/
	if (error_reg != 0) {
		if (mutex_lock_interruptible(&dev_ctxt->dev_status.device_status_lock) == 0) {
			dev_ctxt->dev_status.fw_error |= error_reg;
			mutex_unlock(&dev_ctxt->dev_status.device_status_lock);
		}

		set_driver_error(dev_ctxt, STM32L051_ERROR_FW_ERROR);
	}
}

static void set_state(struct device_context *dev_ctxt, enum device_state next_state)
{
	if (dev_ctxt->state != next_state) {
		dev_info(&dev_ctxt->i2c_client->dev, "%s state changed: %d ==> %d\n", __func__, dev_ctxt->state, next_state);
		dev_ctxt->state = next_state;
	}
}

static void enable_int(struct device_context *dev_ctxt, const char *caller)
{
	if (!dev_ctxt->interrupt_enabled) {
		dev_info(&dev_ctxt->i2c_client->dev, "%s enabling IRQ\n", caller);
		enable_irq(dev_ctxt->i2c_client->irq);
		dev_ctxt->interrupt_enabled = true;
	} else {
		dev_err(&dev_ctxt->i2c_client->dev, "%s interrupt was already enabled\n", caller);
	}
}
static void disable_int(struct device_context *dev_ctxt, const char *caller)
{
	if (dev_ctxt->interrupt_enabled) {
		dev_info(&dev_ctxt->i2c_client->dev, "%s disabling IRQ\n", caller);
		disable_irq(dev_ctxt->i2c_client->irq);
		dev_ctxt->interrupt_enabled = false;
	} else {
		dev_err(&dev_ctxt->i2c_client->dev, "%s interrupt was already disabled\n", caller);
	}
}

#if PRINT_SENSOR_INFO
/**
 * This function prints a sensor info.
 */
static void print_sensor_info(struct device_context *dev_ctxt, struct stm32l051_sensor_t *sensor)
{
	dev_info(&dev_ctxt->i2c_client->dev, "Name = %s\n", sensor->name);
	dev_info(&dev_ctxt->i2c_client->dev, "Type = %d\n", sensor->type);
	dev_info(&dev_ctxt->i2c_client->dev, "Handle = %d\n", sensor->handle);

	switch (sensor->type) {
	case STM32L051_SENSOR_TYPE_ACCELOROMETER:
	case STM32L051_SENSOR_TYPE_GYROSCOPE:
		dev_info(&dev_ctxt->i2c_client->dev, "Range = %d\n", sensor->
uration.vec_sensor_cfg.range);
		dev_info(&dev_ctxt->i2c_client->dev , "Bandwidth = %d\n", sensor->configuration.vec_sensor_cfg.bandwidth);
		break;
	case STM32L051_SENSOR_TYPE_DEPTH_CAMERA:
	case STM32L051_SENSOR_TYPE_MOTION_CAMERA:
		break;
	case STM32L051_SENSOR_TYPE_EXTERNAL:
		dev_info(&dev_ctxt->i2c_client->dev, "enabled = %d\n", sensor->configuration.external_sensor_cfg.enabled);
		dev_info(&dev_ctxt->i2c_client->dev, "falling_edge = %d\n", sensor->configuration.external_sensor_cfg.falling_edge);
		break;
	}
}

/**
 * This function prints a sensor data entry
 */
static void print_sensor_data(struct device_context *dev_ctxt, struct stm32l051_sensor_data *data)
{
	dev_info(&dev_ctxt->i2c_client->dev, "Src id = %d\n", data->handle);
	dev_info(&dev_ctxt->i2c_client->dev, "Frame id = %d\n", data->hw_frame_id);
	dev_info(&dev_ctxt->i2c_client->dev, "Time stamp = %lld\n", data->hw_time_stamp);
}

#endif /* PRINT_SENSOR_INFO */

/**
 * This function initializes a sensor control (common fields).
 */
static void init_sensor_ctrl(struct sensors_ctrl *sensors, struct sensor_ctrl *ctrl, __u32 handle, char *name, enum stm32l051_sensor_type type, enum sensor_interrupt_mask interrupt_mask)
{
	memset(ctrl, 0, sizeof(struct sensor_ctrl));
	strncpy(ctrl->name, name, STM32L051_SENSOR_NAME_MAX_LENGTH);
	ctrl->handle = handle;
	ctrl->type = type;
	ctrl->filter_out = false;
	ctrl->interrupt_mask = interrupt_mask;
	reset_sensor_ctrl(ctrl);
	sensors->controls[handle] = ctrl;
}

static void reset_sensor_ctrl(struct sensor_ctrl *ctrl)
{
	ctrl->was_reset = true;
	ctrl->next_frame_id = 0;
	ctrl->frame_id_multiplier = 0;
}

static void i2c_error_to_drv_error(struct device_context *dev_ctxt, int err)
{
	switch (err)
	{
	case -ETIMEDOUT:
		set_driver_error(dev_ctxt, STM32L051_ERROR_I2C_TIMEOUT);
		break;
	case -EREMOTEIO:
		set_driver_error(dev_ctxt, STM32L051_ERROR_I2C_REMOTE_IO);
		break;
	default:
		set_driver_error(dev_ctxt, STM32L051_ERROR_I2C_UNSPECIFIED_FAILURE);
		break;
	}
}

/**
 * This function reads from I2C
 */
static int uctl_i2c_read(struct i2c_client *client, u8 addr, u16 len, u8 *buf)
{
	struct i2c_msg msg[2];
	int res = 0;
	int i = 0;

	if (!client)
		return -ENODEV;

	/* dev_info(&client->dev, "==> %s reading %d bytes from address 0x%X", __func__, len, addr); */

	if (!client->adapter) {
		dev_err(&client->dev, "%s error, no adapter\n", __func__);
		return -ENODEV;
	}

	if (!buf) {
		dev_err(&client->dev, "%s error, no buf\n", __func__);
		return -ENODEV;
	}

	memset(msg, 0, sizeof(msg));

	msg[0].addr = client->addr;
	msg[0].flags = 0;
	msg[0].len = 1;
	msg[0].buf = &addr;

	msg[1].addr = client->addr;
	msg[1].len = len;
	msg[1].flags = I2C_M_RD;
	msg[1].buf = buf;

	usleep_range(10, 20);

	for (i = 0; i < i2c_read_retry_count; i++) {
		res = i2c_transfer(client->adapter, msg, ARRAY_SIZE(msg));
		if (res == ARRAY_SIZE(msg)) {
			if (i > 0)
				dev_warn(&client->dev, "%s i2c read retry cnt = %d\n", __func__, i);

			return 0;
		} else {
			dev_warn(&client->dev, "%s(): i2c_transfer returned %d\n", __func__, res);
		}
	}

	dev_err(&client->dev, "%s reading from address 0x%x failed (%d retries), error %d",
		__func__, addr, i2c_read_retry_count, res);

	if (res >= 0)
		res = -EIO;

	i2c_error_to_drv_error(i2c_get_clientdata(client), res);

	return res;
}

/**
 * This function reads a register.
 */
static int uctl_read_reg(struct i2c_client *client, u16 reg, u16 data_length, u32 *val)
{
	u8 data[4];
	int err;

	if (!client)
		return -ENODEV;

	if (!val) {
		dev_err(&client->dev, "%s error, no val\n", __func__);
		return -ENODEV;
	}

	/* read only 32 bit values */
	if (data_length != UCTL_32BIT) {
		dev_err(&client->dev, "%s error, invalid data length\n",
			__func__);
		return -EINVAL;
	}

	memset(data, 0, sizeof(data));

	err = uctl_i2c_read(client, reg, data_length, data);
	if (err)
		goto error;

#if PRINT_REGISTER_TRANSACTIONS
	dev_info(&client->dev, "%s UCTL_REG 0x%x is %x %x %x %x\n", __func__, reg, data[0], data[1], data[2], data[3]);
#endif /* PRINT_REGISTER_TRANSACTIONS */

	/* value recieved in little endian format */
	*val = 0;
	*val |= (data[3] << 24);
	*val |= (data[2] << 16);
	*val |= (data[1] << 8);
	*val |= data[0];

#if PRINT_REGISTER_TRANSACTIONS
	dev_info(&client->dev, "%s UCTL_REG is %x\n", __func__, *val);
#endif /* #if PRINT_REGISTER_TRANSACTIONS */

	return 0;

error:
	dev_err(&client->dev, "%s read from offset 0x%x error %d",
		__func__,  reg, err);
	return err;
}

static int uctl_read_reg_with_retries(struct i2c_client *client, u16 reg, u16 data_length, u32 *val, int retries, int sleep_ms)
{
	int res = 0;
	int i = 0;

	for (i = 0; i < retries; i++) {
		res = uctl_read_reg(client, reg, data_length, val);
		if (res == 0)
			return res;

		msleep(sleep_ms);
	}
	return res;
}

/**
 * This function writes to I2C
 */
static int uctl_i2c_write(struct i2c_client *client, u16 len, u8 *data)
{
	int write_msg_cnt = 0;
	struct i2c_msg msg;
	int i = 0;

	if (!client)
		return -ENODEV;

	if (!client->adapter) {
		dev_err(&client->dev, "%s error, no adapter\n", __func__);
		return -ENODEV;
	}

	if (!data) {
		dev_err(&client->dev, "%s error, no data buf\n", __func__);
		return -ENODEV;
	}

	if (len == 0) {
		dev_err(&client->dev, "%s error, len = 0\n", __func__);
		return -ENODEV;
	}

	memset(&msg, 0, sizeof(msg));

	msg.addr = client->addr;
	msg.flags = 0;
	msg.buf = data;
	msg.len = len;

	for (i = 0; i < i2c_write_retry_count; i++) {
		write_msg_cnt = i2c_transfer(client->adapter, &msg, 1);
		if (write_msg_cnt == 1) {
			if (i > 0)
				dev_warn(&client->dev, "%s i2c write retry cnt = %d\n", __func__, i);
			break;
		}
	}

	if (write_msg_cnt != 1) {
		dev_err(&client->dev, "%s i2c_transfer failed %d\n",
				__func__, write_msg_cnt);

		i2c_error_to_drv_error(i2c_get_clientdata(client), write_msg_cnt);

		return -EIO;
	}


	return 0;
}

/**
 * This function writes to a register.
 */
static int uctl_write_reg(struct i2c_client *client,
							u8 reg,
							u16 data_length,
							u32 val)
{
	int ret;
	unsigned char data[5] = {0};
	const u16 len = 1 + data_length; /* 1 byte address + data */

	if (!client)
		return -ENODEV;

	if (!client->adapter) {
		dev_err(&client->dev, "%s error, no adapter\n", __func__);
		return -ENODEV;
	}

	/* write only 32 bit values */
	if (data_length != UCTL_32BIT) {
		dev_err(&client->dev, "%s error, invalid data length\n",
			__func__);
		return -EINVAL;
	}

	/* first byte includes reg offset */
	data[0] = reg;

	/* data to be written in little endian format */
	data[1] = (val >> 0) & 0xFF;
	data[2] = (val >> 8) & 0xFF;
	data[3] = (val >> 16) & 0xFF;
	data[4] = (val >> 24) & 0xFF;

#if PRINT_REGISTER_TRANSACTIONS
	dev_info(&client->dev, "%s UCTL_REG 0x%x is %x %x %x %x\n", __func__, reg, data[1], data[2], data[3], data[4]);
#endif /* PRINT_REGISTER_TRANSACTIONS */

	ret = uctl_i2c_write(client, len, data);
	if (ret) {
		dev_err(&client->dev,
			"write error: wrote 0x%x to offset 0x%x error %d",
			val, reg, ret);
	}

	return ret;
}

/**
 *
 */
static int pmic_ctrl(struct device_context *dev_ctxt, bool power_on)
{
	int ret = 0;
	struct i2c_client *client = dev_ctxt->i2c_client;

	dev_info(&client->dev, "==> %s (power on = %d) v1p8_reg=0x%p, v3p3_reg=0x%p\n", __func__, power_on, dev_ctxt->v1p8_reg, dev_ctxt->v3p3_reg);

	if (power_on) {
		
		/* for 1.8v */
		if (regulator_is_enabled(dev_ctxt->v1p8_reg) == 0) {
			ret = regulator_enable(dev_ctxt->v1p8_reg);
			if (ret)
				dev_err(&client->dev, "enabling V1P8 failed %d\n", ret);
			dev_ctxt->regulator_was_enabled = 0;
		} else {
			dev_ctxt->regulator_was_enabled = 1;
		}
		/* for 3.3v */
		ret = regulator_enable(dev_ctxt->v3p3_reg);
		if (ret)
			dev_err(&client->dev, "enabling V3P3 failed %d\n", ret);

		msleep(SLEEP_TIME_AFTER_REGULATOR_POWER_ON_MS);
		dev_ctxt->pmic_powered_on = true;
		dev_info(&client->dev, "enable: regulator_was_enabled=0x%x\n", dev_ctxt->regulator_was_enabled);
		if (dev_ctxt->regulator_was_enabled)//we care only for the 1.8 volts to the processor. 3.3 volts failure is a dont-care
			ret = -EALREADY;
	} else {
		dev_ctxt->pmic_powered_on = false;
		dev_info(&client->dev, "disable: regulator_was_enabled=0x%x\n", dev_ctxt->regulator_was_enabled);
		if (dev_ctxt->regulator_was_enabled == 0) {
			/* for 1.8v */
			ret = regulator_disable(dev_ctxt->v1p8_reg);
			if (ret)
				dev_err(&client->dev, "disabling V1P8 failed %d\n", ret);
		}

		/* for 3.3v */
		ret = regulator_disable(dev_ctxt->v3p3_reg);
		if (ret)
			dev_err(&client->dev, "disabling V3P3 failed %d\n", ret);

		usleep_range(1000, 1100); /* TODO: define sleep range */
	}


	dev_info(&client->dev, "<== %s\n", __func__);

	return ret;
}

static int check_power_up(struct device_context *dev_ctxt)
{
	int res = 0;
	int retries = 0;
	u32 power_status_reg = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (!dev_ctxt) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s invalid dev_ctxt\n", __func__);
		return -EINVAL;
	}

	while (retries < power_up_retries) {
		res = uctl_read_reg(dev_ctxt->i2c_client,
			UCTL_REG_CURR_PWR_STATE,
			UCTL_32BIT,
			&power_status_reg);

		dev_info(&dev_ctxt->i2c_client->dev, "%s UCTL_REG_CURR_PWR_STATE is %x\n", __func__, power_status_reg);
		if (!res && (power_status_reg | UCTL_PWR_STATE_INIT)) {
			dev_info(&dev_ctxt->i2c_client->dev, "%s UCTL_REG_CURR_PWR_STATE is %x and res = %d\n", __func__, power_status_reg, res);
			break; /* uctl is in INIT state */
		}

		/* i2c read operation failed */
		if (res) {
			dev_info(&dev_ctxt->i2c_client->dev, "%s WARNING, res = %d\n", __func__, res);
			/*dev_warn();*/;
		}

		/*TODO: handle IAP state */

		retries++;
		msleep(50); /* TODO: define sleep time */
	}

	dev_info(&dev_ctxt->i2c_client->dev, "%s num retries = %d\n", __func__, retries);

	/* reached max number of retires waiting for INIT  state*/
	if (5 == retries)
		/*dev_warn();*/;

	dev_info(&dev_ctxt->i2c_client->dev, "%s returned with status %d\n", __func__, res);
	return res;
}

static int start_stream(struct device_context *dev_ctxt)
{
	int res = 0;
	u32 power_status_reg = 0;
	struct i2c_client *client;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (!dev_ctxt) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: invalid dev_ctxt\n", __func__);
		return -EINVAL;
	}

	client = dev_ctxt->i2c_client;
	kfifo_reset(&dev_ctxt->fifo);

	/* TODO: add power state to device_context and verify we are in INIT or PAUSE
	 if() {
		dev_err(&dev_ctxt->i2c_client->dev,
					"%s can't move to UCTL_PWR_STATE_ACTIVE %d\n",
					__func__, res);
		return res;
	}*/

	res = uctl_write_reg(client,
				UCTL_REG_NEXT_PWR_STATE,
				UCTL_32BIT,
				UCTL_PWR_STATE_ACTIVE);

	/*TODO: remove*/
	res = uctl_read_reg(client,
					UCTL_REG_CURR_PWR_STATE,
					UCTL_32BIT,
					&power_status_reg);

	dev_info(&dev_ctxt->i2c_client->dev, "%s UCTL_REG_CURR_PWR_STATE is %x\n", __func__, power_status_reg);

	mod_timer(&dev_ctxt->fallback_timer, jiffies + msecs_to_jiffies(INTERRUPT_FALLBACK_TIMEOUT_MS));

	set_state(dev_ctxt, DEVICE_STATE_STREAMING);

	return res;
}

static int stop_streaming(struct device_context *dev_ctxt)
{
	int res = 0;
	u32 power_status_reg = 0;
	struct i2c_client *client = dev_ctxt->i2c_client;

	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	set_state(dev_ctxt, DEVICE_STATE_STOPPING);

    dev_ctxt->uc_timestamp_polling_enabled=0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s Disabling timer\n", __func__);
	del_timer_sync(&dev_ctxt->fallback_timer);

	res = uctl_write_reg(client,
		UCTL_REG_NEXT_PWR_STATE,
		UCTL_32BIT,
		UCTL_PWR_STATE_PAUSE);
	if (res != 0) {
		dev_info(&dev_ctxt->i2c_client->dev, "<== %s failed to write to UCTL_REG_NEXT_PWR_STATE\n", __func__);
		return res;
	}

	msleep(50);

	/*TODO: remove*/
	res = uctl_read_reg(client,
		UCTL_REG_CURR_PWR_STATE,
		UCTL_32BIT,
		&power_status_reg);

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s UCTL_REG_CURR_PWR_STATE is %x\n", __func__, power_status_reg);

	return res;
}

static int pause_stream(struct device_context *dev_ctxt)
{
	int res = 0;
	struct i2c_client *client;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (!dev_ctxt) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: invalid dev_ctxt\n", __func__);
		return -EINVAL;
	}

	client = dev_ctxt->i2c_client;

	if (dev_ctxt->state == DEVICE_STATE_STREAMING) {
		res = stop_streaming(dev_ctxt);
	}

	else if (dev_ctxt->state != DEVICE_STATE_STOPPED) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Can't switch to pause from state %d\n", __func__, dev_ctxt->state);
		return -ERESTARTSYS;
	}

	set_state(dev_ctxt, DEVICE_STATE_PAUSED);

	return res;
}

static int stop_stream(struct device_context *dev_ctxt)
{
	int res = 0;
	struct i2c_client *client;
	u32 power_status_reg = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	if (!dev_ctxt) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: invalid dev_ctxt\n", __func__);
		return -EINVAL;
	}

	client = dev_ctxt->i2c_client;

	if (dev_ctxt->state == DEVICE_STATE_STREAMING)
		res = stop_streaming(dev_ctxt);

	res = uctl_write_reg(client,
		UCTL_REG_NEXT_PWR_STATE,
		UCTL_32BIT,
		UCTL_PWR_STATE_INIT);
	if (res != 0) {
		dev_info(&dev_ctxt->i2c_client->dev, "%s failed to write to UCTL_REG_NEXT_PWR_STATE\n", __func__);
	}

	reset_sensors(dev_ctxt);

	res = uctl_read_reg(client,
		UCTL_REG_CURR_PWR_STATE,
		UCTL_32BIT,
		&power_status_reg);

	dev_info(&dev_ctxt->i2c_client->dev, "%s UCTL_REG_CURR_PWR_STATE is %x\n", __func__, power_status_reg);

	set_state(dev_ctxt, DEVICE_STATE_STOPPED);

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);

	return res;
}


#if 0
/*TODO: add documentaion, assuming size is pow(2) */
static int cnt_to_end(int head, int tail, int size)
{
	int end = (size) - (tail);
	int n = ((head) + end) & ((size)-1);
	return n < end ? n : end;
}
#endif

/**
 * File operations
 */

/**
 * Read file operation
 */
static ssize_t stm32l051_read(struct file *filp, char __user *buff, size_t len, loff_t *offset)
{
	int res = 0;
	unsigned int copied = 0;	
	struct device_context *dev_ctxt;

	if (!filp)
		return -ENODEV;

	if (!filp->private_data)
		return -ENODEV;

	dev_ctxt = filp->private_data;

	/* Align the length to size of sensor data. */
	len = ((len / sizeof(struct stm32l051_sensor_data)) * sizeof(struct stm32l051_sensor_data));

	/*dev_info(&dev_ctxt->i2c_client->dev, "Entering %s, filp->f_flags = 0x%X, O_NONBLOCK = 0x%X\n", __func__, filp->f_flags, O_NONBLOCK);*/

	if (kfifo_is_empty(&dev_ctxt->fifo)) {
		if (filp->f_flags & O_NONBLOCK) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s: No entries, and in non blocking mode\n", __func__);
			return -EAGAIN;
		} else {
			int wait_result = 0;
			/*dev_info(&dev_ctxt->i2c_client->dev, "%s Blocking mode waiting until FIFO has something\n", __func__);*/
			wait_result = wait_event_interruptible_timeout(
				dev_ctxt->block_read_wait_queue,
				(!kfifo_is_empty(&dev_ctxt->fifo)),
				jiffies + msecs_to_jiffies(1000));

			/*dev_info(&dev_ctxt->i2c_client->dev, "%s wait for fifo ended, num entries = %d\n", __func__, kfifo_len(&dev_ctxt->fifo));*/

			if (kfifo_is_empty(&dev_ctxt->fifo)) {
				dev_err(&dev_ctxt->i2c_client->dev, "%s: Num entries is still 0\n", __func__);
				return -ERESTARTSYS;
			}
		}
	}
	/*else {
		dev_info(&dev_ctxt->i2c_client->dev, "%s FIFO is not empty, num entries = %d\n", __func__, kfifo_len(&dev_ctxt->fifo));
	}*/

	res = kfifo_to_user(&dev_ctxt->fifo, buff, len, &copied);
	/*dev_info(&dev_ctxt->i2c_client->dev, "%s res = %d, copied = %d\n", __func__, res, copied);*/

	return copied;
}

#define FW_IMAGE_PACKET_PAYLOAD_LEN (128)

struct fw_image_packet {
	u8 op_code;
	__be32 address;
	__be16 length;
	u8	dummy;
	u8	data[FW_IMAGE_PACKET_PAYLOAD_LEN];
}__attribute__((packed, aligned(1)));

#define FW_UPDATE_SWITCH_TO_IAP_RETRY_CNT 3

static int switch_to_iap(struct device_context *dev_ctxt)
{
	int retries = 0;
	u32 reg_val = 0;
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	/* 1. Check if the device is already in IAP. */
	reg_val = 0;
	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_CURR_PWR_STATE, UCTL_32BIT, &reg_val);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s failed to get curr power state register\n", __func__);
		return res;
	}

	/*disable_int(dev_ctxt, __func__);*/

	if (reg_val != UCTL_PWR_STATE_IAP) {
		/* 1. Switch to FW Update */
		res = uctl_write_reg(dev_ctxt->i2c_client,
				UCTL_REG_IAP_REG, UCTL_32BIT, 0xAE);
		if (res != 0) {
			dev_err(&dev_ctxt->i2c_client->dev,
					"%s Failed to insert device to IAP mode\n", __func__);
			return res;
		}

		/* 2. Wait until device is ready. */
		reg_val = 0;
		for (retries = 0; retries < FW_UPDATE_SWITCH_TO_IAP_RETRY_CNT; retries++) {
			msleep(50);
			res = uctl_read_reg(dev_ctxt->i2c_client,
					UCTL_REG_CURR_PWR_STATE, UCTL_32BIT, &reg_val);
			if ((res == 0) && (reg_val == UCTL_PWR_STATE_IAP)) {
				dev_info(&dev_ctxt->i2c_client->dev,
						"%s device inserted to IAP mode after %d retries\n",
						__func__, retries);
				break;
			}
		}
	}

	if (reg_val != UCTL_PWR_STATE_IAP) {
		dev_err(&dev_ctxt->i2c_client->dev,
			"%s Wait for device ready for FW update exceeded %d retries\n",
			__func__, FW_UPDATE_SWITCH_TO_IAP_RETRY_CNT);
		return -ENODEV;
	}

	set_state(dev_ctxt, DEVICE_STATE_IAP);

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);

	return 0;
}

static int switch_to_operational(struct device_context *dev_ctxt)
{
	u32 reg_val = 0;
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	/* Reset FW, register value is not important, just touch the register. */
	res = uctl_write_reg(dev_ctxt->i2c_client,
			UCTL_JUMP_TO_APP_REG, UCTL_32BIT, 0x00);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev,
				"%s Failed to reset FW after update\n", __func__);
		return res;
	}

	reg_val = 0;
	res = uctl_read_reg_with_retries(dev_ctxt->i2c_client,
			UCTL_REG_CURR_PWR_STATE, UCTL_32BIT, &reg_val, 10, 100);

	if (res != 0) {
		dev_info(&dev_ctxt->i2c_client->dev,
				"%s failed to jump back to operational\n", __func__);
		set_state(dev_ctxt, DEVICE_STATE_FW_CORRUPTED);
		return res;
	}

	/* Verify that FW is not in IAP any more. */
	if (reg_val == UCTL_PWR_STATE_IAP) {
		dev_err(&dev_ctxt->i2c_client->dev,
			"%s Firmware is stuck in IAP\n",
			__func__);
		set_driver_error(dev_ctxt, STM32L051_ERROR_FW_UPDATE_FAILED_FW_CORRUPTED);
		set_state(dev_ctxt, DEVICE_STATE_FW_CORRUPTED);
		return -ENODEV;
	}

	dev_info(&dev_ctxt->i2c_client->dev,
			"%s device jumped back to operational (power reg = 0x%X)\n",
			__func__, reg_val);

	return 0;
}

/**
 * Write
 */
static ssize_t stm32l051_write(struct file *filp, const char __user *buff, size_t count, loff_t *f_pos)
{
	struct device_context *dev_ctxt;

	if (!filp)
		return -ENODEV;

	if (!filp->private_data)
		return -ENODEV;

	dev_ctxt = filp->private_data;

	dev_err(&dev_ctxt->i2c_client->dev, "%s is not supported anymore\n", __func__);

	return 0;
}

/**
 * This function adds a sensor info to sensors list (used in get sensors list IOCTL).
 */
static char __user *add_sensor_info_to_sensors_list(struct device_context *dev_ctxt, char __user *buf, struct sensor_ctrl *sensor)
{
	struct stm32l051_sensor_t info = {0};
	fill_common_sensor_info(dev_ctxt, &info, sensor);
	if (copy_to_user(buf, &info, sizeof(struct stm32l051_sensor_t)) != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to copy data to sensors list\n", __func__);
		return NULL;
	}
	buf += sizeof(struct stm32l051_sensor_t);

	return buf;
}

/**
 * ACCELEROMETERS
 */

/**
 * This function initializes the accelerometers.
 */
static int init_accelerometer(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle)
{
	init_sensor_ctrl(&dev_ctxt->sensors,
		&dev_ctxt->sensors.accelerometer,
		(*handle),
		"Accelerometer",
		STM32L051_SENSOR_TYPE_ACCELOROMETER,
		SENSOR_INT_MASK_ACC);
	id_config_reg->bit_fields.acc_id = (*handle);
	(*handle)++;
	return 0;
}

/**
 * GYROSCOPES
 */

/**
 * This function initializes the gyroscopes.
 */
static int init_gyroscope(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle)
{
	init_sensor_ctrl(&dev_ctxt->sensors,
		&dev_ctxt->sensors.gyroscope,
		(*handle),
		"Gyroscope",
		STM32L051_SENSOR_TYPE_GYROSCOPE,
		SENSOR_INT_MASK_GYRO);
	id_config_reg->bit_fields.gyro_id = (*handle);

	(*handle)++;
	return 0;
}

/**
 * DEPTH CAMERAS
 */

/**
 * This function initializes the depth cameras.
 */
static int init_depth_camera(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle)
{
	init_sensor_ctrl(&dev_ctxt->sensors,
		&dev_ctxt->sensors.depth_camera,
		(*handle),
		"Depth Camera",
		STM32L051_SENSOR_TYPE_DEPTH_CAMERA,
		SENSOR_INT_MASK_DEPTH);
	id_config_reg->bit_fields.depth_id = (*handle);
	(*handle)++;

	return 0;
}

/**
 * This function initializes the motion cameras.
 */
static int init_motion_camera(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle)
{
	init_sensor_ctrl(&dev_ctxt->sensors,
		&dev_ctxt->sensors.motion_camera,
		(*handle),
		"Motion Camera",
		STM32L051_SENSOR_TYPE_MOTION_CAMERA,
		SENSOR_INT_MASK_MOTION);
	id_config_reg->bit_fields.motion_id = (*handle);
	(*handle)++;
	return 0;
}

/**
 * This function initializes the external sensors.
 */
static int init_external_sensors(struct device_context *dev_ctxt, union id_config_register *id_config_reg, int *handle)
{
	struct external_sensor_ctrl *sensor = NULL;
	struct external_sensors_ctrl *external_sensors = &dev_ctxt->sensors.external_sensors;
	external_sensors->handle_root = (*handle);

	sensor = &external_sensors->sensors[0];
	init_sensor_ctrl(&dev_ctxt->sensors,
		&sensor->common,
		(*handle),
		"External Sensor 0",
		STM32L051_SENSOR_TYPE_EXTERNAL,
		SENSOR_INT_MASK_G0_VSYNC);
	sensor->index = 0;
	id_config_reg->bit_fields.G0Vsync_id = (*handle);
	(*handle)++;

	sensor = &external_sensors->sensors[1];
	init_sensor_ctrl(&dev_ctxt->sensors,
		&sensor->common,
		(*handle),
		"External Sensor 1",
		STM32L051_SENSOR_TYPE_EXTERNAL,
		SENSOR_INT_MASK_G1_VSYNC);
	sensor->index = 1;
	id_config_reg->bit_fields.G1Vsync_id = (*handle);
	(*handle)++;

	return 0;
}

/**
 * This function initializes the sensors.
 */
static int init_sensors(struct device_context *dev_ctxt)
{
	int handle_root = 0;
	union id_config_register id_config_reg = {.value = 0};

	if (init_accelerometer(dev_ctxt, &id_config_reg, &handle_root) != 0)
		return -ENODATA;
	if (init_gyroscope(dev_ctxt, &id_config_reg, &handle_root) != 0)
		return -ENODATA;
	if (init_depth_camera(dev_ctxt, &id_config_reg, &handle_root) != 0)
		return -ENODATA;
	if (init_motion_camera(dev_ctxt, &id_config_reg, &handle_root) != 0)
		return -ENODATA;
	if (init_external_sensors(dev_ctxt, &id_config_reg, &handle_root) != 0)
		return -ENODATA;

	return uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_INT_ID_CONFIG, UCTL_32BIT, id_config_reg.value);
}

static void reset_sensors(struct device_context *dev_ctxt)
{
	struct sensors_ctrl *sensors = &dev_ctxt->sensors;
	reset_sensor_ctrl(&sensors->accelerometer);
	reset_sensor_ctrl(&sensors->gyroscope);
	reset_sensor_ctrl(&sensors->depth_camera);
	reset_sensor_ctrl(&sensors->motion_camera);
	reset_sensor_ctrl(&sensors->external_sensors.sensors[0].common);
	reset_sensor_ctrl(&sensors->external_sensors.sensors[1].common);
}

static void fallback_timer_handler(unsigned long context)
{
	int wake_thread_status = 0;
	struct device_context *dev_ctxt = (struct device_context *)context;
	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	update_time_delta(&dev_ctxt->timing_measurements.fallback_timer_time, &dev_ctxt->statistics.fallback_timer_delta);

	if (dev_ctxt->process_hw_events_thread != NULL)
		wake_thread_status = wake_up_process(dev_ctxt->process_hw_events_thread);

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s (wake status = 0x%X)\n", __func__, wake_thread_status);
}

static int process_hw_events_thread_handler(void *data)
{
	struct device_context *dev_ctxt = (struct device_context *)data;
	struct i2c_client *client = dev_ctxt->i2c_client;

	dev_info(&client->dev, "==> %s\n", __func__);

	while (!kthread_should_stop()) {

		set_current_state(TASK_INTERRUPTIBLE);
		schedule();

		if(dev_ctxt->state == DEVICE_STATE_STREAMING) {
			int res = 0;
			dev_warn(&client->dev, "%s fallback time thread running\n", __func__);

			res = process_hw_events(dev_ctxt);
			if(res != 0)
				dev_err(&client->dev, "%s failed to process status register\n", __func__);

			/* Re triggering the fallback timer only when streaming*/
			mod_timer(&dev_ctxt->fallback_timer, jiffies + msecs_to_jiffies(INTERRUPT_FALLBACK_TIMEOUT_MS));
		}
	}

	dev_info(&client->dev, "<== %s\n", __func__);

	return 0;
}
/**
 * IOCTLS
 */

/**
 * This function implements STM32L051_IOCTL_INIT
 */
static int init_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	int res = 0;
	u32 reg_val = 0;

	enum stm32l051_init_cmd_status status = STM32L051_INIT_STATUS_SUCCESS;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state == DEVICE_STATE_FW_CORRUPTED) || (dev_ctxt->state == DEVICE_STATE_IAP)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s FW corrupted (state = %d)\n", __func__, dev_ctxt->state);
		res = -EPERM;
		status = STM32L051_INIT_STATUS_FW_CORRUPTED;
		goto init_cmd_error;
	}

	if (dev_ctxt->state != DEVICE_STATE_UNINITIALIZED) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s IMU is already initialized (state = %d)\n", __func__, dev_ctxt->state);
		res = -EPERM;
		status = STM32L051_INIT_STATUS_DEVICE_ALREADY_INITIALIZED;
		goto init_cmd_error;
	}

	res = power_up(dev_ctxt);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s failed to power up the device\n", __func__);
		status = STM32L051_INIT_STATUS_FAILED_TO_POWER_UP;
		goto init_cmd_error;
		return res;
	}

	/* Check if FW corrupted. */
	res = uctl_read_reg_with_retries(dev_ctxt->i2c_client, UCTL_REG_CURR_PWR_STATE, UCTL_32BIT, &reg_val, 10, 20);

	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s failed to get curr power state register\n", __func__);
		status = STM32L051_INIT_STATUS_FAILED_TO_POWER_UP;
		goto init_cmd_error;
	}

	if (reg_val == UCTL_PWR_STATE_IAP) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s FW corrupted (in IAP state)\n", __func__);
		status = STM32L051_INIT_STATUS_FW_CORRUPTED;
		set_state(dev_ctxt, DEVICE_STATE_FW_CORRUPTED);
		/* Will not fail the command, but the status will be FW corrupted. */
		res = 0;
		goto init_cmd_error;
	}

	res = check_power_up(dev_ctxt);
	if (res != 0) {
		status = STM32L051_INIT_STATUS_FAILED_TO_POWER_UP;
		goto init_cmd_error;
	}

	res = init_sensors(dev_ctxt);
	if (res != 0) {
		status = STM32L051_INIT_STATUS_INTERNAL_ERROR;
		goto init_cmd_error;
	}

	res = query_timer_prescaler(dev_ctxt);
	if (res != 0) {
		status = STM32L051_INIT_STATUS_INTERNAL_ERROR;
		goto init_cmd_error;
	}

	set_state(dev_ctxt, DEVICE_STATE_STOPPED);

init_cmd_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);
	if (copy_to_user((char __user *) arg, &status, sizeof(status)) != 0)
		dev_info(&dev_ctxt->i2c_client->dev, "%s Failed to copy status to user\n", __func__);

	dev_info(&dev_ctxt->i2c_client->dev, "%s returned with status %d\n", __func__, res);

	return res;
}

/**
 *
 */
static int uninit(struct device_context *dev_ctxt)
{
	int res = 0;
	switch (dev_ctxt->state) {
	case DEVICE_STATE_UNINITIALIZED:
		dev_err(&dev_ctxt->i2c_client->dev, "%s IMU is not initialized\n", __func__);
		return -EPERM;
	case DEVICE_STATE_STREAMING:
	case DEVICE_STATE_PAUSED:
		res = stop_stream(dev_ctxt);
		if (res != 0)
			dev_err(&dev_ctxt->i2c_client->dev, "%s: Failed to stop stream\n", __func__);
		break;
	case DEVICE_STATE_STOPPING:
		dev_err(&dev_ctxt->i2c_client->dev, "%s: Uninit during state %d is forbidden\n", __func__, dev_ctxt->state);
		return -ERESTARTSYS;
	default:
		break;
	}

	res = power_down(dev_ctxt);

	set_state(dev_ctxt, DEVICE_STATE_UNINITIALIZED);

	return 0;
}
/**
 * This function implements STM32L051_IOCTL_UNINIT
 */
static int uninit_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	res = uninit(dev_ctxt);

	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

/**
 * This function implements STM32L051_IOCTL_GET_DEVICE_INFO
 */
static int get_device_info_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	int res = 0;
	struct stm32l051_device_info_cmd device_info = {0};
	u32 reg_val = 0;
	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if (dev_ctxt->state & (DEVICE_STATE_CLOSED | DEVICE_STATE_UNINITIALIZED | DEVICE_STATE_IAP)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto get_device_info_error;
	}

	/* Hard Coded fields */
	device_info.driver_version = stm32l051_driver_version;
	device_info.num_sensors = NUM_SENSORS;

	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_IMG_VER, UCTL_32BIT, &device_info.fw_version);
	if (res != 0)
		goto get_device_info_error;

	res = query_timer_prescaler(dev_ctxt);
	if (res != 0)
		goto get_device_info_error;

	device_info.timestamp_prescaler = dev_ctxt->timer_prescaler;

	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_TIMER_RESET_VALUE, UCTL_32BIT, &device_info.timestamp_reset_value);
	if (res != 0)
		goto get_device_info_error;

	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_REQ_LTR, UCTL_32BIT, &reg_val);
	if (res != 0)
		goto get_device_info_error;

	device_info.ts_fifo_length = (reg_val & FIFO_LENGTH_MASK);
	device_info.imu_fifo_length = ((reg_val >> IMU_FIFO_SHIFT) & FIFO_LENGTH_MASK);

	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_UCTRL_CFG, UCTL_32BIT, &reg_val);
	if (res != 0)
		goto get_device_info_error;

	/* HW Sync is a bit in the UCTL_REG_UCTRL_CFG. */
	if (reg_val & HW_SYNC_ENABLED)
		device_info.hw_sync_enabled = 1; /* device_info is initialized to 0, so implicit else = 0 */

	switch (dev_ctxt->state) {
	case DEVICE_STATE_PAUSED:
		device_info.state = STM32L051_PAUSE;
		break;
	case DEVICE_STATE_STREAMING:
		device_info.state = STM32L051_START;
		break;
	case DEVICE_STATE_CLOSED:
	case DEVICE_STATE_UNINITIALIZED:
	case DEVICE_STATE_STOPPED:
	default:
		device_info.state = STM32L051_STOP;
		break;
	}

	res = copy_to_user((char __user *) arg, &device_info, sizeof(device_info));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to write device info with status %d\n", __func__, res);
		goto get_device_info_error;
	}

get_device_info_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	dev_info(&dev_ctxt->i2c_client->dev, "%s returned with status %d\n", __func__, res);
	return res;
}


/**
 * This function implements STM32L051_IOCTL_GET_DEVICE_INFO
 */
static int get_ts(struct device_context *dev_ctxt, unsigned long arg)
{
	int res = 0;
//	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if (dev_ctxt->state & (DEVICE_STATE_CLOSED | DEVICE_STATE_UNINITIALIZED | DEVICE_STATE_IAP)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto get_device_info_error;
	}

    dev_ctxt->uc_timestamp_polling_enabled=1;
	res = copy_to_user((u64 __user *) arg, &dev_ctxt->last_uc_timestamp_value, sizeof(dev_ctxt->last_uc_timestamp_value));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to write device info with status %d\n", __func__, res);
		goto get_device_info_error;
	}

get_device_info_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

static void fill_common_sensor_info(struct device_context *dev_ctxt, struct stm32l051_sensor_t *info, struct sensor_ctrl *ctrl)
{
	strncpy(info->name, ctrl->name, STM32L051_SENSOR_NAME_MAX_LENGTH);
	info->handle = ctrl->handle;
	info->type = ctrl->type;
	info->enabled = is_sensor_interrupt_enabled(dev_ctxt, ctrl->interrupt_mask);
}

/**
 * This function queries the accelerometer configuration
 * and adds it's data to the info list.
 */
static char __user *add_accl_info_to_sensors_list(char __user *buf, struct device_context *dev_ctxt)
{
	struct stm32l051_sensor_t info = {0};
	struct stm32l051_vec_sensor_cfg *cfg = &info.configuration.vec_sensor_cfg;
	u32 reg_val = 0;
	int err = 0;

	fill_common_sensor_info(dev_ctxt, &info, &dev_ctxt->sensors.accelerometer);

	err = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_BW, UCTL_32BIT, &reg_val);
	if (err != 0)
		return NULL;

	reg_val &= 0x1F;

	switch (reg_val) {
	case ACC_BW_125_HZ:
		cfg->bandwidth = STM32L051_ACC_BW_125_HZ;
		break;
	case ACC_BW_250_HZ:
		cfg->bandwidth = STM32L051_ACC_BW_250_HZ;
		break;
	case ACC_BW_500_HZ:
		cfg->bandwidth = STM32L051_ACC_BW_500_HZ;
		break;
	case ACC_BW_1000_HZ:
		cfg->bandwidth = STM32L051_ACC_BW_1000_HZ;
		break;
	case ACC_BW_2000_HZ:
		cfg->bandwidth = STM32L051_ACC_BW_2000_HZ;
		break;

	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s ERROR, invalid bandwidth 0x%x\n", __func__, reg_val);
		cfg->bandwidth = STM32L051_ACC_BW_INVALID;
		break;
	}

	reg_val = 0;
	err = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_RANGE, UCTL_32BIT, &reg_val);
	if (err != 0)
		return NULL;

	switch (reg_val) {
	case ACC_RANGE_4:
		cfg->range = STM32L051_ACC_RANGE_4;
		break;
	case ACC_RANGE_8:
		cfg->range = STM32L051_ACC_RANGE_8;
		break;
	case ACC_RANGE_16:
		cfg->range = STM32L051_ACC_RANGE_16;
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s ERROR, invalid range 0x%x\n", __func__, reg_val);
		cfg->range = STM32L051_ACC_RANGE_INVALID;
		break;
	}

	if (copy_to_user(buf, &info, sizeof(struct stm32l051_sensor_t)) != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to copy data to sensors list\n", __func__);
		return NULL;
	}
	buf += sizeof(struct stm32l051_sensor_t);
	return buf;
}

/**
 * This function queries the gyroscope configuration
 * and adds it's data to the info list.
 */


static char __user *add_gyro_info_to_sensors_list(char __user *buf, struct device_context *dev_ctxt)
{
	struct stm32l051_sensor_t info = {0};
	struct stm32l051_vec_sensor_cfg *cfg = &info.configuration.vec_sensor_cfg;
	u32 reg_val = 0;
	int err = 0;

	fill_common_sensor_info(dev_ctxt, &info, &dev_ctxt->sensors.gyroscope);

	err = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_BW, UCTL_32BIT, &reg_val);
	if (err != 0)
		return NULL;

	switch (reg_val) {
	case GYRO_BW_100_HZ:
		cfg->bandwidth = STM32L051_GYRO_BW_100_HZ;
		break;
	case GYRO_BW_200_HZ:
		cfg->bandwidth = STM32L051_GYRO_BW_200_HZ;
		break;
	case GYRO_BW_400_HZ:
		cfg->bandwidth = STM32L051_GYRO_BW_400_HZ;
		break;
	case GYRO_BW_1000_HZ:
		cfg->bandwidth = STM32L051_GYRO_BW_1000_HZ;
		break;
	case GYRO_BW_2000_HZ:
		cfg->bandwidth = STM32L051_GYRO_BW_2000_HZ;
		break;
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s ERROR, invalid bandwidth 0x%x\n", __func__, reg_val);
		cfg->bandwidth = STM32L051_GYRO_BW_INVALID;
		break;
	}

	reg_val = 0;
	err = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_RANGE, UCTL_32BIT, &reg_val);
	if (err != 0)
		return NULL;

	switch (reg_val) {
	case GYRO_RANGE_1000:
		cfg->range = STM32L051_GYRO_RANGE_1000;
		break;
	case GYRO_RANGE_2000:
		cfg->range = STM32L051_GYRO_RANGE_2000;
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s ERROR, invalid range 0x%x\n", __func__, reg_val);
		cfg->range = STM32L051_GYRO_RANGE_INVALID;
		break;
	}

	if (copy_to_user(buf, &info, sizeof(struct stm32l051_sensor_t)) != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to copy data to sensors list\n", __func__);
		return NULL;
	}
	buf += sizeof(struct stm32l051_sensor_t);

	return buf;
}

static const u32 g_ext_sensor_falling_edge_mask[2] = {G0_FALLING_EDGE_MSK, G1_FALLING_EDGE_MSK};

static int get_external_sensor_cfg(struct device_context *dev_ctxt, struct external_sensor_ctrl *sensor, struct stm32l051_external_sensor_cfg *cfg)
{
	u32 uctrl_reg = 0;
	u32 int_type_config_reg = 0;
	u32 index = sensor->index;
	int res = 0;
	struct vsync_info info;

	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_UCTRL_CFG, UCTL_32BIT, &uctrl_reg);
	if (res != 0)
		return res;

	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_INT_TYPE_CONFIG, UCTL_32BIT, &int_type_config_reg);
	if (res != 0)
		return res;

	reg_to_vsync_info(dev_ctxt, &info, uctrl_reg);

	cfg->falling_edge = ((int_type_config_reg & g_ext_sensor_falling_edge_mask[index]) != 0) ? 1 : 0;

	return res;
}

static void reg_to_vsync_info(struct device_context *dev_ctxt, struct vsync_info *info, u32 uctrl_cfg_reg)
{
	u32 masked_val = (uctrl_cfg_reg & 0x7);

	/* Updated using SAS 0_57*/
	/* Starting with default value, NA. Notice that the switch case below will only set the non "NA" fields. */
	info->g0_mode = vsync_mode_na;
	info->g1_mode = vsync_mode_na;
	info->g2_mode = vsync_mode_na;
	
	
	switch (masked_val) {
	case 0:
		break;
	case 1:
		info->g0_mode = vsync_mode_timestamp;
		break;
	case 2:
		info->g0_mode = vsync_mode_timestamp;
		info->g1_mode = vsync_mode_timestamp;
		break;
	case 3:
		info->g0_mode = vsync_mode_timestamp;
		info->g1_mode = vsync_mode_timestamp;
		info->g2_mode = vsync_mode_timestamp;
		break;
	case 4:
		info->g2_mode = vsync_mode_interrupt;
		break;
	case 5:
		info->g0_mode = vsync_mode_timestamp;
		info->g2_mode = vsync_mode_interrupt;
		break;
	case 6:
		info->g0_mode = vsync_mode_timestamp;
		info->g1_mode = vsync_mode_timestamp;
		info->g2_mode = vsync_mode_interrupt;
		break;
	case 7:
		info->g1_mode = vsync_mode_uart_tx;
		info->g2_mode = vsync_mode_uart_rx;
		break;
	default:
		break;
	}
}

#define CHECK_VSYNC_INFO(_info, _reg, _g0, _g1, _g2, _alt) \
	do { \
		if ((_info->g0_mode == _g0) && (_info->g1_mode == _g1) && (_info->g2_mode == _g2)) { \
			*_reg |= _alt; \
			return 0; \
		}; \
	} while (0)

static int vsync_info_to_reg(struct device_context *dev_ctxt, struct vsync_info *info, u32 *reg)
{
	/* This method will only convert known configurations (updated according to SAS version 0_57). */
	*reg &= (~0x7);
	/* ALT 0 */
	CHECK_VSYNC_INFO(info, reg, vsync_mode_na, vsync_mode_na, vsync_mode_na, 0);

	/* ALT 1 */
	CHECK_VSYNC_INFO(info, reg, vsync_mode_timestamp, vsync_mode_na, vsync_mode_na, 1);

	/* ALT 2 */
	CHECK_VSYNC_INFO(info, reg, vsync_mode_timestamp, vsync_mode_timestamp, vsync_mode_na, 2);

	/* ALT 3 */
	CHECK_VSYNC_INFO(info, reg, vsync_mode_timestamp, vsync_mode_timestamp, vsync_mode_timestamp, 3);

	/* ALT 4 */
	CHECK_VSYNC_INFO(info, reg, vsync_mode_na, vsync_mode_na, vsync_mode_interrupt, 4);

	/* ALT 5 */
	CHECK_VSYNC_INFO(info, reg, vsync_mode_timestamp, vsync_mode_na, vsync_mode_interrupt, 5);

	/* ALT 6 */
	CHECK_VSYNC_INFO(info, reg, vsync_mode_timestamp, vsync_mode_timestamp, vsync_mode_interrupt, 6);

	/* ALT 7 */
	CHECK_VSYNC_INFO(info, reg, vsync_mode_na, vsync_mode_uart_tx, vsync_mode_uart_rx, 7);

	dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to convert vsync info (g0=%d, g1=%d, g2=%d) to reg value\n", __func__,
		info->g0_mode, info->g1_mode, info->g2_mode);

	return -ERESTARTSYS;
}

/**
 * This function enables / disables a sensor interrupt in the FW.
 */
static int config_sensor_interrupt(struct device_context *dev_ctxt, enum sensor_interrupt_mask mask, bool enable)
{
	u32 interrupts_mask = 0x0;

	int res = uctl_read_reg(dev_ctxt->i2c_client,
		UCTL_REG_INT_ENABLE,
		UCTL_32BIT,
		&interrupts_mask);

	if (res != 0)
		return res;

	dev_info(&dev_ctxt->i2c_client->dev,
		"%s current interrupts mask is 0x%x, sensor_int_mask = 0x%X, enable = %d\n",
		__func__, interrupts_mask, mask, enable);

	if (enable)
		interrupts_mask |= mask;
	else
		interrupts_mask &= ~mask;

	dev_info(&dev_ctxt->i2c_client->dev,
			"%s new interrupts mask is 0x%x\n", __func__, interrupts_mask);

	res = uctl_write_reg(dev_ctxt->i2c_client,
			UCTL_REG_INT_ENABLE,
			UCTL_32BIT,
			interrupts_mask);

	return res;
}

static bool is_sensor_interrupt_enabled(struct device_context *dev_ctxt, enum sensor_interrupt_mask mask)
{
	u32 interrupts_mask = 0x0;

	int res = uctl_read_reg(dev_ctxt->i2c_client,
		UCTL_REG_INT_ENABLE,
		UCTL_32BIT,
		&interrupts_mask);

	if (res != 0)
		return false;

	dev_info(&dev_ctxt->i2c_client->dev,
		"%s current interrupts mask is 0x%x, sensor_int_mask = 0x%X\n",
		__func__, interrupts_mask, mask);

	if (interrupts_mask & mask)
		return true;
	return false;
}

static int set_external_sensor_cfg(struct device_context *dev_ctxt, struct external_sensor_ctrl *sensor, struct stm32l051_external_sensor_cfg_cmd *cmd)
{
	u32 uctrl_reg = 0;
	u32 int_type_config_reg = 0;
	u32 index = sensor->index;
	int res = 0;
	struct vsync_info info;

	/* Config enabled */
	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_UCTRL_CFG, UCTL_32BIT, &uctrl_reg);
	if (res != 0)
		return res;

	reg_to_vsync_info(dev_ctxt, &info, uctrl_reg);

	if (index == 0) {
		info.g0_mode = (cmd->enabled) ? vsync_mode_timestamp : vsync_mode_na;
	}
	else if (index == 1) {
		info.g1_mode = (cmd->enabled) ? vsync_mode_timestamp : vsync_mode_na;
	}
	else
		return -ERESTARTSYS;
	
	res = config_sensor_interrupt(dev_ctxt, sensor->common.interrupt_mask, cmd->enabled);

	if (res != 0)
		return res;

	res = vsync_info_to_reg(dev_ctxt, &info, &uctrl_reg);
	
	
	if (res != 0)
		return res;

	res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_UCTRL_CFG, UCTL_32BIT, uctrl_reg);
	if (res != 0)
		return res;

	/* Config falling edge */
	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_INT_TYPE_CONFIG, UCTL_32BIT, &int_type_config_reg);
	if (res != 0)
		return res;

	if (cmd->configuration.falling_edge)
		int_type_config_reg |= g_ext_sensor_falling_edge_mask[index];
	else
		int_type_config_reg &= (~g_ext_sensor_falling_edge_mask[index]);

	res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_INT_TYPE_CONFIG, UCTL_32BIT, int_type_config_reg);
	if (res != 0)
		return res;

	return res;
}

static char __user *add_external_sensors_info_to_sensors_list(char __user *buf, struct device_context *dev_ctxt)
{
	int i = 0;
	for (i = 0; i < NUM_EXTERNAL_SENSORS; i++) {
		struct stm32l051_sensor_t info = {0};

		fill_common_sensor_info(dev_ctxt, &info, &dev_ctxt->sensors.external_sensors.sensors[i].common);
		if (get_external_sensor_cfg(dev_ctxt, &dev_ctxt->sensors.external_sensors.sensors[i], &info.configuration.external_sensor_cfg) != 0) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get sensor info\n", __func__);
			return NULL;
		}

		if (copy_to_user(buf, &info, sizeof(struct stm32l051_sensor_t)) != 0) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to copy data to sensors list\n", __func__);
			return NULL;
		}
		buf += sizeof(struct stm32l051_sensor_t);
	}

	return buf;
}

/**
 * This function implements STM32L051_IOCTL_GET_SENSORS_LIST
 * This method refers to struct stm32l051_sensors_list_cmd
 */
static int get_sensors_list_cmd(struct device_context *dev_ctxt, unsigned long arg)
{ 
	int num_sensors = 0;
	int expected_num_sensors = NUM_SENSORS;
	char __user *buf = (char __user *) arg;
	struct stm32l051_sensor_t __user *list = NULL;
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if (dev_ctxt->state & (DEVICE_STATE_CLOSED | DEVICE_STATE_UNINITIALIZED | DEVICE_STATE_IAP)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto get_sensors_list_error;
	}


	res = copy_from_user(&num_sensors, (const void __user *)arg, sizeof(num_sensors));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s failed to get num sensors from buffer with resor %d\n", __func__, res);
		goto get_sensors_list_error;
	}

	res = copy_to_user(buf, &expected_num_sensors, sizeof(expected_num_sensors));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s failed to set num sensors to buffer with resor %d\n", __func__, res);
		goto get_sensors_list_error;
	}
	buf += sizeof(expected_num_sensors);
	list = (struct stm32l051_sensor_t __user *)buf;

	if (num_sensors != NUM_SENSORS) {
		dev_err(&dev_ctxt->i2c_client->dev, "num sensors (%d) != num sensors in the device (%d)\n", num_sensors, expected_num_sensors);
		res = -EINVAL;
		goto get_sensors_list_error;
	}

	buf = add_accl_info_to_sensors_list(buf, dev_ctxt);
	if (buf == NULL)
		return -EINVAL;

	buf = add_gyro_info_to_sensors_list(buf, dev_ctxt);
	if (buf == NULL) {
		res = -EINVAL;
		goto get_sensors_list_error;
	}
	buf = add_sensor_info_to_sensors_list(dev_ctxt, buf, &dev_ctxt->sensors.depth_camera);
	if (buf == NULL) {
		res = -EINVAL;
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to copy depth camera data to sensors list\n", __func__);
		goto get_sensors_list_error;
	}
	buf = add_sensor_info_to_sensors_list(dev_ctxt, buf, &dev_ctxt->sensors.motion_camera);
	if (buf == NULL) {
		res = -EINVAL;
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to copy motion camera data to sensors list\n", __func__);
		goto get_sensors_list_error;
	}
	buf = add_external_sensors_info_to_sensors_list(buf, dev_ctxt);
	if (buf == NULL) {
		res = -EINVAL;
		goto get_sensors_list_error;
	}

get_sensors_list_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

/**
 * This function starts the device
 */
static int start(struct device_context *dev_ctxt)
{
	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	switch (dev_ctxt->state) {
	case DEVICE_STATE_STOPPED:
	case DEVICE_STATE_PAUSED:
		return start_stream(dev_ctxt);
	case DEVICE_STATE_STREAMING:
		dev_err(&dev_ctxt->i2c_client->dev, "%s Device is already started\n", __func__);
		return -EINVAL;
	/* These cases can't happen (checked in the IOCTL) but are implemented here to avoid compilation errors. */
	case DEVICE_STATE_CLOSED:
	case DEVICE_STATE_UNINITIALIZED:
	case DEVICE_STATE_IAP:
	default:
		return -EPERM;
	}
}

/**
 * This function pauses the device
 */
static int pause(struct device_context *dev_ctxt)
{
	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	switch (dev_ctxt->state) {
	case DEVICE_STATE_PAUSED:
		dev_err(&dev_ctxt->i2c_client->dev, "%s Device is already paused\n", __func__);
		return -EINVAL;
	case DEVICE_STATE_STOPPED:
		return pause_stream(dev_ctxt);
	case DEVICE_STATE_STREAMING:
		return pause_stream(dev_ctxt);
	/* These cases can't happen (checked in the IOCTL) but are implemented here to avoid compilation errors. */
	case DEVICE_STATE_CLOSED:
	case DEVICE_STATE_UNINITIALIZED:
	case DEVICE_STATE_IAP:
	default:
		return -EPERM;
	}
}

/**
 * This function stops the device
 */
static int stop(struct device_context *dev_ctxt)
{
	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	switch (dev_ctxt->state) {
	case DEVICE_STATE_STOPPED:
		dev_err(&dev_ctxt->i2c_client->dev, "%s Device is already stopped\n", __func__);
		return -EINVAL;
	case DEVICE_STATE_PAUSED:
	case DEVICE_STATE_STREAMING:
		return stop_stream(dev_ctxt);
	/* These cases can't happen (checked in the IOCTL) but are implemented here to avoid compilation errors. */
	case DEVICE_STATE_CLOSED:
	case DEVICE_STATE_UNINITIALIZED:
	case DEVICE_STATE_IAP:
	default:
		return -EPERM;
	}
}

/**
 * This function implements STM32L051_IOCTL_SET_STATE
 */
static int set_state_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_set_state_cmd ioctl_state = {0};
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	dev_info(&dev_ctxt->i2c_client->dev, "%s waiting for mutex\n", __func__);
	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	dev_info(&dev_ctxt->i2c_client->dev, "%s got the mutex\n", __func__);

	if (dev_ctxt->state & (DEVICE_STATE_CLOSED | DEVICE_STATE_UNINITIALIZED | DEVICE_STATE_IAP)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_state_error;
	}

	res = copy_from_user(&ioctl_state, (const void __user *)arg, sizeof(ioctl_state));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_state_error;
	}

	switch (ioctl_state.state) {
	case STM32L051_START:
		res = start(dev_ctxt);
	break;
	case STM32L051_PAUSE:
		res = pause(dev_ctxt);
	break;
	case STM32L051_STOP:
		res = stop(dev_ctxt);
	break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "Invalid state %d", ioctl_state.state);
		res = -EINVAL;
	}

set_state_error:
	dev_info(&dev_ctxt->i2c_client->dev, "%s releasing mutex\n", __func__);
	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

/**
 * This function translates handle to sensor control index.
 */
static int handle_to_sensor_ctrl_index(struct device_context *dev_ctxt, int handle, int handle_root, int num_sensors)
{
	int handle_root_high = (handle_root + (num_sensors - 1));

	if ((handle < handle_root) || (handle > handle_root_high)) {
		dev_err(&dev_ctxt->i2c_client->dev, "invalid handle %d (not in range %d::%d)\n", handle, handle_root, handle_root_high);
		return -ENXIO;
	}
	return handle - handle_root;
}


/**
 * This function implements STM32L051_IOCTL_SET_ACCL_CONFIG
 */
static int set_accl_config_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_vec_sensor_cfg_cmd ioctl_data;
	struct sensor_ctrl *sensor = &dev_ctxt->sensors.accelerometer;
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state & (DEVICE_STATE_STOPPED | DEVICE_STATE_PAUSED)) == 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_accl_config_error;
	}

	res = copy_from_user(&ioctl_data, (const void __user *)arg, sizeof(ioctl_data));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_accl_config_error;
	}

	if (ioctl_data.handle != sensor->handle) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s handle %d mismatch (expected %d)\n", __func__, ioctl_data.handle, sensor->handle);
		res = -ENXIO;
		goto set_accl_config_error;
	}

	switch (ioctl_data.configuration.bandwidth) {
	case STM32L051_ACC_BW_125_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_BW, UCTL_32BIT, ACC_BW_125_HZ);
		break;
	case STM32L051_ACC_BW_250_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_BW, UCTL_32BIT, ACC_BW_250_HZ);
		break;
	case STM32L051_ACC_BW_500_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_BW, UCTL_32BIT, ACC_BW_500_HZ);
		break;
	case STM32L051_ACC_BW_1000_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_BW, UCTL_32BIT, ACC_BW_1000_HZ);
		break;
	case STM32L051_ACC_BW_2000_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_BW, UCTL_32BIT, ACC_BW_2000_HZ);
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s ERROR incorrect bandwidth %d\n", __func__, ioctl_data.configuration.bandwidth);
		res = -EINVAL;
		goto set_accl_config_error;
		break;
	}

	if (res != 0)
		goto set_accl_config_error;

	switch (ioctl_data.configuration.range) {
	case STM32L051_ACC_RANGE_4:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_RANGE, UCTL_32BIT, ACC_RANGE_4);
		break;
	case STM32L051_ACC_RANGE_8:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_RANGE, UCTL_32BIT, ACC_RANGE_8);
		break;
	case STM32L051_ACC_RANGE_16:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_ACC_RANGE, UCTL_32BIT, ACC_RANGE_16);
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s ERROR incorrect range %d\n", __func__, ioctl_data.configuration.range);
		res = -EINVAL;
		goto set_accl_config_error;
		break;
	}

	if (res != 0)
		goto set_accl_config_error;

	res = config_sensor_interrupt(dev_ctxt, sensor->interrupt_mask, ioctl_data.enabled);
	if (res != 0)
		goto set_accl_config_error;

set_accl_config_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

/**
 * This function implements STM32L051_IOCTL_SET_GYRO_CONFIG
 */
static int set_gyro_config_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_vec_sensor_cfg_cmd ioctl_data;
	struct sensor_ctrl *sensor = &dev_ctxt->sensors.gyroscope;
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state & (DEVICE_STATE_STOPPED | DEVICE_STATE_PAUSED)) == 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_gyro_config_error;
	}

	res = copy_from_user(&ioctl_data, (const void __user *)arg, sizeof(ioctl_data));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_gyro_config_error;
	}

	if (ioctl_data.handle != sensor->handle) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s handle %d mismatch (expected %d)\n", __func__, ioctl_data.handle, sensor->handle);
		res = -ENXIO;
		goto set_gyro_config_error;
	}

	switch (ioctl_data.configuration.bandwidth) {
	case STM32L051_GYRO_BW_100_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_BW, UCTL_32BIT, GYRO_BW_100_HZ);
		break;
	case STM32L051_GYRO_BW_200_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_BW, UCTL_32BIT, GYRO_BW_200_HZ);
		break;
	case STM32L051_GYRO_BW_400_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_BW, UCTL_32BIT, GYRO_BW_400_HZ);
		break;
	case STM32L051_GYRO_BW_1000_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_BW, UCTL_32BIT, GYRO_BW_1000_HZ);
		break;
	case STM32L051_GYRO_BW_2000_HZ:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_BW, UCTL_32BIT, GYRO_BW_2000_HZ);
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s ERROR incorrect bandwidth %d\n", __func__, ioctl_data.configuration.bandwidth);
		res = -EPERM;
		goto set_gyro_config_error;
		break;
	}

	if (res != 0)
		goto set_gyro_config_error;

	switch (ioctl_data.configuration.range) {
	case STM32L051_GYRO_RANGE_1000:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_RANGE, UCTL_32BIT, GYRO_RANGE_1000);
		break;
	case STM32L051_GYRO_RANGE_2000:
		res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_GYRO_RANGE, UCTL_32BIT, GYRO_RANGE_2000);
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s ERROR incorrect range %d\n", __func__, ioctl_data.configuration.range);
		res = -EPERM;
		goto set_gyro_config_error;
		break;
	}

	if (res != 0)
		goto set_gyro_config_error;

	res = config_sensor_interrupt(dev_ctxt, sensor->interrupt_mask, ioctl_data.enabled);
	if (res != 0)
		goto set_gyro_config_error;

set_gyro_config_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

/**
 * This function implements STM32L051_IOCTL_SET_EXT_SENSOR_CONFIG
 */
static int set_ext_sensor_config_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_external_sensor_cfg_cmd ioctl_data = {0};
	struct external_sensors_ctrl *external_sensors = &dev_ctxt->sensors.external_sensors;
	int index = 0;
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state & (DEVICE_STATE_STOPPED | DEVICE_STATE_PAUSED)) == 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_ext_sensor_config_error;
	}

	res = copy_from_user(&ioctl_data, (const void __user *)arg, sizeof(ioctl_data));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_ext_sensor_config_error;
	}

	if (ioctl_data.enabled > 1) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s invalid enabled (%d)\n", __func__, ioctl_data.enabled);
		res = -EINVAL;
		goto set_ext_sensor_config_error;
	}

	if (ioctl_data.configuration.falling_edge > 1) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s invalid falling_edge (%d)\n", __func__, ioctl_data.configuration.falling_edge);
		res = -EINVAL;
		goto set_ext_sensor_config_error;
	}

	index = handle_to_sensor_ctrl_index(dev_ctxt, ioctl_data.handle, external_sensors->handle_root, NUM_EXTERNAL_SENSORS);
	if (index < 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid handle\n", __func__);
		res = -EINVAL;
		goto set_ext_sensor_config_error;
	}

	res = set_external_sensor_cfg(dev_ctxt, &external_sensors->sensors[index], &ioctl_data);

set_ext_sensor_config_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	dev_err(&dev_ctxt->i2c_client->dev, "%s returned with status %d\n", __func__, res);
	return res;
}

static int set_depth_cam_config_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_camera_sensor_cfg_cmd ioctl_data = {0};
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state & (DEVICE_STATE_STOPPED | DEVICE_STATE_PAUSED)) == 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_depth_sensor_config_error;
	}

	res = copy_from_user(&ioctl_data, (const void __user *)arg, sizeof(ioctl_data));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_depth_sensor_config_error;
	}

	if (ioctl_data.enabled > 1) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s invalid enabled (%d)\n", __func__, ioctl_data.enabled);
		res = -EINVAL;
		goto set_depth_sensor_config_error;
	}

	res = config_sensor_interrupt(dev_ctxt, dev_ctxt->sensors.depth_camera.interrupt_mask, ioctl_data.enabled);


set_depth_sensor_config_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	dev_err(&dev_ctxt->i2c_client->dev, "%s returned with status %d\n", __func__, res);
	return res;
}

static int set_motion_cam_config_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_camera_sensor_cfg_cmd ioctl_data = {0};
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state & (DEVICE_STATE_STOPPED | DEVICE_STATE_PAUSED)) == 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_motion_sensor_config_error;
	}

	res = copy_from_user(&ioctl_data, (const void __user *)arg, sizeof(ioctl_data));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_motion_sensor_config_error;
	}

	if (ioctl_data.enabled > 1) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s invalid enabled (%d)\n", __func__, ioctl_data.enabled);
		res = -EINVAL;
		goto set_motion_sensor_config_error;
	}

	res = config_sensor_interrupt(dev_ctxt, dev_ctxt->sensors.motion_camera.interrupt_mask, ioctl_data.enabled);


set_motion_sensor_config_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	dev_err(&dev_ctxt->i2c_client->dev, "%s returned with status %d\n", __func__, res);
	return res;
}

/**
 * This function implements STM32L051_IOCTL_HW_SYNC_CONFIG
 */
static int set_hw_sync_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_hw_sync_cfg_cmd ioctl_data = {0};
	int res = 0;
	__u32 reg_val = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state & (DEVICE_STATE_STOPPED | DEVICE_STATE_PAUSED)) == 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_hw_sync_error;
	}

	res = copy_from_user(&ioctl_data, (const void __user *)arg, sizeof(ioctl_data));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_hw_sync_error;
	}

	if (ioctl_data.enabled > 1) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s invalid value (%d)\n", __func__, ioctl_data.enabled);
		res = -EINVAL;
		goto set_hw_sync_error;
	}

	/* HW SYNC enable is a bit in the control register.
	/ Using read, modify, write to keep the rest of the register untouched. */
	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_UCTRL_CFG, UCTL_32BIT, &reg_val);
	if (res != 0)
		goto set_hw_sync_error;

	if (ioctl_data.enabled)
		reg_val |= HW_SYNC_ENABLED;
	else
		reg_val &= (~HW_SYNC_ENABLED);

	res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_UCTRL_CFG, UCTL_32BIT, reg_val);

set_hw_sync_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

/**
 * This function implements STM32L051_IOCTL_SET_LTR_CONFIG
 */
static int set_ltr_config_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_ltr_cfg_cmd ioctl_data = {0};
	int res = 0;
	int reg_val = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state & (DEVICE_STATE_STOPPED | DEVICE_STATE_PAUSED)) == 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_ltr_error;
	}

	res = copy_from_user(&ioctl_data, (const void __user *)arg, sizeof(ioctl_data));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_ltr_error;
	}

	if ((ioctl_data.ts_fifo_length > MAX_LTR_FIFO_LEN) || (ioctl_data.imu_fifo_length > MAX_LTR_FIFO_LEN)) {
		dev_err(&dev_ctxt->i2c_client->dev, "Invalid LTR config: TS FIFO length = %d, IMU FIFO length = %d (max = %d)\n",
			ioctl_data.ts_fifo_length, ioctl_data.imu_fifo_length, MAX_LTR_FIFO_LEN);
		res = -EINVAL;
		goto set_ltr_error;
	}

	reg_val = ((ioctl_data.ts_fifo_length & FIFO_LENGTH_MASK) |
			   ((ioctl_data.imu_fifo_length & FIFO_LENGTH_MASK) << IMU_FIFO_SHIFT));

	res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_REQ_LTR, UCTL_32BIT, reg_val);
	if (res != 0)
		goto set_ltr_error;

set_ltr_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

/**
 * This function implements STM32L051_IOCTL_SET_TIMESTAMP_CONFIG
 */
static int set_timestamp_config_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	struct stm32l051_timestamp_cfg_cmd ioctl_data = {0};
	int res = 0;
	__u32 reg_val = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	if ((dev_ctxt->state & (DEVICE_STATE_STOPPED | DEVICE_STATE_PAUSED)) == 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Invalid state %d\n", __func__, dev_ctxt->state);
		res = -EPERM;
		goto set_timestamp_cfg_error;
	}

	res = copy_from_user(&ioctl_data, (const void __user *)arg, sizeof(ioctl_data));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data\n", __func__);
		goto set_timestamp_cfg_error;
	}

	if ((ioctl_data.prescaler < MIN_TIMESTAMP_PRESCALER) ||
		(ioctl_data.prescaler > MAX_TIMESTAMP_PRESCALER)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Prescaler value %d out of bounds (%d::%d)\n",
			__func__, ioctl_data.prescaler, MIN_TIMESTAMP_PRESCALER, MAX_TIMESTAMP_PRESCALER);
		res = -EINVAL;
		goto set_timestamp_cfg_error;
	}

	/* SAS defines prescaler range as (1 :: 65536) */
	/* FW requires decreasing prescaler by 1 (0 :: 65535). */
	reg_val = (ioctl_data.prescaler -1);
	dev_info(&dev_ctxt->i2c_client->dev, "prescaler = %d, reg_val = 0x%X\n", ioctl_data.prescaler, reg_val);
	res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_TIMER_PRESCALER, UCTL_32BIT, reg_val);
	if (res != 0)
		goto set_timestamp_cfg_error;

	res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_TIMER_RESET_VALUE, UCTL_32BIT, ioctl_data.reset_value);
	if (res != 0)
		goto set_timestamp_cfg_error;

set_timestamp_cfg_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

/**
 * This function implements STM32L051_IOCTL_UPDATE_FIRMWARE
 */
static int update_firmware_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	int res = 0;
	u32 reg_val = 0;
	u32 image_address = 0x8002000;
	u32 image_length = 0;
	u32 length = 0;
	u32 retry = 0;
	enum device_state prev_state;
	const unsigned char __user *data_buffer;
	const unsigned char __user *command_buffer = (const unsigned char __user *) arg;

	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "<== %s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	prev_state = dev_ctxt->state;

	switch (dev_ctxt->state) {
	case DEVICE_STATE_UNINITIALIZED:
		/* The device is not initialized, need to power up before FW update. */
		res = pmic_ctrl(dev_ctxt, true);
		if (res != 0 && res != -EALREADY) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s failed to power up the device\n", __func__);
			goto fw_update_cmd_error;
		}

		res = uctl_read_reg_with_retries(dev_ctxt->i2c_client, UCTL_REG_CURR_PWR_STATE, UCTL_32BIT, &reg_val, 10, 20);
		if (res != 0) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s failed to read power state register\n", __func__);
			goto fw_update_cmd_error;
		}
		break;
	case DEVICE_STATE_IAP:
	case DEVICE_STATE_FW_CORRUPTED:
		/* Can't assume that interrupts are disabled and the fallback thread is stopped */
		disable_int(dev_ctxt, __func__);
		stop_main_thread(dev_ctxt);
		/* Power down and up to bring FW to a stable state */
		res = pmic_ctrl(dev_ctxt, false);
		if (res != 0)
			goto fw_update_cmd_error;
		res = pmic_ctrl(dev_ctxt, true);
		if (res != 0 && res != -EALREADY)
			goto fw_update_cmd_error;
	break;
	case DEVICE_STATE_STOPPED:
		/* In this case, start FW update. */
		disable_int(dev_ctxt, __func__);
		stop_main_thread(dev_ctxt);
		break;
	case DEVICE_STATE_CLOSED:
	case DEVICE_STATE_PAUSED:
	case DEVICE_STATE_STOPPING:
	case DEVICE_STATE_STREAMING:
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s: FW update is not supported while in state %d\n", __func__, dev_ctxt->state);
		goto fw_update_cmd_error;
		break;
	}

	/* See struct stm32l051_fw_update_cmd: first field is length (4 bytes) followed by the image. */
	res = copy_from_user(&image_length, command_buffer, sizeof(image_length));
	command_buffer += sizeof(image_length);


	dev_info(&dev_ctxt->i2c_client->dev, "%s FW image length: %d\n", __func__, image_length);

	/* data buffer is used for retries */
	for(retry = 0; retry < 3; retry ++) {
		data_buffer = command_buffer;
		length = image_length;
		image_address = 0x8002000;

		res = switch_to_iap(dev_ctxt);
		if (res != 0)
			goto fw_update_cmd_error;

		/* Write the image via I2C in packets of 256 bytes. */
		while (length > 0) {
			u16 payload_length = (length > FW_IMAGE_PACKET_PAYLOAD_LEN) ? FW_IMAGE_PACKET_PAYLOAD_LEN : length;
			struct fw_image_packet packet = {
				.op_code = 0x6,
				.address = cpu_to_be32(image_address),
				.length = cpu_to_be16(payload_length),
				.dummy = 0,
			};
			u16 packetLength = (sizeof(packet) - FW_IMAGE_PACKET_PAYLOAD_LEN + payload_length);
			res = copy_from_user(packet.data, data_buffer, payload_length);
			if (res != 0) {
				dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to copy data from user with status: %d\n", __func__, res);
				goto fw_update_cmd_error;
			}

			res = uctl_i2c_write(dev_ctxt->i2c_client,
				packetLength,
				(u8 *) &packet);
			if (res != 0) {
				dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to write FW image with status %d\n", __func__, res);
				break;
			}

			dev_info(&dev_ctxt->i2c_client->dev, "%s Wrote FW image fragment (Addr:0x%X, image addr:0x%p, length %d, packet length %d)\n",
				__func__, image_address, data_buffer, payload_length, packetLength);

			data_buffer += payload_length;
			length -= payload_length;
			image_address += payload_length;
		}

		if(res == 0)
			res = switch_to_operational(dev_ctxt);

		if(res == 0)
			break;
		else {
			/* If failed to write image to FW, power down / up and start again */
			dev_err(&dev_ctxt->i2c_client->dev, "%s FW update attempt %d failed, powering down and up for retry\n", __func__, retry);
			set_driver_error(dev_ctxt, STM32L051_ERROR_FW_UPDATE_FAILED_WRITE);
			if (pmic_ctrl(dev_ctxt, false) != 0)
				goto fw_update_cmd_error;
			res = pmic_ctrl(dev_ctxt, true);
			if (res != 0 && res != -EALREADY)
				goto fw_update_cmd_error;
			continue;
		}
	}

	/* res != 0 at this state means that FW updated exceeded retry count */
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s FW update exceeded retry count\n", __func__);
		goto fw_update_cmd_error;
	}

	switch (prev_state) {
	case DEVICE_STATE_UNINITIALIZED:
		/* The device was not initialized, power down. */
		res = pmic_ctrl(dev_ctxt, false);
		if (res != 0) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s failed to power down the device\n", __func__);
			goto fw_update_cmd_error;
		}
		set_state(dev_ctxt, DEVICE_STATE_UNINITIALIZED);
		break;
	case DEVICE_STATE_FW_CORRUPTED:
	case DEVICE_STATE_STOPPED:
	case DEVICE_STATE_IAP:
		/* In this case, init the FW. */
		enable_int(dev_ctxt, __func__);
		res = create_main_thread(dev_ctxt);
		if (res != 0)
			goto fw_update_cmd_error;
		res = check_power_up(dev_ctxt);
		if (res != 0)
			goto fw_update_cmd_error;

		res = init_sensors(dev_ctxt);
		if (res != 0)
			goto fw_update_cmd_error;
		set_state(dev_ctxt, DEVICE_STATE_STOPPED);
		break;
	case DEVICE_STATE_CLOSED:
	case DEVICE_STATE_PAUSED:
	case DEVICE_STATE_STOPPING:
	case DEVICE_STATE_STREAMING:
		dev_err(&dev_ctxt->i2c_client->dev, "%s: FW update is not supported while in state %d\n", __func__, dev_ctxt->state);
		goto fw_update_cmd_error;
		break;
	}

fw_update_cmd_error:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);
	return res;
}


union eeprom_access_reg {
	struct bit_fields {
		u32 address:13;
		u32 direction:1; /* 0 = read, 1 = write */
		u32 reserved_1:2;
		u32 size:8;
		u32 reserved_2:8;
	} fields;
	u32 value;
};

static int wait_eeprom_ready(struct device_context *dev_ctxt)
{
	int i = 0;
	int res = 0;
	for (i = 0; i < 5; i++) {
		res = process_hw_events(dev_ctxt);
		if(res != 0)
			return res;
		if ((dev_ctxt->status_register & UCTL_READY_FOR_NEXT_EEPROM) != 0) {
			dev_info(&dev_ctxt->i2c_client->dev, "%s EEprom is ready after %d iterations\n", __func__, i);
			return 0;
		}
		msleep(50);
	}
	dev_err(&dev_ctxt->i2c_client->dev, "%s EEprom is not ready\n", __func__);
	return -ERESTARTSYS;
}

static int begin_eeprom_access(struct device_context *dev_ctxt)
{
	int res = 0;
	u32 reg_val = 0;

	switch (dev_ctxt->state) {
	case DEVICE_STATE_UNINITIALIZED:
		/* The device is not initialized, need to power up before FW update. */
		dev_ctxt->status_register = 0;
		res = power_up(dev_ctxt);
		if (res != 0) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s failed to power up the device\n", __func__);
			return res;
		}

		res = uctl_read_reg_with_retries(dev_ctxt->i2c_client, UCTL_REG_CURR_PWR_STATE, UCTL_32BIT, &reg_val, 10, 20);
		if (res != 0)
			dev_err(&dev_ctxt->i2c_client->dev, "%s failed to read power state register\n", __func__);
		if (reg_val == UCTL_PWR_STATE_IAP) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s Can't access EEprom while in IAP state\n", __func__);
			return -ERESTARTSYS;
		}
		res = check_power_up(dev_ctxt);
		if(res != 0)
			break;
		res = process_hw_events(dev_ctxt);
		break;
	case DEVICE_STATE_STOPPED:
		res = process_hw_events(dev_ctxt);
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s EEprom access is not supported at this state\n", __func__);
		res = -ERESTARTSYS;
	}

	if (res == 0)
		res = wait_eeprom_ready(dev_ctxt);

	return res;
}

static int end_eeprom_access(struct device_context *dev_ctxt)
{
	int res = 0;

	res = wait_eeprom_ready(dev_ctxt);
	if(res != 0)
		return res;

	switch (dev_ctxt->state) {
	case DEVICE_STATE_UNINITIALIZED:
		/* The device is not initialized, need to power up before FW update. */
		res = power_down(dev_ctxt);
		if (res != 0)
			dev_err(&dev_ctxt->i2c_client->dev, "%s failed to power down the device\n", __func__);
		break;
	case DEVICE_STATE_STOPPED:
		break;
	default:
		dev_err(&dev_ctxt->i2c_client->dev, "%s EEprom access is not supported at state %d\n", __func__, dev_ctxt->state);
		res = -ERESTARTSYS;
		break;
	}
	return res;
}


/**
 * This function implements STM32L051_IOCTL_READ_EEPROM
 */
static int read_eeprom_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	int res = 0;
	union eeprom_access_reg access_reg;
	const unsigned char __user *command_buffer = (const unsigned char __user *) arg;
	u16 address = 0;
	unsigned char buffer[STM32L051_EEPROM_BLOCK_LENGTH] = { 0 };

	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	/* Reading only the address from the buffer*/
	res = copy_from_user(&address, command_buffer, sizeof(address));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data (address)\n", __func__);
		goto eeprom_read_error_1;
	}
	command_buffer += sizeof(address);

	access_reg.fields.address = address;
	if (access_reg.fields.address != address) {
		res = -EINVAL;
		dev_err(&dev_ctxt->i2c_client->dev, "%s IOCTL address 0x%X > max address length (13 bits)\n", __func__, address);
		goto eeprom_read_error_1;
	}

	access_reg.fields.direction = 0; /* Read direction. */
	access_reg.fields.size = STM32L051_EEPROM_BLOCK_LENGTH;

	dev_info(&dev_ctxt->i2c_client->dev, "%s EEprom read from address 0x%X", __func__, address);

	res = begin_eeprom_access(dev_ctxt);
	if (res != 0)
		goto eeprom_read_error_1;

	res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_EEPROM_CMD, UCTL_32BIT, access_reg.value);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to config EEprom read", __func__);
		goto eeprom_read_error_2;
	}

	res = uctl_i2c_read(dev_ctxt->i2c_client, UCTL_REG_EEPROM_DATA, STM32L051_EEPROM_BLOCK_LENGTH, buffer);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to read from EEprom", __func__);
		goto eeprom_read_error_2;
	}
	res = copy_to_user((void __user *)command_buffer, buffer, STM32L051_EEPROM_BLOCK_LENGTH);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to copy data to user", __func__);
		goto eeprom_read_error_2;
	}

eeprom_read_error_2:
	res = end_eeprom_access(dev_ctxt);
eeprom_read_error_1:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);

	return res;
}

/**
 * This function implements STM32L051_IOCTL_WRITE_EEPROM
 */
static int write_eeprom_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	int res = 0;
	union eeprom_access_reg access_reg;
	const unsigned char __user *command_buffer = (const unsigned char __user *) arg;
	u16 address = 0;
	unsigned char buffer[STM32L051_EEPROM_BLOCK_LENGTH + 1] = { 0 }; /* EEprom data register address + Buffer length*/

	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	buffer[0] = UCTL_REG_EEPROM_DATA;

	/* Reading only the address from the buffer*/
	res = copy_from_user(&address, command_buffer, sizeof(address));
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data (address)\n", __func__);
		goto eeprom_write_error_1;
	}
	command_buffer += sizeof(address);

	res = copy_from_user((buffer + 1), command_buffer, STM32L051_EEPROM_BLOCK_LENGTH);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to get IOCTL data (buffer)\n", __func__);
		goto eeprom_write_error_1;
	}

	access_reg.fields.address = address;
	if (access_reg.fields.address != address) {
		res = -EINVAL;
		dev_err(&dev_ctxt->i2c_client->dev, "%s IOCTL address 0x%X > max address length (13 bits)\n", __func__, address);
		goto eeprom_write_error_1;
	}

	access_reg.fields.direction = 1; /* Write direction. */
	access_reg.fields.size = STM32L051_EEPROM_BLOCK_LENGTH;

	dev_info(&dev_ctxt->i2c_client->dev, "%s EEprom write to address 0x%X", __func__, address);

	res = begin_eeprom_access(dev_ctxt);
	if (res != 0)
		goto eeprom_write_error_1;

	res = uctl_write_reg(dev_ctxt->i2c_client, UCTL_REG_EEPROM_CMD, UCTL_32BIT, access_reg.value);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to config EEprom write", __func__);
		goto eeprom_write_error_2;
	}

	res = uctl_i2c_write(dev_ctxt->i2c_client, (STM32L051_EEPROM_BLOCK_LENGTH + 1), buffer);
	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s Failed to write data to EEprom", __func__);
		goto eeprom_write_error_2;
	}

eeprom_write_error_2:
	res = end_eeprom_access(dev_ctxt);
eeprom_write_error_1:
	mutex_unlock(&dev_ctxt->device_handle_lock);

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);

	return res;
}

static int get_drv_error_cmd(struct device_context *dev_ctxt, unsigned long arg)
{
	char __user *buf = (char __user *) arg;
	int res = 0;
	struct stm32l051_get_device_error_cmd device_error_cmd;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->dev_status.device_status_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	device_error_cmd.drv_error_mask = dev_ctxt->dev_status.major_error;
	dev_ctxt->dev_status.major_error = 0;
	dev_ctxt->dev_status.error_reported = false;
	mutex_unlock(&dev_ctxt->dev_status.device_status_lock);

	res = copy_to_user(buf, &device_error_cmd, sizeof(device_error_cmd));

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);

	return res;
}

static ssize_t sysfs_show_error(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct device_context *dev_ctxt = input_get_drvdata(input);
	size_t ret_cnt = 0;
	size_t cnt = 0;
	int i = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->dev_status.device_status_lock) == 0) {
		for (i = 0; ((i < STM32L051_NUM_ERRORS) && (dev_ctxt->dev_status.major_error != 0)); i++) {
			if (dev_ctxt->dev_status.major_error & 0x1) {
				if (ret_cnt == 0)
					cnt = sprintf(buf, "%s", g_error_codes[i]);
				else
					cnt = sprintf(buf, "|%s", g_error_codes[i]);
				buf += cnt;
				ret_cnt += cnt;
			}
			dev_ctxt->dev_status.major_error = (dev_ctxt->dev_status.major_error >> 1);
		}
		if (ret_cnt > 0) {
			cnt = sprintf(buf, "\n");
			buf += cnt;
			ret_cnt += cnt;
		}

		dev_ctxt->dev_status.error_reported = false;

		mutex_unlock(&dev_ctxt->dev_status.device_status_lock);
	}

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);

	return ret_cnt;
}

static ssize_t sysfs_show_fw_error(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct device_context *dev_ctxt = input_get_drvdata(input);
	size_t ret_cnt = 0;
	size_t cnt = 0;
	int i = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	if (mutex_lock_interruptible(&dev_ctxt->dev_status.device_status_lock) == 0) {
		dev_info(&dev_ctxt->i2c_client->dev, "%s --> fw_error = 0x%X\n", __func__, dev_ctxt->dev_status.fw_error);
		for (i = 0; ((i < NUM_FW_ERRORS) && (dev_ctxt->dev_status.fw_error != 0)); i++) {
			if (dev_ctxt->dev_status.fw_error & 0x1) {
				if (ret_cnt == 0)
					cnt = sprintf(buf, "%s", g_fw_error_codes[i]);
				else
					cnt = sprintf(buf, "|%s", g_fw_error_codes[i]);
				buf += cnt;
				ret_cnt += cnt;
			}
			dev_ctxt->dev_status.fw_error = (dev_ctxt->dev_status.fw_error >> 1);
		}
		if (ret_cnt > 0) {
			cnt = sprintf(buf, "\n");
			buf += cnt;
			ret_cnt += cnt;
		}
		dev_info(&dev_ctxt->i2c_client->dev, "%s <-- fw_error = 0x%X\n", __func__, dev_ctxt->dev_status.fw_error);
		mutex_unlock(&dev_ctxt->dev_status.device_status_lock);
	}

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);
	return ret_cnt;
}

static int get_driver_version(struct device_context *dev_ctxt, char *buf)
{
	if(NULL == buf) {
		dev_err(&dev_ctxt->i2c_client->dev,
				"%s invalid parameters, buf is null\n", __func__);
		return -EINVAL;
	}

	/*Assuiming buf size is greater or equal to 8*/
	buf[0] = ((stm32l051_driver_version & 0xFF000000) >> 24) + '0';
	buf[1] = '.';
	buf[2] = ((stm32l051_driver_version & 0x00FF0000) >> 16)+ '0';
	buf[3] = '.';
	buf[4] = ((stm32l051_driver_version & 0x0000FF00) >> 8) + '0';
	buf[5] = '.';
	buf[6] = (stm32l051_driver_version & 0x000000FF) + '0';
	buf[7] = '\0';

	return 0;
}

static ssize_t sysfs_show_version(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct device_context *dev_ctxt = input_get_drvdata(input);
	size_t ret_cnt = 0;
	u32 fw_version = 0;
	int res = 0;
	char driver_version[8] = {0};


	dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);

	get_driver_version(dev_ctxt, driver_version);
	if (dev_ctxt->pmic_powered_on) {
		res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_IMG_VER, UCTL_32BIT, &fw_version);
		if (res != 0)
			dev_info(&dev_ctxt->i2c_client->dev, "%s failed to read FW version\n", __func__);
		ret_cnt = sprintf(buf, "Driver version %s, FW Version %X\n", driver_version, fw_version);
	} else {
		ret_cnt = sprintf(buf, "Driver version %s, FW not loaded yet\n", driver_version);
	}

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);
	return ret_cnt;
}

#if STATISTICS_ENABLED
static int sysfs_print_statistics_accumulator(char *buf, const char *message, const char *units, const struct statistic_accumulator *accumulator)
{
	if (accumulator->samples > 0) {
		return sprintf(buf, "%s: Average = %lld, count = %lld, samples = %lld, min = %lld, max = %lld (%s)\n",
			message,
			(accumulator->count / accumulator->samples),
			accumulator->count,
			accumulator->samples,
			accumulator->min,
			accumulator->max,
			units);
	}

	return sprintf(buf, "%s not initialized\n", message);
}

static ssize_t sysfs_reset_statistics(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct input_dev *input = to_input_dev(dev);
	struct device_context *dev_ctxt = input_get_drvdata(input);

	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);

	if ((dev_ctxt->state == DEVICE_STATE_STREAMING) || (dev_ctxt->state == DEVICE_STATE_STOPPING)) {
		dev_info(&dev_ctxt->i2c_client->dev, "<== %s can't reset statistics while in state %d\n", __func__, dev_ctxt->state);
		return count;
	}
	reset_statistics(dev_ctxt);
	dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__);
	return count;
}

static ssize_t sysfs_show_statistics(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct input_dev *input = to_input_dev(dev);
	struct device_context *dev_ctxt = input_get_drvdata(input);
	char *orig_buf = buf;
	int i = 0;

	/* Local copy does not assure data corruption (stats might be updated during copy),
	   but it limits it to a reasonable tolerance.
	   The other option is to lock statistics access by mutex which will slow down the driver. */
	struct device_statistics statistics = dev_ctxt->statistics;
	buf += sysfs_print_statistics_accumulator(buf, "Hard IRQ delta", "ms", &statistics.hard_interrupt_delta);
	buf += sysfs_print_statistics_accumulator(buf, "Soft IRQ delta", "ms", &statistics.soft_interrupt_delta);
	buf += sysfs_print_statistics_accumulator(buf, "Fallback timer delta", "ms", &statistics.fallback_timer_delta);
	buf += sysfs_print_statistics_accumulator(buf, "Read IMU data time", "ms", &statistics.read_imu_fifo_delta);
	buf += sysfs_print_statistics_accumulator(buf, "Read TS data time", "ms", &statistics.read_ts_fifo_delta);
	buf += sysfs_print_statistics_accumulator(buf, "Num IMU entries", "per read", &statistics.imu_entries);
	buf += sysfs_print_statistics_accumulator(buf, "Num TS entries", "per read", &statistics.ts_entries);
	buf += sysfs_print_statistics_accumulator(buf, "Read HW FIFO iterations", "per process HW events", &statistics.read_hw_fifo_iterations);
	buf += sysfs_print_statistics_accumulator(buf, "Poll delta", "ms", &statistics.poll_delta);
	for (i = 0; i < NUM_SENSORS; i++)
		if (dev_ctxt->sensors.controls[i]->name != NULL)
			buf += sprintf(buf, "%s frame loss count = %lld\n", dev_ctxt->sensors.controls[i]->name, statistics.sensor_frame_loss[i]);

	return (size_t) buf - (size_t) orig_buf;
}
#endif /*STATISTICS_ENABLED*/

static int stm32l051_unloading;
static unsigned int stm32l051_poll(struct file *filp, struct poll_table_struct *poll_table)
{
	unsigned int mask = 0;
	struct device_context *dev_ctxt;

	if (!filp)
		return -ENODEV;

	if (!filp->private_data)
		return -ENODEV;

	dev_ctxt = filp->private_data;

	if(!dev_ctxt)
		return POLLERR|POLLNVAL;

	if(stm32l051_unloading)
		return POLLERR|POLLNVAL;

	update_time_delta(&dev_ctxt->timing_measurements.poll_time, &dev_ctxt->statistics.poll_delta);
	



    poll_wait(filp, &dev_ctxt->wait_queue, poll_table);

	if (!kfifo_is_empty(&dev_ctxt->fifo))
		mask |= POLLIN | POLLRDNORM;
	if(mutex_lock_interruptible(&dev_ctxt->dev_status.device_status_lock) == 0) {

		if ((dev_ctxt->dev_status.major_error != 0) && (dev_ctxt->dev_status.error_reported==false)) {
			dev_err(&dev_ctxt->i2c_client->dev, "%s Reporting error\n", __func__);
			dev_ctxt->dev_status.error_reported = true;
			mask |= POLLERR;
		}
		mutex_unlock(&dev_ctxt->dev_status.device_status_lock);
	} else {
		dev_err(&dev_ctxt->i2c_client->dev, "==> %s, couldn't lock interruptable\n", __func__);
	}

	return mask;
}

/**
 * This function dispatches IOCTL calls.
 */
static long stm32l051_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct device_context *dev_ctxt;
	
	if (!filp)
		return -ENODEV;
	
	if (!filp->private_data)
		return -ENODEV;
	
	dev_ctxt = filp->private_data;

//	dev_info(&dev_ctxt->i2c_client->dev, "uctl ioctl\n");

	/*
	 * verify correct cmd type received
	 */
	if (_IOC_TYPE(cmd) != STM32L051_IOCTL_MAGIC)
		return -EPERM;

	if (_IOC_NR(cmd) > STM32L051_IOCTL_MAGIC)
		return -EPERM;

	switch (cmd) {
	case STM32L051_IOCTL_INIT:
		return init_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_UNINIT:
		return uninit_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_GET_DEVICE_INFO:
		return get_device_info_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_GET_SENSORS_LIST:
		return get_sensors_list_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_SET_STATE:
		return set_state_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_SET_ACCL_CONFIG:
		return set_accl_config_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_SET_GYRO_CONFIG:
		return set_gyro_config_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_SET_EXT_SENSOR_CONFIG:
		return set_ext_sensor_config_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_HW_SYNC_CONFIG:
		return set_hw_sync_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_SET_LTR_CONFIG:
		return set_ltr_config_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_SET_TIMESTAMP_CONFIG:
		return set_timestamp_config_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_UPDATE_FIRMWARE:
		return update_firmware_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_READ_EEPROM:
		return read_eeprom_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_WRITE_EEPROM:
		return write_eeprom_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_GET_DEVICE_ERROR:
		return get_drv_error_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_SET_DEPTH_CAM_CONFIG:
		return set_depth_cam_config_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_SET_MOTION_CAM_CONFIG:
		return set_motion_cam_config_cmd(dev_ctxt, arg);
	case STM32L051_IOCTL_GET_TIMESTAMP:
		return get_ts(dev_ctxt, arg);
	default:
		return -EPERM;
	}

	return 0;
}

/**
 * This function opens the device (singleton).
 */
static int stm32l051_open(struct inode *node, struct file *filp)
{
	struct device_context *dev_ctxt;
	int res = 0;

	if (!node)
		return -ENODEV;

	if (!filp)
		return -ENODEV;

	/* save dev_ctxt for other file operation */
	dev_ctxt = container_of(node->i_cdev, struct device_context, cdev);
	filp->private_data = dev_ctxt;

	/* TODO: add access policy
	if((filp->f_flags & O_ACCMODE) != O_RDONLY) {
		return -EACCES;
	}*/

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev,
			"%s: mutex_lock_interruptible failed\n",
			 __func__);
		return -ERESTARTSYS;
	}

	if (dev_ctxt->state != DEVICE_STATE_CLOSED) {
		dev_warn(&dev_ctxt->i2c_client->dev,
				"%s: device is already opened (state = %d)\n",
				__func__, dev_ctxt->state);
		res = -EBUSY; /* TODO: correct error value */
	} else {
		set_state(dev_ctxt, DEVICE_STATE_UNINITIALIZED);
		dev_info(&dev_ctxt->i2c_client->dev, "%s deivce is opened suuccessfuly\n", __func__);
	}

	dev_ctxt->gyro_overflow_count=0;
	dev_ctxt->accel_overflow_count=0;
	dev_ctxt->ts_overflow_count=0;

	mutex_unlock(&dev_ctxt->device_handle_lock);

	return res;
}

static DEVICE_ATTR(drv_error, S_IRUGO, sysfs_show_error, NULL);
static DEVICE_ATTR(fw_error, S_IRUGO, sysfs_show_fw_error, NULL);
static DEVICE_ATTR(version, S_IRUGO, sysfs_show_version, NULL);
#if STATISTICS_ENABLED
//static DEVICE_ATTR(statistics, S_IRUGO | S_IWGRP, sysfs_show_statistics, sysfs_reset_statistics);
#endif /* STATISTICS_ENABLED */

static struct attribute *stm32l051_attributes[] = {
	&dev_attr_drv_error.attr,
	&dev_attr_fw_error.attr,
	&dev_attr_version.attr,
#if STATISTICS_ENABLED
	&dev_attr_statistics.attr,
#endif /*STATISTICS_ENABLED*/
	NULL
};

static struct attribute_group stm32l051_attribute_group = {
	.attrs = stm32l051_attributes
};

/**
 * This function closes the device.
 */
static int stm32l051_release(struct inode *node, struct file *filp)
{
	int res = 0;
	struct device_context *dev_ctxt;

	if (!filp)
		return -ENODEV;

	if (!filp->private_data)
		return -ENODEV;

	dev_ctxt = filp->private_data;

	dev_info(&dev_ctxt->i2c_client->dev, "uctl release - state=%d\n", dev_ctxt->state);

	if (mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	switch (dev_ctxt->state) {
	case DEVICE_STATE_STREAMING:
		stop_stream(dev_ctxt);
		/* No break here to continue to uninit. */
	case DEVICE_STATE_PAUSED:
	case DEVICE_STATE_STOPPED:
		dev_info(&dev_ctxt->i2c_client->dev, "uctl release - calling uninit()\n");
		res = uninit(dev_ctxt);
		if (res != 0)
			dev_err(&dev_ctxt->i2c_client->dev, "%s: driver failed to self uninitialize\n", __func__);
		break;
	case DEVICE_STATE_STOPPING:
		dev_err(&dev_ctxt->i2c_client->dev, "%s: Releasing during Stopping state, unexpected behavior\n", __func__);
		break;
	case DEVICE_STATE_IAP:
		dev_err(&dev_ctxt->i2c_client->dev, "%s: Releasing during IAP state, unexpected behavior\n", __func__);
		break;
	case DEVICE_STATE_FW_CORRUPTED:
	case DEVICE_STATE_CLOSED:
	case DEVICE_STATE_UNINITIALIZED:
	default:
		break;
	}

	set_state(dev_ctxt, DEVICE_STATE_CLOSED);

	mutex_unlock(&dev_ctxt->device_handle_lock);

	return 0;
}

static const struct file_operations uctl_fops = {
	.owner			= THIS_MODULE,
	.read			= stm32l051_read,
	.write			= stm32l051_write,
	.poll			= stm32l051_poll,
	.unlocked_ioctl = stm32l051_ioctl,
	.open			= stm32l051_open,
	.release		= stm32l051_release,
};

static void parse_raw_sensor_commond_data(struct device_context *dev_ctxt, struct raw_sensor_common_data *common, struct stm32l051_sensor_data *dest, u32 overflow_count)
{
	u64 hw_time_stamp = overflow_count;
	hw_time_stamp <<= 32;
	hw_time_stamp |= le32_to_cpu(common->timestamp);
	common->header.value = le16_to_cpu(*(__le16 *) &common->header.value);
	dest->handle = common->header.bit_fields.handle;
	dest->hw_frame_id = common->header.bit_fields.frame_num;
	/* Conversion to nanosecond original formula is:
	   time stamp * SEC_TO_NANOSEC / (IMU frequency / timer prescaler).
	   In order to get to the highest accuracy, the formula is changed as follows:
	   ns = (time_stamp * SEC_TO_NANOSEC / (IMU_FREQUENCY / timer_prescaler)) = 
           (time_stamp * SEC_TO_NANOSEC * timer_prescaler) / IMU_FREQUENCY
           */
	/* NOTE: Original formula caused an overflow, so simplified 2 constant division to
	         SEC_TO_NANOSEC (1000000000) / IMU_FREQUENCY (32000000)
	 */
	dest->hw_time_stamp = hw_time_stamp * dev_ctxt->timer_prescaler * (125 / 4);
	dest->sequence_num = 0;
	dest->system_time_stamp = dev_ctxt->softirq_time_stamp;
}

#define MAX_HW_FRAME_ID (1 << 12)
#define HW_FRAME_ID_MASK (MAX_HW_FRAME_ID - 1)

static __u32 process_hw_frame_id(struct device_context *dev_ctxt, __u32 frame_id, struct sensor_ctrl *ctrl)
{
	if (ctrl->was_reset) {
		ctrl->was_reset = false;
		ctrl->next_frame_id = ((frame_id + 1) & HW_FRAME_ID_MASK);
		return frame_id;
	} else {
		if (frame_id != ctrl->next_frame_id) {
			dev_err(&dev_ctxt->i2c_client->dev, "Frame loss detected for %s (expected = %d, received = %d)",
				ctrl->name, ctrl->next_frame_id, frame_id);
			set_driver_error(dev_ctxt, STM32L051_ERROR_FRAME_LOSS_DETECTED);
			dev_ctxt->statistics.sensor_frame_loss[ctrl->handle]++;
			update_fw_error(dev_ctxt);
		}
		if (frame_id == 0)
			ctrl->frame_id_multiplier++;

		ctrl->next_frame_id = ((frame_id + 1) & HW_FRAME_ID_MASK);
		return frame_id + (MAX_HW_FRAME_ID * ctrl->frame_id_multiplier);
	}
}

static int parse_imu_entry(struct device_context *dev_ctxt, struct stm32l051_sensor_data *dest, u8 *src)
{
	struct raw_imu_data *raw_data = (struct raw_imu_data *)src;

	if (!raw_data->common.header.bit_fields.handle) { //accel
		if (raw_data->common.timestamp < dev_ctxt->accel_last_read.common.timestamp) {
			dev_ctxt->accel_overflow_count++;
		}
		memcpy(&dev_ctxt->accel_last_read, raw_data, sizeof(*raw_data));
		parse_raw_sensor_commond_data(dev_ctxt, &raw_data->common, dest, dev_ctxt->accel_overflow_count);
	}else { //gyro
		if (raw_data->common.timestamp < dev_ctxt->gyro_last_read.common.timestamp) {
			dev_ctxt->gyro_overflow_count++;
		}
		memcpy(&dev_ctxt->gyro_last_read, raw_data, sizeof(*raw_data));
		parse_raw_sensor_commond_data(dev_ctxt, &raw_data->common, dest, dev_ctxt->gyro_overflow_count);
	}

	if (dest->handle == dev_ctxt->sensors.accelerometer.handle) {
		dest->hw_frame_id = process_hw_frame_id(dev_ctxt, dest->hw_frame_id, &dev_ctxt->sensors.accelerometer);
		dest->acc_data.x = le16_to_cpu(raw_data->x);
		dest->acc_data.y = le16_to_cpu(raw_data->y);
		dest->acc_data.z = le16_to_cpu(raw_data->z);
		/* dev_info(&dev_ctxt->i2c_client->dev, "ACC data: Handle=%d, Frame Id=%d, HW Timestamp=0x%llX, Vector: X=0x%X, Y=0x%X, Z=0x%X",
				dest->handle, dest->hw_frame_id, dest->hw_time_stamp, dest->acc_data.x, dest->acc_data.y, dest->acc_data.z); */
	} else if (dest->handle == dev_ctxt->sensors.gyroscope.handle) {
		dest->hw_frame_id = process_hw_frame_id(dev_ctxt, dest->hw_frame_id, &dev_ctxt->sensors.gyroscope);
		dest->gyro_data.x = le16_to_cpu(raw_data->x);
		dest->gyro_data.y = le16_to_cpu(raw_data->y);
		dest->gyro_data.z = le16_to_cpu(raw_data->z);
		/* dev_info(&dev_ctxt->i2c_client->dev, "Gyro data: Handle=%d, Frame Id=%d, HW Timestamp=0x%llX, Vector: X=0x%X, Y=0x%X, Z=0x%X",
				dest->handle, dest->hw_frame_id, dest->hw_time_stamp, dest->gyro_data.x, dest->gyro_data.y, dest->gyro_data.z); */
	} else {
		dev_err(&dev_ctxt->i2c_client->dev, "Invalid handle %d, header = 0x%X", dest->handle, raw_data->common.header.value);
		return -1;
	}

	return 0;
}

static int parse_ts_entry(struct device_context *dev_ctxt, struct stm32l051_sensor_data *dest, u8 *src)
{
	struct raw_ts_data *raw_data = (struct raw_ts_data *)src;
	if (raw_data->common.timestamp < dev_ctxt->ts_last_read.common.timestamp) {
		dev_ctxt->ts_overflow_count++;
	}
	memcpy(&dev_ctxt->ts_last_read, raw_data, sizeof(*raw_data));
	parse_raw_sensor_commond_data(dev_ctxt, &raw_data->common, dest, dev_ctxt->ts_overflow_count);

	if (dest->handle == dev_ctxt->sensors.depth_camera.handle)
		dest->hw_frame_id = process_hw_frame_id(dev_ctxt, dest->hw_frame_id, &dev_ctxt->sensors.depth_camera);
	if (dest->handle == dev_ctxt->sensors.motion_camera.handle)
		dest->hw_frame_id = process_hw_frame_id(dev_ctxt, dest->hw_frame_id, &dev_ctxt->sensors.motion_camera);
	if (dest->handle == dev_ctxt->sensors.external_sensors.sensors[0].common.handle)
		dest->hw_frame_id = process_hw_frame_id(dev_ctxt, dest->hw_frame_id, &dev_ctxt->sensors.external_sensors.sensors[0].common);
	if (dest->handle == dev_ctxt->sensors.external_sensors.sensors[1].common.handle)
		dest->hw_frame_id = process_hw_frame_id(dev_ctxt, dest->hw_frame_id, &dev_ctxt->sensors.external_sensors.sensors[1].common);

	return 0;
}

static int read_imu_data(struct i2c_client *client, int num_of_imu_entries)
{
	struct device_context *dev_ctxt = i2c_get_clientdata(client);
	struct read_fifo_configuration cfg = {
		.num_entries = num_of_imu_entries,
		.entry_size = IMU_ENTRY_SIZE,
		.max_read_size = MAX_IMU_READ_SIZE,
		.fifo_address = UCTL_FIFO_DATA_IMU,
		.parse_entry = parse_imu_entry,
		.name = "IMU",
	};

	return read_fifo(dev_ctxt, &cfg);
}

static int read_ts_data(struct i2c_client *client, int num_of_ts_entries)
{
	struct device_context *dev_ctxt = i2c_get_clientdata(client);
	struct read_fifo_configuration cfg = {
		.num_entries = num_of_ts_entries,
		.entry_size = TS_ENTRY_SIZE,
		.max_read_size = MAX_TS_READ_SIZE,
		.fifo_address = UCTL_FIFO_DATA_TS,
		.parse_entry = parse_ts_entry,
		.name = "TS",
	};

	return read_fifo(dev_ctxt, &cfg);
}

static int read_fifo(struct device_context *dev_ctxt, const struct read_fifo_configuration *cfg)
{
	int res = 0;
	int i = 0;
	u32 size_to_read = (cfg->num_entries * cfg->entry_size);

	/* It is legal for num entries to be 0. */
	if (cfg->num_entries == 0)
		return 0;

	if (size_to_read > cfg->max_read_size) {
		dev_err(&dev_ctxt->i2c_client->dev, "Read FIFO %s: size to read %d > max %zu\n",
			cfg->name, size_to_read, cfg->max_read_size);
		return -1;
	}

	/* dev_info(&dev_ctxt->i2c_client->dev, "Read FIFO %s: entries to read = %d, size to read = %zu\n",
		cfg->name, cfg->num_entries, size_to_read); */

	res = uctl_i2c_read(dev_ctxt->i2c_client,
		cfg->fifo_address,
		size_to_read,
		dev_ctxt->hw_fifo_read_buffer);

	if (res != 0) {
		dev_err(&dev_ctxt->i2c_client->dev, "Read FIFO %s: Failed to read data with status %d\n", cfg->name, res);
		return res;
	}

	for (i = 0; i < cfg->num_entries; i++) {
		struct stm32l051_sensor_data dest;
		if (cfg->parse_entry(dev_ctxt, &dest, &dev_ctxt->hw_fifo_read_buffer[i * cfg->entry_size]) == 0) {
			if (!dev_ctxt->sensors.controls[dest.handle]->filter_out) {
				if (kfifo_in(&dev_ctxt->fifo, &dest, 1) != 1)
					set_driver_error(dev_ctxt, STM32L051_ERROR_DRV_FIFO_FULL);
			}
		}
	}

	return res;
}

struct reg_fifo_rd {
	u32 num_of_ts_entries:6;
	u32 num_of_imu_entries:6;
	u32 reserved:20;
};

static int read_hw_fifo(struct i2c_client *client)
{
	int res = 0;
	struct reg_fifo_rd fifo_rd_reg;
#if STATISTICS_ENABLED
	u64 time = 0;
#endif /*STATISTICS_ENABLED*/

	/* dev_info(&client->dev, "%s ==>", __func__); */

	res = uctl_read_reg(client,
				UCTL_REG_FIFO_RD,
				UCTL_32BIT,
				(u32 *)&fifo_rd_reg);
	if (res == 0) {
		/* dev_info(&client->dev, "%s: num of ts entries %d, num of imu entries %d\n",
			__func__, fifo_rd_reg.num_of_ts_entries, fifo_rd_reg.num_of_imu_entries); */

		if ((fifo_rd_reg.num_of_ts_entries == 0) && (fifo_rd_reg.num_of_imu_entries == 0)) {
			/*dev_err(&client->dev, "%s: IMU and TS FIFO's are empty\n", __func__);*/
			return 0;
		}
	} else {
		dev_err(&client->dev, "%s: Failed to read UCTL_REG_FIFO_RD with status %d\n", __func__, res);
		return res;
	}

	statistic_accumulator_add(&dev_ctxt->statistics.imu_entries, fifo_rd_reg.num_of_imu_entries);
	statistic_accumulator_add(&dev_ctxt->statistics.ts_entries, fifo_rd_reg.num_of_ts_entries);

	update_time_delta(&time, NULL);
	res = read_imu_data(client, fifo_rd_reg.num_of_imu_entries);
	if (res != 0) {
		dev_err(&client->dev, "%s: Failed to read IMU data with status %d\n", __func__, res);
		return res;
	}
	update_time_delta(&time, &dev_ctxt->statistics.read_imu_fifo_delta);

	res = read_ts_data(client, fifo_rd_reg.num_of_ts_entries);
	if (res != 0) {
		dev_err(&client->dev, "%s: Failed to read TS data with status %d\n", __func__, res);
		return res;
	}


	update_time_delta(&time, &dev_ctxt->statistics.read_ts_fifo_delta);

	/* dev_info(&client->dev, "%s <==", __func__); */

	return fifo_rd_reg.num_of_ts_entries + fifo_rd_reg.num_of_imu_entries;
}

static irqreturn_t uctl_hard_irq_handler(int irq, void *dev_id, void *data)
{
	struct device_context *dev_ctxt = dev_id;

	update_time_delta(&dev_ctxt->timing_measurements.hard_irq_time,
			&dev_ctxt->statistics.hard_interrupt_delta);

	if(!dev_ctxt->pmic_powered_on) {
		dev_info(&dev_ctxt->i2c_client->dev, "<== %s ignoring, power not on yet\n", __func__);
		return IRQ_HANDLED;
	}

	return IRQ_WAKE_THREAD;
}

/* interrupt handler */
static irqreturn_t uctl_soft_irq_handler(int irq, void *dev_id, void *data)
{
	struct device_context *dev_ctxt = dev_id;
	int res = 0;

    if (dev_ctxt->uc_timestamp_polling_enabled)
    {
        u64 val = dev_ctxt->ts_overflow_count;
    
        uint32_t regval;
        if ((++dev_ctxt->timestamp_polling_counter %1000) ==0)
        {
            u64 time1= ktime_to_ns(ktime_get());
            u64 time2;

            res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_TIMESTAMP, UCTL_32BIT, &regval);
            if (res != 0) {
                dev_err(&dev_ctxt->i2c_client->dev, "%s returned with status %d\n", __func__, res);
	        }
            time2= ktime_to_ns(ktime_get()) - time1;
            val <<= 32;
 	        val |= le32_to_cpu(regval);
            val = val * dev_ctxt->timer_prescaler * (125 / 4);

            if (!mutex_lock_interruptible(&dev_ctxt->device_handle_lock)) {
                dev_ctxt->last_uc_timestamp_value.uc_time = val;
                dev_ctxt->last_uc_timestamp_value.sys_time = time1;
                dev_ctxt->last_uc_timestamp_value.call_time = time2/1000;
            	mutex_unlock(&dev_ctxt->device_handle_lock);
//                dev_warn(&dev_ctxt->i2c_client->dev,"\nUCTL_REG_TIMESTAMP took: %d us!\n",time2/1000);
    	    }
        }
    }


	dev_ctxt->softirq_time_stamp = ktime_to_ns(ktime_get());
	update_time_delta(&dev_ctxt->timing_measurements.soft_irq_time,
			&dev_ctxt->statistics.soft_interrupt_delta);

	/*if (!dev_ctxt->pmic_powered_on) {
		dev_info(&dev_ctxt->i2c_client->dev, "%s PMIC not powered on, can't access device\n", __func__);
		return IRQ_HANDLED;
	}*/
	/*dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);*/

	/* Canceling the fallback timer, it will be re triggered after the hw events are processed. */
	del_timer_sync(&dev_ctxt->fallback_timer);

	res = process_hw_events(dev_ctxt);
	if(res != 0)
		dev_err(&dev_ctxt->i2c_client->dev, "%s failed to process status register\n", __func__);
	/* Retriggering the fallback timer only when streaming*/
	if (dev_ctxt->state == DEVICE_STATE_STREAMING)
		mod_timer(&dev_ctxt->fallback_timer, jiffies + msecs_to_jiffies(INTERRUPT_FALLBACK_TIMEOUT_MS));

	/* dev_info(&dev_ctxt->i2c_client->dev, "<== %s\n", __func__); */
	return IRQ_HANDLED;
}

#define MAX_READ_HW_EVENTS_ITERATIONS 10
#define MAX_READ_HW_FIFO_ENTRIES 21

static int process_status_register(struct device_context *dev_ctxt)
{
	int res = 0;
	u32 status_register = 0;

	/* Read reg can't guarantee atomic set so reading to local variable and then setting the context member.*/
	res = uctl_read_reg(dev_ctxt->i2c_client,
		UCTL_REG_STATUS_REG,
		UCTL_32BIT,
		&status_register);

	if (res != 0)
		return res;

	if (status_register & UCTL_ERROR_DETECTED) {
		if ((dev_ctxt->state == DEVICE_STATE_IAP) || (dev_ctxt->state == DEVICE_STATE_FW_CORRUPTED)) {
			/* Edge case scenario, received an interrupt while FW is in IAP or corrupted. */
			/* There is a very minimal set of registers that's supported (power state & status) */
			/* Therefore the driver should not proceed processing the status register. */
			dev_warn(&dev_ctxt->i2c_client->dev,
				"%s :FW is in IAP or corrupted (state = %d)\n",
				__func__, dev_ctxt->state);
			dev_ctxt->status_register = status_register;
			return res;
		}

		update_fw_error(dev_ctxt);
	}

	if (status_register & UCTL_POWER_MODE_CHANGE_DONE) {
		dev_info(&dev_ctxt->i2c_client->dev,
			"%s :power mode change interrupt received\n",
			__func__);
		dev_ctxt->power_change_int_received = true;
		wake_up_interruptible(&dev_ctxt->wait_power_up);
		}

	if ((dev_ctxt->state == DEVICE_STATE_STREAMING) && (status_register & UCTL_ONE_FIFO_REACHED_LTR)) {
		int i;
		for (i = 0; i < MAX_READ_HW_EVENTS_ITERATIONS; i++) {
			if (read_hw_fifo(dev_ctxt->i2c_client) < MAX_READ_HW_FIFO_ENTRIES)
				break;
		}
		if (i == MAX_READ_HW_EVENTS_ITERATIONS)
			statistic_accumulator_add(&dev_ctxt->statistics.read_hw_fifo_iterations, MAX_READ_HW_EVENTS_ITERATIONS);
		else
			statistic_accumulator_add(&dev_ctxt->statistics.read_hw_fifo_iterations, (i+1));

		wake_up_interruptible(&dev_ctxt->wait_queue); /* epol. */
		wake_up_interruptible(&dev_ctxt->block_read_wait_queue); /* For block read */
	}
	if ((status_register & UCTL_READY_FOR_NEXT_EEPROM) != (dev_ctxt->status_register & UCTL_READY_FOR_NEXT_EEPROM)) {
		dev_info(&dev_ctxt->i2c_client->dev,
			"%s :eeprom ready status changed to 0x%X\n",
			__func__,
			(status_register & UCTL_READY_FOR_NEXT_EEPROM));
		
	}

	dev_ctxt->status_register = status_register;

	return res;
}


static int process_hw_events(struct device_context *dev_ctxt)
{
	int res = 0;

	/*dev_info(&dev_ctxt->i2c_client->dev, "==> %s\n", __func__);*/
	if (mutex_lock_interruptible(&dev_ctxt->hw_events_lock)) {
		dev_err(&dev_ctxt->i2c_client->dev, "%s: mutex_lock_interruptible failed\n", __func__);
		return -ERESTARTSYS;
	}

	res = process_status_register(dev_ctxt);
	if (res != 0)
		dev_err(&dev_ctxt->i2c_client->dev, "<== %s Failed to process status register with res %d\n", __func__, res);

	mutex_unlock(&dev_ctxt->hw_events_lock);

	return res;
}

static int uctl_irq_init(struct device_context *dev_ctxt)
{
	int ret = 0;
	struct i2c_client *client = dev_ctxt->i2c_client;

	/* register interrupt handler */
	if (client->irq) {
		ret = devm_request_threaded_irq(&client->dev,
						client->irq,
						(irq_handler_t) uctl_hard_irq_handler,
						(irq_handler_t) uctl_soft_irq_handler,
						IRQF_TRIGGER_RISING,
						"uctl",
						dev_ctxt);
		if (ret) {
			dev_err(&client->dev,
				"%s: request_threaded_irq failed %d\n",
				__func__, ret);
		}

		dev_ctxt->interrupt_enabled = true;
		disable_int(dev_ctxt, __func__);
	} else {
		dev_err(&client->dev, "IRQ not set\n");
		ret = -EINVAL;
	}

	return ret;
}
#if 0
/**
 *
 */
static int uctl_setup_cdev(struct device_context *dev_ctxt)
{
	int res = 0;
	struct device *dev;
	int devno = MKDEV(dev_ctxt->major, 0);

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	cdev_init(&dev_ctxt->cdev, &uctl_fops);

	dev_ctxt->cdev.owner = THIS_MODULE;
	dev_ctxt->cdev.ops = &uctl_fops;
	res = cdev_add(&dev_ctxt->cdev, devno, 1);
	if (res) {
		dev_err(&dev_ctxt->i2c_client->dev,
			"%s: Failed adding uctl cdev %d\n",
			__func__, res);
	}

	dev_ctxt->class = class_create(THIS_MODULE, DEVICE_NAME);
	if (IS_ERR(dev_ctxt->class)) {
		res = PTR_ERR(dev_ctxt->class);
		dev_err(&dev_ctxt->i2c_client->dev,
					"%s: class_create failed %d\n",
					__func__, res);
	}

	dev = device_create(dev_ctxt->class,
			NULL, devno, NULL, "%s", DEVICE_NAME);
	if (IS_ERR(dev)) {
		res = PTR_ERR(dev);
		dev_err(&dev_ctxt->i2c_client->dev,
					"%s: device_create failed %d\n",
					__func__, res);
	}

	return res;
}
#endif
/**
 *
 */
static int uctl_acpi_probe(struct device_context *dev_ctxt, struct i2c_client *client)
{
	int ret;
	struct device *device;
	const struct acpi_device_id *acpi_id;
	struct gpio_desc *pa8;
	struct gpio_desc *pa9;
	struct gpio_desc *pa10;

	if (!client) {
		dev_info(&client->dev, "%s: invalid cleint\n", __func__);
		return -EINVAL;
	}
	if (!client->adapter) {
		dev_err(&client->dev, "%s invalid client adapter\n", __func__);
		return -ENODEV;
	}
	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	device = &client->dev;
	if (!ACPI_HANDLE(&client->dev)) {
		dev_err(&client->dev, "%s:ACPI_HANDLE failed\n", __func__);
		return -ENODEV;
	}

	acpi_id = acpi_match_device(device->driver->acpi_match_table, device);
	if (!acpi_id) {
		dev_err(&client->dev, "%s: acpi_match_device failed\n",
			 __func__);
		return -ENODEV;
	}


	pa8 = devm_gpiod_get_index(&client->dev, "stm32l051_pa8", 2, GPIOD_OUT_LOW);
	//ret = gpiod_direction_output(pa8, 0);
	/*if (IS_ERR(pa8)) {
		dev_err(&client->dev, "%s acpi gpio pa8 get index failed\n", __func__);
		// return PTR_ERR(gpio); 
	}*/
	dev_ctxt->pa8_gpiod = pa8;

	pa9 = devm_gpiod_get_index(&client->dev, "stm32l051_pa9", 1, GPIOD_IN);
	//gpiod_direction_output(pa9, 0);
	/*if (IS_ERR(pa9)) {
		dev_err(&client->dev, "%s acpi gpio pa9 get index failed\n", __func__);
		// return PTR_ERR(gpio); 
	}*/
	dev_ctxt->pa9_gpiod = pa9;

	pa10 = devm_gpiod_get_index(&client->dev, "stm32l051_int", 0, GPIOD_IN);
	/*if (IS_ERR(pa10)) {
		dev_err(&client->dev, "%s acpi gpio pa10 get index failed\n", __func__);
		// return PTR_ERR(gpio); 
	}*/
	dev_ctxt->pa10_gpiod = pa10;
	//gpiod_direction_input(pa10);

	ret = desc_to_gpio(pa10);
	client->irq = gpiod_to_irq(gpio_to_desc(ret));
	printk("after gpio indexing");
	if (client->irq < 0) {
		dev_err(&client->dev, "%s: invalid irq for the gpio\n", __func__);
		/* return ret; */
	} else {
		dev_info(&client->dev, "%s: irq = %d\n", __func__, client->irq);
	}

	dev_ctxt->v1p8_reg = regulator_get(&client->dev, "V1P8SX");
	dev_ctxt->v3p3_reg = regulator_get(&client->dev, "V3P3SD");

	return 0;
}

static int init_sysfs(struct device_context *dev_ctxt)
{
	int res = 0;

	dev_info(&dev_ctxt->i2c_client->dev, "%s\n", __func__);

	dev_ctxt->idev = input_allocate_device();
	if (!dev_ctxt->idev) {
		dev_err(&dev_ctxt->i2c_client->dev, "input_allocate_device failed");
		return -ENOMEM;
	}

	dev_ctxt->idev->name = DEVICE_NAME;
	dev_ctxt->idev->id.bustype = BUS_I2C;

	input_set_drvdata(dev_ctxt->idev, dev_ctxt);
	res = input_register_device(dev_ctxt->idev);
	if (res < 0)
		dev_err(&dev_ctxt->i2c_client->dev, "input_register_device failed with status %d", res);

	res = sysfs_create_group(&dev_ctxt->idev->dev.kobj,
		&stm32l051_attribute_group);
	if (res < 0)
		dev_err(&dev_ctxt->i2c_client->dev, "sysfs_create_group failed with status %d", res);

	return res;
}

static int query_timer_prescaler(struct device_context *dev_ctxt)
{
	int res = 0;
	u32 reg_val = 0;

	/* SAS defines prescaler range as (1 :: 65536) */
	/* FW requires decreasing prescaler by 1 (0 :: 65535). */

	res = uctl_read_reg(dev_ctxt->i2c_client, UCTL_REG_TIMER_PRESCALER, UCTL_32BIT, &reg_val);
	if(res == 0)
		dev_ctxt->timer_prescaler = reg_val + 1;

	return res;
}


/**
 *
 */
static int uctl_probe(struct i2c_client *client,
		const struct i2c_device_id *id)
{
	int res = 0;
	struct device_context *dev_ctxt;
	struct i2c_adapter *adapter;

	if (!client) {
		dev_err(&client->dev, "%s: invalid cleint\n", __func__);
		return -EINVAL;
	}

	adapter = to_i2c_adapter(client->dev.parent);
	if (!adapter) {
		dev_err(&client->dev, "%s: invalid client adapter\n", __func__);
		return -EINVAL;
	}
	if (!i2c_check_functionality(adapter, I2C_FUNC_I2C)) {
		dev_err(&client->dev,
			"%s: i2c_check_functionlity failed\n", __func__);
		/* TODO:
		return -EIO; */
	}

	/* allocate uctl device, which includes device state */
	dev_ctxt = devm_kzalloc(&client->dev,
			sizeof(struct device_context),
			GFP_KERNEL);
	if (!dev_ctxt) {
		dev_err(&client->dev, "%s allocating device_context failed\n",
			__func__);
		return -ENOMEM;
	}

	/* initialize device members */
	mutex_init(&dev_ctxt->device_handle_lock);
	mutex_init(&dev_ctxt->hw_events_lock);
	dev_ctxt->pmic_powered_on = false;
	dev_ctxt->interrupt_enabled = false;
	reset_statistics(dev_ctxt);
	reset_timing_measurements(dev_ctxt);
	init_waitqueue_head(&dev_ctxt->wait_queue);
	init_waitqueue_head(&dev_ctxt->wait_power_up);
	init_waitqueue_head(&dev_ctxt->block_read_wait_queue);
	dev_ctxt->i2c_client = client;
    client->addr = (unsigned short)i2c_addr;

	init_device_status(dev_ctxt);
	dev_ctxt->status_register = 0;
	INIT_KFIFO(dev_ctxt->fifo);
	setup_timer(&dev_ctxt->fallback_timer, fallback_timer_handler, (unsigned long)dev_ctxt);
	dev_ctxt->state = DEVICE_STATE_CLOSED; /* This is not a real "set state", but just an init to the state machine. */

	i2c_set_clientdata(client, dev_ctxt);
	
	cdev_init(&dev_ctxt->cdev, &uctl_fops);
	dev_ctxt->cdev.owner = THIS_MODULE;
	res = cdev_add(&dev_ctxt->cdev, MKDEV(uctl_major, uctl_base_minor+uctl_live_devs), 1);
	if (res) {
		dev_err(&dev_ctxt->i2c_client->dev,
			"%s: Failed adding uctl cdev %d\n",
			__func__, res);
		goto error0;
	}
	dev_ctxt->chrdev = device_create(uctl_dev_class, &adapter->dev,
					 dev_ctxt->cdev.dev, dev_ctxt,
					 DEVICE_NODE_NAME, MINOR(dev_ctxt->cdev.dev));
	if (IS_ERR(dev_ctxt->chrdev)) {
		res = PTR_ERR(dev_ctxt->chrdev);
		goto error0;
	}
	res = uctl_acpi_probe(dev_ctxt, client);
	if (res)
		goto error1;
	res = uctl_irq_init(dev_ctxt);
	if (res)
		goto error2;


	res = init_sysfs(dev_ctxt);
	if (res)
		goto error3;

	uctl_live_devs++;
	return 0;
#if 0
error4:
	sysfs_remove_group(&dev_ctxt->idev->dev.kobj, &stm32l051_attribute_group);
#endif
error3:
	devm_free_irq(&client->dev, client->irq, dev_ctxt);
error2:
	regulator_put(dev_ctxt->v3p3_reg);
	regulator_put(dev_ctxt->v1p8_reg);
error1:
	device_destroy(uctl_dev_class, MKDEV(uctl_major, adapter->nr));
	cdev_del(&dev_ctxt->cdev);
error0:
	mutex_destroy(&dev_ctxt->device_handle_lock);
	mutex_destroy(&dev_ctxt->hw_events_lock);
	destroy_device_status(dev_ctxt);
	devm_kfree(&client->dev, dev_ctxt);
	return res;
}

/**
 *
 */
static int uctl_remove(struct i2c_client *client)
{
	struct device_context *dev_ctxt;
	int delay = 1;
	struct i2c_adapter *adapter;

	dev_info(&client->dev, "uctl remove\n");
	if (!client) {
		dev_err(&client->dev, "%s: invalid client\n", __func__);
		return -EINVAL;
	}
	adapter = to_i2c_adapter(client->dev.parent);
	if (!adapter) {
		dev_err(&client->dev, "%s: invalid client adapter\n", __func__);
		return -EINVAL;
	}
	dev_ctxt = i2c_get_clientdata(client);
	if (!dev_ctxt) {
		dev_err(&client->dev, "%s: invalid device context\n", __func__);
		return -EINVAL;
	}
	stm32l051_unloading = 1;
	while (waitqueue_active(&dev_ctxt->wait_queue)) {
		dev_info(&client->dev, "%s waiting for poll to end\n", __func__);
		wake_up_interruptible(&dev_ctxt->wait_queue);
		msleep(delay);
		delay += delay;
	}
	regulator_put(dev_ctxt->v3p3_reg);
	regulator_put(dev_ctxt->v1p8_reg);
	device_destroy(uctl_dev_class, dev_ctxt->cdev.dev);
	cdev_del(&dev_ctxt->cdev);
	uctl_live_devs--;
	/* unregister chrdev */
	sysfs_remove_group(&dev_ctxt->idev->dev.kobj, &stm32l051_attribute_group);
	destroy_device_status(dev_ctxt);
	mutex_destroy(&dev_ctxt->device_handle_lock);
	mutex_destroy(&dev_ctxt->hw_events_lock);
	devm_kfree(&client->dev, dev_ctxt);
	return 0;
}

/**
 *
 */
static void uctl_shutdown(struct i2c_client *client)
{
	dev_info(&client->dev, "uctl shutdown\n");
}

/**
 * ACPI
 */
static const struct i2c_device_id uctl_id[] = {
	{ DEVICE_NAME, 0 },
	{ }
};

MODULE_DEVICE_TABLE(i2c, uctl_id);
static struct acpi_device_id uctl_acpi_match[] = {
	{ "INT35EE", 0},
	{},
};
MODULE_DEVICE_TABLE(acpi, uctl_acpi_match);

/* I2C driver */
static struct i2c_driver uctl_driver = {
	.driver = {
		.owner	= THIS_MODULE,
		.name	= DEVICE_NAME,
		.acpi_match_table = ACPI_PTR(uctl_acpi_match),
	},
	.id_table	= uctl_id,
	.probe		= uctl_probe,
	.remove		= uctl_remove,
	.shutdown	= uctl_shutdown,
};

/**
 *
 */
static int __init uctl_init(void)
{
	int res;
	dev_t devid;
	
	res = alloc_chrdev_region(&devid, 0, 2, "uctl");
	if (res < 0) {
		return res;
	}
	uctl_major = MAJOR(devid);
	uctl_base_minor = MINOR(devid);
	uctl_dev_class = class_create(THIS_MODULE, "uctl-dev");
	if (IS_ERR(uctl_dev_class)) {
		res = PTR_ERR(uctl_dev_class);
		goto out;
	}
	res = i2c_add_driver(&uctl_driver);
	if(res == 0)
		return 0;
/* error handling */
	class_destroy(uctl_dev_class);
out:
	unregister_chrdev_region(MKDEV(uctl_major, uctl_base_minor), 2);
	return res;
}

/**
 *
 */
static void __exit uctl_exit(void)
{
	i2c_del_driver(&uctl_driver);
	class_destroy(uctl_dev_class);
	unregister_chrdev_region(MKDEV(uctl_major, uctl_base_minor), 2);
}

module_init(uctl_init);
module_exit(uctl_exit);
