# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import os
import gobject
import gtk
from shutil import copyfile
import gpod
from time import sleep

import stock
import utils
import vfs
from config import config
import threading

from widget.jobs_manager import Job
from widget.dialog import WindowBase
from cover_manager import CoverManager
from hal import hal
from xdg_support import get_xdg_pixmap_file

from source._device import DeviceWrapper, DeviceSource, DeviceSourceItem, DeviceWrapperInitException, DeviceBrowser

from library import ListenDB
from transcode import Transcoder,TrancoderError

import socket
import locale

defaultencoding = locale.getpreferredencoding()

IPOD_IDS = {
    "title":"@title",
    "album":"@album",
    "artist":"@artist",
    "#duration":"tracklen",
    "genre":"@genre",
    "#size":"size",
    "#track":"#track_nr"
}


AUTOSAVE_TIMEOUT = 5000
USE_BUILTIN_TRACK_COPY = True


class IpodTransfertJob(Job):
    __message = _("Loading...")
    def __init__(self,dw):
        
        self.cond = threading.Condition()
        
        self.__current_transcode = None
        self.__transfer_list = []
        self.__dw = dw
        self.uri_convert = None
        
        super(IpodTransfertJob,self).__init__()
        
    def add_item(self,song,callback,*args_cb,**kwargs):
        self.cond.acquire()
        self.__transfer_list.append((song,callback,args_cb,kwargs))
        self.cond.notify()
        self.cond.release()
        
    def job(self):
        i = 0
        
        self.loginfo("%s queued song ready to transfer",len(self.__transfer_list))
        while True:
            self.cond.acquire()
            if not self.__transfer_list:
                self.cond.wait(5)
            if not self.__transfer_list: 
                break;
            total = len(self.__transfer_list) + i 
            song, callback, args_cb, kwargs = self.__transfer_list.pop(0)
            self.cond.release()
            
            tmp_podcast_uri = "file:///tmp/listenpodcasttmp.mp3"
            i += 1
            ispodcast = song.get("podcast_feed_url")
            if ispodcast:
                if song.get("podcast_local_uri") and \
                   vfs.get_scheme(song.get("podcast_local_uri")) == "file" and \
                   vfs.exists(song.get("podcast_local_uri")):
                    uri_to_transfert = song.get("podcast_local_uri")
                else:
                    # Taken from podcast plugins
                    try: os.unlink("/tmp/listentmp.mp3")
                    except: pass
                    uri = song.get("uri")
                    try:
                        for total_size, receive_size, percent in vfs.download_iterator(uri, tmp_podcast_uri):
                            yield _("Transfering files")+" %d/%d"%(i,total)+" (Download podcast... %d%%)"%percent,float(i)/float(total),False    
                    except IOError:
                        continue
                    uri_to_transfert = tmp_podcast_uri
            else:
                uri_to_transfert = song.get("uri")
            
            self.loginfo("transfert: %s",uri_to_transfert)

            if vfs.get_ext(uri_to_transfert) != ".mp3":
                try:
                    tmp_uri = uri_to_transfert[:uri_to_transfert.rfind("/")+1]+".listen-"+uri_to_transfert[uri_to_transfert.rfind("/")+1:uri_to_transfert.rfind(".")]+".mp3"
                except:
                    try: os.unlink("/tmp/listentmp.mp3")
                    except: pass
                    tmp_uri = "file:///tmp/listentmp.mp3"
                if not vfs.exists(tmp_uri):
                    self.uri_convert = tmp_uri
                    try:self.__current_transcode = Transcoder(uri_to_transfert,tmp_uri)
                    except TrancoderError:
                        try: os.unlink("/tmp/listentmp.mp3")
                        except: pass
                        tmp_uri = "file:///tmp/listentmp.mp3"
 
                        try:self.__current_transcode = Transcoder(uri_to_transfert,tmp_uri)
                        except TrancoderError:    
                            self.logerror("Failed transcode %s",uri_to_transfert)
                            continue
                    
                    while not self.__current_transcode.is_finish():
                        sleep(1)
                        # recalculate the total while transcoding
                        self.cond.acquire()
                        total = len(self.__transfer_list) + i 
                        self.cond.release()
                        yield _("Transfering files")+" %d/%d"%(i,total)+" (Encoding... %d%%)"%self.__current_transcode.get_ratio(),float(i)/float(total),False    
           
                    self.uri_convert = None
            else:
                tmp_uri = uri_to_transfert
            

            track = gpod.itdb_track_new()

            track.filetype = "MPEG audio file"
            track.time_added = gpod.itdb_time_get_mac_time()
            track.time_modified = gpod.itdb_time_get_mac_time()

            track.size = long(song.get("#size",0))

            if ispodcast:
                track.skip_when_shuffling = 0x01
                track.remember_playback_position = 0x01
                track.flag4 = 0x01 # Show Title/Album on the 'Now Playing' page
                track.podcasturl = str(song.get("uri"))
                track.podcastrss = str(song.get("podcast_feed_url"))
                track.time_released = int(utils.time_unix_to_mac(song.get("#date",0)))

            else:
                track.skip_when_shuffling = 0x00
                track.remember_playback_position = 0x00
                track.flag4 = 0x00 # Show Title/Album/Artist on the 'Now Playing' page

            filename = str(vfs.get_path_from_uri(tmp_uri))

            for ltag,itag in IPOD_IDS.iteritems():
                real_itag = itag.replace("@","").replace("#","")
                value = song.get(ltag)
                if itag.find("@") == 0:
                    if value:
                        value = str(value)
                    elif itag == "@title":
                        value = str(song.get_filename())
                    else:
                        value = ""
                if itag.find("#") == 0 and value == 0:
                    continue
                if ltag.find("#") == 0:
                    if value is None:
                        value = 0
                    else:
                        value = int(value)
                if value:
                    setattr(track,real_itag,value)


            gpod.itdb_track_add(self.__dw.get_itdb(), track, -1)
            if gpod.itdb_device_supports_artwork(self.__dw.get_itdb().device):
                if CoverManager.has_cover(song):
                    art = CoverManager.get_cover(song, False)
                    gpod.itdb_track_set_thumbnails(track,str(art))

            if ispodcast:
                gpod.itdb_playlist_add_track(self.__dw.iget_playlist_podcasts(),track,-1)
            else:
                gpod.itdb_playlist_add_track(self.__dw.iget_playlist_mpl(),track,-1)


            if USE_BUILTIN_TRACK_COPY:
                # My hack to have a threaded version of itdb_track_copy_to_ipod
                try: 
                    dest_filename = gpod.itdb_cp_get_dest_filename(track, None, filename, None)
                    if not dest_filename:
                        raise IOError("gpod.itdb_cp_get_dest_filename call fail")
                    vfs.makedirs("/".join(dest_filename.split("/")[:-1]))
                    copyfile(filename, dest_filename)
                    if not gpod.itdb_cp_finalize(track, None, str(dest_filename), None):
                        raise IOError("gpod.itdb_cp_finalize call fail")
                except IOError:
                    self.logexception("failed copy %s to %s", filename, dest_filename)
                    yield _("Transfering files")+" %d/%d"%(i,total), float(i)/float(total), False
                    continue
            else: 
                # That don't work good with thread ie: UI freeze
                gpod.itdb_cp_track_to_ipod(track, str(filename), None)

            path = gpod.itdb_filename_on_ipod(track)
            song["uri"] = "file://"+path
            self.__dw.add_to_cache(song["uri"],track)
                        

            self.cond.acquire()
            self.__dw.add_to_verif_cache(song)
            callback(song,*args_cb,**kwargs)
            self.cond.release()

            try: vfs.unlink(tmp_podcast_uri)
            except: pass

            yield _("Tansfering files")+" %d/%d"%(i,total),float(i)/float(total),False
        
        self.loginfo("%d song already transfered",i)

    def __job_cb(self,cb,song,*args_cb,**kwargs):
            cb(song,*args_cb,**kwargs)

    def on_stop(self):
        if self.__current_transcode:
            self.__current_transcode.stop()
        self.__current_transcode = None
        if self.uri_convert:
            try: os.unlink(vfs.get_path_from_uri(self.uri_convert))
            except: pass
        self.cond.acquire()
        self.loginfo("%d song deleted from queue",len(self.__transfer_list))

        # Ugly Idea
        self.__dw.songs_failed_transfer = self.__transfer_list

        self.__transfer_list = []
        self.cond.release()
        gobject.idle_add(self.__idle_on_stop)
        
    def __idle_on_stop(self):
        self.__dw.set_dirty()

class IpodPropertyWindow(WindowBase):
    def __init__(self,dw):
        WindowBase.__init__(self,_("iPod Property"), get_xdg_pixmap_file("ipod_big.png"),
                (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE) )

        info = gpod.sw_ipod_device_to_dict(dw.get_itdb().device)
        model = info["sysinfo"].get("ModelNumStr",_("Unknown"))
        firmware = info["sysinfo"].get("visibleBuildID",_("Unknown"))
        size = hal.get_property(dw.udi,"volume.size")
        strsize = utils.str_size(size ,0, 1000)
        self.label_heading.set_markup( "<span size=\"large\"><b>iPod %s (%s)</b></span>\n%s: %s"
                %(model, strsize, _("Firmware"), firmware) )

        l = gtk.Label(_("Name")+" : ")
        e = gtk.Entry()
        e.set_text(dw.get_name())
        def rename(e): dw.set_name(e.get_text())
        e.connect("changed",rename)
        hbox = gtk.HBox(0,6)
        hbox.pack_start(l,False,False)
        hbox.pack_start(e)
        
        vfsinfo = os.statvfs(dw.get_itdb().mountpoint)
        free = vfsinfo[0] * vfsinfo[3] # bsize * bfree
        used = size - free
        frac = float(used) / float(size)
        pg = gtk.ProgressBar()
        pg.set_fraction(frac)
        pg.set_text("%s Free space"%utils.str_size(free,1))
        self.add_widget(hbox)
        self.add_widget(pg)
        self.show_all()

class IpodDevice(DeviceWrapper):
    capability = [ "writable", "playlist" ,"property", "podcast" ]

    def __init__(self,udi):
        
        DeviceWrapper.__init__(self,udi)
        m = str(hal.get_mount_point(udi))

        if not m: raise DeviceWrapperInitException()

        self.__itdb = gpod.itdb_parse(m,None)
        if not self.__itdb:
            gpod.itdb_init_ipod(m, None, None , None)
            self.__itdb = gpod.itdb_parse(m,None)
            if not self.__itdb:
                raise DeviceWrapperInitException()
        #self.loginfo("detected ipod %s",gpod.sw_ipod_device_to_dict(self.__itdb.device))
        self.__udi = udi 
        self.__itdb.mountpoint = m
        self.__itdb_file = str(os.path.join(m, "iPod_Control","iTunes","iTunesDB"))

        
        self._verif_cache = {}
        self.__track_cache = {}
        self.__playlist_cache = {}
        self.__ipod_transfert_job = None
        self.__autosave_id = None
        self.__remove_songs_cb_id = None
        self.__dirty = False

        # contient tuble with (songs, callback,*args)
        self.__action_waiting_transfert = ()


        # Ugly idea, JobTransfert , set this var to known file not transfered
        self.songs_failed_transfer = []

    def load_data(self):
        ListenDB.set_db_wrapper_load("device_" + self.udi,True)
        ListenDB.set_db_wrapper_load("device_" + self.udi + "_podcast",True)
        pl = self.iget_playlist_mpl()
        [ self.add_to_verif_cache(ListenDB.get_or_create_song(tags, "device_" + self.udi)) for tags in self.__get_all_tags(pl) ]
        pl = self.iget_playlist_podcasts()
        [ self.add_to_verif_cache(ListenDB.get_or_create_song(tags, "device_" + self.udi + "_podcast")) for tags in self.__get_all_tags(pl) ]
        [ self.addplaylist_cache(pl,ListenDB.create_playlist("device_" + self.udi, pl.name, \
                [ ListenDB.get_song("file://"+gpod.itdb_filename_on_ipod(t)) for t in gpod.sw_get_playlist_tracks(pl) if gpod.itdb_filename_on_ipod(t)])) \
                for pl in gpod.sw_get_playlists(self.__itdb) \
                if not gpod.itdb_playlist_is_mpl(pl) and not gpod.itdb_playlist_is_podcasts(pl) ]

        ListenDB.set_db_wrapper_load("device_" + self.udi,False)
        ListenDB.set_db_wrapper_load("device_" + self.udi + "_podcast",False)
        gobject.timeout_add(500, self.check_pending_action)

    def iget_playlist_mpl(self):
        pl = gpod.itdb_playlist_mpl(self.__itdb)
        if pl == None:
            self.logwarn("ipod incorrectly initialized")
            pl = gpod.itdb_playlist_new(_("iPod"),False)
            gpod.itdb_playlist_set_mpl(pl)
            gpod.itdb_playlist_add(self.__itdb,pl,-1)
        return pl

    def iget_playlist_podcasts(self):
        pl = gpod.itdb_playlist_podcasts(self.__itdb)
        if pl == None:
            pl = gpod.itdb_playlist_new(str(_("Podcasts")),False)
            gpod.itdb_playlist_set_podcasts(pl)
            gpod.itdb_playlist_add(self.__itdb,pl,-1)
        return pl

    def show_property(self, menuitem):
        d = IpodPropertyWindow(self)
        d.run()
        d.destroy()


    def add_to_cache(self,uri,track):
        self.__track_cache[uri] = track

    def add_to_verif_cache(self,song):
        self._verif_cache[(song.get("title",""),song.get("album",""),song.get("artist",""),song.get("#duration",""))] = song

    def addplaylist_cache(self,ipodpl,pl):
        self.__playlist_cache[pl.id] = ipodpl
    
    @utils.threaded
    def __autosave(self):
        self.loginfo("iPod database saving...")
        if self.__autosave_id:
            gobject.source_remove(self.__autosave_id)
            self.__autosave_id = None
        if not gpod.itdb_write_file(self.__itdb, self.__itdb_file, None):
            self.logerror("failed write ipod database")
        self.__dirty = False
        self.loginfo("iPod database saved")
        return True

    def save(self):
        if self.__dirty:
            self.__autosave()
            
    def set_dirty(self):
        self.__dirty = True
        if self.__autosave_id:
            gobject.source_remove(self.__autosave_id)
            self.__autosave_id = None
        self.__autosave_id = gobject.timeout_add(AUTOSAVE_TIMEOUT,self.__autosave)

    def get_itdb(self):
        return self.__itdb

    def set_name(self,name):
        gpod.itdb_playlist_mpl(self.__itdb).name = str(name)
        self.set_dirty()
        #from helper import Dispatcher
        #Dispatcher.update_source()

    def get_podcast_name(self):
        return _("Podcasts")

    def get_podcast_stock(self):
        return stock.SRC_PODCAST_IPOD

    def get_name(self):
        return gpod.itdb_playlist_mpl(self.__itdb).name

    def get_stock(self):
        return stock.SRC_IPOD

    def get_authorized_types(self):
        return ["local","local_unknown", "podcast"]

    def destroy(self):
        self._verif_cache = {}
        if self.__dirty:
            if self.__autosave_id:
                gobject.source_remove(self.__autosave_id)
            if not gpod.itdb_write_file(self.__itdb, self.__itdb_file, None):
                self.logerror("failed write ipod database")
        self.__itdb = None
        DeviceWrapper.destroy(self)

    def __get_all_tags(self, pl):
        tags_list = []
        for track in gpod.sw_get_playlist_tracks(pl):
            if not gpod.itdb_filename_on_ipod(track): continue
            tags = self.__get_tags_from_track(track)
            if tags: 
                self.add_to_cache(tags["uri"],track)
                tags_list.append(tags)
        return tags_list

    def check(self,udi):
        udi_parent = hal.get_property(udi,"info.parent")
        if udi_parent and hal.get_property(udi_parent,"storage.model") == "iPod":
            return True
        return False  

    def get_song_in_cache(self,song):
        try: song = self._verif_cache[(song.get("title",""),song.get("album",""),song.get("artist",""),song.get("#duration",""))]
        except KeyError:
            return None
        return song

    def check_pending_action(self):
        next_action_waiting_transfert = []
        for songs, cb, args in self.__action_waiting_transfert:
            finish = True
            newsongs = [ self.get_song_in_cache(song) for song in songs ]
            newsongs = [ song for song in newsongs if song is not None ]
            failedsongs = [ song for song in self.songs_failed_transfer if song in songs]
            if len(newsongs) + len(failedsongs) == len(songs): cb(newsongs,*args)
            else: next_action_waiting_transfert.append((songs,cb,args))
        self.__action_waiting_transfert = next_action_waiting_transfert

        # Ugly idea :(
        self.songs_failed_transfer = []
        return True

    """ Library Wrapper Func """
    def set_playlist_name(self,pl,name):
        try: ipod_pl = self.__playlist_cache[pl.id]
        except KeyError:
            return False
        else:
            ipod_pl.name = str(name)
            self.set_dirty()
            return True

    def add_playlist(self,type, pl,callback,*args_cb):
        songs = pl.get_songs()
        ipod_pl = gpod.itdb_playlist_new(str(pl.name),False)
        gpod.itdb_playlist_add(self.__itdb,ipod_pl,-1)
        self.__playlist_cache[pl.id] = ipod_pl

        if len(songs) == 0:
            self.set_dirty()
            callback(type,pl,*args_cb)
        else:
            [ ListenDB.get_or_create_song(song, "device_" + self.udi) for song in songs if not self.__track_cache.has_key(song["uri"]) ]
            args_cb = (type, pl , callback) + args_cb
            self.__action_waiting_transfert.append( (songs, self.__add_playlist_cb, args_cb) )

    def __add_playlist_cb(self, songs, type, pl, callback, *args_cb):
        try: ipod_pl = self.__playlist_cache[pl.id]
        except KeyError:
            return
        for song in songs:
            track = self.__track_cache[song["uri"]]
            gpod.itdb_playlist_add_track(ipod_pl,track,-1)
        self.set_dirty()
        callback(type, pl, *args_cb)

    def del_playlist(self,type,pl,callback,*args_cb):
        try: ipod_pl = self.__playlist_cache[pl.id]
        except KeyError:
            return
        gpod.itdb_playlist_remove(ipod_pl)
        self.set_dirty()
        callback(type, pl , *args_cb)


    def playlist_update(self,pl,callback,*args_cb):
        ipod_pl = self.__playlist_cache[pl.id]
        name = ipod_pl.name
        gpod.itdb_playlist_remove(ipod_pl)
        ipod_pl = gpod.itdb_playlist_new(str(name),False)
        gpod.itdb_playlist_add(self.__itdb,ipod_pl,-1)
        self.__playlist_cache[pl.id] = ipod_pl
        self.set_dirty()

        songs = pl.get_songs()

        if len(songs) == 0:
            self.set_dirty()
            callback([],*args_cb)
        else:
            [ ListenDB.get_or_create_song(song, "device_" + self.udi) for song in songs if not self.__track_cache.has_key(song["uri"]) ]
            args_cb = (ipod_pl, callback) + args_cb
            self.__action_waiting_transfert.append( (songs, self.__playlist_update_cb, args_cb) )

    def __playlist_update_cb(self, songs, ipod_pl, callback, *args_cb):
        for song in songs:
            track = self.__track_cache[song["uri"]]
            gpod.itdb_playlist_add_track(ipod_pl,track,-1)
        self.set_dirty()
        uris = [ "file://"+gpod.itdb_filename_on_ipod(t) for t in gpod.sw_get_playlist_tracks(ipod_pl) if gpod.itdb_filename_on_ipod(t)]
        callback(uris,*args_cb)

    def add_song(self,song,callback,*args_cb,**kwargs):
        use_cb_when_exist = kwargs.get("use_cb_when_exist",False)
        try: del kwargs["use_cb_when_exist"]
        except KeyError: pass
        if self.get_song_in_cache(song) is not None:
            if use_cb_when_exist: callback(song,*args_cb,**kwargs)
        else:
            if not self.__ipod_transfert_job or not self.__ipod_transfert_job.is_alive():
                self.__ipod_transfert_job = IpodTransfertJob(self)
            self.__ipod_transfert_job.add_item(song,callback,*args_cb,**kwargs)


    def remove_song(self,song,callback,*args_cb,**kwargs):
        try: 
            track = self.__track_cache[song["uri"]]
            del self._verif_cache[(song.get("title",""),song.get("album",""),song.get("artist",""),song.get("#duration",""))]
        except KeyError:
            self.logwarn("ipod track not found %s",song["uri"])
        else:
            for pl in gpod.sw_get_playlists(self.__itdb):
                while gpod.itdb_playlist_contains_track(pl, track):
                    gpod.itdb_playlist_remove_track(pl, track)
            gpod.itdb_track_unlink(track)
            try: vfs.unlink(song["uri"])
            except: pass
            del self.__track_cache[song["uri"]]
        self.set_dirty()

        if self.__remove_songs_cb_id:
            gobject.source_remove(self.__remove_songs_cb_id)
        self.__remove_songs_cb_id = gobject.timeout_add(100, self.__remove_songs_cb)

        callback(song,*args_cb)

    def __remove_songs_cb(self):
        if self.__remove_songs_cb_id:
            self.__remove_songs_cb_id = None
        for pl in ListenDB.get_playlists("device_"+self.udi):
            pl.update()

    def __repair_ipod(self):
        # Check in podcast playlist
        p = self.iget_playlist_mpl()
        for t in gpod.sw_get_tracks(self.__itdb):
            if not gpod.itdb_playlist_contains_track(p,t):
                gpod.itdb_playlist_add_track(p,t,-1)
    
    def __get_tags_from_track(self,track):
        try:
            tags = {}
            def decode_tag(tag):
                if isinstance(tag, str):
                    return tag.decode('utf-8', "replace")
                return tag
            for ltag, itag in IPOD_IDS.iteritems():
                real_itag = itag.replace("@","").replace("#","")
                value = getattr(track,real_itag)
                if itag.find("@") == 0:
                    value = decode_tag(value)
                if itag.find("#") == 0 and value == 0:
                    continue
                if ltag.find("#") == 0:
                    value = int(value)
                tags[ltag] = value

            
            path = gpod.itdb_filename_on_ipod(track)
            if not path :
                self.logwarn("failed read a track (%s)",tags)
                return None
            tags["uri"] = "file://"+path
            tags["#size"] = os.path.getsize(path)

            tags["ext_data"]= {}
            tags["ext_data"]["ipod_track"] = track

            tags["podcast_feed_url"] = track.podcastrss
        except :
            tags = None
            try: self.logexception("failed to load track: %s",track)
            except : self.logexception("failed to load a track")

        return tags

class IpodSource(DeviceSource):
    PLUGIN_NAME = "Ipod"
    PLUGIN_DESC = "Support for ipod device"
    DEVICE_WRAPPER =  IpodDevice
    display_index = 51
    signal_collector_id = "ipod_source"

