
/*
 ***********************************************************************
 * Copyright 2009-2010 by Freescale Semiconductor, Inc.
 * All modifications are confidential and proprietary information
 * of Freescale Semiconductor, Inc. ALL RIGHTS RESERVED.
 ***********************************************************************
 */
 
/**
 * @file MP4Parser.h
 * @MP4 core parser API header file
 */

#ifndef FSL_MP4_PARSER_API_HEADER_INCLUDED
#define FSL_MP4_PARSER_API_HEADER_INCLUDED

/* common files of mmalyer */
#include "common/fsl_types.h"
#include "common/file_stream.h"
#include "common/fsl_media_types.h"
#include "common/fsl_parser.h"


/***************************************************************************************
*                  MP4 parser specific error codes 
*
*  Note: 
*  (a)For parsers' common error codes and argument types, please refer to "fsl_parser.h".
*  (b)API functions do not use enumeration types but int32 as return values. 
*     It's because different compilers can treat enum types as different basic data types 
*     (eg.int or unsigned char).
***************************************************************************************/

enum
{
    MP4HasRootOD            = 102,
        
    MP4EOF                  = PARSER_EOS,
    MP4NoErr                = PARSER_SUCCESS,
    MP4FileNotFoundErr      = PARSER_FILE_OPEN_ERROR,
    MP4BadParamErr          = PARSER_ERR_INVALID_PARAMETER,
    MP4NoMemoryErr          = PARSER_INSUFFICIENT_MEMORY,
    MP4IOErr                = PARSER_READ_ERROR,
    MP4InvalidMediaErr      = PARSER_ERR_INVALID_MEDIA,
    MP4NotImplementedErr    = PARSER_NOT_IMPLEMENTED,
        
    MP4NoLargeAtomSupportErr  = -105,
    MP4BadDataErr             = -106,
    MP4VersionNotSupportedErr = -107, 
    MP4DataEntryTypeNotSupportedErr = -108,
    MP4NoQTAtomErr                  = -120,    
    MP4_ERR_WRONG_SAMPLE_SIZE = -200
     
};



/***************************************************************************************
*                  Data Structures & Constants
***************************************************************************************/
#define MAX_MP4_TRACKS 64

/*
 * Handle of the MP4 core parser created. 
 */
typedef void * MP4ParserHandle;



/***************************************************************************************
*                  API Funtions
* For calling sequence, please refer to the end of this file.
***************************************************************************************/
#ifdef __cplusplus
#define EXTERN extern "C"
#else
#define EXTERN 
#endif

/**
 * function to get the MP4 core parser version.
 *  
 * @return Version string.
 */
EXTERN const char * MP4ParserVersionInfo();


/**
 * Function to create the MP4 core parser.
 * It will parse the MP4 file header and probe whether the movie can be handled by this MP4 parser.
 * 
 * @param stream [in]   Source stream of the MP4 file.
 *                      It implements functions to open, close, tell, read and seek a file.
 *
 * @param memOps [in]   Memory operation callback table.
 *                      It implements the functions to malloc, calloc, realloc and free memory.
 *
 *
 * @param context [in]  Wrapper context for the callback functions. MP4 parser never modify it.
 * @param parserHandle [out] Handle of the MP4 core parser if succeeds. NULL for failure.
 * @return
 */
EXTERN int32  MP4CreateParser(file_stream_t * stream,                                                        
                                ParserMemoryOps * memOps,                                               
                                void * context,
                                MP4ParserHandle * parserHandle);

/**
 * function to delete the MP4 core parser.
 *
 * @param parserHandle Handle of the MP4 core parser.
 * @return
 */
EXTERN int32 MP4DeleteParser(MP4ParserHandle parserHandle);



/**
 * function to tell whether the movie is seekable. A seekable MP4 movie must have the index table.
 * If the file's index table is loaded from file or imported successfully, it's seekable.
 * Seeking and trick mode can be performed on a seekable file.
 * If the index table is corrupted, the file is NOT seekable. This function will fail and return value can tell the error type. 
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param seekable [out] true for seekable and false for non-seekable 
 * @return 
 */ 
EXTERN int32 MP4IsSeekable(MP4ParserHandle parserHandle, bool * seekable);

/**
 * function to tell how many tracks in the movie. 
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param numTracks [out] Number of tracks.
 * @return 
 */ 
EXTERN int32 MP4GetNumTracks(MP4ParserHandle parserHandle, uint32 * numTracks);



/**
 * function to tell the user data information (title, artist, genre etc) of the movie. User data API.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param userDataId [in] User data ID. Type of the user data.
 * @param buffer [out] Buffer containing the information. The core parser manages this buffer and the user shall NOT free it.
 *                               If no such info is availabe, this value will be set to NULL.
 * @param size [out] Length of the information in bytes. The informaiton is usually a null-terminated ASCII string.
 *                               If no such info is available, this value will be set to 0.
 * @return
 */
EXTERN int32 MP4GetUserData(MP4ParserHandle parserHandle, uint32 userDataId, uint8 ** buffer, uint32 *size);



/**
 * function to tell the movie duration. 
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param usDuration [out] Duration in us.
 * @return 
 */ 
EXTERN int32 MP4GetTheMovieDuration(MP4ParserHandle parserHandle,  uint64 * usDuration);


/**
 * function to tell a track's duration. 
 * The tracks may have different durations. 
 * And the movie's duration is usually the video track's duration (maybe not the longest one).
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param usDuration [out] Duration in us.
 * @return 
 */ 
EXTERN int32 MP4GetTheTrackDuration(MP4ParserHandle parserHandle, uint32 trackNum, uint64 * usDuration);

/**
 * function to tell the type of a track. 
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based.
 *
 * @param mediaType [out] Media type of the track. (video, audio, subtitle...)
 *                        "MEDIA_TYPE_UNKNOWN" means the media type is unknown.
 *
 * @param decoderType [out] Decoder type of the track if available. (eg. MPEG-4, H264, AAC, MP3, AMR ...)
 *                          "UNKNOWN_CODEC_TYPE" means the decoder type is unknown.
 *
 * @param decoderSubtype [out] Decoder Subtype type of the track if available. (eg. AMR-NB, AMR-WB ...)
 *                             "UNKNOWN_CODEC_SUBTYPE" means the decoder subtype is unknown.
 * @return 
 */
EXTERN int32 MP4GetTrackType(  MP4ParserHandle parserHandle, 
                                uint32 trackNum, 
                                uint32 * mediaType,
                                uint32 * decoderType,
                                uint32 * decoderSubtype);


/**
 * function to tell the codec specific information of a track. 
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param data [out] Buffer holding the codec specific information. The user shall never free this buffer.
 * @param size [out] Size of the codec specific information, in bytes.
 * @return 
 */
EXTERN int32 MP4GetDecoderSpecificInfo(MP4ParserHandle parserHandle, 
                                        uint32 trackNum, 
                                        uint8 ** data,
                                        uint32 * size);

/**
 * function to tell the maximum sample size of a track. 
 * MP4 parser read A/V tracks sample by sample. The max sample size can help the user to prepare a big enough buffer.
 * Warning!
 * The "max sample size" is not availbe if the index table is invalid.
 * And for MP4 file, if the index table is invalid, playback will be affected.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param size [out] Max sample size of the track.
 * @return:
 * PARSER_SUCCESS
 * PARSER_ERR_INVALID_MEDIA  Index table is invalid so max sample size is not available.
 */
EXTERN int32 MP4GetMaxSampleSize(MP4ParserHandle parserHandle, uint32 trackNum, uint32 * size);

/**
 * function to tell the language of a track used.
 * This is helpful to select an video/audio/subtitle track or menu pages.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param threeCharCode [out] Three character language code. 
 *                  See ISO 639-2/T for the set of three character codes.Eg. 'eng' for English.
 *                  Four special cases:
 *                  mis- "uncoded languages"
 *                  mul- "multiple languages"
 *                  und- "undetermined language"
 *                  zxx- "no linguistic content"
 * @return 
 */
EXTERN int32 MP4GetLanguage(MP4ParserHandle parserHandle, uint32 trackNum, uint8 *threeCharCode);

/* NOT implemented */
EXTERN int32 MP4GetScale(MP4ParserHandle parserHandle, uint32 trackNum, uint32 *scale);
/* NOT implemented */
EXTERN int32 MP4GetRate(MP4ParserHandle parserHandle, uint32 trackNum, uint32 *rate);
/* NOT implemented */
EXTERN int32 MP4GetStartDelay(MP4ParserHandle parserHandle, uint32 trackNum, uint32 *usStartDelay);

/**
 * function to tell the average bitrate of a track.
 * If the average bitrate is not available in the file header, 0 is given.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param bitrate [out] Average bitrate, in bits per second. 
 * @return 
 */
EXTERN int32 MP4GetBitRate(MP4ParserHandle parserHandle, uint32 trackNum, uint32 *bitrate);


/**
 * function to tell the sample duration in us of a track.
 * If the sample duration is not a constant (eg. audio, subtilte), 0 is given.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based. 
 * @param usDuration [out] Sample duration in us. If sample duration is not a constant, this value is 0.
 * @return 
 */
EXTERN int32 MP4GetSampleDuration(MP4ParserHandle parserHandle, uint32 trackNum, uint64 *usDuration);


/**
 * function to tell the width in pixels of a video track. 
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a video track.
 * @param width [out] Width in pixels.
 * @return 
 */
EXTERN int32 MP4GetVideoFrameWidth(MP4ParserHandle parserHandle, uint32 trackNum, uint32 *width) ;

/**
 * function to tell the height in pixels of a video track. 
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a video track.
 * @param height [out] Height in pixels.
 * @return 
 */
EXTERN int32 MP4GetVideoFrameHeight(MP4ParserHandle parserHandle, uint32 trackNum, uint32 *height);


/**
 * function to tell how many channels in an audio track.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to an audio track.
 * @param numchannels [out] Number of the channels. 1 mono, 2 stereo, or more for multiple channels.
 * @return 
 */
EXTERN int32 MP4GetAudioNumChannels(MP4ParserHandle parserHandle, uint32 trackNum, uint32 *numChannels);

/**
 * function to tell the audio sample rate (sampling frequency) of an audio track.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to an audio track.
 * @param sampleRate [out] Audio integer sample rate (sampling frequency).
 * @return 
 */
EXTERN int32 MP4GetAudioSampleRate(MP4ParserHandle parserHandle, uint32 trackNum, uint32 * sampleRate);



/**
 * function to tell the bits per sample for a PCM audio track.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a PCM audio track.
 * @param bitsPerSample [out] Bits per PCM sample.
 * @return 
 */
EXTERN int32 MP4GetAudioBitsPerSample(MP4ParserHandle parserHandle, uint32 trackNum, uint32 * bitsPerSample);

/**
 * Function to tell the width of a text track.
 * The text track defines a window to display the subtitles. 
 * This window shall be positioned in the middle of the screen. 
 * And the sample is displayed in the window.How to position the sample within the window is defined by the sample data.
 * The origin of window is always (0, 0).
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a text track.
 * @param width [out] Width of the text track, in pixels.
 * @return 
 */
EXTERN int32 MP4GetTextTrackWidth(MP4ParserHandle parserHandle, 
                                    uint32 trackNum, 
                                    uint32 * width);
                                                
/**
 * Function to tell the height of a text track.
 * The text track defines a window to display the subtitles. 
 * This window shall be positioned in the middle of the screen. 
 * And the sample is displayed in the window.How to position the sample within the window is defined by the sample data.
 * The origin of window is always (0, 0).
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a text track.
 * @param height [out] Height of the window, in pixels.
 * @return 
 */
EXTERN int32 MP4GetTextTrackHeight(MP4ParserHandle parserHandle, 
                                    uint32 trackNum,
                                    uint32 * height);


/**
 * function to tell the codec specific information of a H264 video track.
 * For H264 video track, both decoder specific information and sample data 
 * are wrapped in a group of NALs in (length + data) format.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. The track must be an H264 video track.
 * @param size [out] Size of the NAL length field, in bytes. The value shall be 1, 2 or 4.
 * @return 
 */
EXTERN int32 MP4GetNALLengthFieldSize(MP4ParserHandle parserHandle, 
                                        uint32 trackNum,
                                        uint32 * size);


/**
 * Function to read the next sample from a track.
 *
 * It supports partial output of large samples:
 * If the entire sample can not be output by calling this function once, its remaining data  
 * can be got by repetitive calling the same function.
 *
 * BSAC audio track is somewhat special: 
 * Parser does not only read this BSAC track. But it will read one sample from each BSAC track 
 * in the proper order and make up a "big sample" for the user. 
 * Partial output is still supported and the sample flags describe this "big sample".
 * So the user shall take all BSAC tracks as one track and use any BSAC track number to call this function.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track to read, 0-based.
 * @param sampleData [in]   Buffer to hold the sample data. 
 *                          If the buffer is not big enough, usually only part of sample is output.
 *
 * @param dataSize [in/out]  Size of the buffer as input, in bytes.
 *                             As output:
 *
 *                              If a sample or part of sample is output successfully (return value is PARSER_SUCCESS ), 
 *                              it's the size of the data actually got.
 *
 *                              If the sample can not be output at all because buffer is too small 
 *                              (the return value is PARSER_INSUFFICIENT_MEMORY), it's the buffer size needed.
 *                              This is for effiency consideration of tracks with very small size samples.
 *                               
 *
 * @param usStartTime [out] Start time of the sample in us (timestamp)
 * @param usEndTime [out] End time of the sample in us.
 *
 * @param sampleFlags [out] Flags of this sample, if a sample or part of it is got successfully.
 *
 *                            FLAG_SYNC_SAMPLE    
 *                                  Whether this sample is a sync sample (key frame). 
 *                                  For non-video media, the wrapper shall take every sample as sync sample.
 *                          
 *                            FLAG_SAMPLE_NOT_FINISHED
 *                                  Sample data output is not finished because the buffer is not big enough.                                 
 *                                  Need to get the remaining data by repetitive calling this func.
 *                                  This feature is only for non-protected clips.
 *
 *
 * @return  PARSER_SUCCESS     An entire sample or part of it is got successfully.
 *          PARSER_EOS     No sample is got because of end of the track.
 *          PARSER_INSUFFICIENT_MEMORY Buffer is not big enough to hold the sample data                                
 *                                  The user can allocate a larger buffer and call this API again. 
 *          PARSER_READ_ERROR  File reading error. No need for further error concealment. 
 *          Others ...
 */
EXTERN int32 MP4GetNextSample( MP4ParserHandle parserHandle, 
                                uint32 trackNum,
                                uint8 * sampleData,  
                                uint32 * dataSize, 
                                uint64 * usStartTime, 
                                uint64 * usEndTime,
                                uint32 * sampleFlags);


/**
 * Function to seek a track to a target time. It will seek to a sync sample of the time stamp 
 * matching the target time. Due to the scarcity of the video sync samples (key frames), 
 * there can be a gap between the target time and the timestamp of the matched sync sample. 
 * So this time stamp will be output to as the accurate start time of the following playback segment.
 *
 * BSAC audio track is somewhat special: 
 * Parser does not only seek this BSAC track. But it will seek all BSAC tracks to the target time
 * and align their reading position.
 * So the user shall take all BSAC tracks as one track and use any BSAC track number to call this function.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track to seek, 0-based.
 * @param usTime [in/out]  Target time to seek as input, in us. 
 *                         Actual seeking time, timestamp of the matched sync sample, as output.
 *
 * @param flag [in]  Control flags to seek.
 *
 *                      SEEK_FLAG_NEAREST   
 *                      Default flag.The matched time stamp shall be the nearest one to the target time (may be later or earlier).
 *                           
 *                      SEEK_FLAG_NO_LATER  
 *                      The matched time stamp shall be no later than the target time.
 *                                                              
 *                      SEEK_FLAG_NO_EARLIER
 *                      The matched time stamp shall be no earlier than the target time.
 *
 * @return  PARSER_SUCCESS    Seeking succeeds.
 *          PARSER_EOS  Can NOT to seek to the target time because of end of stream.
 *          PARSER_ERR_NOT_SEEKABLE    Seeking fails because he movie is not seekable (index not available)            
 *          Others      Seeking fails for other reason.
 */
EXTERN  int32 MP4Seek(MP4ParserHandle parserHandle,
                     uint32 trackNum,
                     uint64 * usTime,
                     uint32 flag);


/**
 * function to get the next or previous sync sample (key frame) from current reading position of a track.
 * For trick mode FF/RW.
 * Also support partial output of large samples.
 * If not the entire sample is got, its remaining data can be got by repetitive calling the same function.
 *
 * Warning: This function does not support BSAC audio tracks because there are dependency between these tracks.
 *
 * @param parserHandle [in] Handle of the MP4 core parser.
 * @param trackNum [in] Number of the track to read, 0-based. 
 * @param direction [in]  Direction to get the sync sample.
 *                           FLAG_FORWARD   Read the next sync sample from current reading position.
 *                           FLAG_BACKWARD  Read the previous sync sample from current reading position.
 *
 * @param sampleData [in]   Buffer to hold the sample data. 
 *                          If the buffer is not big enough, usually only the part of sample is output. 
 *
 * @param dataSize [in/out]  Size of the buffer as input, in bytes.
 *                             As output:
 *
 *                              If a sample or part of sample is output successfully (return value is PARSER_SUCCESS ), 
 *                              it's the size of the data actually got.
 *
 *                              If the sample can not be output at all because buffer is too small 
 *                              (the return value is PARSER_INSUFFICIENT_MEMORY), it's the buffer size needed. 
 *                              This is for effiency consideration of tracks with very small size samples.
 *
 *
 * @param usStartTime [out] Start time of the sample in us (timestamp)
 * @param usEndTime [out] End time of the sample in us.
 * @param flag [out] Flags of this sample.
 *                            FLAG_SYNC_SAMPLE    
 *                                  Whether this sample is a sync sample (key frame). 
 *                                  For non-video media, the wrapper shall take every sample as sync sample.
 *                                  This flag shall always be SET for this API.
 *
 *                            FLAG_SAMPLE_NOT_FINISHED
 *                                  Sample data output is not finished because the buffer is not big enough.                                 
 *                                  Need to get the remaining data by repetitive calling this func.
 *                                  This feature is only for non-protected clips.
 *
 * @return  PARSER_SUCCESS     An entire sync sample or part of it is got successfully.
 *          PARSER_ERR_NOT_SEEKABLE    No sync sample is got  because the movie is not seekable (index not available) *              
 *          PARSER_EOS      Reaching the end of the track, no sync sample is got.
 *          PARSER_BOS      Reaching the beginning of the track, no sync sample is got.
 *          PARSER_INSUFFICIENT_MEMORY Buffer is too small to hold the sample data. 
 *                                  The user can allocate a larger buffer and call this API again.
 *          PARSER_READ_ERROR  File reading error. No need for further error concealment.
 *          PARSER_ERR_CONCEAL_FAIL  There is error in bitstream, and no sample can be found by error concealment. A seeking is helpful.
 *          Others ... Reading fails for other reason.
 */
EXTERN  int32 MP4GetNextSyncSample(MP4ParserHandle parserHandle, 
                                    uint32 trackNum,
                                    uint32 direction,
                                    uint8 * sampleData,  
                                    uint32 * dataSize, 
                                    uint64 * usStartTime,
                                    uint64 * usEndTime,
                                    uint32 * flags);



#endif //FSL_MP4_PARSER_API_HEADER_INCLUDED

