/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socket.h>
#include <sys/un.h>
#include "Log.h"
#include "sensors.h"
#include "libsensorhub.h"
#include "message.h"
#include "sensorhubd.h"
#include "sensor_device.h"

#define ANDROID_SOCKET_NAMESPACE_RESERVED (0)
static int sensorhub_local_client(const char *name, int namespaceId, int type)
{
        int sockFd;
        struct sockaddr_un client;
        sockFd = socket(AF_UNIX, type, 0);
        if (sockFd < 0) {
                ALOGE("%s line: %d create socket error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return -1;
        }

        client.sun_family = AF_UNIX;
        strcpy(client.sun_path, name);
        if (connect(sockFd, (struct sockaddr *) &client, sizeof(struct sockaddr_un)) < 0) {
                ALOGE("%s line: %d connect error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return -1;
        }
        
        return sockFd;
}

static void dump_sensor_info(struct sensor_device_t * device)
{
        ALOGV_IF(VERBOSE, "************************%s************************\n", device->name);
        ALOGV_IF(VERBOSE, "name: %s\n", device->name);
        ALOGV_IF(VERBOSE, "vendor: %s\n", device->vendor);
        ALOGV_IF(VERBOSE, "version: %d\n", device->version);
        ALOGV_IF(VERBOSE, "handle: %d\n", device->handle);
        ALOGV_IF(VERBOSE, "type: %d\n", device->type);
        ALOGV_IF(VERBOSE, "maxRange: %f\n", device->maxRange);
        ALOGV_IF(VERBOSE, "resolution: %f\n", device->resolution);
        ALOGV_IF(VERBOSE, "power: %f\n", device->power);
        ALOGV_IF(VERBOSE, "minDelay: %d\n", device->minDelay);
        ALOGV_IF(VERBOSE, "fifoReservedEventCount: %u\n", device->fifoReservedEventCount);
        ALOGV_IF(VERBOSE, "fifoMaxEventCount: %u\n", device->fifoMaxEventCount);
        ALOGV_IF(VERBOSE, "stringType: %s\n", device->stringType);
        ALOGV_IF(VERBOSE, "requiredPermission: %s\n", device->requiredPermission);
        ALOGV_IF(VERBOSE, "maxDelay: %d\n", device->maxDelay);
        ALOGV_IF(VERBOSE, "flags: 0x%x\n", device->flags);
        ALOGV_IF(VERBOSE, "************************%s************************\n", device->name);
}

static int sensorhub_send_recv_locked(int fd, uint8_t *send_buf, size_t send_size,
                                      uint8_t *recv_buf, size_t recv_size, pthread_mutex_t *lock)
{
        int ret, err;
        err = pthread_mutex_lock(lock);
        if (err) {
                ALOGE("%s line: %d pthread_mutex_lock error: %s", __FUNCTION__, __LINE__, strerror(err));
                ret = -err;
                goto err0;
        }

        ret = send(fd, send_buf, send_size, 0);
        if (ret < 0) {
                ALOGE("%s line: %d send error: %s", __FUNCTION__, __LINE__, strerror(errno));
                ret = -errno;
                goto err0;
        }

        ret = recv(fd, recv_buf, recv_size, 0);
        if (ret < 0) {
                ALOGE("%s line: %d recv error: %s", __FUNCTION__, __LINE__, strerror(errno));
                ret = -errno;
                goto err0;
        }

        ret = 0;

err0:
        err = pthread_mutex_unlock(lock);
        if (err) {
                ALOGE("%s line: %d pthread_mutex_unlock error: %s", __FUNCTION__, __LINE__, strerror(err));
        }

        return ret;
}

static int sensorhub_send_recv_locked_with_additional_buffer(int fd, uint8_t *send_buf, size_t send_size,
                                                             uint8_t *recv_buf, size_t recv_size,
                                                             void **additional_buffer, size_t *additional_size,
                                                             pthread_mutex_t *lock)
{
        int err, ret = 0;
        if (recv_size < sizeof(sensorhub_message_header_t)) {
                ALOGE("%s line: %d invalid size: %u the header size is: %u",
                      __FUNCTION__, __LINE__, (unsigned int)recv_size, (unsigned int)sizeof(sensorhub_message_header_t));
                return -EINVAL;
        }

        if (additional_buffer == NULL || additional_size == NULL) {
                ALOGE("%s line: %d invalid arguments", __FUNCTION__, __LINE__);
                return -EINVAL;
        }

        err = pthread_mutex_lock(lock);
        if (err) {
                ALOGE("%s line: %d pthread_mutex_lock error: %s", __FUNCTION__, __LINE__, strerror(err));
                ret = -err;
                goto err0;
        }

        ret = send(fd, send_buf, send_size, 0);
        if (ret < 0) {
                ALOGE("%s line: %d send error: %s", __FUNCTION__, __LINE__, strerror(errno));
                ret = -errno;
                goto err0;
        }

        ret = recv(fd, recv_buf, recv_size, 0);
        if (ret < 0) {
                ALOGE("%s line: %d recv error: %s", __FUNCTION__, __LINE__, strerror(errno));
                ret = -errno;
                goto err0;
        }

        sensorhub_message_header_t * header = (sensorhub_message_header_t *)(recv_buf);
        *additional_size = header->data_size - (recv_size - sizeof(sensorhub_message_header_t));
        if (*additional_size <= 0) {
                ALOGE("%s line: %d invalid additional buffer size: %u", __FUNCTION__, __LINE__, (unsigned int)*additional_size);
                *additional_buffer = NULL;
                ret = -EIO;
                goto err0;
        }

        *additional_buffer = malloc(*additional_size);
        if (*additional_buffer == NULL) {
                ALOGE("%s line: %d malloc failed", __FUNCTION__, __LINE__);
                *additional_size = 0;
                ret = -ENOMEM;
                goto err0;
        }

        ret = recv(fd, *additional_buffer, *additional_size, 0);
        if (ret < 0) {
                ALOGE("%s line: %d recv error: %s", __FUNCTION__, __LINE__, strerror(errno));
                *additional_buffer = NULL;
                *additional_size = 0;
                ret = -errno;
                goto err0;
        }
        ret = 0;
err0:
        err = pthread_mutex_unlock(lock);
        if (err) {
                ALOGE("%s line: %d pthread_mutex_unlock error: %s", __FUNCTION__, __LINE__, strerror(err));
        }

        return 0;
}

static const struct sensor_t * android_sensors_list = NULL;
static const struct sensor_t * aware_sensors_list = NULL;
static uint8_t *android_sensors_list_buf = NULL;
static uint8_t *aware_sensors_list_buf = NULL;
static int android_sensors_count = 0;
static int aware_sensors_count = 0;

int sensorhub_get_sensors_list(const sensor_list_flag_t list_type, struct sensor_t const** list)
{
        int data_size = sizeof(sensor_list_flag_t);
        msg_get_sensors_list msg;

        if (list_type == FLAG_ANDROID_LIST) {
                if (android_sensors_list != NULL) {
                        *list = android_sensors_list;
                        return android_sensors_count;
                }
        } else if (list_type == FLAG_AWARE_LIST) {
                if (aware_sensors_list != NULL) {
                        *list = aware_sensors_list;
                        return aware_sensors_count;
                }
        } else {
                ALOGE("%s line: %d Error: unknown list type: %d", __FUNCTION__, __LINE__, list_type);
        }

        int fd = sensorhub_local_client(SENSORHUBD_SOCKET_PATH, ANDROID_SOCKET_NAMESPACE_RESERVED, SOCK_STREAM);
        if (fd < 0) {
                ALOGE("%s line: %d socket error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return -1;
        }

        msg.header.require_code = REQUIRE_GET_SENSORS_LIST;
        msg.header.ack = 0;
        msg.header.acknowledge_number = 0;
        msg.header.data_size = data_size;
        msg.flag = list_type;

        int ret = send(fd, &msg, sizeof(msg_get_sensors_list), 0);
        if (ret < 0) {
                ALOGE("%s line: %d send error: %s", __FUNCTION__, __LINE__, strerror(errno));
                close(fd);
                return -1;
        }

        sensorhub_message_header_t ack_header;
        ret = recv(fd, &ack_header, sizeof(ack_header), 0);
        if (ret < 0) {
                ALOGE("%s line: %d recv error: %s", __FUNCTION__, __LINE__, strerror(errno));
                close(fd);
                return -1;
        }

        uint8_t *buf = malloc(ack_header.data_size);
        if (buf == NULL) {
                ALOGE("%s line: %d malloc error!", __FUNCTION__, __LINE__);
                return -ENOMEM;
        }
        ret = recv(fd, buf, ack_header.data_size, 0);
        if (ret < 0) {
                ALOGE("%s line: %d recv error: %s", __FUNCTION__, __LINE__, strerror(errno));
                free(buf);
                close(fd);
                return -1;
        }

        if (ack_header.ack != 1 || ack_header.acknowledge_number != data_size) {
                ALOGE("%s line: %d invaild ack or data size: %d %d", __FUNCTION__, __LINE__, ack_header.ack, ack_header.acknowledge_number);
                close(fd);
                free(buf);
                return -1;
        }

        sensor_list_header_t *list_header = (sensor_list_header_t *)buf;
        struct sensor_device_t *devices = (struct sensor_device_t *)(buf + sizeof(sensor_list_header_t));
        char * str = (char *)(buf + sizeof(sensor_list_header_t) + list_header->list_size);
        char * str_base = list_header->string_buffer_base;

        int count = list_header->list_size / sizeof(struct sensor_device_t);

        int i;
        for (i = 0; i < count; i++) {
                devices[i].name = str + (devices[i].name - str_base);
                devices[i].vendor = str + (devices[i].vendor - str_base);
                devices[i].stringType = str + (devices[i].stringType - str_base);
                devices[i].requiredPermission = str + (devices[i].requiredPermission - str_base);

                if (VERBOSE)
                        dump_sensor_info(devices + i);
        }

        struct sensor_t * sensors_list = malloc(count * sizeof(struct sensor_t));
        if (sensors_list != NULL) {
                for (i = 0; i < count; i++) {
                        sensors_list[i].name = devices[i].name;
                        sensors_list[i].vendor = devices[i].vendor;
                        sensors_list[i].version = devices[i].version;
                        sensors_list[i].handle = devices[i].handle;
                        sensors_list[i].type = devices[i].type;
                        sensors_list[i].maxRange = devices[i].maxRange;
                        sensors_list[i].resolution = devices[i].resolution;
                        sensors_list[i].power = devices[i].power;
                        sensors_list[i].minDelay = devices[i].minDelay;
                        sensors_list[i].fifoReservedEventCount = devices[i].fifoReservedEventCount;
                        sensors_list[i].fifoMaxEventCount = devices[i].fifoMaxEventCount;
                        sensors_list[i].stringType = devices[i].stringType;
                        sensors_list[i].requiredPermission = devices[i].requiredPermission;
                        sensors_list[i].maxDelay = devices[i].maxDelay;
                        sensors_list[i].flags = devices[i].flags;
                        sensors_list[i].reserved[0] = NULL;
                        sensors_list[i].reserved[1] = NULL;
                }
        } else {
                ALOGE("%s line: %d malloc for sensors list failed!", __FUNCTION__, __LINE__);
                count = 0;
        }

        *list = sensors_list;

        if (list_type == FLAG_ANDROID_LIST) {
                android_sensors_list = sensors_list;
                android_sensors_list_buf = buf;
                android_sensors_count = count;
        } else if (list_type == FLAG_AWARE_LIST) {
                aware_sensors_list = sensors_list;
                aware_sensors_list_buf = buf;
                aware_sensors_count = count;
        }

        return count;
}

void sensorhub_release_sensors_list()
{
        if (android_sensors_list != NULL) {
                free((void *)android_sensors_list);
                android_sensors_list = NULL;
        }
        if(android_sensors_list_buf != NULL) {
                free((void *)android_sensors_list_buf);
                android_sensors_list_buf = NULL;
        }

        if(aware_sensors_list != NULL) {
                free((void *)aware_sensors_list);
                aware_sensors_list = NULL;
        }
        if(aware_sensors_list_buf != NULL) {
                free((void *)aware_sensors_list_buf);
                aware_sensors_list_buf = NULL;
        }
}

typedef struct {
        int datafd;
        int ctrlfd;
        pthread_mutex_t ctrlfd_lock;
} session_context_t;

sensorhub_handle_t sensorhub_open(session_category_t category)
{
        int datafd, ret, ctrlfd;
        static int retry = 0;
        session_context_t *context = NULL;

        datafd = sensorhub_local_client(SENSORHUBD_SOCKET_PATH, ANDROID_SOCKET_NAMESPACE_RESERVED, SOCK_STREAM);
        if (datafd < 0)        {
                while (retry < 10) {
                        usleep(500000);
                        datafd = sensorhub_local_client(SENSORHUBD_SOCKET_PATH, ANDROID_SOCKET_NAMESPACE_RESERVED, SOCK_STREAM);
                        if (datafd >= 0)
                                break;
                        retry++;
                }
        }

        if (datafd < 0) {
                ALOGE("%s line: %d socket error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto err1;
        }

        msg_create_session_data_fd msg_data;
        msg_data.header.require_code = REQUIRE_CREATE_SESSION_DATA_FD;
        msg_data.header.ack = 0;
        msg_data.header.acknowledge_number = 0;
        msg_data.header.data_size = 0;

        ret = send(datafd, &msg_data, sizeof(msg_create_session_data_fd), 0);
        if (ret < 0) {
                ALOGE("%s line: %d send error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto err2;
        }

        msg_create_session_data_fd_resp data_resp;
        ret = recv(datafd, &data_resp, sizeof(msg_create_session_data_fd_resp), 0);
        if (ret < 0) {
                ALOGE("%s line: %d recv error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto err2;
        }

        if (data_resp.header.ack != 1 || data_resp.header.acknowledge_number != 0) {
                ALOGE("%s line: %d invaild ack or data size: %d %d", __FUNCTION__, __LINE__, data_resp.header.ack, data_resp.header.acknowledge_number);
                goto err2;
        }

        ctrlfd = sensorhub_local_client(SENSORHUBD_SOCKET_PATH, ANDROID_SOCKET_NAMESPACE_RESERVED, SOCK_STREAM);
        if (ctrlfd < 0) {
                ALOGE("%s line: %d socket error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto err2;
        }

        msg_create_session_ctrl_fd msg_ctrl;
        msg_ctrl.header.require_code = REQUIRE_CREATE_SESSION_CTRL_FD;
        msg_ctrl.header.ack = 0;
        msg_ctrl.header.acknowledge_number = 0;
        msg_ctrl.header.data_size = sizeof(msg_create_session_ctrl_fd) - sizeof(sensorhub_message_header_t);
        msg_ctrl.session = data_resp.session;
        msg_ctrl.category = category;

        ret = send(ctrlfd, &msg_ctrl, sizeof(msg_create_session_ctrl_fd), 0);
        if (ret < 0) {
                ALOGE("%s line: %d send error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto err3;
        }

        msg_simple_resp ctrl_resp;
        ret = recv(ctrlfd, &ctrl_resp, sizeof(msg_simple_resp), 0);
        if (ret < 0) {
                ALOGE("%s line: %d recv error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto err3;
        }

        if (ctrl_resp.header.ack != 1 || ctrl_resp.header.acknowledge_number != msg_ctrl.header.data_size) {
                ALOGE("%s line: %d invaild ack or data size: %d %d", __FUNCTION__, __LINE__, ctrl_resp.header.ack, ctrl_resp.header.acknowledge_number);
                goto err3;
        }

        ret = ctrl_resp.result;
        if (ret == 0) {
                ALOGE("%s line: %d sensorhub open failed!", __FUNCTION__, __LINE__);
                goto err3;
        }

        context = (session_context_t *)malloc(sizeof(session_context_t));
        if (context == NULL) {
                ALOGE("%s line: %d malloc failed!", __FUNCTION__, __LINE__);
                goto err3;
        }
        context->datafd = datafd;
        context->ctrlfd = ctrlfd;
        pthread_mutex_init(&context->ctrlfd_lock, NULL);
        goto success;

err3:
        close(ctrlfd);
err2:
        close(datafd);
err1:
success:
        return (sensorhub_handle_t)context;
}

void sensorhub_close(sensorhub_handle_t sensorhub_handle)
{
        session_context_t * context = (session_context_t *)sensorhub_handle;
        if (context != NULL) {
                close(context->datafd);
                close(context->ctrlfd);
                pthread_mutex_destroy(&context->ctrlfd_lock);
                free(context);
        }
}

int sensorhub_get_data_fd(sensorhub_handle_t sensorhub_handle)
{
        if (sensorhub_handle == NULL) {
                ALOGE("%s line: %d invalid sensorhub handle!", __FUNCTION__, __LINE__);
                return -1;
        }
        session_context_t *context = (session_context_t *)sensorhub_handle;
        return context->datafd;
}

/* 0 on success */
int sensorhub_start_streaming(sensorhub_handle_t sensorhub_handle, int handle, int64_t sampling_period_us, int64_t max_report_latency_us)
{
        if (sensorhub_handle == NULL) {
                ALOGE("%s line: %d invalid sensorhub handle!", __FUNCTION__, __LINE__);
                return -EFAULT;
        }

        session_context_t *context = (session_context_t *)sensorhub_handle;
        msg_start_streaming msg;
        msg_simple_resp resp;

        msg.header.require_code = REQUIRE_START_STREAMING;
        msg.header.ack = 0;
        msg.header.acknowledge_number = 0;
        msg.header.data_size = sizeof(start_streaming_data_t);
        msg.data.handle = handle;
        msg.data.sampling_period_us = sampling_period_us;
        msg.data.max_report_latency_us = max_report_latency_us;

        int ret = sensorhub_send_recv_locked(context->ctrlfd, (uint8_t *)&msg, sizeof(msg_start_streaming),
                                             (uint8_t *)&resp, sizeof(msg_simple_resp), &context->ctrlfd_lock);
        if (ret != 0)
                goto finish;

        if (resp.header.ack != 1 || resp.header.acknowledge_number != msg.header.data_size) {
                ALOGE("%s line: %d invaild ack or data size: %d %d", __FUNCTION__, __LINE__, resp.header.ack, resp.header.acknowledge_number);
                ret = -EBADMSG;
                goto finish;
        }

        ret = resp.result;
        if (ret != 1) {
                ALOGE("%s line: %d operation failed!", __FUNCTION__, __LINE__);
                ret = -EIO;
                goto finish;
        }
        ret = 0;

finish:
        return ret;
}

static int sensorhub_simple_command(sensorhub_handle_t sensorhub_handle, int handle, int require_code)
{
        if (sensorhub_handle == NULL) {
                ALOGE("%s line: %d invalid sensorhub handle!", __FUNCTION__, __LINE__);
                return -EFAULT;
        }

        session_context_t *context = (session_context_t *)sensorhub_handle;
        msg_simple_command msg;
        msg_simple_resp resp;

        msg.header.require_code = require_code;
        msg.header.ack = 0;
        msg.header.acknowledge_number = 0;
        msg.header.data_size = sizeof(handle);
        msg.handle = handle;

        int ret = sensorhub_send_recv_locked(context->ctrlfd, (uint8_t *)&msg, sizeof(msg_simple_command),
                                             (uint8_t *)&resp, sizeof(msg_simple_resp), &context->ctrlfd_lock);
        if (ret != 0)
                goto finish;

        if (resp.header.ack != 1 || resp.header.acknowledge_number != msg.header.data_size) {
                ALOGE("%s line: %d invaild ack or data size: %d %d", __FUNCTION__, __LINE__, resp.header.ack, resp.header.acknowledge_number);
                ret = -EBADMSG;
                goto finish;
        }

        ret = resp.result;
        if (ret != 1) {
                ALOGE("%s line: %d operation failed! ret: %d", __FUNCTION__, __LINE__, ret);
                ret = -EIO;
                goto finish;
        }
        ret = 0;

finish:
        return ret;
}

/* 0 on success */
int sensorhub_stop_streaming(sensorhub_handle_t sensorhub_handle, int handle)
{
        return sensorhub_simple_command(sensorhub_handle, handle, REQUIRE_STOP_STREAMING);
}

/* 0 on success */
int sensorhub_flush_streaming(sensorhub_handle_t sensorhub_handle, int handle)
{
        return sensorhub_simple_command(sensorhub_handle, handle, REQUIRE_FLUSH_STREAMING);
}

/* 0 on success */
int sensorhub_set_property(sensorhub_handle_t sensorhub_handle, int handle, void * property, size_t size)
{
        if (sensorhub_handle == NULL) {
                ALOGE("%s line: %d invalid sensorhub handle!", __FUNCTION__, __LINE__);
                return -EFAULT;
        }

        if (property == NULL || size == 0) {
                ALOGE("%s line: %d invalid argument!", __FUNCTION__, __LINE__);
                return -EINVAL;
        }

        session_context_t *context = (session_context_t *)sensorhub_handle;
        uint8_t *buf = malloc(sizeof(sensorhub_message_header_t) + sizeof(int) + size);
        sensorhub_message_header_t *header = (sensorhub_message_header_t *)buf;
        int *handle_p = (int *)(buf + sizeof(sensorhub_message_header_t));
        uint8_t *property_p = buf + sizeof(sensorhub_message_header_t) + sizeof(int);

        if (buf == NULL) {
                ALOGE("%s line: %d malloc error!", __FUNCTION__, __LINE__);
                return -ENOMEM;
        }

        header->require_code = REQUIRE_SET_PROPERTY;
        header->ack = 0;
        header->acknowledge_number = 0;
        header->data_size = sizeof(int) + size;
        *handle_p = handle;
        memcpy(property_p, property, size);

        int ret = sensorhub_send_recv_locked(context->ctrlfd, buf, sizeof(sensorhub_message_header_t) + sizeof(int) + size,
                                             buf, sizeof(sensorhub_message_header_t) + sizeof(int), &context->ctrlfd_lock);
        if (ret != 0)
                goto finish;

        if (header->ack != 1 || header->acknowledge_number != sizeof(int) + size) {
                ALOGE("%s line: %d invaild ack or data size: %d %d", __FUNCTION__, __LINE__, header->ack, header->acknowledge_number);
                ret = -EBADMSG;
                goto finish;
        }

        ret = *(int *)(buf + sizeof(sensorhub_message_header_t));
        if (ret != 1) {
                ALOGE("%s line: %d operation failed!", __FUNCTION__, __LINE__);
                ret = -EIO;
                goto finish;
        }
        ret = 0;

finish:
        free(buf);
        return ret;
}

size_t sensorhub_get_property(sensorhub_handle_t sensorhub_handle, int handle, void * command, size_t command_size, void ** property)
{
        if (property == NULL)
                return 0;

        if (sensorhub_handle == NULL) {
                ALOGE("%s line: %d invalid sensorhub handle!", __FUNCTION__, __LINE__);
                *property = NULL;
                return 0;
        }

        if (command == NULL || command_size == 0) {
                ALOGE("%s line: %d invalid argument!", __FUNCTION__, __LINE__);
                *property = NULL;
                return 0;
        }

        session_context_t *context = (session_context_t *)sensorhub_handle;
        uint8_t *buf = malloc(sizeof(sensorhub_message_header_t) + sizeof(int) + command_size);
        sensorhub_message_header_t *header = (sensorhub_message_header_t *)buf;
        int *handle_p = (int *)(buf + sizeof(sensorhub_message_header_t));
        uint8_t *command_p = buf + sizeof(sensorhub_message_header_t) + sizeof(int);

        if (buf == NULL) {
                ALOGE("%s line: %d malloc error!", __FUNCTION__, __LINE__);
                *property = NULL;
                return 0;
        }

        header->require_code = REQUIRE_GET_PROPERTY;
        header->ack = 0;
        header->acknowledge_number = 0;
        header->data_size = sizeof(int) + command_size;
        *handle_p = handle;
        memcpy(command_p, command, command_size);

        size_t property_size = 0;
        *property = NULL;
        int ret = sensorhub_send_recv_locked_with_additional_buffer(context->ctrlfd, buf, sizeof(sensorhub_message_header_t) + sizeof(int) + command_size,
                                                                    buf, sizeof(sensorhub_message_header_t) + sizeof(int), property, &property_size,
                                                                    &context->ctrlfd_lock);
        if (ret != 0)
                goto finish;

        if (header->ack != 1 || header->acknowledge_number != sizeof(int) + command_size) {
                ALOGE("%s line: %d invaild ack or data size: %d %d", __FUNCTION__, __LINE__, header->ack, header->acknowledge_number);
                goto finish;
        }

        ret = *(int *)(buf + sizeof(sensorhub_message_header_t));
        if (ret != 1) {
                ALOGE("%s line: %d operation failed!", __FUNCTION__, __LINE__);
                goto finish;
        }

finish:
        free(buf);
        return property_size;
}
