/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "mwb-download-model.h"
#include <clutter-mozembed.h>

#define MWB_TYPE_DOWNLOAD_MODEL_ITER mwb_download_model_iter_get_type()

#define MWB_DOWNLOAD_MODEL_ITER(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
  MWB_TYPE_DOWNLOAD_MODEL_ITER, MwbDownloadModelIter))

#define MWB_DOWNLOAD_MODEL_ITER_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST ((klass), \
  MWB_TYPE_DOWNLOAD_MODEL_ITER, MwbDownloadModelIterClass))

#define MWB_IS_DOWNLOAD_MODEL_ITER(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
  MWB_TYPE_DOWNLOAD_MODEL_ITER))

#define MWB_IS_DOWNLOAD_MODEL_ITER_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE ((klass), \
  MWB_TYPE_DOWNLOAD_MODEL_ITER))

#define MWB_DOWNLOAD_MODEL_ITER_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS ((obj), \
  MWB_TYPE_DOWNLOAD_MODEL_ITER, MwbDownloadModelIterClass))

typedef struct
{
  ClutterModelIter parent;

  MwbDownloadModel *model;
  guint             row;
} MwbDownloadModelIter;

typedef struct
{
  ClutterModelIterClass parent_class;
} MwbDownloadModelIterClass;

GType mwb_download_model_iter_get_type (void);

G_DEFINE_TYPE (MwbDownloadModel, mwb_download_model, CLUTTER_TYPE_MODEL)
G_DEFINE_TYPE (MwbDownloadModelIter, mwb_download_model_iter,
               CLUTTER_TYPE_MODEL_ITER)

#define DOWNLOAD_MODEL_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_DOWNLOAD_MODEL, \
                                MwbDownloadModelPrivate))

enum
{
  PROGRESS,
  COMPLETED,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

struct _MwbDownloadModelPrivate
{
  GPtrArray *downloads;
  gboolean   active;
};

static void
mwb_download_model_get_property (GObject *object, guint property_id,
                              GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_download_model_set_property (GObject *object, guint property_id,
                              const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_download_model_dispose (GObject *object)
{
  MwbDownloadModelPrivate *priv = MWB_DOWNLOAD_MODEL (object)->priv;

  if (priv->downloads)
    {
      gint i;
      for (i = 0; i < priv->downloads->len; i++)
        {
          GObject *download = g_ptr_array_index (priv->downloads, i);
          g_object_unref (download);
        }
      g_ptr_array_free (priv->downloads, TRUE);
      priv->downloads = NULL;
    }

  G_OBJECT_CLASS (mwb_download_model_parent_class)->dispose (object);
}

static void
mwb_download_model_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_download_model_parent_class)->finalize (object);
}

static guint
mwb_download_model_get_n_rows (ClutterModel *model)
{
  MwbDownloadModelPrivate *priv = MWB_DOWNLOAD_MODEL (model)->priv;
  return priv->downloads->len;
}

static guint
mwb_download_model_get_n_columns (ClutterModel *model)
{
  return MWB_COL_LAST;
}

static GType
mwb_download_model_get_column_type (ClutterModel *model, guint column)
{
  switch (column)
    {
    case MWB_COL_DOWNLOAD :
      return CLUTTER_TYPE_MOZEMBED_DOWNLOAD;

    case MWB_COL_SRC :
    case MWB_COL_DEST :
      return G_TYPE_STRING;

    case MWB_COL_PROGRESS :
    case MWB_COL_MAX_PROGRESS :
      return G_TYPE_INT64;

    case MWB_COL_COMPLETED :
    case MWB_COL_CANCELLED :
      return G_TYPE_BOOLEAN;

    default :
      return G_TYPE_INVALID;
    }
}

static ClutterModelIter *
mwb_download_model_get_iter_at_row (ClutterModel *model,
                                    guint         row)
{
  MwbDownloadModelIter *iter =
    g_object_new (MWB_TYPE_DOWNLOAD_MODEL_ITER,
                  "model", model,
                  "row", row,
                  NULL);
  iter->model = (MwbDownloadModel *)model;
  iter->row = row;
  return (ClutterModelIter *)iter;
}

static void
mwb_download_model_class_init (MwbDownloadModelClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterModelClass *model_class = CLUTTER_MODEL_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbDownloadModelPrivate));

  object_class->get_property = mwb_download_model_get_property;
  object_class->set_property = mwb_download_model_set_property;
  object_class->dispose = mwb_download_model_dispose;
  object_class->finalize = mwb_download_model_finalize;

  model_class->get_n_rows = mwb_download_model_get_n_rows;
  model_class->get_n_columns = mwb_download_model_get_n_columns;
  model_class->get_column_type = mwb_download_model_get_column_type;
  model_class->get_iter_at_row = mwb_download_model_get_iter_at_row;

  signals[PROGRESS] =
    g_signal_new ("progress",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbDownloadModelClass, progress),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__DOUBLE,
                  G_TYPE_NONE, 1, G_TYPE_DOUBLE);

  signals[COMPLETED] =
    g_signal_new ("completed",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbDownloadModelClass, completed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

static void
mwb_download_model_init (MwbDownloadModel *self)
{
  MwbDownloadModelPrivate *priv = self->priv = DOWNLOAD_MODEL_PRIVATE (self);

  priv->downloads = g_ptr_array_new ();
}

ClutterModel *
mwb_download_model_new (void)
{
  return g_object_new (MWB_TYPE_DOWNLOAD_MODEL, NULL);
}

static void
mwb_download_model_update_progress (MwbDownloadModel *self)
{
  gint i, downloads;

  gboolean complete = TRUE;
  gdouble total_progress = 0.0;
  MwbDownloadModelPrivate *priv = self->priv;

  /* Iterate over all active downloads */
  for (i = 0, downloads = 0; i < priv->downloads->len; i++)
    {
      ClutterMozEmbedDownload *download = (ClutterMozEmbedDownload *)
        g_ptr_array_index (priv->downloads, i);

      if (!clutter_mozembed_download_get_complete (download) &&
          !clutter_mozembed_download_get_cancelled (download))
        {
          gint64 progress, max_progress;

          complete = FALSE;
          max_progress = clutter_mozembed_download_get_max_progress (download);
          progress = clutter_mozembed_download_get_progress (download);

          if ((progress > 0) && (max_progress > progress))
            {
              downloads ++;
              total_progress += progress / (gdouble)max_progress;
            }
        }
    }

  priv->active = !complete;

  if (complete || !downloads)
    g_signal_emit (self, signals[COMPLETED], 0);
  else
    g_signal_emit (self, signals[PROGRESS], 0,
                   total_progress / (gdouble)downloads);
}

static void
mwb_download_changed_cb (ClutterMozEmbedDownload *download,
                         GParamSpec              *pspec,
                         MwbDownloadModel        *model)
{
  guint row = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (download), "row"));
  ClutterModelIter *iter =
    mwb_download_model_get_iter_at_row((ClutterModel *)model, row);
  g_signal_emit_by_name (model, "row-changed", iter);
  g_object_unref (iter);

  /* Send overall progress/complete signals */
  mwb_download_model_update_progress (model);
}

void
mwb_download_model_add (MwbDownloadModel        *model,
                        ClutterMozEmbedDownload *download)
{
  ClutterModelIter *iter;
  MwbDownloadModelPrivate *priv = model->priv;

  g_ptr_array_add (priv->downloads, g_object_ref (download));
  g_object_set_data (G_OBJECT (download),
                     "row", GUINT_TO_POINTER (priv->downloads->len - 1));

  iter = mwb_download_model_get_iter_at_row ((ClutterModel *)model,
                                             priv->downloads->len - 1);
  g_signal_emit_by_name (model, "row-added", iter);
  g_object_unref (iter);

  g_signal_connect (download, "notify::progress",
                    G_CALLBACK (mwb_download_changed_cb), model);
  g_signal_connect (download, "notify::max-progress",
                    G_CALLBACK (mwb_download_changed_cb), model);
  g_signal_connect (download, "notify::complete",
                    G_CALLBACK (mwb_download_changed_cb), model);
  g_signal_connect (download, "notify::cancelled",
                    G_CALLBACK (mwb_download_changed_cb), model);
}

static void
mwb_download_model_iter_get_value (ClutterModelIter *iter,
                                   guint             column,
                                   GValue           *value)
{
  MwbDownloadModelIter *mwb_iter = (MwbDownloadModelIter *)iter;
  MwbDownloadModelPrivate *priv = mwb_iter->model->priv;
  ClutterMozEmbedDownload *download = (ClutterMozEmbedDownload *)
    g_ptr_array_index (priv->downloads, mwb_iter->row);

  switch (column)
    {
    case MWB_COL_DOWNLOAD :
      g_value_set_object (value, download);
      break;

    case MWB_COL_SRC :
      g_value_set_string (value,
                          clutter_mozembed_download_get_source (download));
      break;

    case MWB_COL_DEST :
      g_value_set_string (value,
                          clutter_mozembed_download_get_destination (download));
      break;

    case MWB_COL_PROGRESS :
      g_value_set_int64 (value,
                         clutter_mozembed_download_get_progress (download));
      break;

    case MWB_COL_MAX_PROGRESS :
      g_value_set_int64 (value,
                         clutter_mozembed_download_get_max_progress (download));
      break;

    case MWB_COL_COMPLETED :
      g_value_set_boolean (value,
                           clutter_mozembed_download_get_complete (download));
      break;

    case MWB_COL_CANCELLED :
      g_value_set_boolean (value,
                           clutter_mozembed_download_get_cancelled (download));
      break;

    default :
      g_warning ("Unrecognised column");
    }
}

static gboolean
mwb_download_model_iter_is_first (ClutterModelIter *iter)
{
  MwbDownloadModelIter *mwb_iter = (MwbDownloadModelIter *)iter;
  return (mwb_iter->row == 0);
}

static gboolean
mwb_download_model_iter_is_last (ClutterModelIter *iter)
{
  MwbDownloadModelIter *mwb_iter = (MwbDownloadModelIter *)iter;
  MwbDownloadModelPrivate *priv = mwb_iter->model->priv;
  return (mwb_iter->row == priv->downloads->len);
}

static ClutterModelIter *
mwb_download_model_iter_next (ClutterModelIter *iter)
{
  MwbDownloadModelIter *mwb_iter = (MwbDownloadModelIter *)iter;
  g_object_set (G_OBJECT (mwb_iter), "row", ++mwb_iter->row, NULL);
  return iter;
}

static ClutterModelIter *
mwb_download_model_iter_prev (ClutterModelIter *iter)
{
  MwbDownloadModelIter *mwb_iter = (MwbDownloadModelIter *)iter;
  g_object_set (G_OBJECT (mwb_iter), "row", --mwb_iter->row, NULL);
  return iter;
}

static ClutterModel *
mwb_download_model_iter_get_model (ClutterModelIter *iter)
{
  MwbDownloadModelIter *mwb_iter = (MwbDownloadModelIter *)iter;
  return (ClutterModel *)mwb_iter->model;
}

static guint
mwb_download_model_iter_get_row (ClutterModelIter *iter)
{
  MwbDownloadModelIter *mwb_iter = (MwbDownloadModelIter *)iter;
  return mwb_iter->row;
}

static ClutterModelIter *
mwb_download_model_iter_copy (ClutterModelIter *iter)
{
  MwbDownloadModelIter *mwb_iter = (MwbDownloadModelIter *)iter;
  MwbDownloadModelIter *new_mwb_iter =
    g_object_new (MWB_TYPE_DOWNLOAD_MODEL_ITER,
                  "model", mwb_iter->model,
                  "row", mwb_iter->row,
                  NULL);
  new_mwb_iter->model = mwb_iter->model;
  new_mwb_iter->row = mwb_iter->row;
  return (ClutterModelIter *)new_mwb_iter;
}

static void
mwb_download_model_iter_class_init (MwbDownloadModelIterClass *klass)
{
  ClutterModelIterClass *iter_class = CLUTTER_MODEL_ITER_CLASS (klass);

  iter_class->get_value = mwb_download_model_iter_get_value;
  iter_class->is_first = mwb_download_model_iter_is_first;
  iter_class->is_last = mwb_download_model_iter_is_last;
  iter_class->next = mwb_download_model_iter_next;
  iter_class->prev = mwb_download_model_iter_prev;
  iter_class->get_model = mwb_download_model_iter_get_model;
  iter_class->get_row = mwb_download_model_iter_get_row;
  iter_class->copy = mwb_download_model_iter_copy;
}

static void
mwb_download_model_iter_init (MwbDownloadModelIter *self)
{
}

gboolean
mwb_download_model_get_active (MwbDownloadModel *model)
{
  return model->priv->active;
}

