/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mwb-page.h"
#include "mwb-mozembed.h"
#include "mwb-utils.h"
#include <moz-headless.h>

G_DEFINE_TYPE (MwbPage, mwb_page, NBTK_TYPE_WIDGET)

#define PAGE_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_PAGE, MwbPagePrivate))

enum
{
  PROP_0,

  PROP_PARENT,
  PROP_MOZEMBED,
  PROP_DLMAN,
};

struct _MwbPagePrivate
{
  ClutterMozEmbed    *parent;
  ClutterActor       *mozembed;
  NbtkWidget         *scroll_view;
  NbtkWidget         *dialog;
  ClutterActor       *dialog_bg;
  MwbDownloadManager *dlman;
};

static void
mwb_page_get_property (GObject *object, guint property_id,
                       GValue *value, GParamSpec *pspec)
{
  MwbPagePrivate *priv = MWB_PAGE (object)->priv;

  switch (property_id)
    {
    case PROP_PARENT:
      g_value_set_object (value, priv->parent);
      break;

    case PROP_MOZEMBED:
      g_value_set_object (value, priv->mozembed);
      break;

    case PROP_DLMAN:
      g_value_set_object (value, priv->dlman);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_page_set_property (GObject *object, guint property_id,
                       const GValue *value, GParamSpec *pspec)
{
  MwbPage *page = MWB_PAGE (object);
  MwbPagePrivate *priv = page->priv;

  switch (property_id)
    {
    case PROP_PARENT:
      priv->parent = g_value_get_object (value);
      break;

    case PROP_MOZEMBED:
      mwb_page_set_mozembed (page, MWB_MOZEMBED (g_value_get_object (value)));
      break;

    case PROP_DLMAN:
      mwb_page_set_download_manager (page, g_value_get_object (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_page_dispose (GObject *object)
{
  MwbPage *self = MWB_PAGE (object);
  MwbPagePrivate *priv = self->priv;

  mwb_page_set_mozembed (self, NULL);

  if (priv->dialog)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->dialog));
      priv->dialog = NULL;
    }

  if (priv->dialog_bg)
    {
      clutter_actor_unparent (priv->dialog_bg);
      priv->dialog_bg = NULL;
    }

  if (priv->scroll_view)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->scroll_view));
      priv->scroll_view = NULL;
    }

  if (priv->dlman)
    {
      g_object_unref (priv->dlman);
      priv->dlman = NULL;
    }

  G_OBJECT_CLASS (mwb_page_parent_class)->dispose (object);
}

static void
mwb_page_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_page_parent_class)->finalize (object);
}

static void
mwb_page_paint (ClutterActor *actor)
{
  MwbPagePrivate *priv = MWB_PAGE (actor)->priv;

  /* Chain up to get background */
  CLUTTER_ACTOR_CLASS (mwb_page_parent_class)->paint (actor);

  if (CLUTTER_ACTOR_IS_MAPPED (priv->scroll_view))
    clutter_actor_paint (CLUTTER_ACTOR (priv->scroll_view));

  if (priv->dialog)
    {
      if (CLUTTER_ACTOR_IS_MAPPED (priv->dialog_bg))
        clutter_actor_paint (priv->dialog_bg);
      if (CLUTTER_ACTOR_IS_MAPPED (priv->dialog))
        clutter_actor_paint (CLUTTER_ACTOR (priv->dialog));
    }
}

static void
mwb_page_pick (ClutterActor *actor, const ClutterColor *color)
{
  MwbPagePrivate *priv = MWB_PAGE (actor)->priv;

  if (priv->dialog)
    clutter_actor_paint (CLUTTER_ACTOR (priv->dialog));
  else
    clutter_actor_paint (CLUTTER_ACTOR (priv->scroll_view));
}

static void
mwb_page_allocate (ClutterActor           *actor,
                   const ClutterActorBox  *box,
                   ClutterAllocationFlags  flags)
{
  NbtkPadding padding;
  ClutterActorBox child_box;

  MwbPagePrivate *priv = MWB_PAGE (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_page_parent_class)->allocate (actor, box, flags);

  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);

  child_box.x1 = padding.left;
  child_box.y1 = padding.top;
  child_box.x2 = box->x2 - box->x1 - padding.right;
  child_box.y2 = box->y2 - box->y1 - padding.bottom;

  clutter_actor_allocate (CLUTTER_ACTOR (priv->scroll_view), &child_box, flags);
  clutter_actor_allocate (priv->dialog_bg, &child_box, flags);

  if (priv->dialog)
    {
      gfloat width, height, nat_width, nat_height;

      nbtk_widget_get_padding (NBTK_WIDGET (priv->dialog), &padding);
      clutter_actor_get_preferred_size (CLUTTER_ACTOR (priv->dialog),
                                        NULL, NULL, &nat_width, &nat_height);

      if (nat_width && nat_height)
        {
          width = MIN (child_box.x2 - child_box.x1,
                       nat_width + padding.left + padding.right);
          height = MIN (child_box.y2 - child_box.y1,
                        nat_height + padding.top + padding.bottom);
        }
      else
       {
         width = (child_box.x2 - child_box.x1) * 0.55;
         height = (child_box.y2 - child_box.y1) * 0.4;
       }

      child_box.x1 = MWB_PIXBOUND (((box->x2 - box->x1) / 2.0) - (width / 2.0));
      child_box.x2 = child_box.x1 + width;
      child_box.y1 = MWB_PIXBOUND (((box->y2 - box->y1) / 2.0) - (height / 2.0));
      child_box.y2 = child_box.y1 + height;
      clutter_actor_allocate (CLUTTER_ACTOR (priv->dialog), &child_box, flags);
    }
}

static ClutterActor *
mwb_page_get_main_mozembed (MwbPage *self)
{
  MwbPagePrivate *priv = self->priv;

  if (priv->dialog)
    return nbtk_bin_get_child (NBTK_BIN (priv->dialog));
  else
    return priv->mozembed;
}

static void
mwb_page_key_focus_in (ClutterActor *actor)
{
  ClutterActor *mozembed = mwb_page_get_main_mozembed (MWB_PAGE (actor));
  CLUTTER_ACTOR_CLASS (G_OBJECT_GET_CLASS (mozembed))->key_focus_in (mozembed);
}

static void
mwb_page_key_focus_out (ClutterActor *actor)
{
  ClutterActor *mozembed = mwb_page_get_main_mozembed (MWB_PAGE (actor));
  CLUTTER_ACTOR_CLASS (G_OBJECT_GET_CLASS (mozembed))->key_focus_out (mozembed);
}

static gboolean
mwb_page_key_press_event (ClutterActor    *actor,
                          ClutterKeyEvent *event)
{
  ClutterActor *mozembed = mwb_page_get_main_mozembed (MWB_PAGE (actor));
  return CLUTTER_ACTOR_CLASS (G_OBJECT_GET_CLASS (mozembed))->
    key_press_event (mozembed, event);
}

static gboolean
mwb_page_key_release_event (ClutterActor    *actor,
                            ClutterKeyEvent *event)
{
  ClutterActor *mozembed = mwb_page_get_main_mozembed (MWB_PAGE (actor));
  return CLUTTER_ACTOR_CLASS (G_OBJECT_GET_CLASS (mozembed))->
    key_release_event (mozembed, event);
}

static void
mwb_page_dialog_anim_completed_cb (ClutterAnimation *animation,
                                   MwbPage *self)
{
  MwbPagePrivate *priv = self->priv;

  g_signal_handlers_disconnect_by_func (animation,
                                        mwb_page_dialog_anim_completed_cb,
                                        self);

  if (clutter_actor_get_opacity (CLUTTER_ACTOR (priv->dialog)) == 0x00)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->dialog));
      priv->dialog = NULL;
      clutter_actor_queue_relayout (CLUTTER_ACTOR (self));

      if (mwb_utils_actor_has_focus (CLUTTER_ACTOR (self)))
        mwb_page_key_focus_in (CLUTTER_ACTOR (self));
    }
}

static void
mwb_page_dialog_closed_cb (ClutterMozEmbed *mozembed, MwbPage *self)
{
  MwbPagePrivate *priv = self->priv;

  g_signal_handlers_disconnect_by_func (mozembed,
                                        mwb_page_dialog_closed_cb,
                                        self);

  clutter_actor_animate (CLUTTER_ACTOR (priv->dialog),
                         CLUTTER_EASE_IN_SINE, 150,
                         "opacity", 0x00,
                         "signal::completed",
                           mwb_page_dialog_anim_completed_cb, self,
                         NULL);
  clutter_actor_animate (priv->dialog_bg,
                         CLUTTER_EASE_IN_SINE, 150,
                         "opacity", 0x00,
                         NULL);
}

static void
mwb_page_dialog_size_request_cb (ClutterMozEmbed *mozembed,
                                 gint             width,
                                 gint             height,
                                 MwbPage         *self)
{
  clutter_actor_set_size (CLUTTER_ACTOR (mozembed), width, height);
  clutter_actor_queue_relayout (CLUTTER_ACTOR (self));
}

static void
mwb_page_download_cb (ClutterMozEmbed         *mozembed,
                      ClutterMozEmbedDownload *download,
                      MwbPage                 *page)
{
  MwbPagePrivate *priv = page->priv;

  if (priv->dlman)
    mwb_download_manager_add (priv->dlman, download);
}

static void
mwb_page_new_window_cb (ClutterMozEmbed  *mozembed,
                        ClutterMozEmbed **new_mozembed,
                        guint             chromemask,
                        MwbPage          *self)
{
  MwbPagePrivate *priv = self->priv;

  if (!(chromemask & MOZ_HEADLESS_FLAG_OPENASCHROME))
    return;

  if (priv->dialog)
    {
      ClutterActor *child = nbtk_bin_get_child (NBTK_BIN (priv->dialog));

      if (child)
        g_signal_handlers_disconnect_by_func (child,
                                              mwb_page_dialog_closed_cb,
                                              self);
      else
        g_warning ("Dialog with no child!");

      clutter_actor_unparent (CLUTTER_ACTOR (priv->dialog));
    }

  *new_mozembed = CLUTTER_MOZEMBED (mwb_mozembed_new_for_new_window ());

  g_signal_connect (*new_mozembed, "closed",
                    G_CALLBACK (mwb_page_dialog_closed_cb), self);
  g_signal_connect (*new_mozembed, "size-request",
                    G_CALLBACK (mwb_page_dialog_size_request_cb), self);
  g_signal_connect (*new_mozembed, "download",
                    G_CALLBACK (mwb_page_download_cb), self);

  priv->dialog = nbtk_bin_new ();
  nbtk_bin_set_fill (NBTK_BIN (priv->dialog), TRUE, TRUE);
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->dialog), CLUTTER_ACTOR (self));

  clutter_container_add_actor (CLUTTER_CONTAINER (priv->dialog),
                               CLUTTER_ACTOR (*new_mozembed));

  if (mwb_utils_actor_has_focus (CLUTTER_ACTOR (self)))
    mwb_page_key_focus_in (CLUTTER_ACTOR (self));

  clutter_actor_set_opacity (CLUTTER_ACTOR (priv->dialog), 0x00);

  clutter_actor_animate (CLUTTER_ACTOR (priv->dialog),
                         CLUTTER_EASE_IN_SINE, 150,
                         "opacity", 0xff,
                         NULL);
  clutter_actor_animate (priv->dialog_bg,
                         CLUTTER_EASE_IN_SINE, 150,
                         "opacity", 0xff,
                         NULL);

  clutter_actor_queue_relayout (CLUTTER_ACTOR (self));
}

static void
mwb_page_constructed (GObject *object)
{
  MwbPage *page = MWB_PAGE (object);
  MwbPagePrivate *priv = page->priv;

  if (!priv->mozembed)
    {
      ClutterActor *mozembed;

      if (priv->parent)
        mozembed = mwb_mozembed_new_with_parent (priv->parent);
      else
        mozembed = mwb_mozembed_new (FALSE);

      mwb_page_set_mozembed (page, MWB_MOZEMBED (mozembed));
    }

  if (G_OBJECT_CLASS (mwb_page_parent_class)->constructed)
    G_OBJECT_CLASS (mwb_page_parent_class)->constructed (object);
}

static void
mwb_page_map (ClutterActor *actor)
{
  MwbPagePrivate *priv = MWB_PAGE (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_page_parent_class)->map (actor);

  clutter_actor_map (CLUTTER_ACTOR (priv->scroll_view));
  clutter_actor_map (priv->dialog_bg);
  if (priv->dialog)
    clutter_actor_map (CLUTTER_ACTOR (priv->dialog));
}

static void
mwb_page_unmap (ClutterActor *actor)
{
  MwbPagePrivate *priv = MWB_PAGE (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_page_parent_class)->unmap (actor);

  clutter_actor_unmap (CLUTTER_ACTOR (priv->scroll_view));
  clutter_actor_unmap (priv->dialog_bg);
  if (priv->dialog)
    clutter_actor_unmap (CLUTTER_ACTOR (priv->dialog));
}

static gboolean
mwb_page_captured_event (ClutterActor *actor,
                         ClutterEvent *event)
{
  if (event->type == CLUTTER_BUTTON_PRESS)
    clutter_actor_grab_key_focus (actor);

  return FALSE;
}

static void
mwb_page_class_init (MwbPageClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbPagePrivate));

  object_class->constructed = mwb_page_constructed;
  object_class->get_property = mwb_page_get_property;
  object_class->set_property = mwb_page_set_property;
  object_class->dispose = mwb_page_dispose;
  object_class->finalize = mwb_page_finalize;

  actor_class->paint = mwb_page_paint;
  actor_class->pick = mwb_page_pick;
  actor_class->allocate = mwb_page_allocate;
  actor_class->map = mwb_page_map;
  actor_class->unmap = mwb_page_unmap;
  actor_class->key_focus_in = mwb_page_key_focus_in;
  actor_class->key_focus_out = mwb_page_key_focus_out;
  actor_class->key_press_event = mwb_page_key_press_event;
  actor_class->key_release_event = mwb_page_key_release_event;
  actor_class->captured_event = mwb_page_captured_event;

  g_object_class_install_property (object_class,
                                   PROP_PARENT,
                                   g_param_spec_object ("parent",
                                                        "Parent",
                                                        "Parent "
                                                        "ClutterMozEmbed.",
                                                        CLUTTER_TYPE_MOZEMBED,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB |
                                                        G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class,
                                   PROP_MOZEMBED,
                                   g_param_spec_object ("mozembed",
                                                        "MozEmbed",
                                                        "Child "
                                                        "ClutterMozEmbed.",
                                                        CLUTTER_TYPE_MOZEMBED,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_DLMAN,
                                   g_param_spec_object ("download-manager",
                                                        "Download manager",
                                                        "The download manager.",
                                                        MWB_TYPE_DOWNLOAD_MANAGER,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));
}

static void
mwb_page_scroll_start_cb (MwbPage *self)
{
  MwbPagePrivate *priv = self->priv;
  mwb_mozembed_set_sync_adjustments (MWB_MOZEMBED (priv->mozembed), FALSE);
}

static void
mwb_page_scroll_stop_cb (MwbPage *self)
{
  MwbPagePrivate *priv = self->priv;
  mwb_mozembed_set_sync_adjustments (MWB_MOZEMBED (priv->mozembed), TRUE);
}

static void
mwb_page_notify_reactive (MwbPage *self)
{
  gboolean reactive = CLUTTER_ACTOR_IS_REACTIVE (self);
  MwbPagePrivate *priv = self->priv;

  clutter_actor_set_reactive (CLUTTER_ACTOR (priv->mozembed), reactive);
  if (priv->dialog)
    clutter_actor_set_reactive (nbtk_bin_get_child (NBTK_BIN (priv->dialog)),
                                reactive);
}

static void
mwb_page_init (MwbPage *self)
{
  ClutterActor *bar;
  const ClutterColor black = (ClutterColor){ 0x00, 0x00, 0x00, 0x5C };
  MwbPagePrivate *priv = self->priv = PAGE_PRIVATE (self);

  priv->scroll_view = nbtk_scroll_view_new ();
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->scroll_view),
                            CLUTTER_ACTOR (self));

  priv->dialog_bg = clutter_rectangle_new_with_color (&black);
  clutter_actor_set_opacity (priv->dialog_bg, 0x00);
  clutter_actor_set_parent (priv->dialog_bg, CLUTTER_ACTOR (self));

  bar = nbtk_scroll_view_get_hscroll_bar (NBTK_SCROLL_VIEW (priv->scroll_view));
  g_signal_connect_swapped (bar, "scroll-start",
                            G_CALLBACK (mwb_page_scroll_start_cb), self);
  g_signal_connect_swapped (bar, "scroll-stop",
                            G_CALLBACK (mwb_page_scroll_stop_cb), self);
  bar = nbtk_scroll_view_get_vscroll_bar (NBTK_SCROLL_VIEW (priv->scroll_view));
  g_signal_connect_swapped (bar, "scroll-start",
                            G_CALLBACK (mwb_page_scroll_start_cb), self);
  g_signal_connect_swapped (bar, "scroll-stop",
                            G_CALLBACK (mwb_page_scroll_stop_cb), self);

  g_signal_connect (self, "notify::reactive",
                    G_CALLBACK (mwb_page_notify_reactive), NULL);
}

NbtkWidget *
mwb_page_new (gboolean private)
{
  if (!private)
    return g_object_new (MWB_TYPE_PAGE, NULL);
  else
    return g_object_new (MWB_TYPE_PAGE,
                         "mozembed", mwb_mozembed_new (TRUE),
                         NULL);
}

NbtkWidget *
mwb_page_new_with_mozembed (MwbMozEmbed *mozembed)
{
  return g_object_new (MWB_TYPE_PAGE, "mozembed", mozembed, NULL);
}

NbtkWidget *
mwb_page_new_with_parent (MwbMozEmbed *parent)
{
  return g_object_new (MWB_TYPE_PAGE, "parent", parent, NULL);
}

void
mwb_page_set_mozembed (MwbPage *page, MwbMozEmbed *mozembed)
{
  MwbPagePrivate *priv = page->priv;

  if (priv->mozembed)
    {
      g_signal_handlers_disconnect_by_func (priv->mozembed,
                                            mwb_page_new_window_cb,
                                            page);
      g_signal_handlers_disconnect_by_func (priv->mozembed,
                                            mwb_page_download_cb,
                                            page);
      clutter_container_remove_actor (CLUTTER_CONTAINER (priv->scroll_view),
                                      priv->mozembed);
      priv->mozembed = NULL;
    }

  priv->mozembed = CLUTTER_ACTOR (mozembed);

  if (priv->mozembed)
    {
      clutter_container_add_actor (CLUTTER_CONTAINER (priv->scroll_view),
                                   priv->mozembed);
      g_signal_connect (priv->mozembed, "new-window",
                        G_CALLBACK (mwb_page_new_window_cb), page);
      g_signal_connect (priv->mozembed, "download",
                        G_CALLBACK (mwb_page_download_cb), page);
    }
}

ClutterMozEmbed *
mwb_page_get_mozembed (MwbPage *page)
{
  MwbPagePrivate *priv = page->priv;
  return CLUTTER_MOZEMBED (priv->mozembed);
}

void
mwb_page_set_download_manager (MwbPage *page, MwbDownloadManager *dlman)
{
  MwbPagePrivate *priv = page->priv;

  if (priv->dlman)
    g_object_unref (priv->dlman);

  priv->dlman = dlman ? g_object_ref (dlman) : NULL;
}

