/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <nm-device-ethernet.h>
#include <nm-device-wifi.h>
#include <nm-gsm-device.h>
#include <nm-cdma-device.h>

#include "nmn-status-icon.h"
#include "nmn-icon-cache.h"

typedef enum {
    STATUS_IMAGE_NO_NETWORK,
    STATUS_IMAGE_ETHERNET,
    STATUS_IMAGE_WWAN,
    STATUS_IMAGE_WIFI_00,
    STATUS_IMAGE_WIFI_25,
    STATUS_IMAGE_WIFI_50,
    STATUS_IMAGE_WIFI_75,
    STATUS_IMAGE_WIFI_100,
    STATUS_IMAGE_ACTIVATING,
} StatusImage;

#define ACTIVATION_STEPS 6

G_DEFINE_TYPE (NmnStatusIcon, nmn_status_icon, GTK_TYPE_STATUS_ICON)

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_STATUS_ICON, NmnStatusIconPrivate))

typedef struct {
    NMClient *client;
    GHashTable *icon_cache;
    StatusImage current_image;
    gboolean active;
    guint activation_step;

    NMActiveConnection *default_ac;
    gulong ac_state_changed_id;

    guint activation_animation_id;
    guint activation_animation_index;
} NmnStatusIconPrivate;

static StatusImage get_active_wifi_icon (NmnStatusIcon *self, NMDeviceWifi *device);

static char *
get_icon_filename (StatusImage image,
                   guint activation_step,
                   gboolean active)
{
    GString *str;

    str = g_string_sized_new (128);
    g_string_append (str, ICON_PATH);

    switch (image) {
    case STATUS_IMAGE_NO_NETWORK:
        g_string_append (str, "nm-no-connection");
        break;
    case STATUS_IMAGE_ETHERNET:
        g_string_append (str, "nm-device-wired");
        break;
    case STATUS_IMAGE_WWAN:
        g_string_append (str, "nm-device-wwan");
        break;
    case STATUS_IMAGE_WIFI_00:
        g_string_append (str, "nm-signal-00");
        break;
    case STATUS_IMAGE_WIFI_25:
        g_string_append (str, "nm-signal-25");
        break;
    case STATUS_IMAGE_WIFI_50:
        g_string_append (str, "nm-signal-50");
        break;
    case STATUS_IMAGE_WIFI_75:
        g_string_append (str, "nm-signal-75");
        break;
    case STATUS_IMAGE_WIFI_100:
        g_string_append (str, "nm-signal-100");
        break;
    case STATUS_IMAGE_ACTIVATING:
        g_string_append_printf (str, "nm-progress-working-%02d", activation_step);
        break;
    }

    g_string_append (str, active ? "-active" : "-normal");
    g_string_append (str, ".png");

    return g_string_free (str, FALSE);
}

static void
update_icon (NmnStatusIcon *self, StatusImage image)
{
    NmnStatusIconPrivate *priv = GET_PRIVATE (self);
    char *filename;
    GdkPixbuf *pixbuf;

    filename = get_icon_filename (image, priv->activation_step, priv->active);

    if (G_UNLIKELY (priv->icon_cache == NULL)) {
        priv->icon_cache = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, g_object_unref);
        pixbuf = NULL;
    } else {
        pixbuf = (GdkPixbuf *) g_hash_table_lookup (priv->icon_cache, filename);
    }

    if (!pixbuf) {
        GError *error = NULL;

        pixbuf = gdk_pixbuf_new_from_file (filename, &error);
        if (pixbuf)
            g_hash_table_insert (priv->icon_cache, filename, pixbuf);

        if (error) {
            g_warning ("Error loading status icon '%s': %s", filename, error->message);
            g_error_free (error);
        }
    }

    g_free (filename);

    if (pixbuf) {
        GtkStatusIcon *s = GTK_STATUS_ICON (self);

        priv->current_image = image;
        if (!(gtk_status_icon_get_storage_type (s) == GTK_IMAGE_PIXBUF &&
              gtk_status_icon_get_pixbuf (s) == pixbuf))

            gtk_status_icon_set_from_pixbuf (s, pixbuf);
    }
}

GtkStatusIcon *
nmn_status_icon_new (void)
{
    return GTK_STATUS_ICON (g_object_new (NMN_TYPE_STATUS_ICON, NULL));
}

void
nmn_status_icon_set_active (NmnStatusIcon *self,
                            gboolean active)
{
    NmnStatusIconPrivate *priv;

    g_return_if_fail (NMN_IS_STATUS_ICON (self));

    priv = GET_PRIVATE (self);
    if (priv->active != active) {
        priv->active = active;
        update_icon (self, priv->current_image);
    }
}

static StatusImage
get_active_ap_icon (NmnStatusIcon *self, NMAccessPoint *ap)
{
    StatusImage icon;
    guint32 strength;

    strength = nm_access_point_get_strength (ap);
    strength = CLAMP (strength, 0, 100);

    if (strength > 80)
        icon = STATUS_IMAGE_WIFI_100;
    else if (strength > 55)
        icon = STATUS_IMAGE_WIFI_75;
    else if (strength > 30)
        icon = STATUS_IMAGE_WIFI_50;
    else if (strength > 5)
        icon = STATUS_IMAGE_WIFI_25;
    else
        icon = STATUS_IMAGE_WIFI_00;

    return icon;
}

static void
ap_strength_changed (NMAccessPoint *ap, GParamSpec *pspec, gpointer user_data)
{
    NmnStatusIcon *self = NMN_STATUS_ICON (user_data);

    update_icon (self, get_active_ap_icon (self, ap));
}

typedef struct {
    NmnStatusIcon *status_icon;
    NMDeviceWifi *device;
    NMAccessPoint *ap;
    gulong ap_strength_id;
    gulong active_ap_id;
} WifiIconInfo;

static void
wifi_icon_info_destroy (gpointer data,
                        GClosure *closure)
{
    WifiIconInfo *info = (WifiIconInfo *) data;

    g_object_unref (info->ap);
    g_slice_free (WifiIconInfo, data);
}

static void
device_active_ap_changed (NMDeviceWifi *device, GParamSpec *pspec, gpointer user_data)
{
    WifiIconInfo *info = (WifiIconInfo *) user_data;
    NmnStatusIcon *self = info->status_icon;

    /* First, remove the old signal handlers */
    g_signal_handler_disconnect (info->device, info->active_ap_id);
    g_signal_handler_disconnect (info->ap, info->ap_strength_id);

    /* If the device is still active, it means we've roamed to another AP,
       set up new signal handlers and update the icon */
    if (nm_device_get_state (NM_DEVICE (device)) == NM_DEVICE_STATE_ACTIVATED)
        update_icon (self, get_active_wifi_icon (self, device));
}

static StatusImage
get_active_wifi_icon (NmnStatusIcon *self, NMDeviceWifi *device)
{
    NMAccessPoint *ap;
    StatusImage icon;

    ap = nm_device_wifi_get_active_access_point (device);
    if (ap) {
        WifiIconInfo *info;

        info = g_slice_new0 (WifiIconInfo);
        info->status_icon = self;
        info->device = device;
        info->ap = g_object_ref (ap);

        info->active_ap_id = g_signal_connect_data (device,
                                                    "notify::" NM_DEVICE_WIFI_ACTIVE_ACCESS_POINT,
                                                    G_CALLBACK (device_active_ap_changed),
                                                    info,
                                                    wifi_icon_info_destroy,
                                                    0);

        info->ap_strength_id = g_signal_connect (ap,
                                                 "notify::" NM_ACCESS_POINT_STRENGTH,
                                                 G_CALLBACK (ap_strength_changed),
                                                 self);

        icon = get_active_ap_icon (self, ap);
    } else
        icon = STATUS_IMAGE_WIFI_00;

    return icon;
}

static StatusImage
get_active_device_icon (NmnStatusIcon *self, NMDevice *device)
{
    StatusImage icon;

    if (NM_IS_DEVICE_ETHERNET (device))
        icon = STATUS_IMAGE_ETHERNET;
    else if (NM_IS_DEVICE_WIFI (device))
        icon = get_active_wifi_icon (self, NM_DEVICE_WIFI (device));
    else if (NM_IS_GSM_DEVICE (device) || NM_IS_CDMA_DEVICE (device))
        icon = STATUS_IMAGE_WWAN;
    else {
        g_warning ("Unhandled device type: '%s'", G_OBJECT_TYPE_NAME (device));
        icon = STATUS_IMAGE_NO_NETWORK;
    }

    return icon;
}

static gboolean
activation_animation (gpointer data)
{
    NmnStatusIcon *self = NMN_STATUS_ICON (data);
    NmnStatusIconPrivate *priv = GET_PRIVATE (self);

    if (++priv->activation_step > ACTIVATION_STEPS)
        priv->activation_step = 1;

    update_icon (self, STATUS_IMAGE_ACTIVATING);

    return TRUE;
}

/***********************************************************************************/

static void
ac_state_changed (NMActiveConnection *ac,
                  GParamSpec *pspec,
                  gpointer user_data)
{
    NmnStatusIcon *self = NMN_STATUS_ICON (user_data);
    NmnStatusIconPrivate *priv = GET_PRIVATE (self);
    const GPtrArray *active_devices;
    NMDevice *device = NULL;
    StatusImage icon;
    NMActiveConnectionState state = NM_ACTIVE_CONNECTION_STATE_UNKNOWN;

    /* Cancel any ongoing activation animantion */
    if (priv->activation_animation_id) {
        g_source_remove (priv->activation_animation_id);
        priv->activation_animation_id = 0;
        priv->activation_animation_index = 0;
    }

    if (ac) {
        active_devices = nm_active_connection_get_devices (ac);
        if (active_devices) {
            device = NM_DEVICE (g_ptr_array_index (active_devices, 0));
            state = nm_active_connection_get_state (ac);
        }
    }

    switch (state) {
    case NM_ACTIVE_CONNECTION_STATE_UNKNOWN:
        icon = STATUS_IMAGE_NO_NETWORK;;
        break;
    case NM_ACTIVE_CONNECTION_STATE_ACTIVATING:
        priv->activation_animation_id = g_timeout_add (200, activation_animation, self);
        activation_animation (self);
        return;
        break;
    case NM_ACTIVE_CONNECTION_STATE_ACTIVATED:
        icon = get_active_device_icon (self, device);
        break;
    }

    update_icon (self, icon);
}

static void
update_best_ac (NmnStatusIcon *self)
{
    NmnStatusIconPrivate *priv = GET_PRIVATE (self);
    const GPtrArray *acs;
    NMActiveConnection *best_ac = NULL;
    int i;

    if (nm_client_get_manager_running (priv->client))
        acs = nm_client_get_active_connections (priv->client);
    else
        acs = NULL;

    for (i = 0; acs && i < acs->len; i++) {
        NMActiveConnection *ac = g_ptr_array_index (acs, i);

        if (nm_active_connection_get_state (ac) == NM_ACTIVE_CONNECTION_STATE_UNKNOWN)
            continue;

        if (!best_ac ||
            nm_active_connection_get_state (ac) > nm_active_connection_get_state (best_ac) ||
            nm_active_connection_get_default (ac) == TRUE)
            best_ac = ac;
    }

    if (priv->default_ac && priv->default_ac != best_ac) {
        g_signal_handler_disconnect (priv->default_ac, priv->ac_state_changed_id);
        g_object_unref (priv->default_ac);
        priv->default_ac = NULL;
    }

    if (best_ac) {
        priv->default_ac = best_ac;
        priv->default_ac = g_object_ref (best_ac);
        priv->ac_state_changed_id = g_signal_connect (best_ac, "notify",
                                                      G_CALLBACK (ac_state_changed), self);
    }

    ac_state_changed (best_ac, NULL, self);
}

static void
active_connections_changed (NMClient *client,
                            GParamSpec *pspec,
                            gpointer user_data)
{
    update_best_ac (NMN_STATUS_ICON (user_data));
}

void
nmn_status_icon_set_client (NmnStatusIcon *self,
                            NMClient *client)
{
    NmnStatusIconPrivate *priv;

    g_return_if_fail (NMN_IS_STATUS_ICON (self));
    g_return_if_fail (NM_IS_CLIENT (client));

    priv = GET_PRIVATE (self);
    priv->client = g_object_ref (client);

    g_signal_connect (priv->client, "notify::" NM_CLIENT_ACTIVE_CONNECTIONS,
	                  G_CALLBACK (active_connections_changed),
	                  self);

    update_best_ac (self);
}

/*****************************************************************************/

static gboolean
set_initial_icon (gpointer data)
{
    update_icon (NMN_STATUS_ICON (data), STATUS_IMAGE_NO_NETWORK);

    return FALSE;
}

static void
nmn_status_icon_init (NmnStatusIcon *self)
{
    g_idle_add (set_initial_icon, self);
}

static void
finalize (GObject *object)
{
    NmnStatusIconPrivate *priv = GET_PRIVATE (object);

    if (priv->activation_animation_id) {
        g_source_remove (priv->activation_animation_id);
        priv->activation_animation_id = 0;
    }

    if (priv->icon_cache)
        g_hash_table_destroy (priv->icon_cache);

    if (priv->client)
        g_object_unref (priv->client);

    G_OBJECT_CLASS (nmn_status_icon_parent_class)->finalize (object);
}

static void
nmn_status_icon_class_init (NmnStatusIconClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnStatusIconPrivate));

    object_class->finalize = finalize;
}
