/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <string.h>
#include <NetworkManager.h>
#include <nm-settings.h>
#include <nm-device-ethernet.h>
#include <nm-device-wifi.h>
#include "nmn-networks.h"
#include "nmn-network-item.h"
#include "nmn-ethernet-item.h"
#include "nmn-wifi-item.h"
#include "nmn-serial-item.h"
#include "nmn-device-handler.h"
#include "nmn-ethernet-handler.h"
#include "nmn-serial-handler.h"
#include "nmn-wifi-handler.h"

#define DBUS_TYPE_G_ARRAY_OF_OBJECT_PATH (dbus_g_type_get_collection ("GPtrArray", DBUS_TYPE_G_OBJECT_PATH))

G_DEFINE_TYPE (NmnNetworks, nmn_networks, NMN_TYPE_LIST)

enum {
    PROP_0,
    PROP_NM_DATA,

    LAST_PROP
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_NETWORKS, NmnNetworksPrivate))

typedef struct {
    NmnNMData *nm_data;
    GSList *handlers;
    gboolean populated;

    gboolean disposed;
} NmnNetworksPrivate;

GtkWidget *
nmn_networks_new (NmnNMData *nm_data)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (nm_data), NULL);

    return GTK_WIDGET (g_object_new (NMN_TYPE_NETWORKS,
                                     NMN_NETWORKS_NM_DATA, nm_data,
                                     NULL));
}

static void
find_ac_for_item (NmnNetworks *self, NmnNetworkItem *item)
{
    NmnNetworksPrivate *priv = GET_PRIVATE (self);
    NMConnection *connection;
    NMConnectionScope scope;
    const char *path;
    const GPtrArray *acs;
    NMActiveConnection *this_ac = NULL;
    int i;

    acs = nm_client_get_active_connections (NM_CLIENT (priv->nm_data));
    connection = nm_exported_connection_get_connection (nmn_network_item_get_connection (item));
    path = nm_connection_get_path (connection);
    scope = nm_connection_get_scope (connection);

    for (i = 0; acs && i < acs->len; i++) {
        NMActiveConnection *ac = g_ptr_array_index (acs, i);

        if (scope == nm_active_connection_get_scope (ac) &&
            !strcmp (path, nm_active_connection_get_connection (ac))) {
            this_ac = ac;
            break;
        }
    }

    nmn_network_item_set_active_connection (item, this_ac);
}

typedef struct {
    NmnItem *item;
    NMExportedConnection *exported;
    NMDevice *device;
    GSList *items_to_remove;
} RemoveInfo;

static void
remove_connections_cb (GtkWidget *widget, gpointer data)
{
    NmnNetworkItem *item = NMN_NETWORK_ITEM (widget);
    RemoveInfo *info = (RemoveInfo *) data;

    if ((info->item && (NmnItem *) item == info->item) ||
        (info->exported && nmn_network_item_get_connection (item) == info->exported) ||
        (info->device && nmn_network_item_get_device (item) == info->device))

        info->items_to_remove = g_slist_prepend (info->items_to_remove, g_object_ref (widget));
}

static void
remove_connections (NmnNetworks *self,
                    NmnItem *item,
                    NMExportedConnection *connection,
                    NMDevice *device)
{
    RemoveInfo info;

    info.item = item;
    info.exported = connection;
    info.device = device;
    info.items_to_remove = NULL;

    gtk_container_foreach (GTK_CONTAINER (self), remove_connections_cb, &info);

    while (info.items_to_remove) {
        gtk_container_remove (GTK_CONTAINER (self), GTK_WIDGET (info.items_to_remove->data));
        g_object_unref (info.items_to_remove->data);

        info.items_to_remove = g_slist_delete_link (info.items_to_remove, info.items_to_remove);
    }
}

static void
item_remove_requested (NmnItem *item,
                       gpointer user_data)
{
    remove_connections (NMN_NETWORKS (user_data), item, NULL, NULL);
}

static void
item_added (NmnDeviceHandler *handler,
            NmnItem *item,
            gpointer user_data)
{
    NmnNetworks *self = NMN_NETWORKS (user_data);

    nmn_list_add_item (NMN_LIST (self), item);
    g_signal_connect (item, "remove-requested", G_CALLBACK (item_remove_requested), self);

    find_ac_for_item (self, NMN_NETWORK_ITEM (item));
}

static void
device_added (NMClient *client,
              NMDevice *device,
              gpointer user_data)
{
    NmnNetworks *self = NMN_NETWORKS (user_data);
    NmnNetworksPrivate *priv = GET_PRIVATE (self);
    NmnDeviceHandler *handler;

    if (NM_IS_DEVICE_WIFI (device))
        handler = nmn_wifi_handler_new (priv->nm_data, NM_DEVICE_WIFI (device));
    else if (NM_IS_DEVICE_ETHERNET (device))
        handler = nmn_ethernet_handler_new (priv->nm_data, NM_DEVICE_ETHERNET (device));
    else if (NM_IS_SERIAL_DEVICE (device))
        handler = nmn_serial_handler_new (priv->nm_data, NM_SERIAL_DEVICE (device));
    else {
        handler = NULL;
        g_warning ("Unhandled device type: '%s'", G_OBJECT_TYPE_NAME (device));
    }

    if (handler) {
        priv->handlers = g_slist_prepend (priv->handlers, handler);
        g_signal_connect (handler, "item-added", G_CALLBACK (item_added), self);
        nmn_device_handler_start (handler);
    }
}

static void
device_removed (NMClient *client,
                NMDevice *device,
                gpointer user_data)
{
    NmnNetworks *self = NMN_NETWORKS (user_data);
    NmnNetworksPrivate *priv = GET_PRIVATE (self);
    GSList *iter;

    for (iter = priv->handlers; iter; iter = iter->next) {
        if (nmn_device_handler_get_device (NMN_DEVICE_HANDLER (iter->data)) == device) {
            remove_connections (self, NULL, NULL, device);
            g_object_unref (iter->data);
            priv->handlers = g_slist_delete_link (priv->handlers, iter);
            break;
        }
    }
}

static void
acs_changed_cb (GtkWidget *widget, gpointer data)
{
    find_ac_for_item (NMN_NETWORKS (data), NMN_NETWORK_ITEM (widget));
}

static void
active_connections_changed (NMClient *client,
                            GParamSpec *pspec,
                            gpointer user_data)
{
    NmnNetworks *self = NMN_NETWORKS (user_data);

    gtk_container_foreach (GTK_CONTAINER (self), acs_changed_cb, self);
}

void
nmn_networks_populate (NmnNetworks *self)
{
    NmnNetworksPrivate *priv;
    const GPtrArray *devices;
    int i;

    g_return_if_fail (NMN_IS_NETWORKS (self));

    priv = GET_PRIVATE (self);
    if (priv->populated)
        return;

    priv->populated = TRUE;

    g_signal_connect (priv->nm_data, "device-added",
                      G_CALLBACK (device_added),
                      self);

    g_signal_connect (priv->nm_data, "device-removed",
                      G_CALLBACK (device_removed),
                      self);

    g_signal_connect (priv->nm_data, "notify::" NM_CLIENT_ACTIVE_CONNECTIONS,
	                  G_CALLBACK (active_connections_changed),
	                  self);

    devices = nm_client_get_devices (NM_CLIENT (priv->nm_data));
    for (i = 0; devices && i < devices->len; i++)
        device_added (NM_CLIENT (priv->nm_data), NM_DEVICE (g_ptr_array_index (devices, i)), self);
}

static void
nmn_networks_init (NmnNetworks *networks)
{
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnNetworksPrivate *priv;

    object = G_OBJECT_CLASS (nmn_networks_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    if (!priv->nm_data) {
        g_warning ("Missing constructor arguments");
        g_object_unref (object);
        return NULL;
    }

    return object;
}

static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
    NmnNetworksPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NM_DATA:
        /* Construct only */
        priv->nm_data = g_value_dup_object (value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
    NmnNetworksPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NM_DATA:
        g_value_set_object (value, priv->nm_data);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
dispose (GObject *object)
{
    NmnNetworksPrivate *priv = GET_PRIVATE (object);

    if (priv->disposed)
        return;

    g_slist_foreach (priv->handlers, (GFunc) g_object_unref, NULL);
    g_slist_free (priv->handlers);

    g_object_unref (priv->nm_data);

    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_networks_parent_class)->dispose (object);
}

static void
nmn_networks_class_init (NmnNetworksClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnNetworksPrivate));

    object_class->constructor = constructor;
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->dispose = dispose;

    /* properties */
    g_object_class_install_property
        (object_class, PROP_NM_DATA,
         g_param_spec_object (NMN_NETWORKS_NM_DATA,
                              "NmnNMData",
                              "NmnNMData",
                              NMN_TYPE_NM_DATA,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}
