/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** json-parser.c - implements json-based resource/value loading
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <gdk/gdk.h>

#include "json-parser.h"

G_DEFINE_TYPE (Parser, parser, G_TYPE_OBJECT);

/*-- internal API ------------------------------------------------------------*/

static void
parser_dispose (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (parser_parent_class)->dispose (gobject);
}

static void
parser_finalize (GObject* gobject)
{
	if (PARSER(gobject)->json_parser != NULL)
		g_object_unref (PARSER(gobject)->json_parser);

	/* chain up to the parent class */
	G_OBJECT_CLASS (parser_parent_class)->finalize (gobject);
}

static void
parser_init (Parser* self)
{
	/* If you need specific construction properties to complete
	** initialization, delay initialization completion until the
	** property is set. */

	self->json_parser = NULL;
	self->root_node   = NULL;
}

static void
parser_get_property (GObject*    gobject,
                     guint       prop,
                     GValue*     value,
                     GParamSpec* spec)
{
	switch (prop)
	{
		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
parser_class_init (ParserClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	gobject_class->dispose      = parser_dispose;
	gobject_class->finalize     = parser_finalize;
	gobject_class->get_property = parser_get_property;
}

JsonObject*
_get_node_object (Parser* self,
                  gchar*  node_name)
{
	JsonObject* object = json_node_get_object (self->root_node);
	if (!object)
		return NULL;
	
	JsonNode* node = json_object_get_member (object, node_name);
	if (!node)
		return NULL;

	return json_node_get_object (node);
}

JsonArray*
_get_array (Parser* self,
            gchar*  node_name,
            gchar*  member_name)
{
	JsonObject* object = _get_node_object (self, node_name);
	if (!object)
		return NULL;

	return json_object_get_array_member (object, member_name);
}

/*-- public API --------------------------------------------------------------*/

Parser*
parser_new ()
{
	Parser*  self   = NULL;

	self = g_object_new (PARSER_TYPE, NULL);
	if (!self)
		return NULL;

	self->json_parser = json_parser_new ();

	return self;
}

gboolean
parser_open (Parser* self,
             gchar*  file_name)
{
	gboolean result = FALSE;
	GError*  error  = NULL;

	/* sanity check */
	if (!self || !IS_PARSER (self) || !file_name)
		return FALSE;

	result = json_parser_load_from_file (self->json_parser, file_name, &error);
	if (!result)
	{
		g_message ("%s() - \"%s\"\n", G_STRFUNC, error->message);
		g_error_free (error);
		return FALSE;
	}

	self->root_node = json_parser_get_root (self->json_parser);

	return result;
}

gint
parser_read_int (Parser* self,
                 gchar*  node_name,
                 gchar*  member_name)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name)
		return 0;

	JsonObject* member_object = _get_node_object (self, node_name);
	if (!member_object)
		return 0;

	return (gint) json_object_get_int_member (member_object, member_name);
}

gint*
parser_read_ints (Parser* self,
                  gchar*  node_name,
                  gchar*  member_name,
                  guint*  size)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name || !size)
		return NULL;

	JsonArray* array = _get_array (self, node_name, member_name);
	if (!array)
		return NULL;

	*size = json_array_get_length (array);

	gint i = 0;
	gint* values = (gint*) g_malloc0 (*size);
	for (i = 0; i < *size; ++i)
		values[i] = json_array_get_int_element (array, i);

	return values;
}

gdouble
parser_read_double (Parser* self,
                    gchar*  node_name,
                    gchar*  member_name)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name)
		return 0;

	JsonObject* member_object = _get_node_object (self, node_name);
	if (!member_object)
		return 0;

	return (gdouble) json_object_get_double_member (member_object, member_name);
}

gdouble*
parser_read_doubles (Parser* self,
                     gchar*  node_name,
                     gchar*  member_name,
                     guint*  size)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name || !size)
		return NULL;

	JsonArray* array = _get_array (self, node_name, member_name);
	if (!array)
		return NULL;

	*size = json_array_get_length (array);

	gint i = 0;
	gdouble* values = (gdouble*) g_malloc0 (*size * sizeof(gdouble));
	for (i = 0; i < *size; ++i)
		values[i] = json_array_get_double_element (array, i);

	return values;
}

gboolean
parser_read_color (Parser*   self,
                   gchar*    node_name,
                   gchar*    member_name,
                   GdkColor* color)
{
	gboolean result = FALSE;

	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name || !color)
		return FALSE;

	JsonObject* member_object = _get_node_object (self, node_name);
	if (!member_object)
		return FALSE;
	
	result = gdk_color_parse (json_object_get_string_member (member_object,
	                                                         member_name),
	                          color);
	if (!result)
		return FALSE;

	return result;
}

GdkColor*
parser_read_colors (Parser* self,
                    gchar*  node_name,
                    gchar*  member_name,
                    guint*  size)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name || !size)
		return NULL;

	JsonArray* array = _get_array (self, node_name, member_name);
	if (!array)
		return NULL;

	*size = json_array_get_length (array);

	gint i = 0;
	GdkColor* colors = (GdkColor*) g_malloc0 (*size * sizeof (GdkColor));
	for (i = 0; i < *size; ++i)
	{
		gdk_color_parse (json_array_get_string_element (array, i),
		                 &colors[i]);
	}

	return colors;
}

FontStyle
parser_read_font_style (Parser* self,
                        gchar*  node_name,
                        gchar*  member_name)
{
	FontStyle style = FONT_STYLE_NONE;

	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name)
		return FONT_STYLE_NONE;

	JsonObject* member_object = _get_node_object (self, node_name);
	if (!member_object)
		return FONT_STYLE_NONE;

	const gchar* string = json_object_get_string_member (member_object,
	                                                     member_name);
	if (!string)
		return FONT_STYLE_NONE;

	if (!g_strcmp0 (string, "normal"))
		style = FONT_STYLE_NORMAL;
	else if (!g_strcmp0 (string, "oblique"))
		style = FONT_STYLE_OBLIQUE;
	else if (!g_strcmp0 (string, "italic"))
		style = FONT_STYLE_ITALIC;
	else
		style = FONT_STYLE_NONE;

	return style;
}

FontStyle*
parser_read_font_styles (Parser* self,
                         gchar*  node_name,
                         gchar*  member_name,
                         guint*  size)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name || !size)
		return NULL;

	JsonArray* array = _get_array (self, node_name, member_name);
	if (!array)
		return NULL;

	*size = json_array_get_length (array);

	gint i = 0;
	FontStyle* styles = (FontStyle*) g_malloc0 (*size * sizeof (FontStyle));
	for (i = 0; i < *size; ++i)
	{
		const gchar* string = json_array_get_string_element (array, i);

		if (!g_strcmp0 (string, FONT_STYLE_NORMAL_NAME))
			styles[i] = FONT_STYLE_NORMAL;
		else if (!g_strcmp0 (string, FONT_STYLE_OBLIQUE_NAME))
			styles[i] = FONT_STYLE_OBLIQUE;
		else if (!g_strcmp0 (string, FONT_STYLE_ITALIC_NAME))
			styles[i] = FONT_STYLE_ITALIC;
		else
			styles[i] = FONT_STYLE_NONE;
	}

	return styles;
}

FontWeight
parser_read_font_weight (Parser* self,
                         gchar*  node_name,
                         gchar*  member_name)
{
	FontWeight weight = FONT_WEIGHT_NONE;

	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name)
		return FONT_WEIGHT_NONE;

	JsonObject* member_object = _get_node_object (self, node_name);
	if (!member_object)
		return FONT_WEIGHT_NONE;

	const gchar* string = json_object_get_string_member (member_object,
	                                                     member_name);
	if (!string)
		return FONT_WEIGHT_NONE;

	if (!g_strcmp0 (string, FONT_WEIGHT_LIGHT_NAME))
		weight = FONT_WEIGHT_LIGHT;
	else if (!g_strcmp0 (string, FONT_WEIGHT_NORMAL_NAME))
		weight = FONT_WEIGHT_NORMAL;
	else if (!g_strcmp0 (string, FONT_WEIGHT_BOLD_NAME))
		weight = FONT_WEIGHT_BOLD;
	else
		weight = FONT_WEIGHT_NONE;

	return weight;
}

FontWeight*
parser_read_font_weights (Parser* self,
                          gchar*  node_name,
                          gchar*  member_name,
                          guint*  size)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name || !size)
		return NULL;

	JsonArray* array = _get_array (self, node_name, member_name);
	if (!array)
		return NULL;

	*size = json_array_get_length (array);

	gint i = 0;
	FontWeight* weights = (FontWeight*) g_malloc0 (*size * sizeof (FontWeight));
	for (i = 0; i < *size; ++i)
	{
		const gchar* string = json_array_get_string_element (array, i);

		if (!g_strcmp0 (string, FONT_WEIGHT_LIGHT_NAME))
			weights[i] = FONT_WEIGHT_LIGHT;
		else if (!g_strcmp0 (string, FONT_WEIGHT_NORMAL_NAME))
			weights[i] = FONT_WEIGHT_NORMAL;
		else if (!g_strcmp0 (string, FONT_WEIGHT_BOLD_NAME))
			weights[i] = FONT_WEIGHT_BOLD;
		else
			weights[i] = FONT_WEIGHT_NONE;
	}

	return weights;
}

BlendMode
parser_read_blend_mode (Parser* self,
                        gchar*  node_name,
                        gchar*  member_name)
{
	BlendMode mode = BLEND_MODE_NONE;

	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name)
		return BLEND_MODE_NONE;

	JsonObject* member_object = _get_node_object (self, node_name);
	if (!member_object)
		return BLEND_MODE_NONE;

	const gchar* string = json_object_get_string_member (member_object,
	                                                     member_name);
	if (!string)
		return BLEND_MODE_NONE;

	if (!g_strcmp0 (string, BLEND_MODE_SOURCE_NAME))
		mode = BLEND_MODE_SOURCE;
	else if (!g_strcmp0 (string, BLEND_MODE_OVER_NAME))
		mode = BLEND_MODE_OVER;
	else
		mode = BLEND_MODE_NONE;

	return mode;
}

BlendMode*
parser_read_blend_modes (Parser* self,
                         gchar*  node_name,
                         gchar*  member_name,
                         guint*  size)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name || !size)
		return NULL;

	JsonArray* array = _get_array (self, node_name, member_name);
	if (!array)
		return NULL;

	*size = json_array_get_length (array);

	gint i = 0;
	BlendMode* modes = (BlendMode*) g_malloc0 (*size * sizeof (BlendMode));
	for (i = 0; i < *size; ++i)
	{
		const gchar* string = json_array_get_string_element (array, i);

		if (!g_strcmp0 (string, BLEND_MODE_SOURCE_NAME))
			modes[i] = BLEND_MODE_SOURCE;
		else if (!g_strcmp0 (string, BLEND_MODE_OVER_NAME))
			modes[i] = BLEND_MODE_OVER;
		else
			modes[i] = BLEND_MODE_NONE;
	}

	return modes;
}

gchar*
parser_read_string (Parser* self,
                    gchar*  node_name,
                    gchar*  member_name)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name)
		return NULL;

	JsonObject* member_object = _get_node_object (self, node_name);
	if (!member_object)
		return NULL;

	const gchar* string = json_object_get_string_member (member_object,
	                                                     member_name);
	if (!string)
		return NULL;

	return string;
}

gchar**
parser_read_strings (Parser* self,
                     gchar*  node_name,
                     gchar*  member_name,
                     guint*  size)
{
	/* sanity check */
	if (!self || !IS_PARSER (self) || !node_name || !member_name || !size)
		return NULL;

	JsonArray* array = _get_array (self, node_name, member_name);
	if (!array)
		return NULL;

	*size = json_array_get_length (array);

	gint i = 0;
	gchar** strings = (gchar**) g_malloc0 (*size * sizeof (gchar*));
	for (i = 0; i < *size; ++i)
		strings[i] = g_strdup (json_array_get_string_element (array, i));

	return strings;
}
