/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** test-button.c - unit-tests for button class used by snap-decisions
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <glib.h>
#include <cairo.h>

#include "config.h"
#include "button.h"
#include "settings.h"

static void
test_button_new ()
{
	Button*   button   = NULL;
	Settings* settings = settings_new ();

	button = button_new (0, 0, 100, 10, FALSE, settings->button, "label");
	g_assert (button != NULL);
	g_assert (IS_BUTTON (button));
	g_object_unref (button);
	button = NULL;

	button = button_new (0, 0, 100, 10, FALSE, settings->button, NULL);
	g_assert (button != NULL);
	g_assert (IS_BUTTON (button));
	g_object_unref (button);
	settings_del (settings);
}

static void
test_button_del ()
{
	Button*   button   = NULL;
	Settings* settings = settings_new ();

	button = button_new (0, 0, 100, 10, FALSE, settings->button, "label");
	g_assert (button != NULL);
	g_assert (IS_BUTTON (button));

	button_del (button);
	g_assert (!IS_BUTTON (button));
	button = NULL;
	settings_del (settings);
}

static void
test_button_getset_position ()
{
	Button*   button   = NULL;
	Settings* settings = settings_new ();

	button = button_new (20, 35, 50, 50, FALSE, settings->button, "label");
	g_assert (button != NULL);
	g_assert (IS_BUTTON (button));

	gint x = 0;
	gint y = 0;

	button_get_position (button, &x, &y);
	g_assert_cmpint (x, ==, 20);
	g_assert_cmpint (y, ==, 35);
	button_set_position (button, 42, 69);
	button_get_position (button, &x, &y);
	g_assert_cmpint (x, ==, 42);
	g_assert_cmpint (y, ==, 69);

	button_del (button);
	settings_del (settings);
}

static void
test_button_is_hit ()
{
	Button*   button   = NULL;
	Settings* settings = settings_new ();

	button = button_new (10, 10, 50, 20, FALSE, settings->button, "label");
	g_assert (button != NULL);
	g_assert (IS_BUTTON (button));

	gboolean result = FALSE;
	result = button_is_hit (button, 2, 2);
	g_assert (!result);
	result = button_is_hit (button, 20, 20);
	g_assert (result);
	result = button_is_hit (button, 70, 40);
	g_assert (!result);

	button_del (button);
	settings_del (settings);
}

static void
test_button_set_state ()
{
	Button*   button   = NULL;
	Settings* settings = settings_new ();

	button = button_new (10, 10, 50, 20, FALSE, settings->button, "label");
	g_assert (button != NULL);
	g_assert (IS_BUTTON (button));

	g_assert_cmpint (button_get_state (button), ==, BUTTON_STATE_NORMAL);
	button_set_state (button, BUTTON_STATE_HOVER);
	g_assert_cmpint (button_get_state (button), ==, BUTTON_STATE_HOVER);
	button_set_state (button, BUTTON_STATE_PRESSED);
	g_assert_cmpint (button_get_state (button), ==, BUTTON_STATE_PRESSED);
	button_set_state (button, BUTTON_STATE_NORMAL);
	g_assert_cmpint (button_get_state (button), ==, BUTTON_STATE_NORMAL);
	
	button_del (button);
	settings_del (settings);
}

static void
test_button_paint ()
{
	Button*   button   = NULL;
	Settings* settings = settings_new ();

	button = button_new (10, 10, 50, 20, FALSE, settings->button, "label");
	g_assert (button != NULL);
	g_assert (IS_BUTTON (button));

	cairo_surface_t* surface = NULL;
	cairo_t*         cr      = NULL;

	surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 130, 100);
	cr = cairo_create (surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

	button_set_button_tint (button, FALSE);
	button_set_state (button, BUTTON_STATE_NORMAL);
	button_set_position (button, 10, 10);
	button_paint (button, cr);
	button_set_state (button, BUTTON_STATE_HOVER);
	button_set_position (button, 10, 40);
	button_paint (button, cr);
	button_set_state (button, BUTTON_STATE_PRESSED);
	button_set_position (button, 10, 70);
	button_paint (button, cr);

	button_set_button_tint (button, TRUE);
	button_set_state (button, BUTTON_STATE_NORMAL);
	button_set_position (button, 70, 10);
	button_paint (button, cr);
	button_set_state (button, BUTTON_STATE_HOVER);
	button_set_position (button, 70, 40);
	button_paint (button, cr);
	button_set_state (button, BUTTON_STATE_PRESSED);
	button_set_position (button, 70, 70);
	button_paint (button, cr);

	cairo_destroy (cr);
	cairo_surface_write_to_png (surface, "/tmp/test-button-paint.png");
	cairo_surface_destroy (surface);

	button_del (button);
	settings_del (settings);
}

GTestSuite *
test_button_create_test_suite (void)
{
	GTestSuite *ts = NULL;

	ts = g_test_create_suite ("button");

#define TC(x) g_test_create_case(#x, 0, NULL, NULL, x, NULL)

	g_test_suite_add(ts, TC(test_button_new));
	g_test_suite_add(ts, TC(test_button_del));
	g_test_suite_add(ts, TC(test_button_getset_position));
	g_test_suite_add(ts, TC(test_button_is_hit));
	g_test_suite_add(ts, TC(test_button_set_state));
	g_test_suite_add(ts, TC(test_button_paint));

	return ts;
}
