#include "iwl-drv.h"
#include "iwl-config.h"
#include "iwl-prph.h"
#include "iwl-trans.h"
#include "pcie/internal.h"
#include "idi_internal.h"
#include "idi_utils.h"
#include "idi_al.h"
#include "iwl-debug.h"
#include "iwl-io.h"
#include "idi_tx.h"
#include "shared.h"
#include "idi_tx_policy.h"

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
#include "iwl-sfdb-hal.h"
#else
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/idi/idi_interface.h>

#include <linux/reset.h>
#include <linux/clk.h>
#include <linux/of.h>
#endif

/* if not running compatible al_emulation - ignore changes */
#ifndef IWL_IDI_INTA
#define iwl_inta_em_clean_irq_called(trans) 0
#define iwl_inta_em_read_called(trans) 0
#else
/* for simulation of irq-register. will be replaced with target access */
#include "iwl-em-intr.h"
#endif

#define IWL_IDI_NAME  "IWL_IDI"
#define IWL_IDI_AL_MEM "wlan"
#define IWL_IDI_IRQ_NAME "wlan_irq"

#define IWL_IDI_IS_Q_PAN(q) (((q) > 3) && ((q) < 8))

#define ADDR_IN_AL_MSK (0x80000000)
#define IS_AL_ADDR(ofs) ((ofs) & (ADDR_IN_AL_MSK))

static u32 iwl_trans_idi_read32(struct iwl_trans *trans, u32 ofs);

static const struct iwl_trans_ops trans_ops_idi;

/**
 * clear the idi status bit as well as the status bit of the incorporated pcie
 */
#define idi_clear_status_bit(_bit, _trans_slv) \
	do { \
		clear_bit(_bit, &(IWL_TRANS_GET_PCIE_TRANS\
			((IWL_TRANS_SLV_GET_IDI_TRANS(_trans_slv))\
			->trans_pcie))->status); \
		clear_bit(_bit, &_trans_slv->status); \
	} while (0)

/**
 * copied from pcie/trans. not dependent on pcie but not accessible from here
 */
static const char *get_fh_string(int cmd)
{
	switch (cmd) {
	case FH_RSCSR_CHNL0_STTS_WPTR_REG:
		return "FH_RSCSR_CHNL0_STTS_WPTR_REG";

	case FH_RSCSR_CHNL0_RBDCB_BASE_REG:
			return "FH_RSCSR_CHNL0_RBDCB_BASE_REG";

	case FH_RSCSR_CHNL0_WPTR:
			return "FH_RSCSR_CHNL0_WPTR";

	case FH_MEM_RCSR_CHNL0_CONFIG_REG:
			return "FH_MEM_RCSR_CHNL0_CONFIG_REG";

	case FH_MEM_RSSR_SHARED_CTRL_REG:
			return "FH_MEM_RSSR_SHARED_CTRL_REG";

	case FH_MEM_RSSR_RX_STATUS_REG:
			return "FH_MEM_RSSR_RX_STATUS_REG";

	case FH_MEM_RSSR_RX_ENABLE_ERR_IRQ2DRV:
			return "FH_MEM_RSSR_RX_ENABLE_ERR_IRQ2DRV";

	case FH_TSSR_TX_STATUS_REG:
			return "FH_TSSR_TX_STATUS_REG";

	case FH_TSSR_TX_ERROR_REG:
				return "FH_TSSR_TX_ERROR_REG";

	default:
		return "UNKNOWN";
	}
}

static const char *get_csr_string(int cmd)
{
	switch (cmd) {
	case CSR_HW_IF_CONFIG_REG: return "CSR_HW_IF_CONFIG_REG";
	case CSR_INT_COALESCING: return "CSR_INT_COALESCING";
	case CSR_INT: return "CSR_INT";
	case CSR_INT_MASK: return "CSR_INT_MASK";
	case CSR_FH_INT_STATUS: return "CSR_FH_INT_STATUS";
	case CSR_GPIO_IN: return "CSR_GPIO_IN";
	case CSR_RESET: return "CSR_RESET";
	case CSR_GP_CNTRL: return "CSR_GP_CNTRL";
	case CSR_HW_REV: return "CSR_HW_REV";
	case CSR_EEPROM_REG: return "CSR_EEPROM_REG";
	case CSR_EEPROM_GP: return "CSR_EEPROM_GP";
	case CSR_OTP_GP_REG: return "CSR_OTP_GP_REG";
	case CSR_GIO_REG: return "CSR_GIO_REG";
	case CSR_GP_UCODE_REG: return "CSR_GP_UCODE_REG";
	case CSR_GP_DRIVER_REG: return "CSR_GP_DRIVER_REG";
	case CSR_UCODE_DRV_GP1: return "CSR_UCODE_DRV_GP1";
	case CSR_UCODE_DRV_GP2: return "CSR_UCODE_DRV_GP2";
	case CSR_LED_REG: return "CSR_LED_REG";
	case CSR_DRAM_INT_TBL_REG: return "CSR_DRAM_INT_TBL_REG";
	case CSR_GIO_CHICKEN_BITS: return "CSR_GIO_CHICKEN_BITS";
	case CSR_ANA_PLL_CFG: return "CSR_ANA_PLL_CFG";
	case CSR_HW_REV_WA_REG: return "CSR_HW_REV_WA_REG";
	case CSR_DBG_HPET_MEM_REG: return "CSR_DBG_HPET_MEM_REG";
	default:
		return "UNKNOWN";
	}
}

/**
 * taken from pcie with idi i/o
 */
static void iwl_idi_dump_csr(struct iwl_trans *trans)
{
	int i;
	static const u32 csr_tbl[] = {
		CSR_HW_IF_CONFIG_REG,
		CSR_INT_COALESCING,
		CSR_INT,
		CSR_INT_MASK,
		CSR_FH_INT_STATUS,
		CSR_GPIO_IN,
		CSR_RESET,
		CSR_GP_CNTRL,
		CSR_HW_REV,
		CSR_EEPROM_REG,
		CSR_EEPROM_GP,
		CSR_OTP_GP_REG,
		CSR_GIO_REG,
		CSR_GP_UCODE_REG,
		CSR_GP_DRIVER_REG,
		CSR_UCODE_DRV_GP1,
		CSR_UCODE_DRV_GP2,
		CSR_LED_REG,
		CSR_DRAM_INT_TBL_REG,
		CSR_GIO_CHICKEN_BITS,
		CSR_ANA_PLL_CFG,
		CSR_HW_REV_WA_REG,
		CSR_DBG_HPET_MEM_REG
	};
	IWL_ERR(trans, "CSR values:\n");
	IWL_ERR(trans,
		"2nd byte of CSR_INT_COALESCING is CSR_INT_PERIODIC_REG\n");
	for (i = 0; i <  ARRAY_SIZE(csr_tbl); i++) {
		IWL_ERR(trans, "  %25s: 0X%08x\n",
			get_csr_string(csr_tbl[i]),
			iwl_trans_idi_read32(trans, csr_tbl[i]));
	}
}

static int iwl_idi_dump_fh(struct iwl_trans *trans, char **buf, bool display)
{
	int i;

	static const u32 fh_tbl[] = {
		FH_RSCSR_CHNL0_STTS_WPTR_REG,
		FH_RSCSR_CHNL0_RBDCB_BASE_REG,
		FH_RSCSR_CHNL0_WPTR,
		FH_MEM_RCSR_CHNL0_CONFIG_REG,
		FH_MEM_RSSR_SHARED_CTRL_REG,
		FH_MEM_RSSR_RX_STATUS_REG,
		FH_MEM_RSSR_RX_ENABLE_ERR_IRQ2DRV,
		FH_TSSR_TX_STATUS_REG,
		FH_TSSR_TX_ERROR_REG
	};

	IWL_ERR(trans, "FH register values:\n");
	for (i = 0; i <  ARRAY_SIZE(fh_tbl); i++) {
		IWL_ERR(trans, "  %34s: 0X%08x\n",
			get_fh_string(fh_tbl[i]),

			iwl_read_direct32(trans, fh_tbl[i]));
	}
	return 0;
}

static inline void atomic_and(int i, atomic_t *v)
{
	int old;
	int new;

	do {
		old = atomic_read(v);
		new = old & i;
	} while (atomic_cmpxchg(v, old, new) != old);
}

static inline void iwl_idi_disable_interrupts(struct iwl_trans *trans)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	idi_clear_status_bit(STATUS_INT_ENABLED, trans_slv);
	/* disable interrupts from uCode/NIC to host */
#endif
	iwl_write32(trans, CSR_INT_MASK, 0x00000000);

	/* acknowledge/clear/reset any interrupts still pending
	 * from uCode or flow handler (Rx/Tx DMA) */
	iwl_write32(trans, CSR_INT, 0xffffffff);
	iwl_write32(trans, CSR_FH_INT_STATUS, 0xffffffff);
	IWL_DEBUG_ISR(trans, "Disabled interrupts\n");
}

/**
 * fit to idi
 * iwl_pcie_irq_handle_error - called for HW or SW error interrupt from card
 */
static void iwl_idi_irq_handle_error(struct iwl_trans *trans)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	/* W/A for WiFi/WiMAX coex and WiMAX own the RF */
	if (trans->cfg->internal_wimax_coex &&
	    (!(iwl_read_prph(trans, APMG_CLK_CTRL_REG) &
		     APMS_CLK_VAL_MRB_FUNC_MODE) ||
		     (iwl_read_prph(trans, APMG_PS_CTRL_REG) &
		     APMG_PS_CTRL_VAL_RESET_REQ))) {

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
		idi_clear_status_bit(STATUS_HCMD_ACTIVE, trans_slv);
#endif
		iwl_op_mode_wimax_active(trans->op_mode);
		wake_up(&trans_slv->wait_command_queue);
		return;
	}
	iwl_idi_dump_csr(trans);
	iwl_idi_dump_fh(trans, NULL, false);

	local_bh_disable();
	iwl_op_mode_nic_error(trans->op_mode);
	local_bh_enable();
}

/* reads the pending idi dependent irqs from the emulation.
 * and adds to the iwl_tran_idi itself.
 * after the emulation will be dropped - will be replaced with target
 * access to the actual hw register showing the irqs
 */
static inline void iwl_idi_read_irq_register(struct iwl_trans *trans)
{
	u32 called = iwl_inta_em_read_called(trans);
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	atomic_or(called, &trans_idi->irq_data);
}

/* clears the pending idi dependent irqs from the emulation.
 * after the emulation will be dropped - will be replaced with target
 *access to the actual hw register showing the irqs
 */
static inline void iwl_idi_clear_irq_register(struct iwl_trans *trans)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	iwl_inta_em_clean_irq_called(trans);
#else
	/* TODO */
#endif
}

/**
 *called when there are pending idi dependent interrupts
 */
static irqreturn_t __maybe_unused idi_irq_handler(unsigned long data)
{
	struct iwl_trans *trans = (struct iwl_trans *)data;
	iwl_idi_read_irq_register(trans);
	iwl_idi_clear_irq_register(trans);
	return IRQ_WAKE_THREAD;
}

/**
 * The handler for idi irq's
 *Does not depend on pcie. for now treats only hw and sw errors,
 *and set a place to add lls interrupt handling.
 */
static irqreturn_t __maybe_unused idi_irq_thread(unsigned long data)
{
	struct iwl_trans *trans = (struct iwl_trans *)data;
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	u32 inta = atomic_read(&trans_idi->irq_data);
	u32 handled = 0;

	/* TODO - HANDLE LLS INTERRUPT! */
	if (inta & IWL_IDI_LLS_INT)
		handled |= IWL_IDI_LLS_INT;

	/* Now service all interrupt bits discovered above. */
	if (inta & IWL_IDI_HW_ERR_INT) {
		IWL_ERR(trans, "Hardware error detected.  Restarting.\n");
		/* Tell the device to stop sending interrupts */
		iwl_idi_disable_interrupts(trans);
		iwl_idi_irq_handle_error(trans);
		handled |= IWL_IDI_HW_ERR_INT;
		return IRQ_HANDLED;
	}

	/* Error detected by uCode */
	if (inta & IWL_IDI_SW_ERR_INT) {
		IWL_ERR(trans,
			"Microcode SW error detected. Restarting 0x%X.\n",
			inta);
		iwl_idi_irq_handle_error(trans);
		handled |= IWL_IDI_SW_ERR_INT;
	}

	if (inta & IWL_IDI_FH_TX_INT) {
		IWL_DEBUG_FW(trans, "IDI irq: uCode load interrupt\n");
		/* Wake up uCode load routine, now that load is complete */
		trans_idi->trans_tx.ucode_write_complete = true;
		wake_up(&trans_idi->trans_tx.ucode_write_waitq);
		handled |= IWL_IDI_FH_TX_INT;
	}

	/* clear handled irqs from irq_data */
	atomic_and(~handled, &trans_idi->irq_data);

	if (inta & ~handled)
		IWL_WARN(trans, "IRQ not handled %d\n", inta & ~handled);

	return IRQ_HANDLED;
}

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI

irqreturn_t iwl_idi_isr(int irq, void *data)
{
	struct iwl_trans *trans = data;

	IWL_DEBUG_ISR(trans, "received interrupt\n");

	/* check WLAN_IRQ_O */
	/* mask int by write to WLAN_IRQ_MASK */
	/* get group from HIDI_APMG_DRV_IRQ_STATUS_IDI_CLK */

	/* clear int by */
	/* clear corresponding bit in the internal register (level 3) */
	/* mask WLAN_IRQ_O by write to WLAN_IRQ_O_MASK */
	/* clear WLAN_IRQ_O */
	/* unmask WLAN_IRQ_O */
	return IRQ_HANDLED;
}

static int iwl_idi_request_irqs(struct idi_peripheral_device *pdev)
{
	struct iwl_trans *trans = dev_get_drvdata(&pdev->device);
	struct idi_resource *idi_res = &pdev->resources;
	struct resource *res;
	int ret;

	res = idi_get_resource_byname(idi_res, IORESOURCE_IRQ,
				      IWL_IDI_IRQ_NAME);
	if (!res) {
		IWL_ERR(trans, "Failed to get resource %s\n", IWL_IDI_IRQ_NAME);
		return -EINVAL;
	}

	ret = request_threaded_irq(res->start, iwl_idi_isr, NULL,
				   IRQF_SHARED, IWL_IDI_IRQ_NAME, trans);
	if (ret) {
		IWL_ERR(trans, "Failed in request_threaded_irq\n");
		return -EINVAL;
	}

	return 0;
}

static void __iomem *
iwl_idi_request_io_byname(struct idi_peripheral_device *pdev,
			  const char *name, bool request)
{
	void __iomem *io;
	struct resource *res;
	struct idi_resource *idi_res = &pdev->resources;

	if (!name)
		return NULL;

	res = idi_get_resource_byname(idi_res, IORESOURCE_MEM, name);
	if (request && (!request_mem_region(res->start, resource_size(res),
					    dev_name(&pdev->device))))
		return NULL;

	io = ioremap(res->start, resource_size(res));
	if (!io && request)
		release_mem_region(res->start, resource_size(res));

	return io;
}

static void iwl_idi_release_io_byname(struct idi_peripheral_device *pdev,
				      const char *name, void __iomem *io,
				      bool release)
{
	struct resource *res;
	struct idi_resource *idi_res = &pdev->resources;

	if (!name)
		return;

	res = idi_get_resource_byname(idi_res, IORESOURCE_MEM, name);

	if (!res)
		return;

	iounmap(io);
	if (release)
		release_mem_region(res->start, resource_size(res));
}
#define WIFI_CLK_KERNEL_NAME "wlan_clk_req"
#define WIFI_CLK_RTC_NAME "rtc_clk_req"
static struct iwl_idi_platdata *
iwl_idi_get_platdata(struct idi_peripheral_device *pdev)
{
	struct device_node *np = pdev->device.of_node;
	struct iwl_idi_platdata *platdata;
	struct device *dev = &pdev->device;

	platdata = kzalloc(sizeof(*platdata), GFP_KERNEL);
	if (!platdata)
		return NULL;

	platdata->wifi_clk = of_clk_get_by_name(np,WIFI_CLK_KERNEL_NAME);
	if(IS_ERR(platdata->wifi_clk)){
		dev_warn(dev,"No Wifi clock available \n");
		platdata->wifi_clk = NULL;
	}

	platdata->rtc_clk = of_clk_get_by_name(np,WIFI_CLK_RTC_NAME);
	if(IS_ERR(platdata->rtc_clk)){
		dev_warn(dev,"No RTC clock available \n");
		platdata->rtc_clk = NULL;
	}

	platdata->pmu_rst = reset_control_get(dev,"pmu");
	platdata->core_rst = reset_control_get(dev,"core");
	/* FIXME: retrieve platform data according to device tree config */

	return platdata;
}

/* FixME: This will declared in idi bus interface */
enum idi_device_state {
	IDI_D0 = 0,
	IDI_D0i2 = 1,
	IDI_D0i3 = 2,
	IDI_D3 = 3,
};

static void iwl_idi_set_clocks_work_around(void)
{
	u32 val;

	#define SPCURCCONF3_RCOUT_EN 0xe6402044
	#define SPCURCCONF4_RCOUT_EN 0xe6402048
	#define SPCURCCONF5_RCOUT_EN 0xe640205c
	#define ABB_RTC_CTRL 0xe6200010
	#define SPCUMEMPOWER_TRACE_MEM_MODE 0xe640201c

	val = ioread32((void __iomem *)SPCURCCONF3_RCOUT_EN);
	iowrite32(val | 0x2, (void __iomem *)SPCURCCONF3_RCOUT_EN);

	val = ioread32((void __iomem *)SPCURCCONF4_RCOUT_EN);
	iowrite32(val | 0x2, (void __iomem *)SPCURCCONF4_RCOUT_EN);

	val = ioread32((void __iomem *)SPCURCCONF5_RCOUT_EN);
	iowrite32(val | 0x2, (void __iomem *)SPCURCCONF5_RCOUT_EN);

	iowrite32(0x00006040, (void __iomem *)ABB_RTC_CTRL);

	val = ioread32((void __iomem *)SPCUMEMPOWER_TRACE_MEM_MODE);
	iowrite32(val | 0x1, (void __iomem *)SPCUMEMPOWER_TRACE_MEM_MODE);
}

#define SPCU_WLAN_POWER 0xE6402024
#define SPCU_WLAN_MISC_OUTPUT 0xE6402050

static int iwl_idi_set_power_state(struct idi_peripheral_device *pdev,
				   enum idi_device_state state)
{
	struct device *dev = &pdev->device;
	struct iwl_idi_platdata *platdata = dev_get_platdata(dev);
	int ret = 0;

	switch(state) {
	case IDI_D0:
		if(platdata->wifi_clk){
			clk_prepare(platdata->wifi_clk);
			clk_enable(platdata->wifi_clk);
		}
		if(platdata->rtc_clk){
			clk_prepare(platdata->rtc_clk);
			clk_enable(platdata->rtc_clk);
		}

		iwl_idi_set_clocks_work_around();
		/* TODO: Power up the Power domain */
		/* FIXME: Use the regulator framework */

		iowrite32(0x1, (void __iomem *)SPCU_WLAN_POWER);

		if(platdata->core_rst)
			reset_control_reset(platdata->core_rst);

		udelay(500);

		break;
	case IDI_D3:
		if(platdata->wifi_clk){
			clk_disable(platdata->wifi_clk);
			clk_unprepare(platdata->wifi_clk);
		}

		iowrite32(0, (void __iomem *)SPCU_WLAN_POWER);
	break;
	default:
	break;
	};

	return ret;
}
#endif /* CPTCFG_IWLWIFI_IDI_OVER_PCI */

static struct iwl_trans *iwl_trans_idi_alloc(void *pdev_void,
					     const void *ent_void,
					     const struct iwl_cfg *cfg)
{
	struct iwl_trans *iwl_trans = NULL;
	struct iwl_trans_idi *trans_idi = NULL;

	int ret = 0;
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_al_config *iwl_al_cfg = NULL;
	struct pci_dev __maybe_unused *pdev = (struct pci_dev *)pdev_void;
	const struct pci_device_id __maybe_unused *ent =
		(const struct pci_device_id *)ent_void;
#else
	struct idi_peripheral_device __maybe_unused *pdev =
				(struct idi_peripheral_device *)pdev_void;
#endif

	/* Allocate IDI transport */
	iwl_trans = kzalloc(sizeof(struct iwl_trans) +
			sizeof(struct iwl_trans_slv) +
			sizeof(struct iwl_trans_idi), GFP_KERNEL);
	if (iwl_trans == NULL)
		return NULL;

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	iwl_trans->dev = &pdev->dev;
#else
	dev_set_drvdata(&pdev->device, iwl_trans);
	iwl_trans->dev = &pdev->device;
#endif
	trans_idi = IWL_TRANS_GET_IDI_TRANS(iwl_trans);
	iwl_trans->ops = &trans_ops_idi;
	iwl_trans->cfg = cfg;
	trans_idi->trans = iwl_trans;
	trans_idi->trans_tx.trans_idi = trans_idi;
	trans_idi->trans_rx.trans_idi = trans_idi;

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	trans_idi->pdev = pdev;
#endif

	trans_lockdep_init(iwl_trans);

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	/* allocate iwl_al_config and include idi irq handler */
	iwl_al_cfg = kzalloc(sizeof(struct iwl_al_config) , GFP_KERNEL);
	if (iwl_al_cfg == NULL)
		goto error_cfg;

	iwl_al_cfg->iwl_al_irq_handler = &idi_irq_handler;
	iwl_al_cfg->iwl_al_irq_thread = &idi_irq_thread;
	atomic_set(&trans_idi->irq_data, 0);

	/* Allocate PCIe transport */
	trans_idi->trans_pcie = iwl_trans_pcie_alloc(pdev, ent, cfg);
	if (trans_idi->trans_pcie == NULL)
		goto error;

	memcpy(iwl_trans->hw_id_str, trans_idi->trans_pcie->hw_id_str,
	       sizeof(iwl_trans->hw_id_str));
	iwl_trans->hw_id = trans_idi->trans_pcie->hw_id;
	iwl_trans->hw_rev = trans_idi->trans_pcie->hw_rev;
#endif

	snprintf(iwl_trans->dev_cmd_pool_name,
		 sizeof(iwl_trans->dev_cmd_pool_name),
		 "iwl_cmd_pool_IDI:%s", dev_name(iwl_trans->dev));

	/* TODO: add a few bytes for the prefix needed in IDI */
	iwl_trans->dev_cmd_headroom = IWL_IDI_TXBU_HEADROOM_SIZE;
	iwl_trans->dev_cmd_pool =
		kmem_cache_create(iwl_trans->dev_cmd_pool_name,
				  sizeof(struct iwl_device_cmd)
				  + iwl_trans->dev_cmd_headroom,
				  sizeof(void *),
				  SLAB_HWCACHE_ALIGN,
				  NULL);

	if (!iwl_trans->dev_cmd_pool)
		goto error;
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	/* Init the Emulation */
	ret = iwl_al_init(iwl_al_cfg, iwl_trans);
	if (ret)
		goto error;

	kfree(iwl_al_cfg);
#else
	spin_lock_init(&trans_idi->reg_lock);

#ifdef CONFIG_OF /* working with device tree */
	pdev->device.platform_data = iwl_idi_get_platdata(pdev);
	if (!pdev->device.platform_data)
		goto error;
#else
	if (!dev_get_platdata(&pdev->device))
		goto error;
#endif

	/* io mapping of the AL memory */
	trans_idi->al_hw_base =
		iwl_idi_request_io_byname(pdev, IWL_IDI_AL_MEM, false);
	if (!trans_idi->al_hw_base)
		goto error;

	ret = iwl_idi_request_irqs(pdev);
	if (ret)
		goto error_irq;

#endif
	return iwl_trans;

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
error_irq:
	iwl_idi_release_io_byname(pdev, IWL_IDI_AL_MEM,
				  trans_idi->al_hw_base, true);
#endif

error:
	if (iwl_trans->dev_cmd_pool)
		kmem_cache_destroy(iwl_trans->dev_cmd_pool);
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	iwl_al_free();
	kfree(iwl_al_cfg);
#endif

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
error_cfg:
#endif
	kfree(iwl_trans);
	return NULL;
}

static int iwl_trans_idi_start_hw(struct iwl_trans *trans)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	trans_idi->trans_pcie->op_mode = trans->op_mode;
	return trans_idi->trans_pcie->ops->start_hw(trans_idi->trans_pcie);
#else
	iwl_idi_set_power_state(trans_idi->pdev, IDI_D0);
	idi_al_init(trans);

#endif
	set_bit(STATUS_DEVICE_ENABLED, &trans_slv->status);
	return 0;
}

static void iwl_trans_idi_stop_hw(struct iwl_trans *trans,
				  bool op_mode_leaving)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	trans_idi->trans_pcie->ops->stop_hw(trans_idi->trans_pcie,
					    op_mode_leaving);
#else
	iwl_idi_set_power_state(trans_idi->pdev, IDI_D3);
#endif
	clear_bit(STATUS_DEVICE_ENABLED, &trans_slv->status);
}

static int iwl_trans_idi_start_fw(struct iwl_trans *trans,
				const struct fw_img *fw,
				bool run_in_rfkill)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	int ret;

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);

	ret = trans_idi->trans_pcie->ops->start_fw(trans_idi->trans_pcie, fw,
						   run_in_rfkill);
	if (ret) {
		IWL_ERR(trans, "PCIe transport start failed.\n");
		return ret;
	}
	iwl_al_start();

#endif
	ret = iwl_idi_rx_init(trans);
	if (ret) {
		IWL_ERR(trans, "Unable to init nic - rx\n");
		return ret;
	}

	ret = iwl_idi_tx_init(trans);
	if (ret) {
		IWL_ERR(trans, "Unable to init nic - tx\n");
		return ret;
	}

	ret = iwl_idi_load_given_ucode(trans, fw);
	if (ret) {
		IWL_ERR(trans, "Unable to init nic - fw\n");
		return ret;
	}

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	/* TODO: WkP bug, must read the first address to wake up the NIC,
	   should be fixed in the second tape-out */
	if (trans->cfg->device_family == IWL_DEVICE_FAMILY_7000) {
		iwl_write32(trans, HBUS_TARG_MEM_RADDR, 0);
		iwl_read32(trans, HBUS_TARG_MEM_RDAT);
	}

	/* After the emulation was initialized - start the uCode */
	iwl_write32(trans, CSR_RESET, 0);
#endif
	set_bit(STATUS_DEVICE_ENABLED, &trans_slv->status);

	return ret;
}

static void iwl_trans_idi_stop_device(struct iwl_trans *trans)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
#endif

	if (test_bit(STATUS_DEVICE_ENABLED, &trans_slv->status)) {
		iwl_idi_rx_stop(trans);
		iwl_idi_tx_stop(trans);
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
		iwl_al_stop();
		trans_idi->trans_pcie->ops->stop_device(trans_idi->trans_pcie);
#endif
		iwl_idi_rx_free(trans);
		iwl_idi_tx_free(trans);
	}

	clear_bit(STATUS_HCMD_ACTIVE, &trans_slv->status);
	clear_bit(STATUS_DEVICE_ENABLED, &trans_slv->status);
	clear_bit(STATUS_TPOWER_PMI, &trans_slv->status);
}

static void iwl_trans_idi_d3_suspend(struct iwl_trans *trans, bool test)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	trans_idi->trans_pcie->ops->d3_suspend(trans_idi->trans_pcie, test);
#endif
}

static int iwl_trans_idi_d3_resume(struct iwl_trans *trans,
				   enum iwl_d3_status *status,
				   bool test)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->d3_resume(trans_idi->trans_pcie,
						     status, test);
#else
	*status = IWL_D3_STATUS_RESET;
	return 0;
#endif
}

static inline void iwl_idi_txq_set_inactive(struct iwl_trans *trans,
					    u16 txq_id)
{
	/* Simply stop the queue, but don't change any configuration;
	 * the SCD_ACT_EN bit is the write-enable mask for the ACTIVE bit. */
	iwl_write_prph(trans,
		       SCD_QUEUE_STATUS_BITS(txq_id),
		       (0 << SCD_QUEUE_STTS_REG_POS_ACTIVE) |
		       (1 << SCD_QUEUE_STTS_REG_POS_SCD_ACT_EN));
}

static void iwl_trans_idi_txq_enable(struct iwl_trans *trans,
				     int txq_id, int fifo, int sta_id, int tid,
				     int frame_limit, u16 ssn)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);


#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	/* Set this queue as a chain-building queue unless it is CMD queue */
	if (txq_id != trans_slv->cmd_queue)
		iwl_set_bits_prph(trans, SCD_QUEUECHAIN_SEL, BIT(txq_id));

	iwl_sfdb_set_queue_context_data(txq_id, trans_tx->scd_base_addr);
	iwl_sfdb_set_queue_write_ptr(txq_id, ssn);
	iwl_sfdb_set_queue_read_ptr(txq_id, ssn);
	iwl_sfdb_set_queue_status(txq_id, true, IWL_IDI_IS_Q_PAN(txq_id),
				  fifo);
#else
	/* Stop this Tx queue before configuring it */
	iwl_idi_txq_set_inactive(trans, txq_id);

	/* Set this queue as a chain-building queue unless it is CMD queue */
	if (txq_id != trans_slv->cmd_queue)
		iwl_set_bits_prph(trans, SCD_QUEUECHAIN_SEL, BIT(txq_id));

	/* If this queue is mapped to a certain station: it is an AGG queue */
	if (sta_id >= 0) {
		/* FIXME: need to implement for AGG queues */
	} else {
		/* FIXME: disable aggregations */
	}

	/* Configure the shadow register of the SCD write pointer with the
	 * first TFD at index corresponding to start sequence number. */
	idi_al_write(trans, AMFH_TG1_WR_BASE_ADDR + HBUS_TARG_WRPTR,
		     (ssn & 0xff) | (txq_id << 8));

	iwl_write_prph(trans, SCD_QUEUE_RDPTR(txq_id), ssn);

	/* tx window and frame sizes for this queue */
	iwl_trans_write_mem32(trans, trans_tx->scd_base_addr +
			      SCD_CONTEXT_QUEUE_OFFSET(txq_id), 0);
	iwl_trans_write_mem32(trans, trans_tx->scd_base_addr +
			      SCD_CONTEXT_QUEUE_OFFSET(txq_id) + sizeof(u32),
			((frame_limit << SCD_QUEUE_CTX_REG2_WIN_SIZE_POS) &
				SCD_QUEUE_CTX_REG2_WIN_SIZE_MSK) |
			((frame_limit << SCD_QUEUE_CTX_REG2_FRAME_LIMIT_POS) &
				SCD_QUEUE_CTX_REG2_FRAME_LIMIT_MSK));

	/* status area: activate queue, map to fifo */
	iwl_write_prph(trans, SCD_QUEUE_STATUS_BITS(txq_id),
		       (1 << SCD_QUEUE_STTS_REG_POS_ACTIVE) |
		       (fifo << SCD_QUEUE_STTS_REG_POS_TXF) |
		       (1 << SCD_QUEUE_STTS_REG_POS_WSL) |
		       SCD_QUEUE_STTS_REG_MSK);

	IWL_DEBUG_TX_QUEUES(trans, "Activate queue %d on FIFO %d wr ptr: %d\n",
			    txq_id, fifo, ssn & 0xff);

#endif
}

static void iwl_trans_idi_txq_disable(struct iwl_trans *trans, int txq_id)
{
#ifdef CPTCFGG_IWLWIFI_IDI_OVER_PCI
	iwl_sfdb_set_queue_write_ptr(txq_id, 0);
	iwl_sfdb_set_queue_read_ptr(txq_id, 0);
	iwl_sfdb_set_queue_status(txq_id, false, IWL_IDI_IS_Q_PAN(txq_id), 0);
#else
	struct iwl_idi_trans_tx *trans_tx;
	u32 stts_addr;
	static const u32 zero_data[4] = {};

	trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);

	iwl_idi_txq_set_inactive(trans, txq_id);

	stts_addr = trans_tx->scd_base_addr +
		SCD_TX_STTS_QUEUE_OFFSET(txq_id);
	iwl_trans_write_mem(trans, stts_addr, (void *)zero_data,
			    ARRAY_SIZE(zero_data));

	/* FIXME: any other configs? */
#endif
	iwl_slv_free_data_queue(trans, txq_id);

	IWL_DEBUG_TX_QUEUES(trans, "Deactivate queue %d\n", txq_id);
}

static void iwl_idi_tx_start(struct iwl_trans *trans, u32 scd_base_addr)
{
	u8 i, chnl;
	u32 reg_val;
#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	__le32 q_addr;
#endif
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	for (i = 0; i < IDI_TX_MAX_Q_COUNT; i++)
		iwl_sfdb_init_tfd_queue(i);
#else
	q_addr = IDI_AL_SFDB_VTFD_BASE_ADDR;
	for (i = 0; i < IDI_TX_MAX_Q_COUNT; i++) {
		iwl_write_direct32(trans,
			   FH_MEM_CBBC_QUEUE(i),
			   q_addr >> 8);
		q_addr += IDI_AL_SFDB_VTFD_SIZE;
	}
#endif

	trans_tx->scd_base_addr =
		iwl_read_prph(trans, SCD_SRAM_BASE_ADDR);

	WARN_ON(scd_base_addr != 0 &&
		scd_base_addr != trans_tx->scd_base_addr);

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	iwl_write_prph(trans, SCD_DRAM_BASE_ADDR,
		       MEMORY_MAP_BC_BASE_ADDRESS >> 10);
#else
	iwl_write_prph(trans, SCD_DRAM_BASE_ADDR,
		       IDI_AL_SFDB_VIRT_BC_BASE_ADDR >> 10);
#endif
	/* Set entries count in queue to 32 TFDs */
	iwl_write_prph(trans, SCD_CB_SIZE, 6);

	for (chnl = 0; chnl < FH_TCSR_CHNL_NUM; chnl++)
		iwl_write_direct32(trans, FH_TCSR_CHNL_TX_CONFIG_REG(chnl),
				   FH_TCSR_TX_CONFIG_REG_VAL_MSG_MODE_TXF |
				   FH_TCSR_TX_CONFIG_REG_VAL_DMA_CREDIT_ENABLE |
				   FH_TCSR_TX_CONFIG_REG_VAL_CIRQ_HOST_NOINT |
				   FH_TCSR_TX_CONFIG_REG_VAL_CIRQ_RTC_NOINT |
				   FH_TCSR_TX_CONFIG_REG_VAL_DMA_CHNL_ENABLE);

	/* Update FH chicken bits */
	reg_val = iwl_read_direct32(trans, FH_TX_CHICKEN_BITS_REG);
	iwl_write_direct32(trans, FH_TX_CHICKEN_BITS_REG,
			   reg_val | FH_TX_CHICKEN_BITS_SCD_AUTO_RETRY_EN);

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	/* configure TX max pending requests to one, to prevent
	 * fast response bug.
	 * This value is the default value except of max pending req */

	/* FIXME: check this configuration: 0x1 taken from AV scripts;
	 * maybe should be 0x7FF8001 as explained above */
	iwl_write_direct32(trans, FH_TSSR_TX_MSG_CONFIG_REG, 0x1);
#endif

	/* write zero to SCD_AGGREGATION_EN */
	iwl_write_prph(trans, SCD_AGGR_SEL, 0);

	/* enable all transmit queue interrupts */
	iwl_write_prph(trans, SCD_INTERRUPT_MASK,
		       IWL_MASK(0, IDI_TX_MAX_Q_COUNT));

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	iwl_write_prph(trans, SCD_ACTIVE, 0x1);
	reg_val = iwl_read_prph(trans, SCD_ACTIVE);
#endif

	/* enable all TX FIFOs */
	iwl_write_prph(trans, SCD_TXFACT, IWL_MASK(0, 7));

	iwl_trans_ac_txq_enable(trans, trans_slv->cmd_queue,
				trans_slv->cmd_fifo);
	IWL_DEBUG_TX(trans, "Tx Started\n");
}

static void iwl_trans_idi_fw_alive(struct iwl_trans *trans, u32 scd_base_addr)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	iwl_pcie_reset_ict(trans_idi->trans_pcie);
#endif
	iwl_idi_tx_start(trans, scd_base_addr);
}

static void iwl_trans_idi_free(struct iwl_trans *trans)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	iwl_al_stop();
	iwl_al_free();
	iwl_trans_pcie_free(trans_idi->trans_pcie);
#endif
	kmem_cache_destroy(trans->dev_cmd_pool);
	kfree(trans);
}

static int iwl_trans_idi_wait_tx_queue_empty(struct iwl_trans *trans)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->
			wait_tx_queue_empty(trans_idi->trans_pcie);
#else
	return 0;
#endif
}

#ifdef CONFIG_PM

static int iwl_trans_idi_suspend(struct iwl_trans *trans)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->suspend(trans_idi->trans_pcie);
#else
	return 0;
#endif
}

static int iwl_trans_idi_resume(struct iwl_trans *trans)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->resume(trans_idi->trans_pcie);
#else
	return 0;
#endif

}
#else /* CONFIG_PM */
static int iwl_trans_idi_suspend(struct iwl_trans *trans)
{ return 0; }

static int iwl_trans_idi_resume(struct iwl_trans *trans)
{ return 0; }

#endif /* CONFIG_PM */

static void iwl_trans_idi_write8(struct iwl_trans *trans, u32 ofs, u8 val)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	trans_idi->trans_pcie->ops->write8(trans_idi->trans_pcie, ofs, val);
#else
	if (IS_AL_ADDR(ofs)) { /* AL address */
		idi_al_write8(trans, (ofs & ~ADDR_IN_AL_MSK), val);
	} else { /* LMAC address */
		/* TODO: write to LMAC */
	}
#endif
}

static void iwl_trans_idi_write32(struct iwl_trans *trans, u32 ofs, u32 val)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	trans_idi->trans_pcie->ops->write32(trans_idi->trans_pcie, ofs, val);
#else
	/* Usually this fucntion is used to access LMAC registers.
	 * Added support to AL registers; differentiation using MS bit.
	 * This is temporal solution to enable SV tools */
	if (IS_AL_ADDR(ofs)) { /* AL address */
		idi_al_write(trans, (ofs & ~ADDR_IN_AL_MSK), val);
	} else { /* LMAC address */
		idi_al_write_lmac(trans, ofs, val);
	}
#endif
}

static u32 iwl_trans_idi_read32(struct iwl_trans *trans, u32 ofs)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->read32(trans_idi->trans_pcie, ofs);
#else
	/* Usually this fucntion is used to access LMAC registers.
	 * Added support to AL registers; differentiation using MS bit.
	 * This is temporal solution to enable SV tools */
	if (IS_AL_ADDR(ofs)) { /* AL address */
		return idi_al_read(trans, (ofs & ~ADDR_IN_AL_MSK));
	} else { /* LMAC address */
		return idi_al_read_lmac(trans, ofs);
	}
#endif
}

static u32 iwl_trans_idi_read_prph(struct iwl_trans *trans, u32 ofs)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->read_prph(trans_idi->trans_pcie,
						     ofs);
#else
	return idi_al_read_lmac_prph(trans, ofs);
#endif
}

static void iwl_trans_idi_write_prph(struct iwl_trans *trans, u32 ofs, u32 val)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	trans_idi->trans_pcie->ops->write_prph(trans_idi->trans_pcie, ofs, val);
#else
	idi_al_write_lmac_prph(trans, ofs, val);
#endif
}

static void iwl_trans_idi_configure(struct iwl_trans *trans,
			      const struct iwl_trans_config *trans_cfg)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	iwl_slv_set_reclaim_cmds(trans_slv, trans_cfg);
	iwl_slv_set_rx_page_order(trans_slv, trans_cfg);

	trans_slv->cmd_queue = trans_cfg->cmd_queue;
	trans_slv->cmd_fifo = trans_cfg->cmd_fifo;
	trans_slv->command_names = trans_cfg->command_names;

	trans_slv->config.max_queues_num = IDI_TX_MAX_Q_COUNT;
	trans_slv->config.tb_size = IDI_TX_PAYLOAD_PAGE_SIZE;
	trans_slv->config.max_desc_count = IWL_IDI_MAX_TXC_COUNT_IN_TXBU;
	trans_slv->config.tfds_num = IDI_TX_TFD_PER_Q;
	trans_slv->config.hcmd_headroom = IWL_IDI_TXBU_HEADROOM_SIZE;
	trans_slv->config.policy_trigger = iwl_idi_tx_policy_trigger;
	trans_slv->config.clean_dtu = iwl_idi_tx_clean_txbu;
	trans_slv->config.free_dtu_mem = iwl_idi_tx_free_txbu_mem;
}

static void iwl_trans_idi_set_pmi(struct iwl_trans *trans, bool state)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);

	trans_idi->trans_pcie->ops->set_pmi(trans_idi->trans_pcie, state);
#else
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	if (state)
		set_bit(STATUS_TPOWER_PMI, &trans_slv->status);
	else
		clear_bit(STATUS_TPOWER_PMI, &trans_slv->status);
#endif
}

static bool
iwl_trans_idi_grab_nic_access(struct iwl_trans *trans, bool silent,
			      unsigned long *flags)
{
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	return trans_idi->trans_pcie->ops->grab_nic_access(
						trans_idi->trans_pcie,
						silent, flags);
#else
	int ret;
	spin_lock_irqsave(&trans_idi->reg_lock, *flags);
	ret = idi_al_request_access(trans, silent);

	if (unlikely(ret < 0)) {
		/* TODO: reset NIC (?) */
		if (!silent) {
			spin_unlock_irqrestore(&trans_idi->reg_lock, *flags);
			return false;
		}
	}

	/*
	 * Fool sparse by faking we release the lock - sparse will
	 * track nic_access anyway.
	 */
	__release(&trans_idi->reg_lock);
	return true;
#endif
}

static void
iwl_trans_idi_release_nic_access(struct iwl_trans *trans, unsigned long *flags)
{
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	trans_idi->trans_pcie->ops->release_nic_access(trans_idi->trans_pcie,
						       flags);
#else
	lockdep_assert_held(&trans_idi->reg_lock);

	/*
	 * Fool sparse by faking we acquiring the lock - sparse will
	 * track nic_access anyway.
	 */
	__acquire(&trans_idi->reg_lock);

	idi_al_release_access(trans);

	/*
	 * we need this write to be performed before any other writes
	 * scheduled on different CPUs (after we drop reg_lock).
	 */
	mmiowb();
	spin_unlock_irqrestore(&trans_idi->reg_lock, *flags);
#endif
}

static int iwl_trans_idi_read_mem(struct iwl_trans *trans, u32 addr, void *buf,
				  int dwords)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->read_mem(trans_idi->trans_pcie,
						    addr, buf, dwords);
#else
	unsigned long flags;
	int ret;
	if (iwl_trans_grab_nic_access(trans, false, &flags)) {
		ret = idi_al_read_lmac_mem(trans, addr, buf, dwords);
		iwl_trans_release_nic_access(trans, &flags);
	} else {
		ret = -EBUSY;
	}

	return ret;
#endif
}

static int iwl_trans_idi_write_mem(struct iwl_trans *trans, u32 addr,
				   const void *buf, int dwords)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->write_mem(trans_idi->trans_pcie,
						     addr, buf, dwords);
#else
	unsigned long flags;
	int ret = 0;

	if (iwl_trans_grab_nic_access(trans, false, &flags)) {
		idi_al_write_lmac_mem(trans, addr, buf, dwords);
		iwl_trans_release_nic_access(trans, &flags);
	} else {
		ret = -EBUSY;
	}
	return ret;
#endif
}

static void iwl_trans_idi_set_bits_mask(struct iwl_trans *trans,
					u32 reg, u32 mask, u32 value)
{
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	trans_idi->trans_pcie->ops->set_bits_mask(trans_idi->trans_pcie,
						  reg, mask, value);
#else
	unsigned long flags;
	u32 v;

	spin_lock_irqsave(&trans_idi->reg_lock, flags);

#ifdef CONFIG_IWLWIFI_DEBUG
	WARN_ON_ONCE(value & ~mask);
#endif
	v = iwl_read32(trans, reg);
	v &= ~mask;
	v |= value;
	iwl_write32(trans, reg, v);

	spin_unlock_irqrestore(&trans_idi->reg_lock, flags);
#endif
}

#ifdef CPTCFG_IWLWIFI_DEBUGFS
static int iwl_trans_idi_dbgfs_register(struct iwl_trans *trans,
					struct dentry *dir)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
	return trans_idi->trans_pcie->ops->
			dbgfs_register(trans_idi->trans_pcie, dir);
#else
	return 0;
#endif
}

#else
static int iwl_trans_idi_dbgfs_register(struct iwl_trans *trans,
					struct dentry *dir)
{ return 0; }

#endif /*CPTCFG_IWLWIFI_DEBUGFS */

static const struct iwl_trans_ops trans_ops_idi = {
	.start_hw = iwl_trans_idi_start_hw,
	.stop_hw = iwl_trans_idi_stop_hw,
	.fw_alive = iwl_trans_idi_fw_alive,
	.start_fw = iwl_trans_idi_start_fw,
	.stop_device = iwl_trans_idi_stop_device,

	.d3_suspend = iwl_trans_idi_d3_suspend,
	.d3_resume = iwl_trans_idi_d3_resume,

	.send_cmd = iwl_trans_slv_send_cmd,

	.tx = iwl_trans_slv_tx_data_send,
	.reclaim = iwl_trans_slv_tx_data_reclaim,

	/* no aggregation on IDI at this stage */
	.txq_enable = iwl_trans_idi_txq_enable,
	.txq_disable = iwl_trans_idi_txq_disable,

	.dbgfs_register = iwl_trans_idi_dbgfs_register,
	.wait_tx_queue_empty = iwl_trans_idi_wait_tx_queue_empty,

#ifdef CONFIG_PM_SLEEP
	.suspend = iwl_trans_idi_suspend,
	.resume = iwl_trans_idi_resume,
#endif
	.write8 = iwl_trans_idi_write8,
	.write32 = iwl_trans_idi_write32,
	.read32 = iwl_trans_idi_read32,
	.read_prph = iwl_trans_idi_read_prph,
	.write_prph = iwl_trans_idi_write_prph,
	.read_mem = iwl_trans_idi_read_mem,
	.write_mem = iwl_trans_idi_write_mem,
	.configure = iwl_trans_idi_configure,
	.set_pmi = iwl_trans_idi_set_pmi,
	.grab_nic_access = iwl_trans_idi_grab_nic_access,
	.release_nic_access = iwl_trans_idi_release_nic_access,
	.set_bits_mask = iwl_trans_idi_set_bits_mask,
};

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI

#define IWL_IDI_NAME  "IWL_IDI"

static int iwl_idi_probe(struct idi_peripheral_device *pdev,
			 const struct idi_device_id *id)
{
	struct iwl_trans *iwl_trans;
	struct iwl_trans_idi *trans_idi;
	int ret;
	const struct iwl_cfg *cfg = &iwl7999_bgn_cfg;

	iwl_trans = iwl_trans_idi_alloc(pdev, id, cfg);
	if (iwl_trans == NULL)
		return -ENOMEM;

	IWL_INFO(iwl_trans, "Selected bus type = IDI\n");

	trans_idi = IWL_TRANS_GET_IDI_TRANS(iwl_trans);
	trans_idi->drv = iwl_drv_start(iwl_trans, cfg);

	if (IS_ERR_OR_NULL(trans_idi->drv)) {
		ret = PTR_ERR(trans_idi->drv);
		goto out_free_trans;
	}

	/* register transport layer debugfs here */
	ret = iwl_trans_dbgfs_register(iwl_trans, iwl_trans->dbgfs_dir);
	if (ret)
		goto out_free_drv;

	idi_al_init(iwl_trans);

	ret = iwl_idi_tx_set_channel_config(pdev);
	if (ret)
		IWL_ERR(iwl_trans, "Failed to set TX channel config, err %#x\n",
			ret);

	return 0;

out_free_drv:
	iwl_drv_stop(trans_idi->drv);
out_free_trans:
	iwl_trans_idi_free(iwl_trans);
	dev_set_drvdata(&pdev->device, NULL);
	return ret;
}

static int iwl_idi_remove(struct idi_peripheral_device *pdev)
{
	struct iwl_trans *trans = dev_get_drvdata(&pdev->device);
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);

	iwl_drv_stop(trans_idi->drv);
	iwl_idi_release_io_byname(pdev, IWL_IDI_AL_MEM,
				  trans_idi->al_hw_base, true);
	iwl_trans_idi_free(trans);
	dev_set_drvdata(&pdev->device, NULL);

	return 0;
}

static struct idi_peripheral_driver iwl_idi_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = IWL_IDI_NAME,
	},
	.p_type = IDI_WLAN,
	.probe  = iwl_idi_probe,
	.remove = iwl_idi_remove,
};

int iwl_idi_register_driver(void)
{
	int ret;

	ret = idi_register_peripheral_driver(&iwl_idi_driver);

	return ret;
}

void iwl_idi_unregister_driver(void)
{
	idi_unregister_peripheral_driver(&iwl_idi_driver);
}

#else

#include <linux/pci.h>

int iwl_idi_register_driver(void)
{ return 0; }

void iwl_idi_unregister_driver(void)
{}

int iwl_pci_probe_idi(struct pci_dev *pdev, const struct pci_device_id *ent)
{
	const struct iwl_cfg *cfg = (struct iwl_cfg *)(ent->driver_data);
	struct iwl_trans *iwl_trans;
	struct iwl_trans_idi *trans_idi;
	int err;

	iwl_trans = iwl_trans_idi_alloc(pdev, ent, cfg);
	if (iwl_trans == NULL)
		err = -ENOMEM;

	IWL_INFO(iwl_trans, "Selected bus type = IDI (emulation)\n");

	pci_set_drvdata(pdev, iwl_trans);

	trans_idi = IWL_TRANS_GET_IDI_TRANS(iwl_trans);
	trans_idi->drv = iwl_drv_start(iwl_trans, cfg);

	if (IS_ERR(trans_idi->drv)) {
		err = PTR_ERR(trans_idi->drv);
		goto out_free_trans;
	}

	return 0;

out_free_trans:
	iwl_trans_idi_free(iwl_trans);
	pci_set_drvdata(pdev, NULL);
	return err;
}

void iwl_pci_remove_idi(struct pci_dev *pdev)
{
	struct iwl_trans *trans = pci_get_drvdata(pdev);
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);

	iwl_drv_stop(trans_idi->drv);
	iwl_trans_idi_free(trans);

	pci_set_drvdata(pdev, NULL);
}

#endif
