/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgm
 * @short_description: A library to create rich application user interfaces.
 * @see_also: #PgmViewport, #PgmDrawable.
 *
 * <refsect2>
 * <para>
 * Pigment is a media rendering library written in C providing an abstraction
 * layer to easily design GUI (Graphical User Interfaces) like simple games,
 * media center/set-top box applications, etc. It has a pluggable backend
 * system so that the same application can run on various hardware and software
 * platforms using rendering libraries such as OpenGL or DirectFB on
 * operating systems such as Windows, GNU/Linux or Mac OS X.
 * </para>
 * <para>
 * Pigment has been designed to handle common issues like non square pixels
 * displays, non square pixels images or video sources, positioning abstraction
 * and scalable user interface support. Moreover, Pigment is a fully thread safe
 * library which means that you can call methods on Pigment objects from
 * multiple threads at the same time even if the rendering backend is not thread
 * safe itself like some OpenGL implementations are for example.
 * </para>
 * <title>Pigment initialization</title>
 * <para>
 * The <application>Pigment</application> library has to be initialized
 * with pgm_init() before it can be used. You should pass pointers to
 * the main argc and argv variables so that Pigment can process its own command
 * line options, as shown in the following example.
 * </para>
 * <para>
 * <example id="pigment-initialization">
 * <title>Initializing the library</title>
 * <programlisting language="c">
 * int
 * main (int argc, char *argv[])
 * {
 *   pgm_init (&amp;argc, &amp;argv);
 *   ...
 * }
 * </programlisting>
 * </example>
 * </para>
 * <para>
 * The pgm_deinit() call is used to clean up all internal resources used
 * by <application>Pigment</application>.
 * </para>
 * <title>Pigment version retrieval</title>
 * <para>
 * Use pgm_version() to query the library version at runtime or use the
 * PGM_VERSION_* macros to find the version at compile time. Optionally
 * pgm_version_string() returns a printable string.
 * </para>
 * <title>Pigment programming model</title>
 * <para>
 * Pigment is a GUI toolkit based on an event-driven programming model. When the
 * user is doing nothing, Pigment sits in the main loop and waits for input.
 * If the user performs some action, like a mouse button click, then the main
 * loop "wakes up" and delivers an event to Pigment. When a viewport receives
 * an event, it emits a signal to notify your program that something happened
 * by invoking functions you connected to the signal with
 * g_signal_connect(). Functions connected to a signal are often termed
 * callbacks. When your callbacks are invoked, you would typically take some
 * action, for example when a key of the keyboard is pressed you might update
 * the properties of your drawables. After a callback finishes, Pigment will
 * return to the main loop and await more user input.
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-09-25 (0.3.1)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdlib.h>
#include <string.h>
#include <gst/gst.h>
#include "pgm.h"

GST_DEBUG_CATEGORY (pgm_debug);
#define GST_CAT_DEFAULT pgm_debug

static GMainLoop *main_loop = NULL;
static gboolean pgm_initialized = FALSE;

/* Private functions */

static void
init_logging_element_from_string (const gchar *element)
{
  gchar **array = NULL;
  char *name = NULL, *debug = NULL;

  if (!element)
    return;

  array = g_strsplit (element, ":", 2);
  if (!(array && array[0] && array[1]))
    goto tidy;

  name = array[0];
  debug = array[1];

  /* the debug level is only one bit */
  if (name && *name && debug && *debug && !debug[1])
    {
      gint debug_level = debug[0] - '0';
      gst_debug_set_threshold_for_name (name, debug_level);
    }

 tidy:
  g_free (name);
  g_free (debug);
  g_free (array);
}

static void
init_logging (const gchar *debug_list)
{
  gchar **array = NULL;
  gchar **walk = NULL;

  array = g_strsplit (debug_list, ",", 0);

  walk = array;
  while (walk && *walk)
    {
      init_logging_element_from_string (*walk);
      g_free (*walk);
      walk++;
    }

  g_free (array);
}

/* Public functions */

/**
 * pgm_init:
 * @argc: the address of the @argc parameter of your main function.
 * @argv: the address of the @argv parameter of your main function.
 *
 * Initializes the Pigment library.
 *
 * <note><para>
 * This function will terminate your program if it was unable to initialize
 * Pigment for some reason. If you want your program to fall back,
 * use pgm_init_check() instead.
 * </para></note>
 */
void
pgm_init (int *argc,
          char **argv[])
{
  if (!pgm_init_check (argc, argv))
    {
      g_print ("Could not initialize Pigment\n");
      exit (1);
    }
}

/**
 * pgm_init_check:
 * @argc: the address of the @argc parameter of your main function.
 * @argv: the address of the @argv parameter of your main function.
 *
 * Initializes the Pigment library without terminating the program if it
 * was unable to initialize successfully.
 *
 * Returns: %TRUE if Pigment was initialized successfully, %FALSE otherwise.
 */
gboolean
pgm_init_check (int *argc,
                char **argv[])
{
  const gchar *debug_list = NULL;
  gchar *pigment_version = NULL;
  gboolean registered = FALSE;

  /* If the user gives us PGM_DEBUG, we use its logging request. Otherwise, we
   * use the default log level which is GST_LEVEL_INFO for Subversion builds and
   * GST_LEVEL_NONE for release builds. */
  debug_list = g_getenv ("PGM_DEBUG");
  if (debug_list)
    init_logging (debug_list);
  else
    gst_debug_set_threshold_for_name ("pgm_*", PGM_DEFAULT_LOG_LEVEL);

  GST_DEBUG_CATEGORY_INIT (pgm_debug, "pgm", 0, "Pigment library");
  GST_INFO ("Initializing Pigment");

  /* We are already initialized */
  if (pgm_initialized)
    {
      GST_DEBUG ("Already initialized Pigment");
      return TRUE;
    }

  /* Initialize GStreamer */
  if (!gst_init_check (argc, argv, NULL))
    return FALSE;

  /* Register our image sink. There's a bug in GStreamer with versions older
   * than 0.10.13 which prevents to register static plugin giving NULL for
   * the plugin parameter. The fix has been pushed in revision 1.131 of
   * gstelementfactory.c in the GStreamer repository. */
#ifdef HAVE_GST_0_10_13
  registered = gst_element_register (NULL, "pgmimagesink", GST_RANK_NONE,
                                     PGM_TYPE_IMAGE_SINK);
#else
  {
    /* Let's start the hack */
    GstPlugin plugin;
    plugin.desc.name = g_strdup ("NULL");
    registered = gst_element_register (&plugin, "pgmimagesink", GST_RANK_NONE,
                                       PGM_TYPE_IMAGE_SINK);
  }
#endif /* HAVE_GST_0_10_13 */

  if (!registered)
    {
      GST_DEBUG ("cannot register the gstreamer image sink");
      return FALSE;
    }
  GST_DEBUG ("registered the gstreamer image sink");

  /* We are now initialized */
  pgm_initialized = TRUE;

  pigment_version = pgm_version_string ();
  GST_DEBUG ("%s initialized successfully", pigment_version);
  g_free (pigment_version);

  return TRUE;
}

/**
 * pgm_deinit:
 *
 * Deinitializes the Pigment library. This is not going to free Pigment
 * objects, you need to explicitely unref these objects using
 * gst_object_unref().
 */
void
pgm_deinit (void)
{
  GST_INFO ("Deinitializing Pigment");

  if (pgm_initialized == TRUE)
    pgm_initialized = FALSE;
}

/**
 * pgm_version:
 * @major: a pointer to a #guint to store the major version number.
 * @minor: a pointer to a #guint to store the minor version number.
 * @micro: a pointer to a #guint to store the micro version number.
 * @nano: a pointer to a #guint to store the nano version number.
 *
 * Gets the version number of the Pigment library.
 */
void
pgm_version (guint *major,
             guint *minor,
             guint *micro,
             guint *nano)
{
  g_return_if_fail (major);
  g_return_if_fail (minor);
  g_return_if_fail (micro);
  g_return_if_fail (nano);

  *major = PGM_VERSION_MAJOR;
  *minor = PGM_VERSION_MINOR;
  *micro = PGM_VERSION_MICRO;
  *nano = PGM_VERSION_NANO;
}

/**
 * pgm_version_string:
 *
 * Returns a string that is useful for describing this version of Pigment to
 * the outside world: user agent strings, logging, etc.
 *
 * Returns: a newly allocated string describing this version of Pigment.
 */
gchar *
pgm_version_string (void)
{
  guint major, minor, micro, nano;

  pgm_version (&major, &minor, &micro, &nano);

  if (nano == 0)
    return g_strdup_printf ("Pigment %d.%d.%d", major, minor, micro);
  else if (nano == 1)
    return g_strdup_printf ("Pigment %d.%d.%d (SVN)", major, minor,
                            micro);
  else
    return g_strdup_printf ("Pigment %d.%d.%d (prerelease)", major, minor,
                            micro);
}

/**
 * pgm_events_pending:
 *
 * Checks if any events are pending. This can be used to update the GUI, while
 * doing some time intensive computation.
 *
 * <example id="pigment-event-pending">
 * <title>Updating the GUI during a long computation.</title>
 * <programlisting>
 *   // computation going on
 *   ...
 *   while (pgm_events_pending ())
 *     pgm_main_iteration ();
 *   ...
 *   // computation continued
 * </programlisting>
 * </example>
 *
 * Returns: %TRUE if any events are pending, %FALSE otherwise.
 */
gboolean
pgm_events_pending (void)
{
  return g_main_context_pending (NULL);
}

/**
 * pgm_main:
 *
 * Runs the main loop until pgm_main_quit() is called. It's ok to use the GLib
 * main loop directly instead, though it involves slightly more typing. See
 * GMainLoop in the GLib documentation.
 */
void
pgm_main (void)
{
  if (G_LIKELY (!main_loop))
    {
      main_loop = g_main_loop_new (NULL, TRUE);

      GST_DEBUG ("Running the main loop");
      g_main_loop_run (main_loop);

      g_main_loop_unref (main_loop);
      main_loop = NULL;
    }
  else
    GST_DEBUG ("Cannot nest calls to pgm_main");
}

/**
 * pgm_main_quit:
 *
 * Quits the main loop.
 */
void
pgm_main_quit (void)
{
  GST_DEBUG ("Quitting the main loop");
  g_main_loop_quit (main_loop);
}

/**
 * pgm_main_iteration:
 *
 * Runs a single iteration of the main loop. If no events are waiting to be
 * processed, Pigment will block until the next event is noticed. If you don't
 * want to block, look at pgm_main_iteration_do() or check if any events are
 * pending with pgm_events_pending() first.
 */
void
pgm_main_iteration (void)
{
  g_main_context_iteration (NULL, TRUE);
}

/**
 * pgm_main_iteration_do:
 * @blocking: %TRUE if you want Pigment to block if no events are pending.
 *
 * Runs a single iteration of the main loop. If no events are available,
 * either returns or blocks dependent on the value of @blocking.
 */
void
pgm_main_iteration_do (gboolean blocking)
{
  g_main_context_iteration (NULL, blocking);
}
