/* places-glib: A GObject wrapper for the Mozilla Places API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>

#include "places-error.h"
#include "places-error-private.h"

#include "places-history.h"
#include "places-history-bindings.h"
#include "places-service.h"
#include "places-marshal.h"

static void places_history_dispose (GObject *object);

G_DEFINE_TYPE (PlacesHistory, places_history, G_TYPE_OBJECT);

#define PLACES_HISTORY_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), PLACES_TYPE_HISTORY, \
                                PlacesHistoryPrivate))

typedef struct _PlacesHistoryFaviconClosure PlacesHistoryFaviconClosure;

struct _PlacesHistoryPrivate
{
  DBusGProxy *proxy;

  guint callback_id;

  GSList *favicon_callbacks;
};

struct _PlacesHistoryFaviconClosure
{
  PlacesHistory *history;
  DBusGProxyCall *proxy_call;
  guint id;

  PlacesHistoryFaviconCallback callback;
  gpointer user_data;
  GDestroyNotify user_data_notify;
};

enum
  {
    AC_RESULT_RECEIVED_SIGNAL,

    LAST_SIGNAL
  };

static guint history_signals[LAST_SIGNAL] = { 0, };

static void
places_history_class_init (PlacesHistoryClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->dispose = places_history_dispose;

  history_signals[AC_RESULT_RECEIVED_SIGNAL] =
    g_signal_new ("ac-result-received",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (PlacesHistoryClass, ac_result_received),
		  NULL, NULL,
                  places_marshal_VOID__UINT_STRING_STRING,
                  G_TYPE_NONE, 3,
                  G_TYPE_UINT,
                  G_TYPE_STRING,
                  G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (PlacesHistoryPrivate));

  dbus_g_object_register_marshaller (places_marshal_VOID__UINT_STRING_STRING,
                                     G_TYPE_NONE,
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
}

static void
places_history_ac_result_received_cb (DBusGProxy *proxy,
                                      guint search_id,
                                      const gchar *value,
                                      const gchar *comment,
                                      gpointer data)
{
  g_signal_emit (data, history_signals[AC_RESULT_RECEIVED_SIGNAL], 0,
                 search_id, value, comment);
}

static void
places_history_init (PlacesHistory *self)
{
  PlacesHistoryPrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;

  priv = self->priv = PLACES_HISTORY_GET_PRIVATE (self);

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      priv->proxy
        = dbus_g_proxy_new_for_name (connection,
                                     PLACES_SERVICE_HISTORY,
                                     PLACES_SERVICE_HISTORY_PATH,
                                     PLACES_SERVICE_HISTORY_INTERFACE);

      dbus_g_connection_unref (connection);

      dbus_g_proxy_add_signal (priv->proxy, "AcResultReceived",
                               G_TYPE_UINT,
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_INVALID);

      dbus_g_proxy_connect_signal
        (priv->proxy, "AcResultReceived",
         G_CALLBACK (places_history_ac_result_received_cb),
         self, NULL);
    }
}

static void
places_history_dispose (GObject *object)
{
  PlacesHistory *self = (PlacesHistory *) object;
  PlacesHistoryPrivate *priv = self->priv;

  if (priv->proxy)
    {
      while (priv->favicon_callbacks)
        {
          PlacesHistoryFaviconClosure *closure
            = priv->favicon_callbacks->data;
          /* This should also destroy the closure and remove it from
             the list */
          dbus_g_proxy_cancel_call (priv->proxy, closure->proxy_call);
        }

      g_object_unref (priv->proxy);
      priv->proxy = NULL;
    }

  G_OBJECT_CLASS (places_history_parent_class)->dispose (object);
}

PlacesHistory *
places_history_new (void)
{
  PlacesHistory *self = (PlacesHistory *) g_object_new (PLACES_TYPE_HISTORY,
                                                        NULL);

  return self;
}

static gboolean
places_history_check_proxy (PlacesHistory *self,
                            GError **error)
{
  if (self->priv->proxy)
    return TRUE;

  g_set_error (error, PLACES_ERROR,
               PLACES_ERROR_PROXY,
               "Failed to initialize DBUS proxy");

  return FALSE;
}

gboolean
places_history_add_uri (PlacesHistory *self,
                        const gchar *uri,
                        GError **error)
{
  PlacesHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!places_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_places_History_add_uri (priv->proxy, uri, error);

  _places_error_translate_from_dbus (error);

  return ret;
}

gboolean
places_history_set_page_title (PlacesHistory *self,
                               const gchar *uri,
                               const gchar *title,
                               GError **error)
{
  PlacesHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!places_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_places_History_set_page_title (priv->proxy,
                                                  uri, title,
                                                  error);

  _places_error_translate_from_dbus (error);

  return ret;
}

gboolean
places_history_start_ac_search (PlacesHistory *self,
                                const gchar *search_str,
                                guint32 *search_id,
                                GError **error)
{
  PlacesHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!places_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_places_History_start_ac_search (priv->proxy,
                                                   search_str, search_id,
                                                   error);

  _places_error_translate_from_dbus (error);

  return ret;
}

gboolean
places_history_stop_ac_search (PlacesHistory *self,
                               guint32 search_id,
                               GError **error)
{
  PlacesHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!places_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_places_History_stop_ac_search (priv->proxy,
                                                  search_id,
                                                  error);

  _places_error_translate_from_dbus (error);

  return ret;
}

gboolean
places_history_set_favicon_url (PlacesHistory *self,
                                const gchar *page_uri,
                                const gchar *favicon_uri,
                                GError **error)
{
  PlacesHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!places_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_places_History_set_favicon_url (priv->proxy,
                                                   page_uri, favicon_uri,
                                                   error);

  _places_error_translate_from_dbus (error);

  return ret;
}

gboolean
places_history_set_default_favicon_url (PlacesHistory *self,
                                        const gchar *page_uri,
                                        GError **error)
{
  PlacesHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!places_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_places_History_set_default_favicon_url (priv->proxy,
                                                           page_uri,
                                                           error);

  _places_error_translate_from_dbus (error);

  return ret;
}

static void
places_history_favicon_closure_destroy_cb (gpointer data)
{
  PlacesHistoryFaviconClosure *closure = data;
  PlacesHistoryPrivate *priv = closure->history->priv;

  if (closure->user_data_notify)
    closure->user_data_notify (closure->user_data);

  priv->favicon_callbacks = g_slist_remove (priv->favicon_callbacks, closure);

  g_slice_free (PlacesHistoryFaviconClosure, closure);
}

static void
places_history_get_favicon_cb (DBusGProxy *proxy,
                               DBusGProxyCall *proxy_call,
                               void *data)
{
  PlacesHistoryFaviconClosure *closure = data;
  GError *error = NULL;
  gchar *mime_type = NULL;
  GArray *icon_data = NULL;

  if (dbus_g_proxy_end_call (proxy, proxy_call, &error,
                             G_TYPE_STRING, &mime_type,
                             DBUS_TYPE_G_UCHAR_ARRAY, &icon_data,
                             G_TYPE_INVALID))
    {
      closure->callback (closure->history,
                         mime_type,
                         (guint8 *) icon_data->data,
                         icon_data->len,
                         NULL,
                         closure->user_data);

      g_free (mime_type);
      g_array_free (icon_data, TRUE);
    }
  else
    {
      _places_error_translate_from_dbus (&error);

      closure->callback (closure->history,
                         NULL,
                         NULL, 0,
                         error,
                         closure->user_data);

      g_error_free (error);
    }
}

guint
places_history_get_favicon (PlacesHistory *self,
                            const gchar *page_uri,
                            gboolean download,
                            PlacesHistoryFaviconCallback callback,
                            gpointer user_data,
                            GDestroyNotify user_data_notify)
{
  PlacesHistoryPrivate *priv;
  PlacesHistoryFaviconClosure *closure;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);

  priv = self->priv;

  if (++priv->callback_id == 0)
    priv->callback_id = 1;

  closure = g_slice_new0 (PlacesHistoryFaviconClosure);
  closure->history = self;
  closure->id = priv->callback_id;
  closure->callback = callback;
  closure->user_data = user_data;
  closure->user_data_notify = user_data_notify;

  closure->proxy_call
    = dbus_g_proxy_begin_call (priv->proxy,
                               "GetFavicon",
                               places_history_get_favicon_cb,
                               closure,
                               places_history_favicon_closure_destroy_cb,
                               G_TYPE_STRING,
                               page_uri,
                               G_TYPE_BOOLEAN,
                               download,
                               G_TYPE_INVALID);

  priv->favicon_callbacks = g_slist_prepend (priv->favicon_callbacks, closure);

  return closure->id;
}

void
places_history_cancel (PlacesHistory *self,
                       guint id)
{
  PlacesHistoryPrivate *priv;
  PlacesHistoryFaviconClosure *closure;
  GSList *l;

  g_return_if_fail (PLACES_IS_HISTORY (self));

  priv = self->priv;

  for (l = priv->favicon_callbacks; l; l = l->next)
    {
      closure = l->data;

      if (closure->id == id)
        {
          dbus_g_proxy_cancel_call (priv->proxy,
                                    closure->proxy_call);
          break;
        }
    }
}
