/* places-glib: A GObject wrapper for the Mozilla Places API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <nsXPCOM.h>
#include <nsCOMPtr.h>
#include <nsIDirectoryService.h>
#include <nsAppDirectoryServiceDefs.h>
#include <nsStringAPI.h>
#include <nsILocalFile.h>
#include <nsXULAppAPI.h>

#include "places-service.h"
#include "places-history.h"
#include "places-error-private.h"
#include "places-event-source.h"
#include "places-enum-types.h"

class PlacesDirectoryServiceProvider : public nsIDirectoryServiceProvider
{
public:
  PlacesDirectoryServiceProvider ();
  virtual ~PlacesDirectoryServiceProvider ();

  NS_DECL_ISUPPORTS
  NS_DECL_NSIDIRECTORYSERVICEPROVIDER

private:
  gchar *profile_dir;
};

NS_IMPL_ISUPPORTS1(PlacesDirectoryServiceProvider, nsIDirectoryServiceProvider)

PlacesDirectoryServiceProvider::PlacesDirectoryServiceProvider ()
: profile_dir (NULL)
{
}

PlacesDirectoryServiceProvider::~PlacesDirectoryServiceProvider()
{
  if (profile_dir)
    g_free (profile_dir);
}

NS_IMETHODIMP
PlacesDirectoryServiceProvider::GetFile (const char *prop,
                                         PRBool *persistent,
                                         nsIFile **retval)
{
  if (!strcmp (prop, NS_APP_USER_PROFILE_50_DIR))
    {
      if (profile_dir == NULL)
        {
          profile_dir = g_build_filename (g_get_home_dir (),
                                          ".moblin-web-browser",
                                          NULL);

          if (!g_file_test (profile_dir, G_FILE_TEST_IS_DIR))
            g_mkdir_with_parents (profile_dir, 0755);
        }

      nsresult rv;
      nsCOMPtr<nsILocalFile> localFile;

      *persistent = PR_TRUE;

      /* Convert the native file string to an nsIFile */
      rv = NS_NewNativeLocalFile (nsDependentCString (profile_dir),
                                  1, getter_AddRefs (localFile));

      if (NS_SUCCEEDED (rv))
        rv = localFile->QueryInterface (nsIFile::GetIID (), (void **) retval);

      return rv;
    }

  return NS_ERROR_FAILURE;
}

static void
places_service_run_main_loop (GMainLoop *main_loop, GError **error)
{
  nsresult rv;

  nsCOMPtr<nsILocalFile> moz_home;
  rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME), PR_TRUE,
                              getter_AddRefs (moz_home));
  if (NS_FAILED (rv))
    places_error_set_from_nsresult (rv, error);
  else
    {
      /* Initialize embedding */
      nsIDirectoryServiceProvider *dir_provider
        = new PlacesDirectoryServiceProvider;
      /* The dir_provider has no reference so XPCom will take
         ownership of it */
      rv = XRE_InitEmbedding (moz_home, moz_home, dir_provider,
                              nsnull, nsnull);

      if (NS_FAILED (rv))
        places_error_set_from_nsresult (rv, error);
      else
        {
          GSource *event_source = places_event_source_new ();

          XRE_NotifyProfile ();

          /* Make sure the objects we created are unrefd before
             shutting down XPCom */
          moz_home = 0;

          g_source_attach (event_source, NULL);
          g_main_loop_run (main_loop);

          g_source_destroy (event_source);
          g_source_unref (event_source);

          XRE_TermEmbedding ();
        }
    }
}

int
main (int argc, char **argv)
{
  int ret = PLACES_SERVICE_EXIT_SUCCESS;
  GError *error = NULL;
  GMainLoop *main_loop;
  DBusGConnection *bus;

  g_thread_init (NULL);
  g_type_init ();

  main_loop = g_main_loop_new (NULL, FALSE);

  if ((bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("%s", error->message);
      g_clear_error (&error);
      ret = PLACES_SERVICE_EXIT_ERROR;
    }
  else
    {
      DBusGProxy *proxy = dbus_g_proxy_new_for_name (bus,
                                                     DBUS_SERVICE_DBUS,
                                                     DBUS_PATH_DBUS,
                                                     DBUS_INTERFACE_DBUS);
      guint32 request_status;

      if (!org_freedesktop_DBus_request_name (proxy,
                                              PLACES_SERVICE_HISTORY,
                                              DBUS_NAME_FLAG_DO_NOT_QUEUE,
                                              &request_status, &error))
        {
          g_warning ("Failed to request name: %s", error->message);
          g_clear_error (&error);
          ret = PLACES_SERVICE_EXIT_ERROR;
        }
      else if (request_status == DBUS_REQUEST_NAME_REPLY_EXISTS)
        {
          g_warning ("Places history service already exists");
          ret = PLACES_SERVICE_EXIT_SERVICE_EXISTS;
        }
      else if (request_status != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
        {
          g_warning ("Failed to request name");
          ret = PLACES_SERVICE_EXIT_ERROR;
        }
      else
        {
          PlacesHistory *history;
          GError *error = NULL;

          dbus_g_error_domain_register (PLACES_ERROR, NULL, PLACES_TYPE_ERROR);

          history = places_history_new ();

          places_service_run_main_loop (main_loop, &error);

          if (error)
            {
              g_warning ("%s", error->message);
              g_error_free (error);
              ret = PLACES_SERVICE_EXIT_ERROR;
            }

          g_object_unref (history);
        }

      g_object_unref (proxy);

      dbus_g_connection_unref (bus);
    }

  g_main_loop_unref (main_loop);

  return ret;
}
