# -*- coding: utf-8; Mode: Python; indent-tabs-mode: nil; tab-width: 4 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import signal
import gettext
import debconf
import parted

from ubiquity.plugin import *
from ubiquity import misc

from urecovery import recipe
from urecovery import util

NAME='recovery-prep'
BEFORE='partman'
WEIGHT=11
DOMAIN='ubuntu-recovery'
OEM=False
gettext.install(DOMAIN)

class ActionNotFound(Exception):    pass

class PageGtk (PluginUI):
    plugin_prefix = NAME

    #{{{def __init__(self, controller, *args, **kwargs):
    def __init__(self, controller, *args, **kwargs):
        import gtk
        self.controller = controller
        builder = gtk.Builder()
        builder.set_translation_domain(DOMAIN)
        builder.add_from_file('/usr/share/ubiquity/gtk/step%s.ui' %
NAME.capitalize())
        builder.connect_signals(self)
        self.controller.add_builder(builder)
        self.page = builder.get_object('stepOEMRecovery')
        self.label = builder.get_object('label_message')
        self.reboot_warning = builder.get_object('reboot_warning')
        self.page.set_name(NAME)
        self.plugin_widgets = self.page
        self.error_dialog = builder.get_object('error_dialog1')
    #}}}

    #{{{def setMessage(self, text, isError = False):
    def setMessage(self, text, isError = False):
        self.label.set_label(text)
        if isError:
            self.reboot_warning.hide()
        else:
            self.reboot_warning.show()
    #}}}

    #{{{def setAllowForward(self, allowed):
    def setAllowForward(self, allowed):
        self.controller.allow_go_forward(allowed)
        self.forward.set_sensitive(allowed)
    #}}}
    def on_button3_clicked(self):
        execute_root("poweroff")
pass

class Page(Plugin):
    state = None
    error_text = None
    reserved_partitions = (0,)
    reserve_userdata = True

    #{{{def prepare(self, unfiltered=False):
    def prepare(self, unfiltered=False):
        try:
            self.default_fstype = self.db.get('partman/default_filesystem') or 'ext4'
        except debconf.DebconfError:
            pass
        with misc.raised_privileges():
            # get preseed value of partman-auto/disk in chroot if possible.
            # this value can be configured in build config of image.
            self.diskname = self.db.get("partman-auto/disk") or "/dev/sda"
            self.dev = parted.Device(self.diskname)
            self.disk = parted.Disk(self.dev)
            self.state = self.getState()
            if self.state == "create":
                self.ok_handler()
        return Plugin.prepare(self, unfiltered=unfiltered);
    #}}}

    #{{{def getState(self):
    def getState(self):
        if not self.state and self.disk.type == 'gpt':
            self.error_text = _("GUID partition table has been detected. This is not supported. Recovery cannot continue.");
            return "error"

        if self.disk.primaryPartitionCount >= 5:
            self.error_text = _("""There are already five or more partitions.
Recovery cannot continue.  Please make sure there are no more than four
partitions and restart recovery.""")
            return "error"

        if util.kopts_haskey('ubuntu-recovery'):
            return "recovery"
        return "create"
    #}}}

    #{{{def ok_handler(self):
    def ok_handler(self):
        self.preseed_bool('partman-basicfilesystems/no_swap',False)
        self.preseed_bool('grub-installer/only_debian',False)
        self.preseed_bool('grub-installer/with_other_os',False)
        action = "do%s" % self.state.capitalize()
        try:
            fn = getattr(self, action)
        except AttributeError:
            raise ActionNotFound("Action %s not found." % action)
        fn()
        Plugin.ok_handler(self)
    #}}}

    #{{{def doError(self):
    def doError(self):
        self.ui.setMessage(self.error_text, isError=True)
        self.ui.setAllowForward(False)
    #}}}

    #{{{def doCreate(self):
    def doCreate(self):
        self.debug("recovery-prep: do create")
        self.preseed('grub-installer/bootdev', self.diskname)
        self.setRecipe('create')
    #}}}

    #{{{def doRecovery(self):
    def doRecovery(self):
        self.debug("recovery-prep: do recovery")
        self.setRecipe('recovery')
        # before we remove system partition grub installed,  we need make sure the machines
        # always can be booted if the recovery process does not be complete, and then user
        # can run recovery tool again. for example, the laptop is running recovery and poweroff
        # after the root partition is deleted can not be booted again.
        # when recovery is done, we will set it boot from grub again.
        self.setBootFromRecoveryPartition()
    #}}}

    #{{{def setBootFromRecoveryPartition(self):
    def setBootFromRecoveryPartition(self):
        "set boot from recovery partition"
#        if not misc.execute_root('dd',
#                                'if=/usr/share/ubuntu-recovery/boot/syslinux_mbr.bin',
#                                'of=/dev/sda', 'bs=512', 'count=1'):
#            raise RuntimeError("Install SysLinux in Recovery Partition failed")
#       misc.execute_root('parted', '-s', '/dev/sda', 'set', '1', 'boot', 'on')
        os.system('/usr/lib/ubiquity/bin/dorecovery.sh &')
        #misc.execute_root('/usr/lib/ubiquity/bin/dorecovery.sh')
        util.set_partition_typenum(self.diskname, '1', util.FS_VFAT)
    #}}}

    #{{{def setRecipe(self, name, auto=True):
    def setRecipe(self, name, auto=True):
        """set partman-auto recipe
        the default partitoins layout
            recovery partition
            root partition
            swap partition

        when doing recovery, we only formate root and swap partitions.

        \param str name recipe name
        \param bool auto without user's confirmation if True
        """
        r = recipe.Recipe("%s-recipe" % name)
        # only create recovery partiton when first installation.
        hiddenpart = recipe.Partition(min=2000, max=2000,
                                pirority=100000,
                                bootable=False,
                                fstype='fat32')
#        homepart = recipe.Partition(min=500, max=1000000,
#                            pirority=100000,
#                            fstype=self.default_fstype,
#                            mountpoint='/home')
        if name == 'recovery':
            hiddenpart.formate = False
#        if self.reserve_userdata:
#            homepart.formate = False
        r.addPartition(hiddenpart)
#        r.addPartition(homepart)
        r.addPartition(recipe.Partition(min='200%', max='200%',
                            pirority=100000,
                            fstype='swap')) 
        r.addPartition(recipe.Partition(min=500, max=10000000,
                            pirority=100000,
                            fstype=self.default_fstype,
                            mountpoint='/'))
        # partman-atuo settings
        recipe_filename = '/tmp/expert_recipe'
        r.saveTo(recipe_filename)
        self.preseed('partman-auto/expert_recipe_file', recipe_filename)
        self.preseed('partman-auto/choose_recipe', r.name)
        self.preseed('partman-auto/disk', self.diskname)
        self.preseed('partman-auto/method', 'regular')
        # This makes partman automatically partition without confirmation, provided
        # that you told it what to do using one of the methods above.
        if auto:
            self.preseed_bool("partman/confirm_write_new_label", True)
            self.preseed('partman/choose_partition', 'finish')
            self.preseed_bool('partman/confirm', True)
    #}}}
pass
