# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Tests for logger utils """
from __future__ import with_statement

import logging
import unittest
from contrib.testing.testcase import MementoHandler
from ubuntuone.syncdaemon.logger import (
    DebugCapture,
    NOTE,
    TRACE,
    root_logger,
    twisted_logger,
    inotify_logger,
    MultiFilter,
)


class DebugCaptureTest(unittest.TestCase):
    """Tests for DebugCapture context manager."""

    def setUp(self):
        """Setup the logger and the handler"""
        self.handler = MementoHandler()
        self.logger = logging.getLogger(self.__class__.__name__)
        self.logger.addHandler(self.handler)
        self.logger.setLevel(logging.DEBUG)

    def tearDown(self):
        """close the handler and restore the logger (Logger's are global)"""
        self.handler.close()
        self.logger.removeHandler(self.handler)
        self.logger.setLevel(logging.DEBUG)

    def test_capture_in_debug_or_lower(self):
        """Test simple capture with the logger in DEBUG level"""
        self.logger.debug('a message')
        self.assertEquals(1, len(self.handler.records))
        self.handler.records = []

        self.logger.setLevel(TRACE)
        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should never reach the handler')
            self.assertEquals(0, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(1, len(self.handler.records), messages)
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should never reach the handler')
            self.logger.warning('a warning')
            self.logger.debug('another debug message')
            self.assertEquals(0, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(3, len(self.handler.records), messages)
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should never reach the handler')
            self.logger.info('a info message')
            self.logger.debug('another debug message')
            self.assertEquals(0, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(3, len(self.handler.records), messages)

    def test_capture_non_debug_levels(self):
        """Test debug log capture in levels > DEBUG"""
        levels = [logging.INFO, logging.ERROR,
                 NOTE, logging.CRITICAL]
        for level in levels:
            self.logger.setLevel(level)
            self._test_capture()
            self.logger.setLevel(logging.DEBUG)
            self.handler.records = []

    def _test_capture(self):
        """Tests for simple debug capture in INFO level"""
        self.logger.debug('a message')
        self.assertEquals(0, len(self.handler.records))
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should never reach the handler')
            self.assertEquals(1, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(0, len(self.handler.records), messages)
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should never reach the handler')
            self.logger.warning('a warning')
            self.logger.debug('another debug message')
            self.assertEquals(2, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(1, len(self.handler.records), messages)
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should never reach the handler')
            self.logger.info('a info message')
            self.logger.debug('another debug message')
            self.assertEquals(2, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(1, len(self.handler.records), messages)

    def test_dump_on_unhandled_error(self):
        """Test that all captured debug info is dumped on a unhandled error and
        the error itself is logged too
        """
        self.logger.setLevel(logging.INFO)
        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler 1')
            self.assertEquals(1, len(dc.records))
            raise Exception('Expected exception!')
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(2, len(self.handler.records), messages)

    def test_dump_on_unhandled_error_in_DEBUG(self):
        """Test that all captured debug info is dumped on a unhandled error and
        the error itself is logged too (in DEBUG level)
        """
        # now with level <= DEBUG
        self.logger.setLevel(logging.DEBUG)
        self.handler.records = []
        self.logger.setLevel(logging.INFO)
        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler 1')
            self.assertEquals(1, len(dc.records))
            raise Exception('Expected exception!')
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(2, len(self.handler.records), messages)

    def test_dump_on_error_log(self):
        """Test that all captured debug info is dumped on ERROR log"""
        self.logger.setLevel(logging.INFO)
        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler')
            self.logger.error('Oops! an error')
            self.assertEquals(2, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(2, len(self.handler.records), messages)
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler')
            self.logger.error('Oops! an error')
            self.logger.debug('another message')
            self.assertEquals(3, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(3, len(self.handler.records), messages)
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler')
            self.logger.error('Oops! an error')
            self.logger.debug('another message')
            self.logger.error('Oh my! another error!')
            self.assertEquals(4, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(4, len(self.handler.records), messages)

    def test_dump_on_error_log_DEBUG(self):
        """Test that all captured debug info is dumped on ERROR log"""
        self.logger.setLevel(logging.DEBUG)
        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler')
            self.logger.error('Oops! an error')
            self.assertEquals(0, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(2, len(self.handler.records), messages)
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler')
            self.logger.error('Oops! an error')
            self.logger.debug('another message')
            self.assertEquals(0, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(3, len(self.handler.records), messages)
        self.handler.records = []

        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler')
            self.logger.error('Oops! an error')
            self.logger.debug('another message')
            self.logger.error('Oh my! another error!')
            self.assertEquals(0, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(4, len(self.handler.records), messages)

    def test_dump_on_critical_log(self):
        """Test that the dump also works on levels > ERROR"""
        self.logger.setLevel(logging.INFO)
        with DebugCapture(self.logger) as dc:
            self.logger.debug('a message that should go to the handler')
            self.logger.critical('Oops! an error')
            self.assertEquals(2, len(dc.records))
        messages = [r.getMessage() for r in self.handler.records]
        self.assertEquals(2, len(self.handler.records), messages)


class FilterTests(unittest.TestCase):
    """Tests log filters"""

    def setUp(self):
        """Setup the logger and the handler"""
        self.handler = MementoHandler()
        self.handler.setLevel(logging.DEBUG)
        root_logger.addHandler(self.handler)
        inotify_logger.addHandler(self.handler)
        twisted_logger.addHandler(self.handler)

    def tearDown(self):
        """close the handler and restore the logger (Logger's are global)"""
        self.handler.close()
        root_logger.removeHandler(self.handler)
        inotify_logger.removeHandler(self.handler)
        twisted_logger.removeHandler(self.handler)

    def test_multiple_filters(self):
        """Tests logging with more than one filter."""
        test_logger = logging.getLogger('ubuntuone.SyncDaemon.FilterTest')
        test_logger.debug('debug info 0')
        self.assertEquals(1, len(self.handler.records))
        self.handler.addFilter(MultiFilter(['ubuntuone.SyncDaemon', 'twisted', 'pyinotify']))
        test_logger.debug('debug info 1')
        self.assertEquals(2, len(self.handler.records))


class MultiFilterTest(unittest.TestCase):
    """Tests for logger.MultiFilter"""

    def setUp(self):
        """Setup the logger and the handler"""
        self.handler = MementoHandler()
        self.logger = logging.getLogger(self.__class__.__name__)
        self.logger.addHandler(self.handler)
        self.logger.setLevel(logging.DEBUG)

    def tearDown(self):
        """close the handler and restore the logger (Logger's are global)"""
        self.handler.close()
        self.logger.removeHandler(self.handler)
        self.logger.setLevel(logging.DEBUG)

    def test_no_filters(self):
        """Tests filtering without any filter in self.filters."""
        self.handler.addFilter(MultiFilter())
        self.logger.debug('this msg should be logged')
        self.assertEquals(1, len(self.handler.records))

    def test_single_filter(self):
        """Tests filtering with one filter."""
        self.handler.addFilter(MultiFilter([self.__class__.__name__]))
        self.logger.debug('this msg should be logged')
        self.assertEquals(1, len(self.handler.records))
        other_logger = logging.getLogger("NO_LOG."+self.__class__.__name__)
        other_logger.debug('this msg shouldn\'t be logged')
        self.assertEquals(1, len(self.handler.records))

    def test_multiple_filters(self):
        """Tests filtering with more than one filter."""
        self.handler.addFilter(MultiFilter([self.__class__.__name__,
                                        self.__class__.__name__ + ".child"]))
        no_logger = logging.getLogger("NO_LOG."+self.__class__.__name__)
        yes_logger = logging.getLogger(self.__class__.__name__ + '.child')
        self.logger.debug('this msg should be logged')
        self.assertEquals(1, len(self.handler.records))
        no_logger.debug('this msg shouldn\'t be logged')
        self.assertEquals(1, len(self.handler.records))
        yes_logger.debug('this msg from a child logger should be logged')
        self.assertEquals(2, len(self.handler.records))

