/*
 * Copyright 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 3, as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *    Cody Russell  <crussell@canonical.com>
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib-bindings.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <libdbusmenu-gtk/menuitem.h>
#include <libdbusmenu-glib/menuitem.h>
#include <libdbusmenu-glib/server.h>

#include "bridge.h"
#include "application-menu-registrar-client.h"

#define APP_MENU_DBUS_NAME   "org.ayatana.AppMenu.Registrar"
#define APP_MENU_DBUS_OBJECT "/org/ayatana/AppMenu/Registrar"
#define APP_MENU_INTERFACE   "org.ayatana.AppMenu.Registrar"
#define APP_MENU_PATH        "/this/is/a/long/object/path"

static void app_menu_bridge_insert         (UbuntuMenuProxy   *proxy,
                                            GtkWidget         *shell,
                                            GtkWidget         *child,
                                            guint              position);
static gboolean app_menu_bridge_show_local (UbuntuMenuProxy   *proxy);
static void     dbus_owner_change          (DBusGProxy        *proxy,
                                            const gchar       *name,
                                            const gchar       *prev,
                                            const gchar       *new,
                                            AppMenuBridge     *bridge);
static void     toplevel_realized          (GtkWidget         *widget,
                                            gpointer           user_data);
static void     toplevel_notify_cb         (GtkWidget         *widget,
                                            GParamSpec        *pspec,
                                            UbuntuMenuProxy   *proxy);
static void     rebuild_window_items       (AppMenuBridge     *bridge,
                                            GtkWidget         *toplevel);
static void     rebuild                    (AppMenuBridge     *bridge,
                                            GtkWidget         *toplevel);
static void app_menu_bridge_proxy_vanished (AppMenuBridge *bridge);
static void app_menu_bridge_proxy_appeared (AppMenuBridge *bridge);

typedef struct _AppWindowContext AppWindowContext;

struct _AppWindowContext
{
  GtkWidget        *window;
  DbusmenuServer   *server;
  DbusmenuMenuitem *root;
  gchar            *path;
  gboolean          registered;
  AppMenuBridge    *bridge;
  GHashTable       *lookup;
};

struct _AppMenuBridgePrivate
{
  GList *windows;

  DBusGConnection  *session_bus;
  DBusGProxy       *dbusproxy;
  DBusGProxy       *appmenuproxy;
  gboolean          online;
};

typedef struct _RecurseContext
{
  AppMenuBridge    *bridge;
  AppWindowContext *context;

  gint count;
  gboolean previous;
  DbusmenuMenuitem *stack[30];
  gboolean          mode[30];
} RecurseContext;

enum {
  SEPARATOR_MODE_SMART,
  SEPARATOR_MODE_VISIBLE,
  SEPARATOR_MODE_HIDDEN
};

G_DEFINE_DYNAMIC_TYPE(AppMenuBridge, app_menu_bridge, UBUNTU_TYPE_MENU_PROXY)

static GHashTable *rebuild_ids = NULL;

static void
app_menu_bridge_setup_proxy (AppMenuBridge *bridge)
{
  bridge->priv->appmenuproxy = dbus_g_proxy_new_for_name_owner (bridge->priv->session_bus,
                                                                APP_MENU_DBUS_NAME,
                                                                APP_MENU_DBUS_OBJECT,
                                                                APP_MENU_INTERFACE,
                                                                NULL);

  bridge->priv->online = TRUE;
}

/* Async callback from asking DBus if the registrar
   exists on the bus */
static void
has_owner_cb (DBusGProxy *proxy, gboolean has_owner, GError *error, gpointer userdata)
{
	if (error != NULL) {
		g_warning("Unable to check for Registrar owner: %s", error->message);
		return;
	}

	if (has_owner) {
		app_menu_bridge_proxy_appeared (APP_MENU_BRIDGE(userdata));
	} else {
		app_menu_bridge_proxy_vanished (APP_MENU_BRIDGE(userdata));
	}

	return;
}

static void
activate_menu (AppMenuBridge *bridge,
               GtkWidget     *widget,
               gpointer       user_data)
{
  GList *list;

  for (list = bridge->priv->windows; list != NULL; list = list->next)
    {
      AppWindowContext *context = list->data;

      if (context->lookup)
        {
          DbusmenuMenuitem *mi = g_hash_table_lookup (context->lookup, widget);

          if (mi != NULL)
            {
              dbusmenu_menuitem_show_to_user (mi, 0);

              return;
            }
        }
    }
}

static void
app_menu_bridge_init (AppMenuBridge *bridge)
{
  bridge->priv = G_TYPE_INSTANCE_GET_PRIVATE (bridge, APP_MENU_TYPE_BRIDGE, AppMenuBridgePrivate);

  bridge->priv->windows = NULL;

  bridge->priv->appmenuproxy = NULL;

  bridge->priv->online = FALSE;

  bridge->priv->session_bus = dbus_g_bus_get(DBUS_BUS_SESSION, NULL);

  bridge->priv->dbusproxy = dbus_g_proxy_new_for_name (bridge->priv->session_bus,
                                                       DBUS_SERVICE_DBUS,
                                                       DBUS_PATH_DBUS,
                                                       DBUS_INTERFACE_DBUS);

  if (bridge->priv->dbusproxy)
    {
      dbus_g_proxy_add_signal (bridge->priv->dbusproxy,
                               "NameOwnerChanged",
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_INVALID);

      dbus_g_proxy_connect_signal (bridge->priv->dbusproxy,
                                   "NameOwnerChanged",
                                   G_CALLBACK (dbus_owner_change),
                                   bridge,
                                   NULL);

      org_freedesktop_DBus_name_has_owner_async (bridge->priv->dbusproxy,
                                                 APP_MENU_DBUS_NAME,
                                                 has_owner_cb,
                                                 bridge);
    }

  g_signal_connect (bridge,
                    "activate-menu",
                    G_CALLBACK (activate_menu),
                    NULL);

  return;
}

static void
app_menu_bridge_dispose (GObject *object)
{
  AppMenuBridge *bridge = APP_MENU_BRIDGE (object);

  if (bridge->priv->appmenuproxy)
    {
      g_object_unref (bridge->priv->appmenuproxy);
      bridge->priv->appmenuproxy = NULL;
    }

  if (bridge->priv->dbusproxy)
    {
      g_object_unref (bridge->priv->dbusproxy);
      bridge->priv->dbusproxy = NULL;
    }

  if (bridge->priv->session_bus)
    {
      g_object_unref (bridge->priv->session_bus);
      bridge->priv->session_bus = NULL;
    }
}

static void
free_context_contents (AppWindowContext *context)
{
  if (context->path != NULL)
    {
      g_free (context->path);
      context->path = NULL;
    }

  if (context->root != NULL)
    {
      g_object_unref (context->root);
      context->root = NULL;
    }

  if (context->server != NULL)
    {
      g_object_unref (context->server);
      context->server = NULL;
    }

  if (context->lookup)
    {
      g_hash_table_unref (context->lookup);
    }
}

static void
app_menu_bridge_finalize (GObject *object)
{
  AppMenuBridge *bridge = APP_MENU_BRIDGE (object);
  GList *tmp = NULL;

  for (tmp = bridge->priv->windows; tmp != NULL; tmp = tmp->next)
    {
      AppWindowContext *context = tmp->data;

      free_context_contents (context);

      g_free (context);
    }

  g_list_free (bridge->priv->windows);
  bridge->priv->windows = NULL;

  G_OBJECT_CLASS (app_menu_bridge_parent_class)->finalize (object);
}

static void
toplevel_destroyed (GtkWidget *widget,
                    gpointer   user_data)
{
  AppWindowContext *context = (AppWindowContext *)user_data;

  if (context)
    {
      free_context_contents (context);

      context->bridge->priv->windows = g_list_remove (context->bridge->priv->windows, context);

      g_free (context);
    }
}

static void
app_menu_bridge_class_init (AppMenuBridgeClass *class)
{
  UbuntuMenuProxyClass *proxy_class;
  GObjectClass *object_class;

  app_menu_bridge_parent_class = g_type_class_peek_parent (class);

  proxy_class = UBUNTU_MENU_PROXY_CLASS (class);
  object_class = G_OBJECT_CLASS (class);

  proxy_class->insert = app_menu_bridge_insert;
  proxy_class->show_local = app_menu_bridge_show_local;

  object_class->dispose  = app_menu_bridge_dispose;
  object_class->finalize = app_menu_bridge_finalize;

  g_type_class_add_private (class, sizeof (AppMenuBridgePrivate));
}

static void
app_menu_bridge_class_finalize (AppMenuBridgeClass *class)
{
}

static void
app_menu_bridge_set_show_local (AppMenuBridge *bridge,
                                gboolean       local)
{
  const gchar *env = g_getenv ("APPMENU_DISPLAY_BOTH");

  if (g_strcmp0 (env, "1") == 0)
    local = TRUE;

  g_object_set (bridge,
                "show-local", local,
                NULL);
}

static void
register_application_window_cb (DBusGProxy     *proxy,
                                GError         *error,
                                void           *user_data)
{
  AppWindowContext *context = (AppWindowContext *)user_data;

  if (error != NULL)
    {
      if (context->bridge != NULL)
        {
          context->registered = FALSE;

          app_menu_bridge_set_show_local (context->bridge, TRUE);

          return;
        }
    }

  context->registered = TRUE;

  if (context->bridge != NULL)
    app_menu_bridge_set_show_local (context->bridge, FALSE);
}

static void
register_application_windows (AppMenuBridge *bridge)
{
  GList *tmp = NULL;

  for (tmp = bridge->priv->windows; tmp != NULL; tmp = tmp->next)
    {
      AppWindowContext *context = tmp->data;
      GtkWidget *widget = context->window;

      if (bridge->priv->appmenuproxy == NULL || bridge->priv->online == FALSE)
        {
          if (context->bridge != NULL)
            {
              app_menu_bridge_set_show_local (context->bridge, TRUE);
            }
        }

      if (!context->registered && context->server != NULL && context->root != NULL && GTK_IS_WINDOW (widget) && bridge->priv->appmenuproxy != NULL)
        {
          org_ayatana_AppMenu_Registrar_register_window_async (bridge->priv->appmenuproxy,
                                                               GDK_WINDOW_XID (gtk_widget_get_window (widget)),
                                                               context->path,
                                                               register_application_window_cb,
                                                               context);
        }
    }
}

static void
unregister_application_windows (AppMenuBridge *bridge)
{
  GList *tmp = NULL;

  for (tmp = bridge->priv->windows; tmp != NULL; tmp = tmp->next)
    {
      AppWindowContext *context = tmp->data;

      context->registered = FALSE;
    }

  app_menu_bridge_set_show_local (bridge, TRUE);
}

static void
app_menu_bridge_proxy_vanished (AppMenuBridge *bridge)
{
  bridge->priv->online = FALSE;

  unregister_application_windows (bridge);
}

static void
app_menu_bridge_proxy_appeared (AppMenuBridge *bridge)
{
  bridge->priv->online = TRUE;

  app_menu_bridge_setup_proxy (bridge);

  register_application_windows (bridge);
}

static void
dbus_owner_change (DBusGProxy    *proxy,
                   const gchar   *name,
                   const gchar   *prev,
                   const gchar   *new,
                   AppMenuBridge *bridge)
{
  if (strlen (new) > 0 && strlen (prev) == 0)
    {
      if (g_strcmp0 (name, APP_MENU_DBUS_NAME) == 0 && (prev == NULL || strlen (prev) == 0))
        {
          app_menu_bridge_proxy_appeared (bridge);
        }
    }
  else if (g_strcmp0 (name, APP_MENU_DBUS_NAME) == 0 && strlen (new) == 0 && strlen (prev) > 0)
    {
      app_menu_bridge_proxy_vanished (bridge);
    }
}

static GtkWidget *
find_menu_label (GtkWidget *widget)
{
  GtkWidget *label = NULL;

  if (GTK_IS_LABEL (widget))
    return widget;

  if (GTK_IS_CONTAINER (widget))
    {
      GList *children;
      GList *l;

      children = gtk_container_get_children (GTK_CONTAINER (widget));

      for (l = children; l; l = l->next)
        {
          label = find_menu_label (l->data);

          if (label)
            break;
        }

      g_list_free (children);
    }

  return label;
}

static const gchar *
get_menu_label_text (GtkWidget *menuitem)
{
  GtkWidget *label = find_menu_label (menuitem);

  if (label)
    return gtk_label_get_text (GTK_LABEL (label));

  return NULL;
}

static void
item_activated (DbusmenuMenuitem *item, guint timestamp, gpointer user_data)
{
  GtkWidget *child;

  if (user_data != NULL)
    {
      child = (GtkWidget *)user_data;

      if (GTK_IS_MENU_ITEM (child))
        {
          gtk_menu_item_activate (GTK_MENU_ITEM (child));
        }
    }
}

static gboolean
should_show_image (GtkImage *image)
{
  GtkWidget *item;

  item = gtk_widget_get_ancestor (GTK_WIDGET (image),
                                  GTK_TYPE_IMAGE_MENU_ITEM);

  if (item)
    {
      GtkSettings *settings;
      gboolean gtk_menu_images;

      settings = gtk_widget_get_settings (item);

      g_object_get (settings, "gtk-menu-images", &gtk_menu_images, NULL);

      if (gtk_menu_images)
        return TRUE;

      return gtk_image_menu_item_get_always_show_image (GTK_IMAGE_MENU_ITEM (item));
    }

  return FALSE;
}

static gboolean
update_stock_item (DbusmenuMenuitem *menuitem,
                   GtkWidget        *widget)
{
  GtkStockItem stock;
  GtkImage *image;

  g_return_val_if_fail (GTK_IS_IMAGE (widget), FALSE);

  image = GTK_IMAGE (widget);

  if (gtk_image_get_storage_type (image) != GTK_IMAGE_STOCK)
    return FALSE;

  gtk_stock_lookup (image->data.stock.stock_id, &stock);

  if (should_show_image (image))
    dbusmenu_menuitem_property_set (menuitem,
                                    DBUSMENU_MENUITEM_PROP_ICON_NAME,
                                    image->data.stock.stock_id);
  else
    dbusmenu_menuitem_property_remove (menuitem,
                                       DBUSMENU_MENUITEM_PROP_ICON_NAME);

  const gchar *label = dbusmenu_menuitem_property_get (menuitem,
                                                       DBUSMENU_MENUITEM_PROP_LABEL);

  if (stock.label != NULL && label != NULL)
    {
      dbusmenu_menuitem_property_set (menuitem,
                                      DBUSMENU_MENUITEM_PROP_LABEL,
                                      stock.label);

      return TRUE;
    }

  return FALSE;
}

static void
update_icon_name (DbusmenuMenuitem *menuitem,
                  GtkWidget        *widget)
{
  GtkImage *image;

  g_return_if_fail (GTK_IS_IMAGE (widget));

  image = GTK_IMAGE (widget);

  if (gtk_image_get_storage_type (image) != GTK_IMAGE_ICON_NAME)
    return;

  if (should_show_image (image))
    dbusmenu_menuitem_property_set (menuitem,
                                    DBUSMENU_MENUITEM_PROP_ICON_NAME,
                                    image->data.name.icon_name);
  else
    dbusmenu_menuitem_property_remove (menuitem,
                                       DBUSMENU_MENUITEM_PROP_ICON_NAME);
}

static void
widget_notify_cb (GtkWidget  *widget,
                  GParamSpec *pspec,
                  gpointer    data)
{
  DbusmenuMenuitem *child = (DbusmenuMenuitem *)data;

  if (pspec->name == g_intern_static_string ("sensitive"))
    {
      dbusmenu_menuitem_property_set_bool (child,
                                           DBUSMENU_MENUITEM_PROP_ENABLED,
                                           gtk_widget_get_sensitive (widget));
    }
  else if (pspec->name == g_intern_static_string ("label"))
    {
      dbusmenu_menuitem_property_set (child,
                                      DBUSMENU_MENUITEM_PROP_LABEL,
                                      gtk_menu_item_get_label (GTK_MENU_ITEM (widget)));
    }
  else if (pspec->name == g_intern_static_string ("visible"))
    {
      dbusmenu_menuitem_property_set_bool (child,
                                           DBUSMENU_MENUITEM_PROP_VISIBLE,
                                           gtk_widget_get_visible (widget));
    }
  else if (pspec->name == g_intern_static_string ("stock"))
    {
      update_stock_item (child, widget);
    }
  else if (pspec->name == g_intern_static_string ("icon-name"))
    {
      update_icon_name (child, widget);
    }
}

static void
menuitem_notify_cb (GtkWidget  *widget,
                    GParamSpec *pspec,
                    gpointer    data)
{
  if (pspec->name == g_intern_static_string ("visible"))
    {
      AppWindowContext *context = (AppWindowContext *)data;
      GtkWidget *toplevel = gtk_widget_get_toplevel (widget);
      GtkWidget *window = NULL;

      if (context != NULL)
        window = context->window;

      if (toplevel == window)
        {
          rebuild (context->bridge, window);
        }

      /* We only care about this once, so let's disconnect now. */
      g_signal_handlers_disconnect_by_func (widget,
                                            G_CALLBACK (menuitem_notify_cb),
                                            data);
    }
}

static void
checkbox_toggled (GtkWidget *widget, DbusmenuMenuitem *mi)
{
  dbusmenu_menuitem_property_set_int (mi,
                                      DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
                                      gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)) ? DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED : DBUSMENU_MENUITEM_TOGGLE_STATE_UNCHECKED);
}

static void
accel_changed (GtkWidget *widget,
               gpointer   data)
{
  DbusmenuMenuitem *mi = (DbusmenuMenuitem *)data;
  dbusmenu_menuitem_property_set_shortcut_menuitem (mi, GTK_MENU_ITEM (widget));
}

static void
action_notify_cb (GtkAction  *action,
                  GParamSpec *pspec,
                  gpointer    data)
{
  DbusmenuMenuitem *mi = (DbusmenuMenuitem *)data;

  if (pspec->name == g_intern_static_string ("active"))
    {
      dbusmenu_menuitem_property_set_bool (mi,
                                           DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
                                           gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action)));
    }
  else if (pspec->name == g_intern_static_string ("label"))
    {
      dbusmenu_menuitem_property_set (mi,
                                      DBUSMENU_MENUITEM_PROP_LABEL,
                                      gtk_action_get_label (action));
    }
}

static DbusmenuMenuitem *
construct_dbusmenu_for_widget (GtkWidget *widget, gboolean previous_separator)
{
  DbusmenuMenuitem *mi = dbusmenu_menuitem_new ();

  if (GTK_IS_MENU_ITEM (widget))
    {
      if (GTK_IS_SEPARATOR_MENU_ITEM (widget))
        {
          dbusmenu_menuitem_property_set (mi,
                                          "type",
                                          "separator");

          gint mode = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (widget), "gtk-separator-mode"));

          dbusmenu_menuitem_property_set_bool (mi,
                                               DBUSMENU_MENUITEM_PROP_VISIBLE,
                                               mode == SEPARATOR_MODE_SMART && !previous_separator);
        }
      else
        {
          gboolean visible = FALSE;
          gboolean label_set = FALSE;

          g_signal_connect (widget,
                            "accel-closures-changed",
                            G_CALLBACK (accel_changed),
                            mi);

          if (GTK_IS_CHECK_MENU_ITEM (widget))
            {
              dbusmenu_menuitem_property_set (mi,
                                              DBUSMENU_MENUITEM_PROP_TOGGLE_TYPE,
                                              gtk_check_menu_item_get_draw_as_radio (GTK_CHECK_MENU_ITEM (widget)) ? DBUSMENU_MENUITEM_TOGGLE_RADIO : DBUSMENU_MENUITEM_TOGGLE_CHECK);

              dbusmenu_menuitem_property_set_int (mi,
                                                  DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
                                                  gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)) ? DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED : DBUSMENU_MENUITEM_TOGGLE_STATE_UNCHECKED);

              g_signal_connect (widget,
                                "toggle",
                                G_CALLBACK (checkbox_toggled),
                                mi);
            }

          if (GTK_IS_IMAGE_MENU_ITEM (widget))
            {
              GtkWidget *image;
              GtkImageType image_type;

              image = gtk_image_menu_item_get_image (GTK_IMAGE_MENU_ITEM (widget));

              if (GTK_IS_IMAGE (image))
                {
                  image_type = gtk_image_get_storage_type (GTK_IMAGE (image));

                  if (image_type == GTK_IMAGE_STOCK)
                    {
                      label_set = update_stock_item (mi, image);
                    }
                  else if (image_type == GTK_IMAGE_ICON_NAME)
                    {
                      update_icon_name (mi, image);
                    }
                  else if (image_type == GTK_IMAGE_PIXBUF)
                    {
                      dbusmenu_menuitem_property_set_image (mi,
                                                            DBUSMENU_MENUITEM_PROP_ICON_DATA,
                                                            gtk_image_get_pixbuf (GTK_IMAGE (image)));
                    }
                }
            }

          dbusmenu_menuitem_property_set (mi,
                                          "label",
                                          get_menu_label_text (widget));

          if (!g_object_get_data (G_OBJECT (widget), "gtk-empty-menu-item") && !GTK_IS_TEAROFF_MENU_ITEM (widget))
            {
              visible = gtk_widget_get_visible (widget);
            }

          dbusmenu_menuitem_property_set_bool (mi,
                                               DBUSMENU_MENUITEM_PROP_VISIBLE,
                                               visible);

          dbusmenu_menuitem_property_set_bool (mi,
                                               DBUSMENU_MENUITEM_PROP_ENABLED,
                                               gtk_widget_get_sensitive (widget));

          dbusmenu_menuitem_property_set_shortcut_menuitem (mi, GTK_MENU_ITEM (widget));

          g_signal_connect (G_OBJECT (widget),
                            "notify",
                            G_CALLBACK (widget_notify_cb),
                            mi);

          if (GTK_IS_ACTIVATABLE (widget))
            {
              GtkActivatable *activatable = GTK_ACTIVATABLE (widget);

              if (gtk_activatable_get_use_action_appearance (activatable))
                {
                  GtkAction *action = gtk_activatable_get_related_action (activatable);

                  if (action)
                    {
                      g_signal_connect_object (action, "notify",
                                               G_CALLBACK (action_notify_cb),
                                               mi,
                                               G_CONNECT_AFTER);
                    }
                }
            }

          g_signal_connect (G_OBJECT (mi),
                            DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
                            G_CALLBACK (item_activated),
                            widget);
        }
    }

  return mi;
}

static void
rebuild_item (GtkWidget      *widget,
              RecurseContext *recurse)
{
  if (GTK_IS_CONTAINER (widget))
    {
      gboolean increment = GTK_IS_MENU_BAR (widget) || GTK_IS_MENU_ITEM (widget);
      gboolean previous_separator = FALSE;

      if (increment)
        recurse->count++;

      /* Okay, this is a little janky and all.. but some applications update some
       * menuitem properties such as sensitivity on the activate callback.  This
       * seems a little weird, but it's not our place to judge when all this code
       * is so crazy.  So we're going to get ever crazier and activate all the
       * menus that are directly below the menubar and force the applications to
       * update their sensitivity.  The menus won't actually popup in the app
       * window due to our gtk+ patches.
       *
       * Note that this will not force menuitems in submenus to be updated as well.
       */
      if (recurse->count == 0 && GTK_IS_MENU_BAR (widget))
        {
          GList *children = gtk_container_get_children (GTK_CONTAINER (widget));

          for (; children != NULL; children = children->next)
            {
              gtk_menu_shell_activate_item (GTK_MENU_SHELL (widget),
                                            children->data,
                                            TRUE);
            }

          g_list_free (children);
        }

      if (recurse->count > -1 && increment)
        {
          /* If this is a separator, find out if we've already displayed a visible separator during
           * this run.  GtkUIManager internally defines the following separator modes:
           *
           * SEPARATOR_MODE_SMART
           * SEPARATOR_MODE_VISIBLE
           * SEPARATOR_MODE_HIDDEN
           *
           * construct_dbusmenu_for_widget() will mark a smart separator as visible in a run of
           * separators unless it is following another smart separator anywhere in that run.
           */
          if (GTK_IS_SEPARATOR_MENU_ITEM (widget))
            {
              if (recurse->stack[recurse->count] != NULL)
                {
                  const gchar *type = dbusmenu_menuitem_property_get (recurse->stack[recurse->count],
                                                                      DBUSMENU_MENUITEM_PROP_TYPE);

                  if (g_strcmp0 (type, "separator") == 0)
                    {
                      /* Get the previous separator mode. */
                      gint mode = recurse->mode[recurse->count];

                      if (mode == SEPARATOR_MODE_SMART)
                        previous_separator = TRUE;
                    }
                }
            }

          DbusmenuMenuitem *dmi = g_hash_table_lookup (recurse->context->lookup, widget);
          if (dmi != NULL)
            {
              if (increment)
                recurse->count--;

              return;
            }
          else
            {
              recurse->stack[recurse->count] = construct_dbusmenu_for_widget (widget, previous_separator);
              g_hash_table_insert (recurse->context->lookup, widget, recurse->stack[recurse->count]);
            }

          if (GTK_IS_SEPARATOR_MENU_ITEM (widget))
            {
              /* If the previous separator was mode 1, let's pretend that this separator is also mode 1.
               * That means for the remainder of this run of separators, all will be marked as mode 1.
               */
              if (previous_separator)
                {
                  recurse->mode[recurse->count] = SEPARATOR_MODE_SMART;
                }
              else
                {
                  recurse->mode[recurse->count] = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (widget), "gtk-separator-mode"));
                }
            }

          if (!gtk_widget_get_visible (widget))
            {
              g_signal_connect (G_OBJECT (widget),
                                "notify",
                                G_CALLBACK (menuitem_notify_cb),
                                recurse->context);
            }

          if (GTK_IS_TEAROFF_MENU_ITEM (widget))
            {
              dbusmenu_menuitem_property_set_bool (recurse->stack[recurse->count],
                                                   DBUSMENU_MENUITEM_PROP_VISIBLE,
                                                   FALSE);
            }

          if (recurse->count > 0)
            {
              GList *children = NULL;
              GList *peek = NULL;

              if (recurse->stack[recurse->count - 1])
                {
                  children = dbusmenu_menuitem_get_children (recurse->stack[recurse->count - 1]);

                  if (children)
                    {
                      peek = g_list_find (children, recurse->stack[recurse->count]);
                    }

                  if (!peek)
                    {
                      dbusmenu_menuitem_child_append (recurse->stack[recurse->count - 1],
                                                      recurse->stack[recurse->count]);
                    }
                }
              else
                {
                  DbusmenuMenuitem *item = g_hash_table_lookup (recurse->context->lookup,
                                                                gtk_widget_get_parent (widget));

                  if (item)
                    {
                      children = dbusmenu_menuitem_get_children (item);

                      if (children)
                        {
                          peek = g_list_find (children, recurse->stack[recurse->count]);
                        }

                      if (!peek)
                        {
                          dbusmenu_menuitem_child_append (item, recurse->stack[recurse->count]);
                        }
                    }
                }
            }
        }

      gtk_container_foreach (GTK_CONTAINER (widget),
                             (GtkCallback)rebuild_item,
                             recurse);

      if (GTK_IS_MENU_ITEM (widget))
        {
          GtkWidget *menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (widget));

          if (menu != NULL)
            {
              rebuild_item (menu, recurse);
            }
        }

      if (increment)
        recurse->count--;
    }
}

typedef struct _RebuildData {
  AppMenuBridge *bridge;
  GtkWidget     *widget;
} RebuildData;

static gboolean
do_rebuild (RebuildData *data)
{
  if (gtk_widget_is_toplevel (data->widget))
    {
      rebuild_window_items (data->bridge,
                            data->widget);

      g_hash_table_remove (rebuild_ids, data->widget);
    }

  g_free (data);

  return FALSE;
}

static void
rebuild (AppMenuBridge *bridge,
         GtkWidget     *toplevel)
{
  guint id = 0;

  if (rebuild_ids != NULL)
    {
      id = GPOINTER_TO_UINT (g_hash_table_lookup (rebuild_ids, toplevel));

      if (id > 0)
        {
          g_source_remove (id);
          g_hash_table_remove (rebuild_ids, toplevel);
          id = 0;
        }
    }

  RebuildData *data = g_new0 (RebuildData, 1);
  data->bridge = bridge;
  data->widget = toplevel;

  id = g_timeout_add (100,
                      (GSourceFunc)do_rebuild,
                      data);

  if (rebuild_ids == NULL)
    {
      rebuild_ids = g_hash_table_new (g_direct_hash, g_direct_equal);
    }

  g_hash_table_insert (rebuild_ids, toplevel, GUINT_TO_POINTER (id));
}

static void
rebuild_window_items (AppMenuBridge *bridge,
                      GtkWidget     *toplevel)
{
  XID xid;
  AppWindowContext *context = NULL;
  RecurseContext recurse;

  memset (&recurse, 0, sizeof (RecurseContext));

  if (!GTK_IS_WINDOW (toplevel))
    {
      g_signal_connect (G_OBJECT (toplevel),
                        "notify",
                        G_CALLBACK (toplevel_notify_cb),
                        bridge);

      return;
    }
  else if (g_object_class_find_property (G_OBJECT_GET_CLASS (toplevel),
                                         "ubuntu-no-proxy") != NULL)

    {
      gboolean no_proxy;

      g_object_get (G_OBJECT (toplevel),
                    "ubuntu-no-proxy", &no_proxy,
                    NULL);

      if (no_proxy)
        {
          return;
        }
    }

  if (!gtk_widget_get_realized (toplevel))
    {
      g_signal_connect (toplevel, "realize",
                        G_CALLBACK (toplevel_realized),
                        bridge);

      return;
    }

  xid = GDK_WINDOW_XID (gtk_widget_get_window (toplevel));

  GList *tmp;
  gboolean found = FALSE;

  for (tmp = bridge->priv->windows; tmp != NULL; tmp = tmp->next)
    {
      context = tmp->data;

      if (context && GTK_IS_WIDGET (context->window))
        {
          XID xid2 = GDK_WINDOW_XID (gtk_widget_get_window (context->window));

          if (xid == xid2)
            {
              found = TRUE;
              break;
            }
        }
    }

  if (!found)
    {
      context = g_new0 (AppWindowContext, 1);
      context->bridge = bridge;
      context->lookup = g_hash_table_new (g_direct_hash, g_direct_equal);
      bridge->priv->windows = g_list_prepend (bridge->priv->windows, context);

      recurse.previous = FALSE;
      recurse.count    = -1;
    }
  else
    {
      recurse.previous = TRUE;
      recurse.count    = 0;
    }

  if (context->window)
    {
      if (context->window != toplevel)
        {
          g_signal_handlers_disconnect_by_func (context->window,
                                                G_CALLBACK (toplevel_destroyed),
                                                context);
        }
    }

  if (context->window != toplevel)
    {
      context->window = toplevel;

      g_signal_connect (toplevel,
                        "destroy",
                        G_CALLBACK (toplevel_destroyed),
                        context);
    }

  if (!context->path)
    context->path = g_strdup_printf ("/org/ayatana/menu/%X", (guint)xid);

  if (!context->server)
    context->server = dbusmenu_server_new (context->path);

  recurse.bridge  = bridge;
  recurse.context = context;

  gtk_container_foreach (GTK_CONTAINER (toplevel),
                         (GtkCallback)rebuild_item,
                         &recurse);

  if (recurse.stack[0] != NULL && DBUSMENU_IS_MENUITEM (recurse.stack[0]))
    {
      context->root = recurse.stack[0];

      dbusmenu_server_set_root (context->server, context->root);
    }

  register_application_windows (bridge);
}


static void
toplevel_realized (GtkWidget *widget,
                   gpointer   user_data)
{
  AppMenuBridge *bridge = APP_MENU_BRIDGE (user_data);

  if (GTK_IS_WINDOW (widget))
    {
      rebuild (bridge, widget);
      //register_application_windows (bridge);

      return;
    }
}

static void
toplevel_notify_cb (GtkWidget       *widget,
                    GParamSpec      *pspec,
                    UbuntuMenuProxy *proxy)
{
  if (pspec->name == g_intern_static_string ("parent"))
    {
      GtkWidget *toplevel = gtk_widget_get_toplevel (widget);
      AppMenuBridge *bridge = APP_MENU_BRIDGE (proxy);

      if (gtk_widget_get_parent (widget) == NULL)
        return;

      if (GTK_IS_WINDOW (toplevel))
        {
          g_signal_handlers_disconnect_by_func (widget,
                                                G_CALLBACK (toplevel_notify_cb),
                                                proxy);
          rebuild (bridge, toplevel);
        }
      else
        {
          g_signal_connect (G_OBJECT (toplevel),
                            "notify",
                            G_CALLBACK (toplevel_notify_cb),
                            proxy);
        }
    }
}

static void
attach_notify_cb (GtkWidget     *widget,
                  GParamSpec    *pspec,
                  AppMenuBridge *bridge)
{
  if (pspec->name == g_intern_static_string ("attach-widget"))
    {
      GtkWidget *attach = NULL;

      g_object_get (widget, "attach-widget", &attach, NULL);

      rebuild (bridge, attach);
    }
}

static void
app_menu_bridge_insert (UbuntuMenuProxy *proxy,
                        GtkWidget       *parent,
                        GtkWidget       *child,
                        guint            position)
{
  AppMenuBridge        *bridge;
  AppMenuBridgePrivate *priv;
  GtkWidget            *toplevel = NULL;
  gboolean              append = FALSE;

  if (GTK_IS_TEAROFF_MENU_ITEM (child))
    return;

  bridge = APP_MENU_BRIDGE (proxy);
  priv = bridge->priv;

  toplevel = gtk_widget_get_toplevel (parent);

  if (GTK_IS_MENU_BAR (parent))
    {
      g_signal_connect (G_OBJECT (toplevel),
                        "notify",
                        G_CALLBACK (toplevel_notify_cb),
                        proxy);

      append = TRUE;
    }
  else if (GTK_IS_MENU (parent))
    {
      GtkWidget *attach;

      g_object_get (parent, "attach-widget", &attach, NULL);

      if (attach == NULL)
        {
          g_signal_connect (G_OBJECT (parent),
                            "notify",
                            G_CALLBACK (attach_notify_cb),
                            bridge);
          return;
        }
      else
        {
          toplevel = gtk_widget_get_toplevel (attach);

          rebuild (bridge, toplevel);
        }
    }

  if (GTK_IS_WINDOW (toplevel))
    {
      g_signal_connect (toplevel, "realize",
                        G_CALLBACK (toplevel_realized),
                        bridge);
    }
}

static gboolean
app_menu_bridge_show_local (UbuntuMenuProxy *proxy)
{
  gboolean local;

  g_object_get (proxy,
                "show-local", &local,
                NULL);

  return local;
}

/* crude blacklist to avoid patching innoncent apps */
static gboolean
app_menu_brige_shouldnt_load (void)
{
  const char *prg = g_get_prgname ();

  if ((g_strrstr (prg, "indicator-applet") != NULL)
      || (g_strcmp0 (prg, "indicator-loader") == 0)
      || (g_strcmp0 (prg, "mutter") == 0)
      || (g_strcmp0 (prg, "firefox-bin") == 0)
      || (g_strcmp0 (prg, "thunderbird-bin") == 0)
      || (g_strcmp0 (prg, "gnome-panel") == 0))
    {
      return TRUE;
    }

  return FALSE;
}


G_MODULE_EXPORT void
menu_proxy_module_load (UbuntuMenuProxyModule *module)
{
  static gboolean registered = FALSE;

  /* Prevent well-known applications to re-export
	 their own dbusmenus */
  if (app_menu_brige_shouldnt_load ())
	  return;

  if (!registered)
    {
      app_menu_bridge_register_type (G_TYPE_MODULE (module));

      registered = TRUE;
    }
}

G_MODULE_EXPORT void
menu_proxy_module_unload (UbuntuMenuProxyModule *module)
{
  if (rebuild_ids)
    {
      g_hash_table_destroy (rebuild_ids);
      rebuild_ids = NULL;
    }
}
