

#define LOCAL_DEBUG
#include "debug.h"
// doesn't work here without logging, always ignore
#ifdef ldbg
#undef ldbg
#endif

#include "acfg.h"
#include "filereader.h"
#include "rechecks.h"

#ifdef HAVE_WORDEXP
#include <wordexp.h>
#elif defined(HAVE_GLOB)
#include <glob.h>
#endif

#include <errno.h>

#include <iostream>
#include <fstream>
#include <string>
#include <meta.h>
#include <list>
#include <map>
#include <algorithm>

#include "acfg_defaults.h"

using namespace MYSTD;

namespace acfg {

// internal stuff:
char alphabet[65] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
string sPopularPath("/debian/");

MapNameToString n2sTbl[] = {
		{  "Port", 			&port , 0}
		,{  "CacheDir", 	&cachedir, 0 }
		,{ "LogDir", 	&logdir , 0}
		,{ "SocketPath", 	&fifopath, 0}
		,{ "PidFile", 	&pidfile, 0}
		,{ "Proxy",		&proxy, 0}
		,{ "ReportPage",&reportpage, 0}
		,{ "VfilePattern", &vfilepat, 0}
		,{ "PfilePattern", &pfilepat, 0}
		,{ "WfilePattern", &wfilepat, 0}
		,{ "AdminAuth",  &adminauth, 0}
		,{ "BindAddress", &bindaddr, 0}
		,{ "UserAgent", &agentname, 0}
		//,{ "Umask",		  &sUmask}
};

MapNameToInt n2iTbl[] = {
		{ "Debug", 		&debug, 0 }
		,{ "OfflineMode", 	&offlinemode , 0}
		,{ "ForeGround", 	&foreground , 0}
		,{ "Verbose", 		0, "Option is deprecated, ignoring the value." }
		,{ "ForceManaged", 	&forcemanaged , 0}
		,{ "StupidFs", 		&stupidfs , 0}
		,{ "VerboseLog",	&verboselog , 0}
		,{ "ExTreshold",	&extreshhold, 0}
		,{ "MaxSpareThreadSets",	&tpstandbymax, "Deprecated option name, mapped to MaxStandbyConThreads"}
		,{ "MaxStandbyConThreads",	&tpstandbymax, 0}
		,{ "MaxConThreads",	&tpthreadmax, 0}
		,{ "DnsCacheSeconds", &dnscachetime, 0}
		,{ "UnbufferLogs", &debug , 0}
		,{ "ExAbortOnProblems", &exfailabort, 0}
		,{ "ExposeOrigin", &exporigin, 0}
		,{ "LogSubmittedOrigin", &logxff, 0}
		,{ "OldIndexUpdater", &oldupdate, "Option is deprecated, ignoring the value." }
		,{ "RecompBz2", &recompbz2, 0}
		,{ "NetworkTimeout", &nettimeout, 0}
		,{ "MinUpdateInterval", &updinterval, 0}
		,{ "ForwardBtsSoap", &forwardsoap, 0}
};

#define _iterPos(it, start) (it-start.begin())/sizeof(it)
#define sProblemLoc szPath<< ':'<< _iterPos(it, lines)

string sFilterSet(SPACECHARS "#");
#define IsValidButIrrelevantLine(x) (x.empty() || stmiss != sFilterSet.find(x[0]))
#define BARF(x) { cerr << x << endl; exit(EXIT_FAILURE); }

void _ReadRewriteFiles(const string & sFile, const string & sRepName);
void _ReadBackendsFiles(const string & sFile, const string &sRepName);


struct fct_lt_host
{
  bool operator()(const tHttpUrl &a, const tHttpUrl &b) const
  {
    return strcasecmp(a.sHost.c_str(), b.sHost.c_str()) < 0;
  }
};
typedef multimap<tHttpUrl,const string*, fct_lt_host> tMapUrl2StringPtr; 
typedef tMapUrl2StringPtr::iterator tUrl2RepIter;
tMapUrl2StringPtr mapUrl2pVname;

typedef map<const string, tHostiVec> tMapString2Hostivec;
tMapString2Hostivec mapRepName2Backends;

string * _GetStringPtr(const string &key)  {
	for(unsigned int i=0; i<_countof(n2sTbl); i++) {
		if(0==strcasecmp(key.c_str(), n2sTbl[i].name))
		{
			if(n2sTbl[i].warn)
				cerr << "Warning, " << key << ": " << n2sTbl[i].warn << endl; 
						
			return n2sTbl[i].ptr;
		}
	}
	return NULL;
}

int * _GetIntPtr(const string &key)  {
	for(unsigned int i=0; i<_countof(n2iTbl); i++) {
		if(0==strcasecmp(key.c_str(), n2iTbl[i].name))
		{
			if(n2iTbl[i].warn)
				cerr << "Warning, " << key << ": " << n2iTbl[i].warn << endl;
			
			return n2iTbl[i].ptr;
		}
	}
	return NULL;
}

inline void _FixPostPreSlashes(string &val)
{
	// fix broken entries

	if (val.empty() || val.at(val.length()-1) != '/')
		val.append("/");
	if (val.at(0) != '/')
		val.insert(0, "/", 1);
}

bool _ReadMainConfiguration(const char * szFilename)
{
	filereader reader;
	reader.OpenFile(szFilename);
	reader.CheckGoodState(true);
	string sLine, key, val;
	for (bool bNotEof=true; bNotEof;)
	{
		bNotEof=reader.GetOneLine(sLine);
		if (IsValidButIrrelevantLine(sLine))
			continue;
		if(! SetOption(sLine))
			BARF("Error reading main options, terminating.");
		// XXX: move that to SetOption... if (debug>4)
		//	cout << key << " -> "<< val <<endl;
	}
	return true;
}

const string * _CheckBEentryGetNamePtr(const string & sRepName)
{
	// needs a reliably stored string for the pointer. Backend description may not exist,
	// then create a dummy one with no contents and point at this string. Iterators on map are
	// guaranteed to be stable so point at its key variable.
	tMapString2Hostivec::iterator
			itHostiVec = mapRepName2Backends.find(sRepName);
	if (itHostiVec == mapRepName2Backends.end())
	{
		mapRepName2Backends[sRepName]=tHostiVec(0);
		itHostiVec=mapRepName2Backends.find(sRepName); // must refer to that string on the heap
		if (debug>4)
			cout << "created empty backend entry for "<< sRepName <<endl;
	}
	return & itHostiVec->first;
}

inline void _AddRemapInfo(bool bAsBackend, const string & token,
		const string &repname)
{
	if (0!=token.compare(0, 5, "file:"))
	{
		tHttpUrl url;
		if(! url.SetHttpUrl(token))
			BARF(token + " <-- bad URL detected");
		_FixPostPreSlashes(url.sPath);
		
		if (bAsBackend)
			mapRepName2Backends[repname].push_back(url);
		else
			mapUrl2pVname.insert(pair<tHttpUrl,const string*>(
					url, _CheckBEentryGetNamePtr(repname)));
	}
	else
	{
		string sPath=token.substr(5);
		if (sPath.empty())
			BARF("Bad file spec for repname, file:?");
		
		if (sPath[0]!=cPathSep)
			sPath.insert(0, confdir+sPathSep);
		
		tStrVec srcs;

#ifdef HAVE_WORDEXP
		wordexp_t p;
		memset(&p, 0, sizeof(p));
		if(0==wordexp(sPath.c_str(), &p, 0))
		{
			for (UINT i=0; i<p.we_wordc; i++)
				srcs.push_back(p.we_wordv[i]);
			wordfree(&p);
		}
		
#elif defined HAVE_GLOB
		glob_t globbuf;
		memset(&globbuf, 0, sizeof(glob_t));
		if(0==glob(sPath.c_str(), GLOB_DOOFFS | GLOB_NOSORT, NULL, &globbuf))
		{
			for (UINT i=0; i<globbuf.gl_pathc; i++)
				srcs.push_back(globbuf.gl_pathv[i]);
			globfree(&globbuf);
		}
#endif
		std::sort(srcs.begin(), srcs.end());
		for(tStrVecIterConst it=srcs.begin(); it!=srcs.end(); it++)
		{
			if (bAsBackend)
				_ReadBackendsFiles(*it, repname);
			else
				_ReadRewriteFiles(*it, repname);
		}
	}
}

bool SetOption(const MYSTD::string &sLine, bool bQuiet)
{
	string::size_type pos = sLine.find(":");
	if (pos==stmiss)
		pos = sLine.find("=");
	if (pos==stmiss)
	{
		if(!bQuiet)
			cerr << "Not a valid configuration directive: " << sLine <<endl;
		return false;
	}
	string key=sLine.substr(0, pos);
	string value=sLine.substr(pos+1);
	trimString(key);
	trimString(value);
	if(key.empty())
		return false; // weird
	
	string * psTarget;
	int * pnTarget;
	if ( NULL != (psTarget = _GetStringPtr(key)))
		*psTarget=value;
	else if ( NULL != (pnTarget = _GetIntPtr(key)))
	{
		const char *pStart=value.c_str();
		if(! *pStart)
		{
			cerr << "Missing value for " << key << " option!" <<endl;
			return false;
		}
		
		errno=0;
		char *pEnd(0);
		long nVal = strtol(pStart, &pEnd, 10);

		if(RESERVED_DEFVAL == nVal)
		{
			cerr << "Bad value for " << key << " (protected value, use another one)" <<endl;
			return false;
		}

		*pnTarget=nVal;

		if (errno)
		{
			cerr << "Invalid number for " << key << " ";
			perror("option");
			return false;
		}
		if(*pEnd)
		{
			cerr << "Bad value for " << key << " option or found trailing garbage: " << pEnd <<endl;
			return false;
		}
				
	}
	else if(0==strncasecmp(key.c_str(), "Remap-", 6))
	{
		string vname=key.substr(6, key.npos);
		tStrVec tokens;
		
		Tokenize(value, SPACECHARS, tokens);
		if(tokens.empty() || vname.empty())
		{
			if(!bQuiet)
				cerr << "Found invalid entry, ignoring " << key << ": " << value <<endl;
			return false;
		}
		bool bIsBackend=false;
		for(UINT i=0; i<tokens.size(); i++)
		{
			if(tokens[i]==";")
			{
				bIsBackend=true;
				continue;
			}
			_AddRemapInfo(bIsBackend, tokens[i], vname);
		}
		
	}
	else
	{
		if(!bQuiet)
			cerr << "Warning, unknown configuration directive: " << key <<endl;
		return false;
	}
	return true;
}


//const string * GetVnameForUrl(string path, string::size_type * nMatchLen)
const string * GetRepNameAndPathResidual(const tHttpUrl & in, MYSTD::string & sRetPathResidual)
{
	sRetPathResidual.clear();
	
	pair<tUrl2RepIter,tUrl2RepIter> range=mapUrl2pVname.equal_range(in);
	if(range.first==range.second)
		return NULL;
	
	tStrPos bestMatchLen(0);
	string const * psBestHit(NULL);
		
	for (tUrl2RepIter & it=range.first; it!=range.second; it++)
	{
		// rewrite rule path must be a real prefix
		// it's also surrounded by /, ensured during construction
		const string & prefix=it->first.sPath;
		tStrPos len=prefix.length();
		if (in.sPath.size() > len && 0==in.sPath.compare(0, len, prefix))
		{
			if (len>bestMatchLen)
			{
				bestMatchLen=len;
				psBestHit=it->second;
			}
		}
	}
		
	if(psBestHit) sRetPathResidual=in.sPath.substr(bestMatchLen);
	return psBestHit;
	
}

tHostiVec * GetBackendVec(const string * vname)
{
	if(!vname)
		return NULL;
	tMapString2Hostivec::iterator it=mapRepName2Backends.find(*vname);
	if(it==mapRepName2Backends.end() || it->second.empty())
		return NULL;
	return & it->second;
}


void _ReadBackendsFiles(const string & sFile, const string &sRepName)
{
	filereader reader;
	reader.OpenFile(sFile.c_str());
	
	if(debug>4)
		cout << "Reading backend file: " << sFile <<endl;
	if(!reader.CheckGoodState(false))
	{
		if(debug>4)
			cout << "No backend data found, file ignored."<<endl;
		return;
	}
	
	string sLine, key, val;
	tHttpUrl entry;
	
	for(bool bNotEof=true;bNotEof;) {
		
		bNotEof=reader.GetOneLine(sLine);
		//if(debug)
		//	cerr << "backends, got line: " << sLine <<endl;
		

		if( (startsWithSz(sLine, "http://") && entry.SetHttpUrl(sLine) )
				||
			(IsValidButIrrelevantLine(sLine) 
					&& ! entry.sHost.empty() 
					&& ! entry.sPath.empty()
			)
		)
		{
			_FixPostPreSlashes(entry.sPath);
#ifdef DEBUG
			cerr << "Backend: " << sRepName << " <-- " << entry.ToURI() <<endl;
#endif		

			mapRepName2Backends[sRepName].push_back(entry);
			entry.clear();
		}
		else if(_ParseLine(sLine, key, val))
		{
			if(keyEq("Site", key))
				entry.sHost=val;
			/* TODO: not supported yet, maybe add later - push to a vector of hosts and add multiple later
			if(keyEq("Aliases", key))
			{
				val+=" ";
				for(string::size_type posA(0), posB(0);
					posA<val.length();
					posA=posB+1)
				{
					posB=val.find_first_of(" \t\r\n\f\v", posA);
					if(posB!=posA)
						hosts.push_back(val.substr(posA, posB-posA));
				}
			}
			*/
			else if(keyEq("Archive-http", key) || keyEq("X-Archive-http", key))
			{
				entry.sPath=val;
			}
		}
		else if(bNotEof && !IsValidButIrrelevantLine(sLine))
		{
			cerr << "Bad backend description, around line "
					<< reader.GetPositionDescription() << endl;
			exit(2);
		}
	}
}

void ShutDown()
{
	mapUrl2pVname.clear();
	mapRepName2Backends.clear();
}

void _ReadRewriteFiles(const string & sFile, const string & sRepName)
{

	filereader reader;
	if(debug>4)
		cout << "Reading rewrite file: " << sFile <<endl;
	reader.OpenFile(sFile.c_str());
	reader.CheckGoodState(true);

	tStrVec hosts, paths;
	string sLine, key, val;
	tHttpUrl url;
	
	for(bool bNotEof=true; bNotEof; )
	{
		bNotEof=reader.GetOneLine(sLine);

		if (0==sLine.compare(0, 7, "http://"))
		{ // TODO: The check above is optional since SetHttpUrl does that too, but it's 
			// more userfriendly on errors (more exact error message)
			if (url.SetHttpUrl(sLine))
			{
				_FixPostPreSlashes(url.sPath);
				pair<tHttpUrl, const string*> info(url,
						_CheckBEentryGetNamePtr(sRepName));
				mapUrl2pVname.insert(info);
			}
			else
			{
				cout << "Parse error, invalid URL" << sLine << " on line "
						<< reader.GetPositionDescription() <<endl;
				exit(2);
			}
			continue;
		}
		else if (IsValidButIrrelevantLine(sLine)) // end of block, eof, ... -> commit it
		{
			if (hosts.empty() && paths.empty())
				continue; // dummy run
			if ( !hosts.empty() && paths.empty())
			{
				cerr << "Warning, missing path spec for the site " << hosts[0] <<", ignoring mirror."<< endl;
				continue;
			}
			if ( !paths.empty() && hosts.empty())
			{
				cout << "Parse error, missing Site: field around line "
						<< reader.GetPositionDescription() <<endl;
				exit(2);
			}
			for (tStrVec::const_iterator itHost=hosts.begin(); 
			itHost!=hosts.end(); 
			itHost++)
			{
				for (tStrVec::const_iterator itPath=paths.begin(); 
				itPath!=paths.end(); 
				itPath++)
				{
					//mapUrl2pVname[*itHost+*itPath]= &itHostiVec->first;
					tHttpUrl url;
					url.sHost=*itHost;
					url.sPath=*itPath;
					pair<tHttpUrl,const string*> info(url, _CheckBEentryGetNamePtr(sRepName));
					mapUrl2pVname.insert(info);

#ifdef DEBUG
						cout << "Mapping: "<< *itHost << *itPath 
						<< " -> "<< sRepName <<endl;
#endif
				}
			}
			hosts.clear();
			paths.clear();
			continue;
		}
		else if(!_ParseLine(sLine, key, val))
		{
			cerr << "Error parsing rewrite definitions, around line " << reader.GetPositionDescription() <<endl;
			exit(1);
		}
		
		// got something, intepret it...
		if( keyEq("Site", key) || keyEq("Alias", key) || keyEq("Aliases", key))
			Tokenize(val, SPACECHARS, hosts, true);
		
		if(keyEq("Archive-http", key) || keyEq("X-Archive-http", key))
		{
			// help STL saving some memory
			if(sPopularPath==val)
				paths.push_back(sPopularPath);
			else
			{
				_FixPostPreSlashes(val);
				paths.push_back(val);
			}
			continue;
		}
	}
}

void ReadConfigDirectory(const char *szPath)
{
	// TODO: early abort when the dir does not exist!

	char buf[PATH_MAX];
	realpath(szPath, buf);
	confdir=buf; // pickup the last config directory

#ifdef HAVE_WORDEXP
	wordexp_t p;
    wordexp((confdir+"/*.conf").c_str(), &p, 0);
    for (UINT i=0; i<p.we_wordc; i++)
    	_ReadMainConfiguration(p.we_wordv[i]);
    wordfree(&p);
#else
    _ReadMainConfiguration((confdir+cPathSep+"acng.conf").c_str());
#endif

}

string _GetBase64Auth(const string & sUserColonPass)
{
	int cols, bits, c, char_count;

	char_count = 0;
	bits = 0;
	cols = 0;
	tStrPos pos(0);
	MYSTD::string out;
	while ( pos<sUserColonPass.size())
	{
		c=sUserColonPass[pos++];
		bits += c;
		char_count++;
		if (char_count == 3)
		{
			out+=(alphabet[bits >> 18]);
			out+=(alphabet[(bits >> 12) & 0x3f]);
			out+=(alphabet[(bits >> 6) & 0x3f]);
			out+=(alphabet[bits & 0x3f]);
			cols += 4;
			bits = 0;
			char_count = 0;
		}
		else
		{
			bits <<= 8;
		}
	}
	if (char_count != 0)
	{
		bits <<= 16 - (8 * char_count);
		out+=(alphabet[bits >> 18]);
		out+=(alphabet[(bits >> 12) & 0x3f]);
		if (char_count == 1)
		{
			out+=('=');
			out+=('=');
		}
		else
		{
			out+=(alphabet[(bits >> 6) & 0x3f]);
			out+=('=');
		}
	}
	return out;
}

void PostProcConfig() 
{
	if(stupidfs)
		cerr << "Warning, support for primitive filesystems is not completely implemented!\n";
	
    // postprocessing

#ifdef FORCE_CUSTOM_UMASK
	if(!sUmask.empty())
	{
		mode_t nUmask=0;
		if(sUmask.size()>4)
		{
			cerr << "Invalid umask length\n";
			exit(EXIT_FAILURE);
		}
		for(unsigned int i=0; i<sUmask.size(); i++)
		{
			unsigned int val = sUmask[sUmask.size()-i-1]-'0';
			if(val>7)
			{
				cerr << "Invalid umask value\n" <<endl;
				exit(EXIT_FAILURE);
			}
			nUmask |= (val<<(3*i));
		
		}
		//cerr << "Got umask: " << nUmask <<endl;
		umask(nUmask);
	}
#endif
	
   if(cachedir.empty() || cachedir[0] != sPathSep[0]) {
      cerr << "Cache directory unknown or not absolute, terminating..." <<endl;
      exit(EXIT_FAILURE);
   }
   
   rechecksEx rex;
   if(rex.IsRegexOk())
   {
	   cerr << "An error occured while compiling file type regular expression!" <<endl;
	   exit(EXIT_FAILURE);
   }
   
   if(acfg::tpthreadmax < 0)
	   acfg::tpthreadmax = numeric_limits<int>::max();
	   
   // get rid of duplicated and trailing slash(es)
   tStrVec tmp;
   Tokenize(cachedir, sPathSep.c_str(), tmp);
   Join(cachedir, sPathSep, tmp);
   
   if(!pidfile.empty() && pidfile.at(0) != cPathSep)
   {
	   cerr << "Pid file path must be absolute, terminating..."  <<endl;
	         exit(EXIT_FAILURE);
   }
   
   if(! proxy.empty()) {
	   if(!proxy_info.SetHttpUrl(proxy))
	   {
		   cerr << "Invalid proxy specification, aborting..." << endl;
		   exit(EXIT_FAILURE);
	   }
	   
	   tStrPos pos=proxy_info.sHost.find('@');
	   if(stmiss != pos)
	   {
		   proxy_info.sPath=string("Proxy-Authorization: Basic ")+_GetBase64Auth(proxy.substr(0, pos))+"\r\n";
		   proxy_info.sHost.erase(0,pos+1);
	   }
     else
        proxy_info.sPath.clear(); // no auth at all, stop confusion

	   if(proxy_info.sHost.empty())
	   {
		   cerr << "Invalid proxy specification, aborting..." << endl;
		   exit(EXIT_FAILURE);
	   }
	   if (proxy_info.sPort.empty())
		{
			if (proxy_info.sPath.empty()) // guess unless there is any risk...
			{
				cerr << "Warning, unknown proxy port, assuming 80." <<endl;
				proxy_info.sPort="80";
			}
			else
			{
				cerr << "Error, unknown proxy port!"<<endl;
				exit(EXIT_FAILURE);
			}
	   }
   }
   
   if(!adminauth.empty())
	   adminauth=string("Basic ")+_GetBase64Auth(adminauth);
   
   // create working paths before something else fails somewhere
   if(!fifopath.empty())
	   mkbasedir(acfg::fifopath);
   if(!cachedir.empty())
	   mkbasedir(acfg::cachedir);
   if(! pidfile.empty())
	   mkbasedir(acfg::pidfile);

   if(nettimeout < 5) {
	   cerr << "Warning, NetworkTimeout too small, assuming 5." << endl;
	   nettimeout = 5;
   }

   if(RESERVED_DEFVAL == forwardsoap)
	   forwardsoap = !forcemanaged;

#ifdef DEBUG
   if (acfg::debug>8)
	{
		for (UINT i=0; i<_countof(n2sTbl); i++)
		{
			if(! n2sTbl[i].ptr)
				continue;

			string tmp, *val=n2sTbl[i].ptr;
			cout << n2sTbl[i].name << " = " << *val <<endl;
			// escaped version
			for (const char *p=val->c_str(); *p; p++)
				tmp.append('\\'==*p ? 2 : 1, *p);
			cout << n2sTbl[i].name << " = " << tmp << endl;
		}
		for (UINT i=0; i<_countof(n2iTbl); i++)
			if(n2iTbl[i].ptr)
				cout << n2iTbl[i].name << " = \"" << *(n2iTbl[i].ptr) << "\"\n";
	}
#else
   if(acfg::debug>2)
	   cout << "\n\nAdditional debugging information not compiled in.\n\n";
#endif
   
   
   /*
   // help STL saving some memory
   for(tUrl2RepIter it=mapUrl2pVname.begin(); it!=mapUrl2pVname.end(); it++)
   {
	   tUrl2RepIter suc=it;
	   suc++;
	   if(suc==mapUrl2pVname.end()) break;
	   if(suc->first.sPath==it->first.sPath)
		   const_cast<string&>(suc->first.sPath)=it->first.sPath;
   }
   */
}
}

/*
int main(int argc, char **argv)
{
	if(argc<2)
		return -1;
	
	acfg::tHostInfo hi;
	cout << "Parsing " << argv[1] << ", result: " << hi.SetUrl(argv[1])<<endl;
	cout << "Host: " << hi.sHost <<", Port: " << hi.sPort << ", Path: " << hi.sPath<<endl;
	return 0;
}
*/

