#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

from atabake.lib.player_session import PlayerSession

class Player(object):
    """Interface for media backends.

    This should be just a basic interface to be used by plugins
    in order to guide them about the basic functions to be used.

    Plugins that do not implement this interface will not be
    loaded by the engine.

    NOTE: all methods must return a boolean

    """

    def __init__(self, url=None, xid=None, get_state=None,
                 eos_cb=None, error_cb=None, buffering_cb=None):
        """You must give an url to the player and a pointer
        to a function that returns the state of the session,
        so the player can know it's own state"""
        self.url = url
        self.xid = xid
        self.preferences = {}
        self.error_code = None
        self.get_state = get_state
        self._eos_callback = eos_cb
        self._error_callback = error_cb
        self.buffering_callback = buffering_cb

    def error_callback(self, error):
        self.error_code = error
        self._error_callback(error)

    def eos_callback(self):
        if self.error_code is None:
            self._eos_callback()

    def reset_player(self):
        self.error_code = None

    def is_paused(self):
        return self.get_state() == PlayerSession.STATE_PAUSED

    def is_playing(self):
        return self.get_state() == PlayerSession.STATE_PLAYING

    def is_idle(self):
        return self.get_state() == PlayerSession.STATE_NONE

    def is_holded(self):
        return self.get_state() == PlayerSession.STATE_HOLD

    def delete(self):
        self._eos_callback = None
        self._error_callback = None
        self.buffering_callback = None

    def play(self):
        raise NotImplementedError("You must implement play() to have "
                                  "a fully working player.")

    def pause(self):
        raise NotImplementedError("You must implement pause() to have "
                                  "a fully working player.")

    def stop(self):
        raise NotImplementedError("You must implement stop() to have "
                                  "a fully working player.")

    def is_seekable(self):
        raise NotImplementedError("You must implement is_seekable() to have "
                                  "a fully working player.")

    def seek(self, position):
        raise NotImplementedError("You must implement seek(position) "
                                  "to have a fully working player.")

    ## Setters

    def set_uri(self, uri):
        self.url = uri
        return True

    def load_preferences(self, preferences):
        self.preferences = preferences
        return True

    def set_volume(self, value):
        raise NotImplementedError("You must implement set_volume(value) "
                                  "to have a fully working player.")

    def set_fullscreen(self, status):
        raise NotImplementedError("You must implement set_fullscreen(status) "
                                  "to have a fully working player.")

    def set_video_window(self, xid):
        raise NotImplementedError("You must implement set_video_window(xid) "
                                  "to have a fully working player.")

    ## Getters

    def get_position(self):
        raise NotImplementedError("You must implement get_position to have "
                                  "a fully working player.")

    def get_duration(self):
        raise NotImplementedError("You must implement get_duration to have "
                                  "a fully working player.")

    def get_media_details(self):
        raise NotImplementedError("You must implement get_media_details to have "
                                  "a fully working player.")

    def __str__(self):
        return '%s(player=%r, url=%r, xid=%s)' % \
               (self.__class__.__name__, id(self),
                self.url, self.xid)
